/* ChromaticNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import java.util.Arrays;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.computers.AbstractInvariantComputer;
import org.grinvin.invariants.values.IntegerValue;

/**
 * computes the chromatic number of the graph.
 */
public class ChromaticNumber extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        //first find the maximum degree: chromatic number <= max.degree + 1
        int[][] adjlist = bundle.adjacencyList();
        int n = adjlist.length;
        if(n==0)
            return IntegerValue.undefinedValue(this);
        if(n==1)
            return new IntegerValue(1, this);
        int maxdeg = 0;
        int mindeg = Integer.MAX_VALUE;
        for (int i = 0; i < n; i++){
            if (adjlist[i].length > maxdeg) 
                maxdeg = adjlist[i].length;
            if (adjlist[i].length < mindeg)
                mindeg = adjlist[i].length;
        }
        
        if(maxdeg == n-1 && mindeg == n-1)
            return new IntegerValue(n, this); //complete graph
        if(maxdeg == 2 && mindeg == 2 && n % 2 == 1)
            return new IntegerValue(3, this); //odd cycle (possible different components)
      
        //we construct the different partitions and label the partitions with the smallest vertex in it.
        int[] graphPartition = new int[n];
        Arrays.fill(graphPartition, 0);
        
        return new IntegerValue(colorGraph(graphPartition, adjlist, 1, maxdeg+1), this);
    }
    
    private int colorGraph(int[] graphPartition, int[][] adjlist, int currentVertex, int minimumColors){
        //first count the number of partition we currently have (only need to count to the current vertex)
        int nrOfPartitions=0;
        for(int i=0; i<currentVertex; i++)
            if(graphPartition[i]==i) // partition is labeled by its smallest member
                nrOfPartitions++;
        if(nrOfPartitions >= minimumColors)
            //we won't find anything better
            return minimumColors;
        if(currentVertex==graphPartition.length)
            //all vertices have been colored and a smaller number of partitions have been used
            return nrOfPartitions;
        
        //now color the current vertex
        
        //first try all the partition that already exist
        for(int i=0; i<currentVertex; i++){
            if(graphPartition[i]==i){
                boolean goodPartition = true;
                for(int vertex : adjlist[currentVertex])
                    goodPartition = goodPartition && (vertex >= currentVertex || graphPartition[vertex] != i);
                if(goodPartition){
                    graphPartition[currentVertex]=i;
                    minimumColors = colorGraph(graphPartition, adjlist, currentVertex + 1, minimumColors);
                }
            }
        }
        //then try to put the vertex in its own partition
        graphPartition[currentVertex]=currentVertex;
        minimumColors = colorGraph(graphPartition, adjlist, currentVertex + 1, minimumColors);
        
        return minimumColors;
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.ChromaticNumber";
    }
}
