/* ExpressionToString.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.expr;

import org.grinvin.invariants.Invariant;

/**
 * Visitor which converts an expression to a string.<p>
 * <b>Note:</b> This class is not thread safe.
 */
public class ExpressionToString implements Expression.Visitor {
    
    //
    private String result;
    
    //
    @SuppressWarnings("PMD.SingularField")
    private ExpressionToString childVisitor;
    
    /** 
     * Get a visitor for visiting children of expressions.
     */
    private ExpressionToString getChildVisitor () {
        if (childVisitor == null)
            childVisitor = new ExpressionToString ();
        return childVisitor;
    }
    
    /**
     * Return the resulting string.
     */
    public String getString () {
        return this.result;
    }
    
    //
    private int strength;
    
    /**
     * Return the strength of the resulting string. (Used to determine
     * whether parentheses are needed.)
     */
    public int getStrength () {
        return this.strength;
    }
    
    public void visitInvariant(Invariant invariant) {
        this.result = invariant.getName();
        this.strength = Integer.MAX_VALUE;
    }
    
    public void visitIntegerLiteral(IntegerLiteral literal) {
        this.result = literal.toString();
        this.strength = Integer.MAX_VALUE;
    }

    public void visitCompound(Compound compound) {
        Operator operator = compound.getOperator();
                ExpressionToString childVisitor = getChildVisitor ();
        int numberOfArguments = compound.getNumberOfChildren();
        this.strength = operator.getStrength();
        if (numberOfArguments == 1) {
            // unary operator
            
            compound.getChild(0).accept (childVisitor);
            String str = childVisitor.getString();
            if (this.strength >= childVisitor.getStrength())
                str = "(" + str + ")";
            this.result = operator.getSymbol() + " " + str;
            
        } else if (numberOfArguments == 2) {
            // binary operator (assumed left associative)
            
            compound.getChild(0).accept (childVisitor);
            String str1 = childVisitor.getString();
            if (this.strength > childVisitor.getStrength())
                str1 = "(" + str1 + ")";
            
            compound.getChild(1).accept (childVisitor);
            String str2 = childVisitor.getString();
            if (this.strength >= childVisitor.getStrength())
                str2 = "(" + str2 + ")";
            
            this.result = str1 + " " + operator.getSymbol() + " " + str2;
            
        } else {
            assert false;
        }
    }
    
}
