package org.inria.bmajwatcher.server.data;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.inria.biomaj.exe.options.GenStatistiquesChart;
import org.inria.biomaj.exe.options.ObjectStat;
import org.inria.biomaj.session.bank.BiomajSQLQuerier;
import org.inria.biomaj.session.bank.ProductionDirectory;
import org.inria.biomaj.utils.BiomajException;
import org.inria.biomaj.utils.BiomajUtils;
import org.inria.bmajwatcher.server.utils.Utils;

/**
 * Queries the database via biomaj to retrieve banks data used
 * to build statistics.
 * 
 * 
 * @author rsabas
 *
 */
public class StatsRetriever {
	
	private static Logger log = Logger.getLogger(StatsRetriever.class);

	public static List<Map<String, String>> getBankStats(String bankName) {
		try {
			List<ObjectStat> v = GenStatistiquesChart.getStatVectorSize(bankName);
			return objectStatToMap(v);
		} catch (BiomajException e) {
			log.error("", e);
			return null;
		}
	}
	
	
	/**
	 * Returns for each bank in the database its name, type,
	 * size and server.
	 * 
	 * @return
	 */
	public static List<Map<String, String>> getBanksSize() {
		
		List<Map<String, String>> res = new ArrayList<Map<String,String>>();
		List<String> banks = BiomajSQLQuerier.getBanks();
		for (String bank : banks) {
			List<ProductionDirectory> dirs = BiomajSQLQuerier.getAvailableProductionDirectories(bank);
			if (dirs.size() == 0)
				continue;
			
			long size = 0;
			for (ProductionDirectory dir : dirs)
				size += dir.getSize();
			
			Map<String, String> map = BiomajSQLQuerier.getBankInfo(bank);
			
			Map<String, String> info = new HashMap<String, String>();
			info.put("name", map.get(BiomajSQLQuerier.BANK_NAME));
			info.put("size", Utils.byteToMB(size));
			info.put("type", map.get(BiomajSQLQuerier.DB_TYPE));
			info.put("server", map.get(BiomajSQLQuerier.SERVER));
			
			res.add(info);
		}
		Map<String, String> freeSpace = new HashMap<String, String>();
		freeSpace.put("name", "Free space");
		freeSpace.put("size", String.valueOf(Utils.byteToGB(BiomajUtils.getFreeSize())));
		freeSpace.put("type", "");
		freeSpace.put("server", "");
		res.add(freeSpace);
		
		return res;
	}
	
	/**
	 * Transforms an objectStat into a list of maps to be sent over the network
	 * to the client.
	 *  
	 * @param list
	 * @return
	 */
	private static List<Map<String, String>> objectStatToMap(List<ObjectStat> list) {
		List<Map<String, String>> res = new ArrayList<Map<String,String>>();
		for (ObjectStat obj : list) {
			Map<String, String> map = new HashMap<String, String>();
			map.put("bandwidth", String.valueOf(obj.getBandWidth()));
			map.put("globaltime", String.valueOf(obj.getGlobalTime() / 1000)); // Time diff between start and end
			map.put("nbfiledownload", String.valueOf(obj.getNbFileDownload()));
			map.put("nbfileflat", String.valueOf(obj.getNbFileFlat()));
			map.put("nbsession", String.valueOf(obj.getNbSession()));
			map.put("release", obj.getRelease());
			map.put("size", Utils.byteToMB(obj.getSize()));
			map.put("sizecompressed", String.valueOf(obj.getSizeCompressed()));
			map.put("time", String.valueOf(obj.getTime().getTime()));
			map.put("truetime", String.valueOf(obj.getTrueTime() / 1000));
			
			res.add(map);
		}
		
		return res;
	}
	
	
	
}
