package org.inria.bmajwatcher.server;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeSet;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.inria.biomaj.exe.bank.BankFactory;
import org.inria.biomaj.exe.bank.BiomajBank;
import org.inria.biomaj.session.bank.BiomajSQLQuerier;
import org.inria.biomaj.sql.SQLConnection;
import org.inria.biomaj.sql.SQLConnectionFactory;
import org.inria.biomaj.utils.BiomajConst;
import org.inria.biomaj.utils.BiomajException;

/**
 * Servlet that returns in JSON format information
 * on banks.
 * 
 * @author rsabas
 *
 */
public class BankQueryingService extends HttpServlet {
	
	private static final String BLAST = "blast";

	private static final long serialVersionUID = 7729370645569188938L;
	private static Logger log = Logger.getLogger(BankQueryingService.class);
	
	private static String FORMAT = "format";
	private static String TYPE = "type";

	@Override
	protected void doGet(HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		
//		fetchDataFromStyledURL(req.getRequestURL().toString(), resp.getWriter());
		fetchDataFromClassicURL(resp.getWriter(), req);
	}
	
	@Override
	protected void doPost(HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		
//		fetchDataFromStyledURL(req.getRequestURL().toString(), resp.getWriter());
		fetchDataFromClassicURL(resp.getWriter(), req);
	}
	
	
	private void fetchDataFromClassicURL(PrintWriter writer, HttpServletRequest req) {
		
		Map<?, ?> params = req.getParameterMap();
		String type = "ALL";
		String format = "ALL";
		String banks = "ALL";
		boolean lightMode = false;
		boolean typeOk = false;
		boolean banksOk = false;
		boolean formatOk = false;
		
		for (Object s : params.keySet()) {
			String paramName = s.toString();
			if (paramName.equalsIgnoreCase("types")) {
				type = req.getParameter(paramName);
				typeOk = true;
			} else if (paramName.equalsIgnoreCase("banks")) {
				banks = req.getParameter(paramName);
				banksOk = true;
			} else if (paramName.equalsIgnoreCase("formats")) {
				format = req.getParameter(paramName);
				formatOk = true;
			} else if (paramName.equalsIgnoreCase("lightmode")) {
				lightMode=true;
			}
			
			if (typeOk && banksOk && formatOk)
				break;
		}
		
		log.debug("REST request parameters = banks : " + banks + " // type : " + type + " // format : " + format + " // lightmode : " + lightMode);
		
		if (typeOk && !banksOk && !formatOk)
			writer.println(collectionToJSON(getDBTypes(), "types"));
		else if (formatOk && !banksOk && !typeOk)
			writer.println(collectionToJSON(getFormats(), "formats"));
		else if (banksOk)
			writer.println(getBanks(type, format, banks, lightMode));
		
		writer.close();
	}
	
	
	
	/*
	private void fetchDataFromStyledURL(String url, PrintWriter writer) {
		
		log.debug("REST request : " + url);
		String[] params = url.substring(url.indexOf("GET/") + 4).split("/");
		if (params.length > 0) {
			if (params[0].equalsIgnoreCase("types")) { // Retrieve types list
				writer.println(collectionToJSON(getDBTypes(), "type"));
			} else if (params[0].equalsIgnoreCase("formats")) { // Retrieve formats list
				writer.println(collectionToJSON(getFormats(), "format"));
			} else if (params[0].equalsIgnoreCase("banks")) { // Retrieve banks list
				if (params.length > 2) {
					if (params[2].equalsIgnoreCase("type")) { // Filter by type
						if (params.length > 3) {
							if (params[3].equalsIgnoreCase("ALL")) { // All types
								writer.println(getBanks("ALL", "ALL", params[1]));
							} else { // Specific type
								if (params.length > 5) { // + specific format
									if (params[4].equalsIgnoreCase("format") && !params[5].equalsIgnoreCase("ALL"))
										writer.println(getBanks(params[3], params[5], params[1]));
									else
										writer.println(getBanks(params[2], "ALL", params[1]));
								} else { // Specific type, all formats
									writer.println(getBanks(params[3], "ALL", params[1]));
								}
							}
						} else { // All types if nothing specified
							writer.println(getBanks("ALL", "ALL", params[1]));
						}
					} else if (params[2].equalsIgnoreCase("format")) { // Filter by format
						if (params.length > 3) {
							if (params[3].equalsIgnoreCase("ALL")) { // All formats
								writer.println(getBanks("ALL", "ALL", params[1]));
							} else { // Banks corresponding to a specific format
								if (params.length > 5) { // + specific type
									if (params[4].equalsIgnoreCase("type") && !params[5].equalsIgnoreCase("ALL"))
										writer.print(getBanks(params[5], params[3], params[1]));
									else
										writer.println(getBanks("ALL", params[3], params[1]));
								} else
									writer.println(getBanks("ALL", params[3], params[1]));
							}
						} else { // All formats if nothing specified
							writer.println(getBanks("ALL", "ALL", params[1]));
						}
					} else { // Get a specific bank
						writer.println(getBanks("ALL", "ALL", params[1]));
					}
				} else if (params.length > 1) {
					writer.println(getBanks("ALL", "ALL", params[1]));
				} else {
					writer.println(getBanks("ALL", "ALL", "ALL"));
				}
			}
		}
		writer.close();
	}*/
	

	/**
	 * Returns the banks with the given formats, types and names.
	 * Each field has to be url encoded.
	 * 
	 * @param _type
	 * @param _format
	 * @param _name
	 * @param lightMode if true, do not list files for each format
	 * @return
	 */
	private String getBanks(String _type, String _format, String _name, boolean _lightMode) {
		String type;
		String format;
		String name;
		try {
			type = URLDecoder.decode(_type, "UTF-8");
			format = URLDecoder.decode(_format, "UTF-8");
			name = URLDecoder.decode(_name, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			return "";
		}
		
		StringBuilder allBanks = new StringBuilder();
		allBanks.append("{\"banks\":[");
		List<String> banks;
		if (name.equalsIgnoreCase("ALL"))
			banks = BiomajSQLQuerier.getBanks();
		else {
			banks = new ArrayList<String>();
			for (String bk : name.split("\\|"))
				banks.add(bk);
		}
		
		Map<String, String> directoryRelease = null;
		for (String bank : banks) {
			
			if ((directoryRelease = getDirectoriesRelease(bank)).size() == 0) {
				continue;
			}
			
			Map<String, String> info = getBankInfo(bank);
			if (format.equalsIgnoreCase("ALL") || containsFormat(format, info.get(FORMAT))) {
				if (type.equalsIgnoreCase("ALL") || typeContains(info.get(TYPE), type)) {
					allBanks.append(infoToJSON(bank, format, info, directoryRelease, _lightMode) + ",");
				}
			}
		}
		int index = allBanks.lastIndexOf(",");
		if (index > 0)
			allBanks.deleteCharAt(index);
		allBanks.append("]}");
		return allBanks.toString();
	}

	/**
	 * 
	 * @param format format as parameter in the url (f1|f2|...)
	 * @param curFormat format as in a properties file (f1,f2,...
	 * 
	 * @return true if any format in curFormat matches any format in format
	 */
	private boolean containsFormat(String format, String curFormat) {
		String[] curformats = curFormat.split(",");
		String[] formats = format.split("\\|");
		for (String curFmt : curformats) {
			for (String fmt : formats) {
				if (fmt.equalsIgnoreCase(curFmt))
					return true;
			}
		}
		return false;
	}
	
	/**
	 * 
	 * @param type type to compare to
	 * @param types list of types in the url (t1|t2|...)
	 * 
	 * @return
	 */
	private boolean typeContains(String type, String types) {
		String[] split = types.split("\\|");
		for (String s : split) {
			if (s.equalsIgnoreCase(type))
				return true;
		}
		
		return false;
	}
	
	/**
	 * Transforms given info about the bank into a JSON formatted string.
	 * 
	 * @param bankName
	 * @param _format
	 * @param info
	 * @param directories
	 * @param lightMode
	 * @return
	 */
	private String infoToJSON(String bankName, String _format, Map<String, String> info, Map<String, String> directories, boolean lightMode) {
		String[] formats = info.get(FORMAT).split(",");
		StringBuilder sb = new StringBuilder();
		int bankId = BiomajSQLQuerier.getBankId(bankName);
		
		sb.append("{");
		sb.append("\"name\":\"" + bankName + "\",");
		sb.append("\"session_date\":\"" + BiomajSQLQuerier.getLatestSessionDate(bankId) + "\",");
		sb.append("\"current_release\":\"" + getCurrentRelease(directories) + "\",");
		sb.append("\"releases\":{");
		for (String path : directories.keySet()) {
			sb.append("\"" + directories.get(path) + "\":{\"path\":\"" + path + "\",");
			
			sb.append("\"formats\":[");
			int fCount = 0;
			for (String format : formats) {
				if ((!format.trim().isEmpty() && _format.equals("ALL")) || _format.contains(format)) {
					sb.append("{\"value\":\"" + format + "\"");
					if (!lightMode) {
						sb.append("," + filesToJSON(path, format));
					}
					sb.append("},");
					fCount++;
				}
			}
			if (fCount > 0)
				sb.deleteCharAt(sb.lastIndexOf(","));
			sb.append("]");
			sb.append("},");
		}
		if (directories.size() > 0)
			sb.deleteCharAt(sb.lastIndexOf(","));
		sb.append("},");
		sb.append("\"db_type\":\"" + info.get(TYPE) + "\"");
		
		sb.append("}");
		
		return sb.toString();
	}

	/**
	 * Returns from the given releases, the path to the current one.
	 * 
	 * @param releases map that contains for each production directory path, its release
	 * @return
	 */
	private String getCurrentRelease(Map<String, String> releases) {
		for (String path : releases.keySet()) {
			String parent = path.substring(0, path.lastIndexOf('/'));
			File currentLink = new File(parent + "/current");
			if (currentLink.exists()) {
				try {
					String fullPath = currentLink.getCanonicalPath();
					String release = releases.get(fullPath);
					if (release != null)
						return release;
					else {
						log.warn("Path to current release was not found in db : " + fullPath);
						return "";
					}
				} catch (IOException e) {
					log.error(e);
				}
			} else {
				log.warn("Current release link does not exist : " + currentLink);
				return "";
			}
		}
		return "";
	}
	
	/**
	 * Returns for the given bank a map that contains the pairs
	 * (productionDirectoryPath, productionDirectoryRelease)
	 * 
	 * @param bank
	 * @return
	 */
	private Map<String, String> getDirectoriesRelease(String bank) {
		Map<String, String> res = new HashMap<String, String>();
		
		String query = "SELECT path,session FROM productionDirectory WHERE ref_idbank=" +
				"(SELECT idbank FROM bank WHERE name='" + bank + "') AND state='available'";
		SQLConnection connection = SQLConnectionFactory.getConnection();
		Statement stat = connection.getStatement();
		ResultSet rs = connection.executeQuery(query, stat);
		try {
			while (rs.next()) {
				query = "SELECT updateRelease FROM updateBank WHERE idupdateBank=" +
						"(SELECT ref_idupdateBank FROM session WHERE idsession=" + rs.getLong(BiomajSQLQuerier.DIR_SESSION) + ")";
				Statement stat2 = connection.getStatement();
				ResultSet rs2 = connection.executeQuery(query, stat2);
				if (rs2.next())
					res.put(rs.getString(BiomajSQLQuerier.DIR_PATH), rs2.getString(BiomajSQLQuerier.UPDATE_RELEASE));
				SQLConnectionFactory.closeConnection(stat2);
			}
			SQLConnectionFactory.closeConnection(stat);
		} catch (SQLException e) {
			log.error(e);
		}
		
		return res;
	}
	
	/**
	 * Transforms a collection of elements into a JSON formatted string.
	 * 
	 * @param elements
	 * @param type
	 * @return
	 */
	private String collectionToJSON(Collection<String> elements, String type) {
		StringBuilder sb = new StringBuilder();
		sb.append("{");
		sb.append("\"" + type + "\":[");
		for (String elt : elements) {
			sb.append("{\"value\":\"" + elt + "\"},");
		}
		int index = sb.lastIndexOf(",");
		if (index > 0)
			sb.deleteCharAt(index);
		sb.append("]}");
		return sb.toString();
	}
	
	
	/**
	 * Returns all the different bank formats.
	 * 
	 * @return
	 */
	private Collection<String> getFormats() {
		List<String> banks = BiomajSQLQuerier.getBanks();
		Collection<String> formats = new TreeSet<String>();
		for (String bank : banks) {
			String[] fmts = getBankInfo(bank).get(FORMAT).split(",");
			for (String format : fmts)
				if (!format.trim().isEmpty())
					formats.add(format);

		}
		return formats;
	}
	
	/**
	 * Returns the formats of the given bank in the corresponding
	 * properties file.
	 * 
	 * @param bankName
	 * @return
	 */
	private Map<String, String> getBankInfo(String bankName) {
		Map<String, String> res = new HashMap<String, String>();
		res.put(FORMAT, "");
		res.put(TYPE, "");
		try {
			BiomajBank b = new BankFactory().createBank(bankName, false);
			Properties props = b.getPropertiesFromBankFile();
			if (props.containsKey(BiomajConst.dbFormatsProperty))
				res.put(FORMAT, props.getProperty(BiomajConst.dbFormatsProperty));
			if (props.containsKey(BiomajConst.typeProperty))
				res.put(TYPE, props.getProperty(BiomajConst.typeProperty));
		} catch (BiomajException e1) {
			log.error(e1.getMessage());
			e1.printStackTrace();
		}
		
		return res;
	}
	
	/**
	 * Returns all the different bank types.
	 * 
	 * @return
	 */
	public static Collection<String> getDBTypes() {
		List<String> banks = BiomajSQLQuerier.getBanks();
		Collection<String> types = new TreeSet<String>();
		SQLConnection connection = SQLConnectionFactory.getConnection();
		Statement stat = connection.getStatement();
		for (String bankName : banks) {
			int bankId = BiomajSQLQuerier.getBankId(bankName);
			String query = "SELECT dbType FROM remoteInfo WHERE idremoteInfo = (SELECT ref_idremoteInfo FROM configuration " +
					"WHERE ref_idbank=" + bankId + " AND date=(SELECT max(date) FROM configuration WHERE ref_idbank=" + bankId + "))";
			ResultSet rs = connection.executeQuery(query, stat);
			try {
				if (rs != null && rs.next())
					types.add(rs.getString(1));
			} catch (SQLException e) {
				e.printStackTrace();
			}
		}
		SQLConnectionFactory.closeConnection(stat);
		
		return types;
	}
	
	/**
	 * Transforms a file list into a JSON expression.
	 * 
	 * @param root directory
	 * @param format sub directory which is usually the format name
	 * @return
	 */
	private String filesToJSON(String root, String format) {
		StringBuilder sections = new StringBuilder();
		File f = new File(root + "/" + format);
		sections.append("\"sections\":[");
		if (f.exists())
			listFiles(f, sections, format);
		sections.append("]");
		String res = sections.toString();
		
		return res.replaceAll(",\"sections\":\\[\\]", "").replaceAll("\"files\":\\[\\],", "");
	}
	
	
	/**
	 * Recursively lists all the files under given directory.
	 * 
	 * @param parentDir
	 * @param sections object that contains the sections
	 * @param format used for format specfic operations
	 */
	private void listFiles(File parentDir, StringBuilder sections, String format) {
		for (File f : parentDir.listFiles()) {
			
			// If it is a directory or a file directly under the format directory and that format is not blast
			// or is blast but the file ends with .nal
			if (f.isDirectory() || (!f.isDirectory() && f.getParentFile().getName().equals(format) &&
					(!format.equalsIgnoreCase(BLAST) || (format.equals(BLAST) && f.getName().endsWith(".nal"))))) {
				if (sections.charAt(sections.length() - 1) == '}')
					sections.append(",");
				
				sections.append("{");
				sections.append("\"name\":" + "\"" + f.getName() + "\",");
				
				sections.append("\"files\":[");
				List<String> currentFiles = filzOnly(f, format);
				for (String s : currentFiles) {
					sections.append("\"" + s + "\",");
				}
				if (currentFiles.size() > 0)
					sections.deleteCharAt(sections.lastIndexOf(","));
				sections.append("]");
			
				if (f.isDirectory()) {
					sections.append(",\"sections\":[");
					listFiles(f, sections, format);
					sections.append("]");
				}
				sections.append("}");
			}
		}
	}
	
	private List<String> filzOnly(File parent, String format) {
		List<String> filz = new ArrayList<String>();
		if (!parent.isDirectory() && parent.getParentFile().getName().equals(format)) { // Files directly under the format directory
			if (format.equalsIgnoreCase(BLAST)) {
				if (parent.getName().endsWith(".nal") || parent.getName().endsWith(".pal")) {
					filz.add(parent.getAbsolutePath());
				}
			} else {
				filz.add(parent.getAbsolutePath());
			}
		} else if (parent.isDirectory()) {
			File[] files = parent.listFiles();
			for (File f : files) {
				if (!f.isDirectory()) {
					if (format.equalsIgnoreCase(BLAST)) {
						if (f.getName().endsWith(".nal") || f.getName().endsWith(".pal")) {
							filz.add(f.getAbsolutePath());
						}
					} else {
						filz.add(f.getAbsolutePath());
					}
				}
			}
		}
		return filz;
	}
}
