/*
 * LSA function
 * Copyright (C) 1999 Yasuhiro Ohara
 *
 * This file is part of GNU Zebra.
 *
 * GNU Zebra is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * GNU Zebra is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Zebra; see the file COPYING.  If not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 * Boston, MA 02111-1307, USA.  
 */

#include "ospf6d.h"

#include <zebra.h>

#include "log.h"
#include "vty.h"
#include "thread.h"
#include "if.h"

#include "ospf6_prefix.h"
#include "ospf6_lsdb.h"
#include "ospf6_top.h"
#include "ospf6_area.h"
#include "ospf6_interface.h"
#include "ospf6_neighbor.h"
#include "ospf6_redistribute.h"
#include "ospf6_lsa.h"

char *ospf6_lsa_type_strings[] =
{
  "None",
  "Router-LSA",
  "Network-LSA",
  "Inter-Area-Prefix-LSA",
  "Inter-Area-Router-LSA",
  "AS-External-LSA",
  "Group-Membership-LSA",
  "Type-7-LSA",
  "Link-LSA",
  "Intra-Area-Prefix-LSA",
  NULL
};

void
ospf6_lsa_remove_all_reference (struct ospf6_lsa *lsa)
{
  listnode n;
  struct ospf6_neighbor *o6n;
  struct ospf6_interface *o6i;

  while (listcount (lsa->summary_nbr))
    {
      n = listhead (lsa->summary_nbr);
      o6n = (struct ospf6_neighbor *) getdata (n);
      ospf6_neighbor_summary_remove (lsa, o6n);
    }

  while (listcount (lsa->request_nbr))
    {
      n = listhead (lsa->request_nbr);
      o6n = (struct ospf6_neighbor *) getdata (n);
      ospf6_neighbor_request_remove (lsa, o6n);
    }

  while (listcount (lsa->retrans_nbr))
    {
      n = listhead (lsa->retrans_nbr);
      o6n = (struct ospf6_neighbor *) getdata (n);
      ospf6_neighbor_retrans_remove (lsa, o6n);
    }

  while (listcount (lsa->delayed_ack_if))
    {
      n = listhead (lsa->delayed_ack_if);
      o6i = (struct ospf6_interface *) getdata (n);
      ospf6_remove_delayed_ack (lsa, o6i);
    }
}

/* test LSAs identity */
int
ospf6_lsa_issame (struct ospf6_lsa_header *lsh1,
                  struct ospf6_lsa_header *lsh2)
{
  assert (lsh1 && lsh2);

  if (lsh1->advrtr != lsh2->advrtr)
    return 0;

  if (lsh1->ls_id != lsh2->ls_id)
    return 0;

  if (lsh1->type != lsh2->type)
    return 0;

  return 1;
}

int
ospf6_lsa_match (u_int16_t type, u_int32_t ls_id, u_int32_t advrtr,
                 struct ospf6_lsa_header *lsh)
{
  if (lsh->advrtr != advrtr)
    return 0;

  if (lsh->ls_id != ls_id)
    return 0;

  if (lsh->type != type)
    return 0;

  return 1;
}

/* ospf6 age functions */
/* calculate birth and set expire timer */
static void
ospf6_lsa_age_set (struct ospf6_lsa *lsa)
{
  struct timeval now;

  assert (lsa && lsa->lsa_hdr);

  if (gettimeofday (&now, (struct timezone *)NULL) < 0)
    zlog_warn ("Lsa: gettimeofday failed, may fail LSA AGEs: %s",
               strerror (errno));

  lsa->birth = now.tv_sec - ntohs (lsa->lsa_hdr->lsh_age);
  lsa->expire = thread_add_timer (master, ospf6_lsa_expire, lsa,
                                  lsa->birth + MAXAGE - now.tv_sec);
  return;
}

/* this function calculates current age from its birth,
   then update age field of LSA header. return value is current age */
u_int16_t
ospf6_lsa_age_current (struct ospf6_lsa *lsa)
{
  struct timeval now;
  u_int32_t ulage;
  u_int16_t age;
  struct ospf6_lsa_header *lsa_header;

  assert (lsa);
  lsa_header = (struct ospf6_lsa_header *) lsa->lsa_hdr;
  assert (lsa_header);

  /* current time */
  if (gettimeofday (&now, (struct timezone *)NULL) < 0)
    zlog_warn ("Lsa: gettimeofday failed, may fail ages: %s",
               strerror (errno));

  /* calculate age */
  ulage = now.tv_sec - lsa->birth;

  /* if over MAXAGE, set to it */
  if (ulage > MAXAGE)
    age = MAXAGE;
  else
    age = ulage;

  lsa_header->age = htons (age);
  return age;
}

/* returns 1 if LSA's age is MAXAGE. else returns 0 */
int
ospf6_lsa_is_maxage (struct ospf6_lsa *lsa)
{
  if (ospf6_lsa_age_current (lsa) == MAXAGE)
    return 1;
  return 0;
}

/* update age field of LSA header with adding InfTransDelay */
void
ospf6_lsa_age_update_to_send (struct ospf6_lsa *lsa,
                              struct ospf6_interface *o6i)
{
  unsigned short age;
  struct ospf6_lsa_header *lsa_header;

  lsa_header = (struct ospf6_lsa_header *) lsa->lsa_hdr;
  age = ospf6_lsa_age_current (lsa) + o6i->transdelay;
  if (age > MAXAGE)
    age = MAXAGE;
  lsa_header->age = htons (age);
  return;
}

void
ospf6_lsa_premature_aging (struct ospf6_lsa *lsa)
{
  /* log */
  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("Lsa: Premature aging: %s", lsa->str);

  if (lsa->expire)
    thread_cancel (lsa->expire);
  lsa->expire = (struct thread *) NULL;
  if (lsa->refresh)
    thread_cancel (lsa->refresh);
  lsa->refresh = (struct thread *) NULL;

  lsa->birth = 0;
  thread_execute (master, ospf6_lsa_expire, lsa, 0);
}

/* check which is more recent. if a is more recent, return -1;
   if the same, return 0; otherwise(b is more recent), return 1 */
int
ospf6_lsa_check_recent (struct ospf6_lsa *a, struct ospf6_lsa *b)
{
  signed long seqnuma, seqnumb;
  u_int16_t agea, ageb;

  assert (a && a->lsa_hdr);
  assert (b && b->lsa_hdr);
  assert (ospf6_lsa_issame ((struct ospf6_lsa_header *) a->lsa_hdr,
                            (struct ospf6_lsa_header *) b->lsa_hdr));

  seqnuma = ((signed long) ntohl (a->lsa_hdr->lsh_seqnum))
             - (signed long)INITIAL_SEQUENCE_NUMBER;
  seqnumb = ((signed long) ntohl (b->lsa_hdr->lsh_seqnum))
             - (signed long)INITIAL_SEQUENCE_NUMBER;

  /* compare by sequence number */
    /* xxx, care about LS sequence number wrapping */
  if (seqnuma > seqnumb)
    return -1;
  else if (seqnuma < seqnumb)
    return 1;

  /* xxx, Checksum */

  /* Age check */
  agea = ospf6_lsa_age_current (a);
  ageb = ospf6_lsa_age_current (b);

    /* MaxAge check */
  if (agea == OSPF6_LSA_MAXAGE && ageb != OSPF6_LSA_MAXAGE)
    return -1;
  else if (agea != OSPF6_LSA_MAXAGE && ageb == OSPF6_LSA_MAXAGE)
    return 1;

  if (agea > ageb && agea - ageb >= OSPF6_LSA_MAXAGEDIFF)
    return 1;
  else if (agea < ageb && ageb - agea >= OSPF6_LSA_MAXAGEDIFF)
    return -1;

  /* neither recent */
  return 0;
}

/* vty function for LSAs */
void
ospf6_lsa_vty_body_unknown (struct vty *vty,
                            struct ospf6_lsa_header *lsa_header)
{
  vty_out (vty, "    Unknown Type%s", VTY_NEWLINE);
}

void
ospf6_lsa_vty_body_router (struct vty *vty,
                           struct ospf6_lsa_header *lsa_header)
{
  int lsdnum;
  char buf[32];
  struct router_lsa *router_lsa;
  struct router_lsd *router_lsd;

  assert (lsa_header);
  router_lsa = (struct router_lsa *)(lsa_header + 1);
  router_lsd = (struct router_lsd *)(router_lsa + 1);

  lsdnum = (ntohs (lsa_header->length) - sizeof (struct ospf6_lsa_header)
            - sizeof (struct router_lsa)) / sizeof (struct router_lsd);
  assert (lsdnum >= 0);

  while (lsdnum --)
    {
      vty_out (vty, "  Type: %s Metric: %d%s",
               rlsatype_name[router_lsd->rlsd_type - 1],
               ntohs(router_lsd->rlsd_metric), VTY_NEWLINE);
      vty_out (vty, "  Interface ID: %lu%s",
               ntohl (router_lsd->rlsd_interface_id), VTY_NEWLINE);
      vty_out (vty, "  Neighbor Interface ID: %lu%s",
               ntohl (router_lsd->rlsd_neighbor_interface_id), VTY_NEWLINE);
      vty_out (vty, "  Neighbor Router ID: %s%s",
               inet_ntop (AF_INET, &router_lsd->rlsd_neighbor_router_id,
                          buf, sizeof (buf)), VTY_NEWLINE);

      router_lsd++;
    }
}

void
ospf6_lsa_vty_body_network (struct vty *vty,
                            struct ospf6_lsa_header *lsa_header)
{
  int lsdnum;
  struct ospf6_network_lsa *lsa;
  u_int32_t *router_id;
  char *ptr, buf[128];

  assert (lsa_header);
  lsa = (struct ospf6_network_lsa *) (lsa_header + 1);
  router_id = (u_int32_t *)(lsa + 1);

  ptr = buf;
  if (OSPF6_OPT_ISSET (lsa->options, OSPF6_OPT_V6))
    {
      *ptr++ = 'V'; *ptr++ = '6'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (lsa->options, OSPF6_OPT_E))
    {
      *ptr++ = 'E'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (lsa->options, OSPF6_OPT_MC))
    {
      *ptr++ = 'M'; *ptr++ = 'C'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (lsa->options, OSPF6_OPT_N))
    {
      *ptr++ = 'N'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (lsa->options, OSPF6_OPT_R))
    {
      *ptr++ = 'R'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (lsa->options, OSPF6_OPT_DC))
    {
      *ptr++ = 'D'; *ptr++ = 'C'; *ptr++ = ',';
    }
  *ptr++ = '\0';
  vty_out (vty, "     Options: %s%s", buf, VTY_NEWLINE);

  lsdnum = (ntohs (lsa_header->length) - sizeof (struct ospf6_lsa_header)
            - sizeof (struct ospf6_network_lsa)) / sizeof (u_int32_t);
  assert (lsdnum >= 0);

  while (lsdnum--)
    {
      inet_ntop (AF_INET, router_id, buf, sizeof (buf));
      vty_out (vty, "     Attached Router: %s%s", buf, VTY_NEWLINE);
      router_id ++;
    }
}

void
ospf6_lsa_vty_body_inter_prefix (struct vty *vty,
                                 struct ospf6_lsa_header *lsa_header)
{
  struct ospf6_lsa_inter_area_prefix_lsa *inter_prefix_lsa;
  struct ospf6_prefix *prefix;
  struct in6_addr in6;
  char buf[128];

  assert (lsa_header);
  inter_prefix_lsa = (struct ospf6_lsa_inter_area_prefix_lsa *) (lsa_header + 1);

  /* Metric */
  vty_out (vty, "     Metric: %d%s", ntohl (inter_prefix_lsa->metric),
           VTY_NEWLINE);

  prefix = (struct ospf6_prefix *)(inter_prefix_lsa + 1);

  /* Prefix Options */
  ospf6_prefix_options_str (prefix, buf, sizeof (buf));
  vty_out (vty, "     Prefix Options: %s%s", buf, VTY_NEWLINE);

  /* Address Prefix */
  ospf6_prefix_in6_addr (prefix, &in6);
  inet_ntop (AF_INET6, &in6, buf, sizeof (buf));
  vty_out (vty, "     Prefix: %s/%d%s",
           buf, prefix->prefix_length, VTY_NEWLINE);
}

void
ospf6_lsa_vty_body_inter_router (struct vty *vty,
                                 struct ospf6_lsa_header *lsa_header)
{
  struct ospf6_lsa_inter_area_router_lsa *inter_router_lsa;
  char *ptr, buf[128];

  assert (lsa_header);
  inter_router_lsa = (struct ospf6_lsa_inter_area_router_lsa *) (lsa_header + 1);

  /* Options */
  ptr = buf;
  if (OSPF6_OPT_ISSET (inter_router_lsa->options, OSPF6_OPT_V6))
    {
      *ptr++ = 'V'; *ptr++ = '6'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (inter_router_lsa->options, OSPF6_OPT_E))
    {
      *ptr++ = 'E'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (inter_router_lsa->options, OSPF6_OPT_MC))
    {
      *ptr++ = 'M'; *ptr++ = 'C'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (inter_router_lsa->options, OSPF6_OPT_N))
    {
      *ptr++ = 'N'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (inter_router_lsa->options, OSPF6_OPT_R))
    {
      *ptr++ = 'R'; *ptr++ = ',';
    }
  if (OSPF6_OPT_ISSET (inter_router_lsa->options, OSPF6_OPT_DC))
    {
      *ptr++ = 'D'; *ptr++ = 'C'; *ptr++ = ',';
    }
  *ptr++ = '\0';
  vty_out (vty, "     Options: %s%s", buf, VTY_NEWLINE);

  /* Metric */
  vty_out (vty, "     Metric: %d%s", ntohl (inter_router_lsa->metric),
           VTY_NEWLINE);

  /* Destination Router ID */
  inet_ntop (AF_INET, &inter_router_lsa->router_id, buf, sizeof (buf));
  vty_out (vty, "     Destination Router ID: %d%s", buf);
}

void
ospf6_lsa_vty_body_as_external (struct vty *vty,
                                struct ospf6_lsa_header *lsa_header)
{
  struct ospf6_as_external_lsa *elsa;
  char buf[128], *ptr;
  struct in6_addr in6;

  assert (lsa_header);
  elsa = (struct ospf6_as_external_lsa *)(lsa_header + 1);

  /* bits */
  ptr = buf;
  if (ASE_LSA_ISSET (elsa, ASE_LSA_BIT_E))
    *ptr++ = 'E';
  else
    *ptr++ = ' ';

  if (ASE_LSA_ISSET (elsa, ASE_LSA_BIT_F))
    *ptr++ = 'F';
  else
    *ptr++ = ' ';

  if (ASE_LSA_ISSET (elsa, ASE_LSA_BIT_T))
    *ptr++ = 'T';
  else
    *ptr++ = ' ';
  *ptr = '\0';

  vty_out (vty, "     Bits: %3s%s", buf, VTY_NEWLINE);
  vty_out (vty, "     Metric: %5hu%s", ntohs (elsa->ase_metric), VTY_NEWLINE);

  ospf6_prefix_options_str (&elsa->ospf6_prefix, buf, sizeof (buf));
  vty_out (vty, "     Prefix Options: %s%s", buf, VTY_NEWLINE);

  vty_out (vty, "     Referenced LSType: %d%s",
           ntohs (elsa->ospf6_prefix.prefix_refer_lstype), VTY_NEWLINE);

  ospf6_prefix_in6_addr (&elsa->ospf6_prefix, &in6);
  inet_ntop (AF_INET6, &in6, buf, sizeof (buf));
  vty_out (vty, "     Prefix: %s/%d%s",
           buf, elsa->ospf6_prefix.prefix_length, VTY_NEWLINE);

  /* Forwarding-Address */
  if (ASE_LSA_ISSET (elsa, ASE_LSA_BIT_F))
    {
      ptr = ((char *)(elsa + 1))
            + OSPF6_PREFIX_SPACE (elsa->ospf6_prefix.prefix_length);
      inet_ntop (AF_INET6, (struct in6_addr *) ptr, buf, sizeof (buf));
      vty_out (vty, "     Forwarding-Address: %s%s", buf, VTY_NEWLINE);
    }
}

void
ospf6_lsa_vty_body_group_membership (struct vty *vty,
                                     struct ospf6_lsa_header *lsa_header)
{
}

void
ospf6_lsa_vty_body_type_7 (struct vty *vty,
                           struct ospf6_lsa_header *lsa_header)
{
}

void
ospf6_lsa_vty_body_link (struct vty *vty,
                         struct ospf6_lsa_header *lsa_header)
{
  struct link_lsa *llsap;
  int prefixnum;
  struct ospf6_prefix *prefix;
  char buf[128];
  struct in6_addr in6;

  assert (lsa_header);

  llsap = (struct link_lsa *) (lsa_header + 1);
  prefixnum = ntohl (llsap->llsa_prefix_num);

  inet_ntop (AF_INET6, (void *)&llsap->llsa_linklocal, buf, sizeof (buf));
  vty_out (vty, "     LinkLocal Address: %s%s", buf, VTY_NEWLINE);
  vty_out (vty, "     Number of Prefix: %d%s", prefixnum, VTY_NEWLINE);

  prefix = (struct ospf6_prefix *)(llsap + 1);

  while (prefixnum--)
    {
      ospf6_prefix_options_str (prefix, buf, sizeof (buf));
      vty_out (vty, "     Prefix Options: %s%s", buf, VTY_NEWLINE);

      ospf6_prefix_in6_addr (prefix, &in6);
      inet_ntop (AF_INET6, &in6, buf, sizeof (buf));
      vty_out (vty, "     Prefix: %s/%d%s",
               buf, prefix->prefix_length, VTY_NEWLINE);

      prefix = OSPF6_NEXT_PREFIX (prefix);
    }
}

void
ospf6_lsa_vty_body_intra_prefix (struct vty *vty,
                                 struct ospf6_lsa_header *lsa_header)
{
  struct intra_area_prefix_lsa *iap_lsa;
  struct ospf6_prefix *prefix;
  unsigned short prefixnum;
  char buf[128];
  struct in6_addr in6;

  assert (lsa_header);
  iap_lsa = (struct intra_area_prefix_lsa *) (lsa_header + 1);
  prefixnum = ntohs (iap_lsa->intra_prefix_num);

  vty_out (vty, "     Number of Prefix: %d%s", prefixnum, VTY_NEWLINE);
  vty_out (vty, "     Referenced LS Type: %s%s",
           ospf6_lsa_type_string (iap_lsa->intra_prefix_refer_lstype),
           VTY_NEWLINE);
  vty_out (vty, "     Referenced LS ID: %d%s",
           ntohl (iap_lsa->intra_prefix_refer_lsid), VTY_NEWLINE);
  inet_ntop (AF_INET, &iap_lsa->intra_prefix_refer_advrtr, buf, sizeof (buf));
  vty_out (vty, "     Referenced Advertising Router: %s%s",
           buf, VTY_NEWLINE);

  prefix = (struct ospf6_prefix *)(iap_lsa + 1);
  while (prefixnum--)
    {
      ospf6_prefix_options_str (prefix, buf, sizeof (buf));
      vty_out (vty, "     Prefix Options: %s%s", buf, VTY_NEWLINE);

      ospf6_prefix_in6_addr (prefix, &in6);
      inet_ntop (AF_INET6, &in6, buf, sizeof (buf));
      vty_out (vty, "     Prefix: %s/%d%s",
               buf, prefix->prefix_length, VTY_NEWLINE);

      prefix = OSPF6_NEXT_PREFIX (prefix);
    }
}

void
(*ospf6_lsa_vty_body[OSPF6_LSA_TYPE_MAX]) (struct vty *,
                                           struct ospf6_lsa_header *) =
{
  ospf6_lsa_vty_body_unknown,
  ospf6_lsa_vty_body_router,
  ospf6_lsa_vty_body_network,
  ospf6_lsa_vty_body_inter_prefix,
  ospf6_lsa_vty_body_inter_router,
  ospf6_lsa_vty_body_as_external,
  ospf6_lsa_vty_body_group_membership,
  ospf6_lsa_vty_body_type_7,
  ospf6_lsa_vty_body_link,
  ospf6_lsa_vty_body_intra_prefix,
};

void
ospf6_lsa_vty (struct vty *vty, struct ospf6_lsa *lsa)
{
  struct ospf6_lsa_header *lsa_header;
  char advrtr[64];

  assert (lsa);
  lsa_header = (struct ospf6_lsa_header *) lsa->lsa_hdr;
  assert (lsa_header);

  inet_ntop (AF_INET, &lsa_header->advrtr, advrtr, sizeof (advrtr));

  vty_out (vty, "Age: %4hu Type: %s%s", ospf6_lsa_age_current (lsa),
           ospf6_lsa_type_string (lsa_header->type), VTY_NEWLINE);
  vty_out (vty, "Link State ID: %lu%s", ntohl (lsa_header->ls_id),
           VTY_NEWLINE);
  vty_out (vty, "Advertising Router: %s%s", advrtr, VTY_NEWLINE);
  vty_out (vty, "LS Sequence Number: %#x%s", ntohl (lsa_header->seqnum),
           VTY_NEWLINE);
  vty_out (vty, "CheckSum: %#hx Length: %hu%s", ntohs (lsa_header->checksum),
           ntohs (lsa_header->length), VTY_NEWLINE);

  if (OSPF6_LSA_TYPESW_ISKNOWN (lsa_header->type))
    (*ospf6_lsa_vty_body [OSPF6_LSA_TYPESW (lsa_header->type)])
       (vty, lsa_header);
  else
    ospf6_lsa_vty_body_unknown (vty, lsa_header);

  vty_out (vty, "%s", VTY_NEWLINE);
}

/* OSPFv3 LSA creation/deletion function */

/* calculate LS sequence number for my new LSA.
   return value is network byte order */
static signed long
ospf6_seqnum_new (u_int16_t type, u_int32_t ls_id, u_int32_t advrtr)
{
  struct ospf6_lsa *lsa;
  signed long seqnum;

  /* get current database copy */
  lsa = ospf6_lsdb_lookup (type, ls_id, advrtr, ospf6);

  /* if current database copy not found, return InitialSequenceNumber */
  if (!lsa)
    seqnum = INITIAL_SEQUENCE_NUMBER;
  else
    seqnum = (signed long) ntohl (lsa->lsa_hdr->lsh_seqnum) + 1;

  return (htonl (seqnum));
}

static void
ospf6_lsa_header_set (u_int16_t type, u_int32_t ls_id, u_int32_t advrtr,
                      struct ospf6_lsa_header *lsa_header, int bodysize)
{
  /* fill LSA header */
  lsa_header->age = 0;
  lsa_header->type = type;
  lsa_header->ls_id = ls_id;
  lsa_header->advrtr = advrtr;
  lsa_header->seqnum =
    ospf6_seqnum_new (lsa_header->type, lsa_header->ls_id,
                      lsa_header->advrtr);
  lsa_header->length = htons (sizeof (struct ospf6_lsa_header) + bodysize);

  /* LSA checksum */
  ospf6_lsa_checksum (lsa_header);
}

struct ospf6_lsa *
ospf6_lsa_create (struct ospf6_lsa_header *source)
{
  struct ospf6_lsa *lsa = NULL;
  struct ospf6_lsa_header *lsa_header = NULL;
  u_int16_t lsa_size = 0;
  char buf[128];

  /* whole length of this LSA */
  lsa_size = ntohs (source->length);

  /* allocate memory for this LSA */
  lsa_header = (struct ospf6_lsa_header *)
    XMALLOC (MTYPE_OSPF6_LSA, lsa_size);
  memset (lsa_header, 0, lsa_size);

  /* copy LSA from source */
  memcpy (lsa_header, source, lsa_size);

  /* LSA information structure */
  /* allocate memory */
  lsa = (struct ospf6_lsa *)
          XMALLOC (MTYPE_OSPF6_LSA, sizeof (struct ospf6_lsa));
  memset (lsa, 0, sizeof (struct ospf6_lsa));

  /* dump string */
  snprintf (lsa->str, sizeof (lsa->str), "%s Advertise:%s LSID:%lu",
            ospf6_lsa_type_string (lsa_header->type),
            inet_ntop (AF_INET, &lsa_header->advrtr, buf, sizeof (buf)),
            (unsigned long) ntohl (lsa_header->ls_id));

  lsa->lsa_hdr = (struct ospf6_lsa_hdr *) lsa_header;
  lsa->summary_nbr = list_init ();
  lsa->request_nbr = list_init ();
  lsa->retrans_nbr = list_init ();

  lsa->delayed_ack_if = list_init ();
  lsa->dbdesc_neighbor = list_init ();

  lsa->summary = 0; /* this is not LSA summary */

  /* calculate birth, expire and refresh of this lsa */
  ospf6_lsa_age_set (lsa);

#if 0
  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("Lsa: created %s(%#x)", lsa->str, lsa);
#endif

  return lsa;
}

struct ospf6_lsa *
ospf6_lsa_summary_create (struct ospf6_lsa_header *source)
{
  struct ospf6_lsa *lsa = NULL;
  struct ospf6_lsa_header *lsa_header = NULL;
  u_int16_t lsa_size = 0;
  char buf[128];

  /* LSA summary contains LSA Header only */
  lsa_size = sizeof (struct ospf6_lsa_header);

  /* allocate memory for this LSA */
  lsa_header = (struct ospf6_lsa_header *)
    XMALLOC (MTYPE_OSPF6_LSA, lsa_size);
  memset (lsa_header, 0, lsa_size);

  /* copy LSA from source */
  memcpy (lsa_header, source, lsa_size);

  /* LSA information structure */
  /* allocate memory */
  lsa = (struct ospf6_lsa *)
          XMALLOC (MTYPE_OSPF6_LSA, sizeof (struct ospf6_lsa));
  memset (lsa, 0, sizeof (struct ospf6_lsa));

  /* dump string */
  snprintf (lsa->str, sizeof (lsa->str), "%s Advertise:%s LSID:%lu",
            ospf6_lsa_type_string (lsa_header->type),
            inet_ntop (AF_INET, &lsa_header->advrtr, buf, sizeof (buf)),
            (unsigned long) ntohl (lsa_header->ls_id));

  lsa->lsa_hdr = (struct ospf6_lsa_hdr *) lsa_header;

  lsa->summary_nbr = list_init ();
  lsa->request_nbr = list_init ();
  lsa->retrans_nbr = list_init ();

  lsa->delayed_ack_if = list_init ();
  lsa->dbdesc_neighbor = list_init ();

  lsa->summary = 1; /* this is LSA summary */

  /* calculate birth, expire and refresh of this lsa */
  ospf6_lsa_age_set (lsa);

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("Lsa: created summary %s(%#x)", lsa->str, lsa);

  return lsa;
}

static void
ospf6_lsa_delete (struct ospf6_lsa *lsa)
{
  /* just to make sure */
  if (lsa->lock != 0)
    {
      zlog_err ("Lsa: illegal deletion intension: %s", lsa->str);
      return;
    }

  /* threads */
  if (lsa->expire)
    thread_cancel (lsa->expire);
  if (lsa->refresh)
    thread_cancel (lsa->refresh);

  /* lists */
  assert (list_isempty (lsa->dbdesc_neighbor));
  list_free (lsa->dbdesc_neighbor);
  assert (list_isempty (lsa->summary_nbr));
  list_free (lsa->summary_nbr);
  assert (list_isempty (lsa->request_nbr));
  list_free (lsa->request_nbr);
  assert (list_isempty (lsa->retrans_nbr));
  list_free (lsa->retrans_nbr);
  assert (list_isempty (lsa->delayed_ack_if));
  list_free (lsa->delayed_ack_if);

#if 0
  if (IS_OSPF6_DUMP_LSA)
    {
      if (lsa->summary)
        zlog_info ("Lsa: delete summary %s(%#x)", lsa->str, lsa);
      else
        zlog_info ("lsa: delete %s(%#x)", lsa->str, lsa);
    }
#endif

  /* do free */
  XFREE (MTYPE_OSPF6_LSA, lsa->lsa_hdr);
  lsa->lsa_hdr = (struct ospf6_lsa_hdr *) NULL;
  XFREE (MTYPE_OSPF6_LSA, lsa);
}

/* increment reference counter of  struct ospf6_lsa */
void
ospf6_lsa_lock (struct ospf6_lsa *lsa)
{
  lsa->lock++;
  return;
}

/* decrement reference counter of  struct ospf6_lsa */
void
ospf6_lsa_unlock (struct ospf6_lsa *lsa)
{
  /* decrement reference counter */
  lsa->lock--;

  /* if no reference, do delete */
  if (lsa->lock == 0)
    {
      ospf6_lsa_delete (lsa);
    }
}

/* RFC2740 3.4.3.1 Router-LSA */
void
ospf6_lsa_update_router (struct ospf6_area *o6a)
{
  char buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_router_lsa *router_lsa;
  struct ospf6_router_lsd *router_lsd;
  listnode i;
  struct ospf6_interface *o6i;
  struct ospf6_neighbor *o6n = NULL;

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("Lsa: Update Router-LSA for %s", o6a->str);

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_ROUTER),
                                htonl (0), o6a->ospf6->router_id,
                                o6a->ospf6);

  size = sizeof (struct ospf6_router_lsa);
  memset (buffer, 0, sizeof (buffer));
  router_lsa = (struct ospf6_router_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));

  OSPF6_OPT_CLEAR_ALL (router_lsa->options);
  OSPF6_OPT_SET (router_lsa->options, OSPF6_OPT_V6);
  OSPF6_OPT_SET (router_lsa->options, OSPF6_OPT_E);
  OSPF6_OPT_CLEAR (router_lsa->options, OSPF6_OPT_MC);
  OSPF6_OPT_CLEAR (router_lsa->options, OSPF6_OPT_N);
  OSPF6_OPT_SET (router_lsa->options, OSPF6_OPT_R);
  OSPF6_OPT_CLEAR (router_lsa->options, OSPF6_OPT_DC);

  OSPF6_ROUTER_LSA_CLEAR_ALL_BITS (router_lsa);
  OSPF6_ROUTER_LSA_CLEAR (router_lsa, OSPF6_ROUTER_LSA_BIT_B);

  if (ospf6_is_asbr (o6a->ospf6))
    OSPF6_ROUTER_LSA_SET (router_lsa, OSPF6_ROUTER_LSA_BIT_E);
  else
    OSPF6_ROUTER_LSA_CLEAR (router_lsa, OSPF6_ROUTER_LSA_BIT_E);

  OSPF6_ROUTER_LSA_CLEAR (router_lsa, OSPF6_ROUTER_LSA_BIT_V);
  OSPF6_ROUTER_LSA_CLEAR (router_lsa, OSPF6_ROUTER_LSA_BIT_W);

  /* describe links for each interfaces */
  router_lsd = (struct ospf6_router_lsd *) (router_lsa + 1);
  for (i = listhead (o6a->if_list); i; nextnode (i))
    {
      o6i = (struct ospf6_interface *) getdata (i);
      assert (o6i);

      /* Interfaces in state Down or Loopback are not described */
      if (o6i->state == IFS_DOWN || o6i->state == IFS_LOOPBACK)
        continue;

      /* Nor are interfaces without any full adjacencies described */
      if (ospf6_interface_count_full_neighbor (o6i) == 0)
        continue;

      /* Point-to-Point interfaces */
      if (if_is_pointopoint (o6i->interface))
        {
          if (listcount (o6i->neighbor_list) == 0)
            continue;

          if (listcount (o6i->neighbor_list) != 1)
            zlog_warn ("LSA: Multiple neighbors on PoinToPoint: %s",
                       o6i->interface->name);

          o6n = (struct ospf6_neighbor *) getdata (listhead (o6i->neighbor_list));
          assert (o6n);

          router_lsd->type = OSPF6_ROUTER_LSD_TYPE_POINTTOPOINT;
          router_lsd->metric = htons (o6i->cost);
          router_lsd->interface_id = htonl (o6i->if_id);
          router_lsd->neighbor_interface_id = htonl (o6n->ifid);
          router_lsd->neighbor_router_id = o6n->rtr_id;

          size += sizeof (struct router_lsd);
          router_lsd ++;

          continue;
        }

      /* Broadcast and NBMA interfaces */
      if (if_is_broadcast (o6i->interface))
        {
          /* If this router is not DR,
             and If this router not fully adjacent with DR,
             this interface is not transit yet: ignore. */
          if (o6i->state != IFS_DR)
            {
              o6n = ospf6_neighbor_lookup (o6i->dr, o6i); /* find DR */
              assert (o6n);
              if (o6n->state != NBS_FULL)
                continue;
            }
          else
            {
              if (ospf6_interface_count_full_neighbor (o6i) == 0)
                continue;
            }

          router_lsd->type = OSPF6_ROUTER_LSD_TYPE_TRANSIT_NETWORK;
          router_lsd->metric = htons (o6i->cost);
          router_lsd->interface_id = htonl (o6i->if_id);
          if (o6i->state != IFS_DR)
            {
              router_lsd->neighbor_interface_id = htonl (o6n->ifid);
              router_lsd->neighbor_router_id = o6n->rtr_id;
            }
          else
            {
              router_lsd->neighbor_interface_id = htonl (o6i->if_id);
              router_lsd->neighbor_router_id = o6i->area->ospf6->router_id;
            }

          size += sizeof (struct router_lsd);
          router_lsd ++;

          continue;
        }

      /* Virtual links */
        /* xxx */
      /* Point-to-Multipoint interfaces */
        /* xxx */
    }

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_ROUTER), htonl (0),
                        o6a->ospf6->router_id,
                        (struct ospf6_lsa_header *)buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6a;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

/* RFC2740 3.4.3.2 Network-LSA */
void
ospf6_lsa_update_network (struct ospf6_interface *o6i)
{
  char buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_network_lsa *nlsa;
  listnode i;
  struct ospf6_neighbor *o6n;
  u_int32_t *router_id;

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_NETWORK),
                                htonl (o6i->if_id),
                                o6i->area->ospf6->router_id,
                                o6i->area->ospf6);

  /* Don't originate Network-LSA if not DR */
  if (o6i->state != IFS_DR)
    {
      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Network: Interface %s is not DR",
                   o6i->interface->name);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  /* If none of neighbor is adjacent to us */
  if (ospf6_interface_count_full_neighbor (o6i) == 0)
    {
      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Network: Interface %s is Stub",
                   o6i->interface->name);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("LSA Update Network: Interface %s", o6i->interface->name);

  /* prepare buffer */
  size = sizeof (struct ospf6_network_lsa);
  memset (buffer, 0, sizeof (buffer));
  nlsa = (struct ospf6_network_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));
  router_id = (u_int32_t *)(nlsa + 1);

  /* set fields of myself */
  *router_id++ = o6i->area->ospf6->router_id;
  size += sizeof (u_int32_t);
  nlsa->options[0] |= o6i->area->options[0];
  nlsa->options[1] |= o6i->area->options[1];
  nlsa->options[2] |= o6i->area->options[2];

  /* Walk through neighbors */
  for (i = listhead (o6i->neighbor_list); i; nextnode (i))
    {
      o6n = (struct ospf6_neighbor *) getdata (i);

      if (o6n->state != NBS_FULL)
        continue;

      /* set this neighbor's Router-ID to LSA */
      *router_id++ = o6n->rtr_id;
      size += sizeof (u_int32_t);

      /* options field is logical OR */
      nlsa->options[0] |= o6n->options[0];
      nlsa->options[1] |= o6n->options[1];
      nlsa->options[2] |= o6n->options[2];
    }

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_NETWORK), htonl (o6i->if_id),
                        o6i->area->ospf6->router_id,
                        (struct ospf6_lsa_header *) buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6i->area;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

/* RFC2740 3.4.3.5 AS-External-LSA */
void
ospf6_lsa_update_as_external (u_int32_t lsid, struct ospf6 *o6)
{
  char buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_as_external_lsa *elsa;
  struct route_node *rn, *target;
  struct ospf6_redistribute_info *info = NULL;
  char *p;

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_AS_EXTERNAL),
                                ntohl (lsid), o6->router_id, o6);

  /* find corresponding route node */
  target = NULL;
  for (rn = route_top (o6->redistribute_map); rn; rn = route_next (rn))
    {
      info = (struct ospf6_redistribute_info *) rn->info;
      if (info && info->ls_id == lsid)
        target = rn;
    }

  if (!target)
    {
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("LSA Update AS-External: LSId:%d", lsid);

  /* prepare buffer */
  size = sizeof (struct ospf6_as_external_lsa);
  memset (buffer, 0, sizeof (buffer));
  elsa = (struct ospf6_as_external_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));
  p = (char *) (elsa + 1);

  info = (struct ospf6_redistribute_info *) target->info;
  if (info->metric_type == 2)
    ASE_LSA_SET (elsa, ASE_LSA_BIT_E);   /* type2 */
  else
    ASE_LSA_CLEAR (elsa, ASE_LSA_BIT_E); /* type1 */

  if (! IN6_IS_ADDR_UNSPECIFIED (&info->forward))
    ASE_LSA_SET (elsa, ASE_LSA_BIT_F);   /* forwarding address */
  else
    ASE_LSA_CLEAR (elsa, ASE_LSA_BIT_F); /* forwarding address */

  ASE_LSA_CLEAR (elsa, ASE_LSA_BIT_T);   /* external route tag */

  /* don't know how to use */
  elsa->ase_pre_metric = 0;

  /* set metric. note: related to E bit */
  elsa->ase_metric = htons (info->metric);

  /* prefixlen */
  elsa->ospf6_prefix.prefix_length = target->p.prefixlen;

  /* PrefixOptions */
  elsa->ospf6_prefix.prefix_options = info->prefix_options;

  /* don't use refer LS-type */
  elsa->ospf6_prefix.prefix_refer_lstype = htons (0);

  /* set Prefix */
  memcpy (p, &target->p.u.prefix6, OSPF6_PREFIX_SPACE (target->p.prefixlen));
  ospf6_prefix_apply_mask (&elsa->ospf6_prefix);
  size += OSPF6_PREFIX_SPACE (target->p.prefixlen);
  p += OSPF6_PREFIX_SPACE (target->p.prefixlen);

  /* Forwarding address */
  if (ASE_LSA_ISSET (elsa, ASE_LSA_BIT_F))
    {
      memcpy (p, &info->forward, sizeof (struct in6_addr));
      size += sizeof (struct in6_addr);
      p += sizeof (struct in6_addr);
    }

  /* External Route Tag */
  if (ASE_LSA_ISSET (elsa, ASE_LSA_BIT_T))
    {
      /* xxx */
    }

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_AS_EXTERNAL), htonl (lsid),
                        o6->router_id,
                        (struct ospf6_lsa_header *) buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

/* RFC2740 3.4.3.7 Intra-Area-Prefix-LSA */
void
ospf6_lsa_update_intra_prefix_transit (struct ospf6_interface *o6i)
{
  char buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_intra_area_prefix_lsa *intra_prefix_lsa;
  listnode i;
  struct ospf6_neighbor *o6n;
  struct ospf6_prefix *o6_pstart, *o6_pcurrent, *o6_p1, *o6_p2;
  struct ospf6_lsa *link_lsa;
  struct ospf6_link_lsa *link_body;
  u_int16_t prefix_number, link_prefix_number;
  struct in6_addr in6;
  char buf[128];
  int duplicate;

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_INTRA_PREFIX),
                                htonl (o6i->if_id),
                                o6i->area->ospf6->router_id,
                                o6i->area->ospf6);

  /* Don't originate Network-LSA if not DR */
  if (o6i->state != IFS_DR)
    {
      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                   "Interface %s is not DR", o6i->interface->name);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  /* If none of neighbor is adjacent to us */
  if (ospf6_interface_count_full_neighbor (o6i) == 0)
    {
      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                   "Interface %s is Stub", o6i->interface->name);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("LSA Update Intra-Area-Prefix(Transit): Interface %s",
               o6i->interface->name);

  /* prepare buffer */
  size = sizeof (struct ospf6_intra_area_prefix_lsa);
  memset (buffer, 0, sizeof (buffer));
  intra_prefix_lsa = (struct ospf6_intra_area_prefix_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));
  o6_pstart = (struct ospf6_prefix *)(intra_prefix_lsa + 1);
  o6_pcurrent = o6_pstart;
  prefix_number = 0;

  /* Set Referenced LSA field */
  intra_prefix_lsa->refer_lstype = htons (OSPF6_LSA_TYPE_NETWORK);
  intra_prefix_lsa->refer_lsid = htonl (o6i->if_id);
  intra_prefix_lsa->refer_advrtr = o6i->area->ospf6->router_id;

  /* foreach Link-LSA associated with this Link */
  for (i = listhead (o6i->lsdb); i; nextnode (i))
    {
      link_lsa = (struct ospf6_lsa *) getdata (i);

      if (link_lsa->lsa_hdr->lsh_type != htons (OSPF6_LSA_TYPE_LINK))
        continue;

      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                   "Checking %s", link_lsa->str);

      if (link_lsa->lsa_hdr->lsh_advrtr != o6i->area->ospf6->router_id)
        {
          o6n = ospf6_neighbor_lookup (link_lsa->lsa_hdr->lsh_advrtr, o6i);
          if (! o6n || o6n->state != NBS_FULL)
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "neighbor not found or not FULL");
              continue;
            }
        }

      /* For each Prefix in this Link-LSA */
      link_body = (struct ospf6_link_lsa *) (link_lsa->lsa_hdr + 1);
      link_prefix_number = ntohl (link_body->llsa_prefix_num);

      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                   "Prefix #%d", link_prefix_number);

      for (o6_p1 = (struct ospf6_prefix *) (link_body + 1);
           link_prefix_number; link_prefix_number --)
        {
          ospf6_prefix_in6_addr (o6_p1, &in6);
          inet_ntop (AF_INET6, &in6, buf, sizeof (buf));

          if (IS_OSPF6_DUMP_LSA)
            zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                       "Prefix %s", buf);

          /* filter linklocal prefix */
          if (IN6_IS_ADDR_LINKLOCAL (&in6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "%s is Linklocal", buf);
              o6_p1 = OSPF6_NEXT_PREFIX (o6_p1);
              continue;
            }

          /* filter unspecified(default) prefix */
          if (IN6_IS_ADDR_UNSPECIFIED (&in6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "%s is Unspecified", buf);
              o6_p1 = OSPF6_NEXT_PREFIX (o6_p1);
              continue;
            }

          /* filter loopback prefix */
          if (IN6_IS_ADDR_LOOPBACK (&in6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "%s is Loopback", buf);
              o6_p1 = OSPF6_NEXT_PREFIX (o6_p1);
              continue;
            }

          /* filter IPv4 compatible prefix */
          if (IN6_IS_ADDR_V4COMPAT (&in6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "%s is v4-Compatible", buf);
              o6_p1 = OSPF6_NEXT_PREFIX (o6_p1);
              continue;
            }

          /* filter IPv4 Mapped prefix */
          if (IN6_IS_ADDR_V4MAPPED (&in6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "%s is v4-Mapped", buf);
              o6_p1 = OSPF6_NEXT_PREFIX (o6_p1);
              continue;
            }

          /* Check duplicate prefix */
          duplicate = 0;
          for (o6_p2 = o6_pstart; o6_p2 < o6_pcurrent;
               o6_p2 = OSPF6_NEXT_PREFIX (o6_p2))
            {
              if (! ospf6_prefix_issame (o6_p1, o6_p2))
                continue;

              duplicate = 1;
              o6_p2->prefix_options |= o6_p1->prefix_options;
            }
          if (duplicate)
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                           "Duplicate %s", buf);
              o6_p1 = OSPF6_NEXT_PREFIX (o6_p1);
              continue;
            }

          /* copy prefix to new LSA */
          if (IS_OSPF6_DUMP_LSA)
            zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                       "including %s", buf);
          o6_pcurrent->prefix_length = o6_p1->prefix_length;
          o6_pcurrent->prefix_options = o6_p1->prefix_options;
          memcpy (o6_pcurrent + 1, o6_p1 + 1,
                  OSPF6_PREFIX_SPACE (o6_p1->prefix_length));

          size += OSPF6_PREFIX_SIZE (o6_pcurrent);
          o6_pcurrent = OSPF6_NEXT_PREFIX (o6_pcurrent);
          prefix_number ++;
        }
    }

  /* if no prefix to advertise, return */
  if (prefix_number == 0)
    {
      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Transit): "
                   "No Prefix to advertise");
      return;
    }

  intra_prefix_lsa->prefix_number = htons (prefix_number);

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_INTRA_PREFIX),
                        htonl (o6i->if_id),
                        o6i->area->ospf6->router_id,
                        (struct ospf6_lsa_header *) buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6i->area;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

/* RFC2740 3.4.3.7 Intra-Area-Prefix-LSA */
void
ospf6_lsa_update_intra_prefix_stub (struct ospf6_area *o6a)
{
  char buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_intra_area_prefix_lsa *intra_prefix_lsa;
  listnode i,j;
  struct ospf6_interface *o6i = NULL;
  struct ospf6_prefix *prefix_dst;
  struct connected *c;
  struct prefix_ipv6 prefix_ipv6;
  char buf[128];
  u_int16_t prefix_number;

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_INTRA_PREFIX),
                                htonl (0),     /* xxx */
                                o6a->ospf6->router_id, o6a->ospf6);

  /* prepare buffer */
  size = sizeof (struct ospf6_intra_area_prefix_lsa);
  memset (buffer, 0, sizeof (buffer));
  intra_prefix_lsa = (struct ospf6_intra_area_prefix_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));
  prefix_dst = (struct ospf6_prefix *)(intra_prefix_lsa + 1);
  prefix_number = 0;

  /* Examin for each interface */
  for (i = listhead (o6a->if_list); i; nextnode (i))
    {
      o6i = (struct ospf6_interface *) getdata (i);

      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                   "Checking Interface %s", o6i->interface->name);

      if (o6i->state == IFS_DOWN)
        {
          if (IS_OSPF6_DUMP_LSA)
            zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                       "Interface %s down", o6i->interface->name);
          continue;
        }

      if (o6i->state != IFS_LOOPBACK && o6i->state != IFS_PTOP &&
          ospf6_interface_count_full_neighbor (o6i) != 0)
        {
          /* This interface's prefix will be included in DR's */
          if (IS_OSPF6_DUMP_LSA)
            zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                       "Interface %s is not stub", o6i->interface->name);
          continue;
        }

      /* copy foreach site-local or global prefix */
      for (j = listhead (o6i->interface->connected); j; nextnode (j))
        {
          c = (struct connected *) getdata (j);

          /* filter prefix not IPv6 */
          if (c->address->family != AF_INET6)
            continue;

          inet_ntop (AF_INET6, &c->address->u.prefix6, buf, sizeof (buf));

          if (IS_OSPF6_DUMP_LSA)
            zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                       "Checking %s", buf);

          /* filter linklocal prefix */
          if (IN6_IS_ADDR_LINKLOCAL (&c->address->u.prefix6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                           "%s is Linklocal", buf);
              continue;
            }

          /* filter unspecified(default) prefix */
          if (IN6_IS_ADDR_UNSPECIFIED (&c->address->u.prefix6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                           "%s is Unspecified", buf);
              continue;
            }

          /* filter loopback prefix */
          if (IN6_IS_ADDR_LOOPBACK (&c->address->u.prefix6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                           "%s is Loopback", buf);
              continue;
            }

          /* filter IPv4 compatible prefix */
          if (IN6_IS_ADDR_V4COMPAT (&c->address->u.prefix6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                           "%s is v4-Compatible", buf);
              continue;
            }

          /* filter IPv4 Mapped prefix */
          if (IN6_IS_ADDR_V4MAPPED (&c->address->u.prefix6))
            {
              if (IS_OSPF6_DUMP_LSA)
                zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                           "%s is v4-Mapped", buf);
              continue;
            }

          /* set ospf6 prefix according to its state */
          /* xxx, virtual links */
          prefix_copy ((struct prefix *) &prefix_ipv6, c->address);
          if (o6i->state == IFS_LOOPBACK || o6i->state == IFS_PTOP
              /* xxx, PoinToMultiPoint I/F type */ )
            {
              prefix_dst->prefix_length = 128;
              prefix_dst->prefix_options = OSPF6_PREFIX_OPTION_LA;
              prefix_dst->prefix_metric = htons (0);
              memcpy (prefix_dst + 1, &prefix_ipv6.prefix,
                      OSPF6_PREFIX_SPACE (prefix_dst->prefix_length));
            }
          else
            {
              /* apply mask */
              apply_mask_ipv6 (&prefix_ipv6);
              inet_ntop (AF_INET6, &c->address->u.prefix6, buf, sizeof (buf));

              prefix_dst->prefix_length = prefix_ipv6.prefixlen;
              prefix_dst->prefix_options = 0;  /* xxx, no options yet */
              prefix_dst->prefix_metric = htons (o6i->cost);
              memcpy (prefix_dst + 1, &prefix_ipv6.prefix,
                      OSPF6_PREFIX_SPACE (prefix_dst->prefix_length));
            }

          if (IS_OSPF6_DUMP_LSA)
            zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                       "include %s", buf);

          /* forward current buffer pointer */
          prefix_number ++;
          size += OSPF6_PREFIX_SIZE (prefix_dst);
          prefix_dst = OSPF6_NEXT_PREFIX (prefix_dst);
        }
    }

  /* If no prefix to advertise */
  if (prefix_number == 0)
    {
      if (IS_OSPF6_DUMP_LSA)
        zlog_info ("LSA Update Intra-Area-Prefix(Stub): "
                   "No prefix to advertise for area %s", o6a->str);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("LSA Update Intra-Area-Prefix(Stub): Area %s", o6a->str);

  /* Set Referenced LSA field */
  intra_prefix_lsa->refer_lstype = htons (OSPF6_LSA_TYPE_ROUTER);
  intra_prefix_lsa->refer_lsid = htonl (0);
  intra_prefix_lsa->refer_advrtr = o6a->ospf6->router_id;

  intra_prefix_lsa->prefix_number = htons (prefix_number);

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_INTRA_PREFIX),
                        htonl (0), /* xxx */
                        o6i->area->ospf6->router_id,
                        (struct ospf6_lsa_header *) buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6a;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

void
ospf6_lsa_update_link (struct ospf6_interface *o6i)
{
  char *cp, buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_link_lsa *llsa;
  struct ospf6_prefix *p;
  list prefix_connected;
  listnode n;
  struct connected *c;

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_LINK),
                                htonl (o6i->if_id),
                                ospf6->router_id, ospf6);

  /* can't make Link-LSA if linklocal address not set */
  if (! o6i->lladdr)
    {
      zlog_err ("LSA Update Link: No Linklocal Address: %s",
                o6i->interface->name);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("LSA Update Link: Interface %s", o6i->interface->name);

  /* check connected prefix */
  prefix_connected = list_init ();
  for (n = listhead (o6i->interface->connected); n; nextnode (n))
    {
      c = (struct connected *) getdata (n);

      /* filter prefix not IPv6 */
      if (c->address->family != AF_INET6)
        continue;

      /* filter linklocal prefix */
      if (IN6_IS_ADDR_LINKLOCAL (&c->address->u.prefix6))
        continue;

      /* filter unspecified(default) prefix */
      if (IN6_IS_ADDR_UNSPECIFIED (&c->address->u.prefix6))
        continue;

      /* filter loopback prefix */
      if (IN6_IS_ADDR_LOOPBACK (&c->address->u.prefix6))
        continue;

      /* filter IPv4 compatible prefix */
      if (IN6_IS_ADDR_V4COMPAT (&c->address->u.prefix6))
        continue;

      /* filter IPv4 Mapped prefix */
      if (IN6_IS_ADDR_V4MAPPED (&c->address->u.prefix6))
        continue;

      /* hold prefix in list. duplicate is filtered in ospf6_prefix_add() */
      p = ospf6_prefix_make (0, 0, (struct prefix_ipv6 *) c->address);
      ospf6_prefix_add (prefix_connected, p);
    }

  /* Note: if no prefix configured, still we have to create Link-LSA
     for next-hop resolution */

  /* fill Link LSA and calculate size */
  size = sizeof (struct ospf6_link_lsa);
  memset (buffer, 0, sizeof (buffer));
  llsa = (struct ospf6_link_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));
  llsa->llsa_rtr_pri = o6i->priority;
  llsa->llsa_options[0] = o6i->area->options[0];
  llsa->llsa_options[1] = o6i->area->options[1];
  llsa->llsa_options[2] = o6i->area->options[2];

  /* linklocal address */
  memcpy (&llsa->llsa_linklocal, o6i->lladdr, sizeof (struct in6_addr));
#ifdef KAME /* clear ifindex */
  if (llsa->llsa_linklocal.s6_addr[3] & 0x0f)
    llsa->llsa_linklocal.s6_addr[3] &= ~((char)0x0f);
#endif /* KAME */

  llsa->llsa_prefix_num = htonl (listcount (prefix_connected));
  cp = (char *)(llsa + 1);
  for (n = listhead (prefix_connected); n; nextnode (n))
    {
      p = (struct ospf6_prefix *) getdata (n);
      size += OSPF6_PREFIX_SIZE (p);
      memcpy (cp, p, OSPF6_PREFIX_SIZE (p));
      cp += OSPF6_PREFIX_SIZE (p);
    }

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_LINK), htonl (o6i->if_id),
                        o6i->area->ospf6->router_id,
                        (struct ospf6_lsa_header *) buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6i;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

#if 0

void
ospf6_lsa_update_inter_prefix (struct ospf6_area *o6a)
{
  char *cp, buffer [MAXLSASIZE];
  u_int16_t size;
  struct ospf6_lsa *lsa, *prev_lsa;

  struct ospf6_lsa_inter_prefix *inter_prefix;
  struct ospf6_prefix *o6p;

  /* find previous LSA */
  prev_lsa = ospf6_lsdb_lookup (htons (OSPF6_LSA_TYPE_INTER_PREFIX),
                                htonl (0), /* xxx */
                                ospf6->router_id, ospf6);

  if (o6a->area_range.prefixlen == 0)
    {
      zlog_err ("LSA Update Inter-Area-Prefix: No Area range Configured for %s",
                o6a->str);
      if (prev_lsa)
        ospf6_lsa_premature_aging (prev_lsa);
      return;
    }

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("LSA Update Inter-Area-Prefix: Area %s", o6a->str);

  size = sizeof (struct ospf6_lsa_inter_prefix_lsa);
  memset (buffer, 0, sizeof (buffer));
  inter_prefix = (struct ospf6_lsa_inter_prefix_lsa *)
    (buffer + sizeof (struct ospf6_lsa_header));

  /* linklocal address */
  memcpy (&llsa->llsa_linklocal, o6i->lladdr, sizeof (struct in6_addr));
#ifdef KAME /* clear ifindex */
  if (llsa->llsa_linklocal.s6_addr[3] & 0x0f)
    llsa->llsa_linklocal.s6_addr[3] &= ~((char)0x0f);
#endif /* KAME */

  llsa->llsa_prefix_num = htonl (listcount (prefix_connected));
  cp = (char *)(llsa + 1);
  for (n = listhead (prefix_connected); n; nextnode (n))
    {
      p = (struct ospf6_prefix *) getdata (n);
      size += OSPF6_PREFIX_SIZE (p);
      memcpy (cp, p, OSPF6_PREFIX_SIZE (p));
      cp += OSPF6_PREFIX_SIZE (p);
    }

  /* Fill LSA Header */
  ospf6_lsa_header_set (htons (OSPF6_LSA_TYPE_LINK), htonl (o6i->if_id),
                        o6i->area->ospf6->router_id,
                        (struct ospf6_lsa_header *) buffer, size);

  /* create LSA */
  lsa = ospf6_lsa_create ((struct ospf6_lsa_header *) buffer);
  ospf6_lsa_lock (lsa);
  lsa->refresh = thread_add_timer (master, ospf6_lsa_refresh, lsa,
                                   OSPF6_LS_REFRESH_TIME);
  lsa->scope = (void *) o6i;

  ospf6_dbex_flood (lsa, NULL);

  ospf6_lsdb_install (lsa);
  ospf6_lsa_unlock (lsa);
}

#endif

void
ospf6_lsa_reoriginate (struct ospf6_lsa *lsa)
{
  struct ospf6_lsa_header *lsa_header;

  struct ospf6 *o6;
  struct ospf6_area *o6a;
  struct ospf6_interface *o6i;

  lsa_header = (struct ospf6_lsa_header *) lsa->lsa_hdr;

  if (IS_OSPF6_DUMP_LSA)
    zlog_info ("Re-originate %s", lsa->str);

  switch (ntohs (lsa_header->type))
    {
      case OSPF6_LSA_TYPE_ROUTER:
        o6a = (struct ospf6_area *) lsa->scope;
        assert (o6a);
        ospf6_lsa_update_router ((struct ospf6_area *) o6a);
        break;

      case OSPF6_LSA_TYPE_NETWORK:
        o6a = (struct ospf6_area *) lsa->scope;
        assert (o6a);
        o6i = ospf6_interface_lookup_by_index
          (ntohl (lsa_header->ls_id), o6a->ospf6);
        if (o6i)
          ospf6_lsa_update_network (o6i);
        else
          ospf6_lsa_premature_aging (lsa);
        break;

      case OSPF6_LSA_TYPE_INTRA_PREFIX:
        /* xxx, assume LS-ID has addressing semantics */
        o6a = (struct ospf6_area *) lsa->scope;
        o6i = NULL;
        assert (o6a);
        if (ntohl (lsa_header->ls_id) != 0)
          o6i = ospf6_interface_lookup_by_index
            (ntohl (lsa_header->ls_id), ospf6);

        if (o6i)
          ospf6_lsa_update_intra_prefix_transit (o6i);
        else if (ntohl (lsa_header->ls_id) == 0)
          ospf6_lsa_update_intra_prefix_stub (o6a);
        else
          ospf6_lsa_premature_aging (lsa);
        break;

      case OSPF6_LSA_TYPE_LINK:
        o6i = (struct ospf6_interface *) lsa->scope;
        assert (o6i);
        ospf6_lsa_update_link (o6i);
        break;

      case OSPF6_LSA_TYPE_AS_EXTERNAL:
        o6 = (struct ospf6 *) lsa->scope;
        assert (o6);
        ospf6_lsa_update_as_external (ntohl (lsa_header->ls_id), o6);
        break;

      default:
        break;
    }
}

/*****************/



struct router_lsd *
get_router_lsd (u_int32_t rtrid, struct ospf6_lsa *lsa)
{
  unsigned short lsh_len;
  struct router_lsa *rlsa;
  struct router_lsd *rlsd;

  if (ntohs (lsa->lsa_hdr->lsh_type) != OSPF6_LSA_TYPE_ROUTER)
    return NULL;

  lsh_len = ntohs (lsa->lsa_hdr->lsh_len);
  rlsa = (struct router_lsa *)(lsa->lsa_hdr + 1);
  rlsd = (struct router_lsd *)(rlsa + 1);

  for ( ; (char *)rlsd < (char *)(lsa->lsa_hdr) + lsh_len; rlsd++)
    if (rtrid == rlsd->rlsd_neighbor_router_id)
      return rlsd;

  return NULL;
}

/* xxx, messy, strange function name */
unsigned long
get_ifindex_to_router (u_int32_t rtrid, struct ospf6_lsa *lsa)
{
  struct router_lsd *rlsd;
  char rtrid_str[64];

  assert (lsa);
  switch (ntohs (lsa->lsa_hdr->lsh_type))
    {
      case OSPF6_LSA_TYPE_ROUTER:
        rlsd = get_router_lsd (rtrid, lsa);
        if (!rlsd)
          {
            inet_ntop (AF_INET, &rtrid, rtrid_str, sizeof (rtrid_str));
            zlog_warn ("*** can't find ifindex to %s", rtrid_str); 
            return 0;
          }
        else
          return (ntohl (rlsd->rlsd_interface_id));
      case OSPF6_LSA_TYPE_NETWORK:
        return (ntohl (lsa->lsa_hdr->lsh_id));
      default:
        return 0;
    }
  return 0;
}


/* ospf6_lsa expired */
int
ospf6_lsa_expire (struct thread *thread)
{
  struct ospf6_lsa *lsa;

  lsa = (struct ospf6_lsa *) THREAD_ARG (thread);
  assert (lsa && lsa->lsa_hdr);

  /* assertion */
  assert (ospf6_lsa_is_maxage (lsa));
  assert (!lsa->refresh);

  lsa->expire = (struct thread *) NULL;

  /* log */
  if (IS_OSPF6_DUMP_LSA)
    {
      zlog_info ("LSA: Expire:");
      ospf6_dump_lsa (lsa);
    }

  /* reflood lsa */
  ospf6_dbex_flood (lsa, NULL);

  /* do not free LSA, and do nothing about lslists.
     wait event (ospf6_lsdb_check_maxage) */
  return 0;
}

int
ospf6_lsa_refresh (struct thread *thread)
{
  struct ospf6_lsa *lsa;

  assert (thread);
  lsa = (struct ospf6_lsa *) THREAD_ARG  (thread);
  assert (lsa && lsa->lsa_hdr);

  /* this will be used later as flag to decide really originate */
  lsa->refresh = (struct thread *)NULL;

  /* log */
  if (IS_OSPF6_DUMP_LSA)
    {
      zlog_info ("LSA Refresh: %s", lsa->str);
    }

  ospf6_lsa_reoriginate (lsa);

  return 0;
}



/* enhanced Fletcher checksum algorithm, RFC1008 7.2 */
#define MODX                4102
#define LSA_CHECKSUM_OFFSET   15

/* XXX, this function assumes that the machine is little endian */
unsigned short
ospf6_lsa_checksum (struct ospf6_lsa_header *lsa_header)
{
  u_char *sp, *ep, *p, *q;
  int c0 = 0, c1 = 0;
  int x, y;
  u_int16_t length;

  lsa_header->checksum = 0;
  length = ntohs (lsa_header->length) - 2;
  sp = (char *) &lsa_header->type;

  for (ep = sp + length; sp < ep; sp = q)
    {
      q = sp + MODX;
      if (q > ep)
        q = ep;
      for (p = sp; p < q; p++)
        {
          c0 += *p;
          c1 += c0;
        }
      c0 %= 255;
      c1 %= 255;
    }

  /* r = (c1 << 8) + c0; */
  x = ((length - LSA_CHECKSUM_OFFSET) * c0 - c1) % 255;
  if (x <= 0)
    x += 255;
  y = 510 - c0 - x;
  if (y > 255)
    y -= 255;

  lsa_header->checksum = x + (y << 8);

  return (lsa_header->checksum);
}

/* RFC905 ANNEX B */
unsigned short
ospf6_lsa_checksum_set (struct ospf6_lsa_header *lsa_header)
{
#if 0
  int    i, L, c0, c1;
  u_char X, Y;

  L = ntohs (lsh->lsh_len) - 2;
#endif
  return 0;
}

int
ospf6_lsa_checksum_ok (struct ospf6_lsa_header *lsa_header)
{
  return 0;
}

u_int16_t
ospf6_lsa_get_scope_type (u_int16_t type)
{
  return (ntohs (type) & OSPF6_LSA_SCOPE_MASK);
}

