/******************************************************************************\
 * $Id: rcfileParser.cpp,v 1.3 2001/05/20 18:08:45 blais Exp $
 * $Date: 2001/05/20 18:08:45 $
 *
 * Copyright (C) 1999-2001  Martin Blais <blais@iro.umontreal.ca>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *****************************************************************************/

/*==============================================================================
 * EXTERNAL DECLARATIONS
 *============================================================================*/

#include <defs.h>

#ifdef XX_USE_RCFILE

#include <rcfileParser.h>
#include <exceptions.h>

#include <stdexcept>
#include <iostream>
#include <fstream>
#include <sstream>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <iostream>

/*==============================================================================
 * LOCAL DECLARATIONS
 *============================================================================*/

namespace {

/*----- classes -----*/

/*==============================================================================
 * CLASS XxRcfileError
 *============================================================================*/

class XxRcfileError : public std::exception {

public:

   /*----- member functions -----*/

   // Constructor from current errno.
   XxRcfileError( const int lineno, const char* msg );

   // Constructor with a string.
   XxRcfileError( const char* msg );

   // Destructor.
   virtual ~XxRcfileError() XX_THROW_NOTHING;

   // See base class.
   virtual const char* what() const XX_THROW_NOTHING;


private:

   /*----- data members -----*/

   std::string _msg;

};

//------------------------------------------------------------------------------
//
XxRcfileError::XxRcfileError( const int lineno, const char* msg )
{
   std::ostringstream oss;
   oss << "Error parsing rcfile, line " << lineno << ": " 
       << msg << std::endl << std::ends;
   _msg = oss.str();
}

//------------------------------------------------------------------------------
//
XxRcfileError::XxRcfileError( const char* msg )
{
   _msg = msg;
}

//------------------------------------------------------------------------------
//
XxRcfileError::~XxRcfileError() XX_THROW_NOTHING
{}

//------------------------------------------------------------------------------
//
const char* XxRcfileError::what() const XX_THROW_NOTHING
{
   return _msg.c_str();
}

}


//==============================================================================
// NAMESPACE XxRcfileParserNS
//==============================================================================

namespace XxRcfileParserNS {

// Suppress warnings under IRIX.
#ifdef COMPILER_MIPSPRO
#pragma set woff 1209
#pragma set woff 3322
#pragma set woff 1110
#pragma set woff 1174
#pragma set woff 1167
#pragma set woff 1506
#endif

#define YY_NEVER_INTERACTIVE 1
#define YY_SKIP_YYWRAP       1

#define YY_INPUT( buf, result, max_size )       \
if ( is->eof() ) {                              \
   result = 0;                                  \
}                                               \
else {                                          \
   is->read( buf, max_size );                   \
   result = is->gcount();                       \
}

#define YY_NO_UNPUT

#define yylex    __yylex
#define yyback   __yyback
#define yyinput  __yyinput
#define yylook   __yylook
#define yyoutput __yyoutput
#define yyracc   __yyracc
#define yyreject __yyreject
#define yyunput __yyunput


/*----- function prototypes -----*/

void yyerror( const char* );
int yywrap();

/*----- variables -----*/

std::istream* is = 0;
int yylineno = 0;


#define YYPARSE_PARAM parser
#define YY_DECL       int yylex( YYSTYPE* yylval )

#include <rcfileParser.lex.c>
#include <rcfileParser.y.c>

// Reset warnings under IRIX.
#ifdef COMPILER_MIPSPRO
#pragma reset woff 1209
#pragma reset woff 3322
#pragma reset woff 1110
#pragma reset woff 1174
#pragma reset woff 1167
#pragma reset woff 1506
#endif

//------------------------------------------------------------------------------
//
void yyerror(
   const char* msg
)
{
   throw new XxRcfileError( yylineno, msg );
}

//------------------------------------------------------------------------------
//
int yywrap()
{
   return 1;
}

}


XX_NAMESPACE_BEGIN
using namespace XxRcfileParserNS;


/*==============================================================================
 * PUBLIC FUNCTIONS
 *============================================================================*/

/*==============================================================================
 * CLASS XxRcfileParser
 *============================================================================*/

//------------------------------------------------------------------------------
//
XxRcfileParser::XxRcfileParser() :
   _resources()
{
   // Decide on filename.
   const char* XXDIFFRC = getenv( "XXDIFFRC" );
   std::string filename;
   if ( XXDIFFRC ) {
      filename = XXDIFFRC;
   }
   else {
      const char* HOME = getenv( "HOME" );
      if ( HOME ) {
         filename = HOME;
      }
      else {
         filename = "~";
      }
      filename.append( "/.xxdiffrc" );
   }

   // Check for empty filename.
   if ( filename.empty() ) {
      throw new XxInternalError( XX_INTERROR_PARAMS );
   }

   // Stat the file.
   struct stat buf;
   if ( stat( filename.c_str(), &buf ) != 0 ) {
      return; // file does not exist.
   }
   if ( !S_ISREG( buf.st_mode )  ) {
      return; // file is not a regular file.
   }

   // Open the file.
   std::ifstream file( filename.c_str() );

   // Parse the file.
   parse( file );

   // Close the file.
   file.close();
}

//------------------------------------------------------------------------------
//
XxRcfileParser::~XxRcfileParser()
{
   // Warn about unused resources.
   for ( ResourceMap::const_iterator iter = _resources.begin();
         iter != _resources.end();
         ++iter ) {
      if ( (*iter).second._used == false ) {
         std::cerr << "Warning: ignored resource: "
                   << (*iter).first
                   << " : " 
                   << (*iter).second._value << std::endl;
      }
   }
}

//------------------------------------------------------------------------------
//
void XxRcfileParser::parse(
   std::istream& input
)
{
#if YYDEBUG != 0
   yydebug = 1;
#endif

   is = & input;

   try {
      yyrestart( 0 );
      yyparse( this );
   }
   catch ( ... ) {
      // FIXME don't crash on error!
      throw;
   }
}

//------------------------------------------------------------------------------
//
void XxRcfileParser::push()
{
   yy_push_state( yy_top_state() );
}

//------------------------------------------------------------------------------
//
void XxRcfileParser::pop()
{
   yy_pop_state();
}

//------------------------------------------------------------------------------
//
void XxRcfileParser::addResource(
   const std::string& name,
   const std::string& content
)
{
   ResValue& rv = _resources[ name ];
   rv._value = content;
   rv._used = false;
}

//------------------------------------------------------------------------------
//
void XxRcfileParser::clearValueText()
{
   _valueText = std::string(); // Note: clear() is not implemented in libstdc++.
}

//------------------------------------------------------------------------------
//
void XxRcfileParser::addValueText(
   const std::string& text
)
{
   if ( ! _valueText.empty() ) {
      _valueText.insert( 0, " " );
   }
   _valueText.insert( 0, text );
}

//------------------------------------------------------------------------------
//
std::string& XxRcfileParser::getValueText()
{
   return _valueText;
}

//------------------------------------------------------------------------------
//
bool XxRcfileParser::query( 
   XxResources::Resource resource,
   const char*           name,
   std::string&          value
)
{
   ResourceMap::iterator iter = _resources.find( name );
   if ( iter == _resources.end() ) {
      return false;
   }
   value = (*iter).second._value;
   (*iter).second._used = true;
   return true;
}


XX_NAMESPACE_END

#endif
