/*
 *
 * Copyright (c) 1997 Charles D. Cranor.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following condition
 * is met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* 
 * note: this is cobbled together from an X book i read once.
 * i'm not an xpert, so this isn't very hi-tech.
 */

/*
 * create_color
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include "libcdcx.h"

/*
 * create_color
 */

u_long create_color(xdpy, colormap, colorname, defaultcol)

struct xdpy *xdpy;
Colormap colormap;
char *colorname;
u_long defaultcol;

{
  XColor hw, exact;
  
  if (XAllocNamedColor(xdpy->dpy, colormap, colorname, &hw, &exact))
    return(hw.pixel);
  else
    return(defaultcol);
}

/*
 * create_font
 */

XFontStruct *create_font(xdpy, font, fallback)

struct xdpy *xdpy;
char *font, *fallback;

{
  XFontStruct *f;

  if ((f = XLoadQueryFont(xdpy->dpy, font)) == NULL &&
      (f = XLoadQueryFont(xdpy->dpy, fallback)) == NULL &&
      (f = XLoadQueryFont(xdpy->dpy, "fixed")) == NULL)
    errx(1, "XLoadQueryFont failed!");

  return(f);
}

/*
 * create_gc
 */

GC create_gc(xdpy, drawable, fg, bg)

struct xdpy *xdpy;
Drawable drawable;
u_long fg, bg;

{
  XGCValues xgcvalues;
  GC gc;

  xgcvalues.foreground = fg;
  xgcvalues.background = bg;
  gc = XCreateGC(xdpy->dpy, drawable, (GCForeground | GCBackground),
		 &xgcvalues);

  return(gc);
}

/*
 * barbox
 */

int draw_barbox(xdpy, win, gc, font, x0, x1, y, n, total, 
		names, vals, colors, backcolor)

struct xdpy *xdpy;
Window win;
GC gc;
XFontStruct *font;
int x0, x1, y, n, total;
char **names;
u_long *vals, *colors, backcolor;

{
  char buf[BUFSIZ];
  int barheight = get_fontheight(font)/2;
  int barwidth = x1 - x0;
  int x, lcv, npix, tmp, digs;
  int leglen, maxleg = 0;
  int cols, colwidth, width;

  /*
   * draw the bar
   */

  x = x0;
  tmp = total;
  digs = 0;
  while (tmp) {
    digs++;
    tmp = tmp / 10;
  }

  for (lcv = 0 ; lcv < n ; lcv++) {
    
    npix = barwidth * ((double) vals[lcv] / (double) total);
    leglen =  XTextWidth(font, names[lcv], strlen(names[lcv]));
    leglen += XTextWidth(font, " = ", 3);
    leglen += XTextWidth(font,"8", 1) * digs;
    if (leglen > maxleg)
      maxleg = leglen;
    if (npix == 0)
      continue;
    XSetForeground(xdpy->dpy, gc, colors[lcv]);
    XFillRectangle(xdpy->dpy, win, gc, x, y, npix, barheight);
    x += npix;

  }

  /*
   * draw legend
   */

  y += barheight + 5;
  x = x0;
  XSetFont(xdpy->dpy, gc, font->fid); 
  cols = barwidth / maxleg;
  if (cols < 1) 
    cols = 1;
  colwidth = barwidth / cols;
  
  for (lcv = 0 ; lcv < n ; lcv++) {
    snprintf(buf, sizeof(buf), "%s = %d", names[lcv], vals[lcv]);
    XSetForeground(xdpy->dpy, gc, colors[lcv]);
    XDrawImageString(xdpy->dpy, win, gc, x, get_fontheight(font) + y,
              buf, strlen(buf));
    XSetForeground(xdpy->dpy, gc, backcolor);
    width = XTextWidth(font, buf, strlen(buf));
    XFillRectangle(xdpy->dpy, win, gc, x + width, y, 
      colwidth - width, get_fontheight(font));
    x += colwidth;
    if (x + colwidth > x1) {
      y = y + get_fontheight(font);
      x = x0;
    }
  }

  if (x != x0)
      y = y + get_fontheight(font) + 5;
  else
      y += 5;

  return(y);
}

/*
 * barbox
 */

int draw_barlvl(xdpy, win, gc, font, x0, x1, y, n, total, names, vals, 
		colors, backcolor)

struct xdpy *xdpy;
Window win;
GC gc;
XFontStruct *font;
int x0, x1, y, n, total;
char **names;
u_long *vals, *colors, backcolor;

{
  char buf[BUFSIZ];
  int barheight = get_fontheight(font)/2;
  int barwidth = x1 - x0;
  int x, lcv, npix, tmp, digs, width;
  int leglen, maxleg = 0;
  int cols, colwidth;

  /*
   * draw the bar
   */

  x = x0;
  tmp = total;
  digs = 0;
  while (tmp) {
    digs++;
    tmp = tmp / 10;
  }

  for (lcv = 0 ; lcv < n ; lcv++) {
    
    npix = barwidth * ((double) vals[lcv] / (double) total);
    leglen =  XTextWidth(font, names[lcv], strlen(names[lcv]));
    leglen += XTextWidth(font, " = ", 3);
    leglen += XTextWidth(font,"8", 1) * (digs+2);
    if (leglen > maxleg)
      maxleg = leglen;
    XSetForeground(xdpy->dpy, gc, colors[lcv]);
    if (npix)
      XFillRectangle(xdpy->dpy, win, gc, x, y, npix, barheight);
    XSetForeground(xdpy->dpy, gc, backcolor);
    if (barwidth - npix)
      XFillRectangle(xdpy->dpy, win, gc, x+npix, y, barwidth - npix, barheight);
    y += barheight + 1;

  }

  /*
   * draw legend
   */

  x = x0;
  XSetFont(xdpy->dpy, gc, font->fid); 
  cols = barwidth / maxleg;
  if (cols < 1) 
    cols = 1;
  colwidth = barwidth / cols;
  
  for (lcv = 0 ; lcv < n ; lcv++) {
    snprintf(buf, sizeof(buf), "%s = %d", names[lcv], vals[lcv]);
    XSetForeground(xdpy->dpy, gc, colors[lcv]);
    XDrawImageString(xdpy->dpy, win, gc, x, get_fontheight(font) + y,
              buf, strlen(buf));
    width = XTextWidth(font, buf, strlen(buf));
    XSetForeground(xdpy->dpy, gc, backcolor);
    XFillRectangle(xdpy->dpy, win, gc, x + width, y, colwidth - width,
		   get_fontheight(font));
    x += colwidth;
    if (x + colwidth > x1) {
      y = y + get_fontheight(font);
      x = x0;
    }
  }

  if (x != x0)
      y = y + get_fontheight(font) + 5;
  else
      y += 5;

  return(y);
}

/*
 * graph
 */

struct graph *create_graph(nlines, title, names, before, after, 
			   colors, backcolor)

int nlines;
char *title;
char **names;
int **before, **after;
u_long *colors;
u_long backcolor;

{
  struct graph *gp;
  gp = (struct graph *) malloc(sizeof(*gp));
  if (gp == NULL)
    errx(1, "malloc failed");
  gp->nlines = nlines;
  gp->title = title;
  gp->names = names;
  gp->before = before;
  gp->after = after;
  gp->colors = colors;
  gp->backcolor = backcolor;
  gp->nslot = gp->ptr = 0;
  gp->lindat = NULL;
  gp->maxval = 1;
  return(gp);
}

void draw_graph(xdpy, win, gc, font, x, y, w, h, gp, was_timeout)

struct xdpy *xdpy;
Window win;
GC gc;
XFontStruct *font;
int x, y, w, h;
struct graph *gp;
int was_timeout;

{
  int lcv, pixleft, legwid, ly, lx, pix, pt1, pt2, y1, y2, orig_pt1;
  int curmax;
  char buf[80];

  if (gp->lindat == NULL) {
    gp->nslot = w / 5;
    gp->lindat = (int **) malloc(gp->nlines * sizeof(int *));
    if (gp->lindat == NULL)
      errx(1, "malloc failed");
    for (lcv = 0 ; lcv < gp->nlines ; lcv++) {
      gp->lindat[lcv] = (int *) malloc(w * sizeof(int));
      if (gp->lindat[lcv] == NULL)
	errx(1, "malloc failed");
      bzero(gp->lindat[lcv], w * sizeof(int));
    }

    pixleft = w - 10;
    gp->leglines = 0;
    for (lcv = 0 ; lcv < gp->nlines ; lcv++) {
      legwid = XTextWidth(font, gp->names[lcv], strlen(gp->names[lcv]));
      legwid += XTextWidth(font, "  ", 2);
    ReTry:
      if (legwid > pixleft && pixleft == w - 10) {
	gp->leglines++;
	continue;
      }
      if (legwid > pixleft) {
	gp->leglines++;
	pixleft = w - 10;
	goto ReTry;
      }
      pixleft -= legwid;
    }
    if (pixleft != w - 10)
      gp->leglines++;
  }

  if (was_timeout) {
    gp->ptr = (gp->ptr + 1) % gp->nslot;
    for (lcv = 0 ; lcv < gp->nlines ; lcv++) {
      gp->lindat[lcv][gp->ptr] = *gp->after[lcv] - *gp->before[lcv];
      if (gp->lindat[lcv][gp->ptr] > gp->maxval) {
	gp->maxval = gp->lindat[lcv][gp->ptr];
      }
    }
  }

  /* title */
  pix = get_fontheight(font) + 4;
  XSetForeground(xdpy->dpy, gc, xdpy->black);
  XDrawRectangle(xdpy->dpy, win, gc, x, y, w, pix);
  XDrawImageString(xdpy->dpy, win, gc, x+4, y+pix-2, gp->title, 
		   strlen(gp->title));
  snprintf(buf, sizeof(buf), "  (max=%d)", gp->maxval);
  legwid = XTextWidth(font, gp->title, strlen(gp->title));
  XDrawImageString(xdpy->dpy, win, gc, x+4+legwid,
                   y+pix-2, buf, strlen(buf));
  legwid += XTextWidth(font, buf, strlen(buf));
  XSetForeground(xdpy->dpy, gc, gp->backcolor);
  XFillRectangle(xdpy->dpy, win, gc, x+legwid+4, y+1, w-legwid-4, pix-2);
  y += pix;
  h -= pix;
  /* title */

  XSetForeground(xdpy->dpy, gc, gp->backcolor);
  XFillRectangle(xdpy->dpy, win, gc, x+1, y+1, w-1, 
		 h - 3 - (gp->leglines * (get_fontheight(font) + 2)));
  if (was_timeout == 0) { /* redraw? */
    XSetForeground(xdpy->dpy, gc, xdpy->black);
    XDrawRectangle(xdpy->dpy, win, gc, x, y, w, h);
    ly = y + h - 2 - (gp->leglines * (get_fontheight(font) + 2));
    lx = x + 5;
    pixleft = w - 10;
    for (lcv = 0 ; lcv < gp->nlines ; lcv++) {
      legwid = XTextWidth(font, gp->names[lcv], strlen(gp->names[lcv]));
      legwid += XTextWidth(font, "  ", 2);
    NextLine:
      if (legwid > pixleft && pixleft == w - 10) {
	XSetForeground(xdpy->dpy, gc, gp->colors[lcv]);
	XDrawImageString(xdpy->dpy, win, gc, lx, get_fontheight(font) + ly + 2,
              gp->names[lcv], strlen(gp->names[lcv]));
	ly += get_fontheight(font)+2;
      }
      if (legwid > pixleft) {
	ly += get_fontheight(font)+2;
	lx = x + 5;
	pixleft = w - 10;
	goto NextLine;
      }
      XSetForeground(xdpy->dpy, gc, gp->colors[lcv]);
      XDrawImageString(xdpy->dpy, win, gc, lx, get_fontheight(font) + ly + 2,
		       gp->names[lcv], strlen(gp->names[lcv]));
      lx += legwid;
      pixleft -= legwid;
    }
  }

  curmax = 0;
  for (lcv = 0 ; lcv < gp->nlines ; lcv++) {
    XSetForeground(xdpy->dpy, gc, gp->colors[lcv]);
    pix = h - 3 - (gp->leglines * get_fontheight(font) + 2);
    lx = x+1;
    orig_pt1 = pt1 = (gp->ptr + 1)  % gp->nslot;
    y1 = y + pix;
    y1 = y1 - (pix * (gp->lindat[lcv][pt1] / (double) gp->maxval));
    if (gp->lindat[lcv][pt1] > curmax)
      curmax = gp->lindat[lcv][pt1];
    pt2 = (pt1 + 1) % gp->nslot;
    y2 = y + pix;
    while (pt2 != orig_pt1) {
      y2 = y + pix;
      y2 = y2  - (pix * (gp->lindat[lcv][pt2] / (double) gp->maxval));
      if (gp->lindat[lcv][pt2] > curmax)
	curmax = gp->lindat[lcv][pt2];
      XDrawLine(xdpy->dpy, win, gc, lx, y1, lx + 5, y2);
      lx += 5;
      pt1 = pt2;
      y1 = y2;
      pt2 = (pt2 + 1) % gp->nslot;
    }
  }

  if (was_timeout && curmax < (gp->maxval/4.0)) {
    gp->maxval = gp->maxval * 0.95;   /* reduce */
  }
}

/*
 * open_display: frontend for XOpenDisplay
 */

Display *open_display(dpy_name, xdpy)

char *dpy_name;
struct xdpy *xdpy;

{
  if ((xdpy->dpy = XOpenDisplay(dpy_name)) == NULL) 
    errx(1, "%s: open failed", XDisplayName(dpy_name));

  xdpy->screen = DefaultScreen(xdpy->dpy);
  xdpy->rootwin = RootWindow(xdpy->dpy, xdpy->screen);

  xdpy->depth = DefaultDepth(xdpy->dpy, xdpy->screen);
  xdpy->width = DisplayWidth(xdpy->dpy, xdpy->screen);
  xdpy->height = DisplayHeight(xdpy->dpy, xdpy->screen);
  xdpy->black = BlackPixel(xdpy->dpy, xdpy->screen);
  xdpy->white = WhitePixel(xdpy->dpy, xdpy->screen);

  return(xdpy->dpy);
}

#define BORDER_WIDTH 2

/*
 * open_window: open a window
 */

Window open_window(xdpy, parent, x, y, w, h, bordercolor, backcolor,
		   event_mask, visual)

struct xdpy *xdpy;
Window parent;
int x, y, w, h;
u_long bordercolor, backcolor, event_mask;
Visual *visual;

{
  Window newwin;
  XSetWindowAttributes attrs;
  u_long attrs_mask;

  /* window attributes */
  attrs.event_mask = event_mask;
  attrs.border_pixel = bordercolor;
  attrs.background_pixel = backcolor;
  attrs_mask = CWEventMask | CWBackPixel | CWBorderPixel;

  /* create */
  newwin = XCreateWindow(xdpy->dpy, parent, x, y, w, h, BORDER_WIDTH,
			 CopyFromParent, /* depth */
			 InputOutput, /* class */
			 visual, attrs_mask, &attrs);

  return(newwin);
}

/*
 * set_wmhints
 */

void set_classhints(xdpy, win, res_name, res_class)

struct xdpy *xdpy;
Window win;
char *res_name, *res_class;

{
  XClassHint ch;
  ch.res_class = res_class;
  ch.res_name = res_name;
  XSetClassHint(xdpy->dpy, win, &ch);
}

/*
 * set_sizehints
 */

void set_sizehints(xdpy, win, x, y, w, h)

struct xdpy *xdpy;
Window win;
int x, y, w, h;

{
  XSizeHints sh;

  /* start: obsolete */
  sh.x = x;
  sh.y = y;
  sh.height = h;
  sh.width = w;
  /* end obsolete */

  sh.min_height = h;
  sh.min_width = w;
  sh.base_width = w;
  sh.base_height = h;
  sh.flags = USPosition | USSize | PMinSize | PBaseSize;

  XSetWMNormalHints(xdpy->dpy, win, &sh);

}

/*
 * set_stdhints
 */

void set_stdhints(xdpy, win, app_name, win_name, x, y, w, h)

struct xdpy *xdpy;
Window win;
char *app_name, *win_name;
int x, y, w, h;

{
  set_sizehints(xdpy, win, x, y, w, h);
  set_windowname(xdpy, win, win_name);
  set_classhints(xdpy, win, app_name, "Examples");
  set_wmhints(xdpy, win, NormalState);
}

/*
 * set_windowname
 */

void set_windowname(xdpy, win, win_name)

struct xdpy *xdpy;
Window win;
char *win_name;

{
  XTextProperty textprop;

  if (XStringListToTextProperty(&win_name, 1, &textprop) == 0)
    errx(1, "XStringListToTextProperty failed");

  XSetWMName(xdpy->dpy, win, &textprop);
}

/*
 * set_wmhints
 */

void set_wmhints(xdpy, win, init_state)

struct xdpy *xdpy;
Window win;
int init_state;

{
  XWMHints wm_hints;

  wm_hints.flags = InputHint | StateHint;
  wm_hints.initial_state = init_state;
  wm_hints.input = True;
  XSetWMHints(xdpy->dpy, win, &wm_hints);
}

/*
 * setup_colormap
 */

int setup_colormap(xdpy, screen, win, visual, colormap)

struct xdpy *xdpy;
int screen;
Window win;
Visual *visual;
Colormap *colormap;

{
XFlush(xdpy->dpy);
  if (visual == DefaultVisual(xdpy->dpy, screen)) {
    *colormap = DefaultColormap(xdpy->dpy, screen);
    return(True);
  }

  *colormap = XCreateColormap(xdpy->dpy, win, visual, AllocNone);
  if (*colormap != None) {
    XSetWindowColormap(xdpy->dpy, win, *colormap);
    return(True);
  }

  *colormap = DefaultColormap(xdpy->dpy, screen);
  return(False);

}

/*
 * setup_visual
 */

int setup_visual(xdpy, screen, visual, depth)

struct xdpy *xdpy;
int screen;
Visual **visual;
int *depth;

{
  int nvis;
  XVisualInfo *visual_array, visual_info_template;

  if (DefaultVisual(xdpy->dpy, screen)->class == PseudoColor) {
    *visual = DefaultVisual(xdpy->dpy, screen);
    *depth = DefaultDepth(xdpy->dpy, screen);
    return(True);
  }

  visual_info_template.class = PseudoColor;
  visual_info_template.screen = screen;
  visual_array = XGetVisualInfo(xdpy->dpy, VisualClassMask | VisualScreenMask,
				&visual_info_template, &nvis);

  if (nvis < 1 || visual_array == NULL) {
    if (visual_array) XFree(visual_array);
    visual_info_template.class = TrueColor;
    visual_info_template.screen = screen;
    visual_array = XGetVisualInfo(xdpy->dpy, VisualClassMask | VisualScreenMask,
				  &visual_info_template, &nvis);
    if (nvis < 1 || visual_array == NULL) {
      if (visual_array) XFree(visual_array);
      *visual = CopyFromParent;
      return(False);
    }
    
  }

  *visual = visual_array[0].visual;
  *depth = visual_array[0].depth;
  XFree(visual_array);
  return(True);

}
