/*
 * Copyright (c) 1994, 1995 Joerg Wunsch
 *
 * All rights reserved.
 *
 * This program is free software.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE DEVELOPERS ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE DEVELOPERS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: xtestpicture.c,v 1.4 1995/08/25 15:21:27 j Exp $
 *
 * xtestpicture
 * Draw a simple picture containing some test patterns useful for
 * monitor adjustment etc., using a window covering the whole screen
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "patchlevel.h"

typedef unsigned short c_value;
#define MAX_CVALUE 65535	/* per X11 definition */

void
scale_saturation(c_value ri, c_value gi, c_value bi,
		 c_value *ro, c_value *go, c_value *bo,
		 double saturation)
{
  double rii, gii, bii, roo, goo, boo, t, b;

  /* normate to [0.0 .. 1.0] */
  rii = (double)ri / (double)MAX_CVALUE;
  gii = (double)gi / (double)MAX_CVALUE;
  bii = (double)bi / (double)MAX_CVALUE;

  /* scale down for `saturation' towards the white-black diagonale */
  t = (rii + gii + bii) / 3;
  roo = t + saturation * (rii - t);
  goo = t + saturation * (gii - t);
  boo = t + saturation * (bii - t);

  /* correct brightness value (optical appearance) */
  if(roo + goo + boo > 1e-10)	/* avoid scaling null value */
    {
      b = (0.30 * rii + 0.59 * gii + 0.11 * bii) /
      (0.30 * roo + 0.59 * goo + 0.11 * boo);
      roo *= b; goo *= b; boo *=b;
    }
  
  /* denormate to [0 .. MAX_CVALUE] */
  *ro = (c_value)(roo * (double)MAX_CVALUE);
  *go = (c_value)(goo * (double)MAX_CVALUE);
  *bo = (c_value)(boo * (double)MAX_CVALUE);
}


static Display *d;
static Window root;

int
main(int argc, char **argv)
{
  int s, i, j, status, tmp = 1, saturated_only = 0;
  char *cp;
  double scale;
  const char *dispname = 0;
  const char *cmdname;
  Screen *scp;
  GC palette[5 * 8];
  const struct
  {
    c_value r, g, b;
  } color_tab[] =
    {
      {0, 0, 0},		/* black */
      {0, 0, MAX_CVALUE},	/* blue */
      {MAX_CVALUE, 0, 0},	/* red */
      {MAX_CVALUE, 0, MAX_CVALUE}, /* magenta */
      {0, MAX_CVALUE, 0},	/* green */
      {0, MAX_CVALUE, MAX_CVALUE}, /* cyan */
      {MAX_CVALUE, MAX_CVALUE, 0}, /* yellow */
      {MAX_CVALUE, MAX_CVALUE, MAX_CVALUE} /* white */
    };
# define gcblack palette[0]
# define gcwhite palette[7]
  
  XSetWindowAttributes a;
  Window mainw;
  XColor c, c1;

  int h, w;

  if((cp = strrchr(argv[0], '/'))) cmdname = cp + 1;
  else cmdname = argv[0];
  
  for(i = 1; i < argc; i++)
    {
      if(argv[i][0] == '-' && argv[i][1] != 0)
	{
	  if(strcmp(argv[i], "-d") == 0 || strcmp(argv[i], "-display") == 0)
	    {
	      if(i == argc - 1)
		{
		  fprintf(stderr,
			  "%s: missing argument for -display\n", cmdname);
		  return 2;
		}
	      dispname = argv[i + 1];
	      i++;
	    }
	  else if(strcmp(argv[i], "-s") == 0 ||
		  strcmp(argv[i], "-saturated") == 0)
	    {
	      saturated_only = 1;
	    }
	  else if(strcmp(argv[i], "-v") == 0 ||
		  strcmp(argv[i], "-version") == 0)
	    {
	      fprintf(stderr, "%s: version %s\n", cmdname, VERSION);
	      exit(0);
	    }
	  else
	    {
	      fprintf(stderr,
		      "%s: unknown option %s\n", cmdname, argv[i]);
	      return 2;
	    }
	}
      else
	break;
    }
  argc -= i;
  argv += i;

  if(argc > 0)
    {
      fprintf(stderr, "usage: %s [-d[isplay] display]\n", cmdname);
      return 2;
    }
  
  if((d = XOpenDisplay(dispname)) == NULL)
    {
      fprintf(stderr, "%s: Cannot open display\n", cmdname);
      return EXIT_FAILURE;
    }

  s = DefaultScreen(d);
  scp = ScreenOfDisplay(d, s);
  root = RootWindow(d, s);

  a.event_mask = ExposureMask | ButtonPressMask;
  a.override_redirect = True;

  w = WidthOfScreen(scp);
  h = HeightOfScreen(scp);
  mainw = XCreateWindow(d, root,
			0, 0, w, h, 0,
			CopyFromParent, InputOutput, CopyFromParent,
			CWOverrideRedirect|CWEventMask, &a);

  XMapWindow(d, mainw);
  XRaiseWindow(d, mainw);

  for(i = 0, status = 1; i < 8; i++)
    {
      c.red = color_tab[i].r;
      c.green = color_tab[i].g;
      c.blue = color_tab[i].b;
      status &=
      XAllocColor(d, DefaultColormap(d, s), &c);
      palette[i] = XCreateGC(d, root, 0, 0);
      XSetForeground(d, palette[i], c.pixel);
      if(!saturated_only)
	{
	  for(j = 1, tmp = 1, scale = 0.75; tmp && j < 5; j++, scale -= 0.25)
	    {
	      scale_saturation(color_tab[i].r, color_tab[i].g, color_tab[i].b,
			       &c1.red, &c1.green, &c1.blue,
			       scale);
	      tmp &= XAllocColor(d, DefaultColormap(d, s), &c1);
	      palette[j * 8 + i] = XCreateGC(d, root, 0, 0);
	      XSetForeground(d, palette[j * 8 + i], c1.pixel);
	    }
	}
    }
  if(status == 0)
    {
      fprintf(stderr, "Cannot allocate basic color cells.\n");
      return 1;
    }
  if(tmp == 0)
    {
      fprintf(stderr, "Warning: cannot allocate enough color cells.\n"
	      "Falling back to -saturated.\n");
      saturated_only = 1;
    }

  while(1)
    {
      XEvent e;

      XNextEvent(d, &e);
      switch(e.type)
	{
	  int i, j;

	case Expose:
	  if(e.xexpose.count)
	    break;

	  /* make a black window */
	  XClearWindow(d, mainw);
	  XFillRectangle(d, mainw, gcblack, 0, 0, w, h);

	  /* first elements useful for convergency and linearity */

	  /* crossing lines thru center of screen */
	  XDrawLine(d, mainw, gcwhite, 0, 0, w-1, h-1);
	  XDrawLine(d, mainw, gcwhite, 0, h-1, w-1, 0);

	  /*
	   * two rectangles at the border,
	   * one rectangle and a circle in the middle
	   */
	  XDrawRectangle(d, mainw, gcwhite, 0, 0, w-1, h-1);
	  XDrawRectangle(d, mainw, gcwhite, 10, 10, w-21, h-21);
	  XDrawRectangle(d, mainw, gcwhite, w/3, h/3, w/3, h/3);
	  XDrawArc(d, mainw, gcwhite, w/2-h/3, h/6, 2*h/3, 2*h/3, 0, 360*64);

	  /*
	   * some vertical lines in the middle, two and one points thick,
	   * useful to evaluate for sharpness/monitor bandwidth
	   */
	  for(i = w/2, j = 0; j < 100; i--, j++)
	    {
	      if((j / 2) & 1)
		XDrawLine(d, mainw, gcwhite, i, h/2-80, i, h/2-20);
	      else
		XDrawLine(d, mainw, gcblack, i, h/2-80, i, h/2-20);
	    }
	  for(i = w/2, j = 0; j < 100; i++, j++)
	    {
	      if(j & 1)
		XDrawLine(d, mainw, gcwhite, i, h/2-80, i, h/2-20);
	      else
		XDrawLine(d, mainw, gcblack, i, h/2-80, i, h/2-20);
	    }
	  XDrawRectangle(d, mainw, gcwhite, w/2-100, h/2-81, 200, 61);

	  /* the palette */
	  for(i = 0; i < 8; i++)
	    if(saturated_only)
	      {
		XFillRectangle(d, mainw, palette[i],
			       w/2-100+i*25, h/2+20, 25, 60);
	      }
	    else
	      {
		for(j = 0; j < 5; j++)
		  XFillRectangle(d, mainw, palette[i + j * 8],
				 w/2-100+i*25, h/2+20+12*j, 25, 12);
	      }
	  XDrawRectangle(d, mainw, gcwhite, w/2-101, h/2+19, 201, 61);
	  
	  break;
	  
	default:
	  /* the first button press will exit the program */
	  return EXIT_SUCCESS;

	}
    }

  /* NOTREACHED */
  return EXIT_SUCCESS;
}
