/*
 * Copyright (c) 2001-2006 International Computer Science Institute
 * See LICENSE file for licensing, conditions, and warranties on use.
 *
 * This file is PROGRAMMATICALLY GENERATED.
 *
 * This instance was generated with:
 *      callback-gen.py -b 6 -l 15
 */

/**
 * @libdoc Callbacks
 *
 * @sect Callback Overview
 *
 * XORP is an asynchronous programming environment and as a result there
 * are many places where callbacks are useful.  Callbacks are typically
 * invoked to signify the completion or advancement of an asynchronous
 * operation.
 *
 * XORP provides a generic and flexible callback interface that utilizes
 * overloaded templatized functions for for generating callbacks
 * in conjunction with many small templatized classes. Whilst this makes
 * the syntax a little ugly, it provides a great deal of flexibility.
 *
 * XorpCallbacks are callback objects are created by the callback()
 * function which returns a reference pointer to a newly created callback
 * object.  The callback is invoked by calling dispatch(), eg.
 *
<pre>

#include <iostream>

#include "config.h"
#include "libxorp/callback.hh"

static void hello_world() {
    cout << "Hello World" << endl;
}

int main() {
    // Typedef callback() return type for readability.  SimpleCallback
    // declares a XorpCallback taking 0 arguments and of return type void.
    typedef XorpCallback0<void>::RefPtr SimpleCallback;

    // Create XorpCallback object using callback()
    SimpleCallback cb = callback(hello_world);

    // Invoke callback, results in call to hello_world.
    cb->dispatch();
    return 0;
}

</pre>
 *
 * The callback() method is overloaded and can also be used to create
 * callbacks to member functions, eg.
 *
<pre>

#include <iostream>

#include "config.h"
#include "libxorp/callback.hh"

class Foo {
public:
    void hello_world() {
	cout << "Foo::Hello World" << endl;
    }
};

int main() {
    typedef XorpCallback0<void>::RefPtr SimpleCallback;

    Foo f;

    // Create a callback to a member function
    SimpleCallback cb = callback(&f, &Foo::hello_world);

    // Invoke f.hello_world
    cb->dispatch();

    return 0;
}

</pre>
 *
 * In addition, to being able to invoke member functions, callbacks can
 * also store arguments to functions. eg.
 *
<pre>

#include <iostream>

#include "config.h"
#include "libxorp/callback.hh"

static int sum(int x, int y) {
    cout << "sum(x = " << x << ", y = " << y << ")" << endl;
    return x + y;
}

int main() {
    // Callback to function returning "int"
    typedef XorpCallback0<int>::RefPtr NoArgCallback;

    NoArgCallback cb1 = callback(sum, 1, 2);
    cout << "cb1->dispatch() returns " << cb1->dispatch() << endl; // "3"
    cout << endl;

    // Callback to function returning int and taking an integer argument
    typedef XorpCallback1<int,int>::RefPtr OneIntArgCallback;

    OneIntArgCallback cb2 = callback(sum, 5);
    cout << "cb2->dispatch(10) returns " << cb2->dispatch(10) << endl; // 15
    cout << endl;

    cout << "cb2->dispatch(20) returns " << cb2->dispatch(20) << endl; // 25
    cout << endl;

    // Callback to function returning int and taking  2 integer arguments
    typedef XorpCallback2<int,int,int>::RefPtr TwoIntArgCallback;

    TwoIntArgCallback cb3 = callback(sum);
    cout << "cb3->dispatch() returns " << cb3->dispatch(50, -50) << endl; // 0

    return 0;
}

</pre>
 *
 * Bound arguments, as with member functions, are implemented by the
 * overloading of the callback() method.  At dispatch time, the bound
 * arguments are last arguments past to the wrappered function.  If you
 * compile and run the program you will see:
 *
<pre>
sum(x = 10, y = 5)
cb2->dispatch(10) returns 15
</pre>
 *
 * and:
 *
<pre>
sum(x = 20, y = 5)
cb2->dispatch(20) returns 25
</pre>
 *
 * for the one bound argument cases.
 *
 * @sect Declaring Callback Types
 *
 * There are a host of XorpCallbackN types.  The N denotes the number
 * of arguments that will be passed at dispatch time by the callback
 * invoker.  The template parameters to XorpCallbackN types are the
 * return value followed by the types of arguments that will be passed
 * at dispatch time.  Thus type:
 *
 * <pre>
XorpCallback1<double, int>::RefPtr
 * </pre>
 *
 * corresponds to callback object returning a double when invoked and
 * requiring an integer argument to passed at dispatch time.
 *
 * When arguments are bound to a callback they are not specified
 * in the templatized argument list. So the above declaration is good
 * for a function taking an integer argument followed by upto the
 * maximum number of bound arguments.
 *
 * Note: In this header file, support is provided for upto %d bound
 * arguments and %d dispatch arguments.
 *
 * @sect Ref Pointer Helpers
 *
 * Callback objects may be set to NULL, since they use reference pointers
 * to store the objects.  Callbacks may be unset using the ref_ptr::release()
 * method:
 *
<pre>
    cb.release();
</pre>
 * and to tested using the ref_ptr::is_empty() method:
<pre>
if (! cb.is_empty()) {
    cb->dispatch();
}
</pre>
 *
 * In many instances, the RefPtr associated with a callback on an object
 * will be stored by the object itself.  For instance, a class may own a
 * timer object and the associated timer expiry callback which is
 * a member function of the containing class.  Because the containing class
 * owns the callback object corresponding the timer callback, there is
 * never an opportunity for the callback to be dispatched on a deleted object
 * or with invalid data.
 */


#ifndef INCLUDED_FROM_CALLBACK_HH
#error "This file should be included through libxorp/callback.hh"
#endif

#ifndef __XORP_CALLBACK_HH__
#define __XORP_CALLBACK_HH__

#include "minitraits.hh"
#include "ref_ptr.hh"
#include "safe_callback_obj.hh"

///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 0 late args
//

/**
 * @short Base class for callbacks with 0 dispatch time args.
 */
template<class R>
struct XorpCallback0 {
    typedef ref_ptr<XorpCallback0> RefPtr;

    virtual ~XorpCallback0() {}
    virtual R dispatch() = 0;
};

/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R>
struct XorpFunctionCallback0B0 : public XorpCallback0<R> {
    typedef R (*F)();
    XorpFunctionCallback0B0(F f)
	: XorpCallback0<R>(),
	  _f(f)
    {}
    R dispatch() {
	R r = (*_f)();
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <>
struct XorpFunctionCallback0B0<void> : public XorpCallback0<void> {
    typedef void (*F)();
    XorpFunctionCallback0B0(F f)
	: XorpCallback0<void>(),
	  _f(f)
    {}
    void dispatch() {
	(*_f)();
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 0 bound arguments.
 */
template <class R>
typename XorpCallback0<R>::RefPtr
callback(R (*f)()) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B0<R>(f));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O>
struct XorpMemberCallback0B0 : public XorpCallback0<R> {
    typedef R (O::*M)() ;
    XorpMemberCallback0B0(O* o, M m)
	 : XorpCallback0<R>(),
	  _o(o), _m(m) {}
    R dispatch() {
	R r = ((*_o).*_m)();
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O>
struct XorpMemberCallback0B0<void, O>
: public XorpCallback0<void> {
    typedef void (O::*M)() ;
    XorpMemberCallback0B0(O* o, M m)
	 : XorpCallback0<void>(),
	  _o(o), _m(m) {}
    void dispatch() {
	((*_o).*_m)();
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O>
struct XorpSafeMemberCallback0B0
    : public XorpMemberCallback0B0<R, O>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B0<R, O>::M M;
    XorpSafeMemberCallback0B0(O* o, M m)
	 : XorpMemberCallback0B0<R, O>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B0() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B0<R, O>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O>
struct XorpSafeMemberCallback0B0<void,O>
    : public XorpMemberCallback0B0<void, O>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B0<void, O>::M M;
    XorpSafeMemberCallback0B0(O* o, M m)
	 : XorpMemberCallback0B0<void, O>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B0() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B0<void, O>::dispatch();
	}
    }
};

template <class R, class O, bool B=true>
struct XorpMemberCallbackFactory0B0
{
    inline static XorpMemberCallback0B0<R, O>*
    make(O* o, R (O::*p)())
    {
	return new XorpSafeMemberCallback0B0<R, O>(o, p);
    }
};

template <class R, class O>
struct XorpMemberCallbackFactory0B0<R, O, false>
{
    inline static XorpMemberCallback0B0<R, O>*
    make(O* o, R (O::*p)())
    {
	return new XorpMemberCallback0B0<R, O>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)())
{
    return XorpMemberCallbackFactory0B0<R,  O, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)())
{
    return XorpMemberCallbackFactory0B0<R,  O, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O>
struct XorpConstMemberCallback0B0 : public XorpCallback0<R> {
    typedef R (O::*M)()  const;
    XorpConstMemberCallback0B0(O* o, M m)
	 : XorpCallback0<R>(),
	  _o(o), _m(m) {}
    R dispatch() {
	R r = ((*_o).*_m)();
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O>
struct XorpConstMemberCallback0B0<void, O>
: public XorpCallback0<void> {
    typedef void (O::*M)()  const;
    XorpConstMemberCallback0B0(O* o, M m)
	 : XorpCallback0<void>(),
	  _o(o), _m(m) {}
    void dispatch() {
	((*_o).*_m)();
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O>
struct XorpConstSafeMemberCallback0B0
    : public XorpConstMemberCallback0B0<R, O>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B0<R, O>::M M;
    XorpConstSafeMemberCallback0B0(O* o, M m)
	 : XorpConstMemberCallback0B0<R, O>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B0() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B0<R, O>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O>
struct XorpConstSafeMemberCallback0B0<void,O>
    : public XorpConstMemberCallback0B0<void, O>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B0<void, O>::M M;
    XorpConstSafeMemberCallback0B0(O* o, M m)
	 : XorpConstMemberCallback0B0<void, O>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B0() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B0<void, O>::dispatch();
	}
    }
};

template <class R, class O, bool B=true>
struct XorpConstMemberCallbackFactory0B0
{
    inline static XorpConstMemberCallback0B0<R, O>*
    make(O* o, R (O::*p)() const)
    {
	return new XorpConstSafeMemberCallback0B0<R, O>(o, p);
    }
};

template <class R, class O>
struct XorpConstMemberCallbackFactory0B0<R, O, false>
{
    inline static XorpConstMemberCallback0B0<R, O>*
    make(O* o, R (O::*p)() const)
    {
	return new XorpConstMemberCallback0B0<R, O>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)() const)
{
    return XorpConstMemberCallbackFactory0B0<R,  const O, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)() const)
{
    return XorpConstMemberCallbackFactory0B0<R,  const O, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class BA1>
struct XorpFunctionCallback0B1 : public XorpCallback0<R> {
    typedef R (*F)(BA1);
    XorpFunctionCallback0B1(F f, BA1 ba1)
	: XorpCallback0<R>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch() {
	R r = (*_f)(_ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class BA1>
struct XorpFunctionCallback0B1<void, BA1> : public XorpCallback0<void> {
    typedef void (*F)(BA1);
    XorpFunctionCallback0B1(F f, BA1 ba1)
	: XorpCallback0<void>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch() {
	(*_f)(_ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 1 bound arguments.
 */
template <class R, class BA1>
typename XorpCallback0<R>::RefPtr
callback(R (*f)(BA1), BA1 ba1) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B1<R, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class BA1>
struct XorpMemberCallback0B1 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1) ;
    XorpMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class BA1>
struct XorpMemberCallback0B1<void, O, BA1>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1) ;
    XorpMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch() {
	((*_o).*_m)(_ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class BA1>
struct XorpSafeMemberCallback0B1
    : public XorpMemberCallback0B1<R, O, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B1<R, O, BA1>::M M;
    XorpSafeMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback0B1<R, O, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B1() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B1<R, O, BA1>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class BA1>
struct XorpSafeMemberCallback0B1<void,O, BA1>
    : public XorpMemberCallback0B1<void, O, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B1<void, O, BA1>::M M;
    XorpSafeMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback0B1<void, O, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B1() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B1<void, O, BA1>::dispatch();
	}
    }
};

template <class R, class O, class BA1, bool B=true>
struct XorpMemberCallbackFactory0B1
{
    inline static XorpMemberCallback0B1<R, O, BA1>*
    make(O* o, R (O::*p)(BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback0B1<R, O, BA1>(o, p, ba1);
    }
};

template <class R, class O, class BA1>
struct XorpMemberCallbackFactory0B1<R, O, BA1, false>
{
    inline static XorpMemberCallback0B1<R, O, BA1>*
    make(O* o, R (O::*p)(BA1), BA1 ba1)
    {
	return new XorpMemberCallback0B1<R, O, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class BA1> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)(BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory0B1<R,  O, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class BA1> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)(BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory0B1<R,  O, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class BA1>
struct XorpConstMemberCallback0B1 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1)  const;
    XorpConstMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class BA1>
struct XorpConstMemberCallback0B1<void, O, BA1>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1)  const;
    XorpConstMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch() {
	((*_o).*_m)(_ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class BA1>
struct XorpConstSafeMemberCallback0B1
    : public XorpConstMemberCallback0B1<R, O, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B1<R, O, BA1>::M M;
    XorpConstSafeMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback0B1<R, O, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B1() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B1<R, O, BA1>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class BA1>
struct XorpConstSafeMemberCallback0B1<void,O, BA1>
    : public XorpConstMemberCallback0B1<void, O, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B1<void, O, BA1>::M M;
    XorpConstSafeMemberCallback0B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback0B1<void, O, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B1() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B1<void, O, BA1>::dispatch();
	}
    }
};

template <class R, class O, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory0B1
{
    inline static XorpConstMemberCallback0B1<R, O, BA1>*
    make(O* o, R (O::*p)(BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback0B1<R, O, BA1>(o, p, ba1);
    }
};

template <class R, class O, class BA1>
struct XorpConstMemberCallbackFactory0B1<R, O, BA1, false>
{
    inline static XorpConstMemberCallback0B1<R, O, BA1>*
    make(O* o, R (O::*p)(BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback0B1<R, O, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class BA1> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)(BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory0B1<R,  const O, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class BA1> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)(BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory0B1<R,  const O, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class BA1, class BA2>
struct XorpFunctionCallback0B2 : public XorpCallback0<R> {
    typedef R (*F)(BA1, BA2);
    XorpFunctionCallback0B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback0<R>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch() {
	R r = (*_f)(_ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class BA1, class BA2>
struct XorpFunctionCallback0B2<void, BA1, BA2> : public XorpCallback0<void> {
    typedef void (*F)(BA1, BA2);
    XorpFunctionCallback0B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback0<void>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch() {
	(*_f)(_ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 2 bound arguments.
 */
template <class R, class BA1, class BA2>
typename XorpCallback0<R>::RefPtr
callback(R (*f)(BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B2<R, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2>
struct XorpMemberCallback0B2 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2) ;
    XorpMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class BA1, class BA2>
struct XorpMemberCallback0B2<void, O, BA1, BA2>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2) ;
    XorpMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2>
struct XorpSafeMemberCallback0B2
    : public XorpMemberCallback0B2<R, O, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B2<R, O, BA1, BA2>::M M;
    XorpSafeMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback0B2<R, O, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B2() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B2<R, O, BA1, BA2>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class BA1, class BA2>
struct XorpSafeMemberCallback0B2<void,O, BA1, BA2>
    : public XorpMemberCallback0B2<void, O, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B2<void, O, BA1, BA2>::M M;
    XorpSafeMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback0B2<void, O, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B2() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B2<void, O, BA1, BA2>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory0B2
{
    inline static XorpMemberCallback0B2<R, O, BA1, BA2>*
    make(O* o, R (O::*p)(BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback0B2<R, O, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class BA1, class BA2>
struct XorpMemberCallbackFactory0B2<R, O, BA1, BA2, false>
{
    inline static XorpMemberCallback0B2<R, O, BA1, BA2>*
    make(O* o, R (O::*p)(BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback0B2<R, O, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class BA1, class BA2> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)(BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory0B2<R,  O, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class BA1, class BA2> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)(BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory0B2<R,  O, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2>
struct XorpConstMemberCallback0B2 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2)  const;
    XorpConstMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class BA1, class BA2>
struct XorpConstMemberCallback0B2<void, O, BA1, BA2>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2)  const;
    XorpConstMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2>
struct XorpConstSafeMemberCallback0B2
    : public XorpConstMemberCallback0B2<R, O, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B2<R, O, BA1, BA2>::M M;
    XorpConstSafeMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback0B2<R, O, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B2() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B2<R, O, BA1, BA2>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class BA1, class BA2>
struct XorpConstSafeMemberCallback0B2<void,O, BA1, BA2>
    : public XorpConstMemberCallback0B2<void, O, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B2<void, O, BA1, BA2>::M M;
    XorpConstSafeMemberCallback0B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback0B2<void, O, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B2() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B2<void, O, BA1, BA2>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory0B2
{
    inline static XorpConstMemberCallback0B2<R, O, BA1, BA2>*
    make(O* o, R (O::*p)(BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback0B2<R, O, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class BA1, class BA2>
struct XorpConstMemberCallbackFactory0B2<R, O, BA1, BA2, false>
{
    inline static XorpConstMemberCallback0B2<R, O, BA1, BA2>*
    make(O* o, R (O::*p)(BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback0B2<R, O, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class BA1, class BA2> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)(BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory0B2<R,  const O, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class BA1, class BA2> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)(BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory0B2<R,  const O, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class BA1, class BA2, class BA3>
struct XorpFunctionCallback0B3 : public XorpCallback0<R> {
    typedef R (*F)(BA1, BA2, BA3);
    XorpFunctionCallback0B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback0<R>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch() {
	R r = (*_f)(_ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class BA1, class BA2, class BA3>
struct XorpFunctionCallback0B3<void, BA1, BA2, BA3> : public XorpCallback0<void> {
    typedef void (*F)(BA1, BA2, BA3);
    XorpFunctionCallback0B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback0<void>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch() {
	(*_f)(_ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 3 bound arguments.
 */
template <class R, class BA1, class BA2, class BA3>
typename XorpCallback0<R>::RefPtr
callback(R (*f)(BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B3<R, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3>
struct XorpMemberCallback0B3 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3) ;
    XorpMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3>
struct XorpMemberCallback0B3<void, O, BA1, BA2, BA3>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3) ;
    XorpMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback0B3
    : public XorpMemberCallback0B3<R, O, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B3<R, O, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback0B3<R, O, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B3() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B3<R, O, BA1, BA2, BA3>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback0B3<void,O, BA1, BA2, BA3>
    : public XorpMemberCallback0B3<void, O, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B3<void, O, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback0B3<void, O, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B3() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B3<void, O, BA1, BA2, BA3>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory0B3
{
    inline static XorpMemberCallback0B3<R, O, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback0B3<R, O, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory0B3<R, O, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback0B3<R, O, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback0B3<R, O, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)(BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory0B3<R,  O, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)(BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory0B3<R,  O, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback0B3 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3)  const;
    XorpConstMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback0B3<void, O, BA1, BA2, BA3>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3)  const;
    XorpConstMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback0B3
    : public XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B3() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback0B3<void,O, BA1, BA2, BA3>
    : public XorpConstMemberCallback0B3<void, O, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B3<void, O, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback0B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback0B3<void, O, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B3() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B3<void, O, BA1, BA2, BA3>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory0B3
{
    inline static XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback0B3<R, O, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory0B3<R, O, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback0B3<R, O, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)(BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory0B3<R,  const O, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)(BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory0B3<R,  const O, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback0B4 : public XorpCallback0<R> {
    typedef R (*F)(BA1, BA2, BA3, BA4);
    XorpFunctionCallback0B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback0<R>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch() {
	R r = (*_f)(_ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback0B4<void, BA1, BA2, BA3, BA4> : public XorpCallback0<void> {
    typedef void (*F)(BA1, BA2, BA3, BA4);
    XorpFunctionCallback0B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback0<void>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch() {
	(*_f)(_ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 4 bound arguments.
 */
template <class R, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback0<R>::RefPtr
callback(R (*f)(BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B4<R, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback0B4 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3, BA4) ;
    XorpMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3, BA4) ;
    XorpMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback0B4
    : public XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B4() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback0B4<void,O, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B4() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory0B4
{
    inline static XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory0B4<R, O, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)(BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory0B4<R,  O, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)(BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory0B4<R,  O, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback0B4 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback0B4
    : public XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B4() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback0B4<void,O, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback0B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B4() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B4<void, O, BA1, BA2, BA3, BA4>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory0B4
{
    inline static XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory0B4<R, O, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback0B4<R, O, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)(BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory0B4<R,  const O, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)(BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory0B4<R,  const O, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback0B5 : public XorpCallback0<R> {
    typedef R (*F)(BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback0B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback0<R>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch() {
	R r = (*_f)(_ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback0B5<void, BA1, BA2, BA3, BA4, BA5> : public XorpCallback0<void> {
    typedef void (*F)(BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback0B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback0<void>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch() {
	(*_f)(_ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 5 bound arguments.
 */
template <class R, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback0<R>::RefPtr
callback(R (*f)(BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B5<R, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback0B5 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback0B5
    : public XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B5() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback0B5<void,O, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B5() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory0B5
{
    inline static XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory0B5<R, O, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory0B5<R,  O, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)(BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory0B5<R,  O, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback0B5 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback0B5
    : public XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B5() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback0B5<void,O, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback0B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B5() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B5<void, O, BA1, BA2, BA3, BA4, BA5>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory0B5
{
    inline static XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory0B5<R, O, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback0B5<R, O, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory0B5<R,  const O, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)(BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory0B5<R,  const O, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback0B6 : public XorpCallback0<R> {
    typedef R (*F)(BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback0B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback0<R>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch() {
	R r = (*_f)(_ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback0B6<void, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback0<void> {
    typedef void (*F)(BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback0B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback0<void>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch() {
	(*_f)(_ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 0 dispatch time arguments and 6 bound arguments.
 */
template <class R, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback0<R>::RefPtr
callback(R (*f)(BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback0<R>::RefPtr(new XorpFunctionCallback0B6<R, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback0B6 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback0B6
    : public XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B6() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback0B6<void,O, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback0B6() {}
    void dispatch() {
	if (valid()) {
	    XorpMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory0B6
{
    inline static XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback0<R>::RefPtr
callback( O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory0B6<R,  O, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 0 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback0<R>::RefPtr
callback( O& o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory0B6<R,  O, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback0B6 : public XorpCallback0<R> {
    typedef R (O::*M)(BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback0<R>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch() {
	R r = ((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback0<void> {
    typedef void (O::*M)(BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback0<void>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch() {
	((*_o).*_m)(_ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback0B6
    : public XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B6() {}
    R dispatch() {
	if (valid()) {
	    R r = XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch();
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 0 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback0B6<void,O, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback0B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback0B6() {}
    void dispatch() {
	if (valid()) {
	    XorpConstMemberCallback0B6<void, O, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch();
	}
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory0B6
{
    inline static XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback0B6<R, O, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback0<R>::RefPtr
callback( const O* o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory0B6<R,  const O, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 0 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback0<R>::RefPtr
callback( const O& o, R (O::*p)(BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory0B6<R,  const O, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 1 late args
//

/**
 * @short Base class for callbacks with 1 dispatch time args.
 */
template<class R, class A1>
struct XorpCallback1 {
    typedef ref_ptr<XorpCallback1> RefPtr;

    virtual ~XorpCallback1() {}
    virtual R dispatch(A1) = 0;
};

/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1>
struct XorpFunctionCallback1B0 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1);
    XorpFunctionCallback1B0(F f)
	: XorpCallback1<R, A1>(),
	  _f(f)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1>
struct XorpFunctionCallback1B0<void, A1> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1);
    XorpFunctionCallback1B0(F f)
	: XorpCallback1<void, A1>(),
	  _f(f)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1)) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B0<R, A1>(f));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1>
struct XorpMemberCallback1B0 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1) ;
    XorpMemberCallback1B0(O* o, M m)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1>
struct XorpMemberCallback1B0<void, O, A1>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1) ;
    XorpMemberCallback1B0(O* o, M m)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1>
struct XorpSafeMemberCallback1B0
    : public XorpMemberCallback1B0<R, O, A1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B0<R, O, A1>::M M;
    XorpSafeMemberCallback1B0(O* o, M m)
	 : XorpMemberCallback1B0<R, O, A1>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B0() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B0<R, O, A1>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1>
struct XorpSafeMemberCallback1B0<void,O, A1>
    : public XorpMemberCallback1B0<void, O, A1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B0<void, O, A1>::M M;
    XorpSafeMemberCallback1B0(O* o, M m)
	 : XorpMemberCallback1B0<void, O, A1>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B0() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B0<void, O, A1>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, bool B=true>
struct XorpMemberCallbackFactory1B0
{
    inline static XorpMemberCallback1B0<R, O, A1>*
    make(O* o, R (O::*p)(A1))
    {
	return new XorpSafeMemberCallback1B0<R, O, A1>(o, p);
    }
};

template <class R, class O, class A1>
struct XorpMemberCallbackFactory1B0<R, O, A1, false>
{
    inline static XorpMemberCallback1B0<R, O, A1>*
    make(O* o, R (O::*p)(A1))
    {
	return new XorpMemberCallback1B0<R, O, A1>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1))
{
    return XorpMemberCallbackFactory1B0<R,  O, A1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1))
{
    return XorpMemberCallbackFactory1B0<R,  O, A1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1>
struct XorpConstMemberCallback1B0 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1)  const;
    XorpConstMemberCallback1B0(O* o, M m)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1>
struct XorpConstMemberCallback1B0<void, O, A1>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1)  const;
    XorpConstMemberCallback1B0(O* o, M m)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1>
struct XorpConstSafeMemberCallback1B0
    : public XorpConstMemberCallback1B0<R, O, A1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B0<R, O, A1>::M M;
    XorpConstSafeMemberCallback1B0(O* o, M m)
	 : XorpConstMemberCallback1B0<R, O, A1>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B0() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B0<R, O, A1>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1>
struct XorpConstSafeMemberCallback1B0<void,O, A1>
    : public XorpConstMemberCallback1B0<void, O, A1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B0<void, O, A1>::M M;
    XorpConstSafeMemberCallback1B0(O* o, M m)
	 : XorpConstMemberCallback1B0<void, O, A1>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B0() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B0<void, O, A1>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, bool B=true>
struct XorpConstMemberCallbackFactory1B0
{
    inline static XorpConstMemberCallback1B0<R, O, A1>*
    make(O* o, R (O::*p)(A1) const)
    {
	return new XorpConstSafeMemberCallback1B0<R, O, A1>(o, p);
    }
};

template <class R, class O, class A1>
struct XorpConstMemberCallbackFactory1B0<R, O, A1, false>
{
    inline static XorpConstMemberCallback1B0<R, O, A1>*
    make(O* o, R (O::*p)(A1) const)
    {
	return new XorpConstMemberCallback1B0<R, O, A1>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1) const)
{
    return XorpConstMemberCallbackFactory1B0<R,  const O, A1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1) const)
{
    return XorpConstMemberCallbackFactory1B0<R,  const O, A1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class BA1>
struct XorpFunctionCallback1B1 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1, BA1);
    XorpFunctionCallback1B1(F f, BA1 ba1)
	: XorpCallback1<R, A1>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class BA1>
struct XorpFunctionCallback1B1<void, A1, BA1> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1, BA1);
    XorpFunctionCallback1B1(F f, BA1 ba1)
	: XorpCallback1<void, A1>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class BA1>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1, BA1), BA1 ba1) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B1<R, A1, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1>
struct XorpMemberCallback1B1 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1) ;
    XorpMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class BA1>
struct XorpMemberCallback1B1<void, O, A1, BA1>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1) ;
    XorpMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1>
struct XorpSafeMemberCallback1B1
    : public XorpMemberCallback1B1<R, O, A1, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B1<R, O, A1, BA1>::M M;
    XorpSafeMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback1B1<R, O, A1, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B1() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B1<R, O, A1, BA1>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class BA1>
struct XorpSafeMemberCallback1B1<void,O, A1, BA1>
    : public XorpMemberCallback1B1<void, O, A1, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B1<void, O, A1, BA1>::M M;
    XorpSafeMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback1B1<void, O, A1, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B1() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B1<void, O, A1, BA1>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, bool B=true>
struct XorpMemberCallbackFactory1B1
{
    inline static XorpMemberCallback1B1<R, O, A1, BA1>*
    make(O* o, R (O::*p)(A1, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback1B1<R, O, A1, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class BA1>
struct XorpMemberCallbackFactory1B1<R, O, A1, BA1, false>
{
    inline static XorpMemberCallback1B1<R, O, A1, BA1>*
    make(O* o, R (O::*p)(A1, BA1), BA1 ba1)
    {
	return new XorpMemberCallback1B1<R, O, A1, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class BA1> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory1B1<R,  O, A1, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class BA1> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory1B1<R,  O, A1, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1>
struct XorpConstMemberCallback1B1 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1)  const;
    XorpConstMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class BA1>
struct XorpConstMemberCallback1B1<void, O, A1, BA1>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1)  const;
    XorpConstMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1>
struct XorpConstSafeMemberCallback1B1
    : public XorpConstMemberCallback1B1<R, O, A1, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B1<R, O, A1, BA1>::M M;
    XorpConstSafeMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback1B1<R, O, A1, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B1() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B1<R, O, A1, BA1>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class BA1>
struct XorpConstSafeMemberCallback1B1<void,O, A1, BA1>
    : public XorpConstMemberCallback1B1<void, O, A1, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B1<void, O, A1, BA1>::M M;
    XorpConstSafeMemberCallback1B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback1B1<void, O, A1, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B1() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B1<void, O, A1, BA1>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory1B1
{
    inline static XorpConstMemberCallback1B1<R, O, A1, BA1>*
    make(O* o, R (O::*p)(A1, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback1B1<R, O, A1, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class BA1>
struct XorpConstMemberCallbackFactory1B1<R, O, A1, BA1, false>
{
    inline static XorpConstMemberCallback1B1<R, O, A1, BA1>*
    make(O* o, R (O::*p)(A1, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback1B1<R, O, A1, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class BA1> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory1B1<R,  const O, A1, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class BA1> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory1B1<R,  const O, A1, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class BA1, class BA2>
struct XorpFunctionCallback1B2 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1, BA1, BA2);
    XorpFunctionCallback1B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback1<R, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class BA1, class BA2>
struct XorpFunctionCallback1B2<void, A1, BA1, BA2> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1, BA1, BA2);
    XorpFunctionCallback1B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback1<void, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class BA1, class BA2>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B2<R, A1, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2>
struct XorpMemberCallback1B2 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2) ;
    XorpMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2>
struct XorpMemberCallback1B2<void, O, A1, BA1, BA2>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2) ;
    XorpMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2>
struct XorpSafeMemberCallback1B2
    : public XorpMemberCallback1B2<R, O, A1, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B2<R, O, A1, BA1, BA2>::M M;
    XorpSafeMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback1B2<R, O, A1, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B2() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B2<R, O, A1, BA1, BA2>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2>
struct XorpSafeMemberCallback1B2<void,O, A1, BA1, BA2>
    : public XorpMemberCallback1B2<void, O, A1, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B2<void, O, A1, BA1, BA2>::M M;
    XorpSafeMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback1B2<void, O, A1, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B2() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B2<void, O, A1, BA1, BA2>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory1B2
{
    inline static XorpMemberCallback1B2<R, O, A1, BA1, BA2>*
    make(O* o, R (O::*p)(A1, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback1B2<R, O, A1, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class BA1, class BA2>
struct XorpMemberCallbackFactory1B2<R, O, A1, BA1, BA2, false>
{
    inline static XorpMemberCallback1B2<R, O, A1, BA1, BA2>*
    make(O* o, R (O::*p)(A1, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback1B2<R, O, A1, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory1B2<R,  O, A1, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory1B2<R,  O, A1, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2>
struct XorpConstMemberCallback1B2 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2)  const;
    XorpConstMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2>
struct XorpConstMemberCallback1B2<void, O, A1, BA1, BA2>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2)  const;
    XorpConstMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2>
struct XorpConstSafeMemberCallback1B2
    : public XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>::M M;
    XorpConstSafeMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B2() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2>
struct XorpConstSafeMemberCallback1B2<void,O, A1, BA1, BA2>
    : public XorpConstMemberCallback1B2<void, O, A1, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B2<void, O, A1, BA1, BA2>::M M;
    XorpConstSafeMemberCallback1B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback1B2<void, O, A1, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B2() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B2<void, O, A1, BA1, BA2>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory1B2
{
    inline static XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>*
    make(O* o, R (O::*p)(A1, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback1B2<R, O, A1, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class BA1, class BA2>
struct XorpConstMemberCallbackFactory1B2<R, O, A1, BA1, BA2, false>
{
    inline static XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>*
    make(O* o, R (O::*p)(A1, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback1B2<R, O, A1, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory1B2<R,  const O, A1, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory1B2<R,  const O, A1, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3>
struct XorpFunctionCallback1B3 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1, BA1, BA2, BA3);
    XorpFunctionCallback1B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback1<R, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class BA1, class BA2, class BA3>
struct XorpFunctionCallback1B3<void, A1, BA1, BA2, BA3> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1, BA1, BA2, BA3);
    XorpFunctionCallback1B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback1<void, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B3<R, A1, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3>
struct XorpMemberCallback1B3 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3) ;
    XorpMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3>
struct XorpMemberCallback1B3<void, O, A1, BA1, BA2, BA3>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3) ;
    XorpMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback1B3
    : public XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B3() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback1B3<void,O, A1, BA1, BA2, BA3>
    : public XorpMemberCallback1B3<void, O, A1, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B3<void, O, A1, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback1B3<void, O, A1, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B3() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B3<void, O, A1, BA1, BA2, BA3>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory1B3
{
    inline static XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback1B3<R, O, A1, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory1B3<R, O, A1, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback1B3<R, O, A1, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory1B3<R,  O, A1, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory1B3<R,  O, A1, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback1B3 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3)  const;
    XorpConstMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback1B3<void, O, A1, BA1, BA2, BA3>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3)  const;
    XorpConstMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback1B3
    : public XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B3() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback1B3<void,O, A1, BA1, BA2, BA3>
    : public XorpConstMemberCallback1B3<void, O, A1, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B3<void, O, A1, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback1B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback1B3<void, O, A1, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B3() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B3<void, O, A1, BA1, BA2, BA3>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory1B3
{
    inline static XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback1B3<R, O, A1, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory1B3<R, O, A1, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback1B3<R, O, A1, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory1B3<R,  const O, A1, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory1B3<R,  const O, A1, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback1B4 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1, BA1, BA2, BA3, BA4);
    XorpFunctionCallback1B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback1<R, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback1B4<void, A1, BA1, BA2, BA3, BA4> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1, BA1, BA2, BA3, BA4);
    XorpFunctionCallback1B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback1<void, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B4<R, A1, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback1B4 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback1B4
    : public XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B4() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback1B4<void,O, A1, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B4() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory1B4
{
    inline static XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory1B4<R, O, A1, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory1B4<R,  O, A1, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory1B4<R,  O, A1, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback1B4 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback1B4
    : public XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B4() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback1B4<void,O, A1, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback1B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B4() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B4<void, O, A1, BA1, BA2, BA3, BA4>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory1B4
{
    inline static XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory1B4<R, O, A1, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback1B4<R, O, A1, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory1B4<R,  const O, A1, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory1B4<R,  const O, A1, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback1B5 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback1B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback1<R, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback1B5<void, A1, BA1, BA2, BA3, BA4, BA5> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback1B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback1<void, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B5<R, A1, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback1B5 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback1B5
    : public XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B5() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback1B5<void,O, A1, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B5() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory1B5
{
    inline static XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory1B5<R,  O, A1, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory1B5<R,  O, A1, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback1B5 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback1B5
    : public XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B5() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback1B5<void,O, A1, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback1B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B5() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B5<void, O, A1, BA1, BA2, BA3, BA4, BA5>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory1B5
{
    inline static XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback1B5<R, O, A1, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory1B5<R,  const O, A1, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory1B5<R,  const O, A1, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback1B6 : public XorpCallback1<R, A1> {
    typedef R (*F)(A1, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback1B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback1<R, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1) {
	R r = (*_f)(a1, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback1B6<void, A1, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback1<void, A1> {
    typedef void (*F)(A1, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback1B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback1<void, A1>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1) {
	(*_f)(a1, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 1 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback1<R, A1>::RefPtr
callback(R (*f)(A1, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback1<R, A1>::RefPtr(new XorpFunctionCallback1B6<R, A1, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback1B6 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback1B6
    : public XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B6() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback1B6<void,O, A1, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback1B6() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory1B6
{
    inline static XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback1<R, A1>::RefPtr
callback( O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory1B6<R,  O, A1, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 1 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback1<R, A1>::RefPtr
callback( O& o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory1B6<R,  O, A1, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback1B6 : public XorpCallback1<R, A1> {
    typedef R (O::*M)(A1, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback1<R, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1) {
	R r = ((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback1<void, A1> {
    typedef void (O::*M)(A1, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback1<void, A1>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1) {
	((*_o).*_m)(a1, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback1B6
    : public XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B6() {}
    R dispatch(A1 a1) {
	if (valid()) {
	    R r = XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 1 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback1B6<void,O, A1, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback1B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback1B6() {}
    void dispatch(A1 a1) {
	if (valid()) {
	    XorpConstMemberCallback1B6<void, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1);
	}
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory1B6
{
    inline static XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback1B6<R, O, A1, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback1<R, A1>::RefPtr
callback( const O* o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory1B6<R,  const O, A1, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 1 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback1<R, A1>::RefPtr
callback( const O& o, R (O::*p)(A1, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory1B6<R,  const O, A1, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 2 late args
//

/**
 * @short Base class for callbacks with 2 dispatch time args.
 */
template<class R, class A1, class A2>
struct XorpCallback2 {
    typedef ref_ptr<XorpCallback2> RefPtr;

    virtual ~XorpCallback2() {}
    virtual R dispatch(A1, A2) = 0;
};

/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2>
struct XorpFunctionCallback2B0 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2);
    XorpFunctionCallback2B0(F f)
	: XorpCallback2<R, A1, A2>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2>
struct XorpFunctionCallback2B0<void, A1, A2> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2);
    XorpFunctionCallback2B0(F f)
	: XorpCallback2<void, A1, A2>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2)) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B0<R, A1, A2>(f));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2>
struct XorpMemberCallback2B0 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2) ;
    XorpMemberCallback2B0(O* o, M m)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2>
struct XorpMemberCallback2B0<void, O, A1, A2>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2) ;
    XorpMemberCallback2B0(O* o, M m)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2>
struct XorpSafeMemberCallback2B0
    : public XorpMemberCallback2B0<R, O, A1, A2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B0<R, O, A1, A2>::M M;
    XorpSafeMemberCallback2B0(O* o, M m)
	 : XorpMemberCallback2B0<R, O, A1, A2>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B0() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B0<R, O, A1, A2>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2>
struct XorpSafeMemberCallback2B0<void,O, A1, A2>
    : public XorpMemberCallback2B0<void, O, A1, A2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B0<void, O, A1, A2>::M M;
    XorpSafeMemberCallback2B0(O* o, M m)
	 : XorpMemberCallback2B0<void, O, A1, A2>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B0() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B0<void, O, A1, A2>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, bool B=true>
struct XorpMemberCallbackFactory2B0
{
    inline static XorpMemberCallback2B0<R, O, A1, A2>*
    make(O* o, R (O::*p)(A1, A2))
    {
	return new XorpSafeMemberCallback2B0<R, O, A1, A2>(o, p);
    }
};

template <class R, class O, class A1, class A2>
struct XorpMemberCallbackFactory2B0<R, O, A1, A2, false>
{
    inline static XorpMemberCallback2B0<R, O, A1, A2>*
    make(O* o, R (O::*p)(A1, A2))
    {
	return new XorpMemberCallback2B0<R, O, A1, A2>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2))
{
    return XorpMemberCallbackFactory2B0<R,  O, A1, A2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2))
{
    return XorpMemberCallbackFactory2B0<R,  O, A1, A2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2>
struct XorpConstMemberCallback2B0 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2)  const;
    XorpConstMemberCallback2B0(O* o, M m)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2>
struct XorpConstMemberCallback2B0<void, O, A1, A2>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2)  const;
    XorpConstMemberCallback2B0(O* o, M m)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2>
struct XorpConstSafeMemberCallback2B0
    : public XorpConstMemberCallback2B0<R, O, A1, A2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B0<R, O, A1, A2>::M M;
    XorpConstSafeMemberCallback2B0(O* o, M m)
	 : XorpConstMemberCallback2B0<R, O, A1, A2>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B0() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B0<R, O, A1, A2>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2>
struct XorpConstSafeMemberCallback2B0<void,O, A1, A2>
    : public XorpConstMemberCallback2B0<void, O, A1, A2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B0<void, O, A1, A2>::M M;
    XorpConstSafeMemberCallback2B0(O* o, M m)
	 : XorpConstMemberCallback2B0<void, O, A1, A2>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B0() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B0<void, O, A1, A2>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, bool B=true>
struct XorpConstMemberCallbackFactory2B0
{
    inline static XorpConstMemberCallback2B0<R, O, A1, A2>*
    make(O* o, R (O::*p)(A1, A2) const)
    {
	return new XorpConstSafeMemberCallback2B0<R, O, A1, A2>(o, p);
    }
};

template <class R, class O, class A1, class A2>
struct XorpConstMemberCallbackFactory2B0<R, O, A1, A2, false>
{
    inline static XorpConstMemberCallback2B0<R, O, A1, A2>*
    make(O* o, R (O::*p)(A1, A2) const)
    {
	return new XorpConstMemberCallback2B0<R, O, A1, A2>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2) const)
{
    return XorpConstMemberCallbackFactory2B0<R,  const O, A1, A2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2) const)
{
    return XorpConstMemberCallbackFactory2B0<R,  const O, A1, A2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class BA1>
struct XorpFunctionCallback2B1 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2, BA1);
    XorpFunctionCallback2B1(F f, BA1 ba1)
	: XorpCallback2<R, A1, A2>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class BA1>
struct XorpFunctionCallback2B1<void, A1, A2, BA1> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2, BA1);
    XorpFunctionCallback2B1(F f, BA1 ba1)
	: XorpCallback2<void, A1, A2>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class BA1>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2, BA1), BA1 ba1) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B1<R, A1, A2, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1>
struct XorpMemberCallback2B1 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1) ;
    XorpMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1>
struct XorpMemberCallback2B1<void, O, A1, A2, BA1>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1) ;
    XorpMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1>
struct XorpSafeMemberCallback2B1
    : public XorpMemberCallback2B1<R, O, A1, A2, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B1<R, O, A1, A2, BA1>::M M;
    XorpSafeMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback2B1<R, O, A1, A2, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B1() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B1<R, O, A1, A2, BA1>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1>
struct XorpSafeMemberCallback2B1<void,O, A1, A2, BA1>
    : public XorpMemberCallback2B1<void, O, A1, A2, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B1<void, O, A1, A2, BA1>::M M;
    XorpSafeMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback2B1<void, O, A1, A2, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B1() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B1<void, O, A1, A2, BA1>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, bool B=true>
struct XorpMemberCallbackFactory2B1
{
    inline static XorpMemberCallback2B1<R, O, A1, A2, BA1>*
    make(O* o, R (O::*p)(A1, A2, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback2B1<R, O, A1, A2, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class BA1>
struct XorpMemberCallbackFactory2B1<R, O, A1, A2, BA1, false>
{
    inline static XorpMemberCallback2B1<R, O, A1, A2, BA1>*
    make(O* o, R (O::*p)(A1, A2, BA1), BA1 ba1)
    {
	return new XorpMemberCallback2B1<R, O, A1, A2, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory2B1<R,  O, A1, A2, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory2B1<R,  O, A1, A2, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1>
struct XorpConstMemberCallback2B1 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1)  const;
    XorpConstMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1>
struct XorpConstMemberCallback2B1<void, O, A1, A2, BA1>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1)  const;
    XorpConstMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1>
struct XorpConstSafeMemberCallback2B1
    : public XorpConstMemberCallback2B1<R, O, A1, A2, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B1<R, O, A1, A2, BA1>::M M;
    XorpConstSafeMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback2B1<R, O, A1, A2, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B1() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B1<R, O, A1, A2, BA1>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1>
struct XorpConstSafeMemberCallback2B1<void,O, A1, A2, BA1>
    : public XorpConstMemberCallback2B1<void, O, A1, A2, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B1<void, O, A1, A2, BA1>::M M;
    XorpConstSafeMemberCallback2B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback2B1<void, O, A1, A2, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B1() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B1<void, O, A1, A2, BA1>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory2B1
{
    inline static XorpConstMemberCallback2B1<R, O, A1, A2, BA1>*
    make(O* o, R (O::*p)(A1, A2, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback2B1<R, O, A1, A2, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class BA1>
struct XorpConstMemberCallbackFactory2B1<R, O, A1, A2, BA1, false>
{
    inline static XorpConstMemberCallback2B1<R, O, A1, A2, BA1>*
    make(O* o, R (O::*p)(A1, A2, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback2B1<R, O, A1, A2, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory2B1<R,  const O, A1, A2, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory2B1<R,  const O, A1, A2, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2>
struct XorpFunctionCallback2B2 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2, BA1, BA2);
    XorpFunctionCallback2B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback2<R, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class BA1, class BA2>
struct XorpFunctionCallback2B2<void, A1, A2, BA1, BA2> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2, BA1, BA2);
    XorpFunctionCallback2B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback2<void, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B2<R, A1, A2, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2>
struct XorpMemberCallback2B2 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2) ;
    XorpMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2>
struct XorpMemberCallback2B2<void, O, A1, A2, BA1, BA2>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2) ;
    XorpMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2>
struct XorpSafeMemberCallback2B2
    : public XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>::M M;
    XorpSafeMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B2() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2>
struct XorpSafeMemberCallback2B2<void,O, A1, A2, BA1, BA2>
    : public XorpMemberCallback2B2<void, O, A1, A2, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B2<void, O, A1, A2, BA1, BA2>::M M;
    XorpSafeMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback2B2<void, O, A1, A2, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B2() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B2<void, O, A1, A2, BA1, BA2>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory2B2
{
    inline static XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback2B2<R, O, A1, A2, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2>
struct XorpMemberCallbackFactory2B2<R, O, A1, A2, BA1, BA2, false>
{
    inline static XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback2B2<R, O, A1, A2, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory2B2<R,  O, A1, A2, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory2B2<R,  O, A1, A2, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2>
struct XorpConstMemberCallback2B2 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2)  const;
    XorpConstMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2>
struct XorpConstMemberCallback2B2<void, O, A1, A2, BA1, BA2>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2)  const;
    XorpConstMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2>
struct XorpConstSafeMemberCallback2B2
    : public XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>::M M;
    XorpConstSafeMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B2() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2>
struct XorpConstSafeMemberCallback2B2<void,O, A1, A2, BA1, BA2>
    : public XorpConstMemberCallback2B2<void, O, A1, A2, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B2<void, O, A1, A2, BA1, BA2>::M M;
    XorpConstSafeMemberCallback2B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback2B2<void, O, A1, A2, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B2() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B2<void, O, A1, A2, BA1, BA2>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory2B2
{
    inline static XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback2B2<R, O, A1, A2, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2>
struct XorpConstMemberCallbackFactory2B2<R, O, A1, A2, BA1, BA2, false>
{
    inline static XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback2B2<R, O, A1, A2, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory2B2<R,  const O, A1, A2, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory2B2<R,  const O, A1, A2, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpFunctionCallback2B3 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2, BA1, BA2, BA3);
    XorpFunctionCallback2B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback2<R, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class BA1, class BA2, class BA3>
struct XorpFunctionCallback2B3<void, A1, A2, BA1, BA2, BA3> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2, BA1, BA2, BA3);
    XorpFunctionCallback2B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback2<void, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B3<R, A1, A2, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpMemberCallback2B3 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3) ;
    XorpMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3) ;
    XorpMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback2B3
    : public XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B3() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback2B3<void,O, A1, A2, BA1, BA2, BA3>
    : public XorpMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B3() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory2B3
{
    inline static XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory2B3<R, O, A1, A2, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory2B3<R,  O, A1, A2, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory2B3<R,  O, A1, A2, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback2B3 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3)  const;
    XorpConstMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3)  const;
    XorpConstMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback2B3
    : public XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B3() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback2B3<void,O, A1, A2, BA1, BA2, BA3>
    : public XorpConstMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback2B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B3() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B3<void, O, A1, A2, BA1, BA2, BA3>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory2B3
{
    inline static XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory2B3<R, O, A1, A2, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback2B3<R, O, A1, A2, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory2B3<R,  const O, A1, A2, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory2B3<R,  const O, A1, A2, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback2B4 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2, BA1, BA2, BA3, BA4);
    XorpFunctionCallback2B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback2<R, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback2B4<void, A1, A2, BA1, BA2, BA3, BA4> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2, BA1, BA2, BA3, BA4);
    XorpFunctionCallback2B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback2<void, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B4<R, A1, A2, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback2B4 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback2B4
    : public XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B4() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback2B4<void,O, A1, A2, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B4() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory2B4
{
    inline static XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory2B4<R, O, A1, A2, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory2B4<R,  O, A1, A2, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory2B4<R,  O, A1, A2, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback2B4 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback2B4
    : public XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B4() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback2B4<void,O, A1, A2, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback2B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B4() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B4<void, O, A1, A2, BA1, BA2, BA3, BA4>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory2B4
{
    inline static XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory2B4<R, O, A1, A2, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback2B4<R, O, A1, A2, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory2B4<R,  const O, A1, A2, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory2B4<R,  const O, A1, A2, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback2B5 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback2B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback2<R, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback2B5<void, A1, A2, BA1, BA2, BA3, BA4, BA5> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback2B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback2<void, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B5<R, A1, A2, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback2B5 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback2B5
    : public XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B5() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback2B5<void,O, A1, A2, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B5() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory2B5
{
    inline static XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory2B5<R,  O, A1, A2, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory2B5<R,  O, A1, A2, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback2B5 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback2B5
    : public XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B5() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback2B5<void,O, A1, A2, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback2B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B5() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B5<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory2B5
{
    inline static XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback2B5<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory2B5<R,  const O, A1, A2, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory2B5<R,  const O, A1, A2, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback2B6 : public XorpCallback2<R, A1, A2> {
    typedef R (*F)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback2B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback2<R, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2) {
	R r = (*_f)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback2B6<void, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback2<void, A1, A2> {
    typedef void (*F)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback2B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback2<void, A1, A2>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2) {
	(*_f)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 2 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback2<R, A1, A2>::RefPtr
callback(R (*f)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback2<R, A1, A2>::RefPtr(new XorpFunctionCallback2B6<R, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback2B6 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback2B6
    : public XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B6() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback2B6<void,O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback2B6() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory2B6
{
    inline static XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory2B6<R,  O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 2 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback2<R, A1, A2>::RefPtr
callback( O& o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory2B6<R,  O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback2B6 : public XorpCallback2<R, A1, A2> {
    typedef R (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback2<R, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2) {
	R r = ((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback2<void, A1, A2> {
    typedef void (O::*M)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback2<void, A1, A2>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2) {
	((*_o).*_m)(a1, a2, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback2B6
    : public XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B6() {}
    R dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    R r = XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 2 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback2B6<void,O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback2B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback2B6() {}
    void dispatch(A1 a1, A2 a2) {
	if (valid()) {
	    XorpConstMemberCallback2B6<void, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2);
	}
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory2B6
{
    inline static XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback2B6<R, O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory2B6<R,  const O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 2 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback2<R, A1, A2>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory2B6<R,  const O, A1, A2, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 3 late args
//

/**
 * @short Base class for callbacks with 3 dispatch time args.
 */
template<class R, class A1, class A2, class A3>
struct XorpCallback3 {
    typedef ref_ptr<XorpCallback3> RefPtr;

    virtual ~XorpCallback3() {}
    virtual R dispatch(A1, A2, A3) = 0;
};

/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3>
struct XorpFunctionCallback3B0 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3);
    XorpFunctionCallback3B0(F f)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3>
struct XorpFunctionCallback3B0<void, A1, A2, A3> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3);
    XorpFunctionCallback3B0(F f)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3)) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B0<R, A1, A2, A3>(f));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3>
struct XorpMemberCallback3B0 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3) ;
    XorpMemberCallback3B0(O* o, M m)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3>
struct XorpMemberCallback3B0<void, O, A1, A2, A3>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3) ;
    XorpMemberCallback3B0(O* o, M m)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3>
struct XorpSafeMemberCallback3B0
    : public XorpMemberCallback3B0<R, O, A1, A2, A3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B0<R, O, A1, A2, A3>::M M;
    XorpSafeMemberCallback3B0(O* o, M m)
	 : XorpMemberCallback3B0<R, O, A1, A2, A3>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B0<R, O, A1, A2, A3>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3>
struct XorpSafeMemberCallback3B0<void,O, A1, A2, A3>
    : public XorpMemberCallback3B0<void, O, A1, A2, A3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B0<void, O, A1, A2, A3>::M M;
    XorpSafeMemberCallback3B0(O* o, M m)
	 : XorpMemberCallback3B0<void, O, A1, A2, A3>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B0<void, O, A1, A2, A3>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, bool B=true>
struct XorpMemberCallbackFactory3B0
{
    inline static XorpMemberCallback3B0<R, O, A1, A2, A3>*
    make(O* o, R (O::*p)(A1, A2, A3))
    {
	return new XorpSafeMemberCallback3B0<R, O, A1, A2, A3>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3>
struct XorpMemberCallbackFactory3B0<R, O, A1, A2, A3, false>
{
    inline static XorpMemberCallback3B0<R, O, A1, A2, A3>*
    make(O* o, R (O::*p)(A1, A2, A3))
    {
	return new XorpMemberCallback3B0<R, O, A1, A2, A3>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3))
{
    return XorpMemberCallbackFactory3B0<R,  O, A1, A2, A3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3))
{
    return XorpMemberCallbackFactory3B0<R,  O, A1, A2, A3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3>
struct XorpConstMemberCallback3B0 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3)  const;
    XorpConstMemberCallback3B0(O* o, M m)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3>
struct XorpConstMemberCallback3B0<void, O, A1, A2, A3>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3)  const;
    XorpConstMemberCallback3B0(O* o, M m)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3>
struct XorpConstSafeMemberCallback3B0
    : public XorpConstMemberCallback3B0<R, O, A1, A2, A3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B0<R, O, A1, A2, A3>::M M;
    XorpConstSafeMemberCallback3B0(O* o, M m)
	 : XorpConstMemberCallback3B0<R, O, A1, A2, A3>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B0<R, O, A1, A2, A3>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3>
struct XorpConstSafeMemberCallback3B0<void,O, A1, A2, A3>
    : public XorpConstMemberCallback3B0<void, O, A1, A2, A3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B0<void, O, A1, A2, A3>::M M;
    XorpConstSafeMemberCallback3B0(O* o, M m)
	 : XorpConstMemberCallback3B0<void, O, A1, A2, A3>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B0<void, O, A1, A2, A3>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, bool B=true>
struct XorpConstMemberCallbackFactory3B0
{
    inline static XorpConstMemberCallback3B0<R, O, A1, A2, A3>*
    make(O* o, R (O::*p)(A1, A2, A3) const)
    {
	return new XorpConstSafeMemberCallback3B0<R, O, A1, A2, A3>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3>
struct XorpConstMemberCallbackFactory3B0<R, O, A1, A2, A3, false>
{
    inline static XorpConstMemberCallback3B0<R, O, A1, A2, A3>*
    make(O* o, R (O::*p)(A1, A2, A3) const)
    {
	return new XorpConstMemberCallback3B0<R, O, A1, A2, A3>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3) const)
{
    return XorpConstMemberCallbackFactory3B0<R,  const O, A1, A2, A3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3) const)
{
    return XorpConstMemberCallbackFactory3B0<R,  const O, A1, A2, A3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class BA1>
struct XorpFunctionCallback3B1 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3, BA1);
    XorpFunctionCallback3B1(F f, BA1 ba1)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class BA1>
struct XorpFunctionCallback3B1<void, A1, A2, A3, BA1> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3, BA1);
    XorpFunctionCallback3B1(F f, BA1 ba1)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class BA1>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3, BA1), BA1 ba1) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B1<R, A1, A2, A3, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1>
struct XorpMemberCallback3B1 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1) ;
    XorpMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1>
struct XorpMemberCallback3B1<void, O, A1, A2, A3, BA1>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1) ;
    XorpMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1>
struct XorpSafeMemberCallback3B1
    : public XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>::M M;
    XorpSafeMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1>
struct XorpSafeMemberCallback3B1<void,O, A1, A2, A3, BA1>
    : public XorpMemberCallback3B1<void, O, A1, A2, A3, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B1<void, O, A1, A2, A3, BA1>::M M;
    XorpSafeMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback3B1<void, O, A1, A2, A3, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B1<void, O, A1, A2, A3, BA1>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, bool B=true>
struct XorpMemberCallbackFactory3B1
{
    inline static XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback3B1<R, O, A1, A2, A3, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1>
struct XorpMemberCallbackFactory3B1<R, O, A1, A2, A3, BA1, false>
{
    inline static XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1), BA1 ba1)
    {
	return new XorpMemberCallback3B1<R, O, A1, A2, A3, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory3B1<R,  O, A1, A2, A3, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory3B1<R,  O, A1, A2, A3, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1>
struct XorpConstMemberCallback3B1 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1)  const;
    XorpConstMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1>
struct XorpConstMemberCallback3B1<void, O, A1, A2, A3, BA1>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1)  const;
    XorpConstMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1>
struct XorpConstSafeMemberCallback3B1
    : public XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>::M M;
    XorpConstSafeMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1>
struct XorpConstSafeMemberCallback3B1<void,O, A1, A2, A3, BA1>
    : public XorpConstMemberCallback3B1<void, O, A1, A2, A3, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B1<void, O, A1, A2, A3, BA1>::M M;
    XorpConstSafeMemberCallback3B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback3B1<void, O, A1, A2, A3, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B1<void, O, A1, A2, A3, BA1>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory3B1
{
    inline static XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback3B1<R, O, A1, A2, A3, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1>
struct XorpConstMemberCallbackFactory3B1<R, O, A1, A2, A3, BA1, false>
{
    inline static XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback3B1<R, O, A1, A2, A3, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory3B1<R,  const O, A1, A2, A3, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory3B1<R,  const O, A1, A2, A3, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2>
struct XorpFunctionCallback3B2 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3, BA1, BA2);
    XorpFunctionCallback3B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class BA1, class BA2>
struct XorpFunctionCallback3B2<void, A1, A2, A3, BA1, BA2> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3, BA1, BA2);
    XorpFunctionCallback3B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B2<R, A1, A2, A3, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpMemberCallback3B2 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2) ;
    XorpMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2) ;
    XorpMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpSafeMemberCallback3B2
    : public XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>::M M;
    XorpSafeMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpSafeMemberCallback3B2<void,O, A1, A2, A3, BA1, BA2>
    : public XorpMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>::M M;
    XorpSafeMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory3B2
{
    inline static XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpMemberCallbackFactory3B2<R, O, A1, A2, A3, BA1, BA2, false>
{
    inline static XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory3B2<R,  O, A1, A2, A3, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory3B2<R,  O, A1, A2, A3, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpConstMemberCallback3B2 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2)  const;
    XorpConstMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpConstMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2)  const;
    XorpConstMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpConstSafeMemberCallback3B2
    : public XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>::M M;
    XorpConstSafeMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpConstSafeMemberCallback3B2<void,O, A1, A2, A3, BA1, BA2>
    : public XorpConstMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>::M M;
    XorpConstSafeMemberCallback3B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B2<void, O, A1, A2, A3, BA1, BA2>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory3B2
{
    inline static XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2>
struct XorpConstMemberCallbackFactory3B2<R, O, A1, A2, A3, BA1, BA2, false>
{
    inline static XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback3B2<R, O, A1, A2, A3, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory3B2<R,  const O, A1, A2, A3, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory3B2<R,  const O, A1, A2, A3, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpFunctionCallback3B3 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3, BA1, BA2, BA3);
    XorpFunctionCallback3B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpFunctionCallback3B3<void, A1, A2, A3, BA1, BA2, BA3> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3, BA1, BA2, BA3);
    XorpFunctionCallback3B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B3<R, A1, A2, A3, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpMemberCallback3B3 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3) ;
    XorpMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3) ;
    XorpMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback3B3
    : public XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback3B3<void,O, A1, A2, A3, BA1, BA2, BA3>
    : public XorpMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory3B3
{
    inline static XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory3B3<R, O, A1, A2, A3, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory3B3<R,  O, A1, A2, A3, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory3B3<R,  O, A1, A2, A3, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback3B3 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3)  const;
    XorpConstMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3)  const;
    XorpConstMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback3B3
    : public XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback3B3<void,O, A1, A2, A3, BA1, BA2, BA3>
    : public XorpConstMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback3B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B3<void, O, A1, A2, A3, BA1, BA2, BA3>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory3B3
{
    inline static XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory3B3<R, O, A1, A2, A3, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback3B3<R, O, A1, A2, A3, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory3B3<R,  const O, A1, A2, A3, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory3B3<R,  const O, A1, A2, A3, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback3B4 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3, BA1, BA2, BA3, BA4);
    XorpFunctionCallback3B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback3B4<void, A1, A2, A3, BA1, BA2, BA3, BA4> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3, BA1, BA2, BA3, BA4);
    XorpFunctionCallback3B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B4<R, A1, A2, A3, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback3B4 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback3B4
    : public XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback3B4<void,O, A1, A2, A3, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory3B4
{
    inline static XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory3B4<R,  O, A1, A2, A3, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory3B4<R,  O, A1, A2, A3, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback3B4 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback3B4
    : public XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback3B4<void,O, A1, A2, A3, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback3B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B4<void, O, A1, A2, A3, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory3B4
{
    inline static XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback3B4<R, O, A1, A2, A3, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory3B4<R,  const O, A1, A2, A3, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory3B4<R,  const O, A1, A2, A3, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback3B5 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback3B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback3B5<void, A1, A2, A3, BA1, BA2, BA3, BA4, BA5> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback3B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B5<R, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback3B5 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback3B5
    : public XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback3B5<void,O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory3B5
{
    inline static XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory3B5<R,  O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory3B5<R,  O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback3B5 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback3B5
    : public XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback3B5<void,O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback3B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B5<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory3B5
{
    inline static XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback3B5<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory3B5<R,  const O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory3B5<R,  const O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback3B6 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (*F)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback3B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback3<R, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = (*_f)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback3B6<void, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback3<void, A1, A2, A3> {
    typedef void (*F)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback3B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback3<void, A1, A2, A3>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	(*_f)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 3 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback(R (*f)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback3<R, A1, A2, A3>::RefPtr(new XorpFunctionCallback3B6<R, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback3B6 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback3B6
    : public XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback3B6<void,O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback3B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory3B6
{
    inline static XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory3B6<R,  O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 3 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory3B6<R,  O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback3B6 : public XorpCallback3<R, A1, A2, A3> {
    typedef R (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback3<R, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	R r = ((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback3<void, A1, A2, A3> {
    typedef void (O::*M)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback3<void, A1, A2, A3>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	((*_o).*_m)(a1, a2, a3, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback3B6
    : public XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    R r = XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 3 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback3B6<void,O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback3B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback3B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3) {
	if (valid()) {
	    XorpConstMemberCallback3B6<void, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory3B6
{
    inline static XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback3B6<R, O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory3B6<R,  const O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 3 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback3<R, A1, A2, A3>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory3B6<R,  const O, A1, A2, A3, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 4 late args
//

/**
 * @short Base class for callbacks with 4 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4>
struct XorpCallback4 {
    typedef ref_ptr<XorpCallback4> RefPtr;

    virtual ~XorpCallback4() {}
    virtual R dispatch(A1, A2, A3, A4) = 0;
};

/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4>
struct XorpFunctionCallback4B0 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4);
    XorpFunctionCallback4B0(F f)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4>
struct XorpFunctionCallback4B0<void, A1, A2, A3, A4> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4);
    XorpFunctionCallback4B0(F f)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4)) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B0<R, A1, A2, A3, A4>(f));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4>
struct XorpMemberCallback4B0 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4) ;
    XorpMemberCallback4B0(O* o, M m)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4>
struct XorpMemberCallback4B0<void, O, A1, A2, A3, A4>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4) ;
    XorpMemberCallback4B0(O* o, M m)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4>
struct XorpSafeMemberCallback4B0
    : public XorpMemberCallback4B0<R, O, A1, A2, A3, A4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B0<R, O, A1, A2, A3, A4>::M M;
    XorpSafeMemberCallback4B0(O* o, M m)
	 : XorpMemberCallback4B0<R, O, A1, A2, A3, A4>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B0<R, O, A1, A2, A3, A4>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4>
struct XorpSafeMemberCallback4B0<void,O, A1, A2, A3, A4>
    : public XorpMemberCallback4B0<void, O, A1, A2, A3, A4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B0<void, O, A1, A2, A3, A4>::M M;
    XorpSafeMemberCallback4B0(O* o, M m)
	 : XorpMemberCallback4B0<void, O, A1, A2, A3, A4>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B0<void, O, A1, A2, A3, A4>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, bool B=true>
struct XorpMemberCallbackFactory4B0
{
    inline static XorpMemberCallback4B0<R, O, A1, A2, A3, A4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4))
    {
	return new XorpSafeMemberCallback4B0<R, O, A1, A2, A3, A4>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4>
struct XorpMemberCallbackFactory4B0<R, O, A1, A2, A3, A4, false>
{
    inline static XorpMemberCallback4B0<R, O, A1, A2, A3, A4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4))
    {
	return new XorpMemberCallback4B0<R, O, A1, A2, A3, A4>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4))
{
    return XorpMemberCallbackFactory4B0<R,  O, A1, A2, A3, A4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4))
{
    return XorpMemberCallbackFactory4B0<R,  O, A1, A2, A3, A4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4>
struct XorpConstMemberCallback4B0 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4)  const;
    XorpConstMemberCallback4B0(O* o, M m)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4>
struct XorpConstMemberCallback4B0<void, O, A1, A2, A3, A4>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4)  const;
    XorpConstMemberCallback4B0(O* o, M m)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4>
struct XorpConstSafeMemberCallback4B0
    : public XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>::M M;
    XorpConstSafeMemberCallback4B0(O* o, M m)
	 : XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4>
struct XorpConstSafeMemberCallback4B0<void,O, A1, A2, A3, A4>
    : public XorpConstMemberCallback4B0<void, O, A1, A2, A3, A4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B0<void, O, A1, A2, A3, A4>::M M;
    XorpConstSafeMemberCallback4B0(O* o, M m)
	 : XorpConstMemberCallback4B0<void, O, A1, A2, A3, A4>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B0<void, O, A1, A2, A3, A4>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, bool B=true>
struct XorpConstMemberCallbackFactory4B0
{
    inline static XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4) const)
    {
	return new XorpConstSafeMemberCallback4B0<R, O, A1, A2, A3, A4>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4>
struct XorpConstMemberCallbackFactory4B0<R, O, A1, A2, A3, A4, false>
{
    inline static XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4) const)
    {
	return new XorpConstMemberCallback4B0<R, O, A1, A2, A3, A4>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4) const)
{
    return XorpConstMemberCallbackFactory4B0<R,  const O, A1, A2, A3, A4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4) const)
{
    return XorpConstMemberCallbackFactory4B0<R,  const O, A1, A2, A3, A4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1>
struct XorpFunctionCallback4B1 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4, BA1);
    XorpFunctionCallback4B1(F f, BA1 ba1)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class BA1>
struct XorpFunctionCallback4B1<void, A1, A2, A3, A4, BA1> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4, BA1);
    XorpFunctionCallback4B1(F f, BA1 ba1)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4, BA1), BA1 ba1) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B1<R, A1, A2, A3, A4, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpMemberCallback4B1 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1) ;
    XorpMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1) ;
    XorpMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpSafeMemberCallback4B1
    : public XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>::M M;
    XorpSafeMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpSafeMemberCallback4B1<void,O, A1, A2, A3, A4, BA1>
    : public XorpMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>::M M;
    XorpSafeMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, bool B=true>
struct XorpMemberCallbackFactory4B1
{
    inline static XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpMemberCallbackFactory4B1<R, O, A1, A2, A3, A4, BA1, false>
{
    inline static XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1), BA1 ba1)
    {
	return new XorpMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory4B1<R,  O, A1, A2, A3, A4, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory4B1<R,  O, A1, A2, A3, A4, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpConstMemberCallback4B1 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1)  const;
    XorpConstMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpConstMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1)  const;
    XorpConstMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpConstSafeMemberCallback4B1
    : public XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>::M M;
    XorpConstSafeMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpConstSafeMemberCallback4B1<void,O, A1, A2, A3, A4, BA1>
    : public XorpConstMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>::M M;
    XorpConstSafeMemberCallback4B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B1<void, O, A1, A2, A3, A4, BA1>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory4B1
{
    inline static XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1>
struct XorpConstMemberCallbackFactory4B1<R, O, A1, A2, A3, A4, BA1, false>
{
    inline static XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback4B1<R, O, A1, A2, A3, A4, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory4B1<R,  const O, A1, A2, A3, A4, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory4B1<R,  const O, A1, A2, A3, A4, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpFunctionCallback4B2 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4, BA1, BA2);
    XorpFunctionCallback4B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpFunctionCallback4B2<void, A1, A2, A3, A4, BA1, BA2> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4, BA1, BA2);
    XorpFunctionCallback4B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B2<R, A1, A2, A3, A4, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpMemberCallback4B2 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2) ;
    XorpMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2) ;
    XorpMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpSafeMemberCallback4B2
    : public XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>::M M;
    XorpSafeMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpSafeMemberCallback4B2<void,O, A1, A2, A3, A4, BA1, BA2>
    : public XorpMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>::M M;
    XorpSafeMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory4B2
{
    inline static XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpMemberCallbackFactory4B2<R, O, A1, A2, A3, A4, BA1, BA2, false>
{
    inline static XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory4B2<R,  O, A1, A2, A3, A4, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory4B2<R,  O, A1, A2, A3, A4, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpConstMemberCallback4B2 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2)  const;
    XorpConstMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpConstMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2)  const;
    XorpConstMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpConstSafeMemberCallback4B2
    : public XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>::M M;
    XorpConstSafeMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpConstSafeMemberCallback4B2<void,O, A1, A2, A3, A4, BA1, BA2>
    : public XorpConstMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>::M M;
    XorpConstSafeMemberCallback4B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B2<void, O, A1, A2, A3, A4, BA1, BA2>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory4B2
{
    inline static XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2>
struct XorpConstMemberCallbackFactory4B2<R, O, A1, A2, A3, A4, BA1, BA2, false>
{
    inline static XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback4B2<R, O, A1, A2, A3, A4, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory4B2<R,  const O, A1, A2, A3, A4, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory4B2<R,  const O, A1, A2, A3, A4, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpFunctionCallback4B3 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4, BA1, BA2, BA3);
    XorpFunctionCallback4B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpFunctionCallback4B3<void, A1, A2, A3, A4, BA1, BA2, BA3> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4, BA1, BA2, BA3);
    XorpFunctionCallback4B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B3<R, A1, A2, A3, A4, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpMemberCallback4B3 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3) ;
    XorpMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3) ;
    XorpMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback4B3
    : public XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback4B3<void,O, A1, A2, A3, A4, BA1, BA2, BA3>
    : public XorpMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory4B3
{
    inline static XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory4B3<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory4B3<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback4B3 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3)  const;
    XorpConstMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3)  const;
    XorpConstMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback4B3
    : public XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback4B3<void,O, A1, A2, A3, A4, BA1, BA2, BA3>
    : public XorpConstMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback4B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B3<void, O, A1, A2, A3, A4, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory4B3
{
    inline static XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback4B3<R, O, A1, A2, A3, A4, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory4B3<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory4B3<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback4B4 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4, BA1, BA2, BA3, BA4);
    XorpFunctionCallback4B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback4B4<void, A1, A2, A3, A4, BA1, BA2, BA3, BA4> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4, BA1, BA2, BA3, BA4);
    XorpFunctionCallback4B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B4<R, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback4B4 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback4B4
    : public XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback4B4<void,O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory4B4
{
    inline static XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory4B4<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory4B4<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback4B4 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback4B4
    : public XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback4B4<void,O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback4B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B4<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory4B4
{
    inline static XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback4B4<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory4B4<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory4B4<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback4B5 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback4B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback4B5<void, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback4B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B5<R, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback4B5 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback4B5
    : public XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback4B5<void,O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory4B5
{
    inline static XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory4B5<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory4B5<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback4B5 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback4B5
    : public XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback4B5<void,O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback4B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B5<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory4B5
{
    inline static XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback4B5<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory4B5<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory4B5<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback4B6 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (*F)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback4B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback4<R, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = (*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback4B6<void, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (*F)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback4B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback4<void, A1, A2, A3, A4>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	(*_f)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 4 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback(R (*f)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr(new XorpFunctionCallback4B6<R, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback4B6 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback4B6
    : public XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback4B6<void,O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback4B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory4B6
{
    inline static XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory4B6<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 4 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory4B6<R,  O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback4B6 : public XorpCallback4<R, A1, A2, A3, A4> {
    typedef R (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback4<R, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback4<void, A1, A2, A3, A4> {
    typedef void (O::*M)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback4<void, A1, A2, A3, A4>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	((*_o).*_m)(a1, a2, a3, a4, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback4B6
    : public XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    R r = XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 4 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback4B6<void,O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback4B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback4B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4) {
	if (valid()) {
	    XorpConstMemberCallback4B6<void, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory4B6
{
    inline static XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback4B6<R, O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory4B6<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 4 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback4<R, A1, A2, A3, A4>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory4B6<R,  const O, A1, A2, A3, A4, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 5 late args
//

/**
 * @short Base class for callbacks with 5 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4, class A5>
struct XorpCallback5 {
    typedef ref_ptr<XorpCallback5> RefPtr;

    virtual ~XorpCallback5() {}
    virtual R dispatch(A1, A2, A3, A4, A5) = 0;
};

/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5>
struct XorpFunctionCallback5B0 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5);
    XorpFunctionCallback5B0(F f)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5>
struct XorpFunctionCallback5B0<void, A1, A2, A3, A4, A5> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5);
    XorpFunctionCallback5B0(F f)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5)) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B0<R, A1, A2, A3, A4, A5>(f));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5>
struct XorpMemberCallback5B0 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5) ;
    XorpMemberCallback5B0(O* o, M m)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5>
struct XorpMemberCallback5B0<void, O, A1, A2, A3, A4, A5>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5) ;
    XorpMemberCallback5B0(O* o, M m)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5>
struct XorpSafeMemberCallback5B0
    : public XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>::M M;
    XorpSafeMemberCallback5B0(O* o, M m)
	 : XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5>
struct XorpSafeMemberCallback5B0<void,O, A1, A2, A3, A4, A5>
    : public XorpMemberCallback5B0<void, O, A1, A2, A3, A4, A5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B0<void, O, A1, A2, A3, A4, A5>::M M;
    XorpSafeMemberCallback5B0(O* o, M m)
	 : XorpMemberCallback5B0<void, O, A1, A2, A3, A4, A5>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B0<void, O, A1, A2, A3, A4, A5>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, bool B=true>
struct XorpMemberCallbackFactory5B0
{
    inline static XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5))
    {
	return new XorpSafeMemberCallback5B0<R, O, A1, A2, A3, A4, A5>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5>
struct XorpMemberCallbackFactory5B0<R, O, A1, A2, A3, A4, A5, false>
{
    inline static XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5))
    {
	return new XorpMemberCallback5B0<R, O, A1, A2, A3, A4, A5>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5))
{
    return XorpMemberCallbackFactory5B0<R,  O, A1, A2, A3, A4, A5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5))
{
    return XorpMemberCallbackFactory5B0<R,  O, A1, A2, A3, A4, A5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5>
struct XorpConstMemberCallback5B0 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5)  const;
    XorpConstMemberCallback5B0(O* o, M m)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5>
struct XorpConstMemberCallback5B0<void, O, A1, A2, A3, A4, A5>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5)  const;
    XorpConstMemberCallback5B0(O* o, M m)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5>
struct XorpConstSafeMemberCallback5B0
    : public XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>::M M;
    XorpConstSafeMemberCallback5B0(O* o, M m)
	 : XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5>
struct XorpConstSafeMemberCallback5B0<void,O, A1, A2, A3, A4, A5>
    : public XorpConstMemberCallback5B0<void, O, A1, A2, A3, A4, A5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B0<void, O, A1, A2, A3, A4, A5>::M M;
    XorpConstSafeMemberCallback5B0(O* o, M m)
	 : XorpConstMemberCallback5B0<void, O, A1, A2, A3, A4, A5>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B0<void, O, A1, A2, A3, A4, A5>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, bool B=true>
struct XorpConstMemberCallbackFactory5B0
{
    inline static XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5) const)
    {
	return new XorpConstSafeMemberCallback5B0<R, O, A1, A2, A3, A4, A5>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5>
struct XorpConstMemberCallbackFactory5B0<R, O, A1, A2, A3, A4, A5, false>
{
    inline static XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5) const)
    {
	return new XorpConstMemberCallback5B0<R, O, A1, A2, A3, A4, A5>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5) const)
{
    return XorpConstMemberCallbackFactory5B0<R,  const O, A1, A2, A3, A4, A5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5) const)
{
    return XorpConstMemberCallbackFactory5B0<R,  const O, A1, A2, A3, A4, A5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpFunctionCallback5B1 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5, BA1);
    XorpFunctionCallback5B1(F f, BA1 ba1)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpFunctionCallback5B1<void, A1, A2, A3, A4, A5, BA1> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5, BA1);
    XorpFunctionCallback5B1(F f, BA1 ba1)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, BA1), BA1 ba1) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B1<R, A1, A2, A3, A4, A5, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpMemberCallback5B1 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1) ;
    XorpMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1) ;
    XorpMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpSafeMemberCallback5B1
    : public XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>::M M;
    XorpSafeMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpSafeMemberCallback5B1<void,O, A1, A2, A3, A4, A5, BA1>
    : public XorpMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>::M M;
    XorpSafeMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, bool B=true>
struct XorpMemberCallbackFactory5B1
{
    inline static XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpMemberCallbackFactory5B1<R, O, A1, A2, A3, A4, A5, BA1, false>
{
    inline static XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1), BA1 ba1)
    {
	return new XorpMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory5B1<R,  O, A1, A2, A3, A4, A5, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory5B1<R,  O, A1, A2, A3, A4, A5, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpConstMemberCallback5B1 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1)  const;
    XorpConstMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpConstMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1)  const;
    XorpConstMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpConstSafeMemberCallback5B1
    : public XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>::M M;
    XorpConstSafeMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpConstSafeMemberCallback5B1<void,O, A1, A2, A3, A4, A5, BA1>
    : public XorpConstMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>::M M;
    XorpConstSafeMemberCallback5B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B1<void, O, A1, A2, A3, A4, A5, BA1>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory5B1
{
    inline static XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1>
struct XorpConstMemberCallbackFactory5B1<R, O, A1, A2, A3, A4, A5, BA1, false>
{
    inline static XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback5B1<R, O, A1, A2, A3, A4, A5, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory5B1<R,  const O, A1, A2, A3, A4, A5, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory5B1<R,  const O, A1, A2, A3, A4, A5, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpFunctionCallback5B2 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5, BA1, BA2);
    XorpFunctionCallback5B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpFunctionCallback5B2<void, A1, A2, A3, A4, A5, BA1, BA2> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5, BA1, BA2);
    XorpFunctionCallback5B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B2<R, A1, A2, A3, A4, A5, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpMemberCallback5B2 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2) ;
    XorpMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2) ;
    XorpMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpSafeMemberCallback5B2
    : public XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>::M M;
    XorpSafeMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpSafeMemberCallback5B2<void,O, A1, A2, A3, A4, A5, BA1, BA2>
    : public XorpMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>::M M;
    XorpSafeMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory5B2
{
    inline static XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpMemberCallbackFactory5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2, false>
{
    inline static XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory5B2<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory5B2<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpConstMemberCallback5B2 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2)  const;
    XorpConstMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpConstMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2)  const;
    XorpConstMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpConstSafeMemberCallback5B2
    : public XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>::M M;
    XorpConstSafeMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpConstSafeMemberCallback5B2<void,O, A1, A2, A3, A4, A5, BA1, BA2>
    : public XorpConstMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>::M M;
    XorpConstSafeMemberCallback5B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B2<void, O, A1, A2, A3, A4, A5, BA1, BA2>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory5B2
{
    inline static XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2>
struct XorpConstMemberCallbackFactory5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2, false>
{
    inline static XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback5B2<R, O, A1, A2, A3, A4, A5, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory5B2<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory5B2<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpFunctionCallback5B3 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3);
    XorpFunctionCallback5B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpFunctionCallback5B3<void, A1, A2, A3, A4, A5, BA1, BA2, BA3> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3);
    XorpFunctionCallback5B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B3<R, A1, A2, A3, A4, A5, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpMemberCallback5B3 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3) ;
    XorpMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3) ;
    XorpMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback5B3
    : public XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback5B3<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3>
    : public XorpMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory5B3
{
    inline static XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory5B3<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory5B3<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback5B3 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3)  const;
    XorpConstMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3)  const;
    XorpConstMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback5B3
    : public XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback5B3<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3>
    : public XorpConstMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback5B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B3<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory5B3
{
    inline static XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback5B3<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory5B3<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory5B3<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback5B4 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4);
    XorpFunctionCallback5B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback5B4<void, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4);
    XorpFunctionCallback5B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B4<R, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback5B4 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback5B4
    : public XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback5B4<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory5B4
{
    inline static XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory5B4<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory5B4<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback5B4 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback5B4
    : public XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback5B4<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback5B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B4<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory5B4
{
    inline static XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback5B4<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory5B4<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory5B4<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback5B5 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback5B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback5B5<void, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback5B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B5<R, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback5B5 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback5B5
    : public XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback5B5<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory5B5
{
    inline static XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory5B5<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory5B5<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback5B5 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback5B5
    : public XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback5B5<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback5B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B5<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory5B5
{
    inline static XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback5B5<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory5B5<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory5B5<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback5B6 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback5B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = (*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback5B6<void, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (*F)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback5B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	(*_f)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 5 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr(new XorpFunctionCallback5B6<R, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback5B6 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback5B6
    : public XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback5B6<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback5B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory5B6
{
    inline static XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory5B6<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 5 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory5B6<R,  O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback5B6 : public XorpCallback5<R, A1, A2, A3, A4, A5> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback5<R, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback5<void, A1, A2, A3, A4, A5> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback5<void, A1, A2, A3, A4, A5>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	((*_o).*_m)(a1, a2, a3, a4, a5, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback5B6
    : public XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    R r = XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 5 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback5B6<void,O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback5B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback5B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5) {
	if (valid()) {
	    XorpConstMemberCallback5B6<void, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory5B6
{
    inline static XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback5B6<R, O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory5B6<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 5 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback5<R, A1, A2, A3, A4, A5>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory5B6<R,  const O, A1, A2, A3, A4, A5, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 6 late args
//

/**
 * @short Base class for callbacks with 6 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpCallback6 {
    typedef ref_ptr<XorpCallback6> RefPtr;

    virtual ~XorpCallback6() {}
    virtual R dispatch(A1, A2, A3, A4, A5, A6) = 0;
};

/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpFunctionCallback6B0 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6);
    XorpFunctionCallback6B0(F f)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpFunctionCallback6B0<void, A1, A2, A3, A4, A5, A6> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6);
    XorpFunctionCallback6B0(F f)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6)) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B0<R, A1, A2, A3, A4, A5, A6>(f));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpMemberCallback6B0 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6) ;
    XorpMemberCallback6B0(O* o, M m)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6) ;
    XorpMemberCallback6B0(O* o, M m)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpSafeMemberCallback6B0
    : public XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>::M M;
    XorpSafeMemberCallback6B0(O* o, M m)
	 : XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpSafeMemberCallback6B0<void,O, A1, A2, A3, A4, A5, A6>
    : public XorpMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>::M M;
    XorpSafeMemberCallback6B0(O* o, M m)
	 : XorpMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, bool B=true>
struct XorpMemberCallbackFactory6B0
{
    inline static XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6))
    {
	return new XorpSafeMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpMemberCallbackFactory6B0<R, O, A1, A2, A3, A4, A5, A6, false>
{
    inline static XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6))
    {
	return new XorpMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6))
{
    return XorpMemberCallbackFactory6B0<R,  O, A1, A2, A3, A4, A5, A6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6))
{
    return XorpMemberCallbackFactory6B0<R,  O, A1, A2, A3, A4, A5, A6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpConstMemberCallback6B0 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6)  const;
    XorpConstMemberCallback6B0(O* o, M m)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpConstMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6)  const;
    XorpConstMemberCallback6B0(O* o, M m)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpConstSafeMemberCallback6B0
    : public XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>::M M;
    XorpConstSafeMemberCallback6B0(O* o, M m)
	 : XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpConstSafeMemberCallback6B0<void,O, A1, A2, A3, A4, A5, A6>
    : public XorpConstMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>::M M;
    XorpConstSafeMemberCallback6B0(O* o, M m)
	 : XorpConstMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B0<void, O, A1, A2, A3, A4, A5, A6>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, bool B=true>
struct XorpConstMemberCallbackFactory6B0
{
    inline static XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6) const)
    {
	return new XorpConstSafeMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6>
struct XorpConstMemberCallbackFactory6B0<R, O, A1, A2, A3, A4, A5, A6, false>
{
    inline static XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6) const)
    {
	return new XorpConstMemberCallback6B0<R, O, A1, A2, A3, A4, A5, A6>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6) const)
{
    return XorpConstMemberCallbackFactory6B0<R,  const O, A1, A2, A3, A4, A5, A6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6) const)
{
    return XorpConstMemberCallbackFactory6B0<R,  const O, A1, A2, A3, A4, A5, A6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpFunctionCallback6B1 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, BA1);
    XorpFunctionCallback6B1(F f, BA1 ba1)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpFunctionCallback6B1<void, A1, A2, A3, A4, A5, A6, BA1> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, BA1);
    XorpFunctionCallback6B1(F f, BA1 ba1)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, BA1), BA1 ba1) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B1<R, A1, A2, A3, A4, A5, A6, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpMemberCallback6B1 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1) ;
    XorpMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1) ;
    XorpMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpSafeMemberCallback6B1
    : public XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>::M M;
    XorpSafeMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpSafeMemberCallback6B1<void,O, A1, A2, A3, A4, A5, A6, BA1>
    : public XorpMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>::M M;
    XorpSafeMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, bool B=true>
struct XorpMemberCallbackFactory6B1
{
    inline static XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpMemberCallbackFactory6B1<R, O, A1, A2, A3, A4, A5, A6, BA1, false>
{
    inline static XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1), BA1 ba1)
    {
	return new XorpMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory6B1<R,  O, A1, A2, A3, A4, A5, A6, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory6B1<R,  O, A1, A2, A3, A4, A5, A6, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpConstMemberCallback6B1 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1)  const;
    XorpConstMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpConstMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1)  const;
    XorpConstMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpConstSafeMemberCallback6B1
    : public XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>::M M;
    XorpConstSafeMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpConstSafeMemberCallback6B1<void,O, A1, A2, A3, A4, A5, A6, BA1>
    : public XorpConstMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>::M M;
    XorpConstSafeMemberCallback6B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B1<void, O, A1, A2, A3, A4, A5, A6, BA1>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory6B1
{
    inline static XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1>
struct XorpConstMemberCallbackFactory6B1<R, O, A1, A2, A3, A4, A5, A6, BA1, false>
{
    inline static XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback6B1<R, O, A1, A2, A3, A4, A5, A6, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory6B1<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory6B1<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpFunctionCallback6B2 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2);
    XorpFunctionCallback6B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpFunctionCallback6B2<void, A1, A2, A3, A4, A5, A6, BA1, BA2> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2);
    XorpFunctionCallback6B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B2<R, A1, A2, A3, A4, A5, A6, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpMemberCallback6B2 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2) ;
    XorpMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2) ;
    XorpMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpSafeMemberCallback6B2
    : public XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::M M;
    XorpSafeMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpSafeMemberCallback6B2<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2>
    : public XorpMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::M M;
    XorpSafeMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory6B2
{
    inline static XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpMemberCallbackFactory6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, false>
{
    inline static XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory6B2<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory6B2<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpConstMemberCallback6B2 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2)  const;
    XorpConstMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpConstMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2)  const;
    XorpConstMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpConstSafeMemberCallback6B2
    : public XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::M M;
    XorpConstSafeMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpConstSafeMemberCallback6B2<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2>
    : public XorpConstMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::M M;
    XorpConstSafeMemberCallback6B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B2<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory6B2
{
    inline static XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2>
struct XorpConstMemberCallbackFactory6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, false>
{
    inline static XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback6B2<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory6B2<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory6B2<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpFunctionCallback6B3 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3);
    XorpFunctionCallback6B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpFunctionCallback6B3<void, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3);
    XorpFunctionCallback6B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B3<R, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpMemberCallback6B3 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3) ;
    XorpMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3) ;
    XorpMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback6B3
    : public XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback6B3<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>
    : public XorpMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory6B3
{
    inline static XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory6B3<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory6B3<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback6B3 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3)  const;
    XorpConstMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3)  const;
    XorpConstMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback6B3
    : public XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback6B3<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>
    : public XorpConstMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback6B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B3<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory6B3
{
    inline static XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback6B3<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory6B3<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory6B3<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback6B4 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4);
    XorpFunctionCallback6B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback6B4<void, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4);
    XorpFunctionCallback6B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B4<R, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback6B4 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback6B4
    : public XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback6B4<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory6B4
{
    inline static XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory6B4<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory6B4<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback6B4 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback6B4
    : public XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback6B4<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback6B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B4<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory6B4
{
    inline static XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback6B4<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory6B4<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory6B4<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback6B5 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback6B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback6B5<void, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback6B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B5<R, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback6B5 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback6B5
    : public XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback6B5<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory6B5
{
    inline static XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory6B5<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory6B5<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback6B5 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback6B5
    : public XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback6B5<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback6B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B5<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory6B5
{
    inline static XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback6B5<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory6B5<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory6B5<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback6B6 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback6B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback6B6<void, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback6B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	(*_f)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 6 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr(new XorpFunctionCallback6B6<R, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback6B6 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback6B6
    : public XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback6B6<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback6B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory6B6
{
    inline static XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory6B6<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 6 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory6B6<R,  O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback6B6 : public XorpCallback6<R, A1, A2, A3, A4, A5, A6> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback6<R, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback6<void, A1, A2, A3, A4, A5, A6> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback6<void, A1, A2, A3, A4, A5, A6>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback6B6
    : public XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    R r = XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 6 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback6B6<void,O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback6B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback6B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6) {
	if (valid()) {
	    XorpConstMemberCallback6B6<void, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory6B6
{
    inline static XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback6B6<R, O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory6B6<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 6 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback6<R, A1, A2, A3, A4, A5, A6>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory6B6<R,  const O, A1, A2, A3, A4, A5, A6, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 7 late args
//

/**
 * @short Base class for callbacks with 7 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpCallback7 {
    typedef ref_ptr<XorpCallback7> RefPtr;

    virtual ~XorpCallback7() {}
    virtual R dispatch(A1, A2, A3, A4, A5, A6, A7) = 0;
};

/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpFunctionCallback7B0 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7);
    XorpFunctionCallback7B0(F f)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpFunctionCallback7B0<void, A1, A2, A3, A4, A5, A6, A7> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7);
    XorpFunctionCallback7B0(F f)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7)) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B0<R, A1, A2, A3, A4, A5, A6, A7>(f));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpMemberCallback7B0 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7) ;
    XorpMemberCallback7B0(O* o, M m)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7) ;
    XorpMemberCallback7B0(O* o, M m)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpSafeMemberCallback7B0
    : public XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>::M M;
    XorpSafeMemberCallback7B0(O* o, M m)
	 : XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpSafeMemberCallback7B0<void,O, A1, A2, A3, A4, A5, A6, A7>
    : public XorpMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>::M M;
    XorpSafeMemberCallback7B0(O* o, M m)
	 : XorpMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, bool B=true>
struct XorpMemberCallbackFactory7B0
{
    inline static XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7))
    {
	return new XorpSafeMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpMemberCallbackFactory7B0<R, O, A1, A2, A3, A4, A5, A6, A7, false>
{
    inline static XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7))
    {
	return new XorpMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7))
{
    return XorpMemberCallbackFactory7B0<R,  O, A1, A2, A3, A4, A5, A6, A7, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7))
{
    return XorpMemberCallbackFactory7B0<R,  O, A1, A2, A3, A4, A5, A6, A7, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpConstMemberCallback7B0 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7)  const;
    XorpConstMemberCallback7B0(O* o, M m)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpConstMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7)  const;
    XorpConstMemberCallback7B0(O* o, M m)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpConstSafeMemberCallback7B0
    : public XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>::M M;
    XorpConstSafeMemberCallback7B0(O* o, M m)
	 : XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpConstSafeMemberCallback7B0<void,O, A1, A2, A3, A4, A5, A6, A7>
    : public XorpConstMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>::M M;
    XorpConstSafeMemberCallback7B0(O* o, M m)
	 : XorpConstMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B0<void, O, A1, A2, A3, A4, A5, A6, A7>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, bool B=true>
struct XorpConstMemberCallbackFactory7B0
{
    inline static XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7) const)
    {
	return new XorpConstSafeMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7>
struct XorpConstMemberCallbackFactory7B0<R, O, A1, A2, A3, A4, A5, A6, A7, false>
{
    inline static XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7) const)
    {
	return new XorpConstMemberCallback7B0<R, O, A1, A2, A3, A4, A5, A6, A7>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7) const)
{
    return XorpConstMemberCallbackFactory7B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7) const)
{
    return XorpConstMemberCallbackFactory7B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpFunctionCallback7B1 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, BA1);
    XorpFunctionCallback7B1(F f, BA1 ba1)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpFunctionCallback7B1<void, A1, A2, A3, A4, A5, A6, A7, BA1> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, BA1);
    XorpFunctionCallback7B1(F f, BA1 ba1)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, BA1), BA1 ba1) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B1<R, A1, A2, A3, A4, A5, A6, A7, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpMemberCallback7B1 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1) ;
    XorpMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1) ;
    XorpMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpSafeMemberCallback7B1
    : public XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>::M M;
    XorpSafeMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpSafeMemberCallback7B1<void,O, A1, A2, A3, A4, A5, A6, A7, BA1>
    : public XorpMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>::M M;
    XorpSafeMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, bool B=true>
struct XorpMemberCallbackFactory7B1
{
    inline static XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpMemberCallbackFactory7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, false>
{
    inline static XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1), BA1 ba1)
    {
	return new XorpMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory7B1<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory7B1<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpConstMemberCallback7B1 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1)  const;
    XorpConstMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpConstMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1)  const;
    XorpConstMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpConstSafeMemberCallback7B1
    : public XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>::M M;
    XorpConstSafeMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpConstSafeMemberCallback7B1<void,O, A1, A2, A3, A4, A5, A6, A7, BA1>
    : public XorpConstMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>::M M;
    XorpConstSafeMemberCallback7B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B1<void, O, A1, A2, A3, A4, A5, A6, A7, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory7B1
{
    inline static XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1>
struct XorpConstMemberCallbackFactory7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, false>
{
    inline static XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback7B1<R, O, A1, A2, A3, A4, A5, A6, A7, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory7B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory7B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpFunctionCallback7B2 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2);
    XorpFunctionCallback7B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpFunctionCallback7B2<void, A1, A2, A3, A4, A5, A6, A7, BA1, BA2> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2);
    XorpFunctionCallback7B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B2<R, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpMemberCallback7B2 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2) ;
    XorpMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2) ;
    XorpMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpSafeMemberCallback7B2
    : public XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::M M;
    XorpSafeMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpSafeMemberCallback7B2<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>
    : public XorpMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::M M;
    XorpSafeMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory7B2
{
    inline static XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpMemberCallbackFactory7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, false>
{
    inline static XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory7B2<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory7B2<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpConstMemberCallback7B2 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2)  const;
    XorpConstMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpConstMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2)  const;
    XorpConstMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpConstSafeMemberCallback7B2
    : public XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::M M;
    XorpConstSafeMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpConstSafeMemberCallback7B2<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>
    : public XorpConstMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::M M;
    XorpConstSafeMemberCallback7B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B2<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory7B2
{
    inline static XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2>
struct XorpConstMemberCallbackFactory7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, false>
{
    inline static XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback7B2<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory7B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory7B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpFunctionCallback7B3 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3);
    XorpFunctionCallback7B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpFunctionCallback7B3<void, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3);
    XorpFunctionCallback7B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B3<R, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpMemberCallback7B3 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3) ;
    XorpMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3) ;
    XorpMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback7B3
    : public XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback7B3<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>
    : public XorpMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory7B3
{
    inline static XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory7B3<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory7B3<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback7B3 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3)  const;
    XorpConstMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3)  const;
    XorpConstMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback7B3
    : public XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback7B3<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>
    : public XorpConstMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback7B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B3<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory7B3
{
    inline static XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback7B3<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory7B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory7B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback7B4 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4);
    XorpFunctionCallback7B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback7B4<void, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4);
    XorpFunctionCallback7B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B4<R, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback7B4 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback7B4
    : public XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback7B4<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory7B4
{
    inline static XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory7B4<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory7B4<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback7B4 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback7B4
    : public XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback7B4<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback7B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B4<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory7B4
{
    inline static XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback7B4<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory7B4<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory7B4<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback7B5 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback7B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback7B5<void, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback7B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B5<R, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback7B5 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback7B5
    : public XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback7B5<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory7B5
{
    inline static XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory7B5<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory7B5<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback7B5 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback7B5
    : public XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback7B5<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback7B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B5<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory7B5
{
    inline static XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback7B5<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory7B5<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory7B5<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback7B6 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback7B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback7B6<void, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback7B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 7 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr(new XorpFunctionCallback7B6<R, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback7B6 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback7B6
    : public XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback7B6<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback7B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory7B6
{
    inline static XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory7B6<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 7 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory7B6<R,  O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback7B6 : public XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback7<void, A1, A2, A3, A4, A5, A6, A7>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback7B6
    : public XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    R r = XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 7 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback7B6<void,O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback7B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback7B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7) {
	if (valid()) {
	    XorpConstMemberCallback7B6<void, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory7B6
{
    inline static XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback7B6<R, O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory7B6<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 7 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback7<R, A1, A2, A3, A4, A5, A6, A7>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory7B6<R,  const O, A1, A2, A3, A4, A5, A6, A7, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 8 late args
//

/**
 * @short Base class for callbacks with 8 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpCallback8 {
    typedef ref_ptr<XorpCallback8> RefPtr;

    virtual ~XorpCallback8() {}
    virtual R dispatch(A1, A2, A3, A4, A5, A6, A7, A8) = 0;
};

/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpFunctionCallback8B0 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8);
    XorpFunctionCallback8B0(F f)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpFunctionCallback8B0<void, A1, A2, A3, A4, A5, A6, A7, A8> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8);
    XorpFunctionCallback8B0(F f)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8)) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B0<R, A1, A2, A3, A4, A5, A6, A7, A8>(f));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpMemberCallback8B0 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8) ;
    XorpMemberCallback8B0(O* o, M m)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8) ;
    XorpMemberCallback8B0(O* o, M m)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpSafeMemberCallback8B0
    : public XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>::M M;
    XorpSafeMemberCallback8B0(O* o, M m)
	 : XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpSafeMemberCallback8B0<void,O, A1, A2, A3, A4, A5, A6, A7, A8>
    : public XorpMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>::M M;
    XorpSafeMemberCallback8B0(O* o, M m)
	 : XorpMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, bool B=true>
struct XorpMemberCallbackFactory8B0
{
    inline static XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8))
    {
	return new XorpSafeMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpMemberCallbackFactory8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, false>
{
    inline static XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8))
    {
	return new XorpMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8))
{
    return XorpMemberCallbackFactory8B0<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8))
{
    return XorpMemberCallbackFactory8B0<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpConstMemberCallback8B0 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8)  const;
    XorpConstMemberCallback8B0(O* o, M m)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpConstMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8)  const;
    XorpConstMemberCallback8B0(O* o, M m)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpConstSafeMemberCallback8B0
    : public XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>::M M;
    XorpConstSafeMemberCallback8B0(O* o, M m)
	 : XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpConstSafeMemberCallback8B0<void,O, A1, A2, A3, A4, A5, A6, A7, A8>
    : public XorpConstMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>::M M;
    XorpConstSafeMemberCallback8B0(O* o, M m)
	 : XorpConstMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, bool B=true>
struct XorpConstMemberCallbackFactory8B0
{
    inline static XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8) const)
    {
	return new XorpConstSafeMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8>
struct XorpConstMemberCallbackFactory8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, false>
{
    inline static XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8) const)
    {
	return new XorpConstMemberCallback8B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8) const)
{
    return XorpConstMemberCallbackFactory8B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8) const)
{
    return XorpConstMemberCallbackFactory8B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpFunctionCallback8B1 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1);
    XorpFunctionCallback8B1(F f, BA1 ba1)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpFunctionCallback8B1<void, A1, A2, A3, A4, A5, A6, A7, A8, BA1> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1);
    XorpFunctionCallback8B1(F f, BA1 ba1)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, BA1), BA1 ba1) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B1<R, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpMemberCallback8B1 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1) ;
    XorpMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1) ;
    XorpMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpSafeMemberCallback8B1
    : public XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::M M;
    XorpSafeMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpSafeMemberCallback8B1<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>
    : public XorpMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::M M;
    XorpSafeMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, bool B=true>
struct XorpMemberCallbackFactory8B1
{
    inline static XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpMemberCallbackFactory8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, false>
{
    inline static XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1), BA1 ba1)
    {
	return new XorpMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory8B1<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory8B1<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpConstMemberCallback8B1 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1)  const;
    XorpConstMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpConstMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1)  const;
    XorpConstMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpConstSafeMemberCallback8B1
    : public XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::M M;
    XorpConstSafeMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpConstSafeMemberCallback8B1<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>
    : public XorpConstMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::M M;
    XorpConstSafeMemberCallback8B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory8B1
{
    inline static XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1>
struct XorpConstMemberCallbackFactory8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, false>
{
    inline static XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback8B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory8B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory8B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpFunctionCallback8B2 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2);
    XorpFunctionCallback8B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpFunctionCallback8B2<void, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2);
    XorpFunctionCallback8B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B2<R, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpMemberCallback8B2 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2) ;
    XorpMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2) ;
    XorpMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpSafeMemberCallback8B2
    : public XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::M M;
    XorpSafeMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpSafeMemberCallback8B2<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>
    : public XorpMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::M M;
    XorpSafeMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory8B2
{
    inline static XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpMemberCallbackFactory8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, false>
{
    inline static XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory8B2<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory8B2<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpConstMemberCallback8B2 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2)  const;
    XorpConstMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpConstMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2)  const;
    XorpConstMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpConstSafeMemberCallback8B2
    : public XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::M M;
    XorpConstSafeMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpConstSafeMemberCallback8B2<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>
    : public XorpConstMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::M M;
    XorpConstSafeMemberCallback8B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory8B2
{
    inline static XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2>
struct XorpConstMemberCallbackFactory8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, false>
{
    inline static XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback8B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory8B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory8B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpFunctionCallback8B3 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3);
    XorpFunctionCallback8B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpFunctionCallback8B3<void, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3);
    XorpFunctionCallback8B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B3<R, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpMemberCallback8B3 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3) ;
    XorpMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3) ;
    XorpMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback8B3
    : public XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback8B3<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>
    : public XorpMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory8B3
{
    inline static XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory8B3<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory8B3<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback8B3 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3)  const;
    XorpConstMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3)  const;
    XorpConstMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback8B3
    : public XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback8B3<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>
    : public XorpConstMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback8B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory8B3
{
    inline static XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback8B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory8B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory8B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback8B4 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4);
    XorpFunctionCallback8B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback8B4<void, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4);
    XorpFunctionCallback8B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B4<R, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback8B4 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback8B4
    : public XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback8B4<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory8B4
{
    inline static XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory8B4<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory8B4<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback8B4 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback8B4
    : public XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback8B4<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback8B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory8B4
{
    inline static XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback8B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory8B4<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory8B4<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback8B5 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback8B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback8B5<void, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback8B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B5<R, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback8B5 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback8B5
    : public XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback8B5<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory8B5
{
    inline static XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory8B5<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory8B5<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback8B5 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback8B5
    : public XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback8B5<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback8B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory8B5
{
    inline static XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback8B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory8B5<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory8B5<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback8B6 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback8B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback8B6<void, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback8B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 8 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr(new XorpFunctionCallback8B6<R, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback8B6 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback8B6
    : public XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback8B6<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback8B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory8B6
{
    inline static XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory8B6<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 8 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory8B6<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback8B6 : public XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback8<void, A1, A2, A3, A4, A5, A6, A7, A8>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback8B6
    : public XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    R r = XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 8 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback8B6<void,O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback8B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback8B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8) {
	if (valid()) {
	    XorpConstMemberCallback8B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory8B6
{
    inline static XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback8B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory8B6<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 8 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback8<R, A1, A2, A3, A4, A5, A6, A7, A8>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory8B6<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 9 late args
//

/**
 * @short Base class for callbacks with 9 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpCallback9 {
    typedef ref_ptr<XorpCallback9> RefPtr;

    virtual ~XorpCallback9() {}
    virtual R dispatch(A1, A2, A3, A4, A5, A6, A7, A8, A9) = 0;
};

/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpFunctionCallback9B0 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9);
    XorpFunctionCallback9B0(F f)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpFunctionCallback9B0<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9);
    XorpFunctionCallback9B0(F f)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9)) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B0<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(f));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpMemberCallback9B0 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9) ;
    XorpMemberCallback9B0(O* o, M m)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9) ;
    XorpMemberCallback9B0(O* o, M m)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpSafeMemberCallback9B0
    : public XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::M M;
    XorpSafeMemberCallback9B0(O* o, M m)
	 : XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpSafeMemberCallback9B0<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9>
    : public XorpMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::M M;
    XorpSafeMemberCallback9B0(O* o, M m)
	 : XorpMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, bool B=true>
struct XorpMemberCallbackFactory9B0
{
    inline static XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9))
    {
	return new XorpSafeMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpMemberCallbackFactory9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, false>
{
    inline static XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9))
    {
	return new XorpMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9))
{
    return XorpMemberCallbackFactory9B0<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9))
{
    return XorpMemberCallbackFactory9B0<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpConstMemberCallback9B0 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9)  const;
    XorpConstMemberCallback9B0(O* o, M m)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpConstMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9)  const;
    XorpConstMemberCallback9B0(O* o, M m)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpConstSafeMemberCallback9B0
    : public XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::M M;
    XorpConstSafeMemberCallback9B0(O* o, M m)
	 : XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpConstSafeMemberCallback9B0<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9>
    : public XorpConstMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::M M;
    XorpConstSafeMemberCallback9B0(O* o, M m)
	 : XorpConstMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, bool B=true>
struct XorpConstMemberCallbackFactory9B0
{
    inline static XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9) const)
    {
	return new XorpConstSafeMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9>
struct XorpConstMemberCallbackFactory9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, false>
{
    inline static XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9) const)
    {
	return new XorpConstMemberCallback9B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9) const)
{
    return XorpConstMemberCallbackFactory9B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9) const)
{
    return XorpConstMemberCallbackFactory9B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpFunctionCallback9B1 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1);
    XorpFunctionCallback9B1(F f, BA1 ba1)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpFunctionCallback9B1<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1);
    XorpFunctionCallback9B1(F f, BA1 ba1)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1), BA1 ba1) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B1<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpMemberCallback9B1 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1) ;
    XorpMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1) ;
    XorpMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpSafeMemberCallback9B1
    : public XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::M M;
    XorpSafeMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpSafeMemberCallback9B1<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>
    : public XorpMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::M M;
    XorpSafeMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, bool B=true>
struct XorpMemberCallbackFactory9B1
{
    inline static XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpMemberCallbackFactory9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, false>
{
    inline static XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1), BA1 ba1)
    {
	return new XorpMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory9B1<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory9B1<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpConstMemberCallback9B1 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1)  const;
    XorpConstMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpConstMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1)  const;
    XorpConstMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpConstSafeMemberCallback9B1
    : public XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::M M;
    XorpConstSafeMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpConstSafeMemberCallback9B1<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>
    : public XorpConstMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::M M;
    XorpConstSafeMemberCallback9B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory9B1
{
    inline static XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1>
struct XorpConstMemberCallbackFactory9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, false>
{
    inline static XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback9B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory9B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory9B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpFunctionCallback9B2 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2);
    XorpFunctionCallback9B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpFunctionCallback9B2<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2);
    XorpFunctionCallback9B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B2<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpMemberCallback9B2 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2) ;
    XorpMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2) ;
    XorpMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpSafeMemberCallback9B2
    : public XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::M M;
    XorpSafeMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpSafeMemberCallback9B2<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>
    : public XorpMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::M M;
    XorpSafeMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory9B2
{
    inline static XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpMemberCallbackFactory9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, false>
{
    inline static XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory9B2<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory9B2<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpConstMemberCallback9B2 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2)  const;
    XorpConstMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpConstMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2)  const;
    XorpConstMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpConstSafeMemberCallback9B2
    : public XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::M M;
    XorpConstSafeMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpConstSafeMemberCallback9B2<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>
    : public XorpConstMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::M M;
    XorpConstSafeMemberCallback9B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory9B2
{
    inline static XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2>
struct XorpConstMemberCallbackFactory9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, false>
{
    inline static XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback9B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory9B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory9B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpFunctionCallback9B3 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3);
    XorpFunctionCallback9B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpFunctionCallback9B3<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3);
    XorpFunctionCallback9B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B3<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpMemberCallback9B3 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3) ;
    XorpMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3) ;
    XorpMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback9B3
    : public XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback9B3<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>
    : public XorpMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory9B3
{
    inline static XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory9B3<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory9B3<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback9B3 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3)  const;
    XorpConstMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3)  const;
    XorpConstMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback9B3
    : public XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback9B3<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>
    : public XorpConstMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback9B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory9B3
{
    inline static XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback9B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory9B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory9B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback9B4 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4);
    XorpFunctionCallback9B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback9B4<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4);
    XorpFunctionCallback9B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B4<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback9B4 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback9B4
    : public XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback9B4<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory9B4
{
    inline static XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory9B4<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory9B4<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback9B4 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback9B4
    : public XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback9B4<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback9B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory9B4
{
    inline static XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback9B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory9B4<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpConstMemberCallbackFactory9B4<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback9B5 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback9B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpFunctionCallback9B5<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5);
    XorpFunctionCallback9B5(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 5 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B5<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(f, ba1, ba2, ba3, ba4, ba5));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback9B5 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5) ;
    XorpMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback9B5
    : public XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpSafeMemberCallback9B5<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>
    : public XorpMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpSafeMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpMemberCallbackFactory9B5
{
    inline static XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpSafeMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpMemberCallbackFactory9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory9B5<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpMemberCallbackFactory9B5<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback9B5 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5)  const;
    XorpConstMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback9B5
    : public XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B5() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 5 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstSafeMemberCallback9B5<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>
    : public XorpConstMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::M M;
    XorpConstSafeMemberCallback9B5(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
	 : XorpConstMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, m, ba1, ba2, ba3, ba4, ba5),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B5() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B5<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, bool B=true>
struct XorpConstMemberCallbackFactory9B5
{
    inline static XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstSafeMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5>
struct XorpConstMemberCallbackFactory9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, false>
{
    inline static XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
    {
	return new XorpConstMemberCallback9B5<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5>(o, p, ba1, ba2, ba3, ba4, ba5);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory9B5<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 5 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5)
{
    return XorpConstMemberCallbackFactory9B5<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5);
}


/**
 * @short Callback object for functions with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback9B6 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback9B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * @short Callback object for void functions with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpFunctionCallback9B6<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6> : public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6);
    XorpFunctionCallback9B6(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	: XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
    BA5 _ba5;
    BA6 _ba6;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 9 dispatch time arguments and 6 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6) {
    return typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr(new XorpFunctionCallback9B6<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(f, ba1, ba2, ba3, ba4, ba5, ba6));
}

/**
 * @short Callback object for member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback9B6 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6) ;
    XorpMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback9B6
    : public XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpSafeMemberCallback9B6<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpSafeMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback9B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpMemberCallbackFactory9B6
{
    inline static XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpSafeMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpMemberCallbackFactory9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory9B6<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 9 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpMemberCallbackFactory9B6<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


/**
 * @short Callback object for const member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback9B6 : public XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>
: public XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6)  const;
    XorpConstMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpCallback9<void, A1, A2, A3, A4, A5, A6, A7, A8, A9>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4), _ba5(ba5), _ba6(ba6) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, _ba1, _ba2, _ba3, _ba4, _ba5, _ba6);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
    BA5 _ba5;	// Bound argument
    BA6 _ba6;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback9B6
    : public XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B6() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    R r = XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 9 dispatch time
 * arguments and 6 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstSafeMemberCallback9B6<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>
    : public XorpConstMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::M M;
    XorpConstSafeMemberCallback9B6(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
	 : XorpConstMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, m, ba1, ba2, ba3, ba4, ba5, ba6),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback9B6() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9) {
	if (valid()) {
	    XorpConstMemberCallback9B6<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6, bool B=true>
struct XorpConstMemberCallbackFactory9B6
{
    inline static XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstSafeMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6>
struct XorpConstMemberCallbackFactory9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6, false>
{
    inline static XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
    {
	return new XorpConstMemberCallback9B6<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6>(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory9B6<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 9 dispatch time arguments and 6 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class BA1, class BA2, class BA3, class BA4, class BA5, class BA6> typename XorpCallback9<R, A1, A2, A3, A4, A5, A6, A7, A8, A9>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4, BA5 ba5, BA6 ba6)
{
    return XorpConstMemberCallbackFactory9B6<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, BA1, BA2, BA3, BA4, BA5, BA6, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4, ba5, ba6);
}


///////////////////////////////////////////////////////////////////////////////
//
// Code relating to callbacks with 10 late args
//

/**
 * @short Base class for callbacks with 10 dispatch time args.
 */
template<class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpCallback10 {
    typedef ref_ptr<XorpCallback10> RefPtr;

    virtual ~XorpCallback10() {}
    virtual R dispatch(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) = 0;
};

/**
 * @short Callback object for functions with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpFunctionCallback10B0 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10);
    XorpFunctionCallback10B0(F f)
	: XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	return r;
    }
protected:
    F   _f;
};


/**
 * @short Callback object for void functions with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpFunctionCallback10B0<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> : public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10);
    XorpFunctionCallback10B0(F f)
	: XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
    }
protected:
    F   _f;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 10 dispatch time arguments and 0 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)) {
    return typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr(new XorpFunctionCallback10B0<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(f));
}

/**
 * @short Callback object for member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpMemberCallback10B0 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) ;
    XorpMemberCallback10B0(O* o, M m)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) ;
    XorpMemberCallback10B0(O* o, M m)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for safe member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpSafeMemberCallback10B0
    : public XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::M M;
    XorpSafeMemberCallback10B0(O* o, M m)
	 : XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpSafeMemberCallback10B0<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>
    : public XorpMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::M M;
    XorpSafeMemberCallback10B0(O* o, M m)
	 : XorpMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, bool B=true>
struct XorpMemberCallbackFactory10B0
{
    inline static XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10))
    {
	return new XorpSafeMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpMemberCallbackFactory10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, false>
{
    inline static XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10))
    {
	return new XorpMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10))
{
    return XorpMemberCallbackFactory10B0<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10))
{
    return XorpMemberCallbackFactory10B0<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for const member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpConstMemberCallback10B0 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)  const;
    XorpConstMemberCallback10B0(O* o, M m)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for void const member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpConstMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10)  const;
    XorpConstMemberCallback10B0(O* o, M m)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
};

/**
 * @short Callback object for const safe member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpConstSafeMemberCallback10B0
    : public XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::M M;
    XorpConstSafeMemberCallback10B0(O* o, M m)
	 : XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B0() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 10 dispatch time
 * arguments and 0 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpConstSafeMemberCallback10B0<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>
    : public XorpConstMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::M M;
    XorpConstSafeMemberCallback10B0(O* o, M m)
	 : XorpConstMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, m),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B0() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpConstMemberCallback10B0<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, bool B=true>
struct XorpConstMemberCallbackFactory10B0
{
    inline static XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) const)
    {
	return new XorpConstSafeMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, p);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10>
struct XorpConstMemberCallbackFactory10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, false>
{
    inline static XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) const)
    {
	return new XorpConstMemberCallback10B0<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(o, p);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) const)
{
    return XorpConstMemberCallbackFactory10B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 0 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10) const)
{
    return XorpConstMemberCallbackFactory10B0<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p);
}


/**
 * @short Callback object for functions with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpFunctionCallback10B1 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1);
    XorpFunctionCallback10B1(F f, BA1 ba1)
	: XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * @short Callback object for void functions with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpFunctionCallback10B1<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1> : public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1);
    XorpFunctionCallback10B1(F f, BA1 ba1)
	: XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1);
    }
protected:
    F   _f;
    BA1 _ba1;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 10 dispatch time arguments and 1 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1), BA1 ba1) {
    return typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr(new XorpFunctionCallback10B1<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(f, ba1));
}

/**
 * @short Callback object for member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpMemberCallback10B1 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1) ;
    XorpMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1) ;
    XorpMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpSafeMemberCallback10B1
    : public XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::M M;
    XorpSafeMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpSafeMemberCallback10B1<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>
    : public XorpMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::M M;
    XorpSafeMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, bool B=true>
struct XorpMemberCallbackFactory10B1
{
    inline static XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1), BA1 ba1)
    {
	return new XorpSafeMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpMemberCallbackFactory10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, false>
{
    inline static XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1), BA1 ba1)
    {
	return new XorpMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory10B1<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1), BA1 ba1)
{
    return XorpMemberCallbackFactory10B1<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for const member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpConstMemberCallback10B1 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1)  const;
    XorpConstMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpConstMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1)  const;
    XorpConstMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpConstSafeMemberCallback10B1
    : public XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::M M;
    XorpConstSafeMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B1() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 10 dispatch time
 * arguments and 1 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpConstSafeMemberCallback10B1<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>
    : public XorpConstMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::M M;
    XorpConstSafeMemberCallback10B1(O* o, M m, BA1 ba1)
	 : XorpConstMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, m, ba1),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B1() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpConstMemberCallback10B1<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, bool B=true>
struct XorpConstMemberCallbackFactory10B1
{
    inline static XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1) const, BA1 ba1)
    {
	return new XorpConstSafeMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, p, ba1);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1>
struct XorpConstMemberCallbackFactory10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, false>
{
    inline static XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1) const, BA1 ba1)
    {
	return new XorpConstMemberCallback10B1<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1>(o, p, ba1);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory10B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 1 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1) const, BA1 ba1)
{
    return XorpConstMemberCallbackFactory10B1<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1);
}


/**
 * @short Callback object for functions with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpFunctionCallback10B2 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2);
    XorpFunctionCallback10B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * @short Callback object for void functions with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpFunctionCallback10B2<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2> : public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2);
    XorpFunctionCallback10B2(F f, BA1 ba1, BA2 ba2)
	: XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1), _ba2(ba2)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 10 dispatch time arguments and 2 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2), BA1 ba1, BA2 ba2) {
    return typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr(new XorpFunctionCallback10B2<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(f, ba1, ba2));
}

/**
 * @short Callback object for member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpMemberCallback10B2 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2) ;
    XorpMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2) ;
    XorpMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpSafeMemberCallback10B2
    : public XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::M M;
    XorpSafeMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpSafeMemberCallback10B2<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>
    : public XorpMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::M M;
    XorpSafeMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, bool B=true>
struct XorpMemberCallbackFactory10B2
{
    inline static XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpSafeMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpMemberCallbackFactory10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, false>
{
    inline static XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2), BA1 ba1, BA2 ba2)
    {
	return new XorpMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory10B2<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2), BA1 ba1, BA2 ba2)
{
    return XorpMemberCallbackFactory10B2<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for const member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpConstMemberCallback10B2 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2)  const;
    XorpConstMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpConstMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2)  const;
    XorpConstMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpConstSafeMemberCallback10B2
    : public XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::M M;
    XorpConstSafeMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B2() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 10 dispatch time
 * arguments and 2 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpConstSafeMemberCallback10B2<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>
    : public XorpConstMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::M M;
    XorpConstSafeMemberCallback10B2(O* o, M m, BA1 ba1, BA2 ba2)
	 : XorpConstMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, m, ba1, ba2),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B2() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpConstMemberCallback10B2<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, bool B=true>
struct XorpConstMemberCallbackFactory10B2
{
    inline static XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstSafeMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, p, ba1, ba2);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2>
struct XorpConstMemberCallbackFactory10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, false>
{
    inline static XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2) const, BA1 ba1, BA2 ba2)
    {
	return new XorpConstMemberCallback10B2<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2>(o, p, ba1, ba2);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory10B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 2 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2) const, BA1 ba1, BA2 ba2)
{
    return XorpConstMemberCallbackFactory10B2<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2);
}


/**
 * @short Callback object for functions with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpFunctionCallback10B3 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3);
    XorpFunctionCallback10B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * @short Callback object for void functions with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpFunctionCallback10B3<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3> : public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3);
    XorpFunctionCallback10B3(F f, BA1 ba1, BA2 ba2, BA3 ba3)
	: XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 10 dispatch time arguments and 3 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3) {
    return typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr(new XorpFunctionCallback10B3<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(f, ba1, ba2, ba3));
}

/**
 * @short Callback object for member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpMemberCallback10B3 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3) ;
    XorpMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3) ;
    XorpMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback10B3
    : public XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpSafeMemberCallback10B3<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>
    : public XorpMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::M M;
    XorpSafeMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, bool B=true>
struct XorpMemberCallbackFactory10B3
{
    inline static XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpSafeMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpMemberCallbackFactory10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, false>
{
    inline static XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory10B3<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3), BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpMemberCallbackFactory10B3<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for const member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback10B3 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3)  const;
    XorpConstMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpConstMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3)  const;
    XorpConstMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback10B3
    : public XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B3() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 10 dispatch time
 * arguments and 3 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpConstSafeMemberCallback10B3<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>
    : public XorpConstMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::M M;
    XorpConstSafeMemberCallback10B3(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3)
	 : XorpConstMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, m, ba1, ba2, ba3),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B3() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpConstMemberCallback10B3<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, bool B=true>
struct XorpConstMemberCallbackFactory10B3
{
    inline static XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstSafeMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3>
struct XorpConstMemberCallbackFactory10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, false>
{
    inline static XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
    {
	return new XorpConstMemberCallback10B3<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3>(o, p, ba1, ba2, ba3);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory10B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3);
}

/**
 * Factory function that creates a callback object targetted at a
 * const member function with 10 dispatch time arguments and 3 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( const O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3) const, BA1 ba1, BA2 ba2, BA3 ba3)
{
    return XorpConstMemberCallbackFactory10B3<R,  const O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3);
}


/**
 * @short Callback object for functions with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback10B4 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4);
    XorpFunctionCallback10B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = (*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * @short Callback object for void functions with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpFunctionCallback10B4<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4> : public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (*F)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4);
    XorpFunctionCallback10B4(F f, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	: XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _f(f), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4)
    {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	(*_f)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    F   _f;
    BA1 _ba1;
    BA2 _ba2;
    BA3 _ba3;
    BA4 _ba4;
};


/**
 * Factory function that creates a callback object targetted at a
 * function with 10 dispatch time arguments and 4 bound arguments.
 */
template <class R, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback(R (*f)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4) {
    return typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr(new XorpFunctionCallback10B4<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(f, ba1, ba2, ba3, ba4));
}

/**
 * @short Callback object for member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback10B4 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4) ;
    XorpMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for safe member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback10B4
    : public XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void safe member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpSafeMemberCallback10B4<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>
    : public XorpMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::M M;
    XorpSafeMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpSafeMemberCallback10B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpMemberCallbackFactory10B4
{
    inline static XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpSafeMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpMemberCallbackFactory10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4, false>
{
    inline static XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    };
};

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory10B4<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(o, p, ba1, ba2, ba3, ba4);
}

/**
 * Factory function that creates a callback object targetted at a
 * member function with 10 dispatch time arguments and 4 bound arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4> typename XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>::RefPtr
callback( O& o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4), BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
{
    return XorpMemberCallbackFactory10B4<R,  O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4, BaseAndDerived<CallbackSafeObject, O>::True>::make(&o, p, ba1, ba2, ba3, ba4);
}


/**
 * @short Callback object for const member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback10B4 : public XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef R (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback10<R, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	R r = ((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3, _ba4);
	return r;
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for void const member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>
: public XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10> {
    typedef void (O::*M)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4)  const;
    XorpConstMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpCallback10<void, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10>(),
	  _o(o), _m(m), _ba1(ba1), _ba2(ba2), _ba3(ba3), _ba4(ba4) {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	((*_o).*_m)(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10, _ba1, _ba2, _ba3, _ba4);
    }
protected:
    O*	_o;	// Callback's target object
    M	_m;	// Callback's target method
    BA1 _ba1;	// Bound argument
    BA2 _ba2;	// Bound argument
    BA3 _ba3;	// Bound argument
    BA4 _ba4;	// Bound argument
};

/**
 * @short Callback object for const safe member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback10B4
    : public XorpConstMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B4() {}
    R dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    R r = XorpConstMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	    return r;
	}
    }
};

/**
 * @short Callback object for void const safe member methods with 10 dispatch time
 * arguments and 4 bound (stored) arguments.
 */
template <class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpConstSafeMemberCallback10B4<void,O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>
    : public XorpConstMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>,
      public SafeCallbackBase {
    typedef typename XorpConstMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::M M;
    XorpConstSafeMemberCallback10B4(O* o, M m, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
	 : XorpConstMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, m, ba1, ba2, ba3, ba4),
	   SafeCallbackBase(o) {}
    ~XorpConstSafeMemberCallback10B4() {}
    void dispatch(A1 a1, A2 a2, A3 a3, A4 a4, A5 a5, A6 a6, A7 a7, A8 a8, A9 a9, A10 a10) {
	if (valid()) {
	    XorpConstMemberCallback10B4<void, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>::dispatch(a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	}
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4, bool B=true>
struct XorpConstMemberCallbackFactory10B4
{
    inline static XorpConstMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstSafeMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>(o, p, ba1, ba2, ba3, ba4);
    }
};

template <class R, class O, class A1, class A2, class A3, class A4, class A5, class A6, class A7, class A8, class A9, class A10, class BA1, class BA2, class BA3, class BA4>
struct XorpConstMemberCallbackFactory10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4, false>
{
    inline static XorpConstMemberCallback10B4<R, O, A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4>*
    make(O* o, R (O::*p)(A1, A2, A3, A4, A5, A6, A7, A8, A9, A10, BA1, BA2, BA3, BA4) const, BA1 ba1, BA2 ba2, BA3 ba3, BA4 ba4)
    {
	return new XorpConstMemberCallback10