/****************************************************************************
*  Copyright (C) 2001 by Leo Khramov
*  email:     leo@xnc.dubna.su
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
 ****************************************************************************/

#include "xdndclass.h"

//Bitmaps and Pixmaps
#include "dnd1.xbm"
#include "dnd1m.xbm"
#include "dnd2.xbm"
#include "dnd2m.xbm"

DndManager* default_dnd_man=0;

/////////////////////////////////DndObject/////////////////////////////////////
int DndObject::dnd_enter(XEvent*, int x, int y)
{
  xncdprintf(("Abstract call to DndObject::dnd_enter(XEvent*, %d, %d)\n",x,y));
  return 1;
}

int DndObject::dnd_leave(XEvent*, int x, int y)
{
  xncdprintf(("Abstract call to DndObject::dnd_leave(XEvent*, %d, %d)\n",x,y));
  return 1;
}

int DndObject::dnd_position(XEvent*, int x, int y)
{
  xncdprintf(("Abstract call to DndObject::dnd_position(XEvent*, %d, %d)\n",x,y));
  return 1;
}

//you MUST DELETE DndData object after use!
int DndObject::dnd_drop(DndData* d, int x, int y,Atom action)
{
  xncdprintf(("Abstract call to DndObject::dnd_drop('%s', %d, %d)\n",d->data,x,y));
  dndman->debug_atom(action);
  delete d;
  return 1;
}

int DndObject::dnd_drag_now(DndData* dat)
{
  
  return dndman->dragging(this,dat);
}

//////////////////////////////////DndData//////////////////////////////////////

int  DndData::calc_entries(char* dat, char sym)
{
  int i=0;
  while(*dat)
  {
    if(*dat==sym)
      i++;
    dat++;
  }
  return i;
}

void DndData::add_file_entry(char *dir, char* fname)
{
  while(*dir)
  {
    data[dat_idx++]=*dir++;
  }
  if(*(dir-1)!='/')
    data[dat_idx++]='/';
  while(*fname)
  {
    data[dat_idx++]=*fname++;
  }
  data[dat_idx++]='\n';
  data[dat_idx]=0;
}

void DndData::free_file_list()
{
  int i;
  if(!file_list)
    return;
  for(i=0;i<file_listlen;i++)
    if(file_list[i])
      delete file_list[i];
  delete file_list;
  file_list=0;
}

int DndData::break_to_filelist()
{
  int len,i=0,j;
  free_file_list();
  len=calc_entries(data,'\n')+1;
  file_list=new char* [len];
  file_listlen=0;
  while(data[i])
  {
    j=0;
    file_list[file_listlen]=new char[L_MAXPATH];
    while(data[i] && data[i]!='\n' && data[i]!='\r')
      file_list[file_listlen][j++]=data[i++];
    file_list[file_listlen][j]=0;
    while(data[i]=='\n' || data[i]=='\r')
      i++;
    file_listlen++;
  }
  return file_listlen;
}

void DndData::show_filelist()
{
  int i;
  for(i=0;i<file_listlen;i++)
  {
    xncdprintf(("Filelist[%d]: [%s]\n",i,file_list[i]));
  }
}

//////////////////////////////////DndWin///////////////////////////////////////

DndWin::DndWin(DndManager* iman,Window iw, DndObject* o, int ix, int iy, uint il, uint ih)
{
  int i;

  obj=o;
  w=iw;
  man=iman;
  rx=ix;ry=iy;
  l=il;h=ih;
  typelist=new Atom[MAX_TYPES_PER_ITEM];
  for(i=0;i<MAX_TYPES_PER_ITEM;i++)
    typelist[i]=0;
}

DndWin::DndWin(DndManager* iman,Window iw, int ix, int iy, uint il, uint ih)
{
  int i;

  obj=0;
  w=iw;
  man=iman;
  rx=ix;ry=iy;
  l=il;h=ih;
  typelist=new Atom[MAX_TYPES_PER_ITEM];
  for(i=0;i<MAX_TYPES_PER_ITEM;i++)
    typelist[i]=0;
}

void DndWin::add_type(char* name)
{
  int i;
  Atom a=XInternAtom(man->display(),name,False);
  for(i=0;typelist[i];i++)
    if(typelist[i]==a)
      return;
  typelist[i]=a;
  xncdprintf(("Added to DndWin:\n"));
  man->debug_atom(a);
  man->add_in_type(a);
}

int DndWin::check_type(Atom t)
{
  int i;
  xncdprintf(("DndWin check_type:\n"));
  man->debug_atom(t);
  for(i=0;typelist[i];i++)
    if(typelist[i]==t)
      return 1;
  return 0;
}


///////////////////////////////DndManager//////////////////////////////////////////////

DndManager::DndManager(Display* d,Window irootw)
{
  int i;

  disp=d;
  rootw=irootw;
  winar=new DndWin* [MAX_WINS];
  our_in_typelist=new Atom[MAX_TYPES];

  for(i=0;i<MAX_WINS;i++)
    winar[i]=0;

  for(i=0;i<MAX_TYPES;i++)
    our_in_typelist[i]=0;

  version=XDND_VERSION;
  typelist=0;
  curtype=-1;
  dnd_data=0;
  curaction=0;
  curwin=0;

  XdndAware =             XInternAtom (disp, "XdndAware", False);
  XdndSelection =         XInternAtom (disp, "XdndSelection", False);
  XdndEnter =             XInternAtom (disp, "XdndEnter", False);
  XdndLeave =             XInternAtom (disp, "XdndLeave", False);
  XdndPosition =          XInternAtom (disp, "XdndPosition", False);
  XdndDrop =              XInternAtom (disp, "XdndDrop", False);
  XdndFinished =          XInternAtom (disp, "XdndFinished", False);
  XdndStatus =            XInternAtom (disp, "XdndStatus", False);
  XdndActionCopy =        XInternAtom (disp, "XdndActionCopy", False);
  XdndActionMove =        XInternAtom (disp, "XdndActionMove", False);
  XdndActionLink =        XInternAtom (disp, "XdndActionLink", False);
  XdndActionAsk =         XInternAtom (disp, "XdndActionAsk", False);
  XdndActionPrivate =     XInternAtom (disp, "XdndActionPrivate", False);
  XdndTypeList =          XInternAtom (disp, "XdndTypeList", False);
  XdndActionList =        XInternAtom (disp, "XdndActionList", False);
  XdndActionDescription = XInternAtom (disp, "XdndActionDescription", False);  
  dnd_data_atom =         XInternAtom (disp, "dnd_data_atom", False);  

  make_dnd_aware();

  create_dnd_cursors();

  xncdprintf(("DndManager initialized. DndEnter is 0x%x\n",XdndEnter));
  //  add_in_type("text/plain");
  //add_in_type("text/uri-list");
}

void DndManager::create_dnd_cursors()
{
  Pixmap dnd1pix,dnd1mpix;
  XColor xc1, xc2;
  xc1.pixel = BlackPixel(disp,DefaultScreen(disp));
  xc1.red = xc1.green = xc1.blue = 0l;
  xc1.flags = DoRed | DoBlue | DoGreen;
  xc2.pixel = WhitePixel(disp,DefaultScreen(disp));
  xc2.red = 65535l;
  xc2.green = 65535l;
  xc2.blue = 65535l;
  xc2.flags = DoRed | DoBlue | DoGreen;
  dnd1pix = XCreateBitmapFromData(disp, rootw, (char *)dnd1_bits, dnd1_width, dnd1_height);
  dnd1mpix = XCreateBitmapFromData(disp, rootw, (char *)dnd1m_bits, dnd1m_width, dnd1m_height);
  curs_one = XCreatePixmapCursor(disp, dnd1pix, dnd1mpix, &xc1, &xc2, 11, 8);
  XFreePixmap(disp, dnd1pix);
  XFreePixmap(disp, dnd1mpix);
  dnd1pix = XCreateBitmapFromData(disp, rootw, (char *)dnd2_bits, dnd2_width, dnd2_height);
  dnd1mpix = XCreateBitmapFromData(disp, rootw, (char *)dnd2m_bits, dnd2m_width, dnd2m_height);
  curs_many = XCreatePixmapCursor(disp, dnd1pix, dnd1mpix, &xc1, &xc2, 12, 13);
  XFreePixmap(disp, dnd1pix);
  XFreePixmap(disp, dnd1mpix);
}


void DndManager::make_dnd_aware()
{
  XWindowAttributes attr;
  XGetWindowAttributes(disp,rootw,&attr);
  xncdprintf(("Root window input mask: %x %x\n",attr.your_event_mask, 
	      attr.all_event_masks));
  XSelectInput(disp,rootw,attr.your_event_mask | EnterWindowMask 
	       | LeaveWindowMask | StructureNotifyMask);
  XChangeProperty (disp, rootw, XdndAware, XA_ATOM, 32, PropModeReplace,
		   (unsigned char *) &version, 1);
  xncdprintf(("Setting XDndAware property to 0x%x\n",rootw));
}

DndManager::~DndManager()
{
  int i;
  for(i=0;i<MAX_WINS;i++)
    if(winar[i])
      delete winar[i];
  delete winar;
  winar=0;
  if(typelist)
    delete typelist;
  if(dnd_data)
    delete dnd_data;
}

int DndManager::find_win_idx(Window w)
{
  int i;
  for(i=0;i<MAX_WINS;i++)
    if(winar[i] && winar[i]->w==w)
      return i;
  return -1;
}

///global DndManager::find_dnd_win(rx,ry,action)
/// Find best dnd window, that can receive our drop
/// Think that top window on stacking order is at the end of our list

DndWin* DndManager::find_dnd_win(int rx, int ry, Atom action)
{
  int i;
  Atom a=None;
  DndWin* last=0;
  if(curtype!=-1)
    a=our_in_typelist[curtype];
  for(i=0;winar[i] && i<MAX_WINS;i++)
    if(rx>=winar[i]->rx && ry>=winar[i]->ry && 
       winar[i]->rx+winar[i]->l>rx && winar[i]->ry+winar[i]->h>ry)
    {
      xncdprintf(("find_dnd_win: found entry %d\n",i));
      if(winar[i]->check_type(a))
      {
	xncdprintf(("check_type - ok\n"));
	last=winar[i];
      }
       else
       {
	 xncdprintf(("check_type - fail\n"));
	 last=0;
       }
    }
  return last;
}

int DndManager::new_idx()
{
  int i;
  for(i=0;i<MAX_WINS;i++)
    if(winar[i]==0)
      return i;
  return -1;
}

DndWin* DndManager::add_dnd_win(Window w)
{
  int idx;
  int x,y;
  uint l,h;
  if((idx=find_win_idx(w))!=-1)
    return winar[idx];
  idx=new_idx();
  trace_win_geom(w,x,y,l,h);
  winar[idx]=new DndWin(this,w,x,y,l,h);
  return winar[idx];
}

DndWin* DndManager::add_dnd_object(DndObject* o, Window w)
{
  int idx;
  int x,y;
  uint l,h;
  if((idx=find_win_idx(w))!=-1)
    return winar[idx];
  idx=new_idx();
  trace_win_geom(w,x,y,l,h);
  winar[idx]=new DndWin(this,w,o,x,y,l,h);
  o->dnd_init(winar[idx],this);
  return winar[idx];
}

void DndManager::del_dnd_win(Window w)
{
  int i,j;
  for(i=0;i<MAX_WINS;i++)
    if(winar[i] && winar[i]->w==w)
    {
      delete winar[i];
      for(j=i+1;j<MAX_WINS;j++)
	winar[j-1]=winar[j];
      winar[MAX_WINS-1]=0;
      xncdprintf(("DndObject deleted from DndManager list\n"));
      return;
    }
}

int DndManager::move_to_top(Window w)
{
  int i;
  DndWin* tmp;
  if((i=find_win_idx(w))==-1)
    return 0;
  tmp=winar[i];
  for(;i<MAX_WINS-1;i++)
  {
    winar[i]=winar[i+1];
    if(winar[i]==0)
      break;
  }
  winar[i]=tmp;
  return 1;
}

//Translate window coords into coords on root window
void DndManager::trace_win_geom(Window w, int& x, int& y, uint& l, uint& h)
{
  Window screenw=DefaultRootWindow(disp);
  Window subw;
  uint bw,dep;
  XGetGeometry(disp,w,&subw,&x,&y,&l,&h,&bw,&dep);
  XTranslateCoordinates(disp,w,screenw,0,0,&x,&y,&subw);
  xncdprintf(("Traced coords for 0x%x: %d,%d %dx%d\n",w,x,y,l,h));
}

//Translate global coords to window coords 
void DndManager::global2win_coords(Window w, int fromx, int fromy, int& x, int& y)
{
  Window screenw=DefaultRootWindow(disp);
  Window subw;
  XTranslateCoordinates(disp,screenw,w,fromx,fromy,&x,&y,&subw);
}


void DndManager::update_geoms()
{
  int i;
  for(i=0;i<MAX_WINS;i++)
    if(winar[i])
      trace_win_geom(winar[i]->w,winar[i]->rx, winar[i]->ry, 
		     winar[i]->l, winar[i]->h);
}

//Look on event and process it, if event is for us -> return 1
//If event is not processed then return 0
int DndManager::process_event(XEvent* ev)
{
  switch(ev->type)
  {
  case ClientMessage:
    //XdndEnter received
    if(XDND_MESTYPE(ev)==XdndEnter)
    {
      xncdprintf(("XdndEnter for 0x%x: VER[%d] SRCWIN[0x%x] 3TYPES[%d]\n",ev->xany.window,
		  XDND_ENTER_VERSION(ev),XDND_ENTER_SOURCE_WIN(ev),XDND_ENTER_THREE_TYPES(ev)));
      process_XdndEnter(ev,XDND_ENTER_SOURCE_WIN(ev),XDND_ENTER_VERSION(ev));
      return 1;
    } 

    //XdndPosition received
    if(XDND_MESTYPE(ev)==XdndPosition)
    {
      xncdprintf(("XdndPosition:\n"));
      process_XdndPosition(ev,XDND_POSITION_SOURCE_WIN(ev));
      return 1;
    }
    //XdndPosition received
    if(XDND_MESTYPE(ev)==XdndLeave)
    {
      xncdprintf(("XdndLeave: cleanups\n"));
      process_XdndLeave(ev);
      return 1;
    }
    if(XDND_MESTYPE(ev)==XdndDrop)
    {
      xncdprintf(("XdndDrop: retreiving the data\n"));
      process_XdndDrop(ev);
      return 1;
    }
    break;
    
  case SelectionNotify:
    if(ev->xany.window==rootw && ev->xselection.property==dnd_data_atom)
    {
      data_ready_get_it();
    }
    break;
  }
  return 0;
}


//Processing XDndEnter message
int DndManager::process_XdndEnter(XEvent* ev, Window srcw, int vers)
{
  if(vers>version)
  {
    xncdprintf(("Source has newer version that we, reject XdndEnter\n"));
    return 0;
  }
  srcwin=srcw;
  if(XDND_ENTER_THREE_TYPES(ev))
  {
    xncdprintf(("Source provide three types only\n"));
    build_threetype_list(ev,srcw);
  } else
  {
    xncdprintf(("Source provide more than three types\n"));
    build_bigtype_list(ev,srcw);
  }
  if((curtype=check_type_list(ev))==-1)
    return 0;

  return 1;
}


//Processing XDndPosition message
int DndManager::process_XdndPosition(XEvent* ev, Window srcw)
{
  DndWin* pwin;
  int rx=XDND_POSITION_ROOT_X(ev);
  int ry=XDND_POSITION_ROOT_Y(ev);

  curaction=(Atom)XDND_POSITION_ACTION(ev);
  xncdprintf(("SrcWin[0x%x] X=%d, Y=%d, Action:\n",srcw,rx,ry));
  debug_atom((Atom)XDND_POSITION_ACTION(ev));
  if(srcw!=srcwin)
  {
    xncdprintf(("Wrong source window, rejecting\n"));
    return 0;
  }
  pwin=find_dnd_win(rx,ry,curaction);
  if(pwin!=curwin)
  {
    if(curwin)
      curwin->obj->dnd_leave(ev,rx,ry);
    curwin=pwin;
    if(curwin)
      curwin->obj->dnd_enter(ev,rx,ry);
  }
  if(curwin)
  {
    curx=rx;
    cury=ry;
    curwin->obj->dnd_position(ev,rx,ry);
    send_status(1,curaction,1);
  }
  else 
    send_status(0);
  return 1;
}


//Processing XdndLeave message - doing clean-ups
int DndManager::process_XdndLeave(XEvent* ev)
{
  if(dnd_data)
    delete dnd_data;
  dnd_data=0;
  curtype=-1;
  if(curwin)
    curwin->obj->dnd_leave(ev,curx,cury);
  curwin=0;
  return 1;
}

//Processing XdndDrop message - get data and kick object
int DndManager::process_XdndDrop(XEvent* ev)
{
  if(!try_convert_drag_data(ev,srcwin))
  {
    send_finished();
    return 0;
  }
  if(curwin==0)
  {
    send_finished();
    return 0;
  }
    
  return 1;
}


int DndManager::send_finished()
{
  XEvent xevent;
  
  memset (&xevent, 0, sizeof (xevent));
  
  xevent.xany.type = ClientMessage;
  xevent.xany.display = disp;
  xevent.xclient.window = srcwin;
  xevent.xclient.message_type = XdndFinished;
  xevent.xclient.format = 32;
  xevent.xclient.data.l[0]=rootw;

  send_event(xevent);
  process_XdndLeave(0);
  return 1;
}

int DndManager::send_status(int will_accept,Atom action,int want_position, int x, int y, 
			    uint l, uint h)
{
  XEvent xevent;
  
  memset (&xevent, 0, sizeof (xevent));
  
  xevent.xany.type = ClientMessage;
  xevent.xany.display = disp;
  xevent.xclient.window = srcwin;
  xevent.xclient.message_type = XdndStatus;
  xevent.xclient.format = 32;

  XDND_STATUS_TARGET_WIN (&xevent) = rootw;
  XDND_STATUS_WILL_ACCEPT_SET (&xevent, will_accept);
  if (will_accept)
    XDND_STATUS_WANT_POSITION_SET (&xevent, want_position);
  if (want_position)
    XDND_STATUS_RECT_SET (&xevent, x, y, l, h);
  XDND_STATUS_ACTION (&xevent) = action;
  send_event(xevent);
  return 1;
}

int DndManager::try_convert_drag_data(XEvent* ev,Window srcw)
{
  Window w;
  if (curtype==-1)
    return 0;
  if (!(w = XGetSelectionOwner(disp, XdndSelection))) 
  {
    xncdprintf(("XGetSelectionOwner failed"));
    return 0;
  }
  /*  if(w!=srcwin)
  {
    xncdprintf(("Source not an owner of XdndSelection [0x%x/0x%x], rejecting\n",w,srcwin));
    return 0;
    } */
  XConvertSelection(disp, XdndSelection, our_in_typelist[curtype],
		     dnd_data_atom, rootw, CurrentTime);
  return 1;
}

int DndManager::data_ready_get_it()
{
  DndData *dd;
  Atom type, *a;
  int format, i;
  unsigned long count, remaining;
  unsigned char *data = NULL;

  XGetWindowProperty (disp, rootw, dnd_data_atom,
		      0, 0x8000000L, True, our_in_typelist[curtype],
		      &type, &format, &count, &remaining, &data);

  if (type != our_in_typelist[curtype] || format != 8 || count == 0 || !data) {
    if (data)
      XFree (data);
    xncdprintf(("data_ready_get_it failed: format=%d, count=%d, type:", format,count));
    debug_atom(type);
    if(curwin)
      send_finished();
    return 0;
  }
  
  if(dnd_data)
    delete dnd_data;
  dnd_data=new char[count+1];
  strncpy(dnd_data,(char*)data,count);
  dnd_data[count]=0;
  xncdprintf(("Got the following data: '%s'\n",dnd_data));
  if(curwin)
  {
    dd=new DndData((char*)data,count,type);
    curwin->obj->dnd_drop(dd,curx,cury,curaction);
    send_finished();
  }
  XFree(data);
  return 1;
}

//Building list of types from xclient data structure
int DndManager::build_threetype_list(XEvent* ev,Window)
{
  int i;
  if(typelist)
    delete typelist;
  typelist = new Atom [XDND_THREE + 1];
  for (i = 0; i < XDND_THREE; i++)
    typelist[i] = XDND_ENTER_TYPE (ev, i);
  typelist[XDND_THREE] = 0;	/* although typelist[1] or typelist[2] may also be set to nill */
  return XDND_THREE;
}

//Building list of types from xclient data structure
int DndManager::build_bigtype_list(XEvent* ev,Window srcw)
{
  Atom type, *a;
  int format, i;
  unsigned long count, remaining;
  unsigned char *data = NULL;

  if(typelist)
    delete typelist;

  typelist = 0;

  XGetWindowProperty (disp, srcw, XdndTypeList,
		      0, 0x8000000L, False, XA_ATOM,
		      &type, &format, &count, &remaining, &data);

  if (type != XA_ATOM || format != 32 || count == 0 || !data) {
    if (data)
      XFree (data);
    xncdprintf(("XGetWindowProperty failed [XdndTypeList = %ld]", XdndTypeList));
    return 0;
  }
  typelist = new Atom [count + 1];
  a = (Atom *) data;
  for (i = 0; i < count; i++)
  {
    typelist[i] = a[i];
    debug_atom(a[i]);
  }
  typelist[count] = 0;

  XFree (data);
  
  xncdprintf(("XdndTypeList: Received %d types\n",count));

  return count;
}

void DndManager::debug_atom(Atom a)
{
#ifdef DEBUG_XNC
  if(!a)
    return;
  char *str=XGetAtomName(disp,a);
  xncdprintf(("Atom: '%s' - %d - 0x%x\n",str,a,a));
  XFree(str);
#endif
}


//Checking typelist for type we need
int DndManager::check_type_list(XEvent* ev)
{
  int i,j;
  if(!typelist || !our_in_typelist)
    return -1;
  for(i=0;typelist[i];i++)
    for(j=0;our_in_typelist[j];j++)
      if(typelist[i]==our_in_typelist[j])
      {
	xncdprintf(("Found entry in our_in_typelist:\n"));
	debug_atom(our_in_typelist[j]);
	return j;
      }
  return -1;
}


//Adding type for our_in_types that we will accept
void DndManager::add_in_type(Atom intype)
{
  int i;
  for(i=0;our_in_typelist[i];i++)
    if(our_in_typelist[i]==intype)
      return;
  our_in_typelist[i]=intype;
  xncdprintf(("Atom added to our_in_typelist:\n"));
  debug_atom(intype);
}

void DndManager::add_in_type(char* name)
{
  Atom a=XInternAtom(disp,name,False);
  add_in_type(a);
}


void DndManager::send_event(XEvent& ev)
{
  XSendEvent(disp,srcwin,False,0L,&ev);
}

int DndManager::is_dnd_aware(DndWin* dndw, Window window)
{
    Atom actual;
    int format;
    unsigned long count, remaining;
    unsigned char *data = 0;
    Atom *types, *t;
    int result = 1;

    version_in = 0;
    XGetWindowProperty (disp, window, XdndAware,
			0, 0x8000000L, False, XA_ATOM,
			&actual, &format,
			&count, &remaining, &data);
    
    if (actual != XA_ATOM || format != 32 || count == 0 || !data) {
      xncdprintf(("XGetWindowProperty failed in xdnd_is_dnd_aware - XdndAware = %ld", XdndAware));
      if (data)
	XFree (data);
      return 0;
    }
    types = (Atom *) data;
#if XDND_VERSION >= 3
    if (types[0] < 3) {
      if (data)
	XFree (data);
      return 0;
    }
#endif
    version_in = XDND_VERSION < types[0] ? XDND_VERSION : types[0];	/* minimum */
    xncdprintf(("Using XDND version %d\n", version));
    if (count > 1) {
      result = 0;
      for (t = dndw->typelist; *t; t++) 
      {
	int j;
	for (j = 1; j < count; j++) 
	{
	  if (types[j] == *t) 
	  {
	    result = 1;
	    break;
	  }
	}
	if (result)
	  break;
      }
    }
    XFree (data);
    return result;
}

int DndManager::dragging(DndObject* o,DndData* dat)
{
  XEvent xevent,xevent_temp;
  int dnd_aware=0;
  Window dropper_toplevel;

  xncdprintf(("Starting drag from object %x\n",o));
  if (XGrabPointer (disp, rootw, False,
		    ButtonMotionMask | PointerMotionMask | ButtonPressMask 
		    | ButtonReleaseMask,
		    GrabModeAsync, GrabModeAsync, None,
		    curs_one, CurrentTime) != GrabSuccess)
  {
    xncdprintf(("Unable to grab pointer"));
    return 0;
  }

  while (xevent.xany.type != ButtonRelease) 
  {
    XAllowEvents (disp, SyncPointer, CurrentTime);
    XNextEvent (disp, &xevent);
    switch (xevent.type) 
    {
    case Expose:
      break;
    case EnterNotify:
      /* this event is not actually reported, so we find out by ourselves from motion events */
      break;
    case LeaveNotify:
      /* this event is not actually reported, so we find out by ourselves from motion events */
      break;
    case ButtonRelease:
      /* done, but must send a leave event */
      break;
    case MotionNotify:
      dnd_aware = 0;
      memcpy (&xevent_temp, &xevent, sizeof (xevent));
      xevent.xmotion.subwindow = xevent.xmotion.window;
      {
	Window root_return, child_return;
	int x_temp, y_temp;
	unsigned int mask_return;
	while (XQueryPointer (disp, xevent.xmotion.subwindow, 
			      &root_return, &child_return,
			      &x_temp, &y_temp, &xevent.xmotion.x,
			      &xevent.xmotion.y, &mask_return)) 
	{
#if XDND_VERSION >= 3
	  if (!dnd_aware) 
	  {
	    if ((dnd_aware = is_dnd_aware(o->get_dndwin(), xevent.xmotion.subwindow))) 
	    {
	      dropper_toplevel = xevent.xmotion.subwindow;
	      xevent.xmotion.x_root = x_temp;
	      xevent.xmotion.y_root = y_temp;
	      xncdprintf(("Window 0x%x is supported XDND version %d\n",dropper_toplevel,version_in));
	    }
	  }

	  dnd_aware=0;
#else
	  xevent.xmotion.x_root = x_temp;
	  xevent.xmotion.y_root = y_temp;
#endif
	  if (!child_return)
	    goto found_descendent;
	  xevent.xmotion.subwindow = child_return;
	}
	break;
      }
    found_descendent:
      break;
    }
  }
  XUngrabPointer(disp,CurrentTime);
  return 1;
}
