/*
 * xpm.cxx - load routine for X11 XPM v3 format pictures
 *
 * LoadXPM(fname,pinfo)
 */

/*
 * Written by Chris P. Ross (cross@eng.umd.edu) to support XPM v3
 * format images.
 *
 * Thanks go to Sam Yates (syates@spam.maths.adelaide.edu.au) for
 * provideing inspiration.
 */
/* Modified for use with Image Engine by Leo [Leonid V. KHramov] (old@sunct2.jinr.dubna.su)
 * Copyright 1997.
 */
 
#define VALUES_LEN	80	/* Max length of values line */
#define TOKEN_LEN	8	/* Max length of color token */

#include "image.h"
#include "picinfo.h"
#include "accel.h"


/*
 * File format:
 *  (A file format def should go here)
 *
 */

typedef struct hent {
  char        token[TOKEN_LEN];
  union {
    byte      index;
    byte      rgb[3];
  } color_val;
  struct hent *next;
} hentry;

#define cv_index	color_val.index
#define cv_rgb		color_val.rgb



/* Local (Global) Variables */
byte	hex[256];
hentry **hashtab;        /* Hash table */
int      hash_len;       /* number of hash buckets */
int	bufchar;	/* Buffered character from XpmGetc */
short	in_quote;	/* Is the current point in the file in */
                                /*  a quoted string? */

/* Local Functions */
int     XpmLoadError  (char*, char*);
int     XpmGetc	      (FILE*);
int     hash          (char *);
int     hash_init     (int);
int     hash_insert   (hentry *);
hentry *hash_search   (char *);
void    hash_destroy  (void);


/**************************************/
int LoadXPM(char* fname,PICINFO* pinfo)
{
  /* returns '1' on success */
  
  FILE    *fp;
  hentry   item;
  int      c;
  char    *bname;
  char     values[VALUES_LEN];
  byte    *pic;
  byte    *i_sptr;		/* image search pointer */
  long     filesize;
  int      w, h, nc, cpp, line_pos;
  short    i, j, k;		/* for() loop indexes */
  hentry  *clmp;		/* colormap hash-table */
  hentry  *c_sptr;		/* cmap hash-table search pointer*/
  XColor   col;
  
  bname =fname;
  fp = fopen(fname, "r");
  if (!fp)
    return (XpmLoadError(bname, "couldn't open file"));
  
  fseek(fp, 0L, 2);
  filesize = ftell(fp);
  fseek(fp, 0L, 0);
  
  bufchar = -2;
  in_quote = FALSE;
  
  /* Read in the values line.  It is the first string in the
   * xpm, and contains four numbers.  w, h, num_colors, and
   * chars_per_pixel. */
  
  /* First, get to the first string */
  while (((c = XpmGetc(fp))!=EOF) && (c != '"')) ;
  line_pos = 0;
  
  /* Now, read in the string */
  while (((c = XpmGetc(fp))!=EOF) && (line_pos < VALUES_LEN) && (c != '"')) {
    values[line_pos++] = c;
  }
  if (c != '"')
    return (XpmLoadError(bname, "error parsing values line"));
  
  values[line_pos] = '\0';
  sscanf(values, "%d%d%d%d", &w, &h, &nc, &cpp);
  if (nc <= 0 || cpp <= 0)
    return (XpmLoadError(bname, "No colours in Xpm?"));
  
  if (nc > 256)
    pinfo->type = PIC24;
  else
    pinfo->type = PIC8;
  
  if (!hash_init(nc))
    return (XpmLoadError(bname, "Not enough memory to hash colormap"));
  
  clmp = (hentry *) malloc(nc * sizeof(hentry)); /* Holds the colormap */
  if (pinfo->type == PIC8) pic = (byte *) malloc((size_t) (w*h));
                      else pic = (byte *) malloc((size_t) (w*h*3));
  
  if (!clmp || !pic)
    return (XpmLoadError(bname, "Not enough memory to load pixmap"));
  
  c_sptr = clmp;
  i_sptr = pic;
  
  /* initialize the 'hex' array for zippy ASCII-hex -> int conversion */
  
  for (i = 0 ; i < 256 ; i++)   hex[i] = 0;
  for (i = '0'; i <= '9' ; i++) hex[i] = i - '0';
  for (i = 'a'; i <= 'f' ; i++) hex[i] = i - 'a' + 10;
  for (i = 'A'; i <= 'F' ; i++) hex[i] = i - 'A' + 10;
  
  /* Now, we need to read the colormap. */
  pinfo->colType = F_BWDITHER;
  for (i = 0 ; i < nc ; i++) {
    while (((c = XpmGetc(fp))!=EOF) && (c != '"')) ;
    if (c != '"')
      return (XpmLoadError(bname, "Error reading colormap"));
    
    for (j = 0 ; j < cpp ; j++)
      c_sptr->token[j] = XpmGetc(fp);
    c_sptr->token[j] = '\0';
    
    while (((c = XpmGetc(fp))!=EOF) && ((c == ' ') || (c == '\t'))) ;
    if (c == EOF)		/* The failure condition of getc() */
      return (XpmLoadError(bname, "Error parsing colormap line"));
    
    do {
      char  key[3];
      char  color[40];	/* Need to figure a good size for this... */
      short hd;		/* Hex digits per R, G, or B */
      
      for (j=0; j<2 && (c != ' ') && (c != '\t') && (c != EOF); j++) {
	key[j] = c;
	c = XpmGetc(fp);
      }
      key[j] = '\0';

      while (((c = XpmGetc(fp))!=EOF) && ((c == ' ') || (c == '\t'))) ;
      if (c == EOF)	/* The failure condition of getc() */
	return (XpmLoadError(bname, "Error parsing colormap line"));

      for (j=0; j<39 && (c!=' ') && (c!='\t') && (c!='"') && c!=EOF; j++) {
	color[j] = c;
	c = XpmGetc(fp);
      }
      color[j]='\0';

      while ((c == ' ') || (c == '\t'))
	c = XpmGetc(fp);
      
      if (key[0] == 's')	/* Don't find a color for a symbolic name */
	continue;
      
      if (XParseColor(disp,defcmp,color,&col)) {
	if (pinfo->type == PIC8) {
	  pinfo->pal[i*3] = col.red >> 8;
	  pinfo->pal[i*3+1] = col.green >> 8;
	  pinfo->pal[i*3+2] = col.blue >> 8;
	  c_sptr->cv_index = i;

	  /* Is there a better way to do this? */
	  if (pinfo->colType != F_FULLCOLOR)
	    if (pinfo->colType == F_GREYSCALE)
	      if (pinfo->pal[i*3] == pinfo->pal[i*3+1] &&
		  pinfo->pal[i*3+1] == pinfo->pal[i*3+2])
		/* Still greyscale... */
		;
	      else
		/* It's color */
		pinfo->colType = F_FULLCOLOR;
	    else
	      if (pinfo->pal[i*3] == pinfo->pal[i*3+1] &&
		  pinfo->pal[i*3+1] == pinfo->pal[i*3+2])
		if ((pinfo->pal[i*3] == 0 || pinfo->pal[i*3] == 0xff) &&
		    (pinfo->pal[i*3+1] == 0 || pinfo->pal[i*3+1] == 0xff) &&
		    (pinfo->pal[i*3+2] == 0 || pinfo->pal[i*3+2] == 0xff))
		  /* It's B/W */
		  ;
		else
		  /* It's greyscale */
		  pinfo->colType = F_GREYSCALE;
	      else
		/* It's color */
		pinfo->colType = F_FULLCOLOR;
	  
	}
	else {   /* PIC24 */
	  c_sptr->cv_rgb[0] = col.red >> 8;
	  c_sptr->cv_rgb[1] = col.green >> 8;
	  c_sptr->cv_rgb[2] = col.blue >> 8;
	}
      }

      else {      /* 'None' or unrecognized color spec */
	int rgb;

	if (strcmp(color, "None") == 0) rgb = 0xb2c0dc;  /* infobg */
	else {
	  fprintf(stderr, "XPM: %s:  unknown color spec '%s'", bname, color);
	  rgb = 0x808080;
	}
	
	if (pinfo->type == PIC8) {
	  pinfo->pal[i*3] = (rgb>>16) & 0xff;
	  pinfo->pal[i*3+1] = (rgb>> 8) & 0xff;
	  pinfo->pal[i*3+2] = (rgb>> 0) & 0xff;
	  c_sptr->cv_index = i;
	}
	else {
	  c_sptr->cv_rgb[0] = (rgb>>16) & 0xff;
	  c_sptr->cv_rgb[1] = (rgb>> 8) & 0xff;
	  c_sptr->cv_rgb[2] = (rgb>> 0) & 0xff;
	}
      }

      
      xvbcopy((char *) c_sptr, (char *) &item, sizeof(item));
      hash_insert(&item);
      
      
      if (*key == 'c') {	/* This is the color entry, keep it. */
	while (c!='"' && c!=EOF) c = XpmGetc(fp);
	break;
      }
      
    } while (c != '"');
    c_sptr++;

  } /* for */
  

  
  /* Now, read the pixmap. */
  for (i = 0 ; i < h ; i++) {
    while (((c = XpmGetc(fp))!=EOF) && (c != '"')) ;
    if (c != '"')
      return (XpmLoadError(bname, "Error reading colormap"));
    
    for (j = 0 ; j < w ; j++) {
      char pixel[TOKEN_LEN];
      hentry *mapentry;
      
      for (k = 0 ; k < cpp ; k++)
	pixel[k] = XpmGetc(fp);
      pixel[k] = '\0';

      if (!(mapentry = (hentry *) hash_search(pixel))) {
	/* No colormap entry found.  What do we do?  Bail for now */
	return (XpmLoadError(bname, "Can't map resolve into colormap"));
      }
      
      if (pinfo->type == PIC8)
	*i_sptr++ = mapentry->cv_index;
      else {
	*i_sptr++ = mapentry->cv_rgb[0];
	*i_sptr++ = mapentry->cv_rgb[1];
	*i_sptr++ = mapentry->cv_rgb[2];
      }
    }  /* for ( j < w ) */
    (void)XpmGetc(fp);		/* Throw away the close " */
  
  }  /* for ( i < h ) */
  
  pinfo->pic = pic;
  pinfo->w = w;
  pinfo->h = h;
  pinfo->frmType = F_XPM;

  sprintf(pinfo->fullInfo, "Xpm v3 Pixmap %dx%d (%ld bytes)", w,h, filesize);
  sprintf(pinfo->shrtInfo, "%dx%d Xpm.", w, h);
  pinfo->comment = (char *)NULL;
  
  hash_destroy();
  free(clmp);
  
  if (fp != stdin)
    fclose(fp);
  
  return(1);
}


/***************************************/
int XpmLoadError(char *fname,char* st)
{
  fprintf(stderr,"XPM Error: %s:  %s", fname, st);
  return 0;
}


/***************************************/
int XpmGetc(FILE* f)
{
  int	c, d, lastc;
  
  if (bufchar != -2) {
    /* The last invocation of this routine read the character... */
    c = bufchar;
    bufchar = -2;
    return(c);
  }
  
  if ((c = getc(f)) == EOF)
    return(EOF);
  
  if (c == '"')
    in_quote = !in_quote;
  else if (!in_quote && c == '/') {	/* might be a C-style comment */
    if ((d = getc(f)) == EOF)
      return(EOF);
    if (d == '*') {				/* yup, it *is* a comment */
      if ((lastc = getc(f)) == EOF)
	return(EOF);
      do {				/* skip through to the end */
	if ((c = getc(f)) == EOF)
	  return(EOF);
	if (lastc != '*' || c != '/')	/* end of comment */
	  lastc = c;
      } while (lastc != '*' || c != '/');
      if ((c = getc(f)) == EOF)
	return(EOF);
    } else					/* nope, not a comment */
      bufchar = d;
  }
  return(c);
}


/***************************************/
/*         hashing functions           */
/***************************************/


/***************************************/
int hash(char* token) 
{
  int i, sum;

  for (i=sum=0; token[i] != '\0'; i++)
    sum += token[i];
  
  sum = sum % hash_len;
  return (sum);
}


/***************************************/
int hash_init(int hsize)
{
  /*
   * hash_init() - This function takes an arg, but doesn't do anything with
   * it.  It could easily be expanded to figure out the "optimal" number of
   * buckets.  But, for now, a hard-coded 257 will do.  (Until I finish the
   * 24-bit image writing code.  :-)
   */

  int i;
  
  hash_len = 257;

  hashtab = (hentry **) malloc(sizeof(hentry *) * hash_len);
  if (!hashtab) {
    fprintf(stderr, "XPM: Couldn't malloc hashtable in LoadXPM()!\n");
    return 0;
  }

  for (i = 0 ; i < hash_len ; i++)
    hashtab[i] = NULL;
  
  return 1;
}


/***************************************/
int hash_insert(hentry* entry)
{
  int     key;
  hentry *tmp;
  
  key = hash(entry->token);
  
  tmp = (hentry *) malloc(sizeof(hentry));
  if (!tmp) {
    fprintf(stderr, "XPM: Couldn't malloc hash entry in LoadXPM()!\n");
    return 0;
  }
  
  xvbcopy((char *)entry, (char *)tmp, sizeof(hentry));
  
  if (hashtab[key]) tmp->next = hashtab[key];
               else tmp->next = NULL;
  
  hashtab[key] = tmp;
  
  return 1;
}


/***************************************/
hentry *hash_search(char* token)
{
  int     key;
  hentry *tmp;
  
  key = hash(token);
  
  tmp = hashtab[key];
  while (tmp && strcmp(token, tmp->token)) {
    tmp = tmp->next;
  }

  return tmp;
}


/***************************************/
void hash_destroy()
{
  int     i;
  hentry *tmp;
  
  for (i=0; i<hash_len; i++) {
    while (hashtab[i]) {
      tmp = hashtab[i]->next;
      free(hashtab[i]);
      hashtab[i] = tmp;
    }
  }
  
  free(hashtab);
  return;
}






