/*	$Csoft: audio.c,v 1.1 2001/08/31 00:09:03 vedge Exp $	*/

/*
 * Copyright (c) 2001, CubeSoft Communications, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistribution in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of CubeSoft Communications, nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
#include <errno.h>
#include "sun.h"
#include "libxmms/util.h"
#include "resample.h"


static	void	sun_setup_format(AFormat, gint, gint);
static	int	sun_downsample(gpointer, guint, guint, guint);
void		sun_set_audio_params(void);
static	void	sun_calc_device_buffer_used(void);
gint		sun_used(void);
gint		sun_free(void);
static	int	sun_write_audio(gpointer, gint);
static	void	swap_words(guint16 *, gint);
int		sun_downsample(gpointer, guint, guint, guint);
void		*sun_loop(void *);
void		sun_set_audio_params(void);


static void
sun_setup_format(AFormat fmt, gint rate, gint nch)
{
	audio.format = fmt;
	audio.frequency = rate;
	audio.channels = nch;

	switch (fmt) {
	case FMT_U8:		/* 8-bit unsigned linear */
		audio.sun_format = AUDIO_ENCODING_PCM8;
		break;
	case FMT_S8:		/* 8-bit signed linear */
		audio.sun_format = AUDIO_ENCODING_SLINEAR;
		break;
	case FMT_U16_LE:	/* 16-bit unsigned linear, LSB */
		audio.sun_format = AUDIO_ENCODING_ULINEAR_LE;
		break;
	case FMT_U16_BE:	/* 16-bit unsigned linear, MSB */
		audio.sun_format = AUDIO_ENCODING_ULINEAR_BE;
		break;
	case FMT_U16_NE:	/* 16-bit unsigned linear, native */
#ifdef WORDS_BIGENDIAN
		audio.sun_format = AUDIO_ENCODING_ULINEAR_BE;
#else
		audio.sun_format = AUDIO_ENCODING_ULINEAR_LE;
#endif
		break;
	case FMT_S16_LE:	/* 16-bit signed linear, LSB */
		audio.sun_format = AUDIO_ENCODING_SLINEAR_LE;
		break;
	case FMT_S16_BE:	/* 16-bit signed linear, MSB */
		audio.sun_format = AUDIO_ENCODING_SLINEAR_BE;
		break;
	case FMT_S16_NE:	/* 16-bit signed linear, native */
#ifdef WORDS_BIGENDIAN
		audio.sun_format = AUDIO_ENCODING_SLINEAR_BE;
#else
		audio.sun_format = AUDIO_ENCODING_SLINEAR_LE;
#endif
		break;
	}

	audio.bps = rate * nch;
	if (audio.sun_format == AUDIO_ENCODING_ULINEAR_BE
	    || audio.sun_format == AUDIO_ENCODING_ULINEAR_LE
	    || audio.sun_format == AUDIO_ENCODING_SLINEAR_BE
	    || audio.sun_format == AUDIO_ENCODING_SLINEAR_LE) {
		audio.bps *= 2;
	}
}

static void
sun_calc_device_buffer_used(void)
{
	audio_offset_t ooffs;

	if (audio.paused) {
		audio.device_buffer_used = 0;
	} else if (ioctl(audio.fd, AUDIO_GETOOFFS, &ooffs) >= 0) {
		audio.device_buffer_used = ooffs.offset;
	}
}

gint
sun_get_written_time(void)
{
	if (!audio.going) {
		return (0);
	}
	return ((gint) ((audio.written * 1000) / audio.input_bps));
}

gint
sun_get_output_time(void)
{
	guint64 bytes;

	if (!audio.fd || !audio.going) {
		return (0);
	}

	if (audio.realtime) {
		sun_calc_device_buffer_used();
	}

	bytes = MAX(0, audio.output_bytes - audio.device_buffer_used);
	return (audio.output_time_offset +
	    (gint) ((bytes * 1000) / audio.ebps));
}

gint
sun_used(void)
{
	if (audio.realtime) {
		return (0);
	}

	if (audio.wr_index >= audio.rd_index) {
		return (audio.wr_index - audio.rd_index);
	}
	return (audio.buffer_size - (audio.rd_index - audio.wr_index));
}

gint
sun_playing(void)
{
	if (!audio.going) {
		return (0);
	}
	if (audio.realtime) {
		sun_calc_device_buffer_used();
	}
	if (!sun_used() && (audio.device_buffer_used -
	    (3 * audio.blocksize)) <= 0) {
		return (FALSE);
	}

	return (TRUE);
}

gint
sun_free(void)
{
	if (audio.realtime) {
		if (audio.paused) {
			return (0);
		} else {
			return (1000000);
		}
	} else {
		if (audio.remove_prebuffer && audio.prebuffer) {
			audio.prebuffer = FALSE;
			audio.remove_prebuffer = FALSE;
		}
		if (audio.prebuffer) {
			audio.remove_prebuffer = TRUE;
		}

		if (audio.rd_index > audio.wr_index) {
			return ((audio.rd_index - audio.wr_index)
				- audio.blocksize - 1);
		}

		return ((audio.buffer_size - (audio.wr_index - audio.rd_index))
			- audio.blocksize - 1);
	}
}

static int
sun_write_audio(gpointer data, gint length)
{
	AFormat new_format;
	gint new_frequency, new_channels, output = 0;
	EffectPlugin *ep;

	new_format = audio.input_format;
	new_frequency = audio.input_frequency;
	new_channels = audio.input_channels;

	ep = get_current_effect_plugin();
	if (effects_enabled() && ep && ep->query_format) {
		ep->query_format(&new_format, &new_frequency, &new_channels);
	}

	if (new_format != audio.format || new_frequency != audio.frequency
	    || new_channels != audio.channels) {
		audio.output_time_offset +=
			(gint) ((audio.output_bytes * 1000) / audio.ebps);

		audio.output_bytes = 0;
		sun_setup_format(new_format, new_frequency, new_channels);
		audio.frequency = new_frequency;
		audio.channels = new_channels;

		close(audio.fd);
		audio.fd = open(audio.devaudio, O_WRONLY);
		sun_set_audio_params();
	}
	if (effects_enabled() && ep && ep->mod_samples) {
		length = ep->mod_samples(&data, length, audio.input_format,
		    audio.input_frequency, audio.input_channels);
	}

	if (audio.frequency == audio.efrequency) {
		output = write(audio.fd, data, length);
	} else {
		output += sun_downsample(data, length, audio.frequency,
		    audio.efrequency);
	}

	audio.output_bytes += output;

	/* FIXME: Handle short writes when downsampling */
	if (audio.frequency != audio.efrequency) {
		output = length;
	}

	return (output);
}

static void
swap_words(guint16 * buffer, gint length)
{
	guint16 *ptr = buffer;
	gint i;

	for (i = 0; i < length; i++, ptr++) {
		*ptr = ((*ptr & 0x00FF) << 8) | (*ptr >> 8);
	}
}

int
sun_downsample(gpointer ob, guint length, guint speed, guint espeed)
{
	guint w = 0;
	static gpointer nbuffer = NULL;
	static gint nbuffer_size = 0;
#ifdef WORDS_BIGENDIAN
	const gboolean big_endian = TRUE;
#else
	const gboolean big_endian = FALSE;
#endif

	switch (audio.sun_format) {
	case AUDIO_ENCODING_SLINEAR_BE:
	case AUDIO_ENCODING_SLINEAR_LE:
		if (audio.channels == 2) {
			RESAMPLE_STEREO(gint16);
		} else {
			RESAMPLE_MONO(gint16);
		}
		break;
	case AUDIO_ENCODING_ULINEAR_BE:
	case AUDIO_ENCODING_ULINEAR_LE:
		if (audio.channels == 2) {
			RESAMPLE_STEREO(guint16);
		} else {
			RESAMPLE_MONO(guint16);
		}
		break;
	case AUDIO_ENCODING_SLINEAR:
		if (audio.channels == 2) {
			RESAMPLE_STEREO(gint8);
		} else {
			RESAMPLE_MONO(gint8);
		}
		break;
	case AUDIO_ENCODING_ULINEAR:
		if (audio.channels == 2) {
			RESAMPLE_STEREO(guint8);
		} else {
			RESAMPLE_MONO(guint8);
		}
		break;
	}
	return (w);
}

void
sun_write(gpointer ptr, gint length)
{
	gint cnt, off = 0;

	if (audio.realtime) {
		if (audio.paused) {
			return;
		}
		audio.written += sun_write_audio(ptr, length);
	} else {
		audio.remove_prebuffer = FALSE;

		audio.written += length;
		while (length > 0) {
			cnt = MIN(length, audio.buffer_size - audio.wr_index);
			memcpy((gchar *) audio.buffer + audio.wr_index,
			    (gchar *) ptr + off, cnt);
			audio.wr_index = (audio.wr_index + cnt) %
			    audio.buffer_size;
			length -= cnt;
			off += cnt;
		}
	}
}

void
sun_close(void)
{
	if (!audio.going) {
		return;
	}
	audio.going = 0;
	pthread_join(audio.buffer_thread, NULL);
	audio.wr_index = 0;
	audio.rd_index = 0;
}

void
sun_flush(gint time)
{
	ioctl(audio.fd, AUDIO_FLUSH, NULL);

	audio.output_time_offset = time;
	audio.written = (guint16)(time / 10) * (guint64)(audio.input_bps / 100);
	audio.output_bytes = 0;
}

void
sun_pause(short p)
{
	if (audio.realtime) {
		audio.paused = p;
	} else {
		if (p == TRUE) {
			audio.do_pause = TRUE;
		} else {
			audio.unpause = TRUE;
		}
	}
}

void *
sun_loop(void *arg)
{
	gint length, cnt;
	fd_set set;
	struct timeval tv;

	while (audio.going) {
		if (sun_used() > audio.prebuffer_size) {
			audio.prebuffer = FALSE;
		}

		if (sun_used() > 0 && !audio.paused && !audio.prebuffer) {

			tv.tv_sec = 0;
			tv.tv_usec = 10000;
			FD_ZERO(&set);
			FD_SET(audio.fd, &set);

			if (select(audio.fd + 1, NULL, &set, NULL, &tv) > 0) {
				length = MIN(audio.blocksize, sun_used());
				while (length > 0) {
					cnt = MIN(length, audio.buffer_size -
					    audio.rd_index);
					cnt = sun_write_audio(
					    (gchar *)audio.buffer +
					    audio.rd_index, cnt);
					audio.rd_index =
					    (audio.rd_index + cnt) %
					    audio.buffer_size;
					length -= cnt;
				}
			}

		} else {
			xmms_usleep(10000);
		}

		if (audio.do_pause && !audio.paused) {
			audio.do_pause = FALSE;
			audio.paused = TRUE;
			audio.rd_index -= audio.device_buffer_used;
			audio.output_bytes -= audio.device_buffer_used;
			if (audio.rd_index < 0) {
				audio.rd_index += audio.buffer_size;
			}
			ioctl(audio.fd, AUDIO_FLUSH, NULL);
		}
		if (audio.unpause && audio.paused) {
			audio.unpause = FALSE;
			close(audio.fd);
			audio.fd = open(audio.devaudio, O_WRONLY);
			sun_set_audio_params();
			audio.paused = FALSE;
		}
	}

	close(audio.fd);
	g_free(audio.buffer);
	pthread_exit(NULL);
}

void
sun_set_audio_params(void)
{
	audio_encoding_t enc;
	audio_info_t info;
	int ochannels;

	AUDIO_INITINFO(&info);

	info.mode = AUMODE_PLAY;
	if (ioctl(audio.fd, AUDIO_SETINFO, &info) < 0) {
		g_error("%s: could not switch to play mode.", audio.devaudio);
		return;
	}

	enc.index = 0;
	while ((ioctl(audio.fd, AUDIO_GETENC, &enc) >= 0) &&
	    enc.encoding != audio.sun_format) {
		enc.index++;
	}

	info.play.encoding = audio.sun_format;
	info.play.precision = enc.precision;
	strcpy(audio.format_name, enc.name);

	if (ioctl(audio.fd, AUDIO_SETINFO, &info) < 0) {
		g_error("%s: cannot handle encoding: %i-bit \"%s\" (0x%x)",
		    audio.devaudio, enc.precision, enc.name, audio.sun_format);
		return;
	}

	ochannels = info.play.channels;
	info.play.channels = audio.channels;
	if (ioctl(audio.fd, AUDIO_SETINFO, &info) < 0) {
		g_warning("%s: cannot handle %i channels", audio.devaudio,
		    audio.channels);
		audio.channels = ochannels;
		info.play.channels = ochannels;
		return;
	}

	audio.efrequency = audio.frequency;
	info.play.sample_rate = audio.frequency;
	if (ioctl(audio.fd, AUDIO_SETINFO, &info) < 0) {
		g_error("%s: cannot handle %i Hz playback", audio.devaudio,
		    audio.frequency);
		return;
	}

	if (ioctl(audio.fd, AUDIO_GETINFO, &info) < 0) {
		audio.blocksize = SUN_DEFAULT_BLOCKSIZE;
	} else {
		audio.blocksize = info.blocksize;
	}
	audio.ebps = audio.efrequency * audio.channels;
	if (audio.sun_format == AUDIO_ENCODING_ULINEAR_BE
	    || audio.sun_format == AUDIO_ENCODING_ULINEAR_LE
	    || audio.sun_format == AUDIO_ENCODING_SLINEAR_BE
	    || audio.sun_format == AUDIO_ENCODING_SLINEAR_LE) {
		audio.ebps *= 2;
	}
}


gint
sun_open(AFormat fmt, gint rate, gint nch)
{
	audio_info_t info;

	AUDIO_INITINFO(&info);

	/* Open the audio device */
	if ((audio.fd = open(audio.devaudio, O_WRONLY)) < 0) {
		g_warning("%s: %s", audio.devaudio, strerror(errno));
		return (0);
	}
	sun_setup_format(fmt, rate, nch);

	/* Obtain the H/W block size */
	if (ioctl(audio.fd, AUDIO_GETINFO, &info) != 0) {
		audio.blocksize = SUN_DEFAULT_BLOCKSIZE;
	} else {
		audio.blocksize = info.blocksize;
	}

	audio.realtime = xmms_check_realtime_priority();
	audio.input_format = audio.format;
	audio.input_channels = audio.channels;
	audio.input_frequency = audio.frequency;
	audio.input_bps = audio.bps;

	if (!audio.realtime) {
		audio.buffer_size = audio.req_buffer_size;

		if (audio.buffer_size < SUN_MIN_BUFFER_SIZE) {
			audio.buffer_size = SUN_MIN_BUFFER_SIZE;
		}

		audio.prebuffer_size =
		    (audio.buffer_size * audio.req_prebuffer_size) / 100;

		audio.buffer_size += audio.blocksize;

#ifdef SUN_DEBUG
		fprintf(stderr, "
Input:			%i kbps
Requested buffer size:	%i bytes
Hardware R / W block size:%i bytes
Buffer size:		%i bytes
Prebuffer size:	%i bytes(%i %%)
			",
			audio.input_bps / 1000,
			audio.req_buffer_size,
			audio.blocksize,
			audio.buffer_size,
			audio.prebuffer_size,
			audio.req_prebuffer_size);
#endif /* SUN_DEBUG */

		audio.buffer = g_malloc0(audio.buffer_size);
	}

	audio.prebuffer = 1;
	audio.wr_index = 0;
	audio.rd_index = 0;
	audio.output_time_offset = 0;
	audio.written = 0;
	audio.output_bytes = 0;

	audio.paused = FALSE;
	audio.do_pause = FALSE;
	audio.unpause = FALSE;
	audio.remove_prebuffer = FALSE;

	sun_set_audio_params();

	audio.going++;
	if (!audio.realtime) {
		pthread_create(&audio.buffer_thread, NULL, sun_loop, NULL);
	}
	return (1);
}
