/*  seek.c - functions related to real-time seeking
 *  Copyright (C) 2000-2001  Jason Jordan (shnutils@freeshell.org)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * $Id: seek.c,v 1.4 2001/12/30 05:12:04 jason Exp $
 */

#include <stdlib.h>
#include <glib.h>
#include <xmms/util.h>
#include <xmms/configfile.h>
#include "shorten.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

shn_seek_entry *seek_entry_search(shn_seek_entry *table,ulong goal,ulong min,ulong max)
{
	ulong med = (min + max) / 2;
	shn_seek_entry *middle = table + med;
	ulong sample = uchar_to_ulong_le(middle->data);

	if (goal < sample)
		return seek_entry_search(table,goal,min,med-1);
	if (goal > sample + 25600)
		return seek_entry_search(table,goal,med+1,max);
	return middle;
}

int load_separate_seek_table(char *filename,shn_file *this_shn)
{
	FILE *f;
	slong seek_table_len;

	debug("Looking for seek table in separate file: '%s'",filename);

	if (!(f=fopen(filename,"r")))
	{
		return 0;
	}

	fseek(f,0,SEEK_END);
	seek_table_len = (slong)ftell(f) - SEEK_HEADER_SIZE;
	fseek(f,0,SEEK_SET);

	if (fread((void *)this_shn->seek_header.data,1,SEEK_HEADER_SIZE,f) == SEEK_HEADER_SIZE)
	{
		this_shn->seek_header.version = uchar_to_ulong_le(this_shn->seek_header.data+4);
		this_shn->seek_header.shnFileSize = uchar_to_ulong_le(this_shn->seek_header.data+8);
		if (memcmp(this_shn->seek_header.data,SEEK_HEADER_SIGNATURE,strlen(SEEK_HEADER_SIGNATURE)) == 0)
		{
			if (this_shn->seek_header.shnFileSize == this_shn->wave_header.actual_size)
			{
				if ((this_shn->seek_table = malloc(seek_table_len)))
				{
					if (fread((void *)this_shn->seek_table,1,seek_table_len,f) == seek_table_len)
					{
						debug("Successfully loaded seek table in separate file: '%s'",filename);
						this_shn->vars.seek_table_entries = seek_table_len / SEEK_ENTRY_SIZE;
						fclose(f);
						return 1;
					}
				}
			}
			else
			{
				debug("Seek table .shn file size and actual .shn file size differ");
			}
		}
	}

	fclose(f);
	return 0;
}

void load_seek_table(shn_file *this_shn,char *filename)
{
	char *altfilename,*p;
	int len;

	debug("Looking for seek table appended to file: '%s'",filename);

	fseek(this_shn->vars.fd,-SEEK_TRAILER_SIZE,SEEK_END);
	if (fread((void *)this_shn->seek_trailer.data,1,SEEK_TRAILER_SIZE,this_shn->vars.fd) == SEEK_TRAILER_SIZE)
	{
		this_shn->seek_trailer.seekTableSize = uchar_to_ulong_le(this_shn->seek_trailer.data);
		if (memcmp(this_shn->seek_trailer.data+4,SEEK_TRAILER_SIGNATURE,strlen(SEEK_TRAILER_SIGNATURE)) == 0)
		{
			fseek(this_shn->vars.fd,-this_shn->seek_trailer.seekTableSize,SEEK_END);
			this_shn->seek_trailer.seekTableSize -= (SEEK_HEADER_SIZE + SEEK_TRAILER_SIZE);
			if (fread((void *)this_shn->seek_header.data,1,SEEK_HEADER_SIZE,this_shn->vars.fd) == SEEK_HEADER_SIZE)
			{
				if ((this_shn->seek_table = malloc(this_shn->seek_trailer.seekTableSize)))
				{
					if (fread((void *)this_shn->seek_table,1,this_shn->seek_trailer.seekTableSize,this_shn->vars.fd) == this_shn->seek_trailer.seekTableSize)
					{
						debug("Successfully loaded seek table appended to file: '%s'",filename);
						this_shn->vars.seek_table_entries = this_shn->seek_trailer.seekTableSize / SEEK_ENTRY_SIZE;
						return;
					}
				}
			}
		}
	}

	/* otherwise, look for it in a separate file */

	if (!(altfilename = malloc(strlen(filename)+sizeof(SEEK_SUFFIX)+1)))
	{
		debug("Could not allocate memory for alternate filename");
		return;
	}

	strcpy(altfilename,filename);
	if (filename_contains_a_dot(altfilename))
		p = strrchr(altfilename,'.');
	else
		p = altfilename + strlen(altfilename);
	strcpy(p,SEEK_SUFFIX);

	if (load_separate_seek_table(altfilename,this_shn))
	{
		free(altfilename);
		return;
	}

	free(altfilename);

	if ((p=strrchr(filename,'/')))
	{
		p++;
		len = filename + strlen(filename) - p + 1;
	}
	else
	{
		p = filename;
		len = strlen(filename);
	}

	if (!(altfilename = malloc(strlen(shn_cfg.seek_tables_path)+1+len+sizeof(SEEK_SUFFIX)+1)))
	{
		debug("Could not allocate memory for alternate filename");
		return;
	}

	strcpy(altfilename,shn_cfg.seek_tables_path);
	strcat(altfilename,"/");
	strcat(altfilename,p);
	if (filename_contains_a_dot(altfilename))
		p = strrchr(altfilename,'.');
	else
		p = altfilename + strlen(altfilename);
	strcpy(p,SEEK_SUFFIX);

	if (load_separate_seek_table(altfilename,this_shn))
	{
		free(altfilename);
		return;
	}

	debug("Could not find any seek tables");
	free(altfilename);
}
