/*  gtk.c - functions to build and display GTK windows
 *  Copyright (C) 2000-2001  Jason Jordan (shnutils@freeshell.org)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * $Id: gtk.c,v 1.6 2001/12/30 05:12:04 jason Exp $
 */

#include <stdlib.h>
#include <glib.h>
#include <xmms/util.h>
#include <xmms/configfile.h>
#include <xmms/dirbrowser.h>
#include "shorten.h"

static GtkWidget *shn_configurewin = NULL,
		 *about_box,
		 *vbox,
		 *options_vbox,
		 *bbox,
		 *notebook,
		 *output_frame,
		 *output_vbox,
		 *output_label,
		 *output_error_stderr,
		 *output_error_window,
		 *output_error_devnull,
		 *misc_frame,
		 *misc_vbox,
		 *swap_bytes_toggle,
		 *verbose_toggle,
		 *path_frame,
		 *path_vbox,
		 *path_label,
		 *path_entry,
		 *path_entry_hbox,
		 *path_browse,
		 *path_dirbrowser = NULL,
		 *ok,
		 *cancel,
		 *apply;

void display_shn_about(void)
{
	if (about_box != NULL)
	{
		gdk_window_raise(about_box->window);
		return;
	}

	about_box = xmms_show_message(
		(gchar *) "About " PACKAGE,
		(gchar *) PACKAGE " version " VERSION "\n"
			  "Copyright 2001 Jason Jordan <shnutils@freeshell.org>\n"
			  "Portions Copyright (C) 1992-1995 Tony Robinson\n"
			  "\n"
			  "shorten utilities pages:\n"
			  "\n"
			  "http://shnutils.etree.org/\n"
			  "http://shnutils.freeshell.org/",
		(gchar *) "Cool",
		FALSE, NULL, NULL);
	gtk_signal_connect(GTK_OBJECT(about_box), "destroy",
		gtk_widget_destroyed, &about_box);
}

static void path_dirbrowser_cb(gchar * dir)
{
        gtk_entry_set_text(GTK_ENTRY(path_entry), dir);
}

void destroy_path_dirbrowser()
{
	if (path_dirbrowser)
		gtk_widget_destroy(path_dirbrowser);
}

static void path_browse_cb(GtkWidget * w, gpointer data)
{
        if (!path_dirbrowser)
        {
                path_dirbrowser = xmms_create_dir_browser("Select alternate seek table directory:", shn_cfg.seek_tables_path, GTK_SELECTION_SINGLE, path_dirbrowser_cb);
                gtk_signal_connect(GTK_OBJECT(path_dirbrowser), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &path_dirbrowser);
                gtk_window_set_transient_for(GTK_WINDOW(path_dirbrowser), GTK_WINDOW(shn_configurewin));
                gtk_widget_show(path_dirbrowser);
        }
}

void shn_configurewin_save(void)
{
	ConfigFile *cfg;
	gchar *filename;

	shn_cfg.error_output_method = ERROR_OUTPUT_DEVNULL;
	if (GTK_TOGGLE_BUTTON(output_error_stderr)->active)
		shn_cfg.error_output_method = ERROR_OUTPUT_STDERR;
	else if (GTK_TOGGLE_BUTTON(output_error_window)->active)
		shn_cfg.error_output_method = ERROR_OUTPUT_WINDOW;

	g_free(shn_cfg.seek_tables_path);
	shn_cfg.seek_tables_path = g_strdup(gtk_entry_get_text(GTK_ENTRY(path_entry)));

	shn_cfg.verbose = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(verbose_toggle));

	shn_cfg.swap_bytes = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(swap_bytes_toggle));

	filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
	cfg = xmms_cfg_open_file(filename);
	if (!cfg)
		cfg = xmms_cfg_new();

	xmms_cfg_write_int(cfg, XMMS_SHN_VERSION_TAG, shn_cfg.error_output_method_config_name, shn_cfg.error_output_method);
	xmms_cfg_write_boolean(cfg, XMMS_SHN_VERSION_TAG, shn_cfg.verbose_config_name, shn_cfg.verbose);
	xmms_cfg_write_string(cfg, XMMS_SHN_VERSION_TAG, shn_cfg.seek_tables_path_config_name, shn_cfg.seek_tables_path);
	xmms_cfg_write_boolean(cfg, XMMS_SHN_VERSION_TAG, shn_cfg.swap_bytes_config_name, shn_cfg.swap_bytes);

	xmms_cfg_write_file(cfg, filename);
	xmms_cfg_free(cfg);
	g_free(filename);
	destroy_path_dirbrowser();
}

void shn_configurewin_apply()
{
	shn_configurewin_save();
}

void shn_configurewin_ok(void)
{
	shn_configurewin_save();
	gtk_widget_destroy(shn_configurewin);
}

void display_shn_configure(void)
{
	if (shn_configurewin != NULL)
	{
		gdk_window_raise(shn_configurewin->window);
		return;
	}

	shn_configurewin = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_signal_connect(GTK_OBJECT(shn_configurewin), "destroy", GTK_SIGNAL_FUNC(destroy_path_dirbrowser), NULL);
	gtk_signal_connect(GTK_OBJECT(shn_configurewin), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &shn_configurewin);
	gtk_window_set_title(GTK_WINDOW(shn_configurewin), (gchar *)"SHN Player Configuration");
	gtk_window_set_policy(GTK_WINDOW(shn_configurewin), FALSE, FALSE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(shn_configurewin), 10);

	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(shn_configurewin), vbox);

	notebook = gtk_notebook_new();

	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

	options_vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(options_vbox), 5);

/* begin error output box */

	output_frame = gtk_frame_new((gchar *)" Error display options ");
	gtk_container_set_border_width(GTK_CONTAINER(output_frame), 5);

	output_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(output_vbox), 5);
	gtk_container_add(GTK_CONTAINER(output_frame), output_vbox);

	output_label = gtk_label_new((gchar *)"When an error occurs, display it to:");
	gtk_misc_set_alignment(GTK_MISC(output_label), 0, 0);
	gtk_label_set_justify(GTK_LABEL(output_label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(output_vbox), output_label, FALSE, FALSE, 0);
	gtk_widget_show(output_label);

	output_error_window = gtk_radio_button_new_with_label(NULL, (gchar *)"a window");
	if (shn_cfg.error_output_method == ERROR_OUTPUT_WINDOW)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(output_error_window), TRUE);
	gtk_box_pack_start(GTK_BOX(output_vbox), output_error_window, FALSE, FALSE, 0);
	gtk_widget_show(output_error_window);

	output_error_stderr = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(output_error_window)), (gchar *)"stderr");
	if (shn_cfg.error_output_method == ERROR_OUTPUT_STDERR)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(output_error_stderr), TRUE);
	gtk_box_pack_start(GTK_BOX(output_vbox), output_error_stderr, FALSE, FALSE, 0);
	gtk_widget_show(output_error_stderr);

	output_error_devnull = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(output_error_window)), (gchar *)"/dev/null");
	if (shn_cfg.error_output_method == ERROR_OUTPUT_DEVNULL)
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(output_error_devnull), TRUE);
	gtk_box_pack_start(GTK_BOX(output_vbox), output_error_devnull, FALSE, FALSE, 0);
	gtk_widget_show(output_error_devnull);

	gtk_widget_show(output_vbox);
	gtk_widget_show(output_frame);

/* end error output box */

/* begin misc box */

	misc_frame = gtk_frame_new((gchar *)" Miscellaneous options ");
	gtk_container_set_border_width(GTK_CONTAINER(misc_frame), 5);

	misc_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(misc_vbox), 5);
	gtk_container_add(GTK_CONTAINER(misc_frame), misc_vbox);

	swap_bytes_toggle = gtk_check_button_new_with_label((gchar *)"Swap audio bytes on output (toggle if all you hear is static)");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(swap_bytes_toggle), shn_cfg.swap_bytes);
	gtk_box_pack_start(GTK_BOX(misc_vbox), swap_bytes_toggle, FALSE, FALSE, 0);
	gtk_widget_show(swap_bytes_toggle);

	verbose_toggle = gtk_check_button_new_with_label((gchar *)"Display debug info to stderr");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(verbose_toggle), shn_cfg.verbose);
	gtk_box_pack_start(GTK_BOX(misc_vbox), verbose_toggle, FALSE, FALSE, 0);
	gtk_widget_show(verbose_toggle);

	gtk_widget_show(misc_vbox);
	gtk_widget_show(misc_frame);

/* end misc box */

/* begin seek table path box */

	path_frame = gtk_frame_new((gchar *)" Alternate seek table file location ");
	gtk_container_set_border_width(GTK_CONTAINER(path_frame), 5);

	path_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(path_vbox), 5);
	gtk_container_add(GTK_CONTAINER(path_frame), path_vbox);

	path_label = gtk_label_new((gchar *)"Enter an alternate directory where .skt files can be found (searched\nwhen all other options fail, see the README file for more info):");
	gtk_misc_set_alignment(GTK_MISC(path_label), 0, 0);
	gtk_label_set_justify(GTK_LABEL(path_label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(path_vbox), path_label, FALSE, FALSE, 0);
	gtk_widget_show(path_label);

	path_entry_hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(path_entry_hbox), 5);
	gtk_box_pack_start(GTK_BOX(path_vbox), path_entry_hbox, TRUE, TRUE, 0);

	path_entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(path_entry), shn_cfg.seek_tables_path);
	gtk_box_pack_start(GTK_BOX(path_entry_hbox), path_entry, TRUE, TRUE, 0);
	gtk_widget_show(path_entry);

	path_browse = gtk_button_new_with_label("Browse");
	gtk_signal_connect(GTK_OBJECT(path_browse), "clicked", GTK_SIGNAL_FUNC(path_browse_cb), NULL);
	gtk_box_pack_start(GTK_BOX(path_entry_hbox), path_browse, FALSE, FALSE, 0);
	gtk_widget_show(path_browse);

	gtk_widget_show(path_entry_hbox);
	gtk_widget_show(path_vbox);
	gtk_widget_show(path_frame);

/* end seek table path box */

	gtk_box_pack_start(GTK_BOX(options_vbox), output_frame, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(options_vbox), path_frame, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(options_vbox), misc_frame, FALSE, FALSE, 0);

	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), options_vbox, gtk_label_new((gchar *)"Options"));

	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);

	ok = gtk_button_new_with_label((gchar *)"OK");
	gtk_signal_connect(GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(shn_configurewin_ok), NULL);
	GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), ok, TRUE, TRUE, 0);
	gtk_widget_show(ok);
	gtk_widget_grab_default(ok);

	cancel = gtk_button_new_with_label((gchar *)"Cancel");
	gtk_signal_connect_object(GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(shn_configurewin));
	GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), cancel, TRUE, TRUE, 0);
	gtk_widget_show(cancel);

	apply = gtk_button_new_with_label((gchar *)"Apply");
	gtk_signal_connect_object(GTK_OBJECT(apply), "clicked", GTK_SIGNAL_FUNC(shn_configurewin_apply), NULL);
	GTK_WIDGET_SET_FLAGS(apply, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), apply, TRUE, TRUE, 0);
	gtk_widget_show(apply);

	gtk_widget_show(bbox);
	gtk_widget_show(options_vbox);
	gtk_widget_show(notebook);
	gtk_widget_show(vbox);
	gtk_widget_show(shn_configurewin);
}

void message_box(char *message)
{
	GtkWidget *mbox_win,
		  *mbox_vbox1,
		  *mbox_vbox2,
		  *mbox_frame,
		  *mbox_label,
		  *mbox_bbox,
		  *mbox_ok;

	mbox_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_signal_connect(GTK_OBJECT(mbox_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &mbox_win);
	gtk_window_set_title(GTK_WINDOW(mbox_win), (gchar *)"Shorten file information");
	gtk_window_set_policy(GTK_WINDOW(mbox_win), FALSE, FALSE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(mbox_win), 10);

	mbox_vbox1 = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(mbox_win), mbox_vbox1);

	mbox_frame = gtk_frame_new((gchar *)" " PACKAGE " error ");
	gtk_container_set_border_width(GTK_CONTAINER(mbox_frame), 5);
	gtk_box_pack_start(GTK_BOX(mbox_vbox1), mbox_frame, FALSE, FALSE, 0);

	mbox_vbox2 = gtk_vbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(mbox_vbox2), 5);
	gtk_container_add(GTK_CONTAINER(mbox_frame), mbox_vbox2);

	mbox_label = gtk_label_new((gchar *)message);
	gtk_misc_set_alignment(GTK_MISC(mbox_label), 0, 0);
	gtk_label_set_justify(GTK_LABEL(mbox_label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(mbox_vbox2), mbox_label, TRUE, TRUE, 0);
	gtk_widget_show(mbox_label);

	mbox_bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(mbox_bbox), GTK_BUTTONBOX_SPREAD);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(mbox_bbox), 5);
	gtk_box_pack_start(GTK_BOX(mbox_vbox2), mbox_bbox, FALSE, FALSE, 0);

	mbox_ok = gtk_button_new_with_label((gchar *)"OK");
	gtk_signal_connect_object(GTK_OBJECT(mbox_ok), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(mbox_win));
	GTK_WIDGET_SET_FLAGS(mbox_ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(mbox_bbox), mbox_ok, TRUE, TRUE, 0);
	gtk_widget_show(mbox_ok);
	gtk_widget_grab_default(mbox_ok);

	gtk_widget_show(mbox_bbox);
	gtk_widget_show(mbox_vbox2);
	gtk_widget_show(mbox_frame);
	gtk_widget_show(mbox_vbox1);
	gtk_widget_show(mbox_win);
}

void display_shn_info(shn_file *tmp_file)
{
	char props[BUF_SIZE];
	char details[BUF_SIZE];
	char props_label[BUF_SIZE];
	char details_label[BUF_SIZE];
	GtkWidget *info_win,
		  *info_notebook,
		  *props_hbox,
		  *props_vbox,
		  *props_frame,
		  *props_label_left,
		  *props_label_right,
		  *details_hbox,
		  *details_vbox,
		  *details_frame,
		  *details_label_left,
		  *details_label_right,
		  *main_vbox,
		  *info_bbox,
		  *info_ok;

#ifdef HAVE_SNPRINTF
	snprintf(props_label,BUF_SIZE-1,
#else
	sprintf(props_label,
#endif
		" Properties for %s ",
		strrchr(tmp_file->wave_header.filename,'/') ?
		strrchr(tmp_file->wave_header.filename,'/') + 1 :
		tmp_file->wave_header.filename);

#ifdef HAVE_SNPRINTF
	snprintf(details_label,BUF_SIZE-1,
#else
	sprintf(details_label,
#endif
		" Details for %s ",
		strrchr(tmp_file->wave_header.filename,'/') ?
		strrchr(tmp_file->wave_header.filename,'/') + 1 :
		tmp_file->wave_header.filename);

#ifdef HAVE_SNPRINTF
	snprintf(props,BUF_SIZE-1,
#else
	sprintf(props,
#endif
		"\n"
		"%s\n"
		"%s\n"
		"%0.2f%% of original size\n"
		"\n"
		"%s\n"
		"%s\n"
		"%s\n"
		"\n"
		"%s\n"
		"%s"
		"\n",
		tmp_file->wave_header.m_ss,
		(tmp_file->vars.seek_table_entries == NO_SEEK_TABLE) ? "no":"yes",
		100.0 * ((double)tmp_file->wave_header.actual_size/(double)tmp_file->wave_header.total_size),
		(PROB_NOT_CD(tmp_file->wave_header))?"no":"yes",
		(PROB_NOT_CD(tmp_file->wave_header))?"n/a":((PROB_BAD_BOUND(tmp_file->wave_header))?"no":"yes"),
		(PROB_NOT_CD(tmp_file->wave_header))?"n/a":((PROB_TOO_SHORT(tmp_file->wave_header))?"no":"yes"),
		(PROB_HDR_NOT_CANONICAL(tmp_file->wave_header))?"yes":"no",
		(PROB_EXTRA_CHUNKS(tmp_file->wave_header))?"yes":"no"
		);

#ifdef HAVE_SNPRINTF
	snprintf(details,BUF_SIZE-1,
#else
	sprintf(details,
#endif
		"0x%04x (%s)\n"
		"%hu\n"
		"%hu\n"
		"%lu\n"
		"%lu\n"
		"%lu bytes/sec\n"
		"%hu\n"
		"%d bytes\n"
		"%lu bytes\n"
		"%lu bytes\n"
		"%lu bytes\n"
		"%lu bytes",
		tmp_file->wave_header.wave_format,
		format_to_str(tmp_file->wave_header.wave_format),
		tmp_file->wave_header.channels,
		tmp_file->wave_header.bits_per_sample,
		tmp_file->wave_header.samples_per_sec,
		tmp_file->wave_header.avg_bytes_per_sec,
		tmp_file->wave_header.rate,
		tmp_file->wave_header.block_align,
		tmp_file->wave_header.header_size,
		tmp_file->wave_header.data_size,
		tmp_file->wave_header.chunk_size,
		tmp_file->wave_header.total_size,
		tmp_file->wave_header.actual_size
		);

	unload_shn(tmp_file);

	info_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_signal_connect(GTK_OBJECT(info_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &info_win);
	gtk_window_set_title(GTK_WINDOW(info_win), (gchar *)"Shorten file information");
	gtk_window_set_policy(GTK_WINDOW(info_win), FALSE, FALSE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(info_win), 10);

	main_vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(info_win), main_vbox);

	info_notebook = gtk_notebook_new();
	gtk_container_add(GTK_CONTAINER(main_vbox), info_notebook);

/* begin properties page */

	props_vbox = gtk_vbox_new(FALSE, 10);

	props_frame = gtk_frame_new((gchar *)props_label);
	gtk_container_set_border_width(GTK_CONTAINER(props_frame), 10);
	gtk_box_pack_start(GTK_BOX(props_vbox), props_frame, FALSE, FALSE, 0);

	props_hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(props_hbox), 5);
	gtk_container_add(GTK_CONTAINER(props_frame), props_hbox);

	props_label_left = gtk_label_new((gchar *)
		"\n"
		"Length:\n"
		"Seekable:\n"
		"Compression ratio:\n"
		"CD-quality properties:\n"
		"  CD-quality:\n"
		"  Cut on sector boundary:\n"
		"  Long enough to be burned:\n"
		"WAVE properties:\n"
		"  Non-canonical header:\n"
		"  Extra RIFF chunks:"
		"\n"
		);

	props_label_right = gtk_label_new((gchar *)props);

	gtk_misc_set_alignment(GTK_MISC(props_label_left), 0, 0);
	gtk_label_set_justify(GTK_LABEL(props_label_left), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(props_hbox), props_label_left, TRUE, TRUE, 0);
	gtk_widget_show(props_label_left);

	gtk_misc_set_alignment(GTK_MISC(props_label_right), 0, 0);
	gtk_label_set_justify(GTK_LABEL(props_label_right), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(props_hbox), props_label_right, TRUE, TRUE, 0);
	gtk_widget_show(props_label_right);

/* end properties page */

/* begin details page */

	details_vbox = gtk_vbox_new(FALSE, 10);

	details_frame = gtk_frame_new((gchar *)details_label);
	gtk_container_set_border_width(GTK_CONTAINER(details_frame), 10);
	gtk_box_pack_start(GTK_BOX(details_vbox), details_frame, FALSE, FALSE, 0);

	details_hbox = gtk_hbox_new(FALSE, 10);
	gtk_container_set_border_width(GTK_CONTAINER(details_hbox), 5);
	gtk_container_add(GTK_CONTAINER(details_frame), details_hbox);

	details_label_left = gtk_label_new((gchar *)
		"WAVE format:\n"
		"Channels:\n"
		"Bits per sample:\n"
		"Samples per second:\n"
		"Average bytes per second:\n"
		"Rate (calculated):\n"
		"Block align:\n"
		"Header size:\n"
		"WAVE data size:\n"
		"Chunk size:\n"
		"Total size (chunk size + 8):\n"
		"Actual file size:"
		);

	details_label_right = gtk_label_new((gchar *)details);

	gtk_misc_set_alignment(GTK_MISC(details_label_left), 0, 0);
	gtk_label_set_justify(GTK_LABEL(details_label_left), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(details_hbox), details_label_left, TRUE, TRUE, 0);
	gtk_widget_show(details_label_left);

	gtk_misc_set_alignment(GTK_MISC(details_label_right), 0, 0);
	gtk_label_set_justify(GTK_LABEL(details_label_right), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(details_hbox), details_label_right, TRUE, TRUE, 0);
	gtk_widget_show(details_label_right);

/* end details page */

	info_bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(info_bbox), GTK_BUTTONBOX_SPREAD);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(info_bbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), info_bbox, FALSE, FALSE, 0);

	info_ok = gtk_button_new_with_label((gchar *)"OK");
	gtk_signal_connect_object(GTK_OBJECT(info_ok), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(info_win));
	GTK_WIDGET_SET_FLAGS(info_ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(info_bbox), info_ok, TRUE, TRUE, 0);
	gtk_widget_show(info_ok);
	gtk_widget_grab_default(info_ok);

	gtk_notebook_append_page(GTK_NOTEBOOK(info_notebook), props_vbox, gtk_label_new((gchar *)"Properties"));
	gtk_notebook_append_page(GTK_NOTEBOOK(info_notebook), details_vbox, gtk_label_new((gchar *)"Details"));

	gtk_widget_show(props_frame);
	gtk_widget_show(props_hbox);
	gtk_widget_show(props_vbox);
	gtk_widget_show(details_frame);
	gtk_widget_show(details_hbox);
	gtk_widget_show(details_vbox);
	gtk_widget_show(info_notebook);
	gtk_widget_show(main_vbox);
	gtk_widget_show(info_bbox);
	gtk_widget_show(info_win);
}
