/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-vifi.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : routines which allow to change the FILEINFO information  *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcMakePatAnonymous()     - Make patient anonymous       *
 *                MdcGivePatInformation()   - Give patient information     *
 *                MdcEditFI()               - Edit FILEINFO structure      *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-vifi.c,v 1.1.1.1 2000/10/28 16:50:56 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

void MdcMakePatAnonymous(FILEINFO *fi)
{

  MdcStringCopy(fi->patient_sex,"X",1);
  MdcStringCopy(fi->patient_name,"X",1);
  MdcStringCopy(fi->patient_id,"X",1);
  MdcStringCopy(fi->study_descr,"X",1);
  MdcStringCopy(fi->study_name,"X",1);

}

void MdcGivePatInformation(FILEINFO *fi)
{
   int a;

   MdcPrintLine('-',MDC_FULL_LENGTH);
   printf("\tPATIENT/STUDY  INFORMATION\t\tFILE: %s\n",fi->ifname);
   MdcPrintLine('-',MDC_FULL_LENGTH);
   printf("\n\tNote: all strings are limited to %d characters\n\n",MDC_MAXSTR);
   printf("\n\tGive patient name [%s]: ",fi->patient_name);
   if (! MdcPutDefault(mdcbufr)) 
     MdcStringCopy(fi->patient_name,mdcbufr,MDC_MAXSTR);
  
   printf("\n\tGive patient id [%s]: ",fi->patient_id);
   if (! MdcPutDefault(mdcbufr)) 
     MdcStringCopy(fi->patient_id,mdcbufr,MDC_MAXSTR);

   printf("\n\tSelect patient sex [%s]:\n",fi->patient_sex);
   printf("\n\t\t   1    ->  male");
   printf("\n\t\t   2    ->  female");
   printf("\n\t\t   3    ->  other");
   printf("\n\t\t<enter> ->  default"); 
   printf("\n\n\tYour choice? ");
   if (! MdcPutDefault(mdcbufr)) {
     a = atoi(mdcbufr); 
     switch (a) {
       case 1 : MdcStringCopy(fi->patient_sex,"M",1); break;
       case 2 : MdcStringCopy(fi->patient_sex,"F",1); break;
       default: MdcStringCopy(fi->patient_sex,"O",1);
     }
   }
   

   printf("\n\tGive study description [%s]: ",fi->study_descr);
   if (! MdcPutDefault(mdcbufr)) 
     MdcStringCopy(fi->study_descr,mdcbufr,MDC_MAXSTR);

   printf("\n\tGive study name/p-number [%s]: ",fi->study_name);
   if (! MdcPutDefault(mdcbufr))
     MdcStringCopy(fi->study_name,mdcbufr,MDC_MAXSTR);

   MdcPrintLine('-',MDC_FULL_LENGTH);
}


char *MdcEditFI(FILEINFO *fi)
{
   IMG_DATA *id=NULL;
   Uint32 i, number, dflt, dim[8];
   Int8 a, LAST_FOUND=MDC_NO, PET_MOD=MDC_YES;
   float pixel_size, slice_width, slice_spacing;
   char *msg, modality[3], *badvalue="Bad dim[]-value supplied";

   MdcPrintLine('#',MDC_FULL_LENGTH);
   printf("\tEDIT  FILEINFO  STRUCTURE\t\tFILE: %s\n",fi->ifname);
   MdcPrintLine('#',MDC_FULL_LENGTH);

   /* patient/slice orientation */
   MdcPrintLine('-',MDC_HALF_LENGTH);
   printf("\tPATIENT/SLICE  ORIENTATION\n");
   MdcPrintLine('-',MDC_HALF_LENGTH);

   printf("\n\tSelect Patient/Slice orientation:\n");

   for (a=0; a < 13; a++) 
      printf("\n\t\t %2hd  ->  %s",a,MdcGetStrPatSlOrient(a));
 
   printf("\n\n\tYour choice [%d]? ",fi->image[0].pat_slice_orient);

   if (MdcPutDefault(mdcbufr)) a = fi->image[0].pat_slice_orient;
   else a = (Int8)atoi(mdcbufr);

   MdcPrintLine('-',MDC_HALF_LENGTH);

   for (i=0; i < fi->number; i++) {
     id = &fi->image[i];
      id->pat_slice_orient = (Int8)a;
      strcpy(id->pat_pos,MdcGetStrPatientPos(id->pat_slice_orient));
      strcpy(id->pat_orient,MdcGetStrPatientOrient(id->pat_slice_orient));
   }


   /* pixel/slice dimensions */
   MdcPrintLine('-',MDC_HALF_LENGTH);
   printf("\tPIXEL/SLICE DIMENSIONS\n");
   MdcPrintLine('-',MDC_HALF_LENGTH);


   /* fill in default values */
   id = &fi->image[0];
   pixel_size   = id->pixel_xsize;
   slice_width  = id->slice_width;
   slice_spacing= id->slice_spacing;

   printf("\n\tNote: The following entries require float values");
   printf("\n\t      Examples: 10.0 or 1.0e+1\n");

   printf("\n\tGive pixel size in mm [%e]: ",pixel_size);
   if (! MdcPutDefault(mdcbufr)) pixel_size = (float)atof(mdcbufr);

   printf("\n\tGive slice width in mm [%e]: ",slice_width);
   if (! MdcPutDefault(mdcbufr)) slice_width = (float)atof(mdcbufr);

   printf("\n\tGive centre-centre slice separation in mm [%e]: ",slice_spacing);
   if (! MdcPutDefault(mdcbufr)) slice_spacing = (float)atof(mdcbufr);

   MdcPrintLine('-',MDC_HALF_LENGTH);

   for (i=0; i<fi->number; i++) {

      id = &fi->image[i];

      id->pixel_xsize = pixel_size;
      id->pixel_ysize = pixel_size;
      id->slice_width = slice_width;
      id->slice_spacing = slice_spacing;

   }

   fi->pixdim[1] = pixel_size;
   fi->pixdim[2] = pixel_size;
   fi->pixdim[3] = slice_width;


   /* array dimensions */
   MdcPrintLine('-',MDC_HALF_LENGTH);
   printf("\tARRAY  DIMENSIONS\n");
   MdcPrintLine('-',MDC_HALF_LENGTH);

   printf("\n\tNote: Each array entry must be a 1-based integer and the");
   printf("\n\t      product of dim[]-values = total numer of images\n");


   /* fill in some defaults */
   dflt = fi->number;
   for (i=0; i<8; i++) dim[i] = 1;

   printf("\n\t            ( PET   |    NucMed    )    default");
   printf("\n\t             ----------------------     -------");
   printf("\n\tGive dim[3] (planes | (time) slices)     [%u]: ",dflt);
   if (MdcPutDefault(mdcbufr)) dim[3] = dflt;
   else dim[3] = (Uint32)atoi(mdcbufr);
   if (dim[3] == 0) return(badvalue);
 
   dflt /= dim[3]; if(dflt == 0) return(badvalue);
   printf("\n\tGive dim[4] (frames | time slots|phases) [%u]: ",dflt);
   if (MdcPutDefault(mdcbufr)) dim[4] = dflt;
   else dim[4] = (Uint32)atoi(mdcbufr);
   if (dim[4] == 0) return(badvalue);

   dflt /= dim[4]; if (dflt == 0) return(badvalue); 
   printf("\n\tGive dim[5] (gates  | R-R intervals)     [%u]: ",dflt);
   if (MdcPutDefault(mdcbufr)) dim[5] = dflt;
   else dim[5] = (Uint32)atoi(mdcbufr);
   if (dim[5] == 0) return(badvalue);

   dflt /= dim[5]; if (dflt == 0) return(badvalue);
   printf("\n\tGive dim[6] (beds   | detector heads)    [%u]: ",dflt);
   if (MdcPutDefault(mdcbufr)) dim[6] = dflt;
   else dim[6] = (Uint32)atoi(mdcbufr);
   if (dim[6] == 0) return(badvalue);

   dflt /= dim[6]; if (dflt == 0) return(badvalue);
   printf("\n\tGive dim[7] (????   | energy windows)    [%u]: ",dflt);
   if (MdcPutDefault(mdcbufr)) dim[7] = dflt;
   else dim[7] = (Uint32)atoi(mdcbufr);
   if (dim[7] == 0) return(badvalue);

   number = 1;
   for (i=3; i<8; i++) number*=dim[i];

   if (number != fi->number) return(badvalue);

   /* ok, use the values */ 
   for (i=7; i>=3; i--) {
      fi->dim[i] = dim[i];
      if ((LAST_FOUND == MDC_NO) && (dim[i] > 1)) {
        fi->dim[0] = i; LAST_FOUND = MDC_YES;
      }
   }

   MdcPrintLine('-',MDC_HALF_LENGTH);


   /* study information */
   MdcPrintLine('-',MDC_HALF_LENGTH);
   printf("\tSTUDY  PARAMETERS\n");
   MdcPrintLine('-',MDC_HALF_LENGTH);

   printf("\n\tReconstructed ");
   if (fi->reconstructed == MDC_YES) {
     printf("([y]/n)");
   }else{
     printf("(y/[n])");
   }
   printf(" ? ");

   if (!MdcPutDefault(mdcbufr)) {
     if ( mdcbufr[0] == 'y' || mdcbufr[0] == 'Y' ) {
       fi->reconstructed = MDC_YES;
     }else if ( mdcbufr[0] == 'n' || mdcbufr[0] == 'N' ) {
       fi->reconstructed = MDC_NO;
     }
   }

   if (strncmp(fi->image[0].image_mod,"PT",2) == 0) {
     strcpy(modality,"PT"); PET_MOD = MDC_YES;
   }else{
     strcpy(modality,"NM"); PET_MOD = MDC_NO;
   }
   printf("\n\tSelect Modality:\n");
   printf("\n\t\t   1    ->  NM (NucMed)");
   printf("\n\t\t   2    ->  PT (PET)");
   printf("\n\t\t<enter> ->  default");
   printf("\n\n\tYour choice ");
   if (PET_MOD == MDC_NO) printf("[1] ? ");
   else printf("[2] ? ");

   if (! MdcPutDefault(mdcbufr)) {
     a = atoi(mdcbufr);
     switch (a) {
       case 1: strcpy(modality,"NM"); break;
       case 2: strcpy(modality,"PT"); break;
     } 
   } 

   printf("\n\tSelect Acquisition type:\n"); 
   for (a=0; a < 4; a++)
      printf("\n\t\t %2hd  ->  %s",a,MdcGetStrAcquisition(a));

   printf("\n\n\tYour choice [%d]? ",fi->acquisition_type);

   if (!MdcPutDefault(mdcbufr)) {
     fi->acquisition_type = (Int16)atoi(mdcbufr);
   }
   
   MdcPrintLine('-',MDC_HALF_LENGTH);



   /* some final completions */
   msg = MdcImagesPixelFiddle(fi);
   if (msg != NULL) return(msg);

   for (i=0; i < fi->number; i++) {
      strcpy(fi->image[i].image_mod,modality);
      MdcFillImgPos(fi,i,i%fi->dim[3],0.0);
      MdcFillImgOrient(fi,i);
   }

   MdcPrintLine('#',MDC_FULL_LENGTH);

   return(NULL);
}   

