/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-raw.c                                                       *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : read (interactive) and write raw images                  *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcReadRAW()          - Read raw images interactive      *
 *                MdcWriteRAW()         - Write raw images to file         * 
 *                MdcInitRawPrevInput() - Initialize previous inputs       *
 *                                                                         *
 * Notes        : Reading is an interactive process to determine           *
 *                the headersize to skip and the pixel data type           *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-raw.c,v 1.1.1.1 2000/10/28 16:51:11 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

MdcRawInputStruct mdcrawinput;
MdcRawPrevInputStruct mdcrawprevinput;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/
void MdcInitRawPrevInput(void)
{
   MdcRawPrevInputStruct *prev = &mdcrawprevinput;

   prev->XDIM=0; prev->YDIM=0; prev->NRIMGS=1;
   prev->GENHDR=0; prev->IMGHDR=0; prev->ABSHDR=0;
   prev->PTYPE = BIT16_S;
   prev->DIFF  = MDC_NO; prev->HDRREP=MDC_NO; prev->PSWAP=MDC_NO;
}


/* read raw images */
char *MdcReadRAW(FILEINFO *fi)
{
  MdcRawInputStruct *input = &mdcrawinput;
  MdcRawPrevInputStruct *prev = &mdcrawprevinput;
  IMG_DATA *id=NULL;
  Uint32 i, p, bytes;
  int AGAIN;
  double *pix=NULL;
  char *err=NULL;

#if XSUPPORTED
  if (XMDC_MEDCON) {
    XMdcBeginProgressBar("Reading RAW:");
  }
#endif

  if (XMDC_MEDCON == MDC_NO) 
  do { /* ask command-line info */
    input->gen_offset=0;
    input->img_offset=0;
    input->REPEAT=MDC_NO;
    input->DIFF=MDC_NO;
    input->abs_offset=NULL;
    AGAIN=MDC_NO;

    MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;
 
    MdcPrintLine('-',MDC_FULL_LENGTH);
    printf("\tINTERACTIVE  PROCEDURE\n");
    MdcPrintLine('-',MDC_FULL_LENGTH);

    fi->number = prev->NRIMGS;

    printf("\n\tFilename: %s\n\n",fi->ifname);
    printf("\tNumber of images [%u]? ",fi->number);
    if (!MdcPutDefault(mdcbufr)) 
      fi->number = (Uint32)atol(mdcbufr); prev->NRIMGS=fi->number;
    if (fi->number == 0) return("RAW  No images specified");

    if (!MdcGetStructID(fi)) return("RAW  Bad malloc IMG_DATA structs");

    printf("\tGeneral header offset to binary data [%u bytes]? ",prev->GENHDR);
      if (MdcPutDefault(mdcbufr)) input->gen_offset = prev->GENHDR; 
      else{
        input->gen_offset = (Uint32)atol(mdcbufr);
        prev->GENHDR = input->gen_offset;
      }
    printf("\tImage   header offset to binary data [%u bytes]? ",prev->IMGHDR);
      if (MdcPutDefault(mdcbufr)) input->img_offset = prev->IMGHDR;
      else{
        input->img_offset = (Uint32)atol(mdcbufr);
        prev->IMGHDR = input->img_offset;
      }
    printf("\tImage   header repeated before each image ");
    sprintf(mdcbufr,"%s",MdcGetStrYesNo(prev->HDRREP));
    printf("[%s]? ",mdcbufr); 
      if (!MdcPutDefault(mdcbufr)) { 
        if (mdcbufr[0]=='y' || mdcbufr[0]=='Y') {
          input->REPEAT = MDC_YES; prev->HDRREP = MDC_YES;
        }else{
          input->REPEAT = MDC_NO;  prev->HDRREP = MDC_NO;
        }
      }else{
         input->REPEAT = prev->HDRREP;
      }
      
    printf("\tSwap the pixel bytes ");
    sprintf(mdcbufr,"%s",MdcGetStrYesNo(prev->PSWAP));
    printf("[%s]? ",mdcbufr);
      if (!MdcPutDefault(mdcbufr)) {
        if (mdcbufr[0]=='y' || mdcbufr[0]=='Y') {
          MDC_FILE_ENDIAN = !MDC_HOST_ENDIAN; prev->PSWAP = MDC_YES;
         }else{
          MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;  prev->PSWAP = MDC_NO; 
         }
      }else{
        if (prev->PSWAP == MDC_YES) {
          MDC_FILE_ENDIAN = !MDC_HOST_ENDIAN;
        }else{
          MDC_FILE_ENDIAN = MDC_HOST_ENDIAN;
        }
      }
          
    printf("\tSame characteristics for all images ");
    sprintf(mdcbufr,"%s",MdcGetStrYesNo(!prev->DIFF));
    printf("[%s]? ",mdcbufr);
      if (!MdcPutDefault(mdcbufr)) {
        if (mdcbufr[0]=='n' || mdcbufr[0]=='N') {
          input->DIFF=MDC_YES; prev->DIFF = MDC_YES;
        }else{
          input->DIFF=MDC_NO;  prev->DIFF = MDC_NO;
        }
      }else{
        input->DIFF = prev->DIFF;
      }

    /* malloc abs_offset[] */
    input->abs_offset=(Uint32 *)malloc(fi->number*sizeof(Uint32));
    if ( input->abs_offset == NULL ) {
      return("Couldn't malloc abs_offset[]");
    }else{
      memset(input->abs_offset,0,fi->number*sizeof(Uint32));
    } 
   
 
    for (i=0; i<fi->number; i++) {
#if XSUPPORTED
       if (XMDC_MEDCON) {
         pvalue += 1./(float)fi->number;
         XMdcUpdateProgressBar(NULL);
       }
#endif
 
       id = &fi->image[i];

       if (input->DIFF) {
         printf("\n\tIMAGE #%03u\n",i+1);
         printf("\t----------\n");
       }else if (i==0) {
         printf("\n\tALL IMAGES\n"); 
         printf("\t----------\n");
       }

       /* put default */
       if (i==0) id->type = prev->PTYPE;

       if (input->DIFF || (!input->DIFF && i==0)) {

         printf("\tAbsolute offset in bytes [%u]? ",prev->ABSHDR);
           if (MdcPutDefault(mdcbufr)) input->abs_offset[i] = prev->ABSHDR;
           else{ 
             input->abs_offset[i] = (Uint32)atol(mdcbufr);
             prev->ABSHDR = input->abs_offset[i];
           }
         printf("\tImage columns [%u]? ",prev->XDIM); 
           if (MdcPutDefault(mdcbufr)) id->width = prev->XDIM;
           else{ id->width = (Uint32)atol(mdcbufr); prev->XDIM = id->width; }
           
           if(id->width == 0) return("RAW  No width specified");
         printf("\tImage rows    [%u]? ",prev->YDIM); 
           if (MdcPutDefault(mdcbufr)) id->height = prev->YDIM;
           else{ id->height = (Uint32)atol(mdcbufr); prev->YDIM = id->height; }
           if (id->height == 0) return("RAW  No height specified");
         printf("\tPixel data type:\n\n");
         printf("\t\t  1  ->  bit\n");
         printf("\t\t  2  ->  Int8 \t\t  3 -> Uint8\n");
         printf("\t\t  4  ->  Int16\t\t  5 -> Uint16\n");
         printf("\t\t  6  ->  Int32\t\t  7 -> Uint32\n");
#ifdef HAVE_8BYTE_INT
         printf("\t\t  8  ->  Int64\t\t  9 -> Uint64\n");
#endif
         printf("\t\t 10  ->  float\t\t 11 -> double\n");
         printf("\t\t 12  ->  ascii\n\n");
         printf("\tYour choice [%hu]? ", prev->PTYPE);

         if (MdcPutDefault(mdcbufr)) id->type = prev->PTYPE;
         else{ id->type = (Int16)atoi(mdcbufr); prev->PTYPE = id->type; }
         printf("\n"); 
       }else{
         id->width = prev->XDIM;
         id->height = prev->YDIM; 
         id->type  = prev->PTYPE;
         input->abs_offset[i]= prev->ABSHDR;
       }
 
       switch (id->type) {
         case  BIT1   : 
         case  BIT8_S :
         case  BIT8_U :
         case  BIT16_S:
         case  BIT16_U:
         case  BIT32_S:
         case  BIT32_U:
#ifdef HAVE_8BYTE_INT
         case  BIT64_S:
         case  BIT64_U:
#endif
         case  FLT32  :
         case  FLT64  : 
         case  ASCII  : id->bits = MdcType2Bits(id->type); break;
         default      : return("RAW  Unsupported data type");
       }
     }

     fi->endian = MDC_FILE_ENDIAN;
     fi->dim[0] = 3;
     fi->dim[3] = fi->number;

     MdcPrintImageLayout(fi,input->gen_offset,input->img_offset,
                            input->abs_offset,input->REPEAT);

     printf("\n\tRedo input [no]? "); mdcbufr[0]='n'; AGAIN=MDC_NO;
      if (!MdcPutDefault(mdcbufr))
        if (mdcbufr[0]=='y' || mdcbufr[0]=='Y') {
          AGAIN=MDC_YES;
          MdcFreeIDs(fi);
          MdcFree(input->abs_offset);
        }

  }while ( AGAIN );

  if (MDC_VERBOSE) MdcPrntMesg("RAW  Reading <%s> ...",fi->ifname);

  fseek(fi->ifp,input->gen_offset,SEEK_SET);
 
  /* read the images */
  fi->type = FLT64; /* read ascii as double */
  for (i = 0; i<fi->number; i++) {

     if ( i==0  || input->REPEAT) fseek(fi->ifp,input->img_offset,SEEK_CUR);

     if (input->abs_offset[i] != 0)
       fseek(fi->ifp,input->abs_offset[i],SEEK_SET);

     id = &fi->image[i];
     bytes = id->width * id->height * MdcType2Bytes(id->type);
     id->buf = MdcGetImgBuffer(bytes);
     if (id->buf == NULL) {
       MdcFree(input->abs_offset);
       return("RAW  Bad malloc image buffer");
     }
     if (id->type == ASCII) {
       pix = (double *)id->buf;
       for (p=0; p < (id->width * id->height); p++) {
          fscanf(fi->ifp,"%le",&pix[p]);
          if (ferror(fi->ifp)) {
            err=MdcHandleTruncated(fi,i+1,MDC_YES);
            if (err != NULL) {
              MdcFree(input->abs_offset);
              return(err);
            }
            break;
          }
       }
       id->type = FLT64; /* read ascii as double */
     }else{
      if (fread(id->buf,1,bytes,fi->ifp) != bytes) {
        err=MdcHandleTruncated(fi,i+1,MDC_YES); 
        if (err != NULL) {
          MdcFree(input->abs_offset);
          return(err);
        }
      }
     } 
     if (id->type == BIT1)  {
       MdcMakeBIT8_U(id->buf, fi, i);
       id->type = BIT8_U;
       id->bits = MdcType2Bits(id->type);
       if (i==0) { fi->type = id->type; fi->bits = id->bits; } 
     }
     if (fi->truncated) break;
  }

  MdcFree(input->abs_offset);

  MdcFileClose(fi->ifp);

  if (fi->truncated) return("RAW  Truncated image file");

  return NULL;
}

char *MdcWriteRAW(FILEINFO *fi)
{
  IMG_DATA *id;
  Uint32 size, i, p;
  Uint8 *new_buf=NULL, *pbuf=NULL;
  double pix;

  MDC_FILE_ENDIAN = MDC_WRITE_ENDIAN;


  switch (fi->rawconv) {
   case MDC_FRMT_RAW:   
       if (XMDC_MEDCON == MDC_NO) 
         MdcDefaultName(MDC_FRMT_RAW,fi->ofname,fi->ifname);
       break;
   case MDC_FRMT_ASCII: 
       if (XMDC_MEDCON == MDC_NO)
         MdcDefaultName(MDC_FRMT_ASCII,fi->ofname,fi->ifname);
       break;
   default: return("Internal ## Improper `fi->rawconv' value");
  }

#if XSUPPORTED
  if (XMDC_MEDCON) {
    switch (fi->rawconv) {
     case MDC_FRMT_RAW  : XMdcBeginProgressBar("Writing RAW:");   break;
     case MDC_FRMT_ASCII: XMdcBeginProgressBar("Writing ASCII:"); break;
    } 
  }
#endif
  if (MDC_VERBOSE) MdcPrntMesg("RAW  Writing <%s> ...",fi->ofname);

  if (MdcFileExists(fi->ofname))
    return("RAW  File exists!!");
  if ( (fi->ofp=fopen(fi->ofname,"wb")) == NULL )
    return("RAW  Couldn't open file");


  /* check some supported things */
  if (MDC_FORCE_INT != MDC_NO) {
    /* Sorry, no message.  The user should know ... */ 
  }else if (MDC_QUANTIFY || MDC_CALIBRATE) {
    if (fi->rawconv == MDC_FRMT_RAW) {
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayWarn("RAW  Quantification to `float' type");
      }else
#endif
      {
      MdcPrntWarn("RAW  Quantification to `float' type");
      }
    }
  }


  for (i=0; i<fi->number; i++) {
#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif
     id = &fi->image[i];
     size = id->width * id->height;
     if (fi->rawconv == MDC_FRMT_RAW) {         /* RAW Binary */
       if (MDC_FORCE_INT != MDC_NO) {
         switch (MDC_FORCE_INT) {
          case BIT8_U : 
              new_buf=MdcGetImgBIT8_U(fi,i);
              if (new_buf == NULL) return("RAW  Bad malloc Uint8 buffer");
              break;
          case BIT16_S:
              new_buf=MdcGetImgBIT16_S(fi,i);
              if (new_buf == NULL) return("RAW  Bad malloc Int16 buffer");
              break;
          default:
              new_buf=MdcGetImgBIT16_S(fi,i);
              if (new_buf == NULL) return("RAW  Bad malloc Int16 buffer");
         }

         if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN)
           MdcMakeImgSwapped(new_buf,fi,i,id->width,id->height,MDC_FORCE_INT);
         if (fwrite(new_buf,MdcType2Bytes(MDC_FORCE_INT),size,fi->ofp) != size){
           MdcFree(new_buf);
           return("RAW  Bad write raw image");
         } 
       }else if (MDC_QUANTIFY || MDC_CALIBRATE) {
         new_buf=MdcGetImgFLT32(fi,i);
         if (new_buf == NULL) return("RAW  Quantification failed!");
         if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN) 
           MdcMakeImgSwapped(new_buf,fi,i,id->width,id->height,FLT32);
         if (fwrite(new_buf,MdcType2Bytes(FLT32),size,fi->ofp) != size) {
           MdcFree(new_buf);
           return("RAW  Bad write image file");
         }
       }else{ /* same pixel type */
        if (MDC_FILE_ENDIAN != MDC_HOST_ENDIAN) {
          new_buf = MdcGetImgSwapped(fi,i);
          if (fwrite(new_buf,MdcType2Bytes(id->type),size,fi->ofp) != size) {
            MdcFree(new_buf);
            return("RAW Bad write image file");
          }
        }else if (fwrite(id->buf,MdcType2Bytes(id->type),size,fi->ofp) != size)
                return("RAW  Bad write image file");
       }
     }else{                                     /* RAW Ascii  */
       if (MDC_QUANTIFY || MDC_CALIBRATE) {
         new_buf=MdcGetImgFLT32(fi,i);
         if (new_buf == NULL) return("RAW  Quantification failed!");

         pbuf = new_buf;
         for (p=0; p < size; p++, pbuf+=MdcType2Bytes(FLT32)) {
            pix = MdcGetDoublePixel(pbuf, FLT32);
            fprintf(fi->ofp," %+e",pix);
            if ( ((p+1) % id->width) == 0 ) fprintf(fi->ofp,"\n");
         }
       }else{
         pbuf = id->buf;
         for (p=0; p < size; p++, pbuf+=MdcType2Bytes(id->type)) {
            pix = MdcGetDoublePixel(pbuf, id->type);
            fprintf(fi->ofp," %+e",pix);
            if ( ((p+1) % id->width) == 0 ) fprintf(fi->ofp,"\n");
         }
       }
     }

     MdcFree(new_buf); /* free when allocated */
  }

  MdcFileClose(fi->ofp);
 
  return NULL;

}
