/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-fancy.c                                                     *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : Nice output, edit strings & print defaults               *
 *                                                                         * 
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcPrintLine()          - Print a full or half line      *
 *                MdcPrintChar()          - Print a char                   *
 *                MdcPrintStr()           - Print a string                 *
 *                MdcPrintBoxLine()       - Print horizontal line of a box *
 *                MdcPrintYesNo()         - Print Yes, No or Unknown       *
 *                MdcPrintImageLayout()   - Print layout of a raw image    *
 *                MdcLowStr()             - Make string lower case         *
 *                MdcKillSpaces()         - Remove first/last spaces       *
 *                MdcRemoveEnter()        - Remove <enter> from string     *
 *                MdcGetStrInput()        - Get string & remove <enter>    *
 *                MdcGetSafeString()      - Copy & add terminating char    *
 *                MdcPutDefault()         - Get (default) answer           *
 *                MdcHandleEcatList()     - Get a list in ecat style       *
 *                MdcHandleNormList()     - Get a list in normal style     *
 *                MdcGetPixelRange()      - Get the col/row ranges of list *
 *                MdcHandlePixelList()    - Get a list of pixels           *
 *                MdcGetStrAcquisition()  - Get string for acquisition type*
 *                MdcGetStrRawConv()      - Get string of raw type         *
 *                MdcGetStrEndian()       - Get string of endian type      *
 *                MdcGetStrCompression()  - Get string of compression type *
 *                MdcGetStrPixelType()    - Get string of pixel type       *
 *                MdcGetStrColorMap()     - Get string of colormap         *
 *                MdcGetStrYesNo()        - Get string "yes" or "no"       *
 *                MdcGetStrPatSlOrient()  - Get string patient/slice orient*
 *                MdcGetStrPatientPos()   - Get string patient position    *
 *                MdcGetStrPatientOrient()- Get string patient orientation *
 *                MdcGetStrPatientRot()   - Get string patient rotation    * 
 *                MdcGetStrSliceOrient()  - Get string slice   orientation *
 *                MdcGetStrRotation()     - Get string rotation direction  *
 *                MdcGetStrMotion()       - Get string detector motion     *
 *                MdcGetLibLongVersion()  - Get string of library version  *
 *                MdcGetLibShortVersion() - Get string of short   version  *
 *                MdcCheckStrSize()       - Check if we can add a string   *
 *                MdcMakeScanInfoStr()    - Make string with scan info     *
 *                MdcIsDigit()            - Test if char is a digit        *
 *                MdcWaitForEnter()       - Wait until <enter> key press   *
 *                MdcGetSelectionType()   - Get select type (norm,ecat,...)*
 *                MdcFlushInput()         - Flush the input stream         *
 *                MdcWhichDecompress()    - Give supported decompression   *
 *                MdcWhichCompression()   - Give compression type of file  *
 *                MdcAddCompressionExt()  - Add  compression extension     *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-fancy.c,v 1.1.1.1 2000/10/28 16:51:00 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "m-fancy.h"

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

void MdcPrintLine(char c, int length)
{
  int i;

  for (i=0; i<length; i++) printf("%c",c);
  printf("\n");

}

void MdcPrintChar(int c)
{
  if (c == '\0') printf("<null>");
  else if (c==9 || c==13 || c==10) putchar(c);
  else if (c >= 32) putchar(c);
  else if (c==EOF) printf("<EOF>");
  else printf("<%u>",c);
}

void MdcPrintStr(char *str)
{
  int t=strlen(str);

  if ( t == 0 ) printf("<null>");
  else printf("%s",str);

  printf("\n");

}

void MdcPrintBoxLine(char c, int t)
{
  int i;

  printf("\t\t#");
  for (i=-1;i<=t;i++) printf("%c",c);
  printf("#\n");
  
}

void MdcPrintYesNo(int value )
{
  switch ( value ) {
    case MDC_NO : printf("(= No)\n");      break;
    case MDC_YES: printf("(= Yes)\n");     break;
    default     : printf("(= Unknown)\n"); break;
  }
}

void MdcPrintImageLayout(FILEINFO *fi, Uint32 gen, Uint32 img
                                     , Uint32 *abs, int repeat)
{
  IMG_DATA *id;
  Uint32 i;

  MdcPrintLine('-',MDC_FULL_LENGTH);
  printf("\t\t\tSUMMARY OF IMAGE LAYOUT\n");
  MdcPrintLine('-',MDC_FULL_LENGTH);

  if ((gen==0) && (img==0))  MdcPrintBoxLine('-',MDC_BOX_SIZE);
  if (gen!=0) {
    MdcPrintBoxLine('-',MDC_BOX_SIZE);
    printf("\t\t| General Header   |  \t(%u)\n",gen);
    MdcPrintBoxLine('-',MDC_BOX_SIZE);
  }
  for (i=0; i<fi->number; i++) {
     id = &fi->image[i];
     if ( ((i==0) && (img>0)) || (repeat) ) {
       if ( ! ((i==0) && (gen>0)) )MdcPrintBoxLine('-',MDC_BOX_SIZE); 
       printf("\t\t| Image   Header   |  \t(%u)\n",img);
       MdcPrintBoxLine('-',MDC_BOX_SIZE);
     }
     printf("\t\t! Image #%-4u ",i+1);
     if (fi->endian != MDC_HOST_ENDIAN) printf("swap !"); 
     else printf("     !");
     printf("\t(%ux%ux%u)",id->width,id->height,MdcType2Bytes(id->type));
     if (abs!=NULL) if (abs[i] > 0) printf("\tOFFSET: %u",abs[i]);
     printf("\n");

  }
  MdcPrintBoxLine('-',MDC_BOX_SIZE);
}

void MdcLowStr(char *str)
{ 
  char *c;

  c=str;

  while(*c) { *c=tolower((int)*c); c++; }

}


void MdcKillSpaces(char string[]) /* kill first and last spaces */
{
  int i=0, shift=0, length;

  length = strlen(string);

  if (length > 0) { 
    /* kill the first spaces */
    while (isspace((int)string[i])) { 
         if (i < length) {
           i+=1; shift+=1; 
         }else break;
    }
    if (shift) for (i=0; i<=length; i++) string[i] = string[i+shift];

    /* kill the last  spaces */
    length = strlen(string);
    if (length > 0) { 
      i = length - 1;
      while (isspace((int)string[i])) { 
           if (i > 0 ) {
             string[i] = '\0'; i-=1;
           }else break;
      }
    }
  } 
}

void MdcRemoveEnter(char string[])
{
  char *p;

  p = strchr(string,'\r'); if (p != NULL) p[0] = '\0';
  p = strchr(string,'\n'); if (p != NULL) p[0] = '\0';

} 

void MdcGetStrInput(char string[], Uint32 maxchars)
{ 
  fscanf(stdin,"%s",string); 
  string[maxchars-1]='\0';         /* make sure it's null terminated */
}
 
void MdcGetSafeString(char *dest, char *src, Uint32 length, Uint32 maximum)
{
   Uint32 MAX = maximum - 1; /* let's be really safe */

   if (length < MAX) {
     memcpy(dest,src,length);
     dest[length]='\0';
   }else{
     memcpy(dest,src,MAX);
     dest[MAX]='\0';
   }
} 
                    /* string[] = MDC_2KB_OFFSET               */
int MdcPutDefault(char string[]) /* 1=default  or 0=no default */
{
  fgets(string,MDC_2KB_OFFSET-1,stdin);
  if (string[0] == '\n' || string[0] == '\r') return(1);
  MdcKillSpaces(string);
  return(0);

}

int MdcHandleEcatList(char *s, Uint32 **item, Uint32 max)
{

  Uint32 a1, a2, tmp, i;

  while (1) {

    if (atol(s) < 0 ) { break; } 

    if ( strstr(s,"...") != 0 ) { /* a range ... */
      sscanf(s,"%u...%u",&a1,&a2);
      if (a1 > max) {
        MdcPrntWarn("Number %u too big!",a1);
        MdcFlushInput(); return(MDC_BAD_CODE);
      }
      if (a2 > max) {
        MdcPrntWarn("Number %u too big!",a2);
        MdcFlushInput(); return(MDC_BAD_CODE);
      } 
      if ( (a1==0) || (a2==0) ) {
        for (i=1; i<=max; i++) (*item)[i]=MDC_YES;
        (*item)[0]=max;
        break;
      }
      if (a1 > a2) {
        tmp = a1;
        a1  = a2;
        a2  = tmp;
      }
      for (i=a1; i<=a2; i++) {
         if ((*item)[i] == MDC_NO) {
           (*item)[i] = MDC_YES; 
           (*item)[0] += 1;
         }
      }
    }else{ /* a single ... */
      sscanf(s,"%u",&a1);
      if (a1 > max) {
        MdcPrntWarn("Number %u too big!",a1);
        MdcFlushInput(); return(MDC_BAD_CODE);
      }

      if (a1 == 0) {
        for (i=1; i<=max; i++) (*item)[i]=MDC_YES;
        (*item)[0]=max;
        break;
      }

      if ((*item)[a1] == MDC_NO) {
        (*item)[a1] = MDC_YES;
        (*item)[0] += 1;
      }
    }

    MdcGetStrInput(s,MDC_2KB_OFFSET);

  }

  MdcFlushInput(); return(MDC_OK);

}

int MdcHandleNormList(char *s,Uint32 **inrs,Uint32 *it,Uint32 *bt,Uint32 max)
{
  Uint32 a1, a2, i;

  while(1) {
 
    if (atol(s) < 0 ) { break; }

    if ( strstr(s,"...") != 0 ) { /* a range ... */
      sscanf(s,"%u...%u",&a1,&a2);
      if (a1 > max) {
        MdcPrntWarn("Number %u too big!",a1);
        break;
      }
      if (a2 > max) {
        MdcPrntWarn("Number %u too big!",a2);
        break;
      }
      if ( (a1==0) || (a2==0) ) {
        (*inrs)[1] = 0;
        *it = 2; 
        break;
      }

      if (a1 > a2) {
        for (i=a1; i>=a2; i--) {
          (*inrs)[*it] = i;
          *it += 1;
          if ( (*it % MDC_BUF_ITMS) == 0 ) {
            if (((*inrs)=(Uint32 *)MdcRealloc((*inrs),
                         (*bt)*MDC_BUF_ITMS*sizeof(Uint32)))==NULL){
              MdcPrntWarn("Couldn't realloc images number buffer");
              MdcFlushInput(); return(MDC_BAD_ALLOC);
            }
            *bt += 1;
          }
        }
      }else {
        for (i=a1; i<=a2; i++) {
          (*inrs)[*it] = i;
          *it += 1;
          if ( (*it % MDC_BUF_ITMS) == 0 ) {
            if (((*inrs)=(Uint32 *)MdcRealloc((*inrs),
                         (*bt)*MDC_BUF_ITMS*sizeof(Uint32)))==NULL){
              MdcPrntWarn("Couldn't realloc images number buffer");
              MdcFlushInput(); return(MDC_BAD_ALLOC);
            }
            *bt += 1;
          }
        }
      }

    }else{ /* a single ... */
      sscanf(s,"%u",&a1);
      if (a1 > max) {
        MdcPrntWarn("Number %u too big!",a1);
        break;
      }
      if ( a1 == 0 ) {
        (*inrs)[1] = 0;
        *it = 2;
        break;
      }
      (*inrs)[*it] = a1;
      (*it) += 1;
      if ( (*it % MDC_BUF_ITMS) == 0 ) {
        if (((*inrs)=(Uint32 *)MdcRealloc((*inrs),
                     (*bt)*MDC_BUF_ITMS*sizeof(Uint32)))==NULL){
          MdcPrntWarn("Couldn't realloc images number buffer");
          MdcFlushInput(); return(MDC_BAD_ALLOC);
        }
        *bt += 1;
      }
    }

    MdcGetStrInput(s,MDC_2KB_OFFSET);

  }

  MdcFlushInput(); return(MDC_OK);

}

int MdcGetPixelRange(char *string, Uint32 *from, Uint32 *to)
{
  Uint32 a1, a2, tmp;

  if ( strstr(string,"...") != 0 ) { /* a range ... */
    sscanf(string,"%d...%d",&a1,&a2);
    if ( (a1==0) || (a2==0) ) {
      *from = 0; *to = 0;
    }else{
     if (a1 > a2) {
       tmp = a1;
       a1  = a2;
       a2  = tmp;
     }
     *from = a1; *to = a2;
    }
  }else{ /* a single */
    sscanf(string,"%u",&a1);
    if (a1 == 0) {
      *from=0; *to=0;
    }else{
      *from=a1; *to=a1;
    }
  }

  return(MDC_OK);
}

int MdcHandlePixelList(char *list, Uint32 **cols, Uint32 **rows, 
                    Uint32 *it, Uint32 *bt)
{
   int error;
   Uint32 r_from, r_to, c_from, c_to, r, c;
   char *col, *row;
   
   while (1) {
     if (atol(list) < 0 ) { MdcFlushInput(); return(MDC_OK); }
     col=list;
     row=strchr(list,',');
     if ( row == NULL) {
       MdcPrntWarn("Wrong input!\n");
       MdcFlushInput(); return(MDC_BAD_CODE);
     }else{
       *row = '\0';
       row += 1;
     }

     if ((error=MdcGetPixelRange(col,&c_from,&c_to)) != MDC_OK) {
       MdcFlushInput(); return(error);
     }
     if ((error=MdcGetPixelRange(row,&r_from,&r_to)) != MDC_OK) {
       MdcFlushInput(); return(error);
     }
       
     for (r=r_from; r<=r_to; r++)
     for (c=c_from; c<=c_to; c++) {
        (*cols)[*it] = c;
        (*rows)[*it] = r;
        *it+=1;
        if ( (*it % MDC_BUF_ITMS) == 0 ) {
          if ( ((*cols)=(Uint32 *)MdcRealloc((*cols),
                      (*bt)*MDC_BUF_ITMS*sizeof(Uint32)))==NULL){
            MdcPrntWarn("Couldn't realloc pixels column buffer");
            MdcFlushInput(); return(MDC_BAD_ALLOC);
          }
          if (((*rows)=(Uint32 *)MdcRealloc((*rows),
                      (*bt)*MDC_BUF_ITMS*sizeof(Uint32)))==NULL){
            MdcPrntWarn("Couldn't realloc pixels row buffer");
            MdcFlushInput(); return(MDC_BAD_ALLOC);
          }
          *bt+=1;
        }
     }

     MdcGetStrInput(list,MDC_2KB_OFFSET);

   }

   MdcFlushInput(); return(MDC_OK);
}

char *MdcGetStrAcquisition(int acq_type)
{ 
  switch (acq_type) {
    case MDC_ACQUISITION_STATIC : return("Static");
                                       break;
    case MDC_ACQUISITION_DYNAMIC: return("Dynamic");
                                       break;
    case MDC_ACQUISITION_TOMO   : return("Tomographic");
                                       break;
    default                     : return("Unknown");
  }
} 

char *MdcGetStrRawConv(int rawconv)
{
  switch (rawconv) {
    case MDC_NO            : return("No"); break;
    case MDC_FRMT_RAW      : return("Binary"); break;
    case MDC_FRMT_ASCII    : return("Ascii"); break;
    default                : return("Unknown");
  }
}

char *MdcGetStrEndian(int endian)
{
  switch (endian) {
    case MDC_BIG_ENDIAN   : return("Big"); break;
    case MDC_LITTLE_ENDIAN: return("Little"); break;
    default               : return("Unknown");
  }
}

char *MdcGetStrCompression(int compression)
{
  switch (compression) {
    case MDC_NO       : return("None"); break;
    case MDC_COMPRESS : return("Compress"); break;
    case MDC_GZIP     : return("Gzipped"); break;
    default           : return("Unknown");
  }
}

char *MdcGetStrPixelType(int type)
{
  switch (type) {
   case     BIT1: return("1-bit"); break;
   case   BIT8_S: return("Int8"); break;
   case   BIT8_U: return("Uint8"); break;
   case  BIT16_S: return("Int16"); break;
   case  BIT16_U: return("Uint16"); break;
   case  BIT32_S: return("Int32"); break;
   case  BIT32_U: return("Uint32"); break;
   case  BIT64_S: return("Int64"); break;
   case  BIT64_U: return("Uint64"); break;
   case    FLT32: return("IEEE float"); break;
   case    FLT64: return("IEEE double"); break;
   case  VAXFL32: return("VAX  float"); break;
   case    ASCII: return("ASCII"); break;
   default      : return("Unknown");
  }
}

char *MdcGetStrColorMap(int map)
{
  switch (map) {
   case MDC_MAP_PRESENT : return("present"); break;
   case MDC_MAP_GRAY    : if (MDC_CMAP_INVERS == MDC_NO)
                             return("gray normal");
                           else
                             return("gray invers");
                          break;
   case MDC_MAP_RAINBOW : return("rainbow"); break;
   case MDC_MAP_COMBINED: return("combined"); break; 
   case MDC_MAP_HOTMETAL: return("hotmetal"); break;
   case MDC_MAP_LOADED  : return("loaded LUT"); break;
   default              : return("Unknown");
  }
}

char *MdcGetStrYesNo(int boolean)
{
  switch (boolean) {
    case MDC_NO : return("No");      break;
    case MDC_YES: return("Yes");     break;
    default     : return("Unknown");
  }
}

char *MdcGetStrPatSlOrient(int patient_slice_orient)
{
  switch (patient_slice_orient) {

   case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
    strcpy(mdcbufr,"Supine;HeadFirst;Transversal"); break;
   case MDC_SUPINE_HEADFIRST_SAGITTAL   : 
    strcpy(mdcbufr,"Supine;HeadFirst;Sagittal");    break;
   case MDC_SUPINE_HEADFIRST_CORONAL    : 
    strcpy(mdcbufr,"Supine;HeadFirst;Coronal");     break;
   case MDC_SUPINE_FEETFIRST_TRANSVERSAL: 
    strcpy(mdcbufr,"Supine;FeetFirst;Transversal"); break;
   case MDC_SUPINE_FEETFIRST_SAGITTAL   : 
    strcpy(mdcbufr,"Supine;FeetFirst;Sagittal");    break;
   case MDC_SUPINE_FEETFIRST_CORONAL    : 
    strcpy(mdcbufr,"Supine;FeetFirst;Coronal");     break;
   case MDC_PRONE_HEADFIRST_TRANSVERSAL : 
    strcpy(mdcbufr,"Prone;HeadFirst;Transversal");  break;
   case MDC_PRONE_HEADFIRST_SAGITTAL    : 
    strcpy(mdcbufr,"Prone;HeadFirst;Sagittal");     break;
   case MDC_PRONE_HEADFIRST_CORONAL     : 
    strcpy(mdcbufr,"Prone;HeadFirst;Coronal");      break;
   case MDC_PRONE_FEETFIRST_TRANSVERSAL : 
    strcpy(mdcbufr,"Prone;FeetFirst;Transversal");   break;
   case MDC_PRONE_FEETFIRST_SAGITTAL    : 
    strcpy(mdcbufr,"Prone;FeetFirst;Sagittal");     break;
   case MDC_PRONE_FEETFIRST_CORONAL     :
    strcpy(mdcbufr,"Prone;FeetFirst;Coronal");      break;
   default                              : 
    strcpy(mdcbufr,"Unknown");

  }
  
  return(mdcbufr);
} 

char *MdcGetStrPatientPos(int patient_slice_orient)
{
  switch (patient_slice_orient) {

   case MDC_SUPINE_HEADFIRST_TRANSVERSAL: 
   case MDC_SUPINE_HEADFIRST_SAGITTAL   :
   case MDC_SUPINE_HEADFIRST_CORONAL    :
   case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
   case MDC_SUPINE_FEETFIRST_SAGITTAL   :
   case MDC_SUPINE_FEETFIRST_CORONAL    :
    strcpy(mdcbufr,"supine"); break;
   case MDC_PRONE_HEADFIRST_TRANSVERSAL :
   case MDC_PRONE_HEADFIRST_SAGITTAL    :
   case MDC_PRONE_HEADFIRST_CORONAL     :
   case MDC_PRONE_FEETFIRST_TRANSVERSAL :
   case MDC_PRONE_FEETFIRST_SAGITTAL    :
   case MDC_PRONE_FEETFIRST_CORONAL     :
    strcpy(mdcbufr,"prone"); break;
   default                              :
    strcpy(mdcbufr,"Unknown");

  }

  return(mdcbufr);

}

char *MdcGetStrPatientOrient(int patient_slice_orient)
{
  switch (patient_slice_orient) {
     case MDC_SUPINE_HEADFIRST_TRANSVERSAL: strcpy(mdcbufr,"L\\P"); break;
     case MDC_SUPINE_HEADFIRST_SAGITTAL   : strcpy(mdcbufr,"P\\F"); break;
     case MDC_SUPINE_HEADFIRST_CORONAL    : strcpy(mdcbufr,"L\\F"); break;
     case MDC_SUPINE_FEETFIRST_TRANSVERSAL: strcpy(mdcbufr,"R\\P"); break;
     case MDC_SUPINE_FEETFIRST_SAGITTAL   : strcpy(mdcbufr,"P\\H"); break;
     case MDC_SUPINE_FEETFIRST_CORONAL    : strcpy(mdcbufr,"R\\H"); break;
     case MDC_PRONE_HEADFIRST_TRANSVERSAL : strcpy(mdcbufr,"R\\A"); break;
     case MDC_PRONE_HEADFIRST_SAGITTAL    : strcpy(mdcbufr,"A\\F"); break;
     case MDC_PRONE_HEADFIRST_CORONAL     : strcpy(mdcbufr,"R\\F"); break;
     case MDC_PRONE_FEETFIRST_TRANSVERSAL : strcpy(mdcbufr,"L\\A"); break;
     case MDC_PRONE_FEETFIRST_SAGITTAL    : strcpy(mdcbufr,"A\\H"); break;
     case MDC_PRONE_FEETFIRST_CORONAL     : strcpy(mdcbufr,"L\\H"); break;
     default                              : strcpy(mdcbufr,"Unknown");
  }

  return(mdcbufr);
}

char *MdcGetStrPatientRot(int patient_slice_orient)
{

  switch (patient_slice_orient) {
     case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
     case MDC_SUPINE_HEADFIRST_SAGITTAL   :
     case MDC_SUPINE_HEADFIRST_CORONAL    :
     case MDC_PRONE_HEADFIRST_TRANSVERSAL :
     case MDC_PRONE_HEADFIRST_SAGITTAL    :
     case MDC_PRONE_HEADFIRST_CORONAL     : 
      strcpy(mdcbufr,"head_in"); break;
     case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
     case MDC_SUPINE_FEETFIRST_SAGITTAL   :
     case MDC_SUPINE_FEETFIRST_CORONAL    :
     case MDC_PRONE_FEETFIRST_TRANSVERSAL :
     case MDC_PRONE_FEETFIRST_SAGITTAL    : 
     case MDC_PRONE_FEETFIRST_CORONAL     :
      strcpy(mdcbufr,"feet_in"); break;
     default                              :
      strcpy(mdcbufr,"unknown"); break;
  }

  return(mdcbufr);
}

char *MdcGetStrSliceOrient(int patient_slice_orient)
{
  switch (patient_slice_orient) {
     case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
     case MDC_PRONE_HEADFIRST_TRANSVERSAL :
     case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
     case MDC_PRONE_FEETFIRST_TRANSVERSAL :
      strcpy(mdcbufr,"Transverse"); break;
     case MDC_SUPINE_HEADFIRST_SAGITTAL   :
     case MDC_PRONE_HEADFIRST_SAGITTAL    :
     case MDC_SUPINE_FEETFIRST_SAGITTAL   :
     case MDC_PRONE_FEETFIRST_SAGITTAL    :
      strcpy(mdcbufr,"Sagittal");  break;
     case MDC_SUPINE_HEADFIRST_CORONAL    :
     case MDC_PRONE_HEADFIRST_CORONAL     :
     case MDC_SUPINE_FEETFIRST_CORONAL    :
     case MDC_PRONE_FEETFIRST_CORONAL     :
      strcpy(mdcbufr,"Coronal");   break;
     default                              : 
      strcpy(mdcbufr,"unknown");
  }

  return(mdcbufr);
}

char *MdcGetStrRotation(int rotation)
{ 
  switch (rotation) {
    case MDC_ROTATION_CW: strcpy(mdcbufr,"clockwise");         break;
    case MDC_ROTATION_CC: strcpy(mdcbufr,"counter-clockwise"); break;
    default             : strcpy(mdcbufr,"unknown");
  }

  return(mdcbufr);
}

char *MdcGetStrMotion(int motion)
{
  switch (motion) {
    case MDC_MOTION_STEP: strcpy(mdcbufr,"step and shoot");  break;
    case MDC_MOTION_CONT: strcpy(mdcbufr,"continuous");    break;
    case MDC_MOTION_DRNG: strcpy(mdcbufr,"during step");   break;
    default             : strcpy(mdcbufr,"unknown");
  }

  return(mdcbufr);
}

const char *MdcGetLibLongVersion(void)
{
  return(MDC_LIBVERS);
}

const char *MdcGetLibShortVersion(void)
{
  return(MDC_VERSION);
}
  
/* returns the new stringsize value or 0 in case of error to add */
Uint32 MdcCheckStrSize(char *str_to_add, Uint32 current_size, Uint32 max)
{
  Uint32 max_value = MDC_2KB_OFFSET;
  Uint32 new_size;

  if (max != 0) max_value = max;

  new_size = current_size + (Uint32)strlen(str_to_add);

  if ( new_size >= max_value ) {
    MdcPrntWarn("Internal Problem -- Information string too small");
    return(0);
  }

  return(new_size);
} 

/* print to global `mdcbufr' array */
int MdcMakeScanInfoStr(FILEINFO *fi)
{
  char strbuf[100];
  Uint32 size=0;

  sprintf(mdcbufr,"\n\n\
******************************\n\
Short Patient/Scan Information\n\
******************************\n");
  size = (Uint32)strlen(mdcbufr);
  sprintf(strbuf,"Patient Name: %s\n",fi->patient_name);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Patient Sex : %s\n",fi->patient_sex);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Patient ID  : %s\n",fi->patient_id);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Study Date  : %02d/%02d/%04d\n",fi->study_date_day
                                               ,fi->study_date_month
                                               ,fi->study_date_year);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Study Time  : %02d:%02d:%02d\n",fi->study_time_hour
                                               ,fi->study_time_minute
                                               ,fi->study_time_second);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Study Name  : %s\n",fi->study_name);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Study Descr : %s\n",fi->study_descr);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Acquisition Type     : %s\n",
                                 MdcGetStrAcquisition(fi->acquisition_type));
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Reconstructed        : %s\n",
                                 MdcGetStrYesNo(fi->reconstructed));
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;

  if (fi->reconstructed == MDC_YES) {
  sprintf(strbuf,"Reconstruction Method: %s\n",fi->recon_method);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Filter Type          : %s\n",fi->filter_type);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Decay Corrected      : %s\n",
                                       MdcGetStrYesNo(fi->decay_corrected));
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Flood Corrected      : %s\n",
                                       MdcGetStrYesNo(fi->flood_corrected));
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Radiopharmaceutical  : %s\n",fi->radiopharma);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  }
  sprintf(strbuf,"Isotope Code         : %s\n",fi->isotope_code);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Isotope Halflife     : %+e [sec]\n",
                                       fi->isotope_halflife);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;
  sprintf(strbuf,"Gantry Tilt          : %+e degrees\n",
                                       fi->gantry_tilt);
  if ((size=MdcCheckStrSize(strbuf,size,0))) strcat(mdcbufr,strbuf);
  else return MDC_NO;

  return(MDC_YES);

}

int MdcIsDigit(char c) 
{
  if (c >= '0' && c <= '9') return MDC_YES;
  
  return(MDC_NO);
}

void MdcWaitForEnter(int page)
{
  if (page > 0) {
    printf("\t\t*********** Press <enter> for page #%d **********",page);
  }
  if (page == 0) {
    printf("\t\t********** Press <enter> for next page **********");
  }
  if (page < 0 ) {
    printf("Press <enter> to continue ...");
  }
  while ( fgetc(stdin) != '\n' ) { /* wait until <enter> key pressed */ }
}

Int32 MdcGetSelectionType(void)
{
   Int32 type=-1;

   printf("\n\tSelection Type:\n");
   printf("\n\ttype  %d  ->  normal",MDC_INPUT_NORM_STYLE);
   printf("\n\t      %d  ->  ecat\n",MDC_INPUT_ECAT_STYLE);
   printf("\n\tYour choice [%d]? ",MDC_INPUT_NORM_STYLE);
   fgets(mdcbufr,MDC_2KB_OFFSET-1,stdin);

   type=(Int32)atol(mdcbufr);

   if (type != MDC_INPUT_ECAT_STYLE) type = MDC_INPUT_NORM_STYLE;

   return(type);
}

void MdcFlushInput(void)
{
   while( fgetc(stdin) != '\n' ) { }
}

int MdcWhichDecompress(void)
{
  if (strcmp(MDC_DECOMPRESS,"gunzip")     == 0) return(MDC_GZIP);
  if (strcmp(MDC_DECOMPRESS,"uncompress") == 0) return(MDC_COMPRESS);

  return(MDC_NO);
}

int MdcWhichCompression(char *fname)
{
  char *ext=NULL;
  int compression = MDC_NO;

  /* get filename extension */
  if (fname != NULL) ext = strrchr(fname,'.');
  if (ext != NULL) {
    /* check for supported compression */
    switch (MdcWhichDecompress()) {

      case MDC_COMPRESS: if (strcmp(ext,".Z") == 0 )        /* only .Z files */
                           compression = MDC_COMPRESS;
                         break;
      case MDC_GZIP    : if (strcmp(ext,".gz") == 0 ) {
                           compression = MDC_GZIP;
                         }else if (strcmp(ext,".Z")  == 0 ) {
                           compression = MDC_COMPRESS;
                         }
                         break;
    }
  }

  return(compression);

}

void MdcAddCompressionExt(int ctype, char *fname)
{
   switch (ctype) {
     case MDC_COMPRESS: strcat(fname,".Z");  break;

     case MDC_GZIP    : strcat(fname,".gz"); break;
   } 
}
