/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-algori.c                                                    *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : endian/image algorithms                                  *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcRealloc()            - if (NULL) malloc else realloc  *
 *                MdcDoSwap()             - Test to swap                   *
 *                MdcHostBig()            - Test if host is bigendian      *
 *                MdcSwapBytes()          - Swap the bytes                 *
 *                MdcForceSwap()          - Forced bytes swapping          *
 *                MdcIEEEfl_to_VAXfl()    - Change hostfloat to VAX float  *
 *                MdcVAXfl_to_IEEEfl()    - Change VAX float to hostfloat  *
 *                MdcType2Bytes()         - Pixel data type in bytes       *
 *                MdcType2Bits()          - Pixel data type in bits        *
 *                MdcTypeIntMax()         - Give maximum of integer type   *
 *                MdcImagesPixelFiddle()  - Process all pixels & images    *
 *                MdcGetDoublePixel()     - Get pixel from memory buffer   *
 *                MdcPutDoublePixel()     - Put pixel to memory buffer     *
 *                MdcDoSimpleCast()       - Test cast sufficient to rescale*
 *                MdcGetResizedImage()    - Make images of same size       *
 *                MdcGetDisplBIT8_U()     - Get  an Uint8 display image    *
 *                MdcMakeBIT8_U()         - Make an Uint8 image            *
 *                MdcGetImgBIT8_U()       - Get  an Uint8 image            *
 *                MdcMakeBIT16_S()        - Make an Int16 image            *
 *                MdcGetImgBIT16_S()      - Get  an Int16 image            *
 *                MdcMakeBIT32_S()        - Make an Int32 image            *
 *                MdcGetImgBIT32_S()      - Get  an Int32 image            *
 *                MdcMakeFLT32()          - Make a  float image            *
 *                MdcGetImgFLT32()        - Get  a  float image            *
 *                MdcMakeImgSwapped()     - Make an endian swapped image   *
 *                MdcGetImgSwapped()      - Get  an endian swapped image   *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-algori.c,v 1.1.1.1 2000/10/28 16:51:07 enlf Exp $ 
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

#define maxmin(x, max, min)          { maxmin_t=(x);                     \
                                       if (maxmin_t > max) max=maxmin_t; \
                                       if (maxmin_t < min) min=maxmin_t; \
                                     }

static Uint8 MDC_ALLOW_CAST = MDC_YES;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/


void *MdcRealloc(void *p, Uint32 bytes)
{
  void *ptmp;

  if (p == NULL) {
    ptmp = malloc(bytes);
  }else{
    ptmp = realloc(p, bytes);
  }

  return(ptmp);
}



/* the machine-endian soup */
/*
READING:   host     infile    swap
-----------------------------------
           big0      big0      0
           big0      little1   1
           little1   big0      1 
           little1   little1   0 
----------------------------------
READING:   host  XOR file  =  swap


WRITING:   host     outfile   swap 
----------------------------------
           big0      big0      0
           little1   big0      1

           big0      little1   1 
           little1   little1   0 
----------------------------------
WRITING    host  XOR file  =  swap
*/

int MdcDoSwap(void) 
{
  return(MDC_HOST_ENDIAN^MDC_FILE_ENDIAN); 
}

int MdcHostBig(void)
{
  if (MDC_HOST_ENDIAN == MDC_BIG_ENDIAN) return 1;
  return 0;
}

void MdcSwapBytes(Uint8 *ptr, int bytes)
{
  int i, j;

  if ( MdcDoSwap() ) 
    for (i=0,j=bytes-1;i < (bytes/2); i++, j--) {
       ptr[i]^=ptr[j]; ptr[j]^=ptr[i]; ptr[i]^=ptr[j];
    }
}

void MdcForceSwap(Uint8 *ptr, int bytes)
{
  int i, j;

  for (i=0,j=bytes-1;i < (bytes/2); i++, j--) {
     ptr[i]^=ptr[j]; ptr[j]^=ptr[i]; ptr[i]^=ptr[j];
  }
}

void MdcIEEEfl_to_VAXfl(float *f)
{ 
  Uint16 exp;
  union { 
          Uint16 t[2];
          float t4;
  } test;

  test.t4 = *f;

  if (test.t4 != 0.0) {
    if (!MdcHostBig()) { /* swap words */
      Uint16 temp;
      memcpy((void *)&temp,(void *)&test.t[0],2);
      memcpy((void *)&test.t[0],(void *)&test.t[1],2);
      memcpy((void *)&test.t[1],(void *)&temp,2);
    }

    exp =  ((test.t[0] & 0x7f00) + 0x0100) & 0x7f00;
    test.t[0] = (test.t[0] & 0x80ff) + exp;

    MdcSwapBytes((Uint8 *)&test.t[0],2);
    MdcSwapBytes((Uint8 *)&test.t[1],2);

  }

  memcpy((void *)f,(void *)&test.t4,4);

}

void MdcVAXfl_to_IEEEfl(float *f)
{
  Uint16  t1, t2;
  union {
          Uint16 n[2];
          float  n4;
  } number;

  union {
          Uint32 t3;
          float  t4;
  } test;

  number.n4 = *f;

  if (MdcHostBig()) {
    Uint16 temp;
    temp = number.n[0]; number.n[0]=number.n[1]; number.n[1]=temp;
  } 

  MdcSwapBytes((Uint8 *)&number.n4,4);

  if ((number.n[0] != 0) || (number.n[1] != 0) ) { 
    t1 = number.n[0] & 0x80ff;
    t2 = (((number.n[0])&0x7f00)+0xff00)&0x7f00;
    test.t3 = (t1+t2)<<16;
    test.t3 = test.t3+number.n[1];

    number.n4 = test.t4;
  }

  memcpy((void *)f,(void *)&number.n4,4);

} 

int MdcType2Bytes(int type)
{
  int bytes = 0; 

  switch (type) {

    case  BIT1  :
    case  BIT8_S:
    case  BIT8_U: bytes = 1; break;

    case BIT16_S:
    case BIT16_U: bytes = 2; break;

    case BIT32_S:
    case BIT32_U:
    case FLT32  :
    case VAXFL32: bytes = 4; break;
   
    case ASCII  : /* read as double */
#ifdef HAVE_8BYTE_INT
    case BIT64_S:
    case BIT64_U:
#endif
    case FLT64  : bytes = 8; break;
  }
 
  return(bytes);

}



int MdcType2Bits(int type)
{
  int bits = 0; 

  switch (type) {

    case  BIT1  : bits = 1; break;

    case  BIT8_S:
    case  BIT8_U: bits = 8; break;

    case BIT16_S:
    case BIT16_U: bits = 16; break;

    case BIT32_S:
    case BIT32_U:
    case FLT32  :
    case VAXFL32: bits = 32; break;

    case ASCII  : /* read as double */
#ifdef HAVE_8BYTE_INT
    case BIT64_S:
    case BIT64_U:
#endif
    case FLT64  : bits = 64; break;
  }
 
  return(bits);

}

double MdcTypeIntMax(int type)
{
  switch (type) {

    case BIT8_S : return(127.);
    case BIT8_U : return(255.);
    case BIT16_S: return(32767.);
    case BIT16_U: return(65535.);
    case BIT32_S: return(2147483647.);
    case BIT32_U: return(4294967295.);
#ifdef HAVE_8BYTE_INT
    case BIT64_S: return(9223372036854775807.);
    case BIT64_U: return(18446744073709551615.);
#endif

  }

  return(0.0);

}

void SwapImage(Uint8 *buf, Uint32 width, Uint32 height, Uint16 type)
{
  Uint32 i,  size = width*height;

  switch (type) {

   case BIT16_S:
   case BIT16_U:
       for (i=0; i<size; i++, buf+=2 )  MdcSwapBytes(buf,2); break;
   case BIT32_S:
   case BIT32_U:
   case   FLT32:
       for (i=0; i<size; i++, buf+=4 )  MdcSwapBytes(buf,4); break;
#ifdef HAVE_8BYTE_INT
   case BIT64_S:
   case BIT64_U:
#endif
   case   FLT64:
       for (i=0; i<size; i++, buf+=8 )  MdcSwapBytes(buf,8); break;
  }
}


/*
 pixel by pixel processes: yes, THE all in one routine 
     - swap bytes 
     - make positive 
     - get global & image  variables
 check some parameters 
 Notification: double pixel types may get corrupted quantification values
               because our quantification is stripped down to a float!!
*/
char *MdcImagesPixelFiddle(FILEINFO *fi)
{
  IMG_DATA *id, *idprev;
  int type, BAD=MDC_NO;
  Uint32 i, j, n, t;
  double fmin=0., fmax=0., qfmin=0., qfmax=0.; 

  /* initial checks for FILEINFO integrity */
  if (fi->number == 0) return("Internal ## Improper fi->number value");

  for (i=1, t=3; t <= fi->dim[0]; t++) {
     MdcDebugPrint("DEBUG: dim[] TEST : fi->dim[%d] = %u\n",t,fi->dim[t]);
     i*=fi->dim[t];
  }
 
  if (fi->number != i) {
    /* return("Internal ## Improper fi->dim values"); */
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayWarn("Internal Error ## Improper fi->dim values\n" \
                      "** falling back to one dimensional array **\n" \
                      "** WE WOULD LIKE TO RECEIVE THIS IMAGE FILE **");
    }else
#endif
    {
      MdcPrntWarn("Internal Error ## Improper fi->dim values\n" \
                  "\t\t - falling back to one dimensional array\n" \
                  "\t\t(WE WOULD LIKE TO RECEIVE THIS IMAGE FILE)" );
    }

    fi->dim[0] = 3; fi->dim[3] = fi->number;
    for (i=4; i<8; i++) fi->dim[i] = 1;
 
  }

  /* make sure fi->dim[] are 1-based */
  for (i=0; i<8; i++) if (fi->dim[i] <= 0) fi->dim[i] = 1;

  fi->mwidth = fi->image[0].width;
  fi->mheight = fi->image[0].height;

  /* short check of ACQ_DATA stuff */
  if ((fi->acqnr > 0) && (fi->acqdata == NULL)) fi->acqnr = 0;
  
#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Processing images:");
#endif

  /* check images as well */
  for (j=0; j<fi->number; j++) {

     id = &fi->image[j];

     /* check dimension/pixeltype values */
     if ( id->width == 0 || id->height == 0 || 
          id->bits == 0  || id->type == 0   ||
          id->buf == NULL) {
       BAD=MDC_YES;
       break;
     }
  }

  if (BAD == MDC_YES) return("Internal ## Improper IMG_DATA values");


  for (j=0; j<fi->number; j++) {

#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif

     id = &fi->image[j];
     n = id->width * id->height;
     type = id->type;

     /* check some IMG_DATA things */
     if (id->pixel_xsize   <= 0.0 ) id->pixel_xsize = 1.0;
     if (id->pixel_ysize   <= 0.0 ) id->pixel_ysize = 1.0;
     if (id->slice_width   <= 0.0 ) id->slice_width = 1.0;
     if (id->slice_spacing <= 0.0 ) id->slice_spacing = id->slice_width;
     if (id->ct_zoom_fctr  <= 0.0 ) id->ct_zoom_fctr= 1.0;

     if (strcmp(id->pat_pos,"Unknown") == 0)
       strcpy(id->pat_pos,MdcGetStrPatientPos(id->pat_slice_orient));
     if (strcmp(id->pat_orient,"Unknown") == 0)
       strcpy(id->pat_orient,MdcGetStrPatientOrient(id->pat_slice_orient));
 
     id->bits = MdcType2Bits(type);

     if (MDC_QUANTIFY) 
       id->rescale_fctr = id->quant_scale;
     else if (MDC_CALIBRATE)
       id->rescale_fctr = id->quant_scale * id->calibr_fctr;

     switch (type) {

      case BIT8_U:
       {
         Uint8 *pix = (Uint8 *) id->buf;
         Uint8 max, min, maxmin_t;

         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) maxmin(*pix, max, min);


         id->max = (double)max;
         id->min = (double)min;
       }
       break;
      case BIT8_S:
       { 
         Int8 *pix = (Int8 *) id->buf;
         Int8 max, min, maxmin_t;

         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];
 
         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) {
            if (!MDC_NEGATIVE && (*pix < 0)) *pix=0;
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
      case BIT16_U:
       {
         Uint16 *pix = (Uint16 *) id->buf;
         Uint16 max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 2);
 
         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 2);
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
      case BIT16_S:
       {
         Int16 *pix = (Int16 *) id->buf;
         Int16 max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 2);

         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 2);
            if (!MDC_NEGATIVE && (*pix < 0)) *pix=0;
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
      case BIT32_U:
       {
         Uint32 *pix = (Uint32 *) id->buf;
         Uint32 max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 4);

         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 4);
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
      case BIT32_S:
       {
         Int32 *pix = (Int32 *) id->buf;
         Int32 max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 4);

         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 4);
            if (!MDC_NEGATIVE && (*pix < 0)) *pix = 0;
            maxmin(*pix, max, min); 
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
#ifdef HAVE_8BYTE_INT
      case BIT64_U:
       { 
         Uint64 *pix = (Uint64 *) id->buf;
         Uint64 max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 8);
         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];
         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 8);
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
      case BIT64_S:
       {
         Int64 *pix = (Int64 *) id->buf;
         Int64 max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 8);
         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];
         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 8);
            if (!MDC_NEGATIVE && (*pix < 0)) *pix = 0;
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;
#endif
      case FLT32:
       {
         float *pix = (float *) id->buf;
         float max, min, maxmin_t;

         MdcSwapBytes((Uint8 *)pix, 4);
         if (j == 0) fi->glmin = fi->glmax = (double) pix[0];

         min = max = pix[0];

         for (i=1, pix+=1; i<n; i++, pix++) {
            MdcSwapBytes((Uint8 *)pix, 4);
            if (!MDC_NEGATIVE && (*pix < 0.)) *pix = 0.;
            maxmin(*pix, max, min);
         }
         id->max = (double)max;
         id->min = (double)min;
       }
       break;

     case FLT64:
      {
        double *pix = (double *) id->buf;
        double max, min, maxmin_t;

        MdcSwapBytes((Uint8 *)pix, 8);
        if (j == 0 ) fi->glmin = fi->glmax = (double) pix[0];

        min = max = pix[0];
        for (i=1, pix+=1; i<n; i++, pix++) {
           MdcSwapBytes((Uint8 *)pix, 8);
           if (!MDC_NEGATIVE && (*pix < 0.)) *pix = 0.;
           maxmin(*pix, max, min);
        }
        id->max = max;
        id->min = min;
      }
      break;
     }

     if (j == 0) {
       fi->glmin = id->min;  fi->glmax = id->max;
     }else{
       if ( id->max > fi->glmax ) fi->glmax = id->max;
       if ( id->min < fi->glmin ) fi->glmin = id->min;
     }

     /* do it even if not MDC_QUANTIFY or not MDC_CALIBRATE */
     /* handle global min values */
     if (!MDC_NEGATIVE) { fi->qglmin =  fi->glmin = id->min = 0.; }
     else { 
        id->qmin = (double)((float)id->min * id->rescale_fctr);
        if (j == 0) {
          fi->qglmin = id->qmin;
        }else{
          if ( id->qmin < fi->qglmin) fi->qglmin = id->qmin;
        }
     }
     /* handle global max values */ 
     id->qmax = (double)((float)id->max * id->rescale_fctr);

     if (j == 0) { 
       fi->qglmax = id->qmax;
     }else{
       if ( id->qmax > fi->qglmax ) fi->qglmax = id->qmax;
     }

     /* handle the max/min values for the frame group */
     if ( (j % fi->dim[3]) == 0 ) {
       /* a frame boundary */
       if (j == 0) { 
         /* the beginning frame group */
         fmin  = id->min;  fmax  = id->max; 
         qfmin = id->qmin; qfmax = id->qmax;
       }else{
         /* new frame group, fill in the values for previous frame */
         for (t=j - fi->dim[3]; t<j ; t++) {
            idprev = &fi->image[t];

            idprev->fmin  = fmin;  idprev->fmax  = fmax;
            idprev->qfmin = qfmin; idprev->qfmax = qfmax;
         }
         /* re-initialize the values for the new frame group */
         fmin  = id->min;  fmax  = id->max;
         qfmin = id->qmin; qfmax = id->qmax;
       }
     }else{ 
       /* inside a frame group, determine min/max values */
       if (id->min < fmin ) fmin = id->min;
       if (id->max > fmax ) fmax = id->max;

       if (id->qmin < qfmin ) qfmin = id->qmin;
       if (id->qmax > qfmax ) qfmax = id->qmax;

     } 


     if (id->type != fi->image[0].type) fi->diff_type = MDC_YES;
     if (id->width != fi->mwidth) {
       fi->diff_size = MDC_YES; 
       if (id->width > fi->mwidth ) fi->mwidth = id->width;
     }
     if (id->height != fi->mheight) {
       fi->diff_size = MDC_YES;
       if (id->height > fi->mheight ) fi->mheight = id->height;
     }
      
  }

  /* don't forget to fill in the min/max values for the last frame group */
  for (t=j - fi->dim[3]; t<j ; t++) {        
     idprev = &fi->image[t];

     idprev->fmin  = fmin;  idprev->fmax  = fmax;
     idprev->qfmin = qfmin; idprev->qfmax = qfmax;
  }

  /* MARK: Prevent strange side effect of negative & raw reading: */
  /*            raw reading => enables negative values            */
  /*       In this case min value is really the min value found   */
  /*       and not set to zero. In case min value is >0 this value*/
  /*       will be displayed in XMedCon as black which is not     */
  /*       wanted sometimes (ex. DICOM's converted to pgm files)  */
  /*       Here we check if fi->glmin > 0  and fi->qglmin > 0     */
  /*       in order to set the min values to zero                 */
  if (MDC_NEGATIVE == MDC_YES && fi->glmin > 0. && fi->qglmin > 0.) {
    fi->glmin = 0.; fi->qglmin = 0.;
    for (t = 0; t < fi->number; t++) {
       fi->image[t].min   = 0.;
       fi->image[t].qmin  = 0.;
       fi->image[t].fmin  = 0.;
       fi->image[t].qfmin = 0.;
    }
  }
 
  fi->dim[1] = (Int16) fi->mwidth;
  fi->dim[2] = (Int16) fi->mheight;
 
  id = &fi->image[0];

  fi->bits  = id->bits;
  fi->type  = id->type;

  /* final checks for FILEINFO integrity */
  /* really ugly things */
  if (fi->dim[0] <= 2 ) {
    sprintf(mdcbufr,"Internal ## fi->dim[0]=%d",fi->dim[0]);
    return(mdcbufr);
  }else{
    for (t=1; t<=fi->dim[0]; t++) {
       if (fi->dim[t] <= 0 ) {
         sprintf(mdcbufr,"Internal ## fi->dim[%d]=%d",t,fi->dim[t]);
         return(mdcbufr);
       }
    }
  }
  /* fixable things */
  if (fi->pixdim[0] == 3.0 ||
      fi->pixdim[0] == 4.0 ||
      fi->pixdim[0] == 5.0 ||
      fi->pixdim[0] == 6.0 ||
      fi->pixdim[0] == 7.0 ) { 
    for (t=1; t<=(Int32)fi->pixdim[0]; t++) {
       if (fi->pixdim[t] <= 0.0 ) fi->pixdim[t] = 1.;
    }
  }else{
    fi->pixdim[0] = 3.;
    fi->pixdim[1] = 1.;
    fi->pixdim[2] = 1.;
    fi->pixdim[3] = 1.;
  }

  return(NULL);

} 

double MdcGetDoublePixel(Uint8 *buf, Uint16 type)
{
  double value=0.0;

  switch (type) {
    case BIT8_U:
        {
           Uint8 *pix = (Uint8 *)buf;
           value = (double)pix[0];
        }
        break;
    case BIT8_S:
        {
           Int8 *pix = (Int8 *)buf;
           value = (double)pix[0];
        }
        break;
    case BIT16_U:
        {
           Uint16 *pix = (Uint16 *)buf;
           value = (double)pix[0];
        }
        break;
    case BIT16_S:
        {
           Int16 *pix = (Int16 *)buf;
           value = (double)pix[0];
        }
        break;
    case BIT32_U:
        {
           Uint32 *pix = (Uint32 *)buf;
           value = (double)pix[0];
        }
        break;
    case BIT32_S:
        {
           Int32 *pix = (Int32 *)buf;
           value = (double)pix[0];
        }
        break;
#ifdef HAVE_8BYTE_INT
    case BIT64_U:
        {
           Uint64 *pix = (Uint64 *)buf;
           value = (double)pix[0];
        }
        break;
    case BIT64_S:
        {
           Int64 *pix = (Int64 *)buf;
           value = (double)pix[0];
        }
        break;
#endif
    case FLT32:
        {
           float *pix = (float *)buf;
           value = (double)pix[0];
        }
        break;
    case FLT64:
        {
           double *pix = (double *)buf;
           value = pix[0];
        }
        break;
  }

  return(value);

}

void MdcPutDoublePixel(Uint8 *buf, double pix, Uint16 type)
{
 int bytes = MdcType2Bytes(type);

 switch (type) {
  case BIT8_S:
   {
     Int8 c = (Int8) pix;
     buf[0] = c; 
   }
   break; 
  case BIT8_U:
   {
     Uint8 c = (Uint8) pix;
     buf[0] = c;
   }
   break;
  case BIT16_S:
   {
     Int16 c = (Int16) pix;
     memcpy(buf,(Uint8 *)&c,bytes); 
   }
   break;
  case BIT16_U:
   {
     Uint16 c = (Uint16) pix;
     memcpy(buf,(Uint8 *)&c,bytes);
   }
   break;
  case BIT32_S:
   {
     Int32 c = (Int32) pix; 
     memcpy(buf,(Uint8 *)&c,bytes); 
   }
   break;
  case BIT32_U:
   {
     Uint32 c = (Uint32) pix;
     memcpy(buf,(Uint8 *)&c,bytes); 
   }
   break;
#ifdef HAVE_8BYTE_INT
  case BIT64_S:
   {
     Int64 c = (Int64) pix;
     memcpy(buf,(Uint8 *)&c,bytes);
   }
   break;
  case BIT64_U:
   {
     Uint64 c = (Uint64) pix;
     memcpy(buf,(Uint8 *)&c,bytes);
   }
   break;
#endif 
  case FLT32:
   {
     float c = (float) pix;
     memcpy(buf,(Uint8 *)&c,bytes); 
   }
   break;
  case FLT64:
   {
     double c = (double) pix;
     memcpy(buf,(Uint8 *)&c,bytes); 
   }
   break;
 }

}

int MdcDoSimpleCast(double minv, double maxv, double negmin, double posmax)
{
  Int32 casted;

  /* Rescaling to new integer values (Int32, Int16, Uint8): when original
   * values are integers and within the range of the new pixel type, a 
   * simple cast would do - without rescaling
   *                      - preserving original values 
   */

  if (MDC_ALLOW_CAST == MDC_NO) return(MDC_NO);

  /* TEST #1:  simple cast test -> original values = integer ? */
  casted = (Int32)minv; if ((double)casted != minv) return(MDC_NO);
  casted = (Int32)maxv; if ((double)casted != maxv) return(MDC_NO);

  /* TEST #2: within new pixel type range ? */
  if (minv < negmin || maxv > posmax) return(MDC_NO);

  return(MDC_YES);

}


Uint8 *MdcGetResizedImage(FILEINFO *fi,Uint8 *buffer,Uint16 type,Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint32 d, w, bytes, linesize, size;
  double pval;
  Uint8 *maxbuf, *pbuf;

  if (id->rescaled) {
    pval = id->rescaled_min;
  }else{
    pval = id->min;
  }

  bytes = MdcType2Bytes(type);

  linesize = id->width * bytes;

  size = fi->mwidth * fi->mheight * bytes;
  maxbuf = MdcGetImgBuffer(size);
  if (maxbuf == NULL) return NULL;

  pbuf = maxbuf;

  for (d=0; d < id->height; buffer+=linesize, d++) {
     memcpy(pbuf,buffer,linesize); 
     pbuf += linesize;
     /* fill line ? */
     for (w=id->width; w < fi->mwidth; w++) {
        MdcPutDoublePixel(pbuf,pval,type);
        pbuf += bytes;
     }
  }

  for (d=id->height; d < fi->mheight; d++) 
     for (w=0; w < fi->mwidth; w++) { /* add lines */
        MdcPutDoublePixel(pbuf,pval,type);
        pbuf += bytes;
  }

  return(maxbuf);

}

/* special case of GetImgBIT8_U: disable simple casting for */
/* display buffer any max should be rescaled to 255 white   */
Uint8 *MdcGetDisplBIT8_U(FILEINFO *fi, Uint32 img)
{
   Uint8 *buf, RESTORE=MDC_ALLOW_CAST;

   MDC_ALLOW_CAST = MDC_NO;

   buf = MdcGetImgBIT8_U(fi,img);

   MDC_ALLOW_CAST = RESTORE;

   return(buf);
}


Uint8 *MdcMakeBIT8_U(Uint8 *cbuf, FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint8 *buf=(Uint8 *)cbuf, *pixel, DO_QUANT_CALIBR;

  Uint32 i, n = id->width * id->height;
  double pixval, min, max, scale=1.0;
  float  newval;

  /* get proper maximum/minimum value */
  if (MDC_QUANTIFY || MDC_CALIBRATE) {
    DO_QUANT_CALIBR = MDC_YES;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->qfmin;
      max = id->qfmax;
    }else{
      min = fi->qglmin;
      max = fi->qglmax;
    }
    scale = (max == 0.) ? 1. : 255./max;   
  }else{
    DO_QUANT_CALIBR = MDC_NO;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->fmin;
      max = id->fmax;
    }else{
      min = fi->glmin;
      max = fi->glmax;
    }
    scale = (max == min) ? 1. : 255./(max - min);
  }

  if (MdcDoSimpleCast(min,max,0.,255.) == MDC_YES) {
    scale = 1.;  min = 0.;
  }

  switch( id->type ) {
    case BIT1: /* convert bits to byte  */
        {
          /* to avoid a premature overwrite, we must begin from the end */
          Uint8 masktable[8]={0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01};

          for (i=n; i>0; i--)
             if(buf[(i-1) >> 3] & masktable[(i-1) & 7]) buf[i-1]=0xff;
             else buf[i-1]=0x00;
        }
        break;

    default:   /* anything else to byte */
        {
          for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {
             pixval = MdcGetDoublePixel(pixel,id->type);
             if (DO_QUANT_CALIBR) {
               pixval *= (double)id->rescale_fctr;
               newval  = (float) (scale * pixval);
               buf[i]  = (Uint8) newval;
             }else{
               newval  = (float) (scale * (pixval - min));
               buf[i]  = (Uint8) newval;
             }
          }
        }
  }

  id->rescaled = MDC_YES;
  if (DO_QUANT_CALIBR) {
    id->rescaled_fctr = 1./scale;
    id->rescaled_max  = (double)((Uint8)(scale * id->qmax));
    id->rescaled_min  = (double)((Uint8)(scale * id->qmin));
  }else{ 
    id->rescaled_fctr = 1.;
    id->rescaled_max  = (double)((Uint8)(scale * (id->max - min)));
    id->rescaled_min  = (double)((Uint8)(scale * (id->min - min)));
  }

  return(buf);

} 

/* converts to Uint8 */
Uint8 *MdcGetImgBIT8_U(FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint32 size = id->width * id->height * MdcType2Bytes(BIT8_U);
  Uint8 *buffer;

  if ( (buffer=(Uint8 *)malloc(size)) == NULL )  return NULL;

  buffer=MdcMakeBIT8_U(buffer,fi,img);

  return((Uint8 *)buffer); 
  
} 

Uint8 *MdcMakeBIT16_S(Uint8 *cbuf, FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint8 *pixel, DO_QUANT_CALIBR;
  Int16 *buf = (Int16 *)cbuf;
  Uint32 i, n = id->width * id->height;
  double pixval, min, max, scale=1.0;
  double SMAX, UMAX, negmin, posmax;
  float newval;

  UMAX = (double)(1 << MDC_INT16_BITS_USED);     /* 16-bits: 65536 */
  SMAX = (double)(1 << (MDC_INT16_BITS_USED-1)); /* 16-bits: 32768 */

  /* get proper maximum/minimum value */
  if (MDC_QUANTIFY || MDC_CALIBRATE) {
    DO_QUANT_CALIBR = MDC_YES;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->qfmin;
      max = id->qfmax;
    }else{
      min = fi->qglmin;
      max = fi->qglmax;
    }
    switch (MDC_INT16_BITS_USED) {
      case 16: /* signed */
          scale = (max == 0.) ? 1. : (SMAX - 1.)/max; 
          break;
      default: /* unsigned */
          scale = (max == 0.) ? 1. : (UMAX - 1.)/max;
    }
  }else{
    DO_QUANT_CALIBR = MDC_NO;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->fmin;
      max = id->fmax;
    }else{
      min = fi->glmin;
      max = fi->glmax;
    }
    switch (MDC_INT16_BITS_USED) {
      case 16: /* signed */
        if (MDC_NEGATIVE)
          scale = (max == min) ? 1. : (UMAX - 1.)/(max - min);
        else
          scale = (max == min) ? 1. : (SMAX - 1.)/(max - min);
        break;
      default: /* unsigned */
          scale = (max == min) ? 1. : (UMAX - 1.)/(max - min);
    }
  }

  switch (MDC_INT16_BITS_USED) {
    case 16: /* signed */
        negmin = -SMAX; posmax = SMAX - 1.; break;
    default: /* unsigned */
        negmin = 0.;    posmax = UMAX - 1.;
  }
  if (MdcDoSimpleCast(min,max,negmin,posmax) == MDC_YES) {
    scale = 1.; min = 0.; SMAX = 0.;
  }

  for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {

     pixval = MdcGetDoublePixel(pixel,id->type);
     if (DO_QUANT_CALIBR) {
       pixval *= (double)id->rescale_fctr;
       newval = (float) (scale * pixval);
       buf[i] = (Int16) newval;
     }else{ 
       switch (MDC_INT16_BITS_USED) {
         case 16: /* signed */
           if (MDC_NEGATIVE) {
             newval = (float) ((scale * (pixval - min)) - SMAX);
             buf[i] = (Int16) newval;
           }else{
             newval = (float)  (scale * (pixval - min));
             buf[i] = (Int16) newval;
           }
           break;
         default: /* unsigned */
             newval = (float)  (scale * (pixval - min));
             buf[i] = (Int16) newval;
       }
     }
  }

  id->rescaled = MDC_YES;
  if (DO_QUANT_CALIBR) {
    id->rescaled_fctr = 1./scale;
    id->rescaled_max  = (double)((Int16)(scale * id->qmax));
    id->rescaled_min  = (double)((Int16)(scale * id->qmin));
  }else{
    id->rescaled_fctr = 1.;
    switch (MDC_INT16_BITS_USED) {
     case 16: /* signed */
       if (MDC_NEGATIVE) {
         id->rescaled_max = (double)((Int16)((scale * (id->max - min)) - SMAX));
         id->rescaled_min = (double)((Int16)((scale * (id->min - min)) - SMAX));
       }else{
         id->rescaled_max = (double)((Int16)(scale * (id->max - min)));
         id->rescaled_min = (double)((Int16)(scale * (id->min - min)));
       }
       break;
     default: /* unsigned */
         id->rescaled_max = (double)((Int16)(scale * (id->max - min)));
         id->rescaled_max = (double)((Int16)(scale * (id->min - min)));
    }
  }

  return((Uint8 *)buf);

}

/* converts to Int16 */
Uint8 *MdcGetImgBIT16_S(FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint32 bytes = id->width * id->height * MdcType2Bytes(BIT16_S);
  Uint8 *buffer;

  if ( (buffer=(Uint8 *)malloc(bytes)) == NULL ) return NULL;

  buffer=MdcMakeBIT16_S(buffer,fi,img);

  return(buffer);

}


Uint8 *MdcMakeBIT32_S(Uint8 *cbuf, FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint8 *pixel, DO_QUANT_CALIBR, BITS = 32;
  Int32 *buf = (Int32 *)cbuf;
  Uint32 i, n = id->width * id->height;
  double pixval, min, max, scale=1.0;
  double SMAX, UMAX;
  float newval;

  UMAX = (double)(1 <<  BITS);     /* 4294967296 */
  SMAX = (double)(1 << (BITS-1));  /* 2147483648 */

  /* get proper maximum/minimum value */
  if (MDC_QUANTIFY || MDC_CALIBRATE) {
    DO_QUANT_CALIBR = MDC_YES;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->qfmin;
      max = id->qfmax;
    }else{
      min = fi->qglmin;
      max = fi->qglmax;
    }
    scale = (max == 0.) ? 1. : (SMAX - 1.)/max;
  }else{
    DO_QUANT_CALIBR = MDC_NO;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->fmin;
      max = id->fmax;
    }else{
      min = fi->glmin;
      max = fi->glmax;
    }
    if (MDC_NEGATIVE)
      scale = (max == min) ? 1. : (UMAX - 1.)/(max - min);
    else
      scale = (max == min) ? 1. : (SMAX - 1.)/(max - min);

  }

  if (MdcDoSimpleCast(min,max,-SMAX,SMAX-1.) == MDC_YES) {
    scale = 1.; min = 0.; SMAX = 0.; 
  }

  for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {

     pixval = MdcGetDoublePixel(pixel,id->type);
     if (DO_QUANT_CALIBR) {
       pixval *= (double)id->rescale_fctr;
       newval  = (float) (scale * pixval);
       buf[i]  = (Int32) newval;
     }else{
       if (MDC_NEGATIVE) {
         newval = (float) ((scale * (pixval - min)) - SMAX);
         buf[i] = (Int32) newval;
       }else{
         newval = (float)  (scale * (pixval - min));
         buf[i] = (Int32) newval;
       }
     }
  }

  id->rescaled = MDC_YES;
  if (DO_QUANT_CALIBR) {
    id->rescaled_fctr = 1./scale;
    id->rescaled_max  = (double)((Int32)(scale * id->qmax));
    id->rescaled_min  = (double)((Int32)(scale * id->qmin));
  }else{
    id->rescaled_fctr = 1.;
    if (MDC_NEGATIVE) {
      id->rescaled_max = (double)((Int32)((scale * (id->qmax - min)) - SMAX));
      id->rescaled_min = (double)((Int32)((scale * (id->qmin - min)) - SMAX));
    }else{
      id->rescaled_max = (double)((Int32)(scale * (id->qmax - min)));
      id->rescaled_min = (double)((Int32)(scale * (id->qmin - min)));
    }
  }

  return((Uint8 *)buf);

}

/* converts to Int32 */
Uint8 *MdcGetImgBIT32_S(FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint32 size = id->width * id->height * MdcType2Bytes(BIT32_S); 
  Uint8 *buffer;

  if ( (buffer=(Uint8 *)malloc(size)) == NULL ) return NULL;

  buffer=MdcMakeBIT32_S(buffer,fi,img);

  return(buffer);

}

Uint8 *MdcMakeFLT32(Uint8 *cbuf, FILEINFO *fi, Uint32 img)
{

  IMG_DATA *id = &fi->image[img];
  Uint8 *pixel, DO_QUANT_CALIBR, DO_CAST=MDC_NO;
  float *buf = (float *)cbuf;
  Uint32 i, n = id->width * id->height;
  double pixval, min, max, scale=1.0;


  /* get proper maximum/minimum value */
  if (MDC_QUANTIFY || MDC_CALIBRATE) {
    /* do the real quantification */
    DO_QUANT_CALIBR = MDC_YES; 
    min = id->qmin;
    max = id->qmax;            

    if (id->type == FLT64) { 
      /* probably be too big for float. if global too */
      /* big, don't quantify an image but do a simple */
      /* downscaling to float and warn the user!      */ 
      if (fi->qglmax > 3.40282347e+38) {
        MdcPrntWarn("Values `double' too big for `quantified float'");
        DO_QUANT_CALIBR = MDC_NO;
        if (MDC_NORM_OVER_FRAMES) {
          min = id->fmin;
          max = id->fmax;
        }else{
          min = fi->glmin;
          max = fi->glmax;
        }
      }
    }
  }else{
    DO_QUANT_CALIBR = MDC_NO;
    if (MDC_NORM_OVER_FRAMES) {
      min = id->fmin;
      max = id->fmax;
    }else{
      min = fi->glmin;
      max = fi->glmax;
    }
  }

  if (DO_QUANT_CALIBR) {
    scale = (double)id->rescale_fctr; /* anything else fits in float */
  }else{
    /* try preserving pixel values with simple cast */
    if (id->type < FLT32 ) { 
      scale = 1.; min = 0.; DO_CAST = MDC_YES; /* ok, integers */
    }else if ( id->type == FLT64 && fabs(fi->glmax) < 3.40282347e+38 
                                 && fabs(fi->glmin) > 1e-37 ) {
      scale = 1.; min = 0.; DO_CAST = MDC_YES; /* ok, doubles fit in float */
    }else{ /* need rescaling */
      scale = (max == min) ? 1. : 3.40282347e+38 / (max - min);
      min = 0.; DO_CAST = MDC_NO;
    }
  }

  for (pixel=id->buf, i=0; i<n; i++, pixel+=MdcType2Bytes(id->type)) {

     pixval = MdcGetDoublePixel(pixel,id->type);

     if (DO_QUANT_CALIBR) {
       buf[i] = (float) (scale *  pixval);
     }else{
       buf[i] = (float) (scale * (pixval - min));
     }
  }

  id->rescaled = MDC_YES;
  if (DO_QUANT_CALIBR) {
    id->rescaled_fctr = 1.; /* got the real quantified values this time! */
    id->rescaled_max  = max; 
    id->rescaled_min  = min;
  }else if (DO_CAST == MDC_NO) {
      id->rescaled_fctr = 1.;
      id->rescaled_max  = 3.40282347e+38;
      id->rescaled_min  = 0.;
  }else{
      id->rescaled = MDC_NO;
  }

  return((Uint8 *)buf);

}
  


/* converts from FLT64 to FLT32                              */
/* or in case of quantification all other types into a FLT32 */
Uint8 *MdcGetImgFLT32(FILEINFO *fi, Uint32 img) 
{
  IMG_DATA *id = &fi->image[img];
  Uint32 bytes  = id->width * id->height * MdcType2Bytes(FLT32);
  Uint8 *buffer = NULL;

  if ( (buffer=(Uint8 *)malloc(bytes)) == NULL ) return NULL;

  buffer=MdcMakeFLT32(buffer,fi,img);

  if (buffer == NULL) return NULL;

  return(buffer);

}

Uint8 *MdcMakeImgSwapped(Uint8 *cbuf, FILEINFO *fi, Uint32 img, 
                         Uint32 width, Uint32 height, Int16 type)
{
  IMG_DATA *id = &fi->image[img];
  Uint8 *pixel=NULL;
  Uint32 pixbytes;
  Uint32 i;

  /* giving a width, heigth & type, allows to use this function directly */
  /* for swapping of none IMG_DATA image buffers                         */

  if ((type == BIT8_U) || (type == BIT8_S)) return(cbuf); /* no swap needed */

  if (width  == 0) width  = id->width;
  if (height == 0) height = id->height;
  if (type   <= 0) type   = id->type;

  pixbytes = (Uint32)MdcType2Bytes(type);

  for (i=0; i<width*height; i++) {
    pixel = cbuf + (i * pixbytes);
    MdcForceSwap(pixel,pixbytes);
  }

  return(cbuf); 

}


Uint8 *MdcGetImgSwapped(FILEINFO *fi, Uint32 img)
{
  IMG_DATA *id = &fi->image[img];
  Uint32 bytes  = id->width * id->height * MdcType2Bytes(id->type);
  Uint8 *buffer = NULL;

  if ( (buffer=(Uint8 *)malloc(bytes)) == NULL ) return NULL;

  memcpy(buffer,id->buf,bytes);

  buffer=MdcMakeImgSwapped(buffer,fi,img,0,0,0);

  return(buffer);

}
