/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-getopt.c                                                    *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : routines for handling the command-line options           *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcPrintGlobalOptions() - Print only global option usage *
 *                MdcPrintLocalOptions()  - Print only local  option usage *
 *                MdcPrintUsage()         - Print usage of medcon options  *
 *                MdcHandleArgs()         - Handle the arguments           *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-getopt.c,v 1.1.1.1 2000/10/28 16:51:04 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

/* global arrays for argument handling */
char *mdc_arg_files[MDC_MAX_FILES];    /* pointers to filenames         */
int   mdc_arg_convs[MDC_MAX_FRMTS];    /* counter for each conversion   */
int   mdc_arg_total[2];                /* totals for files & conversion */

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

/* options useful for derived programs */
void MdcPrintGlobalOptions(void)
{

  if (XMDC_MEDCON == MDC_NO) {

  printf("\n");
  printf("    -c  specify a list of conversion \"<format>\" strings:\n");
  printf("\n");
  printf("         \"ascii\" = %s (.%s)\n",FrmtString[MDC_FRMT_ASCII],
                                        FrmtExt[MDC_FRMT_ASCII]);
  printf("         \"bin\"   = %s (.%s)\n",FrmtString[MDC_FRMT_RAW],
                                        FrmtExt[MDC_FRMT_RAW]);
#if MDC_INCLUDE_GIF
  printf("         \"gif\"   = %s (.%s)\n",FrmtString[MDC_FRMT_GIF],
                                        FrmtExt[MDC_FRMT_GIF]);
#endif
#if MDC_INCLUDE_ACR
  printf("         \"acr\"   = %s (.%s)\n",FrmtString[MDC_FRMT_ACR],
                                        FrmtExt[MDC_FRMT_ACR]);
#endif
#if MDC_INCLUDE_INW
  printf("         \"inw\"   = %s (.%s)\n",FrmtString[MDC_FRMT_INW],
                                        FrmtExt[MDC_FRMT_INW]);
#endif
#if MDC_INCLUDE_INTF
  printf("         \"intf\"  = %s (.%s)+(.i33)\n",FrmtString[MDC_FRMT_INTF],
                                               FrmtExt[MDC_FRMT_INTF]);
#endif
#if MDC_INCLUDE_ECAT
  printf("         \"ecat\"  = %s (.%s)\n",FrmtString[MDC_FRMT_ECAT],
                                        FrmtExt[MDC_FRMT_ECAT]);
#endif
#if MDC_INCLUDE_ANLZ
  printf("         \"anlz\"  = %s (.%s)+(.img)\n",FrmtString[MDC_FRMT_ANLZ],
                                               FrmtExt[MDC_FRMT_ANLZ]);
#endif
#if MDC_INCLUDE_DICM
  printf("         \"dicom\" = %s (.%s)\n",FrmtString[MDC_FRMT_DICM],
                                        FrmtExt[MDC_FRMT_DICM]);
#endif
  }

  printf("\n");
  MdcWaitForEnter(0); 

  printf("\n");
  printf("======>> Pixels: [-n|-qs|-qc] [-nf] [-b8|-b16[.12]] [-big|little]\n");
  printf("\n");
  printf("    -n   negative pixels             -nf  normalize over frames\n");
  printf("    -qs  quantification (1 x scale)  -b8  write Uint8 pixel types\n");
  printf("    -qc  calibration    (2 x scale)  -b16 write Int16 pixel types\n");
  printf("                                     -b16.12  (only 12 bits used)\n");
  printf("\n");
  printf("    -big     write file in big    endian\n");
  printf("    -little  write file in little endian\n"); 
  printf("\n");
  printf("======>> Format Gif89a: [-mh|-mr|-mi|-mc|-lut file] [-g -optgif]\n");
  printf("\n");
#if MDC_INCLUDE_GIF
  if (XMDC_MEDCON == MDC_NO) {
  printf("    -mh  colormap hotmetal          -g       remap to grayscale\n");
  printf("    -mr  colormap rainbow           -optgif  give GIF89a options\n");
  }else{
  printf("    -mh  colormap hotmetal          -g       (*unused*)\n");
  printf("    -mr  colormap rainbow           -optgif  (*unused*)\n");
  }
  printf("    -mi  colormap inverted (gray)   -lut     load specified LUT\n");
  printf("    -mc  colormap combined\n");
#else
  printf("    -mh  (*unused*)                 -g       (*unused*)\n");
  printf("    -mr  (*unused*)                 -optgif  (*unused*)\n");
  printf("    -mi  (*unused*)                 -lut     (*unused*)\n");
  printf("    -mc  (*unused*)\n");
#endif
  printf("\n");
  printf("======>> Format Analyze: [-ar -spm -optspm]\n");
  printf("\n");
#if MDC_INCLUDE_ANLZ
  printf("    -ar  analyze reverse            -spm     analyze for SPM\n");
  if (XMDC_MEDCON == MDC_NO) {
  printf("                                    -optspm  give SPM options\n");
  }else{
  printf("                                    -optspm  (*unused*)\n");
  }
#else
  printf("    -ar  (*unused*)                 -spm     (*unused*)\n");
  printf("                                    -optspm  (*unused*)\n");
#endif
  printf("\n");
  MdcWaitForEnter(0);
  printf("\n");
  printf("======>> Fallback Read Format: [-fb-none|-fb-anlz|-fb-ecat|fb-dicom]\n");
  printf("\n");
  printf("    -fb-none    fallback disabled\n");
#if MDC_INCLUDE_ANLZ
  printf("    -fb-anlz    fallback on Analyze (SPM)\n");
#else
  printf("    -fb-anlz    (*unused*)\n");
#endif
#if MDC_INCLUDE_ECAT
  printf("    -fb-ecat    fallback on ECAT 6.4\n");
#else
  printf("    -fb-ecat    (*unused*)\n");
#endif
#if MDC_INCLUDE_DICM
  printf("    -fb-dicom   fallback on DICOM 3.0\n");
#else
  printf("    -fb-dicom   (*unused*)\n");
#endif
  printf("\n");
}

/* options for MedCon in particular */
void MdcPrintLocalOptions(void)
{
  printf("======>> Patient/Slice/Study: [-anon|-ident] [-vifi]\n");
  printf("\n");
  printf("    -anon     make patient/study anonymous\n");
  printf("    -ident    give patient/study information\n");
  printf("    -vifi     edit FILEINFO entries (image/slice/orient)\n");
  printf("\n");
  printf("======>> Debugging: [-d -v -hackacr]\n");
  printf("\n");
  printf("    -d   debug mode (printout FI)    -hackacr   try to find tags");
#if MDC_INCLUDE_ACR
  printf("\n");
#else
  printf(" (*unused*)\n");
#endif
  printf("    -v   verbose mode\n");
  printf("\n");
  printf("======>> Personnal: [-db]\n");
  printf("\n");
  printf("    -db  print database info (ECAT)\n");
  printf("\n");
}

void MdcPrintUsage(char *pgrname)
{

  if (pgrname == NULL) {

  /* usage for MedCon in particular */
  MdcPrintLine('*',MDC_FULL_LENGTH);
  printf("%s (C) %s ",MDC_LIBVERS,MDC_DATE);
  printf("\t  Medical Image Conversion\n");
  MdcPrintLine('*',MDC_FULL_LENGTH);
  printf("Usage  : medcon [options] -f <files> ...\n");
  printf("\n");
  printf("Options: General: [-i -e -r] [-p|-c <format> ...]\n");
  printf("\n");
  printf("    -i  interactive read     -e  extract images   -r  rename file\n");
  printf("    -p  show pixel values\n");
  MdcPrintGlobalOptions();
  MdcPrintLocalOptions();

  }else{

  /* usage for any derived program */
  printf("\n");
  printf("Usage  : %s [options] -f files ...\n",pgrname);
  if (XMDC_MEDCON == MDC_NO) {
  printf("\n");
  printf("Options: General: [-c <format> ...]\n");
  }
  MdcPrintGlobalOptions();

  }

  exit(0);

}

int MdcHandleArgs(FILEINFO *fi, int argc, char *argv[], int MAXFILES)
{
  int a,ARG=-1;
  char **files = mdc_arg_files;
  int   *convs = mdc_arg_convs;
  int   *total = mdc_arg_total;

  /* limit number of files */
  if (MAXFILES > MDC_MAX_FILES || MAXFILES <= 0) MAXFILES = MDC_MAX_FILES;
 
  fi->map = MDC_MAP_GRAY;      /* default color map */
 
  if ( argc == 1 ) return(MDC_BAD_CODE);


  /* initialize some stuff */
  files[0]=NULL;  
  memset(convs,0,sizeof(int)*MDC_MAX_FRMTS);
  memset(total,0,sizeof(int)*2);

  for (a=1; a<argc; a++) {

     if ( strcasecmp(argv[a],"-c") == 0 ) {       /* begin list of convs */
       if (XMDC_MEDCON == MDC_NO) {
         MDC_INFO=MDC_NO; MDC_CONVERT=MDC_YES; ARG=MDC_ARG_CONV;
       }
       continue;
     }else if ( strcasecmp(argv[a],"-i") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_INTERACTIVE=MDC_YES;
       }
       continue;
     }else if ( strcasecmp(argv[a],"-e") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_EXTRACT=MDC_YES; 
       }
       continue;
     }else if ( strcasecmp(argv[a],"-db") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_INFO_DB = MDC_YES; MDC_INFO = MDC_NO; 
       }
       continue;
     }else if ( strcasecmp(argv[a],"-anon") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_PATIENT_ANON = MDC_YES;
       }
       continue;
     }else if ( strcasecmp(argv[a],"-ident") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_PATIENT_IDENT = MDC_YES; 
       }
       continue;
     }else if ( strcasecmp(argv[a],"-p") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_PIXELS=MDC_YES;   MDC_INFO=MDC_NO;
         MDC_NEGATIVE=MDC_YES; MDC_CALIBRATE=MDC_YES; 
       }
       continue;
     }else if ( strcasecmp(argv[a],"-r") == 0 ) {
       if (XMDC_MEDCON == MDC_NO) {
         MDC_RENAME = MDC_YES; 
       }
       continue;
     }else if ( strcasecmp(argv[a],"-f") == 0 ) { /* begin list of files */
       ARG=MDC_ARG_FILE; continue;
     }else if ( strcasecmp(argv[a],"-n") == 0 ) {
       MDC_NEGATIVE=MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-q") == 0 ) {
       MdcPrntWarn("Option `-q' obsolete!  Using '-qc' option as default");
       MDC_CALIBRATE=MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-qs") == 0 ) {
       MDC_QUANTIFY=MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-qc") == 0 ) {
       MDC_CALIBRATE=MDC_YES; continue;
#if MDC_INCLUDE_GIF
     }else if ( strcasecmp(argv[a],"-mr") == 0 ) {
       MDC_COLOR_MAP = MDC_MAP_RAINBOW;  MDC_MAKE_GRAY = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-mc") == 0 ) {
       MDC_COLOR_MAP = MDC_MAP_COMBINED; MDC_MAKE_GRAY = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-mh") == 0 ) {
       MDC_COLOR_MAP = MDC_MAP_HOTMETAL; MDC_MAKE_GRAY = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-mi") == 0 ) {
       MDC_COLOR_MAP = MDC_MAP_GRAY;   MDC_CMAP_INVERS = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-optgif") == 0 ) {
       MDC_GIF_OPTIONS = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-g") == 0 ) {
       MDC_COLOR_MAP = MDC_MAP_GRAY;     MDC_MAKE_GRAY = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-lut") == 0 ) {
       a+=1; 
       if ((a < argc) && (MdcLoadLUT(argv[a]) == MDC_YES)) {
         MDC_COLOR_MAP = MDC_MAP_LOADED; MDC_MAKE_GRAY = MDC_YES;
       }
       continue;
#endif
#if MDC_INCLUDE_ACR
     }else if ( strcasecmp(argv[a],"-hackacr") == 0 ) {
       MDC_HACK_ACR = MDC_YES; continue;
#endif
#if MDC_INCLUDE_ANLZ
     }else if ( strcasecmp(argv[a],"-ar") == 0 ) {
       MDC_ANLZ_REV = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-spm") == 0 ) {
       MDC_ANLZ_SPM = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-optspm") == 0 ) {
       MDC_ANLZ_OPTIONS = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-fb-anlz") == 0 ) {
       MDC_FALLBACK_FRMT = MDC_FRMT_ANLZ; continue;
#endif
#if MDC_INCLUDE_ECAT
     }else if ( strcasecmp(argv[a],"-fb-ecat") == 0 ) {
       MDC_FALLBACK_FRMT = MDC_FRMT_ECAT; continue;
#endif
#if MDC_INCLUDE_DICM
     }else if ( strcasecmp(argv[a],"-fb-dicom") == 0 ) {
       MDC_FALLBACK_FRMT = MDC_FRMT_DICM; continue;
#endif
     }else if ( strcasecmp(argv[a],"-fb-none") == 0 ) {
       MDC_FALLBACK_FRMT = MDC_FRMT_BAD; continue;
     }else if ( strcasecmp(argv[a],"-d") == 0 ) {
       MDC_DEBUG = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-nf") == 0 ) {
       MDC_NORM_OVER_FRAMES = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-v") == 0 ) {
       MDC_VERBOSE  = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-b8") == 0 ) {
       MDC_FORCE_INT = BIT8_U;  continue;
     }else if ( strcasecmp(argv[a],"-b16") == 0 ) {
       MDC_FORCE_INT = BIT16_S; MDC_INT16_BITS_USED=16; continue;
     }else if ( strcasecmp(argv[a],"-b16.12") == 0 ) {
       MDC_FORCE_INT = BIT16_S; MDC_INT16_BITS_USED=12; continue;
     }else if ( strcasecmp(argv[a],"-sigfpe") == 0 ) { /* debug: undocumented */
       MDC_SIGFPE   = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-debug") == 0 )  { /* debug: undocumented */
       MDC_MY_DEBUG = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-vifi") == 0 ) {
       MDC_EDIT_FI = MDC_YES; continue;
     }else if ( strcasecmp(argv[a],"-big") == 0 ) {
       MDC_WRITE_ENDIAN=MDC_BIG_ENDIAN; continue;
     }else if ( strcasecmp(argv[a],"-little") == 0 ) {
       MDC_WRITE_ENDIAN=MDC_LITTLE_ENDIAN; continue;
     }

     
     if ( ARG == -1 ) return(MDC_BAD_CODE);              /* no '-c' or '-f' */

     switch (ARG) {

       case MDC_ARG_CONV:
         if ( strcasecmp(argv[a],"ascii")== 0) {
             convs[MDC_FRMT_ASCII]+=1; total[MDC_CONVS]+=1;
         }else
         if ( strcasecmp(argv[a],"bin")  == 0) {
             convs[MDC_FRMT_RAW]+=1; total[MDC_CONVS]+=1;
         }else
#if MDC_INCLUDE_ACR
         if ( strcasecmp(argv[a],"acr")  == 0 ) {
             convs[MDC_FRMT_ACR]+=1; total[MDC_CONVS]+=1;
         }else
#endif
#if MDC_INCLUDE_GIF
         if ( strcasecmp(argv[a],"gif")  == 0 ) {
             convs[MDC_FRMT_GIF]+=1; total[MDC_CONVS]+=1;
         }else 
#endif
#if MDC_INCLUDE_INW
         if ( strcasecmp(argv[a],"inw")  == 0 ) {
             convs[MDC_FRMT_INW]+=1; total[MDC_CONVS]+=1;
         }else
#endif
#if MDC_INCLUDE_ECAT
         if ( strcasecmp(argv[a],"ecat") == 0 ) {
             convs[MDC_FRMT_ECAT]+=1; total[MDC_CONVS]+=1;
         }else
#endif
#if MDC_INCLUDE_INTF
         if ( strcasecmp(argv[a],"intf") == 0 ) {
             convs[MDC_FRMT_INTF]+=1; total[MDC_CONVS]+=1;
         }else
#endif
#if MDC_INCLUDE_ANLZ
         if ( strcasecmp(argv[a],"anlz") == 0 ) {
             convs[MDC_FRMT_ANLZ]+=1; total[MDC_CONVS]+=1;
         }else
#endif
#if MDC_INCLUDE_DICM
         if ( strcasecmp(argv[a],"dicom") == 0) {
             convs[MDC_FRMT_DICM]+=1; total[MDC_CONVS]+=1;
         }else
#endif
         MdcPrntErr(MDC_BAD_CODE,"Unsupported conversion `%s'",argv[a]);
 
         break;

       case MDC_ARG_FILE:
         if ( total[MDC_FILES] == MAXFILES )
           MdcPrntErr(MDC_OVER_FLOW,"Too many files specified (max=%d)"
                                   ,MAXFILES);
         else
           files[total[MDC_FILES]]=argv[a]; total[MDC_FILES]+=1;

         break;
     }

  }

  /* need an image file, except for XMedCon */
  if ( total[MDC_FILES] == 0  && XMDC_MEDCON == MDC_NO) 
    MdcPrntErr(MDC_NO_CODE,"No files specified");


  /* check the options */

  /* quantification troubles */
  if (MDC_INFO == MDC_YES) { 
    if (MDC_QUANTIFY == MDC_YES) {
      MDC_NEGATIVE=MDC_YES; MDC_CALIBRATE=MDC_NO;
    }else{
      MDC_NEGATIVE=MDC_YES; MDC_CALIBRATE=MDC_YES; 
    }
  }

  if ((MDC_ANLZ_SPM == MDC_YES) && (MDC_QUANTIFY == MDC_NO) 
                                && (MDC_CALIBRATE == MDC_NO) ) 
    MdcPrntWarn("For SPM scaling you should select a quantification option");

  if (MDC_INTERACTIVE == MDC_YES) {
    MdcPrntWarn("Enabling negative pixels & disabling quantification");
    MDC_NEGATIVE  = MDC_YES;
    MDC_QUANTIFY  = MDC_NO;
    MDC_CALIBRATE = MDC_NO;
  }

  if ( (MDC_PIXELS == MDC_YES) && (MDC_QUANTIFY == MDC_YES) )
    MDC_CALIBRATE = MDC_NO;

  /* conversion related issues */
  if (MDC_CONVERT  == MDC_YES || (XMDC_MEDCON == MDC_YES)) {
    /* with output file */
    if (XMDC_MEDCON == MDC_NO) {
      if ( total[MDC_CONVS] == 0 ) 
        MdcPrntErr(MDC_NO_CODE,"No conversion formats specified");
      if (MDC_PIXELS   == MDC_YES)
        MdcPrntErr(MDC_NO_CODE,"Options '-c' and '-p' are mutually exclusive");
    }
    if (MDC_NEGATIVE == MDC_YES) {
      if (MDC_QUANTIFY == MDC_YES)
     MdcPrntErr(MDC_NO_CODE,"Options '-n' & '-qs' are mutually exclusive");
      else if (MDC_CALIBRATE == MDC_YES)
     MdcPrntErr(MDC_NO_CODE,"Options '-n' & '-qc' are mutually exclusive");
    }
  }else{
    /* without output file */
    if (MDC_RENAME == MDC_YES)
      MdcPrntWarn("Option '-r' rename file ignored");
    if (MDC_EXTRACT == MDC_YES)
      MdcPrntErr(MDC_NO_CODE,"Need conversion specification");
  }

  /* study identification */
  if ( (MDC_PATIENT_ANON == MDC_YES) && (MDC_PATIENT_IDENT == MDC_YES) )
    MdcPrntErr(MDC_NO_CODE,"Options '-anon' & '-ident' are mutually exclusive");

  /* the must do FINAL quantification/calibration check */ 
  if ( (MDC_QUANTIFY == MDC_YES) && (MDC_CALIBRATE == MDC_YES) )
    MdcPrntErr(MDC_NO_CODE,"Options '-qs' & '-qc' are mutually exclusive");

  /* giving an overview of settings */
  if (MDC_VERBOSE ) {

    if (MDC_WRITE_ENDIAN == MDC_LITTLE_ENDIAN)
      MdcPrntMesg("Writing in little endian as default");
    else
      MdcPrntMesg("Writing in big endian as default");

    switch (MDC_FALLBACK_FRMT) {
      case MDC_FRMT_ANLZ:
         MdcPrntMesg("Read fallback format Analyze (SPM)"); break;
      case MDC_FRMT_DICM:
         MdcPrntMesg("Read fallback format DICOM 3.0");     break;
      case MDC_FRMT_ECAT:
         MdcPrntMesg("Read fallback format ECAT 6.4");      break;
    }

    if (MDC_QUANTIFY == MDC_YES) 
    MdcPrntMesg("Quantification     is ON  (ECAT units=[counts/second/pixel])");
    if (MDC_CALIBRATE == MDC_YES)
    MdcPrntMesg("Calibration        is ON  (ECAT units=[uCi/ml])");
    if (MDC_NEGATIVE  == MDC_YES)
    MdcPrntMesg("Negative pixels    is ON");
    if (MDC_ANLZ_REV  == MDC_YES)
    MdcPrntMesg("Analyze reverse    is ON");
    if (MDC_ANLZ_SPM  == MDC_YES)
    MdcPrntMesg("Analyze for SPM    is ON");
    if (MDC_NORM_OVER_FRAMES == MDC_YES) 
    MdcPrntMesg("Norm over frames   is ON");
    if (MDC_FORCE_INT != MDC_NO) {
      switch (MDC_FORCE_INT) {
        case BIT8_U : 
            MdcPrntMesg("Writing Uint8 pixs is ON  (quantified values lost!)");
            break;
        case BIT16_S: 
            MdcPrntMesg("Writing Int16 pixs is ON  (quantified values lost!)");
            break;
        default     : 
            MdcPrntMesg("Writing Int16 pixs is ON  (quantified values lost!)");
      }

      if (MDC_INT16_BITS_USED == 12) 
            MdcPrntMesg("Using only 12 bits is ON");
      
    }
  }

  return(MDC_OK);

}

