/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-files.c                                                     *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : Files edit, file I/O, process FI/ID, image buffers       *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcOpenFile()         - Open file                        *
 *                MdcReadFile()         - Read file                        *
 *                MdcWriteFile()        - Write file                       * 
 *                MdcDecompressFile()   - Decompress file                  *
 *                MdcCheckFI()          - Check FILEINFO struct integrity  * 
 *                MdcStringCopy()       - Copy a string                    *
 *                MdcFileSize()         - Get the size of a file           *
 *                MdcFileExists()       - Check if file exists             *
 *                MdcGetFrmt()          - Get format of imagefile          *
 *                MdcGetImgBuffer()     - Malloc a buffer                  *
 *                MdcHandleTruncated()  - Reset FILEINFO for truncated file*
 *                MdcWriteLine()        - Write image line                 *
 *                MdcWriteDoublePixel() - Write single pixel double input  *
 *                MdcGetStructID()      - Get IMG_DATA structs             *
 *                MdcInitID()           - Initialize IMG_DATA structs      *
 *                MdcResetIDs()         - Reset IMG_DATA structs           *
 *                MdcGetStructAD()      - Get ACQ_DATA structs             *
 *                MdcInitAD()           - Initialize ACQ_DATA structs      *
 *                MdcInitFI()           - Initialize FILEINFO struct       *
 *                MdcFreeIDs()          - Free IMG_DATA structs            *
 *                MdcCleanUpFI()        - Clean up FILEINFO struct         *
 *                MdcGetFname()         - Get filename                     *
 *                MdcGetExtension()     - Get filename original extension  *
 *                MdcGetLastPathSlash() - Get pointer to last path-slash   * 
 *                MdcMySplitPath()      - Split path from filename         *
 *                MdcMyMergePath()      - Merge path to filename           *
 *                MdcNewExt()           - Give filename a new extension    *
 *                MdcNewName()          - Create new filename              *
 *                MdcDefaultName()      - Create new filename for format   *
 *                MdcRenameFile()       - Let user give a new name         *
 *                MdcFillImgPos()       - Fill the image_pos(_dev/_pat)    *
 *                MdcFillImgOrient()    - Fill the image_orient(_dev/_pat) *
 *                MdcGetPatSliceOrient()- Get patient_slice_orient         *
 *                MdcTryPatSliceOrient()- Try to get it from pat_orient    *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-files.c,v 1.1.1.1 2000/10/28 16:51:03 enlf Exp $ 
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */
 
/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/
int MdcOpenFile(FILEINFO *fi, char *path)
{
  int ctype;

  if ( (ctype = MdcWhichCompression(path)) != MDC_NO ) {
    if ((MdcDecompressFile(path)) != MDC_OK) {

      strcpy(mdcbufr,"Decompression failed");
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayErr(mdcbufr);
      }else
#endif
        MdcPrntWarn(mdcbufr);

      return(MDC_BAD_OPEN);
    }
  }

  MdcInitFI(fi, path);  fi->compression = (Int8)ctype;


  if ( (fi->ifp=fopen(fi->ipath,"rb")) == NULL) {
    sprintf(mdcbufr,"Couldn't open <%s> for reading",fi->ipath);
#if XSUPPORTED
     if (XMDC_MEDCON) {
       XMdcDisplayWarn(mdcbufr);
     }else
#endif
       MdcPrntWarn(mdcbufr);

    return(MDC_BAD_OPEN);
  }

  if (ctype != MDC_NO) unlink(path);

  MdcSplitPath(fi->ipath,fi->idir,fi->ifname);
 
  return(MDC_OK);
  
}

int MdcReadFile(FILEINFO *fi, int filenr)
{
  int FORMAT=MDC_FRMT_BAD;
  char *msg=NULL;

  /* get the format or fallback */
  if ( (FORMAT=MdcGetFrmt(fi) ) == MDC_FRMT_BAD ) {
    MdcFileClose(fi->ifp);
    sprintf(mdcbufr,"Unsupported format in <%s>",fi->ifname);
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayWarn(mdcbufr);
    }else
#endif
      MdcPrntWarn(mdcbufr);

    return(MDC_BAD_CODE);

  }else if (FORMAT < 0) {
    MdcFileClose(fi->ifp);
    sprintf(mdcbufr,"Unsuccessful read from <%s>",fi->ifname);
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayErr(mdcbufr);
    }else
#endif
      MdcPrntWarn(mdcbufr);

    return(MDC_BAD_READ);
  }

  /* print a header message */
  if (MDC_INFO && !MDC_INTERACTIVE) {
    printf("\n");
    MdcPrintLine('*',MDC_FULL_LENGTH);
    printf("FILE %03d : %s\t\t\t",filenr,fi->ifname);
    printf("FORMAT: %s\n",FrmtString[fi->format]);
    MdcPrintLine('*',MDC_FULL_LENGTH);
    printf("\n");
  }

  /* read the appropriate format */
  switch (FORMAT) {

    case MDC_FRMT_RAW:  msg=MdcReadRAW(fi);  break;
#if MDC_INCLUDE_ACR
    case MDC_FRMT_ACR:  msg=MdcReadACR(fi);  break;
#endif
#if MDC_INCLUDE_GIF
    case MDC_FRMT_GIF:  msg=MdcReadGIF(fi);  break;
#endif
#if MDC_INCLUDE_INW
    case MDC_FRMT_INW:  msg=MdcReadINW(fi);  break;
#endif
#if MDC_INCLUDE_ECAT
    case MDC_FRMT_ECAT: msg=MdcReadECAT(fi); break;
#endif
#if MDC_INCLUDE_INTF 
    case MDC_FRMT_INTF: msg=MdcReadINTF(fi); break;
#endif
#if MDC_INCLUDE_ANLZ
    case MDC_FRMT_ANLZ: msg=MdcReadANLZ(fi); break;
#endif
#if MDC_INCLUDE_DICM
    case MDC_FRMT_DICM: msg=MdcReadDICM(fi); break;
#endif
    default:
      sprintf(mdcbufr,"Reading: Unsupported format");
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayErr(mdcbufr);
      }else
#endif
        MdcPrntWarn(mdcbufr);

      return(MDC_BAD_FILE);
  }

  /* read error handling */
  if (msg != NULL) {
    sprintf(mdcbufr,"Reading: %s",msg);
    if (strstr(msg,"Truncated image") == NULL) {
      MdcCleanUpFI(fi);
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayErr(mdcbufr);
      }else
#endif
        MdcPrntWarn(mdcbufr);

      return(MDC_BAD_READ);

    }else{
      MdcFileClose(fi->ifp);
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayWarn(mdcbufr);
      }else
#endif
        MdcPrntWarn(mdcbufr);
    }
  }

  /* database info | dangerous: not all image info read */
  if (MDC_INFO_DB == MDC_YES) return(MDC_OK);

  /* set the color map */
  if (MDC_COLOR_MAP) {
    fi->map = MDC_COLOR_MAP;
    MdcGetColorMap(fi->map,fi->palette);
  }
 
  /* the obligated pixel handling */
  msg = MdcImagesPixelFiddle(fi);
  if (msg != NULL) {
    MdcCleanUpFI(fi);
    sprintf(mdcbufr,"Reading: %s",msg);
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayErr(mdcbufr);
     }else
#endif
      MdcPrntErr(MDC_BAD_CODE,"%s",mdcbufr);

    return(MDC_BAD_CODE);
  }
 
  return(MDC_OK);
 
}


int MdcWriteFile(FILEINFO *fi, int format, int prefixnr)
{
  char *msg=NULL;

  /* reset ID's rescaled stuff from any previous write */
  MdcResetIDs(fi);

  /* negative value = self made prefix */
  if (prefixnr >= 0 ) MdcPrefix(prefixnr);

  switch (format) {
    case MDC_FRMT_RAW  : fi->rawconv = MDC_FRMT_RAW; 
                         msg=MdcWriteRAW(fi);
                         break;
    case MDC_FRMT_ASCII: fi->rawconv = MDC_FRMT_ASCII;
                         msg=MdcWriteRAW(fi);      
                         break;
#if MDC_INCLUDE_ACR
    case MDC_FRMT_ACR  : msg=MdcWriteACR(fi);  break; 
#endif
#if MDC_INCLUDE_GIF
    case MDC_FRMT_GIF  : msg=MdcWriteGIF(fi);  break;
#endif
#if MDC_INCLUDE_INW
    case MDC_FRMT_INW  : msg=MdcWriteINW(fi);  break;
#endif
#if MDC_INCLUDE_ECAT
    case MDC_FRMT_ECAT : msg=MdcWriteECAT(fi); break;
#endif
#if MDC_INCLUDE_INTF
    case MDC_FRMT_INTF : msg=MdcWriteINTF(fi); break;
#endif
#if MDC_INCLUDE_ANLZ
    case MDC_FRMT_ANLZ : msg=MdcWriteANLZ(fi); break;
#endif
#if MDC_INCLUDE_DICM
    case MDC_FRMT_DICM : msg=MdcWriteDICM(fi); break;
#endif
    default:
      sprintf(mdcbufr,"Writing: Unsupported format");
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayErr(mdcbufr);
      }else
#endif
        MdcPrntWarn(mdcbufr);

      return(MDC_BAD_FILE);

  } 

  MdcFileClose(fi->ofp);

  if (msg != NULL) {
    sprintf(mdcbufr,"Writing: %s",msg);
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcDisplayErr(mdcbufr);
    }else
#endif
      MdcPrntWarn(mdcbufr);

    return(MDC_BAD_WRITE);
  }

  return(MDC_OK);
 
}

int MdcDecompressFile(char *path)
{
  char *ext;

  /* print some info */
#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Decompression (Waiting)");
#endif

  if (MDC_VERBOSE) MdcPrntMesg("Decompression ...");

  /* get last extension (.gz or .Z) */
  ext = strrchr(path,'.');
  /* build system call, put paths between quotes     */
  /* in order to catch at least some weird filenames */
  sprintf(mdcbufr,"%s -c \"%s\" > \"",MDC_DECOMPRESS,path);
  /* remove extension from filename */
  *ext = '\0';
  /* add to pipe of system call */
  strcat(mdcbufr,path); strcat(mdcbufr,"\"");

#if XSUPPORTED
  if (XMDC_MEDCON) XMdcUpdateProgressBar(NULL);
#endif

  /* check if decompressed file already exists */
  if (MdcFileExists(path) == MDC_YES) {
    strcpy(mdcbufr,"Decompressed filename already exists");
#if XSUPPORTED
    if (XMDC_MEDCON) {
      XMdcEndProgressBar(); XMdcDisplayWarn(mdcbufr);
    }else
#endif
      MdcPrntWarn(mdcbufr);

    return(MDC_BAD_CODE);
  }

#if XSUPPORTED
  if (XMDC_MEDCON) XMdcUpdateProgressBar(NULL);
#endif

  if (system(mdcbufr)) {

#if XSUPPORTED
    if (XMDC_MEDCON) XMdcEndProgressBar();
#endif

    unlink(path); return(MDC_BAD_CODE);

  }

  return(MDC_OK);

}


char *MdcCheckFI(FILEINFO *fi)
{
  Uint32 i, t;

  /* check fi->dim[] values */ 
  if (fi->dim[0] <= 2 ) {
    sprintf(mdcbufr,"Internal ## fi->dim[0]=%d",fi->dim[0]);
    return(mdcbufr);
  }else{
    for (i=1; i<=fi->dim[0]; i++) {
       if (fi->dim[i] <= 0 ) {
         sprintf(mdcbufr,"Internal ## fi->dim[%d]=%d",i,fi->dim[i]);
         return(mdcbufr);
       }
    }
  }

  /* all fi->dim[] are 1-based, even unused */
  for (i=0; i<8; i++) if (fi->dim[i] <= 0)
     return("Internal ## Dangerous negative fi->dim values");


  /* check fi->number value */
  for (i=1, t=3; t <= fi->dim[0]; t++) {
     i*=fi->dim[t];
  }
  if (fi->number != i) return("Internal ## Improper fi->dim values");

  return(NULL);

}


void MdcStringCopy(char *s1, char *s2, Uint32 length)
{
  if ( length < MDC_MAXSTR) {
    memcpy(s1,s2,length);
    s1[length] = '\0';
  }else{
    memcpy(s1,s2,MDC_MAXSTR);
    s1[MDC_MAXSTR-1] = '\0';
  }
}
 

int MdcFileSize(FILE *fp)
{
  int size;

  fseek(fp,0,SEEK_END);

  size=ftell(fp);

  fseek(fp,0,SEEK_SET);

  return(size);

}

int MdcFileExists(char *fname)
{

  FILE *fp;

  if ((fp=fopen(fname,"rb")) == NULL) return MDC_NO;

  MdcFileClose(fp);

  return MDC_YES;

} 


int MdcGetFrmt(FILEINFO *fi) 
{
  int i, format=MDC_FRMT_BAD;

  if (MDC_INTERACTIVE) { fi->format = MDC_FRMT_RAW; return(MDC_FRMT_RAW); }

  for (i=3; i<MDC_MAX_FRMTS; i++) {
   
    switch (i) {

#if MDC_INCLUDE_ACR
     case MDC_FRMT_ACR:   format = MdcCheckACR(fi);  break;
#endif
#if MDC_INCLUDE_GIF
     case MDC_FRMT_GIF:   format = MdcCheckGIF(fi);  break;
#endif
#if MDC_INCLUDE_INW
     case MDC_FRMT_INW:   format = MdcCheckINW(fi);  break; 
#endif
#if MDC_INCLUDE_INTF
     case MDC_FRMT_INTF:  format = MdcCheckINTF(fi); break;
#endif
#if MDC_INCLUDE_ANLZ
     case MDC_FRMT_ANLZ:  format = MdcCheckANLZ(fi); break;
#endif
#if MDC_INCLUDE_ECAT
     case MDC_FRMT_ECAT:  format = MdcCheckECAT(fi); break;
#endif
#if MDC_INCLUDE_DICM
     case MDC_FRMT_DICM:  format = MdcCheckDICM(fi); break;
#endif
    }

    fseek(fi->ifp,0,SEEK_SET); 

    if ( format != MDC_FRMT_BAD ) break;

  }

  if (format == MDC_FRMT_BAD) {
    if (MDC_FALLBACK_FRMT != MDC_FRMT_BAD) {
      sprintf(mdcbufr,"Image format unknown - trying fallback format");
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayWarn(mdcbufr);
      }else
#endif
        MdcPrntWarn(mdcbufr);
 
      format = MDC_FALLBACK_FRMT;

    }
  }

  fi->format = format;

  return(format);

}


Uint8 *MdcGetImgBuffer(Uint32 bytes)
{
  return((Uint8 *)calloc(1,bytes));
}

char *MdcHandleTruncated(FILEINFO *fi, Uint32 images, int remap)
{  
   if (images == 0) images = 1;
   if ((remap == MDC_YES) && (images < fi->number)) {
     fi->image=(IMG_DATA *)MdcRealloc(fi->image, sizeof(IMG_DATA)*images);
     if (fi->image == NULL) 
       return("Couldn't realloc truncated IMG_DATA structs");
   }

   fi->truncated = MDC_YES;
   fi->number = images;
   fi->dim[0] = 3;
   fi->dim[3] = fi->number;
   fi->dim[4] = fi->dim[5] = fi->dim[6] = fi->dim[7] = 0;

   return NULL;
}

/* put whole line but MdcSWAP if necessary */
/* on success 1 on error 0 */
int MdcWriteLine(IMG_DATA *id, Uint8 *buf, int type, FILE *fp)
{
  Uint32 i, bytes = MdcType2Bytes(type);
  Uint8 *pbuf;

  if (bytes == 1) {

    fwrite(buf,id->width,bytes,fp); /* no MdcSWAP necessary */

  }else for (i=0; i<id->width; i++) {

   pbuf = buf + (i * bytes);

   switch (type) {
    case BIT16_S:
     {
        Int16 pix;

        memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
        fwrite((char *)&pix,1,bytes,fp);
     }
     break;
    case BIT16_U:
     { 
        Uint16 pix;

        memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
        fwrite((char *)&pix,1,bytes,fp);
     }
     break;
    case BIT32_S:
     {
        Int32 pix;

        memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
        fwrite((char *)&pix,1,bytes,fp);
     }
     break;
    case BIT32_U:
     {
        Uint32 pix;

        memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
        fwrite((char *)&pix,1,bytes,fp);
     }
     break;
#ifdef HAVE_8BYTE_INT
    case BIT64_S:
     {
        Int64 pix;

        memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
        fwrite((char *)&pix,1,bytes,fp);
     }
     break;
    case BIT64_U:
     {
        Uint64 pix;

        memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
        fwrite((char *)&pix,1,bytes,fp);
     }
     break;
#endif
    case FLT32:
     {
       float pix;

       memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
       fwrite((char *)&pix,1,bytes,fp);
     }
     break;
    case FLT64:
     {
       double pix;

       memcpy(&pix,pbuf,bytes); MdcSWAP(pix);
       fwrite((char *)&pix,1,bytes,fp);
     }
     break;
    case VAXFL32:
     {
       float  flt;

       memcpy(&flt,pbuf,bytes); 
       MdcMakeVAXfl(flt); 
       fwrite((char *)&flt,1,bytes,fp);

     }
     break;

   }
 }
 
 if (ferror(fp)) return MDC_NO;

 return MDC_YES;

} 

/* Put pixel but MdcSWAP if necessary */
/* on success 1  on error 0 */
int MdcWriteDoublePixel(double pix, int type, FILE *fp)
{
 int bytes = MdcType2Bytes(type);

 switch (type) {
  case BIT8_S:
   {
     Int8 c = (Int8)pix;
     fwrite((char *)&c,1,bytes,fp);
   }
   break; 
  case BIT8_U:
   {
     Uint8 c = (Uint8)pix;
     fwrite((char *)&c,1,bytes,fp);
   }
   break;
  case BIT16_S:
   {
     Int16 c = (Int16)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
  case BIT16_U:
   {
     Uint16 c = (Uint16)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
  case BIT32_S:
   {
     Int32 c = (Int32)pix; 
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
  case BIT32_U:
   {
     Uint32 c = (Uint32)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
#ifdef HAVE_8BYTE_INT
  case BIT64_S:
   {
     Int64 c = (Int64)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
  case BIT64_U:
   {
     Uint64 c = (Uint64)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
#endif 
  case FLT32:
   {
     float c = (float)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
  case VAXFL32:
   {
     float flt = (float)pix;

     MdcMakeVAXfl(flt);
     fwrite((char *)&flt,1,bytes,fp);

   }
   break; 
  case FLT64:
   {
     double c = (double)pix;
     MdcSWAP(c); fwrite((char *)&c,1,bytes,fp);
   }
   break;
 }

 if (ferror(fp)) return MDC_NO;
 
 return MDC_YES;

}


/* returns 1 on success and 0 on failure        */
/* all at once (malloc) or one by one (realloc) */
int  MdcGetStructID(FILEINFO *fi)
{
  Uint32 i, begin; 

  if (fi->image == NULL) {
    fi->image=(IMG_DATA *)malloc(sizeof(IMG_DATA)*fi->number); 
    begin = 0; 
  }else{ 
    fi->image=(IMG_DATA *)realloc(fi->image,sizeof(IMG_DATA)*fi->number);
    begin = fi->number - 1;
  }

  if (fi->image == NULL) return MDC_NO; 

  for (i=begin; i<fi->number; i++) MdcInitID(&fi->image[i]);  

  return MDC_YES;
}

void MdcInitID(IMG_DATA *id)
{
  int i;

  memset(id,'\0',sizeof(IMG_DATA));

  id->rescaled = MDC_NO;
  id->quant_scale = id->calibr_fctr = id->rescale_fctr = 1.;
  id->quant_units = id->calibr_units = 1;
  id->buf = NULL;

  id->pixel_xsize = 1.;
  id->pixel_ysize = 1.;
  id->slice_width = 1.;

  id->slice_spacing = 0.; /* = slice_width; when not found no gap */

  id->pat_slice_orient = MDC_UNKNOWN; 
  strcpy(id->pat_pos,"Unknown");
  strcpy(id->pat_orient,"Unknown");
  strcpy(id->image_mod,"NM");

  for(i=0;i<3;i++) { id->image_pos_dev[i]=0.;    id->image_pos_pat[i]=0.;}
  for(i=0;i<6;i++) { id->image_orient_dev[i]=0.; id->image_orient_pat[i]=0.;}

}

void MdcResetIDs(FILEINFO *fi)
{
  Uint32 i;

  for (i=0; i<fi->number; i++)  {
     fi->image[i].rescaled = MDC_NO;
     fi->image[i].rescaled_max = 0.;
     fi->image[i].rescaled_min = 0.;
     fi->image[i].rescaled_fctr= 1.;
  } 

}

/* returns 1 on success and 0 on failure        */
int MdcGetStructAD(FILEINFO *fi)
{
  Uint32 i;

  if ((fi->acqnr > 0) && (fi->acqdata == NULL)) {
    fi->acqdata = (ACQ_DATA *)malloc(sizeof(ACQ_DATA)*fi->acqnr);
    if (fi->acqdata == NULL) return MDC_NO;

    for (i=0; i<fi->acqnr; i++) MdcInitAD(&fi->acqdata[i]);

  } 

  if (fi->acqdata != NULL) return(MDC_YES); /* (already) allocated */

  return(MDC_NO);

}

void MdcInitAD(ACQ_DATA *acq)
{
  if (acq != NULL) {
     acq->rotation_direction = MDC_UNKNOWN;
     acq->detector_motion = MDC_UNKNOWN;
     acq->angle_start = 0.;
     acq->angle_step = 0.; 
  }
}

void MdcInitFI(FILEINFO *fi, char *path)
{  
   fi->ifp = NULL; fi->ofp = NULL;
   fi->idir = NULL; fi->ifname = NULL;
   fi->odir = NULL; fi->ofname = NULL;
   fi->image = NULL;
   fi->format = MDC_FRMT_BAD;
   fi->diff_type = fi->diff_size = MDC_NO;
   fi->rawconv = MDC_NO;
   fi->endian = MDC_UNKNOWN;
   fi->compression = MDC_NO;
   fi->truncated=MDC_NO;
   fi->number = 0;
   fi->mwidth=fi->mheight=0;
   fi->bits = 8; fi->type = BIT8_U;
   fi->ifname = fi->ipath;
   memset(fi->ipath,'\0',MDC_MAX_PATH);
   strncpy(fi->ipath,path,MDC_MAX_PATH);
   fi->ofname = fi->opath;
   memset(fi->opath,'\0',MDC_MAX_PATH);
   fi->study_date_day   = 0;
   fi->study_date_month = 0;
   fi->study_date_year  = 0;
   fi->study_time_hour  = 0;
   fi->study_time_minute= 0;
   fi->study_time_second= 0;

   fi->decay_corrected   = MDC_NO;
   fi->flood_corrected   = MDC_NO;
   fi->acquisition_type  = MDC_ACQUISITION_UNKNOWN;
   fi->reconstructed     = MDC_NO;

   strcpy(fi->recon_method,"Unknown");
   strcpy(fi->patient_name,"Unknown");
   strcpy(fi->patient_id,"Unknown");
   strcpy(fi->patient_sex,"Unknown");
   strcpy(fi->study_descr,"Unknown");
   strcpy(fi->study_name,"Unknown");
   strcpy(fi->radiopharma,"Unknown");
   strcpy(fi->filter_type,"Unknown");
   strcpy(fi->isotope_code,"Unknown");
  
   fi->isotope_halflife = 0.;
   fi->gantry_tilt      = 0.;

   fi->dim[0] = 3;
   fi->dim[1] = 1;
   fi->dim[2] = 1;
   fi->dim[3] = 1;
   fi->dim[4] = 1;
   fi->dim[5] = 1;
   fi->dim[6] = 1;
   fi->dim[7] = 1;
   fi->pixdim[0] = 3.;
   fi->pixdim[1] = 1.;
   fi->pixdim[2] = 1.;
   fi->pixdim[3] = 1.;
   fi->pixdim[4] = 1.;
   fi->pixdim[5] = 1.;
   fi->pixdim[6] = 1.;
   fi->pixdim[7] = 1.;


   fi->map = MDC_MAP_GRAY;
   MdcGetColorMap(fi->map,fi->palette);
   fi->comment = NULL;
   fi->comm_length = 0;
   fi->glmin = fi->glmax = fi->qglmin = fi->qglmax = 0.;

   fi->acqnr = 0; fi->acqdata = NULL;

}

void MdcFreeIDs(FILEINFO *fi)
{
  Uint32 i;
 
  if ( fi->image != NULL ) {
    for ( i=0; i<fi->number; i++) {
       if (fi->image[i].buf != NULL) {
         MdcFree(fi->image[i].buf);
         fi->image[i].buf = NULL;
       }
    }
    MdcFree(fi->image);
  }
  fi->image = NULL;
} 

void MdcCleanUpFI(FILEINFO  *fi)
{
  if (fi->acqdata != NULL) MdcFree(fi->acqdata);
  if (fi->comment != NULL) MdcFree(fi->comment);
  MdcFreeIDs(fi);

  MdcFileClose(fi->ifp);
  MdcFileClose(fi->ofp);
}

char *MdcGetFname(char path[])
{
  char *p;

  p = MdcGetLastPathSlash(path);

  if ( p == NULL ) return(path);

  return(p+1); 
}

char *MdcGetExtension(char path[])
{
  char *p=NULL, *ext=NULL;

  if (path != NULL) {
    p=(char *)strrchr(path,'.');

    if (p != NULL) {
      ext=malloc(strlen(p)+1);
      p+=1; /* without the dot */
      if (ext != NULL) strcpy(ext,p);
    }
  }

  return(ext);

} 
    
    
void MdcNewExt(char dest[], char *src, char *ext)
{
  char *p;

  if ((src != NULL) && (src[0] != '\0'))  strcat(dest,src);
  /* MARK: 8.3 DOS filename problem for GtkWin32 XMedCon        */ 
  /* MARK: because full path is supplied -> removed restriction */

  p=(char *)strrchr(dest,'.');

  if ( p != NULL )  *p = '\0';

  strcat(dest,".");
  strcat(dest,ext);

} 

/* create a prefix of the form: 
00 ...  ... 99,A0...A9,AA...AZ,B0...B9,BA...BZ...ZZ
 <- normal -> | <------------- extra ------------>

normal prefixes =
      integer value from 00 to 99         (total: 100)
extra  prefixes = 
      first char: A...Z
     second char: 0...9,A...Z             (total: 936)

With this construction we keep the previous prefixes, 
but now enables us to create more than 1000 filenames
without overlapping. Even a sort of an `ls -l` gives
the filenames in sequence as they were created. */
        
void MdcPrefix(int n)
{
  int t, c1, c2, v1, v2;
  int A='A', Zero='0';     /* ascii values of A and Zero */

  if (n < 100) {
    sprintf(prefix,"m%02d-",n);
  }else{
    t = n - 100;
    v1 = t / 36;
    v2 = t % 36;

    if (v1 >= 26) {  
#ifdef XSUPPORT
      if (XMDC_MEDCON) {
        XMdcDisplayErr("%d-th conversion creates overlapping filenames", n); 
      }else 
#endif
       MdcPrntErr(MDC_OVER_FLOW,
                 "%d-th conversion creates overlapping filenames", n);
    }

    /* first  char */
    c1 = A + v1;                   /* A...Z */
    /* second char */
    if (v2 < 10) c2 = Zero + v2;   /* 0...9 */
    else c2 = A + v2 - 10;         /* A...Z */

    sprintf(prefix,"m%c%c-",(char)c1,(char)c2);
  }
}

void MdcNewName(char dest[], char *src, char *ext)
{
  strcpy(dest,prefix);
  MdcNewExt( dest, src, ext);
}

void MdcDefaultName(Int8 format, char dest[], char *src)
{
  switch (format) {
   case MDC_FRMT_RAW  : MdcNewName(dest,src,FrmtExt[MDC_FRMT_RAW]);    break;
   case MDC_FRMT_ASCII: MdcNewName(dest,src,FrmtExt[MDC_FRMT_ASCII]);  break;
#if MDC_INCLUDE_ACR
   case MDC_FRMT_ACR  : MdcNewName(dest,src,FrmtExt[MDC_FRMT_ACR]);    break;
#endif
#if MDC_INCLUDE_GIF
   case MDC_FRMT_GIF  : MdcNewName(dest,src,FrmtExt[MDC_FRMT_GIF]);    break;
#endif
#if MDC_INCLUDE_INW
   case MDC_FRMT_INW  : MdcNewName(dest,src,FrmtExt[MDC_FRMT_INW]);    break;
#endif
#if MDC_INCLUDE_ECAT
   case MDC_FRMT_ECAT : MdcNewName(dest,src,FrmtExt[MDC_FRMT_ECAT]);   break;
#endif
#if MDC_INCLUDE_INTF
   case MDC_FRMT_INTF : MdcNewName(dest,src,FrmtExt[MDC_FRMT_INTF]);   break;
#endif
#if MDC_INCLUDE_ANLZ
   case MDC_FRMT_ANLZ : MdcNewName(dest,src,FrmtExt[MDC_FRMT_ANLZ]);   break;
#endif
#if MDC_INCLUDE_DICM
   case MDC_FRMT_DICM : MdcNewName(dest,src,FrmtExt[MDC_FRMT_DICM]);   break;
#endif
   default            : MdcNewName(dest,src,FrmtExt[MDC_FRMT_BAD]);    break; 
  } 
}



void MdcRenameFile(char *name)
{
  char *pbegin = NULL, *pend = NULL;

  MdcPrintLine('-',MDC_FULL_LENGTH);
  printf("\tRENAME FILE\n");
  MdcPrintLine('-',MDC_FULL_LENGTH);

  pbegin = MdcGetLastPathSlash(name);    /* get the basename  */

  if (pbegin == NULL) pbegin = name;
  else pbegin = pbegin + 1;

  strcpy(mdcbufr,pbegin);
  pend   = (char *)strrchr(mdcbufr,'.'); /* without extension */
  if (pend != NULL) pend[0] = '\0';

  printf("\n\tOld Filename: %s\n",mdcbufr);
  printf("\n\tNew Filename: ");
  fgets(mdcbufr,MDC_MAX_PATH-1,stdin);
  mdcbufr[MDC_MAX_PATH]='\0'; 
  MdcRemoveEnter(mdcbufr);
  strcpy(name,mdcbufr);

  MdcPrintLine('-',MDC_FULL_LENGTH);

}

char *MdcGetLastPathSlash(char *path)
{
  char *p=NULL;

  if (path == NULL) return NULL;

#ifdef MDC_DOS
  p = (char *)strrchr(path,'\\');
  if (p != NULL) return(p);
#endif
  p = (char *)strrchr(path,'/');
  return(p);

}


void MdcMySplitPath(char path[], char **dir, char **fname) 
{
  char *p=NULL;

  p = MdcGetLastPathSlash(path);       /* last slash becomes '\0'      */

  if ( p == NULL ) {
    *fname=&path[0];
    *dir=NULL;
  } else {
    *p='\0';
    *dir=&path[0];
    *fname=p+1;
  }

}

void MdcMyMergePath(char path[], char *dir, char **fname) 
                                      /* first '\0' becomes slash again */
{
  char *p;

  if ( dir != NULL ) {
    p=(char *)strchr(path,'\0');
    if ( p != NULL )
#ifdef MDC_DOS
      *p='\\';
#else
      *p='/';
#endif
  }

  *fname = &path[0];

}

void MdcFillImgPos(FILEINFO *fi, Uint32 nr, Uint32 plane, float translation)
{
  IMG_DATA *id = &fi->image[nr];

  /* according to device coordinates */

  switch (id->pat_slice_orient) {

   case MDC_SUPINE_HEADFIRST_TRANSVERSAL: 
   case MDC_PRONE_HEADFIRST_TRANSVERSAL :
   case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
   case MDC_PRONE_FEETFIRST_TRANSVERSAL :
    id->image_pos_dev[0]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_dev[1]=-(id->pixel_ysize*(float)id->height);
    id->image_pos_dev[2]=-((id->slice_spacing*(float)(plane+1))+translation);
    break;
   case MDC_SUPINE_HEADFIRST_SAGITTAL   :
   case MDC_PRONE_HEADFIRST_SAGITTAL    :
   case MDC_SUPINE_FEETFIRST_SAGITTAL   :
   case MDC_PRONE_FEETFIRST_SAGITTAL    :
    id->image_pos_dev[0]=-((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_dev[1]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_dev[2]=-(id->pixel_ysize*(float)id->height);
    break;
   case MDC_SUPINE_HEADFIRST_CORONAL    :
   case MDC_PRONE_HEADFIRST_CORONAL     :
   case MDC_SUPINE_FEETFIRST_CORONAL    :
   case MDC_PRONE_FEETFIRST_CORONAL     :
    id->image_pos_dev[0]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_dev[1]=-((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_dev[2]=-(id->pixel_ysize*(float)id->height);
    break;
   default                              : { } /* do nothing */

  }

  /* according to the patient coordinate system */

  switch (id->pat_slice_orient) {
   case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
    id->image_pos_pat[0]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=-(id->pixel_ysize*(float)id->height);
    id->image_pos_pat[2]=-((id->slice_spacing*(float)(plane+1))+translation);
    break;
   case MDC_SUPINE_HEADFIRST_SAGITTAL   :
    id->image_pos_pat[0]=-((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[1]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[2]=-(id->pixel_ysize*(float)id->height);
    break;
   case MDC_SUPINE_HEADFIRST_CORONAL    :
    id->image_pos_pat[0]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=-((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[2]=-(id->pixel_ysize*(float)id->height);
    break;
   case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
    id->image_pos_pat[0]=+(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=-(id->pixel_ysize*(float)id->height);
    id->image_pos_pat[2]=+((id->slice_spacing*(float)(plane+1))+translation);
    break;
   case MDC_SUPINE_FEETFIRST_SAGITTAL   :
    id->image_pos_pat[0]=+((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[1]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[2]=+(id->pixel_ysize*(float)id->height);
    break;
   case MDC_SUPINE_FEETFIRST_CORONAL    :
    id->image_pos_pat[0]=+(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=-((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[2]=+(id->pixel_ysize*(float)id->height);
    break;
   case MDC_PRONE_HEADFIRST_TRANSVERSAL :
    id->image_pos_pat[0]=+(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=+(id->pixel_ysize*(float)id->height);
    id->image_pos_pat[2]=-((id->slice_spacing*(float)(plane+1))+translation);
    break;
   case MDC_PRONE_HEADFIRST_SAGITTAL    :
    id->image_pos_pat[0]=+((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[1]=+(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[2]=-(id->pixel_ysize*(float)id->height);
    break;
   case MDC_PRONE_HEADFIRST_CORONAL     :
    id->image_pos_pat[0]=+(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=+((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[2]=-(id->pixel_ysize*(float)id->height);
    break;
   case MDC_PRONE_FEETFIRST_TRANSVERSAL :
    id->image_pos_pat[0]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=+(id->pixel_ysize*(float)id->height);
    id->image_pos_pat[2]=+((id->slice_spacing*(float)(plane+1))+translation);
    break;
   case MDC_PRONE_FEETFIRST_SAGITTAL    : 
    id->image_pos_pat[0]=-((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[1]=+(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[2]=+(id->pixel_ysize*(float)id->height);
    break;
   case MDC_PRONE_FEETFIRST_CORONAL     :
    id->image_pos_pat[0]=-(id->pixel_xsize*(float)id->width);
    id->image_pos_pat[1]=+((id->slice_spacing*(float)(plane+1))+translation);
    id->image_pos_pat[2]=+(id->pixel_ysize*(float)id->height);
    break;
   default                              : { } /* do nothing */
  }
}

void MdcFillImgOrient(FILEINFO *fi, Uint32 nr)
{
  IMG_DATA *id = &fi->image[nr];

  /* according to device coordinate system */

  switch (id->pat_slice_orient) {
     case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
     case MDC_PRONE_HEADFIRST_TRANSVERSAL :
     case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
     case MDC_PRONE_FEETFIRST_TRANSVERSAL :
         id->image_orient_dev[0]=+1.0;    id->image_orient_dev[3]=+0.0;
         id->image_orient_dev[1]=-0.0;    id->image_orient_dev[4]=+1.0;
         id->image_orient_dev[2]=+0.0;    id->image_orient_dev[5]=-0.0;
         break; 

     case MDC_SUPINE_HEADFIRST_SAGITTAL   :
     case MDC_PRONE_HEADFIRST_SAGITTAL    :
     case MDC_SUPINE_FEETFIRST_SAGITTAL   :
     case MDC_PRONE_FEETFIRST_SAGITTAL    :
         id->image_orient_dev[0]=+0.0;    id->image_orient_dev[3]=+0.0; 
         id->image_orient_dev[1]=+1.0;    id->image_orient_dev[4]=-0.0;
         id->image_orient_dev[2]=-0.0;    id->image_orient_dev[5]=-1.0;
         break;

     case MDC_SUPINE_HEADFIRST_CORONAL    :
     case MDC_PRONE_HEADFIRST_CORONAL     :
     case MDC_SUPINE_FEETFIRST_CORONAL    :
     case MDC_PRONE_FEETFIRST_CORONAL     :
         id->image_orient_dev[0]=+1.0;    id->image_orient_dev[3]=+0.0; 
         id->image_orient_dev[1]=-0.0;    id->image_orient_dev[4]=-0.0;
         id->image_orient_dev[2]=+0.0;    id->image_orient_dev[5]=-1.0;
         break;

     default                              : { } 
  } 

  /* according to patient coordinate system */

  switch (id->pat_slice_orient) {

   case MDC_SUPINE_HEADFIRST_TRANSVERSAL:
         id->image_orient_pat[0]=+1.0;    id->image_orient_pat[3]=+0.0;
         id->image_orient_pat[1]=-0.0;    id->image_orient_pat[4]=+1.0;
         id->image_orient_pat[2]=+0.0;    id->image_orient_pat[5]=-0.0;
         break;
   case MDC_SUPINE_HEADFIRST_SAGITTAL   :
         id->image_orient_pat[0]=+0.0;    id->image_orient_pat[3]=+0.0;
         id->image_orient_pat[1]=+1.0;    id->image_orient_pat[4]=-0.0;
         id->image_orient_pat[2]=-0.0;    id->image_orient_pat[5]=-1.0;
         break;
   case MDC_SUPINE_HEADFIRST_CORONAL    :
         id->image_orient_pat[0]=+1.0;    id->image_orient_pat[3]=+0.0;
         id->image_orient_pat[1]=-0.0;    id->image_orient_pat[4]=-0.0;
         id->image_orient_pat[2]=+0.0;    id->image_orient_pat[5]=-1.0;
         break;
   case MDC_SUPINE_FEETFIRST_TRANSVERSAL:
         id->image_orient_pat[0]=-1.0;    id->image_orient_pat[3]=+0.0;
         id->image_orient_pat[1]=+0.0;    id->image_orient_pat[4]=+1.0;
         id->image_orient_pat[2]=-0.0;    id->image_orient_pat[5]=-0.0;
         break;
   case MDC_SUPINE_FEETFIRST_SAGITTAL   :
         id->image_orient_pat[0]=+0.0;    id->image_orient_pat[3]=-0.0;
         id->image_orient_pat[1]=+1.0;    id->image_orient_pat[4]=+0.0;
         id->image_orient_pat[2]=-0.0;    id->image_orient_pat[5]=+1.0;
         break;
   case MDC_SUPINE_FEETFIRST_CORONAL    :
         id->image_orient_pat[0]=-1.0;    id->image_orient_pat[3]=-0.0;
         id->image_orient_pat[1]=+0.0;    id->image_orient_pat[4]=+0.0;
         id->image_orient_pat[2]=-0.0;    id->image_orient_pat[5]=+1.0;
         break;
   case MDC_PRONE_HEADFIRST_TRANSVERSAL :
         id->image_orient_pat[0]=-1.0;    id->image_orient_pat[3]=-0.0;
         id->image_orient_pat[1]=+0.0;    id->image_orient_pat[4]=-1.0;
         id->image_orient_pat[2]=-0.0;    id->image_orient_pat[5]=+0.0;
         break;
   case MDC_PRONE_HEADFIRST_SAGITTAL    :
         id->image_orient_pat[0]=-0.0;    id->image_orient_pat[3]=+0.0;
         id->image_orient_pat[1]=-1.0;    id->image_orient_pat[4]=-0.0;
         id->image_orient_pat[2]=+0.0;    id->image_orient_pat[5]=-1.0;
         break;
   case MDC_PRONE_HEADFIRST_CORONAL     :
         id->image_orient_pat[0]=-1.0;    id->image_orient_pat[3]=+0.0;
         id->image_orient_pat[1]=+0.0;    id->image_orient_pat[4]=-0.0;
         id->image_orient_pat[2]=-0.0;    id->image_orient_pat[5]=-1.0;
         break;
   case MDC_PRONE_FEETFIRST_TRANSVERSAL :
         id->image_orient_pat[0]=+1.0;    id->image_orient_pat[3]=-0.0;
         id->image_orient_pat[1]=-0.0;    id->image_orient_pat[4]=-1.0;
         id->image_orient_pat[2]=+0.0;    id->image_orient_pat[5]=+0.0;
         break;
   case MDC_PRONE_FEETFIRST_SAGITTAL    :
         id->image_orient_pat[0]=-0.0;    id->image_orient_pat[3]=-0.0;
         id->image_orient_pat[1]=-1.0;    id->image_orient_pat[4]=+0.0;
         id->image_orient_pat[2]=+0.0;    id->image_orient_pat[5]=+1.0;
         break;
   case MDC_PRONE_FEETFIRST_CORONAL     :
         id->image_orient_pat[0]=+1.0;    id->image_orient_pat[3]=-0.0;
         id->image_orient_pat[1]=-0.0;    id->image_orient_pat[4]=+0.0;
         id->image_orient_pat[2]=+0.0;    id->image_orient_pat[5]=+1.0;
         break;
   default                              : { } /* do nothing */
  }
}


Int8 MdcGetPatSliceOrient(FILEINFO *fi, Uint32 i)
{
  IMG_DATA *id = &fi->image[i];
  float f0,f1,f4,f5;

  f0 = id->image_orient_pat[0];
  f1 = id->image_orient_pat[1];
  f4 = id->image_orient_pat[4];
  f5 = id->image_orient_pat[5];

  if ((f0 == +1.0) && (f4 == +1.0)) return MDC_SUPINE_HEADFIRST_TRANSVERSAL;
  if ((f0 == -1.0) && (f4 == +1.0)) return MDC_SUPINE_FEETFIRST_TRANSVERSAL;
  if ((f0 == -1.0) && (f4 == -1.0)) return MDC_PRONE_HEADFIRST_TRANSVERSAL;
  if ((f0 == +1.0) && (f4 == -1.0)) return MDC_PRONE_FEETFIRST_TRANSVERSAL;
  
  if ((f1 == +1.0) && (f5 == -1.0)) return MDC_SUPINE_HEADFIRST_SAGITTAL;
  if ((f1 == +1.0) && (f5 == +1.0)) return MDC_SUPINE_FEETFIRST_SAGITTAL;
  if ((f1 == -1.0) && (f5 == -1.0)) return MDC_PRONE_HEADFIRST_SAGITTAL;
  if ((f1 == -1.0) && (f5 == +1.0)) return MDC_PRONE_FEETFIRST_SAGITTAL;

  if ((f0 == +1.0) && (f5 == -1.0)) return MDC_SUPINE_HEADFIRST_CORONAL;
  if ((f0 == -1.0) && (f5 == +1.0)) return MDC_SUPINE_FEETFIRST_CORONAL;
  if ((f0 == -1.0) && (f5 == -1.0)) return MDC_PRONE_HEADFIRST_CORONAL;
  if ((f0 == +1.0) && (f5 == +1.0)) return MDC_PRONE_FEETFIRST_CORONAL;

  return(MDC_UNKNOWN);
}

Int8 MdcTryPatSliceOrient(char *pat_orient)
{
  char buffer[MDC_MAXSTR], *p1, *p2;
  Int8 orient1=MDC_UNKNOWN, orient2=MDC_UNKNOWN;

  MdcStringCopy(buffer,pat_orient,MDC_MAXSTR);

  p1 = buffer;
  p2 = strrchr(buffer, '\\');

  if (p2 == NULL) return MDC_UNKNOWN;

  p2[0] = '\0'; p2+=1;

  if      (strchr(p1,'L') != NULL) orient1 = MDC_LEFT;
  else if (strchr(p1,'R') != NULL) orient1 = MDC_RIGHT;
  else if (strchr(p1,'A') != NULL) orient1 = MDC_ANTERIOR;
  else if (strchr(p1,'P') != NULL) orient1 = MDC_POSTERIOR;
  else if (strchr(p1,'H') != NULL) orient1 = MDC_HEAD;
  else if (strchr(p1,'F') != NULL) orient1 = MDC_FEET;

  if      (strchr(p2,'L') != NULL) orient2 = MDC_LEFT;
  else if (strchr(p2,'R') != NULL) orient2 = MDC_RIGHT;
  else if (strchr(p2,'A') != NULL) orient2 = MDC_ANTERIOR;
  else if (strchr(p2,'P') != NULL) orient2 = MDC_POSTERIOR;
  else if (strchr(p2,'H') != NULL) orient2 = MDC_HEAD;
  else if (strchr(p2,'F') != NULL) orient2 = MDC_FEET;


  if (orient1 == MDC_LEFT      && orient2 == MDC_POSTERIOR)
    return MDC_SUPINE_HEADFIRST_TRANSVERSAL;
  if (orient1 == MDC_POSTERIOR && orient2 == MDC_FEET)
    return MDC_SUPINE_HEADFIRST_SAGITTAL;
  if (orient1 == MDC_LEFT      && orient2 == MDC_FEET)
    return MDC_SUPINE_HEADFIRST_CORONAL;

  if (orient1 == MDC_RIGHT     && orient2 == MDC_POSTERIOR)
    return MDC_SUPINE_FEETFIRST_TRANSVERSAL;
  if (orient1 == MDC_POSTERIOR && orient2 == MDC_HEAD)
    return MDC_SUPINE_FEETFIRST_SAGITTAL;
  if (orient1 == MDC_RIGHT     && orient2 == MDC_HEAD)
    return MDC_SUPINE_FEETFIRST_CORONAL;

  if (orient1 == MDC_RIGHT     && orient2 == MDC_ANTERIOR)
    return MDC_PRONE_HEADFIRST_TRANSVERSAL;
  if (orient1 == MDC_ANTERIOR  && orient2 == MDC_FEET)
    return MDC_PRONE_HEADFIRST_SAGITTAL;
  if (orient1 == MDC_RIGHT     && orient2 == MDC_FEET)
    return MDC_PRONE_HEADFIRST_CORONAL;

  if (orient1 == MDC_LEFT      && orient2 == MDC_ANTERIOR)
    return MDC_PRONE_FEETFIRST_TRANSVERSAL;
  if (orient1 == MDC_ANTERIOR  && orient2 == MDC_HEAD)
    return MDC_PRONE_FEETFIRST_SAGITTAL;
  if (orient1 == MDC_LEFT      && orient2 == MDC_HEAD)
    return MDC_PRONE_FEETFIRST_CORONAL;

  return MDC_UNKNOWN;
}

