/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-ecat64.c                                                    *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : Read and Write ECAT 6.4 files                            *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcCheckECAT()          - Check for ECAT format          * 
 *                MdcReadECAT()           - Read ECAT 6.4 file             *
 *                MdcWriteECAT()          - Write ECAT 6.4 file            *
 *                MdcGetFilterCode()      - Get code number of filter      *
 *                MdcFillMainHeader()     - Fill in Main Header            *
 *                MdcFillImageSubHeader() - Fill in Image SubHeader        *
 *                MdcPrintEcatInfoDB()    - Print ECAT database info       *
 *                                                                         *
 *                                                                         *
 * Notes        : source needs m-matrix.h & m-matrix.c                     *
 *                                                                         *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-ecat64.c,v 1.1.1.1 2000/10/28 16:51:26 enlf Exp $
 */

/*
   Copyright (C) 1997-2000 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "medcon.h"

/****************************************************************************
                        G L O B A L   D E F I N E S
****************************************************************************/

#define MDC_ECAT_MAX_PLANES  1024
#define MDC_ECAT_MAX_FRAMES   512
#define MDC_ECAT_MAX_GATES     64
#define MDC_ECAT_MAX_BEDS      16

#define MdcMatrixBlocks(x)   (((x+(MatBLKSIZE-1))/MatBLKSIZE)*MatBLKSIZE);

static Uint32 saved_mwidth;
static Uint32 saved_mheight;

char MdcEcatDataTypes
        [MDC_MAX_ECATDATATYPES][MDC_MAX_ECATDATATYPES_SIZE]=
                      {"Unknown","ByteData","VAX Int16","VAX Int32",
                       "VAX float","IEEE float","SUN Int16","SUN Int32"};
char MdcEcatFileTypes
        [MDC_MAX_ECATFILETYPES][MDC_MAX_ECATFILETYPES_SIZE]=
                      {"Unknown","Sinogram","PetImage","Attenuation",
                       "Normalization","Smooth File"};
char MdcEcatAcquisitionTypes
        [MDC_MAX_ECATACQTYPES][MDC_MAX_ECATACQTYPES_SIZE]=
                      {"Undefined","Blank","Transmission",
                       "Static Emission","Dynamic Emission","Gated Emission",
                       "Transmission Rectilinear","Emission Rectilinear",
                       "Whole Body Transmission","Whole Body Static"};

char MdcEcatFilterTypes
        [MDC_MAX_ECATFLTRTYPES][MDC_MAX_ECATFLTRTYPES_SIZE]=
                      {"None","Ramp","Butter","Hann",
                       "Hamm","Parzen","Shepp","Unknown"};

char MdcEcatQuantificationUnits
        [MDC_MAX_ECATQUANTTYPES][MDC_MAX_ECATQUANTTYPES_SIZE]=
                      {"Total Counts","Undefined",
                       "ECAT counts/second/pixel","uCi/ml [1uCi = 37Bq]",
                       "LMRGlu","LMRGlu umol/min/100g","LMRGlu mg/min/100g",
                       "nCi/ml","Well counts","Becquerels","ml/min/100g",
                       "ml/min/g"};

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/
int MdcCheckECAT(FILEINFO *fi)
{ 
  Main_header mh;

  if (mat_read_main_header(fi->ifp,&mh)) return MDC_BAD_READ;

  if (mh.system_type == MDC_ECAT_SYST_TYPE) return MDC_FRMT_ECAT;

  return MDC_FRMT_BAD;

}


char *MdcReadECAT(FILEINFO *fi)
{
  FILE *fp = fi->ifp;
  IMG_DATA *id=NULL;
  int i, error;
  char *err, *str;
  Uint32 bytes, img=0;
  Main_header mh;
  Image_subheader ish;
  Scan_subheader ssh;
  Attn_subheader ash;
  Norm_subheader nsh;
  struct MatDir entry;
  Int16 data_type=BIT16_S;
  int bed,gate,frame,plane, data=0, matnum, startblk, endblk;

  fi->endian=MDC_FILE_ENDIAN=MDC_LITTLE_ENDIAN; /* default for ECAT files */

#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Reading ECAT:");
#endif 

  if (MDC_VERBOSE) MdcPrntMesg("ECAT Reading <%s> ...",fi->ifname);

  /* read the main header */
  error = mat_read_main_header(fp, &mh);
  if (error) return("ECAT Bad read main header");

  if (MDC_INFO_DB) {
    MdcPrintEcatInfoDB(&mh);
    return NULL;
  }

  if (MDC_INFO) { 

    printf("Main Header (%d bytes)\n",MH_64_SIZE);
    MdcPrintLine('-',MDC_HALF_LENGTH);
 
    printf("Original Filename        : "); 
       MdcPrintStr(mh.original_file_name);
    printf("Software Version         : %d\n",mh.sw_version);
    printf("Data Type                : %d ",mh.data_type);
    if ((mh.data_type > -1) && (mh.data_type < 8))
      printf("(= %s)\n",MdcEcatDataTypes[mh.data_type]);
    else
      printf("(= Unknown)\n");
    printf("System Type              : %d\n",mh.system_type);
    printf("File Type                : %d ",mh.file_type);
    if ((mh.file_type > -1) && (mh.file_type < 5)) 
      printf("(= %s)\n",MdcEcatFileTypes[mh.file_type]);
    else
      printf("(= Unknown)\n");
    printf("Node Id                  : "); MdcPrintStr(mh.node_id);
    printf("Scan Date - Day          : %d\n",mh.scan_start_day);
    printf("          - Month        : %d\n",mh.scan_start_month);
    printf("          - Year         : %d\n",mh.scan_start_year);
    printf("          - Hour         : %d\n",mh.scan_start_hour);
    printf("          - Minute       : %d\n",mh.scan_start_minute);
    printf("          - Second       : %d\n",mh.scan_start_second);
    printf("Isotope Code             : "); MdcPrintStr(mh.isotope_code);
    printf("Isotope Halflife         : %f [sec]\n",mh.isotope_halflife);
    printf("Radiopharmaceutical      : "); 
      MdcPrintStr(mh.radiopharmaceutical);
    printf("Gantry Tilt              : %f [degrees]\n",mh.gantry_tilt);
    printf("Gantry Rotation          : %f [degrees]\n",mh.gantry_rotation);
    printf("Bed Elevation            : %f [cm]\n",mh.bed_elevation);
    printf("Rotating Source Speed    : %d [revolutions/minute]\n"
                                       ,mh.rot_source_speed);
    printf("Wobble Control Speed     : %d [revolutions/minute]\n"
                                       ,mh.wobble_speed);
    printf("Transmission Source      : %d\n",mh.transm_source_type);
    printf("Axial Field of View      : %f [cm]\n",mh.axial_fov);
    printf("Transaxial Field of View : %f [cm]\n",mh.transaxial_fov);
    printf("Transaxial Sampling Mode : %d\n",mh.transaxial_samp_mode);
    printf("Coincidence Sampling Mode: %d\n",mh.coin_samp_mode);
    printf("Axial Sampling Mode      : %d\n",mh.axial_samp_mode);
    printf("Calibration Factor       : %f\n",mh.calibration_factor);
    printf("Calibration Units        : %d ",mh.calibration_units);
    if ((mh.calibration_units > -1) && (mh.calibration_units < 12))
      printf("(= %s)\n",MdcEcatQuantificationUnits[mh.calibration_units]);
    else 
      printf("(= Unknown)\n");
    printf("Compression Code         : %d\n",mh.compression_code);
    printf("Study Name               : "); MdcPrintStr(mh.study_name);
    printf("Patient Id               : "); MdcPrintStr(mh.patient_id);
    printf("Patient Name             : "); MdcPrintStr(mh.patient_name);
    printf("Patient Sex              : "); MdcPrintChar(mh.patient_sex);  
    printf("\n");
    printf("Patient Age              : "); MdcPrintStr(mh.patient_age);
    printf("Patient Height           : "); MdcPrintStr(mh.patient_height);
    printf("Patient Weight           : "); MdcPrintStr(mh.patient_weight);
    printf("Patient Dexterity        : "); 
       MdcPrintChar(mh.patient_dexterity);
    printf("\n");
    printf("Physician Name           : "); MdcPrintStr(mh.physician_name);
    printf("Operator Name            : "); MdcPrintStr(mh.operator_name);
    printf("Study Description        : "); MdcPrintStr(mh.study_description);
    printf("Acquisition Type         : %d ",mh.acquisition_type);
    if ((mh.acquisition_type > -1) && (mh.acquisition_type <= 9))
      printf("(= %s)\n",MdcEcatAcquisitionTypes[mh.acquisition_type]);
    else
      printf("(= Unknown)\n");
    printf("Bed Type                 : %d\n",mh.bed_type);
    printf("Septa Type               : %d\n",mh.septa_type);
    printf("Facility Name            : "); MdcPrintStr(mh.facility_name);
    printf("Number of Planes         : %d\n",mh.num_planes);
    printf("Number of Frames         : %d\n",mh.num_frames);
    printf("Number of Gates          : %d\n",mh.num_gates);
    printf("Number of Bed Positions  : %d\n",mh.num_bed_pos);
    printf("Initial Bed Position     : %f [cm]\n",mh.init_bed_position);
    for (i=0; i<15; i++)
    printf("Bed Offset[%02d]           : %f [cm]\n",i+1,mh.bed_offset[i]);
    printf("Plane Separation         : %f [cm]\n",mh.plane_separation);
    printf("Lower Scatter Treshold   : %d [KeV]\n",mh.lwr_sctr_thres);
    printf("Lower True Treshold      : %d [KeV]\n",mh.lwr_true_thres);
    printf("Upper True Treshold      : %d [KeV]\n",mh.upr_true_thres);
    printf("Collimator               : %6.0f\n",mh.collimator);
    printf("User Process Code        : "); MdcPrintStr(mh.user_process_code);
    printf("Acquisition Mode         : %d\n",mh.acquisition_mode);

  }

  if ((mh.file_type!=MDC_ECAT_SCAN_FILE)  && 
      (mh.file_type!=MDC_ECAT_IMAGE_FILE) &&
      (mh.file_type!=MDC_ECAT_ATTN_FILE)  && 
      (mh.file_type!=MDC_ECAT_NORM_FILE) ) 
    return("ECAT Unsupported file type");

  if (mh.num_frames  <= 0 ) mh.num_frames  = 1;
  if (mh.num_gates   <= 0 ) mh.num_gates   = 1;
  if (mh.num_bed_pos <  0 ) mh.num_bed_pos = 0;

  /* fill in global FILEINFO data */
  fi->dim[0]= 6;
  fi->dim[3]= mh.num_planes;
  fi->dim[4]= mh.num_frames;
  fi->dim[5]= mh.num_gates;
  fi->dim[6]= mh.num_bed_pos + 1; /* must be 1-based */

  for (i=3, fi->number=1; i<=6; i++) fi->number*=fi->dim[i];

  if (fi->number == 0) return("ECAT No valid images specified");

  fi->patient_sex[0] = mh.patient_sex; fi->patient_sex[1]='\0';
  MdcStringCopy(fi->patient_name,mh.patient_name,32);
  MdcStringCopy(fi->patient_id,mh.patient_id,16);
  fi->study_date_day   = mh.scan_start_day;
  fi->study_date_month = mh.scan_start_month;
  fi->study_date_year  = mh.scan_start_year;
  fi->study_time_hour  = mh.scan_start_hour;
  fi->study_time_minute= mh.scan_start_minute;
  fi->study_time_second= mh.scan_start_second;

  if ((mh.file_type==MDC_ECAT_SCAN_FILE) || 
      (mh.file_type==MDC_ECAT_IMAGE_FILE)) {
    switch (mh.acquisition_type) {
      case MDC_ECAT_ACQTYPE_UNKNOWN : 
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN; break;
      case MDC_ECAT_ACQTYPE_BLANK : 
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN; break;
      case MDC_ECAT_ACQTYPE_TRANSMISSION : 
          fi->acquisition_type = MDC_ACQUISITION_TOMO;    break;
      case MDC_ECAT_ACQTYPE_STATIC_EMISSION : 
          fi->acquisition_type = MDC_ACQUISITION_TOMO;    break;
      case MDC_ECAT_ACQTYPE_DYNAMIC_EMISSION : 
          fi->acquisition_type = MDC_ACQUISITION_DYNAMIC; break;
      case MDC_ECAT_ACQTYPE_GATED_EMISSION : 
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN; break;
      case MDC_ECAT_ACQTYPE_TRANSMISSION_RECT : 
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN; break;
      case MDC_ECAT_ACQTYPE_EMISSION_RECT : 
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN; break;
      case MDC_ECAT_ACQTYPE_WHOLE_BODY_TRANSM :
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN; break;
      case MDC_ECAT_ACQTYPE_WHOLE_BODY_STATIC :
          fi->acquisition_type = MDC_ACQUISITION_TOMO;    break;
      default: 
          fi->acquisition_type = MDC_ACQUISITION_UNKNOWN;
    }
  }else{
    fi->acquisition_type = MDC_ACQUISITION_UNKNOWN;
  }
  MdcStringCopy(fi->study_descr,mh.study_description,32);
  MdcStringCopy(fi->study_name,mh.study_name,12);
  MdcStringCopy(fi->radiopharma,mh.radiopharmaceutical,32);
  MdcStringCopy(fi->isotope_code,mh.isotope_code,8);

  fi->isotope_halflife = mh.isotope_halflife;
  fi->gantry_tilt      = mh.gantry_tilt;


  if (!MdcGetStructID(fi))  return("ECAT Bad malloc IMG_DATA structs");


  for (bed=0; bed<fi->dim[6]; bed++)
  for (gate=1; gate<=fi->dim[5]; gate++)
  for (frame=1; frame<=fi->dim[4]; frame++)
  for (plane=0; plane<=fi->dim[3]; plane++) {
#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif
     matnum = mat_numcod(frame,plane,gate,data,bed);
     if (!mat_lookup(fp, matnum,&entry)) continue;
     startblk = entry.strtblk + 1;
     endblk = entry.endblk - entry.strtblk;

     switch (mh.file_type) {

      case MDC_ECAT_SCAN_FILE:

       error = mat_read_scan_subheader(fp, startblk-1, &ssh);
       if (error) return("ECAT Bad read scan subheader");
       if (MDC_INFO) {
        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("SINOGRAM SUBHEADER %05d:  ",img+1);
        printf("Frame: %d  Plane: %d  Gate: %d  Data: %d  Bed: %d\n"
               ,frame,plane,gate,data,bed);
        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("Data Type           : %d ",ssh.data_type);
        if ((ssh.data_type > -1) && (ssh.data_type < 8))
          printf("(= %s)\n",MdcEcatDataTypes[ssh.data_type]);
        else
          printf("(= Unknown)\n");
        printf("Number of Elements  : %d (width)\n",ssh.dimension_1);
        printf("Number of Views     : %d (height)\n",ssh.dimension_2);
        printf("Smoothing           : %d ",ssh.smoothing);
        switch (ssh.smoothing) {
          case 0: printf("(= Not Smoothed)\n"); break;
          case 1: printf("(= 9x9 Smoothing)\n"); break;
         default: printf("(= Unknown)\n"); 
        }
        printf("Processing Code     : %d\n",ssh.processing_code);
        printf("Sample Distance     : %f [cm]\n",ssh.sample_distance);
        printf("Isotope Halflife    : %f [sec]\n",ssh.isotope_halflife);
        printf("Frame Duration (sec): %d [sec]\n",ssh.frame_duration_sec);
        printf("Gate Duration       : %d [msec]\n",ssh.gate_duration);
        printf("R-Wave Offset       : %d [msec]\n",ssh.r_wave_offset);
        printf("Scale factor        : %f\n",ssh.scale_factor);
        printf("Minimum Scan Value  : %d\n",ssh.scan_min);
        printf("Maximum Scan Value  : %d\n",ssh.scan_max);
        printf("Total Prompts       : %d\n",ssh.prompts);
        printf("Total Delayed Events: %d\n",ssh.delayed);
        printf("Total Multiples     : %d\n",ssh.multiples);
        printf("Total Net Trues     : %d (Prompts - Random)\n"
                                      ,ssh.net_trues);
        for (i=0; i<16; i++) 
        printf("Corrected   Singles [%2d] : %f\n",i+1,ssh.cor_singles[i]);
        for (i=0; i<16; i++)
        printf("Uncorrected Singles [%2d] : %f\n",i+1,ssh.uncor_singles[i]);
        printf("Total Average Corrected   Singles: %f\n",ssh.tot_avg_cor);
        printf("Total Average Uncorrected Singles: %f\n",ssh.tot_avg_uncor);
        printf("Total Coincidence Rage  : %d (from IPCP)\n"
                                          ,ssh.total_coin_rate);
        printf("Frame Start Time        : %d [msec]\n",ssh.frame_start_time);
        printf("Frame Duration          : %d [msec]\n",ssh.frame_duration);
        printf("Loss Correction Factor  : %f\n",ssh.loss_correction_fctr);
        for (i=0; i<8; i++)
        printf("Phy_Planes [%d]          : %d\n",i+1,ssh.phy_planes[i]);
       }

       /* fill in IMG_DATA struct */
       id = &fi->image[img];
       id->width = (Uint32)ssh.dimension_1;
       id->height = (Uint32)ssh.dimension_2;
       id->pixel_xsize = id->pixel_ysize = ssh.sample_distance * 10.;/* mm */
       id->frame_start = (float) ssh.frame_start_time;
       id->frame_duration = (float) ssh.frame_duration;
       data_type = ssh.data_type; 
       switch( data_type ) {
         case  BYTE_TYPE:  id->bits = 8; id->type = BIT8_U; break;
         case  M68K_I2  :
     /*  case  SUN_I2   : */
         case  VAX_I2   :  id->bits =16; id->type = BIT16_S; break;
         case  M68K_I4  :
     /*  case  SUN_I4   : */
         case  VAX_I4   :  id->bits =32; id->type = BIT32_S; break;
         case  IEEE_R4  :
     /*  case  SUN_R4   : */
         case  VAX_R4   :  id->bits =32; id->type = FLT32; break;
       }

       break;

      case MDC_ECAT_IMAGE_FILE:

       error = mat_read_image_subheader(fp, startblk-1, &ish);
       if (error) return("ECAT Bad read image subheader");
       if (MDC_INFO) { 
        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("IMAGE SUBHEADER %05d:  ",img+1);
        printf("Frame: %d  Plane: %d  Gate: %d  Data: %d  Bed: %d\n"
               ,frame,plane,gate,data,bed);
        MdcPrintLine('-',MDC_FULL_LENGTH);

        printf("Data Type            : %d ",ish.data_type);
        if ((ish.data_type > -1) && (ish.data_type < 8))
          printf("(= %s)\n",MdcEcatDataTypes[ish.data_type]);
        else
          printf("(= Unknown)\n");
        printf("Number of Dimensions : %d\n",ish.num_dimensions);
        printf("X Dimension          : %d\n",ish.dimension_1);
        printf("Y Dimension          : %d\n",ish.dimension_2);
        printf("X Offset             : %f [cm]\n",ish.x_origin);
        printf("Y Offset             : %f [cm]\n",ish.y_origin);
        printf("Recon Magnification Factor  : %f\n",ish.recon_scale);
        printf("Quantification Scale Factor : %e\n",ish.quant_scale);
        printf("Image Minimum Pixel Value   : %d\n",ish.image_min);
        printf("Image Maximum Pixel Value   : %d\n",ish.image_max);
        printf("Pixel Size         : %f [cm]\n",ish.pixel_size);
        printf("Slice Width        : %f [cm]\n",ish.slice_width);
        printf("Frame Duration     : %d [ms]\n",ish.frame_duration);
        printf("Frame Start Time   : %d\n",ish.frame_start_time);
        printf("Slice Location     : %d [cm]\n",ish.slice_location);
        printf("Recon Start Hour   : %d\n",ish.recon_start_hour);
        printf("Recon Start Minute : %d\n",ish.recon_start_minute);
        printf("Recon Start Second : %d\n",ish.recon_start_sec);
        printf("Gate Duration      : %d [ms]\n",ish.gate_duration); 
        printf("Filter code        : %d ",ish.filter_code);
        ish.filter_code = abs(ish.filter_code); 
        if ((ish.filter_code > -1) && (ish.filter_code < 7))
          printf("(= %s)\n",MdcEcatFilterTypes[ish.filter_code]);
        else
          printf("(= Unknown)\n");
        printf("Scan Matrix Number : %d\n",ish.scan_matrix_num);
        printf("Normalization Matrix Number : %d\n",ish.norm_matrix_num);
        printf("Attenuation Matrix Number   : %d\n"
                                              ,ish.atten_cor_matrix_num);
        printf("Image Rotation              : %f [degrees]\n"
                                              ,ish.image_rotation);
        printf("Plane Efficiency Correction Factor: %f\n"
                                                  ,ish.plane_eff_corr_fctr);
        printf("Decay Correction Factor    : %f\n",ish.decay_corr_fctr);
        printf("Loss Correction Factor     : %f\n",ish.loss_corr_fctr);
        printf("Processing Code            : %d\n",ish.processing_code);
        printf("Quantification Units       : %d ",ish.quant_units);
        if ((ish.quant_units > -1) && (ish.quant_units < 13))
          printf("(= %s)\n",MdcEcatQuantificationUnits[ish.quant_units]);
        else
          printf("(= Unknown)\n");
        printf("Reconstruction Start Day   : %d\n",ish.recon_start_day);
        printf("Reconstruction Start Month : %d\n",ish.recon_start_month);
        printf("Reconstruction Start Year  : %d\n",ish.recon_start_year);
        printf("Ecat Calibration Factor    : %f\n"
                                             ,ish.ecat_calibration_fctr);
        printf("Well Counter Calibribration Factor : %f\n"
                                             ,ish.well_counter_cal_fctr);
        printf("Filter Params - Cutoff Frequency : %f\n"
                                             ,ish.filter_params[0]);
        printf("Filter Params - DC Component     : %f\n"
                                             ,ish.filter_params[1]);
        printf("Filter Params - Ramp Slope       : %f\n"
                                             ,ish.filter_params[2]);
        printf("Filter Params - (4)              : %f\n"
                                             ,ish.filter_params[3]);
        printf("Filter Params - Scatter Comp 1   : %f\n"
                                             ,ish.filter_params[4]);
        printf("Filter Params - Scatter Comp 2   : %f\n"
                                             ,ish.filter_params[5]);
        printf("Annotation : "); MdcPrintStr(ish.annotation);

       }

       /* fill in IMG_DATA struct */

       id = &fi->image[img];
       id->width = (Uint32)ish.dimension_1;
       id->height = (Uint32)ish.dimension_2;
       id->recon_scale = ish.recon_scale;

       id->quant_units = 1;
       id->quant_scale = 1;
       id->calibr_units= 1;
       id->calibr_fctr = 1;

       if ( MDC_CALIBRATE ) {
         id->quant_units = ish.quant_units;
         id->quant_scale = ish.quant_scale;
         id->calibr_units= mh.calibration_units;
         id->calibr_fctr = ish.ecat_calibration_fctr;
       }else if ( MDC_QUANTIFY ) {
         id->quant_units = ish.quant_units;
         id->quant_scale = ish.quant_scale;
         id->calibr_units= ish.quant_units;
         id->calibr_fctr = 1;
       }
       id->pixel_xsize = id->pixel_ysize = ish.pixel_size * 10.; /* in mm */
       id->slice_width = ish.slice_width * 10.; /* in mm */
       id->frame_start = (float) ish.frame_start_time;
       id->frame_duration = (float) ish.frame_duration;
       data_type = ish.data_type;
       switch( data_type ) {
         case  BYTE_TYPE:  id->bits = 8; id->type = BIT8_U;  break;
         case  M68K_I2  :
     /*  case  SUN_I2   : */
         case  VAX_I2   :  id->bits =16; id->type = BIT16_S; break;
         case  M68K_I4  :
     /*  case  SUN_I4   : */ 
         case  VAX_I4   :  id->bits =32; id->type = BIT32_S; break;
         case  IEEE_R4  :
     /*  case  SUN_R4   : */
         case  VAX_R4   :  id->bits =32; id->type = FLT32;   break;
       }

       /* some of the Acr/Nema tags */
       id->pat_slice_orient = MDC_SUPINE_HEADFIRST_TRANSVERSAL; /* default! */

       id->slice_spacing = mh.plane_separation*10.;  /* separation in mm */

       str = MdcGetStrPatientPos(id->pat_slice_orient);
       MdcStringCopy(id->pat_pos,str,strlen(str));

       if ( (strncmp(mh.user_process_code,"COR",10)==0) ||
            (strncmp(mh.user_process_code,"SAG",10)==0)) {
         /* CORONAL SLICES  or SAGITTAL SLICES 
            The images Ecat 6.4 software writes are useless:
            128x128 images with the small coronal/sagittal slice in it ...
            This means their pixel_xsize & pixel_ysize doesn't quite
            fit the real world dimensions any more !! 
            Therefore we don't even try to attempt writing the proper 
            Acr/Nema variables ... 
          */
       }else{ /* "TRA" or nothing */
         /* TRANSAXIAL SLICES (Transverse)             */
         /* Writing the proper Acr/Nema variables ...  */
         /* See man-page `m-acr.4' for more info (!)   */

         str = MdcGetStrPatientOrient(id->pat_slice_orient);
         MdcStringCopy(id->pat_orient,str,strlen(str));

         MdcFillImgPos(fi,img,plane-1,ish.slice_location*10.);
                  /* gamble: don't really know if this ish.slice_location
                     gives Z-dim translation for a next bed position */
         MdcFillImgOrient(fi,img);
         
       }

       break;

     case MDC_ECAT_ATTN_FILE:

       error = mat_read_attn_subheader(fp, startblk-1, &ash);
       if (error) return("ECAT Bad read attenuation subheader");
       if (MDC_INFO) {

        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("ATTENUATION SUBHEADER %05d:  ",img+1);
        printf("Frame: %d  Plane: %d  Gate: %d  Data: %d  Bed: %d\n"
               ,frame,plane,gate,data,bed);
        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("Data Type                     : %d ",ash.data_type);
        if ((ash.data_type > -1) && (ash.data_type < 8))
          printf("(= %s)\n",MdcEcatDataTypes[ash.data_type]);
        else
          printf("(= Unknown)\n");
        printf("Attenuation Correction Method : %d\n",ash.attenuation_type);
        printf("Number of Elements            : %d (width)\n"
                                                ,ash.dimension_1);
        printf("Number of Views               : %d (height)\n"
                                                ,ash.dimension_2);
        printf("Attenuation Scale Factor      : %f\n",ash.scale_factor);
        printf("Ellipse X Offset              : %f [cm]\n",ash.x_origin);
        printf("Ellipse Y Offset              : %f [cm]\n",ash.y_origin);
        printf("Ellipse X Radius              : %f [cm]\n",ash.x_radius);
        printf("Ellipse Y Radius              : %f [cm]\n",ash.y_radius);
        printf("Ellipse Tilt Angle            : %f [degrees]\n"
                                                ,ash.tilt_angle);
        printf("Mu-Absorption Coefficient     : %f [1/cm]\n"
                                                ,ash.attenuation_coeff);
        printf("Sample Distance               : %f [cm]\n" 
                                                ,ash.sample_distance);
       } 
        

       /* fill in IMG_DATA struct */
       id = &fi->image[img];
       id->width = (Uint32)ash.dimension_1;
       id->height = (Uint32)ash.dimension_2;
       id->calibr_fctr = ash.scale_factor;
       data_type = ash.data_type;
       switch( data_type ) {
         case  BYTE_TYPE:  id->bits = 8; id->type = BIT8_U;  break;
         case  M68K_I2  :
     /*  case  SUN_I2   : */
         case  VAX_I2   :  id->bits =16; id->type = BIT16_S; break;
         case  M68K_I4  :
     /*  case  SUN_I4   : */
         case  VAX_I4   :  id->bits =32; id->type = BIT32_S; break;
         case  IEEE_R4  :
     /*  case  SUN_R4   : */
         case  VAX_R4   :  id->bits =32; id->type = FLT32;   break;
       }

       break;

     case MDC_ECAT_NORM_FILE:

       error = mat_read_norm_subheader(fp, startblk-1, &nsh);
       if (error) return("ECAT Bad read normalization subheader");
       if (MDC_INFO) {
        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("NORMALIZATION SUBHEADER %05d:  ",img+1);
        printf("Frame: %d  Plane: %d  Gate: %d  Data: %d  Bed: %d\n"
               ,frame,plane,gate,data,bed);
        MdcPrintLine('-',MDC_FULL_LENGTH);
        printf("Data Type           : %d ",nsh.data_type);
        if ((nsh.data_type > -1) && (nsh.data_type < 8))
          printf("(= %s)\n",MdcEcatDataTypes[nsh.data_type]);
        else
          printf("(= Unknown)\n");
        printf("Number of Elements  : %d (width)\n",nsh.dimension_1);
        printf("Number of Views     : %d (height)\n",nsh.dimension_2);
        printf("Normalization Scale Factor  : %f\n",nsh.scale_factor);
        printf("Normalization Start Hour    : %d\n",nsh.norm_hour);
        printf("Normalization Start Minute  : %d\n",nsh.norm_minute);
        printf("Normalization Start Second  : %d\n",nsh.norm_second);
        printf("Normalization Start Day     : %d\n",nsh.norm_day);
        printf("Normalization Start Month   : %d\n",nsh.norm_month);
        printf("Normalization Start Year    : %d\n",nsh.norm_year);
        printf("Field of View Source Width  : %f [cm]\n",nsh.fov_source_width);
        printf("Ecat Calibration Factor     : %f\n",nsh.ecat_calib_factor);
       }

       /* fill in IMG_DATA struct */
       id = &fi->image[img];
       id->width = (Uint32)nsh.dimension_1;
       id->height = (Uint32)nsh.dimension_2;
       data_type = nsh.data_type;
       id->calibr_fctr = nsh.scale_factor;
       switch( data_type ) {
         case  BYTE_TYPE:  id->bits = 8; id->type = BIT8_U; break;
         case  M68K_I2  :
     /*  case  SUN_I2   : */
         case  VAX_I2   :  id->bits =16; id->type = BIT16_S; break;
         case  M68K_I4  :
     /*  case  SUN_I4   : */
         case  VAX_I4   :  id->bits =32; id->type = BIT32_S; break;
         case  IEEE_R4  :
     /*  case  SUN_R4   : */
         case  VAX_R4   :  id->bits =32; id->type = FLT32; break;
       }

       break;

    }

    strcpy(id->image_mod,"PT");

    bytes = id->width*id->height*MdcType2Bytes(id->type);
    bytes = MdcMatrixBlocks(bytes);

    id->buf = MdcGetImgBuffer(bytes);   
    if (id->buf == NULL) return("ECAT Bad malloc image buffer");

    error = mat_read_matrix_data(fp,startblk,endblk,(Int16 *)id->buf);
    if (error) {
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayErr("ECAT Bad read matrix data");
      }else
#endif
      {
      MdcPrntWarn("ECAT Bad read matrix data"); 
      }
      err=MdcHandleTruncated(fi,img+1,MDC_YES);
      if(err != NULL) return(err);
    }

    if (fi->truncated) break;
      
    img+=1;
  }

  /* check the images really found */
  if (img != fi->number) {
    err=MdcHandleTruncated(fi,img,MDC_YES);
    if (err != NULL) return(err);
  }

  /* fill in other FILEINFO variables */
  id = &fi->image[0];  /* first image */
  fi->bits = id->bits;
  fi->type = id->type;
  fi->pixdim[0]=3;
  fi->pixdim[1]=id->pixel_xsize;
  fi->pixdim[2]=id->pixel_ysize;
  fi->pixdim[3]=id->slice_width;
  if (mh.file_type == MDC_ECAT_IMAGE_FILE) {
    MdcStringCopy(fi->filter_type,MdcEcatFilterTypes[abs(ish.filter_code)],
                                  MDC_MAX_ECATFLTRTYPES_SIZE);
    fi->reconstructed = MDC_YES;
    if (ish.decay_corr_fctr > 1.0 ) fi->decay_corrected = MDC_YES;

    MdcStringCopy(fi->recon_method,ish.annotation,MDC_MAXSTR);
    if (strcmp(fi->recon_method,MDC_ECAT_RECON_METHOD)  == 0 ) {
      strcpy(fi->recon_method,"Filtered Backprojection");
    }
  }else{
    fi->reconstructed = MDC_NO;
    strcpy(fi->recon_method,"None");
  }

  switch( data_type ) {
    case  BYTE_TYPE:  MDC_FILE_ENDIAN = MDC_HOST_ENDIAN; break;
/*  case  SUN_I2   : */
    case  M68K_I2  :  MDC_FILE_ENDIAN = MDC_BIG_ENDIAN;  break;
    case  VAX_I2   :  MDC_FILE_ENDIAN = MDC_HOST_ENDIAN; break;
/*  case  SUN_I4   : */
    case  M68K_I4  :  MDC_FILE_ENDIAN = MDC_BIG_ENDIAN;  break;
    case  VAX_I4   :  MDC_FILE_ENDIAN = MDC_HOST_ENDIAN; break;
/*  case  SUN_R4   : */
    case  IEEE_R4  :  MDC_FILE_ENDIAN = MDC_BIG_ENDIAN;  break;
    case  VAX_R4   :  MDC_FILE_ENDIAN = MDC_HOST_ENDIAN; break;
  }

  MdcFileClose(fi->ifp);

  if (fi->truncated) return("ECAT Truncated image file");

  return NULL;
}

int MdcGetFilterCode(char *string)
{
  int i = 0;

  for (i=0; i<MDC_MAX_ECATFLTRTYPES; i++) 
     if (strstr(string,MdcEcatFilterTypes[i]) != NULL) break;

  return(i);
}

void MdcFillMainHeader(FILEINFO *fi, Main_header *mh)
{
  int i;

  /* memset(mh,0,MH_64_SIZE); */
  memset(mh,0,sizeof(Main_header));

  sprintf(mh->original_file_name,"%.19s",fi->ofname);
  mh->sw_version = 6;
  mh->system_type= 951;
  mh->file_type  = 2;

  mh->data_type = 2;     /* ECAT 6 only reads VAX Int16 */

  sprintf(mh->isotope_code,"%.8s",fi->isotope_code);
  mh->isotope_halflife = fi->isotope_halflife;
  sprintf(mh->radiopharmaceutical,"%.31s",fi->radiopharma);
  mh->calibration_units = fi->image[0].calibr_units;
  if (fi->pixdim[0] >= 3.) /* only valid for TRANSVERSE slices */
  mh->axial_fov = ((float)fi->dim[3] + 1.) * fi->pixdim[3] / 10.;
  mh->scan_start_day   = fi->study_date_day;
  mh->scan_start_month = fi->study_date_month;
  mh->scan_start_year  = fi->study_date_year;
  mh->scan_start_hour  = fi->study_time_hour;
  mh->scan_start_minute= fi->study_time_minute;
  mh->scan_start_second= fi->study_time_second;
  mh->plane_separation = fi->image[0].slice_spacing/10.; /* in cm */
  sprintf(mh->study_name,"%.11s",fi->study_name);
  mh->gantry_tilt = fi->gantry_tilt;
  sprintf(mh->patient_id,"%.15s",fi->patient_id); 
  sprintf(mh->patient_name,"%.31s",fi->patient_name);
  mh->patient_sex = fi->patient_sex[0];
  sprintf(mh->study_description,"%.31s",fi->study_descr);
  switch (fi->acquisition_type ) {
    case MDC_ACQUISITION_STATIC : 
        mh->acquisition_type = MDC_ECAT_ACQTYPE_STATIC_EMISSION;  break;
    case MDC_ACQUISITION_TOMO   :   
        mh->acquisition_type = MDC_ECAT_ACQTYPE_STATIC_EMISSION;  break;
    case MDC_ACQUISITION_DYNAMIC:   
        mh->acquisition_type = MDC_ECAT_ACQTYPE_DYNAMIC_EMISSION; break;
    default                     : 
        mh->acquisition_type = MDC_ECAT_ACQTYPE_UNKNOWN;

  }
  sprintf(mh->facility_name,"%.19s",MDC_INSTITUTION);
  sprintf(mh->user_process_code,"%.10s",MDC_PRGR);

  mh->num_planes = mh->num_frames = mh->num_gates = 1;
  mh->num_bed_pos = 1;
 
  for ( i=3; i<=fi->dim[0]; i++) {
     switch (i) {
       case 3: mh->num_planes  = fi->dim[i];     break;
       case 4: mh->num_frames  = fi->dim[i];     break;
       case 5: mh->num_gates   = fi->dim[i];     break;
       case 6: mh->num_bed_pos = fi->dim[i];     break;
       case 7: mh->num_bed_pos*= fi->dim[i];     break;
     }
  }
  mh->num_bed_pos -= 1; /* zero-based */

}

void MdcFillImageSubHeader(FILEINFO *fi,Image_subheader *ish
                            ,int type,Int32 img, Int32 matnum, Uint32 NEWSIZE)
{
  IMG_DATA *id = &fi->image[img];

  /* memset(ish,0,ISH_64_SIZE); */
  memset(ish,0,sizeof(Image_subheader));

  ish->data_type = 2;           /* ECAT 6 only reads VAX Int16 */

  ish->num_dimensions = 2;
  if (fi->diff_size || NEWSIZE) {
    ish->dimension_1 = fi->mwidth;
    ish->dimension_2 = fi->mheight;
  }else{
    ish->dimension_1 = id->width;
    ish->dimension_2 = id->height;
  }
  ish->recon_scale = id->recon_scale; 
 
  if (ish->data_type == 1 || ish->data_type == 2) {
    if (id->rescaled) {
      ish->image_min = (Int16) id->rescaled_min;
      ish->image_max = (Int16) id->rescaled_max;
    }else{
      ish->image_min = (Int16) id->min;
      ish->image_max = (Int16) id->max; 
    }
  }else{ /* data types too big for an Int16 */
    ish->image_min   = 0;
    ish->image_max   = 0;
  }
  ish->pixel_size  = ((id->pixel_xsize + id->pixel_ysize)/2.) / 10.;
  ish->slice_width = id->slice_width / 10.;
#ifdef MDC_USE_SLICE_SPACING
  if (fi->number > 1) ish->slice_width = id->slice_spacing / 10.;
#endif
  ish->frame_duration   = (Int32) id->frame_duration;
  ish->frame_start_time = (Int32) id->frame_start;
  ish->filter_code      = -(MdcGetFilterCode(fi->filter_type));

  ish->scan_matrix_num  = matnum;
  ish->norm_matrix_num  = matnum;
  ish->atten_cor_matrix_num = matnum;

  ish->quant_units      = id->quant_units;

  if (id->rescaled) {
    ish->quant_scale = id->rescaled_fctr;
    ish->ecat_calibration_fctr = 1.;
  }else{  
    ish->quant_scale = id->quant_scale;
    ish->ecat_calibration_fctr = id->calibr_fctr;
  }

  if (strcmp(fi->recon_method,"Filtered Backprojection") == 0 ) {
    sprintf(ish->annotation,"%.40s",MDC_ECAT_RECON_METHOD);
  }else{ 
    sprintf(ish->annotation,"%.40s",fi->recon_method);
  }
}

static void MdcResetSizes(FILEINFO *fi)
{
  fi->mwidth = saved_mwidth;
  fi->mheight= saved_mheight;
}
 
char *MdcWriteECAT(FILEINFO *fi)
{
  IMG_DATA *id;
  Main_header mh;
  Image_subheader ish; 
  Uint8 *buf, *maxbuf;
  Uint16 type, FREE;
  Int32 matnum, data=0, bed, gate, frame, plane, img=0;
  Uint32 size, NEWSIZE=0;

  MDC_WRITE_ENDIAN = MDC_LITTLE_ENDIAN; /* always (VAX) */

  if (XMDC_MEDCON == MDC_NO)
    MdcDefaultName(MDC_FRMT_ECAT,fi->ofname,fi->ifname);

#if XSUPPORTED
  if (XMDC_MEDCON) XMdcBeginProgressBar("Writing ECAT:");
#endif

  if (MDC_VERBOSE) MdcPrntMesg("ECAT Writing <%s> ...",fi->ofname);

  if (MdcFileExists(fi->ofname)) {
    MdcResetSizes(fi);
    return("ECAT File exists!!");
  }

  if (MDC_FORCE_INT != MDC_NO) {
    if (MDC_FORCE_INT != BIT16_S) {
#if XSUPPORTED
      if (XMDC_MEDCON) {
        XMdcDisplayWarn("ECAT Only Int16 pixels supported");
      }else
#endif
      {
      MdcPrntWarn("ECAT Only Int16 pixels supported");
      }
    }
  } 

  /* check some integrities */

  /* check integrity of planes, frames, gates, beds */
  if (fi->dim[3] > MDC_ECAT_MAX_PLANES)
    return("ECAT number of planes too big (1024)");
  if (fi->dim[4] > MDC_ECAT_MAX_FRAMES)
    return("ECAT number of frames too big (512)");
  if (fi->dim[5] > MDC_ECAT_MAX_GATES)
    return("ECAT number of gates  too big (64)");
  if ((fi->dim[6]*fi->dim[7]) > MDC_ECAT_MAX_GATES)
    return("ECAT number of beds   too big (16)");


  /* check dimensions (ECAT only 64, 128, 256) */
  /* we don't do downsaling                    */
  if (fi->mwidth > MDC_ECAT_MAX_DIMS || fi->mheight > MDC_ECAT_MAX_DIMS)
    return("ECAT dimensions too big (256)");

  /* get maximum dimension */
  if (fi->mwidth > fi->mheight) size = fi->mwidth;
  else size = fi->mheight;

  /* allow only 64, 128, 256 */
  if      (size <=  64) NEWSIZE=64;
  else if (size <= 128) NEWSIZE=128;
  else if (size <= 256) NEWSIZE=256;

  /* save the original dimensions anyway */
  saved_mwidth = fi->mwidth;
  saved_mheight= fi->mheight;

  /* change to new dimensions */
  if (NEWSIZE) {
    fi->mwidth = NEWSIZE;
    fi->mheight= NEWSIZE;
  }

  MdcFillMainHeader(fi,&mh);

  if ( (fi->ofp = mat_create(fi->ofname,&mh)) == NULL) {
    MdcResetSizes(fi);
    return("Couldn't create file");
  }

  for (bed=0; bed <= mh.num_bed_pos; bed++)
  for (gate=1; gate <= mh.num_gates; gate++)
  for (frame=1; frame <= mh.num_frames; frame++)
  for (plane=1; plane <= mh.num_planes; plane++) {

#if XSUPPORTED
     if (XMDC_MEDCON) {
       pvalue += 1./(float)fi->number;
       XMdcUpdateProgressBar(NULL);
     }
#endif

     id = &fi->image[img];

     if (id->type != BIT16_S) {
       buf = MdcGetImgBIT16_S( fi, img);
       FREE=MDC_YES;  type=BIT16_S;
     }else{
       buf = id->buf;
       FREE=MDC_NO; type=id->type;
     }

     matnum = mat_numcod(frame,plane,gate,data,bed);
     MdcFillImageSubHeader(fi,&ish,type,img,matnum,NEWSIZE); 

     if (fi->diff_size || NEWSIZE) {
       size = fi->mwidth * fi->mheight * MdcType2Bytes(type);

       maxbuf = MdcGetResizedImage(fi, buf, type, img);
       if (maxbuf == NULL) {
         MdcResetSizes(fi);
         return("ECAT Bad malloc maxbuf");
       }

       if (FREE) MdcFree(buf); 

       FREE=MDC_YES;

     }else{ /* NEWSIZE is normally always set */
        size = id->width * id->height * MdcType2Bytes(type);
        maxbuf = buf;
     }
 
     matnum = mat_numcod(frame,plane,gate,data,bed);  
     if (mat_write_image(fi->ofp,matnum,&ish,(Uint16 *)maxbuf,(Int32)size)) {
       MdcResetSizes(fi);
       return("ECAT Bad write image matrix");
     }

     img+=1;

     if (FREE) MdcFree(maxbuf);
  }

  MdcFileClose(fi->ofp);

  MdcResetSizes(fi);

  return NULL;
}

void MdcPrintEcatInfoDB(Main_header *mh)
{
  char Unknown[8]="Unknown";

  Uint32 i, patient_strlen, study_strlen;

  patient_strlen = strlen(mh->patient_name);
  study_strlen   = strlen(mh->study_name);

  /* remove # from strings, because it is used as field separator! */
  for (i=0; i<patient_strlen; i++) {
     if ( mh->patient_name[i] == '#' ) {
       mh->patient_name[i]='$';
     }
  }

  /* print database info: study_name */
  if (study_strlen != 6) {
    printf("%s",Unknown);
  }else{
    printf("%s",mh->study_name);
  }
  printf("# ");

  /* print database info: patient_name */
  if (patient_strlen == 0) {
    printf("%-35s",Unknown);
  }else{
    printf("%-35s",mh->patient_name);
  }

  printf("#");

  /* print database info: scan date */
  printf("%02d-",mh->scan_start_day);
  switch (mh->scan_start_month) {
    case  1: printf("Jan"); break;
    case  2: printf("Feb"); break;
    case  3: printf("Mar"); break;
    case  4: printf("Apr"); break;
    case  5: printf("May"); break;
    case  6: printf("Jun"); break;
    case  7: printf("Jul"); break;
    case  8: printf("Aug"); break;
    case  9: printf("Sep"); break;
    case 10: printf("Oct"); break;
    case 11: printf("Nov"); break;
    case 12: printf("Dec"); break;
  }
  printf("-%4d",mh->scan_start_year);
  printf("\n");
}

