/*************************
 * libdicom by Tony Voet *
 *************************/

#include <stdlib.h>
#include <string.h>
#include "dicom.h"

static S32 dicom_pixel(const ELEMENT *);

static SINGLE single;

/**********
 * single *
 **********/

SINGLE *dicom_single(void)
{
  ELEMENT	*e;
  S32		length;
  U32		i;
  char		*interpretation[]=
  {
    "MONOCHROME2",
    "MONOCHROME1",
    "PALETTE COLOR",
    "RGB",
    "HSV",
    "ARGB",
    "CMYK",
    "UNKNOWN"
  };

  dicom_log(DEBUG,"dicom_single()");

  memset(&single,0,sizeof(SINGLE));
  single.frames=1;
  single.samples=1;

  for (;;)
  {
    e=dicom_element();
    if (!e)
      break;

    if (e->group==0x0028)
    {
      if (e->element==0x0002)
      {
        if (dicom_load(US))
          break;
        single.samples=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0004)
      {
        if (dicom_load(CS))
          break;
        dicom_clean();

        for (single.photometric=0; single.photometric<UNKNOWN;
          single.photometric++)
          if ( !strncmp(*e->value.CS,interpretation[single.photometric],
            strlen(interpretation[single.photometric])) )
            break;

        if (single.photometric==UNKNOWN)
          dicom_log(WARNING,"Unknown PhotometricInterpretation");

        eNlfSafeFree(e->value.CS);
        continue;
      }

      if (e->element==0x0006)
      {
        if (dicom_load(US))
          break;
        single.planar=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0008)
      {
        if (dicom_load(IS))
          break;
        dicom_clean();
        single.frames=atoi(*e->value.IS);
        eNlfSafeFree(e->value.IS);
        continue;
      }

      if (e->element==0x0010)
      {
        if (dicom_load(US))
          break;
        single.h=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0011)
      {
        if (dicom_load(US))
          break;
        single.w=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0100)
      {
        if (dicom_load(US))
          break;
        single.alloc=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0101)
      {
        if (dicom_load(US))
          break;
        single.bit=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0102)
      {
        if (dicom_load(US))
          break;
        single.high=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (e->element==0x0103)
      {
        if (dicom_load(US))
          break;
        single.sign=*e->value.US;
        eNlfSafeFree(e->value.US);
        continue;
      }

      if (0x1101<=e->element && e->element<=0x1103)
      {
        if (dicom_load(US))
          break;

	if (e->vm!=3)
	  dicom_log(WARNING,"Wrong VM for PaletteColorLookupTableDescriptor");
	else
	{
	  i=e->element-0x1101;
	  single.clut[i].size=e->value.US[0];
	  single.clut[i].threshold.u16=e->value.US[1];
	  single.clut[i].bit=e->value.US[2];
	}

        eNlfSafeFree(e->value.US);
        continue;
      }

      if (0x1201<=e->element && e->element<=0x1203)
      {
        if (dicom_load(US))
          break;
	single.clut[e->element-0x1201].data.u16=e->value.US;
        continue;
      }
    }

    if (!(e->group&1))
      if (0x7F00<=e->group && e->group<0x7FFF)
        if (e->element==0x0010)
	{
          /* first fix bad VR values (OB instead of OW) */
          /* necessary for pixel image data endian swap */
          if (e->vr==OB && single.alloc==16) {
            dicom_log(WARNING,"Incorrect OB value representation (fixed)");
            e->vr=OW;
          }

	  length=dicom_pixel(e);
	  if (length<0)
	    break;

          if (length!=
            single.frames*single.w*single.h*single.samples*single.alloc>>3)
            dicom_log(WARNING,"Incorrect PixelData length");

          return &single;
	}

    if (dicom_skip())
      break;
  }

  dicom_single_free();

  return 0L;
}

/***************
 * single free *
 ***************/

void dicom_single_free(void)
{
  int i;

  dicom_log(DEBUG,"dicom_single_free()");

  for (i=0; i<3; i++)
    eNlfSafeFree(single.clut[i].data.u16);

  eNlfSafeFree(single.data);

  memset(&single,0,sizeof(SINGLE));
}

/*********
 * pixel *
 *********/

static S32 dicom_pixel(const ELEMENT *e)
{
  int error;

  dicom_log(DEBUG,"dicom_pixel()");

  if (e->length!=0xFFFFFFFF)
  {
    if (single.alloc==16)
      error=dicom_load(OW);
    else
      error=dicom_load(OB);

    if (error)
      return -1;

    single.data=e->value.OW;

    return e->length;
  }

  if (dicom_skip())
    return -2;

  dicom_log(EMERGENCY,"Encapsulated PixelData is not implemented yet");

  return -3;
}
