/*
   XMascot Ver 2.5   image-lib
   Copyright(c) 1996 Go Watanabe     go@cclub.tutcc.tut.ac.jp
                     Tsuyoshi IIda   iida@cclub.tutcc.tut.ac.jp
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "image.h"

typedef struct {
	Display *dpy;
	int screen;
	Window root;
	int depth;
#ifdef SHADOW
    int shadow;
#endif
} ImageLibData;

static ImageLibData ImgLibDat;

ImageData load_pnm( FILE *fp,int col0,int rgb0 ); /* pnm.c */
ImageData load_mag( FILE *fp,int col0,int rgb0 ); /* mag.c */
ImageData load_tif( FILE *fp,int col0,int rgb0 ); /* tif.c */

/* $B=i4|2=(B */
void imagelib_init( Display *ds,int sc ,Window r, int de )
{
	ImgLibDat.dpy    = ds;
	ImgLibDat.screen = sc;
	ImgLibDat.root   = r;
	ImgLibDat.depth  = de;
#ifdef SHADOW
	ImgLibDat.shadow = 0;
#endif
}

#ifdef SHADOW
void image_set_shadow_len( int s )
{
	ImgLibDat.shadow = s;
}

int image_get_shadow_len()
{
	return ImgLibDat.shadow;
}
#endif

/* $B%$%a!<%8%G!<%?=i4|2=(B */
void init_image( ImageData *img )
{
	img->w = img->h = img->n = 0;
	img->data = NULL;
#ifdef SHAPE
	img->mask = NULL;
#endif
	img->pixels = NULL;
}
	
/* $B%$%a!<%8%G!<%?$N2rJ|(B */
void free_image( ImageData *img )
{
	if( img->data != NULL ) XtFree((char*)(img->data));
	img->data = NULL;
#ifdef SHAPE
	if( img->mask != NULL ) XtFree((char*)(img->mask));
	img->mask = NULL;
#endif
}	

/* $B?'$r2rJ|$9$k(B */
void free_col( ImageData *img )
{
	if( img->pixels != 0 ){
		XFreeColors(ImgLibDat.dpy,img->colormap,img->pixels,img->n,0);
		XtFree( (char*)(img->pixels) );
		img->pixels = NULL;
		img->n      = 0;
	}
}

/* $B%+%i!<$r$H$j$"$($:3d$jEv$F$k(B & $BIQEY7W;;(B */
int get_col( ImageData *img, int r, int g, int b )
{
	int i;
	PalletData *pal = img->pal;
	int *use        = img->use;

	for(i=0;i<img->n;i++)
		if(pal[i].r == r && pal[i].g == g && pal[i].b == b){
			use[i]++;
			return i;
		}
	if( img->n < MAX_COL_NUM ){
		pal[img->n].r = r;
		pal[img->n].g = g;
		pal[img->n].b = b;
		use[img->n]   = 1;
		return img->n++;
	}
	return 0;
}

/*
   $B?'$KM%@h=g0L$r$D$1$F3JG<$9$k!#(B
   $BM%@hEY$O!";HMQIQEY$H3NJ]$5$l$F$$$kA4$F$N?'$H$N?'%Y%/%H%k$N5wN%(B
   ($B$J$k$Y$/1s$$$b$N$rM%@h(B)$B$r$b$H$K7W;;$5$l$k!#(B

   $B7k2L(B:
   img->pal.{r,g,b} $BM%@h=g$K%=!<%H$5$l$?%Q%l%C%H%G!<%?(B
   img->use         $B85$N$b$N$H$N4V$NJQ49%F!<%V%k(B

   $B$3$N$"$?$j$N%k!<%A%s$O(B XV $B$r;29M$K$7$F$^$9!#(B

*/
static void sort_col( ImageData *img ) 
{
	struct colmap{
		PalletData pal;
		int use;
		int trans;	     	   /* its index in the old colormap */
		int min_dist;          /* min distance to a selected color */
	} *c0, *c;

	PalletData *pal = img->pal;
	int        *use = img->use; 
	int         n   = img->n;

	unsigned r, g, b;
	int i, j, m, entry, d;

	c0 = (struct colmap*)XtMalloc( sizeof(struct colmap) * n );

	/* $B%G!<%?$N%3%T!<(B */
	c = c0;
	for(c=c0,i=0;i<n;i++,c++){
		c->pal      = pal[i];
		c->use      = use[i];
		c->trans    = i;
		c->min_dist = 1000000; /* 255^2 * 3 = 195075 */
	}

	/* $B:G=i$O:GBg;HMQ2s?t$NB?$$$b$N$rA*$V(B */
	for(m=0,j=0;j<n;j++)
		if( m < c0[j].use )
			m = c0[entry = j].use;

	/* 0$BHV$a$NEPO?(B */
	r = pal[0].r = c0[entry].pal.r;
	g = pal[0].g = c0[entry].pal.g;
	b = pal[0].b = c0[entry].pal.b;
	use[c0[entry].trans] = 0;	
	
	c0[entry].use = 0;		/* $B;HMQ$7$?(B */

	for(i=1;i<n;i++){
		/* $B:G=i$N(B 10$B$3!"$^$?$O4q?t8D$a$O!"5wN%$,1s$$$b$N$rMQ$$$k(B */ 
		if( (i & 1) || (i>0 && i<10) ){
			m = 0;
			for(j=0,c=c0;j<n;j++,c++){
				if( !c->use ) continue;			/* $B$9$G$K;HMQ:Q(B */
				/* $B5wN%$N7W;;(B ($B%Y%/%H%k(B) */
				d = (c->pal.r - r) * (c->pal.r - r) + 
					(c->pal.g - g) * (c->pal.g - g) + 
					(c->pal.b - b) * (c->pal.b - b);
				if( d < c->min_dist ) 
					c->min_dist = d;
				/* $B:GBg5wN%$N$b$N$rC5$9(B */
				if( m < c->min_dist ){
					m = c->min_dist;
					entry = j;
				}
			}
		/* $B$=$l$i0J30$O(B $B;HMQIQEY(B */	
		}else{
			m = 0;
			for(j=0,c=c0;j<n;j++,c++){
				if( !c->use ) continue;
				/* $B5wN%$N7W;;(B ($B%Y%/%H%k(B) */
				d = (c->pal.r - r) * (c->pal.r - r) + 
					(c->pal.g - g) * (c->pal.g - g) + 
					(c->pal.b - b) * (c->pal.b - b);
				if( d < c->min_dist ) 
					c->min_dist = d;
				/* $B:GBg;HMQIQEY$N$b$N$rC5$9(B */
				if( m < c->use ){
					m = c->use;
					entry = j;
				}
			}
		}
		/* i $BHV$a$NEPO?(B */
		r = pal[i].r = c0[entry].pal.r;
		g = pal[i].g = c0[entry].pal.g;
		b = pal[i].b = c0[entry].pal.b;
		use[c0[entry].trans] = i;
		/* $B;HMQ:Q(B */
		c0[entry].use = 0;	
	}

	XtFree( (char *)c0 );
}

/* $B?'$N3d$jEv$F(B */
void set_col(ImageData *img, Colormap cmap)
{
	int i,j,nc;
	XColor defs[MAX_COL_NUM], ctab[MAX_COL_NUM];
	int    failed[MAX_COL_NUM], map[MAX_COL_NUM];
	unsigned long *pixels;

	PalletData *pal = img->pal;
	int n           = img->n;

	int get_n = 0;

	pixels = (unsigned long*)XtMalloc( sizeof( u_long ) * n );

	img->pixels = pixels;
	img->colormap = cmap;

	/* $B?'$N<hF@M%@h=g=x$r7hDj(B */
	sort_col( img );

	/* 1$B2sL\(B $B?75,3d$jEv$F(B */

	for(i=0;i<n;i++){
		defs[i].red   = pal[i].r << 8;
		defs[i].green = pal[i].g << 8;
		defs[i].blue  = pal[i].b << 8;
		defs[i].flags = DoRed | DoGreen | DoBlue;
		if( !XAllocColor( ImgLibDat.dpy, cmap, &defs[i]) ){
			/* $B<hF@<:GT(B */
			failed[i] = 1;
		}else { 
			map[get_n++] = i;
			failed[i] = 0;
			pixels[i] = defs[i].pixel;
		}
	}
	
	msg_out( "Color 1st session %d/%d\n",get_n,n);

	/* $BA4$F$N?'$r<hF@$7$?(B */
	if( get_n >= n-1 )
		return;

	/* 2$B2sL\(B $B4{B8$N?'$+$i$H$C$F$/$k(B */
	nc   = DisplayCells( ImgLibDat.dpy, ImgLibDat.screen );
	nc   = ( nc < MAX_COL_NUM )? nc:MAX_COL_NUM;

	/* $BA4?'>pJs$N<hF@(B */
	for(i=0;i<nc;i++) 
		ctab[i].pixel = i;
	XQueryColors(ImgLibDat.dpy, cmap, ctab, nc );

	for(i=0;i<n;i++){
		int d, mdist = 1000000, close = -1;
		int rd, gd, bd, ri, gi, bi;
		if( !failed[i] )
			continue; /* $B$9$G$K3d$jEv$F40N;(B */

		ri = pal[i].r;  
		gi = pal[i].g;  
		bi = pal[i].b;
		/* $B0lHV6a$$?'$rC5$9(B */
		for(j=0;j<nc;j++){
			rd = ri - (ctab[j].red   >> 8);
			gd = gi - (ctab[j].green >> 8);
			bd = bi - (ctab[j].blue  >> 8);
			d = rd * rd + gd * gd + bd * bd;
			if(d < mdist) 
				mdist = d, close = j;
		}
		/* $B?'$rMW5a(B */
		if( XAllocColor(ImgLibDat.dpy, cmap, &ctab[close]) ){ 
			map[get_n++] = i;
			defs[i]   = ctab[close];
			failed[i] = 0;
			pixels[i] = ctab[close].pixel;
		}
		/* else $B?'$N<hF@<:GT(B */
	}

	msg_out( "Color 2nd session %d/%d\n",get_n,n);

	/* $BA4$F$N?'$r<hF@$7$?(B */
	if( get_n >= n )
		return;
	
	/* 3 $B2sL\(B $B<+J,$,$9$G$K$H$C$F$$$k?'$NCf$+$i<hF@(B */

	for(i=0;i<n;i++){
		int d, k, mdist = 1000000, close = -1;
		int rd,gd,bd, ri,gi,bi;
		if(!failed[i]) 
			continue;  /* $B$9$G$K<hF@(B */
		ri = pal[i].r;
		gi = pal[i].g;
		bi = pal[i].b;
		for(j=0;j<get_n;j++){
			k = map[j];
			rd = ri - (defs[k].red   >> 8);
			gd = gi - (defs[k].green >> 8);
			bd = bi - (defs[k].blue  >> 8);
			d = rd * rd + gd * gd + bd * bd;
			if(d < mdist) 
				mdist = d,close = k;
		}
		pixels[i] = defs[i].pixel;
	}

	msg_out( "Color 3rd session %d/%d\n",get_n,n);

}

/* $B%$%a!<%8%G!<%?$N<hF@(B */
ImageData get_image( char *name, int c0, int r0 )
{
	ImageData ret;
	FILE *fp;
	char *suffix;
	char *fname;

	init_image( &ret );

	if( (fname = search(name)) == NULL )
		return ret;
	if((suffix = strrchr(name,'.')) == NULL) {
		err_out("File \"%s\" has no suffix.\n",name);
		return ret;
	}
	suffix++;
	if(  !strcmp(suffix,"pnm")
	   ||!strcmp(suffix,"ppm")
	   ||!strcmp(suffix,"pgm")
	   ||!strcmp(suffix,"pbm") ) {
		if( (fp = fopen(fname,"r")) == NULL ){
			perror("getmascot"); exit(1);
		}
		ret = load_pnm(fp,c0,r0);
	}else if(!strcmp(suffix,"mag")) {
		if( (fp = fopen(fname,"r")) == NULL ){
			perror("getmascot"); exit(1);
		}
		ret = load_mag(fp,c0,r0);
	}else if(!strcmp(suffix,"tif")) {
		if( (fp = fopen(fname,"r")) == NULL ){
			perror("getmascot"); exit(1);
		}
		ret = load_tif(fp,c0,r0);
	}else{
		/* ???top[npgm]m $B$r8F$S=P$9(B */

		char cmdline[100];
		int fd[2];
		pid_t pid;

		if( pipe(fd) < 0 ){
			perror("pipe");	exit(1);
		}
		if( (pid = fork()) < 0 ){
			perror("fork");	exit(1);
		}
		if( pid == 0 ){
			if( close(1) < 0 ){
				perror("close"); exit(1);
			}
			if( dup(fd[1]) != 1 ){
				perror("dup"); exit(1);
			}
			close(fd[1]);
			sprintf(cmdline,"%stopnm",suffix);
			execlp( cmdline, cmdline, fname, (char*)NULL );
			sprintf(cmdline,"%stoppm",suffix);
			execlp( cmdline, cmdline, fname, (char*)NULL );
			sprintf(cmdline,"%stopgm",suffix);
			execlp( cmdline, cmdline, fname, (char*)NULL );
			sprintf(cmdline,"%stopbm",suffix);
			execlp( cmdline, cmdline, fname, (char*)NULL );
			perror("execlp");
			err_out("Can't find %s\n",cmdline);
			err_out("Bad suffix \"%s\".\n",suffix);
			exit(1);
		}
		close(fd[1]);
		if( (fp = fdopen( fd[0], "r" )) == NULL ){
			perror("fdopen");
			exit(1);
		}
		ret = load_pnm(fp,c0,r0);
	}
	fclose(fp);
	return ret;
}

/* $B%$%a!<%8$r(Bpixmap$B$KJQ49$9$k(B */
Pixmap image2pixmap(ImageData *img, Pixmap *m)
{
	int i,j;
	GC gc;
	unsigned w     = img->w;
	unsigned h     = img->h;
	u_long *pixels = img->pixels;
	int      *use  = img->use;
	
	u_char   *data = img->data;
	XImage *img_data;
	Pixmap p;

#ifdef SHADOW
	unsigned long black = BlackPixel(ImgLibDat.dpy,ImgLibDat.screen);
#endif

#ifdef SHAPE
	u_char   *mask = img->mask;
	XImage *img_mask;
	Pixmap b;
#endif

	p = XCreatePixmap(ImgLibDat.dpy,ImgLibDat.root,
					  w+SHADOW_LEN,h+SHADOW_LEN,ImgLibDat.depth);
	img_data = XGetImage(ImgLibDat.dpy,p,0,0,w+SHADOW_LEN,h+SHADOW_LEN,
						 AllPlanes,ZPixmap);

#ifdef SHAPE
	if(m){
		b = XCreatePixmap(ImgLibDat.dpy,ImgLibDat.root,
						  w+SHADOW_LEN,h+SHADOW_LEN,1);
		img_mask = XGetImage(ImgLibDat.dpy,b,0,0,w+SHADOW_LEN,h+SHADOW_LEN,
							 1,ZPixmap);
	}
#endif


#ifdef SHAPE
	if(m){
#ifdef SHADOW
		if(SHADOW_LEN){
			for(i=h+SHADOW_LEN-1;i>=0;i--)
				for(j=w+SHADOW_LEN-1;j>=0;j--)
					if( i<h && j<w && mask[i*w+j] ){
						XPutPixel(img_data,j,i,pixels[use[data[i*w+j]]]);
						XPutPixel(img_mask,j,i,1);
						if( (i+j) & 1 )
							XPutPixel(img_mask,j+SHADOW_LEN,i+SHADOW_LEN,1);
					}else{
						XPutPixel(img_data,j,i,black);
						XPutPixel(img_mask,j,i,0);
					}
		}else
#endif
		{
			for(i=0;i<h;i++)
				for(j=0;j<w;j++)
					if( mask[i*w+j] ){
						XPutPixel(img_data,j,i,pixels[use[data[i*w+j]]]);
						XPutPixel(img_mask,j,i,1);
					}else{
						XPutPixel(img_mask,j,i,0);
					}
		}
	}else
#endif /* SHAPE */
	{
		for(i=0;i<h;i++)
			for(j=0;j<w;j++)
				XPutPixel(img_data,j,i,pixels[use[data[i*w+j]]]);
	}

	gc = XCreateGC(ImgLibDat.dpy,p,0,NULL);
	XPutImage(ImgLibDat.dpy,p,gc,img_data,0,0,0,0,w+SHADOW_LEN,h+SHADOW_LEN);
	XFreeGC(ImgLibDat.dpy,gc);
	XDestroyImage(img_data);

#ifdef SHAPE
	if(m){
		gc = XCreateGC(ImgLibDat.dpy,b,0,NULL);
		XPutImage(ImgLibDat.dpy,b,gc,img_mask,0,0,0,0,
				  w+SHADOW_LEN,h+SHADOW_LEN);
		*m = b;
		XFreeGC(ImgLibDat.dpy,gc);
		XDestroyImage(img_mask);
	}
#endif
	return p;
}

