/*
Konami "Twin16"

driver by Phil Stroffolino

CPU   : 68000(10M)x2 Z-80
SOUND : YM2151 007232 uPD7759C
OSC.  : 3.579545MHz 18432.00KHz

Main processors are a pair of 68000 CPUs
Sounds are generated by a Z80, a Yamaha 2151 and 3012, a Konami custom IC and a UPD7759C

		Dark Adventure / Devil World / Majuu no Ohkoku
		Vulcan Venture / Gradius II
		Cuebrick
		MIA (Japan)
		Final Round / Hard Puncher (Japan)

Known Issues:
	- battery-backed RAM isn't handled properly (CueBrick)
	- some rogue sprites in Devil World
	- sprite-background priority isn't correct (see intro screen of Devil World)

68000 Memory Map for Konami Twin System

	CPUA										CPUB
	0x000000..0x03ffff	ROM						0x000000..0x03ffff
	0x040000..0x043fff	communication RAM		0x040000..0x043fff (shared)
	0x060000..0x063fff	work RAM				0x060000..0x063fff
	0x080000..0x080fff	palette
	0x080000..0x09ffff	ROM (extra tile data)
	0x0a0000..0x0a0001	IRQ control				0x0a0000..0x0a0001
	0x0a0008..0x0a0009	sound command
	0x0a0010..0xa00011	watchdog
	0x0c0000..0x0c0001	screenflip
	0x0c0002..0x0c000f	scroll registers

	0x100000..0x103fff	FIXRAM (text layer)
	0x120000..0x123fff	VIDRAM (tilemaps)		0x480000..0x483fff (shared)
	0x140000..0x143fff	OBJRAM (sprites)		0x400000..0x403fff (shared)
						ZIP RAM	(tiles)			0x500000..0x53ffff
						gfx ROM (banked)		0x600000..0x77ffff
						sprite gfx RAM			0x780000..0x79ffff
*/

#include "driver.h"
#include "vidhrdw/generic.h"
#include "cpu/m68000/m68000.h"
#include "cpu/z80/z80.h"

WRITE16_HANDLER( fround_gfx_bank_w );
WRITE16_HANDLER( twin16_video_register_w );

extern int twin16_vh_start( void );
extern void twin16_vh_stop( void );
extern void twin16_vh_screenrefresh( struct osd_bitmap *bitmap, int fullrefresh );

extern void twin16_spriteram_process( void );

/******************************************************************************************/

UINT16 twin16_custom_vidhrdw;
data16_t *twin16_gfx_rom;
data16_t *twin16_sprite_gfx_ram;
data16_t *twin16_tile_gfx_ram;
data16_t *twin16_fixram; /* text layer */

static data16_t twin16_CPUA_register, twin16_CPUB_register;
#define CPUA_IRQ_ENABLE (twin16_CPUA_register&0x20)
#define CPUB_IRQ_ENABLE (twin16_CPUB_register&0x02)

static UINT8 twin16_soundlatch;
static data16_t twin16_sound_command;

static data16_t *battery_backed_ram;



int twin16_spriteram_process_enable( void )
{
	return (twin16_CPUA_register&0x40)==0;
}

enum
{
	CPU_SOUND,
	CPU_B,
	CPU_A
};


/******************************************************************************************/

#define COMRAM_r					MRA16_BANK1
#define COMRAM_w					MWA16_BANK1

static void cuebrick_nvram_handler(void *file,int read_or_write)
{
	if (read_or_write)
		osd_fwrite(file,battery_backed_ram,0x4000);
	else
	{
		if (file)
			osd_fread(file,battery_backed_ram,0x4000);
		else
			memset(battery_backed_ram,0,0x4000);
	}
}

/******************************************************************************************/


static READ16_HANDLER( videoram16_r )
{
	return videoram16[offset];
}

static WRITE16_HANDLER( videoram16_w )
{
	COMBINE_DATA(videoram16 + offset);
}

static READ16_HANDLER( extra_rom_r )
{
	return ((data16_t *)memory_region(REGION_GFX3))[offset];
}

static READ16_HANDLER( twin16_gfx_rom1_r )
{
	return twin16_gfx_rom[offset];
}

static READ16_HANDLER( twin16_gfx_rom2_r )
{
	return twin16_gfx_rom[offset + 0x80000 + ((twin16_CPUB_register&0x04)?0x40000:0)];
}

static WRITE16_HANDLER( twin16_paletteram_word_w )
{ /* identical to tmnt_paletteram_w */
	int r,g,b;

	COMBINE_DATA(paletteram16 + offset);
	offset &= ~1;

	data = ((paletteram16[offset] & 0xff) << 8) | (paletteram16[offset+1] & 0xff);

	r = (data >>  0) & 0x1f;
	g = (data >>  5) & 0x1f;
	b = (data >> 10) & 0x1f;

	r = (r << 3) | (r >> 2);
	g = (g << 3) | (g >> 2);
	b = (b << 3) | (b >> 2);

	palette_change_color(offset / 2,r,g,b);
}


/******************************************************************************************/

static WRITE16_HANDLER( sound_command_w )
{
	COMBINE_DATA(&twin16_sound_command);
	soundlatch_w( 0, twin16_sound_command&0xff );
}

static int CPUA_interrupt( void )
{
	return CPUA_IRQ_ENABLE?MC68000_IRQ_5:MC68000_INT_NONE;
}

static int CPUB_interrupt( void )
{
	return CPUB_IRQ_ENABLE?MC68000_IRQ_5:MC68000_INT_NONE;
}

static READ16_HANDLER( twin16_sprite_status_r )
{
	/*
		return value indicates whether the spriteram16-processing circuitry
		is busy.

		for now, we'll just alternate the value every time it is read
	*/
	static int k;
	k = 1-k;
	return k;
}

static WRITE16_HANDLER( twin16_CPUA_register_w )
{
	/*
		7	6	5	4	3	2	1	0
			?							sprite protection disable
				X						IRQ5 enable (CPUA)
					X					0->1 trigger IRQ6 on CPUB
						X				0->1 trigger IRQ on sound CPU
								x	x	coin counters
	*/
	data16_t old = twin16_CPUA_register;
	COMBINE_DATA(&twin16_CPUA_register);
	if( twin16_CPUA_register!=old )
	{
		if( (old&0x08)==0 && (twin16_CPUA_register&0x08) )
		{
			cpu_cause_interrupt( CPU_SOUND, 0xff );
		}

		if( (old&0x40) && (twin16_CPUA_register&0x40)==0 )
		{
			twin16_spriteram_process();
		}

		if( (old&0x10)==0 && (twin16_CPUA_register&0x10) )
		{
			cpu_cause_interrupt( CPU_B, MC68000_IRQ_6 );
		}
		coin_counter_w( 0, twin16_CPUA_register&0x01 );
		coin_counter_w( 1, twin16_CPUA_register&0x02 );
	}
}

static WRITE16_HANDLER( twin16_CPUB_register_w )
{
	/*
		7	6	5	4	3	2	1	0
							X			gfx bank select
								X		IRQ5 enable
									X	0->1 trigger IRQ6 on CPUA
	*/
	data16_t old = twin16_CPUB_register;
	COMBINE_DATA(&twin16_CPUB_register);
	if( twin16_CPUB_register!=old )
	{
		if( (old&0x01)==0 && (twin16_CPUB_register&0x1) )
		{
			cpu_cause_interrupt( CPU_A, MC68000_IRQ_6 );
		}
	}
}

static WRITE16_HANDLER( fround_CPU_register_w )
{
	data16_t old = twin16_CPUA_register;
	COMBINE_DATA(&twin16_CPUA_register);
	if( twin16_CPUA_register!=old )
	{
		if( (old&0x08)==0 && (twin16_CPUA_register&0x08) )
			cpu_cause_interrupt( CPU_SOUND, 0xff ); /* trigger IRQ on sound CPU */
	}
}

/******************************************************************************************/

static READ16_HANDLER( twin16_input_r )
{
	switch( offset )
	{
		case 0x00: return readinputport(0); /* coin */
		case 0x01: return readinputport(1); /* p1 */
		case 0x02: return readinputport(2); /* p2 */
		case 0x03: return readinputport(3); /* p3? (Devils World) */
		case 0x08: return readinputport(5); /* DSW1 */
		case 0x09: return readinputport(4); /* DSW2 */
		case 0x0c: return readinputport(6); /* DSW3 */
	}
	return 0;
}

/******************************************************************************************/
/* sound system */

READ_HANDLER( twin16_sres_r )
{
	return twin16_soundlatch;
}

WRITE_HANDLER( twin16_sres_w )
{
	/* bit 1 resets the UPD7795C sound chip */
	if ((data & 0x02) == 0)
	{
		UPD7759_reset_w(0,(data & 0x02) >> 1);
	}
	twin16_soundlatch = data;
}


/* Added by Takahiro Nogi. (1999/10/27) */
static WRITE_HANDLER( twin16_UPD7759_start_w )
{
	UPD7759_start_w(offset, (!(data & 0x01)));
}

static MEMORY_READ_START( readmem_sound )
	{ 0x0000, 0x7fff, MRA_ROM },
	{ 0x8000, 0x8fff, MRA_RAM },
	{ 0x9000, 0x9000, twin16_sres_r },
	{ 0xa000, 0xa000, soundlatch_r },
	{ 0xb000, 0xb00d, K007232_read_port_0_r },
	{ 0xc001, 0xc001, YM2151_status_port_0_r },
	{ 0xf000, 0xf000, UPD7759_0_busy_r },
MEMORY_END

static MEMORY_WRITE_START( writemem_sound )
	{ 0x0000, 0x7fff, MWA_ROM },
	{ 0x8000, 0x8fff, MWA_RAM },
	{ 0x9000, 0x9000, twin16_sres_w },
	{ 0xb000, 0xb00d, K007232_write_port_0_w  },
	{ 0xc000, 0xc000, YM2151_register_port_0_w },
	{ 0xc001, 0xc001, YM2151_data_port_0_w },
	{ 0xd000, 0xd000, UPD7759_0_message_w },
	{ 0xe000, 0xe000, twin16_UPD7759_start_w },	/* Changed by Takahiro Nogi. (1999/10/27) */
MEMORY_END

/******************************************************************************************/

static MEMORY_READ16_START( readmem )
	{ 0x000000, 0x03ffff, MRA16_ROM },
	{ 0x040000, 0x043fff, COMRAM_r },
	{ 0x060000, 0x063fff, MRA16_RAM },
	{ 0x080000, 0x080fff, MRA16_RAM },
	{ 0x0a0000, 0x0a001b, twin16_input_r },
	{ 0x0b0000, 0x0b3fff, MRA16_RAM }, /* cuebrick only */
	{ 0x0c000e, 0x0c000f, twin16_sprite_status_r },
	{ 0x100000, 0x103fff, MRA16_RAM },
	{ 0x120000, 0x123fff, MRA16_RAM },
	{ 0x140000, 0x143fff, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( writemem )
	{ 0x000000, 0x03ffff, MWA16_ROM },
	{ 0x040000, 0x043fff, COMRAM_w },
	{ 0x060000, 0x063fff, MWA16_RAM },
	{ 0x080000, 0x080fff, twin16_paletteram_word_w, &paletteram16 },
	{ 0x081000, 0x081fff, MWA16_NOP },
	{ 0x0a0000, 0x0a0001, twin16_CPUA_register_w },
	{ 0x0a0008, 0x0a0009, sound_command_w },
	{ 0x0a0010, 0x0a0011, MWA16_NOP }, /* watchdog */
	{ 0x0b0000, 0x0b3fff, MWA16_RAM, &battery_backed_ram }, /* cuebrick only */
	{ 0x0c0000, 0x0c000f, twin16_video_register_w },
	{ 0x100000, 0x103fff, MWA16_RAM, &twin16_fixram },
	{ 0x120000, 0x123fff, MWA16_RAM, &videoram16 },
	{ 0x140000, 0x143fff, MWA16_RAM, &spriteram16 },
MEMORY_END

static MEMORY_READ16_START( readmem_sub )
	{ 0x000000, 0x03ffff, MRA16_ROM },
	{ 0x040000, 0x043fff, COMRAM_r },
	{ 0x060000, 0x063fff, MRA16_RAM },
	{ 0x080000, 0x09ffff, extra_rom_r },
	{ 0x400000, 0x403fff, spriteram16_r },
	{ 0x480000, 0x483fff, videoram16_r },
	{ 0x500000, 0x53ffff, MRA16_RAM },
	{ 0x600000, 0x6fffff, twin16_gfx_rom1_r },
	{ 0x700000, 0x77ffff, twin16_gfx_rom2_r },
	{ 0x780000, 0x79ffff, MRA16_RAM },
MEMORY_END

static MEMORY_WRITE16_START( writemem_sub )
	{ 0x000000, 0x03ffff, MWA16_ROM },
	{ 0x040000, 0x043fff, COMRAM_w },
	{ 0x060000, 0x063fff, MWA16_RAM },
	{ 0x0a0000, 0x0a0001, twin16_CPUB_register_w },
	{ 0x400000, 0x403fff, spriteram16_w },
	{ 0x480000, 0x483fff, videoram16_w },
	{ 0x500000, 0x53ffff, MWA16_RAM, &twin16_tile_gfx_ram },
	{ 0x780000, 0x79ffff, MWA16_RAM, &twin16_sprite_gfx_ram },
MEMORY_END

/******************************************************************************************/

static MEMORY_READ16_START( fround_readmem )
	{ 0x000000, 0x03ffff, MRA16_ROM },
	{ 0x040000, 0x043fff, COMRAM_r },
	{ 0x060000, 0x063fff, MRA16_RAM },
	{ 0x080000, 0x080fff, MRA16_RAM },
	{ 0x0a0000, 0x0a001b, twin16_input_r },
	{ 0x0c000e, 0x0c000f, twin16_sprite_status_r },
	{ 0x100000, 0x103fff, MRA16_RAM },
	{ 0x120000, 0x123fff, MRA16_RAM },
	{ 0x140000, 0x143fff, MRA16_RAM },
	{ 0x500000, 0x6fffff, twin16_gfx_rom1_r },
MEMORY_END

static MEMORY_WRITE16_START( fround_writemem )
	{ 0x000000, 0x03ffff, MWA16_ROM },
	{ 0x040000, 0x043fff, COMRAM_w },
	{ 0x060000, 0x063fff, MWA16_RAM },
	{ 0x080000, 0x080fff, twin16_paletteram_word_w, &paletteram16 },
	{ 0x0a0000, 0x0a0001, fround_CPU_register_w },
	{ 0x0a0008, 0x0a0009, sound_command_w },
	{ 0x0a0010, 0x0a0011, MWA16_NOP }, /* watchdog */
	{ 0x0c0000, 0x0c000f, twin16_video_register_w },
	{ 0x0e0000, 0x0e0001, fround_gfx_bank_w },
	{ 0x100000, 0x103fff, MWA16_RAM, &twin16_fixram },
	{ 0x120000, 0x123fff, MWA16_RAM, &videoram16 },
	{ 0x140000, 0x143fff, MWA16_RAM, &spriteram16 },
MEMORY_END

/******************************************************************************************/

#define KONAMI_TWIN_COINAGE PORT_START \
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coin_A ) ) \
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) ) \
	PORT_DIPSETTING(    0x00, DEF_STR( Free_Play ) ) \
	PORT_DIPNAME( 0xf0, 0xf0, DEF_STR( Coin_B ) ) \
	PORT_DIPSETTING(    0x20, DEF_STR( 4C_1C ) ) \
	PORT_DIPSETTING(    0x50, DEF_STR( 3C_1C ) ) \
	PORT_DIPSETTING(    0x80, DEF_STR( 2C_1C ) ) \
	PORT_DIPSETTING(    0x40, DEF_STR( 3C_2C ) ) \
	PORT_DIPSETTING(    0x10, DEF_STR( 4C_3C ) ) \
	PORT_DIPSETTING(    0xf0, DEF_STR( 1C_1C ) ) \
	PORT_DIPSETTING(    0x30, DEF_STR( 3C_4C ) ) \
	PORT_DIPSETTING(    0x70, DEF_STR( 2C_3C ) ) \
	PORT_DIPSETTING(    0xe0, DEF_STR( 1C_2C ) ) \
	PORT_DIPSETTING(    0x60, DEF_STR( 2C_5C ) ) \
	PORT_DIPSETTING(    0xd0, DEF_STR( 1C_3C ) ) \
	PORT_DIPSETTING(    0xc0, DEF_STR( 1C_4C ) ) \
	PORT_DIPSETTING(    0xb0, DEF_STR( 1C_5C ) ) \
	PORT_DIPSETTING(    0xa0, DEF_STR( 1C_6C ) ) \
	PORT_DIPSETTING(    0x90, DEF_STR( 1C_7C ) ) \
	PORT_DIPSETTING(    0x00, "Disabled" )

INPUT_PORTS_START( twin16 ) /* generic */
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( fround )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x03, "Player's Basic Power" )
	PORT_DIPSETTING(    0x03, "18" )
	PORT_DIPSETTING(    0x02, "20" )
	PORT_DIPSETTING(    0x01, "22" )
	PORT_DIPSETTING(    0x00, "24" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x08, 0x08, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x60, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( vulcan )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x18, "20K and every 70K" )
	PORT_DIPSETTING(    0x10, "30K and every 80K" )
	PORT_DIPSETTING(    0x08, "20K only" )
	PORT_DIPSETTING(    0x00, "70K only" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

/* same as vulcan, different bonus */
INPUT_PORTS_START( gradius2 )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "4" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Cocktail ) )
	PORT_DIPNAME( 0x18, 0x18, DEF_STR (Bonus_Life ) )
	PORT_DIPSETTING(    0x18, "20K and every 150K" )
	PORT_DIPSETTING(    0x10, "30K and every 200K" )
	PORT_DIPSETTING(    0x08, "20K only" )
	PORT_DIPSETTING(    0x00, "70K only" )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END

INPUT_PORTS_START( darkadv )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 ) /* "map" button (also advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE1 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 |IPF_PLAYER2 ) /* SERVICE2 */

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 ) /* START1 */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER3 ) /* SERVICE3 */

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 ) /* START2 */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 ) /* START3 */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* Coinage */
	PORT_DIPNAME( 0x0f, 0x0f, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x08, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 3C_2C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 4C_3C ) )
	PORT_DIPSETTING(    0x0f, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 3C_4C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(    0x0e, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 2C_5C ) )
	PORT_DIPSETTING(    0x0d, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(    0x0c, DEF_STR( 1C_4C ) )
	PORT_DIPSETTING(    0x0b, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(    0x0a, DEF_STR( 1C_6C ) )
	PORT_DIPSETTING(    0x09, DEF_STR( 1C_7C ) )
	PORT_DIPSETTING(    0x00, "Invalid" )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Special" )/*#players? */
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0019 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
/*	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED ) */
INPUT_PORTS_END

INPUT_PORTS_START( devilw )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_BUTTON4 ) /* "map" button (also advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON5 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON6 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON7 )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START1 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON3 )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 ) /* also used in-game */
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER3 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START3 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x08, 0x08, "Special" )/*#players? */
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x60, 0x40, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0019 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x00, "Controls" )
	PORT_DIPSETTING(    0x02, "Single" )
	PORT_DIPSETTING(    0x00, "Dual" )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x08, "Reserved" )
	PORT_DIPSETTING(    0x08, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x10, 0x10, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x10, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x20, 0x20, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x20, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x40, 0x40, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x40, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x80, 0x80, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

/*	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED ) */
INPUT_PORTS_END

INPUT_PORTS_START( miaj )
	PORT_START      /* 0xa0001 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_START1 ) /* advances through tests */
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0003 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER1 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0005 */
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_UP    | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN  | IPF_PLAYER2 | IPF_8WAY )
	PORT_BIT( 0x10, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_UNKNOWN )

	PORT_START      /* 0xa0007 */
	PORT_BIT( 0xff, 0xff, IPT_UNUSED )

	KONAMI_TWIN_COINAGE

	PORT_START	/* DSW2 */
	PORT_DIPNAME( 0x03, 0x02, DEF_STR( Lives ) )
	PORT_DIPSETTING(    0x03, "2" )
	PORT_DIPSETTING(    0x02, "3" )
	PORT_DIPSETTING(    0x01, "5" )
	PORT_DIPSETTING(    0x00, "7" )
	PORT_DIPNAME( 0x04, 0x04, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x04, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x18, 0x08, DEF_STR( Bonus_Life ) )
	PORT_DIPSETTING(    0x18, "30K and every 80K" )
	PORT_DIPSETTING(    0x10, "50K and every 100K" )
	PORT_DIPSETTING(    0x08, "50K only" )
	PORT_DIPSETTING(    0x00, "100K only" )
	PORT_DIPNAME( 0x60, 0x60, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0x60, "Easy" )
	PORT_DIPSETTING(    0x40, "Normal" )
	PORT_DIPSETTING(    0x20, "Difficult" )
	PORT_DIPSETTING(    0x00, "Very Difficult" )
	PORT_DIPNAME( 0x80, 0x00, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(    0x80, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )

	PORT_START	/* DSW3 0xa0018 */
	PORT_DIPNAME( 0x01, 0x01, DEF_STR( Flip_Screen ) )
	PORT_DIPSETTING(    0x01, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_DIPNAME( 0x02, 0x02, DEF_STR( Unknown ) )
	PORT_DIPSETTING(    0x02, DEF_STR( Off ) )
	PORT_DIPSETTING(    0x00, DEF_STR( On ) )
	PORT_SERVICE( 0x04, IP_ACTIVE_LOW )
	PORT_DIPNAME( 0x08, 0x00, DEF_STR( Cabinet ) )
	PORT_DIPSETTING(    0x00, DEF_STR( Upright ) )
	PORT_DIPSETTING(    0x08, DEF_STR( Cocktail ) )
	PORT_BIT( 0xf0, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END



static struct GfxLayout alpha_layout =
{
	8,8,
	0x200,
	4,
	{ 0,1,2,3 },
	{ 0*4,1*4,2*4,3*4,4*4,5*4,6*4,7*4 },
	{ 0*32,1*32,2*32,3*32,4*32,5*32,6*32,7*32 },
	8*32
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0x00000, &alpha_layout, 0, 16 },
	{ -1 }
};



static struct YM2151interface ym2151_interface =
{
	1,			/* 1 chip */
	7159160/2,	/* 3.58 MHz ? */
	{ YM3012_VOL(100,MIXER_PAN_LEFT,100,MIXER_PAN_RIGHT) },
	{ 0 }
};

static void volume_callback(int v)
{
	K007232_set_volume(0,0,(v >> 4) * 0x11,0);
	K007232_set_volume(0,1,0,(v & 0x0f) * 0x11);
}

static struct K007232_interface k007232_interface =
{
	1,		/* number of chips */
	{ REGION_SOUND1 }, /* memory regions */
	{ K007232_VOL(20,MIXER_PAN_CENTER,20,MIXER_PAN_CENTER) },	/* volume */
	{ volume_callback }	/* external port callback */
};

static struct UPD7759_interface upd7759_interface =
{
	1,		/* number of chips */
	UPD7759_STANDARD_CLOCK,
	{ 20 }, /* volume */
	{ REGION_SOUND2 }, /* memory region */
	UPD7759_STANDALONE_MODE, /* chip mode */
	{0}
};



#define MACHINE_DRIVER(NAME,NVRAM)				\
static const struct MachineDriver machine_driver_##NAME =	\
{	\
	{	\
		{	\
			CPU_Z80 | CPU_AUDIO_CPU,	\
			3579545,	\
			readmem_sound,writemem_sound,0,0,	\
			ignore_interrupt,1	\
		},	\
		{	\
			CPU_M68000,	\
			10000000,	\
			readmem_sub,writemem_sub,0,0,	\
			CPUB_interrupt,1	\
		},	\
		{	\
			CPU_M68000,	\
			10000000,	\
			readmem,writemem,0,0,	\
			CPUA_interrupt,1	\
		},	\
	},	\
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,	\
	100, /* CPU slices */	\
	0, /* init machine */	\
	\
	/* video hardware */	\
	320, 256, { 0, 319, 0+16, 255-16 },	\
	gfxdecodeinfo,	\
	0x400,0x400,	\
	0,	\
	\
	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,	\
	0,	\
	twin16_vh_start,	\
	twin16_vh_stop,	\
	twin16_vh_screenrefresh,	\
	\
	/* sound hardware */	\
	SOUND_SUPPORTS_STEREO,0,0,0,	\
	{	\
		{	\
			SOUND_YM2151,	\
			&ym2151_interface	\
		},	\
		{	\
			SOUND_K007232,	\
			&k007232_interface,	\
		},	\
		{	\
			SOUND_UPD7759,	\
			&upd7759_interface	\
		}	\
	},	\
	NVRAM	\
};

MACHINE_DRIVER(twin16,0)
MACHINE_DRIVER(cuebrick,cuebrick_nvram_handler)

static const struct MachineDriver machine_driver_heavysync =
{
	{
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			3579545,
			readmem_sound,writemem_sound,0,0,
			ignore_interrupt,1
		},
		{
			CPU_M68000,
			10000000,
			readmem_sub,writemem_sub,0,0,
			CPUB_interrupt,1
		},
		{
			CPU_M68000,
			10000000,
			readmem,writemem,0,0,
			CPUA_interrupt,1
		},
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,
	1000, /* CPU slices */
	0, /* init machine */

	/* video hardware */
	320, 256, { 0, 319, 0+16, 255-16 },
	gfxdecodeinfo,
	0x400,0x400,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,/* | VIDEO_SUPPORTS_16BIT, */
	0,
	twin16_vh_start,
	twin16_vh_stop,
	twin16_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		{
			SOUND_YM2151,
			&ym2151_interface
		},
		{
			SOUND_K007232,
			&k007232_interface,
		},
		{
			SOUND_UPD7759,
			&upd7759_interface
		}
	}
};

static const struct MachineDriver machine_driver_fround =
{
	{
		{
			CPU_Z80 | CPU_AUDIO_CPU,
			3579545,
			readmem_sound,writemem_sound,0,0,
			ignore_interrupt,1
		},
		{
			CPU_M68000,
			10000000,
			fround_readmem,fround_writemem,0,0,
			CPUA_interrupt,1
		},
	},
	60, DEFAULT_REAL_60HZ_VBLANK_DURATION,
	100, /* CPU slices */
	0, /* init machine */

	/* video hardware */
	320, 256, { 0, 319, 0+16, 255-16 },
	gfxdecodeinfo,
	0x400,0x400,
	0,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,/* | VIDEO_SUPPORTS_16BIT, */
	0,
	twin16_vh_start,
	twin16_vh_stop,
	twin16_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		{
			SOUND_YM2151,
			&ym2151_interface
		},
		{
			SOUND_K007232,
			&k007232_interface,
		},
		{
			SOUND_UPD7759,
			&upd7759_interface
		}
	}
};

/******************************************************************************************/

ROM_START( devilw )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "dw-m03.rom",	0x00000,  0x8000, 0x7201983c )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "dw-m14.rom",	0x00000,  0x4000, 0xd7338557 ) /* characters */

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "dw-r07.rom",	0x00000, 0x10000, 0x53110c0b )
	ROM_LOAD16_BYTE( "dw-r06.rom",	0x00001, 0x10000, 0x9c53a0c5 )
	ROM_LOAD16_BYTE( "dw-r13.rom",	0x20000, 0x10000, 0x36ae6014 )
	ROM_LOAD16_BYTE( "dw-r12.rom",	0x20001, 0x10000, 0x6d012167 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "dw-t05.rom",	0x00000, 0x10000, 0x8ab7dc61 )
	ROM_LOAD16_BYTE( "dw-t04.rom",	0x00001, 0x10000, 0xc69924da )
	ROM_LOAD16_BYTE( "dw-t09.rom",   0x20000, 0x10000, 0xfae97de0 )
	ROM_LOAD16_BYTE( "dw-t08.rom",   0x20001, 0x10000, 0x8c898d67 )

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD16_WORD(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD16_WORD(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD16_WORD(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD16_BYTE( "dw-m10.rom",	0x00001, 0x10000, 0x117c91ee )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "dw-ic5a.rom",	0x00000, 0x20000, 0xd4992dfb )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "dw-ic7c.rom",	0x00000, 0x20000, 0xe5947501 )
ROM_END

ROM_START( majuu )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "dw-m03.rom",	0x00000,  0x8000, 0x7201983c )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "dw-r07.rom",	0x00000, 0x10000, 0x53110c0b )
	ROM_LOAD16_BYTE( "dw-r06.rom",	0x00001, 0x10000, 0x9c53a0c5 )
	ROM_LOAD16_BYTE( "dw-r13.rom",	0x20000, 0x10000, 0x36ae6014 )
	ROM_LOAD16_BYTE( "dw-r12.rom",	0x20001, 0x10000, 0x6d012167 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "687-s05.6n",	0x00000, 0x10000, 0xbd99b434 )
	ROM_LOAD16_BYTE( "687-s04.4n",	0x00001, 0x10000, 0x3df732e2 )
	ROM_LOAD16_BYTE( "687-s09.6r",	0x20000, 0x10000, 0x1f6efec3 )
	ROM_LOAD16_BYTE( "687-s08.4r",	0x20001, 0x10000, 0x8a16c8c6 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "687-l14.d8",	0x00000,  0x4000, 0x20ecccd6 ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD16_WORD(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD16_WORD(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD16_WORD(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD16_BYTE( "dw-m10.rom",	0x00001, 0x10000, 0x117c91ee )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "dw-ic5a.rom",	0x00000, 0x20000, 0xd4992dfb )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "dw-ic7c.rom",	0x00000, 0x20000, 0xe5947501 )
ROM_END

ROM_START( darkadv )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "n03.10a",	0x00000,  0x8000, 0xa24c682f )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "n07.10n",	0x00000, 0x10000, 0x6154322a )
	ROM_LOAD16_BYTE( "n06.8n",	0x00001, 0x10000, 0x37a72e8b )
	ROM_LOAD16_BYTE( "n13.10s",	0x20000, 0x10000, 0xf1c252af )
	ROM_LOAD16_BYTE( "n12.8s",	0x20001, 0x10000, 0xda221944 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "n05.6n",	0x00000, 0x10000, 0xa9195b0b )
	ROM_LOAD16_BYTE( "n04.4n",	0x00001, 0x10000, 0x65b55105 )
	ROM_LOAD16_BYTE( "n09.6r",	0x20000, 0x10000, 0x1c6b594c )
	ROM_LOAD16_BYTE( "n08.4r",	0x20001, 0x10000, 0xa9603196 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "n14.3f",	0x0000,  0x4000, 0xc76ac6d2 ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"dw-10p.rom",	0x000000, 0x80000, 0x66cb3923 )
	ROM_LOAD16_WORD(	"dw-10r.rom",	0x080000, 0x80000, 0xa1c7d0db )
	ROM_LOAD16_WORD(	"dw-10l.rom",	0x100000, 0x80000, 0xeec8c5b2 )
	ROM_LOAD16_WORD(	"dw-10m.rom",	0x180000, 0x80000, 0x746cf48b )

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "dw-m11.rom",	0x00000, 0x10000, 0x399deee8 )
	ROM_LOAD16_BYTE( "dw-m10.rom",	0x00001, 0x10000, 0x117c91ee ) /* tiles */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "dw-ic5a.rom",	0x00000, 0x20000, 0xd4992dfb )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "dw-ic7c.rom",	0x00000, 0x20000, 0xe5947501 )
ROM_END

/******************************************************************************************/

ROM_START( cuebrick )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "903-d03.10a",	0x00000,  0x8000, 0x455e855a )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "903-d07.10n",	0x00000, 0x10000, 0xfc0edce7 )
	ROM_LOAD16_BYTE( "903-d06.8n",	0x00001, 0x10000, 0xb2cef6fe )
	ROM_LOAD16_BYTE( "903-e13.10s",	0x20000, 0x10000, 0x4fb5fb80 )
	ROM_LOAD16_BYTE( "903-e12.8s",	0x20001, 0x10000, 0x883e3097 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "903-e05.6n",	0x00000, 0x10000, 0x8b556220 )
	ROM_LOAD16_BYTE( "903-e04.4n",	0x00001, 0x10000, 0xbf9c7927 )
	ROM_LOAD16_BYTE( "903-e09.6r",	0x20000, 0x10000, 0x2a77554d )
	ROM_LOAD16_BYTE( "903-e08.4r",	0x20001, 0x10000, 0xc0a430c1 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "903-e14.d8",	0x0000, 0x4000, 0xddbebbd5 ) /* characters */

	ROM_REGION( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	/* unpopulated */

	ROM_REGION16_BE( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */
	ROM_LOAD16_BYTE( "903-e11.10r",	0x00000, 0x10000, 0x5c41faf8 )
	ROM_LOAD16_BYTE( "903-e10.8r",	0x00001, 0x10000, 0x417576d4 )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	/* unpopulated */

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	/* unpopulated */
ROM_END

ROM_START( vulcan )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "vulcan.g03",	0x00000,  0x8000, 0x67a3b50d )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "vulcan.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD16_BYTE( "vulcan.p06", 0x00001, 0x10000, 0x70c94bee )
	ROM_LOAD16_BYTE( "vulcan.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD16_BYTE( "vulcan.p12", 0x20001, 0x10000, 0x38ea402a )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "vulcan.w05", 0x00000, 0x10000, 0x6e0e99cd )
	ROM_LOAD16_BYTE( "vulcan.w04", 0x00001, 0x10000, 0x23ec74ca )
	ROM_LOAD16_BYTE( "vulcan.w09", 0x20000, 0x10000, 0x377e4f28 )
	ROM_LOAD16_BYTE( "vulcan.w08", 0x20001, 0x10000, 0x813d41ea )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "vulcan.h14",	0x0000, 0x4000, 0x02f4b16f ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "vulcan.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD16_WORD( "vulcan.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD16_WORD( "vulcan.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD16_WORD( "vulcan.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "vulcan.f01",	0x00000, 0x20000, 0xa0d8d69e )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "vulcan.f02",	0x00000, 0x20000, 0xc39f5ca4 )
ROM_END

ROM_START( gradius2 )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "vulcan.g03",	0x00000,  0x8000, 0x67a3b50d )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "vulcan.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD16_BYTE( "vulcan.p06", 0x00001, 0x10000, 0x70c94bee )
	ROM_LOAD16_BYTE( "vulcan.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD16_BYTE( "vulcan.p12", 0x20001, 0x10000, 0x38ea402a )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "785x05.bin", 0x00000, 0x10000, 0x8a23a7b8 )
	ROM_LOAD16_BYTE( "785x04.bin", 0x00001, 0x10000, 0x88e466ce )
	ROM_LOAD16_BYTE( "785x09.bin", 0x20000, 0x10000, 0x3f3d7d7a )
	ROM_LOAD16_BYTE( "785x08.bin", 0x20001, 0x10000, 0xc39c8efd )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "gradius2.g14",	0x0000, 0x4000, 0x9dcdad9d ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "vulcan.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD16_WORD( "vulcan.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD16_WORD( "vulcan.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD16_WORD( "vulcan.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "vulcan.f01",	0x00000, 0x20000, 0xa0d8d69e )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "vulcan.f02",	0x00000, 0x20000, 0xc39f5ca4 )
ROM_END

ROM_START( grdius2a )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "vulcan.g03",	0x00000,  0x8000, 0x67a3b50d )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "vulcan.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD16_BYTE( "vulcan.p06", 0x00001, 0x10000, 0x70c94bee )
	ROM_LOAD16_BYTE( "vulcan.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD16_BYTE( "vulcan.p12", 0x20001, 0x10000, 0x38ea402a )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "gradius2.p05", 0x00000, 0x10000, 0x4db0e736 )
	ROM_LOAD16_BYTE( "gradius2.p04", 0x00001, 0x10000, 0x765b99e6 )
	ROM_LOAD16_BYTE( "785t09.bin",   0x20000, 0x10000, 0x4e3f4965 )
	ROM_LOAD16_BYTE( "gradius2.j08", 0x20001, 0x10000, 0x2b1c9108 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "gradius2.g14",	0x0000, 0x4000, 0x9dcdad9d ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "vulcan.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD16_WORD( "vulcan.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD16_WORD( "vulcan.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD16_WORD( "vulcan.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "vulcan.f01",	0x00000, 0x20000, 0xa0d8d69e )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "vulcan.f02",	0x00000, 0x20000, 0xc39f5ca4 )
ROM_END

ROM_START( grdius2b )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "vulcan.g03",	0x00000,  0x8000, 0x67a3b50d )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "vulcan.p07", 0x00000, 0x10000, 0x686d549d )
	ROM_LOAD16_BYTE( "vulcan.p06", 0x00001, 0x10000, 0x70c94bee )
	ROM_LOAD16_BYTE( "vulcan.p13", 0x20000, 0x10000, 0x478fdb0a )
	ROM_LOAD16_BYTE( "vulcan.p12", 0x20001, 0x10000, 0x38ea402a )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "gradius2.p05", 0x00000, 0x10000, 0x4db0e736 )
	ROM_LOAD16_BYTE( "gradius2.p04", 0x00001, 0x10000, 0x765b99e6 )
	ROM_LOAD16_BYTE( "gradius2.j09", 0x20000, 0x10000, 0x6d96a7e3 )
	ROM_LOAD16_BYTE( "gradius2.j08", 0x20001, 0x10000, 0x2b1c9108 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "gradius2.g14",	0x0000, 0x4000, 0x9dcdad9d ) /* characters */

	ROM_REGION16_BE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD( "vulcan.f17",	0x000000, 0x80000, 0x8fbec1a4 )
	ROM_LOAD16_WORD( "vulcan.f18",	0x080000, 0x80000, 0x50d61e38 )
	ROM_LOAD16_WORD( "vulcan.f15",	0x100000, 0x80000, 0xaf96aef3 )
	ROM_LOAD16_WORD( "vulcan.f16",	0x180000, 0x80000, 0xb858df1f )

	ROM_REGION( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "vulcan.f01",	0x00000, 0x20000, 0xa0d8d69e )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "vulcan.f02",	0x00000, 0x20000, 0xc39f5ca4 )
ROM_END

/******************************************************************************************/

ROM_START( hpuncher )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "870g03.10a",	0x00000,  0x8000, 0xdb9c10c8 )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE( "870h07.10n",	0x00000, 0x10000, 0xb4dda612 )
	ROM_LOAD16_BYTE( "870h06.8n",	0x00001, 0x10000, 0x696ba702 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "870h05.6n", 0x00000, 0x10000, 0x2bcfeef3 )
	ROM_LOAD16_BYTE( "870h04.4n", 0x00001, 0x10000, 0xb9f97fd3 )
	ROM_LOAD16_BYTE( "870h09.6r", 0x20000, 0x10000, 0x96a4f8b1 )
	ROM_LOAD16_BYTE( "870h08.4r", 0x20001, 0x10000, 0x46d65156 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "870f14.d8",	0x0000, 0x4000, 0xc9b46615 ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"870c17.p16",	0x000000, 0x80000, 0x2bc99ff8 )
	ROM_LOAD16_WORD(	"870c18.p18",	0x080000, 0x80000, 0x07927fe8 )
	ROM_LOAD16_WORD(	"870c15.p13",	0x100000, 0x80000, 0x8c9281df )
	ROM_LOAD16_WORD(	"870c16.p15",	0x180000, 0x80000, 0x41df6a1b )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "870c01.5a",	0x00000, 0x20000, 0x6af96546 )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "870c02.7c",	0x00000, 0x20000, 0x54e12c6d )
ROM_END

ROM_START( fround )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "frf03.bin",	0x00000,  0x8000, 0xa645c727 )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "870m21.bin", 0x00000, 0x20000, 0x436dbffb )
	ROM_LOAD16_BYTE( "870m20.bin", 0x00001, 0x20000, 0xb1c79d6a )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "870f14.d8",	0x0000, 0x4000, 0xc9b46615 ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"870c17.p16",	0x080000, 0x80000, 0x2bc99ff8 )
	ROM_LOAD16_WORD(	"870c18.p18",	0x000000, 0x80000, 0x07927fe8 )
	ROM_LOAD16_WORD(	"870c15.p13",	0x180000, 0x80000, 0x8c9281df )
	ROM_LOAD16_WORD(	"870c16.p15",	0x100000, 0x80000, 0x41df6a1b )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "870c01.5a",	0x00000, 0x20000, 0x6af96546 )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "870c02.7c",	0x00000, 0x20000, 0x54e12c6d )
ROM_END

ROM_START( froundl )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "frf03.bin",	0x00000,  0x8000, 0xa645c727 )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE( "frl21.bin", 0x00000, 0x20000, 0xe21a3a19 )
	ROM_LOAD16_BYTE( "frl20.bin", 0x00001, 0x20000, 0x0ce9786f )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "870f14.d8",	0x0000, 0x4000, 0xc9b46615 ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"870c17.p16",	0x080000, 0x80000, 0x2bc99ff8 )
	ROM_LOAD16_WORD(	"870c18.p18",	0x000000, 0x80000, 0x07927fe8 )
	ROM_LOAD16_WORD(	"870c15.p13",	0x180000, 0x80000, 0x8c9281df )
	ROM_LOAD16_WORD(	"870c16.p15",	0x100000, 0x80000, 0x41df6a1b )

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD( "870c01.5a",	0x00000, 0x20000, 0x6af96546 )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
	ROM_LOAD( "870c02.7c",	0x00000, 0x20000, 0x54e12c6d )
ROM_END


/******************************************************************************************/

ROM_START( miaj )
	ROM_REGION( 0x10000, REGION_CPU1, 0 ) /* Z80 code (sound CPU) */
	ROM_LOAD( "808e03.f4",	0x00000,  0x8000, 0x3d93a7cd )

	ROM_REGION( 0x40000, REGION_CPU2, 0 ) /* 68000 code (CPU B) */
	ROM_LOAD16_BYTE(	"808e07.bin",	0x00000, 0x10000, 0x297bdcea )
	ROM_LOAD16_BYTE(	"808e06.bin",	0x00001, 0x10000, 0x8f576b33 )
	ROM_LOAD16_BYTE(	"808e13.h28",	0x20000, 0x10000, 0x1fa708f4 )
	ROM_LOAD16_BYTE(	"808e12.f28",	0x20001, 0x10000, 0xd62f1fde )

	ROM_REGION( 0x40000, REGION_CPU3, 0 ) /* 68000 code (CPU A) */
	ROM_LOAD16_BYTE(	"808r05.bin", 0x00000, 0x10000, 0x91fd83f4 )
	ROM_LOAD16_BYTE(	"808r04.bin", 0x00001, 0x10000, 0xf1c8c597 )
	ROM_LOAD16_BYTE(	"808r09.bin", 0x20000, 0x10000, 0xf74d4467 )
	ROM_LOAD16_BYTE(	"808r08.bin", 0x20001, 0x10000, 0x26f21704 )

	ROM_REGION( 0x4000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD(	"808e14.bin",	0x0000, 0x4000, 0xb9d36525 ) /* characters */

	ROM_REGION16_LE( 0x200000, REGION_GFX2, 0 )	/* gfx data used at runtime */
	ROM_LOAD16_WORD(	"808d17.j4",	0x000000, 0x80000, 0xd1299082 )
	ROM_LOAD16_WORD(	"808d15.h4",	0x100000, 0x80000, 0x2b22a6b6 )

	ROM_REGION( 0x20000, REGION_GFX3, 0 ) /* tile data; mapped at 0x80000 on CPUB */

	ROM_REGION( 0x20000, REGION_SOUND1, 0 ) /* samples */
	ROM_LOAD(	"808d01.d4",	0x00000, 0x20000, 0xfd4d37c0 )

	ROM_REGION( 0x20000, REGION_SOUND2, 0 ) /* samples */
ROM_END

/******************************************************************************************/

static void gfx_untangle( void )
{ /* sprite, tile data */
	int i;
	data16_t *temp = malloc(0x200000);
	if( temp )
	{
		twin16_gfx_rom = (data16_t *)memory_region(REGION_GFX2);
		memcpy( temp, twin16_gfx_rom, 0x200000 );

		for( i=0; i<0x080000; i++ )
		{
			twin16_gfx_rom[i*2+0] = temp[i+0x080000];
			twin16_gfx_rom[i*2+1] = temp[i];
		}
		free( temp );
	}
}

static void init_twin16(void)
{
	gfx_untangle();
	twin16_custom_vidhrdw = 0;
}

static void init_fround(void)
{
	gfx_untangle();
	twin16_custom_vidhrdw = 1;
}


GAME( 1987, devilw,   0,      heavysync, devilw,   twin16, ROT0, "Konami", "Devil World" )
GAME( 1987, majuu,    devilw, heavysync, devilw,   twin16, ROT0, "Konami", "Majuu no Ohkoku" )
GAME( 1987, darkadv,  devilw, heavysync, darkadv,  twin16, ROT0, "Konami", "Dark Adventure" )
GAME( 1988, vulcan,   0,      twin16,    vulcan,   twin16, ROT0, "Konami", "Vulcan Venture" )
GAME( 1988, gradius2, vulcan, twin16,    gradius2, twin16, ROT0, "Konami", "Gradius II - Gofer no Yabou (Japan set 1)" )
GAME( 1988, grdius2a, vulcan, twin16,    gradius2, twin16, ROT0, "Konami", "Gradius II - Gofer no Yabou (Japan set 2)" )
GAME( 1988, grdius2b, vulcan, twin16,    gradius2, twin16, ROT0, "Konami", "Gradius II - Gofer no Yabou (Japan set 3)" )
GAME( 1989, cuebrick, 0,      cuebrick,  twin16,   twin16, ROT0, "Konami", "Cuebrick" )
GAME( 1988, fround,   0,      fround,    fround,   fround, ROT0, "Konami", "Final Round (version M)" )
GAME( 1988, froundl,  fround, fround,    fround,   fround, ROT0, "Konami", "Final Round (version L)" )
GAME( 1988, hpuncher, fround, twin16,    fround,   twin16, ROT0, "Konami", "Hard Puncher (Japan)" )
GAME( 1989, miaj,     mia,    twin16,    miaj,     twin16, ROT0, "Konami", "Missing in Action (Japan)" )
