/***************************************************************************

	Midway MCR-68k system

	Currently implemented:
		* Zwackery (Chip Squeak Deluxe)
		* Xenopohobe (Sounds Good)
		* Spy Hunter 2 (Sounds Good/Turbo Chip Squeak)
		* Blasted (Sounds Good)
		* Arch Rivals
		* Tri-Sports
		* Pigskin 621AD

	Emulation by Bryan McPhail, mish@tendril.co.uk and
	Aaron Giles agiles@pobox.com

****************************************************************************

	Memory map

****************************************************************************

	========================================================================
	CPU #1
	========================================================================
	000000-03FFFF   R     xxxxxxxx xxxxxxxx    Program ROM
	060000-063FFF   R/W   xxxxxxxx xxxxxxxx    Program RAM
	070000-070FFF   R/W   xxxxxxxx xxxxxxxx    Background video RAM
	071000-071FFF   R/W   xxxxxxxx xxxxxxxx    Additional RAM
	080000-080FFF   R/W   xxxxxxxx xxxxxxxx    Sprite RAM
	090000-0900FF     W   xxxxxxxx xxxxxxxx    Palette RAM
	0A0000-0A000F   R/W   xxxxxxxx --------    M6840 I/O
	0B0000-0B0001     W   -------- --------    Watchdog
	0C0000-0C0001     W   -------- xxxxxxxx    Sound I/O
	0D0000-0D0001   R     xxxxxxxx xxxxxxxx    External inputs
	0E0000-0E0001   R     xxxxxxxx xxxxxxxx    External inputs
	0F0000-0F0001   R     xxxxxxxx xxxxxxxx    External inputs
	========================================================================
	Interrupts:
		NMI ???
		INT generated by CTC
	========================================================================
	NOTE: Pigskin and Tri-Sports have similar memory maps; the addresses
			are simply shuffled around
	========================================================================

***************************************************************************/

#include "driver.h"
#include "machine/mcr.h"
#include "sndhrdw/mcr.h"
#include "sndhrdw/williams.h"
#include "vidhrdw/generic.h"



/*************************************
 *
 *	Externals
 *
 *************************************/

extern UINT8 mcr68_sprite_clip;
extern INT8 mcr68_sprite_xoffset;


WRITE16_HANDLER( mcr68_videoram_w );
WRITE16_HANDLER( mcr68_paletteram_w );
void mcr68_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);

WRITE16_HANDLER( zwackery_videoram_w );
WRITE16_HANDLER( zwackery_paletteram_w );
WRITE16_HANDLER( zwackery_spriteram_w );
void zwackery_convert_color_prom(unsigned char *palette, unsigned short *colortable, const unsigned char *color_prom);
void zwackery_vh_screenrefresh(struct osd_bitmap *bitmap, int full_refresh);



/*************************************
 *
 *	Statics
 *
 *************************************/

static data16_t control_word;



/*************************************
 *
 *	Zwackery-specific handlers
 *
 *************************************/

READ16_HANDLER( zwackery_port_2_r )
{
	int result = readinputport(2);
	int wheel = readinputport(5);

	return result | ((wheel >> 2) & 0x3e);
}


static READ16_HANDLER( zwackery_6840_r )
{
	/* Zwackery does a timer test:                          */
	/* It loads $1388 into one of the timers clocked by E   */
	/* Then it sits in a tight loop counting down from $4E4 */
	/*       BTST #$1,($2,A0)                               */
	/*       DBNE D1,*-6                                    */
	/* It expects D1 to end up between 0 and 5; in order to */
	/* make this happen, we must assume that reads from the */
	/* 6840 take 14 additional cycles                       */
	*cpuintf[Machine->drv->cpu[0].cpu_type & ~CPU_FLAGS_MASK].icount -= 14;
	return mcr68_6840_upper_r(offset,0);
}



/*************************************
 *
 *	Xenophobe-specific handlers
 *
 *************************************/

static WRITE16_HANDLER( xenophobe_control_w )
{
	COMBINE_DATA(&control_word);
/*	soundsgood_reset_w(~control_word & 0x0020);*/
	soundsgood_data_w(offset, ((control_word & 0x000f) << 1) | ((control_word & 0x0010) >> 4));
}



/*************************************
 *
 *	Blasted-specific handlers
 *
 *************************************/

static WRITE16_HANDLER( blasted_control_w )
{
	COMBINE_DATA(&control_word);
/*	soundsgood_reset_w(~control_word & 0x0020);*/
	soundsgood_data_w(offset, (control_word >> 8) & 0x1f);
}



/*************************************
 *
 *	Spy Hunter 2-specific handlers
 *
 *************************************/

static READ16_HANDLER( spyhunt2_port_0_r )
{
	int result = readinputport(0);
	int which = (control_word >> 3) & 3;
	int analog = readinputport(3 + which);
	return result | ((soundsgood_status_r(0) & 1) << 5) | (analog << 8);
}


static READ16_HANDLER( spyhunt2_port_1_r )
{
	int result = readinputport(1);
	return result | ((turbocs_status_r(0) & 1) << 7);
}


static WRITE16_HANDLER( spyhunt2_control_w )
{
	COMBINE_DATA(&control_word);

/* 	turbocs_reset_w(~control_word & 0x0080);*/
	turbocs_data_w(offset, (control_word >> 8) & 0x001f);

/*	soundsgood_reset_w(~control_word & 0x2000);*/
	soundsgood_data_w(offset, (control_word >> 8) & 0x001f);
}



/*************************************
 *
 *	Arch Rivals-specific handlers
 *
 *************************************/

static READ16_HANDLER( archrivl_port_1_r )
{
	int joystick = readinputport(3);
	int result = 0;

	/* each axis of the 49-way joystick is mapped like this:*/
	/*      0 8      = neutral                              */
	/*      1        = slightly left/up                     */
	/*      2 3      = middle left/up                       */
	/*      4 5 6 7  = full left/up                         */
	/*      C        = slightly right/down                  */
	/*      A E      = middle right/down                    */
	/*      9 B D F  = full right/down                      */

	if (joystick & 0x0001) result |= 0x0040;
	else if (joystick & 0x0002) result |= 0x0090;

	if (joystick & 0x0004) result |= 0x0004;
	else if (joystick & 0x0008) result |= 0x0009;

	if (joystick & 0x0010) result |= 0x4000;
	else if (joystick & 0x0020) result |= 0x9000;

	if (joystick & 0x0040) result |= 0x0400;
	else if (joystick & 0x0080) result |= 0x0900;

	return result;
}


static WRITE16_HANDLER( archrivl_control_w )
{
	COMBINE_DATA(&control_word);
	williams_cvsd_reset_w(~control_word & 0x0400);
	williams_cvsd_data_w(control_word & 0x3ff);
}



/*************************************
 *
 *	Pigskin-specific handlers
 *
 *************************************/

static UINT8 protection_data[5];
static WRITE16_HANDLER( pigskin_protection_w )
{
	/* ignore upper-byte only */
	if (ACCESSING_LSB)
	{
		/* track the last 5 bytes */
		protection_data[0] = protection_data[1];
		protection_data[1] = protection_data[2];
		protection_data[2] = protection_data[3];
		protection_data[3] = protection_data[4];
		protection_data[4] = data & 0xff;

		logerror("%06X:protection_w=%02X\n", cpu_getpreviouspc(), data & 0xff);
	}
}


static READ16_HANDLER( pigskin_protection_r )
{
	/* based on the last 5 bytes return a value */
	if (protection_data[4] == 0xe3 && protection_data[3] == 0x94)
		return 0x00;	/* must be <= 1 */
	if (protection_data[4] == 0xc7 && protection_data[3] == 0x7b && protection_data[2] == 0x36)
		return 0x00;	/* must be <= 1 */
	if (protection_data[4] == 0xc7 && protection_data[3] == 0x7b)
		return 0x07;	/* must be > 5 */
	if (protection_data[4] == 0xc7 && protection_data[3] == 0x1f && protection_data[2] == 0x03 &&
		protection_data[1] == 0x25 && protection_data[0] == 0x36)
		return 0x00;	/* must be < 3 */

	logerror("Protection read after unrecognized sequence: %02X %02X %02X %02X %02X\n",
			protection_data[0], protection_data[1], protection_data[2], protection_data[3], protection_data[4]);

	return 0x00;
}


static READ16_HANDLER( pigskin_port_1_r )
{
	int joystick = readinputport(3);
	int result = readinputport(1);

	/* see archrivl_port_1_r for 49-way joystick description */

	if (joystick & 0x0001) result |= 0x4000;
	else if (joystick & 0x0002) result |= 0x9000;

	if (joystick & 0x0004) result |= 0x0400;
	else if (joystick & 0x0008) result |= 0x0900;

	return result;
}


static READ16_HANDLER( pigskin_port_2_r )
{
	int joystick = readinputport(3);
	int result = readinputport(2);

	/* see archrivl_port_1_r for 49-way joystick description */

	if (joystick & 0x0010) result |= 0x4000;
	else if (joystick & 0x0020) result |= 0x9000;

	if (joystick & 0x0040) result |= 0x0400;
	else if (joystick & 0x0080) result |= 0x0900;

	return result;
}



/*************************************
 *
 *	Tri-Sports-specific handlers
 *
 *************************************/

static READ16_HANDLER( trisport_port_1_r )
{
	int xaxis = (INT8)readinputport(3);
	int yaxis = (INT8)readinputport(4);
	int result = readinputport(1);

	result |= (xaxis & 0x3c) << 6;
	result |= (yaxis & 0x3c) << 10;

	return result;
}



/*************************************
 *
 *	Main CPU memory handlers
 *
 *************************************/

static MEMORY_READ16_START( mcr68_readmem )
	{ 0x000000, 0x03ffff, MRA16_ROM },
	{ 0x060000, 0x063fff, MRA16_RAM },
	{ 0x070000, 0x070fff, MRA16_RAM },
	{ 0x071000, 0x071fff, MRA16_RAM },
	{ 0x080000, 0x080fff, MRA16_RAM },
	{ 0x0a0000, 0x0a000f, mcr68_6840_upper_r },
	{ 0x0d0000, 0x0dffff, input_port_0_word_r },
	{ 0x0e0000, 0x0effff, input_port_1_word_r },
	{ 0x0f0000, 0x0fffff, input_port_2_word_r },
MEMORY_END


static MEMORY_WRITE16_START( mcr68_writemem )
	{ 0x000000, 0x03ffff, MWA16_ROM },
	{ 0x060000, 0x063fff, MWA16_RAM },
	{ 0x070000, 0x070fff, mcr68_videoram_w, &videoram16, &videoram_size },
	{ 0x071000, 0x071fff, MWA16_RAM },
	{ 0x080000, 0x080fff, MWA16_RAM, &spriteram16, &spriteram_size },
	{ 0x090000, 0x09007f, mcr68_paletteram_w, &paletteram16 },
	{ 0x0a0000, 0x0a000f, mcr68_6840_upper_w },
	{ 0x0b0000, 0x0bffff, watchdog_reset16_w },
MEMORY_END



/*************************************
 *
 *	Zwackery main CPU memory handlers
 *
 *************************************/

static MEMORY_READ16_START( zwackery_readmem )
	{ 0x000000, 0x037fff, MRA16_ROM },
	{ 0x080000, 0x080fff, MRA16_RAM },
	{ 0x084000, 0x084fff, MRA16_RAM },
	{ 0x100000, 0x10000f, zwackery_6840_r },
	{ 0x104000, 0x104007, pia_2_msb_r },
	{ 0x108000, 0x108007, pia_3_lsb_r },
	{ 0x10c000, 0x10c007, pia_4_lsb_r },
	{ 0x800000, 0x800fff, MRA16_RAM },
	{ 0x802000, 0x803fff, MRA16_RAM },
	{ 0xc00000, 0xc00fff, MRA16_RAM },
MEMORY_END


static MEMORY_WRITE16_START( zwackery_writemem )
	{ 0x000000, 0x037fff, MWA16_ROM },
	{ 0x080000, 0x080fff, MWA16_RAM },
	{ 0x084000, 0x084fff, MWA16_RAM },
	{ 0x100000, 0x10000f, mcr68_6840_upper_w },
	{ 0x104000, 0x104007, pia_2_msb_w },
	{ 0x108000, 0x108007, pia_3_lsb_w },
	{ 0x10c000, 0x10c007, pia_4_lsb_w },
	{ 0x800000, 0x800fff, zwackery_videoram_w, &videoram16, &videoram_size },
	{ 0x802000, 0x803fff, zwackery_paletteram_w, &paletteram16 },
	{ 0xc00000, 0xc00fff, zwackery_spriteram_w, &spriteram16, &spriteram_size },
MEMORY_END



/*************************************
 *
 *	Pigskin CPU memory handlers
 *
 *************************************/

static MEMORY_READ16_START( pigskin_readmem )
	{ 0x000000, 0x03ffff, MRA16_ROM },
	{ 0x080000, 0x08ffff, pigskin_port_1_r },
	{ 0x0a0000, 0x0affff, pigskin_port_2_r },
	{ 0x100000, 0x100fff, MRA16_RAM },
	{ 0x120000, 0x120001, pigskin_protection_r },
	{ 0x140000, 0x143fff, MRA16_RAM },
	{ 0x160000, 0x1607ff, MRA16_RAM },
	{ 0x180000, 0x18000f, mcr68_6840_upper_r },
	{ 0x1e0000, 0x1effff, input_port_0_word_r },
MEMORY_END


static MEMORY_WRITE16_START( pigskin_writemem )
	{ 0x000000, 0x03ffff, MWA16_ROM },
	{ 0x0c0000, 0x0c007f, mcr68_paletteram_w, &paletteram16 },
	{ 0x0e0000, 0x0effff, watchdog_reset16_w },
	{ 0x100000, 0x100fff, mcr68_videoram_w, &videoram16, &videoram_size },
	{ 0x120000, 0x120001, pigskin_protection_w },
	{ 0x140000, 0x143fff, MWA16_RAM },
	{ 0x160000, 0x1607ff, MWA16_RAM, &spriteram16, &spriteram_size },
	{ 0x180000, 0x18000f, mcr68_6840_upper_w },
	{ 0x1a0000, 0x1affff, archrivl_control_w },
MEMORY_END



/*************************************
 *
 *	Tri-Sports CPU memory handlers
 *
 *************************************/

static MEMORY_READ16_START( trisport_readmem )
	{ 0x000000, 0x03ffff, MRA16_ROM },
	{ 0x080000, 0x08ffff, trisport_port_1_r },
	{ 0x0a0000, 0x0affff, input_port_2_word_r },
	{ 0x100000, 0x103fff, MRA16_RAM },
	{ 0x140000, 0x1407ff, MRA16_RAM },
	{ 0x160000, 0x160fff, MRA16_RAM },
	{ 0x180000, 0x18000f, mcr68_6840_upper_r },
	{ 0x1e0000, 0x1effff, input_port_0_word_r },
MEMORY_END


static MEMORY_WRITE16_START( trisport_writemem )
	{ 0x000000, 0x03ffff, MWA16_ROM },
	{ 0x100000, 0x103fff, MWA16_RAM },
	{ 0x120000, 0x12007f, mcr68_paletteram_w, &paletteram16 },
	{ 0x140000, 0x1407ff, MWA16_RAM, &spriteram16, &spriteram_size },
	{ 0x160000, 0x160fff, mcr68_videoram_w, &videoram16, &videoram_size },
	{ 0x180000, 0x18000f, mcr68_6840_upper_w },
	{ 0x1a0000, 0x1affff, archrivl_control_w },
	{ 0x1c0000, 0x1cffff, watchdog_reset16_w },
MEMORY_END



/*************************************
 *
 *	Port definitions
 *
 *************************************/

INPUT_PORTS_START( zwackery )
	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_SERVICE( 0x0010, IP_ACTIVE_LOW )
	PORT_BIT( 0x20, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON1 )	/* sword */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY )
	PORT_BIT( 0x02, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY )
	PORT_BIT( 0x04, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY )
	PORT_BIT( 0x08, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY )
	PORT_BIT( 0xf0, IP_ACTIVE_HIGH, IPT_UNUSED )	/* sound communications */

	PORT_START
	PORT_BIT( 0x01, IP_ACTIVE_LOW, IPT_BUTTON3 )	/* spell up */
	PORT_BIT( 0x3e, IP_ACTIVE_HIGH, IPT_UNUSED )	/* encoder wheel */
	PORT_BIT( 0x40, IP_ACTIVE_LOW, IPT_BUTTON2 )	/* shield */
	PORT_BIT( 0x80, IP_ACTIVE_LOW, IPT_BUTTON4 )	/* spell down */

	PORT_START
	PORT_BIT( 0xff, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x07, 0x00, DEF_STR( Coinage ) )
	PORT_DIPSETTING(    0x05, DEF_STR( 6C_1C ) )
	PORT_DIPSETTING(    0x04, DEF_STR( 5C_1C ) )
	PORT_DIPSETTING(    0x03, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(    0x02, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(    0x01, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(    0x00, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(    0x06, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(    0x07, DEF_STR( Free_Play ) )
	PORT_DIPNAME( 0x38, 0x00, "Buy-in" )
	PORT_DIPSETTING(    0x00, "1 coin" )
	PORT_DIPSETTING(    0x08, "2 coins" )
	PORT_DIPSETTING(    0x10, "3 coins" )
	PORT_DIPSETTING(    0x18, "4 coins" )
	PORT_DIPSETTING(    0x20, "5 coins" )
	PORT_DIPSETTING(    0x28, "6 coins" )
	PORT_DIPSETTING(    0x30, "7 coins" )
	PORT_DIPSETTING(    0x38, "None" )
	PORT_DIPNAME( 0xc0, 0x00, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(    0xc0, "Easier" )
	PORT_DIPSETTING(    0x00, "Normal" )
	PORT_DIPSETTING(    0x40, "Harder" )
	PORT_DIPSETTING(    0x80, "Hardest" )

	PORT_START
	PORT_ANALOGX( 0xff, 0x00, IPT_DIAL | IPF_REVERSE, 50, 10, 0, 0, KEYCODE_Z, KEYCODE_X, 0, 0 )
INPUT_PORTS_END


INPUT_PORTS_START( xenophob )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_COIN3 )
	PORT_BIT( 0x0018, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER3 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER3 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER3 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER3 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0003, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0008, 0x0008, "Coins per Life Unit" )
	PORT_DIPSETTING(      0x0008, "1" )
	PORT_DIPSETTING(      0x0000, "2" )
	PORT_DIPNAME( 0x0010, 0x0010, "Life Unit" )
	PORT_DIPSETTING(      0x0010, "1000" )
	PORT_DIPSETTING(      0x0000, "2000" )
	PORT_DIPNAME( 0x0020, 0x0020, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0020, DEF_STR( On ) )
	PORT_DIPNAME( 0x00c0, 0x0000, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0040, "Easy" )
	PORT_DIPSETTING(      0x0000, "Medium" )
	PORT_DIPSETTING(      0x0080, "Hard" )
/*	PORT_DIPSETTING(      0x00c0, "Medium" )*/
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( spyhunt2 )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_UNKNOWN ) /* SG status */
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_SERVICE )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNKNOWN )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER2 ) /* 1st gear */
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER2 ) /* 2nd gear */
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_BUTTON7 | IPF_PLAYER2 ) /* 3rd gear */
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON5 | IPF_PLAYER1 ) /* 1st gear */
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON6 | IPF_PLAYER1 ) /* 2nd gear */
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON7 | IPF_PLAYER1 ) /* 3rd gear */
	PORT_BIT( 0x0080, IP_ACTIVE_HIGH, IPT_UNUSED )               /* TCS status */
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 ) /* Left Trigger */
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 ) /* Left Button */
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 ) /* Right Trigger */
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER2 ) /* Right Button */
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 ) /* Left Trigger */
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 ) /* Left Button */
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 ) /* Right Trigger */
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON4 | IPF_PLAYER1 ) /* Right Button */

	PORT_START	/* IN3 -- dipswitches */
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_2C ) )
	/*PORT_DIPSETTING(      0x0000, DEF_STR( 1C_2C ) )*/
	PORT_DIPNAME( 0x0004, 0x0004, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0018, 0x0008, "Point Threshholds" )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x0018, "Medium" )
	PORT_DIPSETTING(      0x0010, "Hard" )
	PORT_DIPSETTING(      0x0000, "Hardest" )
	PORT_DIPNAME( 0x0060, 0x0060, "Free Timer After" )
	PORT_DIPSETTING(      0x0000, "30 sec" )
	PORT_DIPSETTING(      0x0040, "45 sec" )
	PORT_DIPSETTING(      0x0060, "60 sec" )
	PORT_DIPSETTING(      0x0020, "90 sec" )
	PORT_BITX( 0x0080,    0x0080, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Advance", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* analog ports for steering and pedals */
	PORT_ANALOG( 0xff, 0x30, IPT_PEDAL | IPF_PLAYER2 | IPF_REVERSE, 100, 10, 0x30, 0xff )

	PORT_START
	PORT_ANALOG( 0xff, 0x30, IPT_PEDAL | IPF_PLAYER1 | IPF_REVERSE, 100, 10, 0x30, 0xff )

	PORT_START
	PORT_ANALOG( 0xff, 0x80, IPT_PADDLE | IPF_PLAYER2 | IPF_REVERSE, 80, 10, 0x10, 0xf0 )

	PORT_START
	PORT_ANALOG( 0xff, 0x80, IPT_PADDLE | IPF_PLAYER1 | IPF_REVERSE, 80, 10, 0x10, 0xf0 )
INPUT_PORTS_END


INPUT_PORTS_START( blasted )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_UNUSED/* credit w/bill */ )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_UNUSED/* credit 1 w/bill */ )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_UNUSED/* credit 2 w/bill */ )
	PORT_BIT( 0xf000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_DIPNAME( 0x0003, 0x0003, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( 1C_2C ) )
	/*PORT_DIPSETTING(      0x0000, DEF_STR( 1C_2C ) )*/
	PORT_DIPNAME( 0x000c, 0x0000, DEF_STR( Difficulty ) )
	PORT_DIPSETTING(      0x0008, "Easy" )
	PORT_DIPSETTING(      0x0000, "Medium" )
	PORT_DIPSETTING(      0x0004, "Hard" )
/*	PORT_DIPSETTING(      0x000c, "Medium" )*/
	PORT_DIPNAME( 0x0020, 0x0020, "Dollar Receptor" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_BITX( 0x0080,    0x0080, IPT_DIPSWITCH_NAME | IPF_CHEAT, "Rack Advance", KEYCODE_F1, IP_JOY_NONE )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )
INPUT_PORTS_END


INPUT_PORTS_START( archrivl )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_SERVICE( 0x0080, IP_ACTIVE_LOW )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0c00, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )

	PORT_START
	PORT_BIT( 0xffff, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 1/2 joysticks go here */

	PORT_START
	PORT_DIPNAME( 0x0003, 0x0003, "Game Time" )
	PORT_DIPSETTING(      0x0003, "Preset Time" )
	PORT_DIPSETTING(      0x0002, "Preset + 10sec" )
	PORT_DIPSETTING(      0x0001, "Preset + 20sec" )
	PORT_DIPSETTING(      0x0000, "Preset + 30sec" )
	PORT_DIPNAME( 0x001c, 0x001c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0014, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0018, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x001c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( 2C_3C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_5C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( 1C_6C ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Team Names" )
	PORT_DIPSETTING(      0x0020, "Default" )
	PORT_DIPSETTING(      0x0000, "Hometown Heroes" )
	PORT_DIPNAME( 0x0040, 0x0040, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0040, DEF_STR( On ) )
	PORT_DIPNAME( 0x0080, 0x0080, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0080, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START	/* 49-way joystick simulator */
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
INPUT_PORTS_END

INPUT_PORTS_START( pigskin )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x0004, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x0008, IP_ACTIVE_LOW, IPT_START2 )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0100, IP_ACTIVE_LOW, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0200, IP_ACTIVE_LOW, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0400, IP_ACTIVE_LOW, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0800, IP_ACTIVE_LOW, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0x4000, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER1 )
	PORT_BIT( 0x8000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x000f, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_BUTTON3 | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 1 joystick goes here */

	PORT_START
	PORT_DIPNAME( 0x0003, 0x0003, "Game Time" )
	PORT_DIPSETTING(      0x0000, "Shortest" )
	PORT_DIPSETTING(      0x0002, "Short" )
	PORT_DIPSETTING(      0x0003, "Medium" )
	PORT_DIPSETTING(      0x0001, "Long" )
	PORT_DIPNAME( 0x000c, 0x000c, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0008, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x000c, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0004, "Set Your Own" )
	PORT_DIPNAME( 0x0010, 0x0010, DEF_STR( Demo_Sounds ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0010, DEF_STR( On ) )
	PORT_DIPNAME( 0x0020, 0x0020, "Test Switch" )
	PORT_DIPSETTING(      0x0020, DEF_STR( Off ) )
	PORT_DIPSETTING(      0x0000, DEF_STR( On ) )
	PORT_DIPNAME( 0x0040, 0x0040, "Coin Chutes" )
	PORT_DIPSETTING(      0x0000, "Individual" )
	PORT_DIPSETTING(      0x0040, "Common" )
	PORT_DIPNAME( 0x0080, 0x0080, "Joystick" )
	PORT_DIPSETTING(      0x0080, "Standard" )
	PORT_DIPSETTING(      0x0000, "Rotated" )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* player 2 joystick goes here */

	PORT_START	/* 49-way joystick simulator */
	PORT_BIT( 0x0001, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0002, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0004, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0008, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER1 )
	PORT_BIT( 0x0010, IP_ACTIVE_HIGH, IPT_JOYSTICK_RIGHT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0020, IP_ACTIVE_HIGH, IPT_JOYSTICK_LEFT | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0040, IP_ACTIVE_HIGH, IPT_JOYSTICK_UP | IPF_8WAY | IPF_PLAYER2 )
	PORT_BIT( 0x0080, IP_ACTIVE_HIGH, IPT_JOYSTICK_DOWN | IPF_8WAY | IPF_PLAYER2 )
INPUT_PORTS_END


INPUT_PORTS_START( trisport )
	PORT_START
	PORT_BIT( 0x0001, IP_ACTIVE_LOW, IPT_COIN1 )
	PORT_BIT( 0x0002, IP_ACTIVE_LOW, IPT_COIN2 )
	PORT_BIT( 0x000c, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x0010, IP_ACTIVE_LOW, IPT_TILT )
	PORT_SERVICE( 0x0020, IP_ACTIVE_LOW )
	PORT_BIT( 0x0040, IP_ACTIVE_LOW, IPT_COIN4 )
	PORT_BIT( 0x0080, IP_ACTIVE_LOW, IPT_UNKNOWN )
	PORT_BIT( 0x0f00, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_BUTTON1 | IPF_PLAYER1 )
	PORT_BIT( 0x1000, IP_ACTIVE_LOW, IPT_START1 )
	PORT_BIT( 0x2000, IP_ACTIVE_LOW, IPT_BUTTON2 | IPF_PLAYER1 )
	PORT_BIT( 0xc000, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
	PORT_BIT( 0x00ff, IP_ACTIVE_LOW, IPT_UNUSED )
	PORT_BIT( 0xff00, IP_ACTIVE_HIGH, IPT_UNUSED )	/* analog controls go here */

	PORT_START
	PORT_DIPNAME( 0x0007, 0x0007, DEF_STR( Coinage ) )
	PORT_DIPSETTING(      0x0002, DEF_STR( 4C_1C ) )
	PORT_DIPSETTING(      0x0003, DEF_STR( 3C_1C ) )
	PORT_DIPSETTING(      0x0006, DEF_STR( 2C_1C ) )
	PORT_DIPSETTING(      0x0007, DEF_STR( 1C_1C ) )
	PORT_DIPSETTING(      0x0005, DEF_STR( 1C_2C ) )
	PORT_DIPSETTING(      0x0004, DEF_STR( 1C_3C ) )
	PORT_DIPSETTING(      0x0001, DEF_STR( Free_Play ) )
	PORT_DIPSETTING(      0x0000, "Battery Options" )
	PORT_DIPNAME( 0x0018, 0x0018, "Pool Turns" )
	PORT_DIPSETTING(      0x0010, "5" )
	PORT_DIPSETTING(      0x0008, "6" )
	PORT_DIPSETTING(      0x0018, "7" )
	PORT_DIPSETTING(      0x0000, "8" )
	PORT_DIPNAME( 0x0020, 0x0020, "Bowling Difficulty" )
	PORT_DIPSETTING(      0x0020, "Standard" )
	PORT_DIPSETTING(      0x0000, "Advanced" )
	PORT_DIPNAME( 0x0040, 0x0040, "Shot Timer" )
	PORT_DIPSETTING(      0x0000, "Slower" )
	PORT_DIPSETTING(      0x0040, "Standard" )
	PORT_DIPNAME( 0x0080, 0x0080, "Golf Holes" )
	PORT_DIPSETTING(      0x0080, "3" )
	PORT_DIPSETTING(      0x0000, "4" )
	PORT_BIT( 0xff00, IP_ACTIVE_LOW, IPT_UNUSED )

	PORT_START
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_X | IPF_PLAYER1, 100, 20, 0, 0 )

	PORT_START
    PORT_ANALOG( 0xff, 0, IPT_TRACKBALL_Y | IPF_REVERSE | IPF_PLAYER1, 100, 20, 0, 0 )
INPUT_PORTS_END



/*************************************
 *
 *	Graphics definitions
 *
 *************************************/

static struct GfxLayout zwackery_layout =
{
	16,16,
	RGN_FRAC(1,2),
	8,
	{ 0, 0, 0, 0, 0, 0, 0, 0 },
	{  3,  2,  1,  0, 11, 10,  9,  8,
	  19, 18, 17, 16, 27, 26, 25, 24 },
	{ 4, RGN_FRAC(1,2)+4, 0, RGN_FRAC(1,2)+0, 36, RGN_FRAC(1,2)+36, 32, RGN_FRAC(1,2)+32,
	  68, RGN_FRAC(1,2)+68, 64, RGN_FRAC(1,2)+64, 100, RGN_FRAC(1,2)+100, 96, RGN_FRAC(1,2)+96 },
	128
};

static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &mcr_bg_layout,     0, 4 },
	{ REGION_GFX2, 0, &mcr_sprite_layout, 0, 4 },
	{ -1 } /* end of array */
};

static struct GfxDecodeInfo zwackery_gfxdecodeinfo[] =
{
	{ REGION_GFX1, 0, &zwackery_layout,       0, 16 },
	{ REGION_GFX2, 0, &mcr_sprite_layout, 0x800, 32 },
	{ REGION_GFX1, 0, &zwackery_layout,       0, 16 },	/* yes, an extra copy */
	{ -1 } /* end of array */
};



/*************************************
 *
 *	Machine drivers
 *
 *************************************/

/*=================================================================

	Timing for these games is crucial. They all use the 6840 timer
	to preemptively multitask during each frame. The clock on the
	6840 timer is taken from the 68000's E clock, which runs at
	1/10th the speed of the 68000 itself.

	All the games run in a sequence of six steps per frame, using
	counter 1 on the 6840 to time each step. The sum total of the
	6 programmed steps for each game determines how many E clocks
	should be generated per frame, which in turn determines the
	clock we should expect the CPU to have.

	Ideal CPU timings, based on counter usage:

		Zwackery:     7652400
		Xenophobe:    7723800
		Spy Hunter 2: 7723800
		Blasted:      7798800
		Arch Rivals:  7799100
		Pigskin:      9211200
		Tri-Sports:   9211200

	Currently, we are using the Xenophobe CPU for the first four
	until we spot problems.

=================================================================*/

static const struct MachineDriver machine_driver_zwackery =
{
	/* basic machine hardware */
	{
		{
			CPU_M68000,
			7652400,	/* 8 MHz */
			zwackery_readmem,zwackery_writemem,0,0,
			mcr68_interrupt,1
		},
		SOUND_CPU_CHIP_SQUEAK_DELUXE
	},
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,
	1,
	zwackery_init_machine,

	/* video hardware */
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },
	zwackery_gfxdecodeinfo,
	4096, 4096,
	zwackery_convert_color_prom,

	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,
	0,
	generic_vh_start,
	generic_vh_stop,
	zwackery_vh_screenrefresh,

	/* sound hardware */
	SOUND_SUPPORTS_STEREO,0,0,0,
	{
		SOUND_CHIP_SQUEAK_DELUXE
	},
	0
};


#define MACHINE_DRIVER_MCR68(NAME, MEMMAP, SOUND) 		\
static const struct MachineDriver machine_driver_##NAME =		\
{														\
	/* basic machine hardware */						\
	{													\
		{												\
			CPU_M68000,									\
			7723800,	/* 8 MHz */						\
			MEMMAP##_readmem,MEMMAP##_writemem,0,0,		\
			mcr68_interrupt,1							\
		},												\
		SOUND_CPU_##SOUND								\
	},													\
	30, DEFAULT_REAL_30HZ_VBLANK_DURATION,				\
	1,													\
	mcr68_init_machine,									\
														\
	/* video hardware */								\
	32*16, 30*16, { 0, 32*16-1, 0, 30*16-1 },			\
	gfxdecodeinfo,										\
	8*16, 8*16,											\
	0,													\
														\
	VIDEO_TYPE_RASTER | VIDEO_MODIFIES_PALETTE,			\
	0,													\
	generic_vh_start,									\
	generic_vh_stop,									\
	mcr68_vh_screenrefresh,								\
														\
	/* sound hardware */								\
	SOUND_SUPPORTS_STEREO,0,0,0,						\
	{													\
		SOUND_##SOUND									\
	},													\
	0													\
};

MACHINE_DRIVER_MCR68(xenophob, mcr68,    SOUNDS_GOOD)
MACHINE_DRIVER_MCR68(spyhunt2, mcr68,    TURBO_CHIP_SQUEAK_PLUS_SOUNDS_GOOD)
MACHINE_DRIVER_MCR68(archrivl, mcr68,    WILLIAMS_CVSD)
MACHINE_DRIVER_MCR68(pigskin,  pigskin,  WILLIAMS_CVSD)
MACHINE_DRIVER_MCR68(trisport, trisport, WILLIAMS_CVSD)



/*************************************
 *
 *	ROM decoding
 *
 *************************************/

static void rom_decode(void)
{
	int i;

	/* tile graphics are inverted */
	for (i = 0; i < memory_region_length(REGION_GFX1); i++)
		memory_region(REGION_GFX1)[i] ^= 0xff;
}



/*************************************
 *
 *	ROM definitions
 *
 *************************************/

ROM_START( zwackery )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "pro0.bin",   0x00000, 0x4000, 0x6fb9731c )
	ROM_LOAD16_BYTE( "pro1.bin",   0x00001, 0x4000, 0x84b92555 )
	ROM_LOAD16_BYTE( "pro2.bin",   0x08000, 0x4000, 0xe6977a2a )
	ROM_LOAD16_BYTE( "pro3.bin",   0x08001, 0x4000, 0xf5d0a60e )
	ROM_LOAD16_BYTE( "pro4.bin",   0x10000, 0x4000, 0xec5841d9 )
	ROM_LOAD16_BYTE( "pro5.bin",   0x10001, 0x4000, 0xd7d99ce0 )
	ROM_LOAD16_BYTE( "pro6.bin",   0x18000, 0x4000, 0xb9fe7bf5 )
	ROM_LOAD16_BYTE( "pro7.bin",   0x18001, 0x4000, 0x5e261b3b )
	ROM_LOAD16_BYTE( "pro8.bin",   0x20000, 0x4000, 0x55e380a5 )
	ROM_LOAD16_BYTE( "pro9.bin",   0x20001, 0x4000, 0x12249dca )
	ROM_LOAD16_BYTE( "pro10.bin",  0x28000, 0x4000, 0x6a39a8ca )
	ROM_LOAD16_BYTE( "pro11.bin",  0x28001, 0x4000, 0xad6b45bc )
	ROM_LOAD16_BYTE( "pro12.bin",  0x30000, 0x4000, 0xe2d25e1f )
	ROM_LOAD16_BYTE( "pro13.bin",  0x30001, 0x4000, 0xe131f9b8 )

	ROM_REGION( 0x20000, REGION_CPU2, 0 )
	ROM_LOAD16_BYTE( "csd7.bin",  0x00000, 0x2000, 0x5501f54b )
	ROM_LOAD16_BYTE( "csd17.bin", 0x00001, 0x2000, 0x2e482580 )
	ROM_LOAD16_BYTE( "csd8.bin",  0x04000, 0x2000, 0x13366575 )
	ROM_LOAD16_BYTE( "csd18.bin", 0x04001, 0x2000, 0xbcfe5820 )

	ROM_REGION( 0x8000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "tileh.bin",    0x00000, 0x4000, 0xa7237eb1 )
	ROM_LOAD( "tileg.bin",    0x04000, 0x4000, 0x626cc69b )

	ROM_REGION( 0x20000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "spr6h.bin",    0x00000, 0x4000, 0xa51158dc )
	ROM_LOAD( "spr7h.bin",    0x04000, 0x4000, 0x941feecf )
	ROM_LOAD( "spr6j.bin",    0x08000, 0x4000, 0xf3eef316 )
	ROM_LOAD( "spr7j.bin",    0x0c000, 0x4000, 0xa8a34033 )
	ROM_LOAD( "spr10h.bin",   0x10000, 0x4000, 0xa99daea6 )
	ROM_LOAD( "spr11h.bin",   0x14000, 0x4000, 0xc1a767fb )
	ROM_LOAD( "spr10j.bin",   0x18000, 0x4000, 0x4dd04376 )
	ROM_LOAD( "spr11j.bin",   0x1c000, 0x4000, 0xe8c6a880 )

	ROM_REGION( 0x8000, REGION_GFX3, 0 )	/* bg color maps */
	ROM_LOAD16_BYTE( "tilef.bin",  0x0000, 0x4000, 0xa0dfcd7e )
	ROM_LOAD16_BYTE( "tilee.bin",  0x0001, 0x4000, 0xab504dc8 )
ROM_END

ROM_START( xenophob )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "xeno_pro.3c",  0x00000, 0x10000, 0xf44c2e60 )
	ROM_LOAD16_BYTE( "xeno_pro.3b",  0x00001, 0x10000, 0x01609a3b )
	ROM_LOAD16_BYTE( "xeno_pro.2c",  0x20000, 0x10000, 0xe45bf669 )
	ROM_LOAD16_BYTE( "xeno_pro.2b",  0x20001, 0x10000, 0xda5d39d5 )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "xeno_snd.u7",  0x00000, 0x10000, 0x77561d15 )
	ROM_LOAD16_BYTE( "xeno_snd.u17", 0x00001, 0x10000, 0x837a1a71 )
	ROM_LOAD16_BYTE( "xeno_snd.u8",  0x20000, 0x10000, 0x6e2915c7 )
	ROM_LOAD16_BYTE( "xeno_snd.u18", 0x20001, 0x10000, 0x12492145 )

	ROM_REGION( 0x10000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "xeno_bg.11d",  0x00000, 0x08000, 0x3d2cf284 )
	ROM_LOAD( "xeno_bg.12d",  0x08000, 0x08000, 0xc32288b1 )

	ROM_REGION( 0x40000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "xeno_fg.7j",   0x00000, 0x10000, 0xb12eddb2 )
	ROM_LOAD( "xeno_fg.8j",   0x10000, 0x10000, 0x20e682f5 )
	ROM_LOAD( "xeno_fg.9j",   0x20000, 0x10000, 0x82fb3e09 )
	ROM_LOAD( "xeno_fg.10j",  0x30000, 0x10000, 0x6a7a3516 )
ROM_END

ROM_START( spyhunt2 )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "sh23c.bin",  0x00000, 0x10000, 0x30b91c90 )
	ROM_LOAD16_BYTE( "sh23b.bin",  0x00001, 0x10000, 0xf64513c6 )
	ROM_LOAD16_BYTE( "sh22c.bin",  0x20000, 0x10000, 0x8ee65009 )
	ROM_LOAD16_BYTE( "sh22b.bin",  0x20001, 0x10000, 0x850c21ad )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbo-cs.u5", 0x08000, 0x4000, 0x4b1d8a66 )
	ROM_LOAD( "turbo-cs.u4", 0x0c000, 0x4000, 0x3722ce48 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "sh2u7.bin",  0x00000, 0x10000, 0x02362ea4 )
	ROM_LOAD16_BYTE( "sh2u17.bin", 0x00001, 0x10000, 0xe29a2c37 )

	ROM_REGION( 0x10000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "sh2bg0.bin",  0x00000, 0x08000, 0xcb3c3d8e )
	ROM_LOAD( "sh2bg1.bin",  0x08000, 0x08000, 0x029d4af1 )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg0.7j",   0x00000, 0x20000, 0x55ce12ea )
	ROM_LOAD( "fg1.8j",   0x20000, 0x20000, 0x692afb67 )
	ROM_LOAD( "fg2.9j",   0x40000, 0x20000, 0xf1aba383 )
	ROM_LOAD( "fg3.10j",  0x60000, 0x20000, 0xd3475ff8 )
ROM_END

ROM_START( spyhnt2a )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "3c",  0x00000, 0x10000, 0x5b92aadf )
	ROM_LOAD16_BYTE( "3b",  0x00001, 0x10000, 0x6ed0a25f )
	ROM_LOAD16_BYTE( "2c",  0x20000, 0x10000, 0xbc834f3f )
	ROM_LOAD16_BYTE( "2b",  0x20001, 0x10000, 0x8a9f7ef3 )

	ROM_REGION( 0x10000, REGION_CPU2, 0 )  /* 64k for the Turbo Cheap Squeak */
	ROM_LOAD( "turbo-cs.u5", 0x08000, 0x4000, 0x4b1d8a66 )
	ROM_LOAD( "turbo-cs.u4", 0x0c000, 0x4000, 0x3722ce48 )

	ROM_REGION( 0x40000, REGION_CPU3, 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "sh2u7.bin",  0x00000, 0x10000, 0x02362ea4 )
	ROM_LOAD16_BYTE( "sh2u17.bin", 0x00001, 0x10000, 0xe29a2c37 )

	ROM_REGION( 0x10000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "bg0.11d",  0x00000, 0x08000, 0x81efef7a )
	ROM_LOAD( "bg1.12d",  0x08000, 0x08000, 0x6a902e4d )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg0.7j",   0x00000, 0x20000, 0x55ce12ea )
	ROM_LOAD( "fg1.8j",   0x20000, 0x20000, 0x692afb67 )
	ROM_LOAD( "fg2.9j",   0x40000, 0x20000, 0xf1aba383 )
	ROM_LOAD( "fg3.10j",  0x60000, 0x20000, 0xd3475ff8 )
ROM_END

ROM_START( blasted )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "3c",  0x00000, 0x10000, 0xb243b7df )
	ROM_LOAD16_BYTE( "3b",  0x00001, 0x10000, 0x627e30d3 )
	ROM_LOAD16_BYTE( "2c",  0x20000, 0x10000, 0x026f30bf )
	ROM_LOAD16_BYTE( "2b",  0x20001, 0x10000, 0x8e0e91a9 )

	ROM_REGION( 0x40000, REGION_CPU2, 0 )  /* Sounds Good board */
	ROM_LOAD16_BYTE( "blasted.u7",  0x00000, 0x10000, 0x8d7c8ef6 )
	ROM_LOAD16_BYTE( "blasted.u17", 0x00001, 0x10000, 0xc79040b9 )
	ROM_LOAD16_BYTE( "blasted.u8",  0x20000, 0x10000, 0xc53094c0 )
	ROM_LOAD16_BYTE( "blasted.u18", 0x20001, 0x10000, 0x85688160 )

	ROM_REGION( 0x10000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "11d",  0x00000, 0x08000, 0xd8ed5cbc )
	ROM_LOAD( "12d",  0x08000, 0x08000, 0x60d00c69 )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "fg0",  0x00000, 0x20000, 0x5034ae8a )
	ROM_LOAD( "fg1",  0x20000, 0x20000, 0x4fbdba58 )
	ROM_LOAD( "fg2",  0x40000, 0x20000, 0x8891f6f8 )
	ROM_LOAD( "fg3",  0x60000, 0x20000, 0x18e4a130 )
ROM_END

ROM_START( archrivl )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "3c-rev2",  0x00000, 0x10000, 0x60d4b760 )
	ROM_LOAD16_BYTE( "3b-rev2",  0x00001, 0x10000, 0xe0c07a8d )
	ROM_LOAD16_BYTE( "2c-rev2",  0x20000, 0x10000, 0xcc2893f7 )
	ROM_LOAD16_BYTE( "2b-rev2",  0x20001, 0x10000, 0xfa977050 )

	ROM_REGION( 0x70000, REGION_CPU2, 0 )  /* Audio System board */
	ROM_LOAD( "u4.snd",  0x10000, 0x08000, 0x96b3c652 )
	ROM_LOAD( "u19.snd", 0x30000, 0x08000, 0xc4b3dc23 )
	ROM_LOAD( "u20.snd", 0x50000, 0x08000, 0xf7907a02 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "11d-rev1",  0x00000, 0x10000, 0x7eb3d7c6 )
	ROM_LOAD( "12d-rev1",  0x10000, 0x10000, 0x31e68050 )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "7j-rev1",   0x00000, 0x20000, 0x148ce28c )
	ROM_LOAD( "8j-rev1",   0x20000, 0x20000, 0x58187ac2 )
	ROM_LOAD( "9j-rev1",   0x40000, 0x20000, 0x0dd1204e )
	ROM_LOAD( "10j-rev1",  0x60000, 0x20000, 0xeb3d0344 )
ROM_END

ROM_START( archriv2 )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "archrivl.4",  0x00000, 0x10000, 0x3c545740 )
	ROM_LOAD16_BYTE( "archrivl.2",  0x00001, 0x10000, 0xbc4df2b9 )
	ROM_LOAD16_BYTE( "archrivl.3",  0x20000, 0x10000, 0xd6d08ff7 )
	ROM_LOAD16_BYTE( "archrivl.1",  0x20001, 0x10000, 0x92f3a43d )

	ROM_REGION( 0x70000, REGION_CPU2, 0 )  /* Audio System board */
	ROM_LOAD( "u4.snd",  0x10000, 0x08000, 0x96b3c652 )
	ROM_LOAD( "u19.snd", 0x30000, 0x08000, 0xc4b3dc23 )
	ROM_LOAD( "u20.snd", 0x50000, 0x08000, 0xf7907a02 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "11d-rev1",  0x00000, 0x10000, 0x7eb3d7c6 )
	ROM_LOAD( "12d-rev1",  0x10000, 0x10000, 0x31e68050 )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "7j-rev1",   0x00000, 0x20000, 0x148ce28c )
	ROM_LOAD( "8j-rev1",   0x20000, 0x20000, 0x58187ac2 )
	ROM_LOAD( "9j-rev1",   0x40000, 0x20000, 0x0dd1204e )
	ROM_LOAD( "10j-rev1",  0x60000, 0x20000, 0xeb3d0344 )
ROM_END

ROM_START( pigskin )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "pigskin.a5",  0x00000, 0x10000, 0xab61c29b )
	ROM_LOAD16_BYTE( "pigskin.b5",  0x00001, 0x10000, 0x55a802aa )
	ROM_LOAD16_BYTE( "pigskin.a6",  0x20000, 0x10000, 0x4d8b7e50 )
	ROM_LOAD16_BYTE( "pigskin.b6",  0x20001, 0x10000, 0x1194f187 )

	ROM_REGION( 0x70000, REGION_CPU2, 0 )  /* Audio System board */
	ROM_LOAD( "pigskin.u4",  0x10000, 0x10000, 0x6daf2d37 )
	ROM_LOAD( "pigskin.u19", 0x30000, 0x10000, 0x56fd16a3 )
	ROM_LOAD( "pigskin.u20", 0x50000, 0x10000, 0x5d032fb8 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "pigskin.e2",  0x00000, 0x10000, 0x12d5737b )
	ROM_LOAD( "pigskin.e1",  0x10000, 0x10000, 0x460202a9 )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "pigskin.h15", 0x00000, 0x20000, 0x2655d03f )
	ROM_LOAD( "pigskin.h17", 0x20000, 0x20000, 0x31c52ea7 )
	ROM_LOAD( "pigskin.h18", 0x40000, 0x20000, 0xb36c4109 )
	ROM_LOAD( "pigskin.h14", 0x60000, 0x20000, 0x09c87104 )
ROM_END

ROM_START( trisport )
	ROM_REGION( 0x40000, REGION_CPU1, 0 )
	ROM_LOAD16_BYTE( "la3.a5",  0x00000, 0x10000, 0xfe1e9e37 )
	ROM_LOAD16_BYTE( "la3.b5",  0x00001, 0x10000, 0xf352ec81 )
	ROM_LOAD16_BYTE( "la3.a6",  0x20000, 0x10000, 0x9c6a1398 )
	ROM_LOAD16_BYTE( "la3.b6",  0x20001, 0x10000, 0x597b564c )

	ROM_REGION( 0x70000, REGION_CPU2, 0 )  /* Audio System board */
	ROM_LOAD( "sl1-snd.u4",  0x10000, 0x10000, 0x0ed8c904 )
	ROM_LOAD( "sl1-snd.u19", 0x30000, 0x10000, 0xb57d7d7e )
	ROM_LOAD( "sl1-snd.u20", 0x50000, 0x08000, 0x3ae15c08 )

	ROM_REGION( 0x20000, REGION_GFX1, ROMREGION_DISPOSE )
	ROM_LOAD( "la2.e2",  0x00000, 0x10000, 0xf61149a0 )
	ROM_LOAD( "la2.e1",  0x10000, 0x10000, 0xcf753497 )

	ROM_REGION( 0x80000, REGION_GFX2, ROMREGION_DISPOSE )
	ROM_LOAD( "la2.h15", 0x00000, 0x20000, 0x18a44d43 )
	ROM_LOAD( "la2.h17", 0x20000, 0x20000, 0x874cd237 )
	ROM_LOAD( "la2.h18", 0x40000, 0x20000, 0xf7637a18 )
	ROM_LOAD( "la2.h14", 0x60000, 0x20000, 0x403f9401 )
ROM_END



/*************************************
 *
 *	Driver initialization
 *
 *************************************/

static void init_zwackery(void)
{
	MCR_CONFIGURE_SOUND(MCR_CHIP_SQUEAK_DELUXE);

	/* Zwackery doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	rom_decode();
}


static void init_xenophob(void)
{
	MCR_CONFIGURE_SOUND(MCR_SOUNDS_GOOD);

	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = 0;

	/* Xenophobe doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* install control port handler */
	install_mem_write16_handler(0, 0x0c0000, 0x0cffff, xenophobe_control_w);

	rom_decode();
}


static void init_spyhunt2(void)
{
	MCR_CONFIGURE_SOUND(MCR_TURBO_CHIP_SQUEAK | MCR_SOUNDS_GOOD);

	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = -6;

	/* Spy Hunter 2 doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* analog port handling is a bit tricky */
	install_mem_write16_handler(0, 0x0c0000, 0x0cffff, spyhunt2_control_w);
	install_mem_read16_handler(0, 0x0d0000, 0x0dffff, spyhunt2_port_0_r);
	install_mem_read16_handler(0, 0x0e0000, 0x0effff, spyhunt2_port_1_r);

	rom_decode();
}


static void init_blasted(void)
{
	MCR_CONFIGURE_SOUND(MCR_SOUNDS_GOOD);

	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = 0;

	/* Blasted checks the timing of VBLANK relative to the 493 interrupt */
	/* VBLANK is required to come within 220-256 E clocks (i.e., 2200-2560 CPU clocks) */
	/* after the 493; we also allow 16 E clocks for latency  */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* handle control writes */
	install_mem_write16_handler(0, 0x0c0000, 0x0cffff, blasted_control_w);

	/* 6840 is mapped to the lower 8 bits */
	install_mem_write16_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_w);
	install_mem_read16_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_r);

	rom_decode();
}


static void init_archrivl(void)
{
	MCR_CONFIGURE_SOUND(MCR_WILLIAMS_SOUND);

	mcr68_sprite_clip = 16;
	mcr68_sprite_xoffset = 0;

	/* Arch Rivals doesn't care too much about this value; currently taken from Blasted */
	mcr68_timing_factor = (256.0 + 16.0) / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	/* handle control writes */
	install_mem_write16_handler(0, 0x0c0000, 0x0cffff, archrivl_control_w);

	/* 49-way joystick handling is a bit tricky */
	install_mem_read16_handler(0, 0x0e0000, 0x0effff, archrivl_port_1_r);

	/* 6840 is mapped to the lower 8 bits */
	install_mem_write16_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_w);
	install_mem_read16_handler(0, 0x0a0000, 0x0a000f, mcr68_6840_lower_r);

	/* expand the sound ROMs */
	memcpy(&memory_region(REGION_CPU2)[0x18000], &memory_region(REGION_CPU2)[0x10000], 0x08000);
	memcpy(&memory_region(REGION_CPU2)[0x20000], &memory_region(REGION_CPU2)[0x10000], 0x10000);
	memcpy(&memory_region(REGION_CPU2)[0x38000], &memory_region(REGION_CPU2)[0x30000], 0x08000);
	memcpy(&memory_region(REGION_CPU2)[0x40000], &memory_region(REGION_CPU2)[0x30000], 0x10000);
	memcpy(&memory_region(REGION_CPU2)[0x58000], &memory_region(REGION_CPU2)[0x50000], 0x08000);
	memcpy(&memory_region(REGION_CPU2)[0x60000], &memory_region(REGION_CPU2)[0x50000], 0x10000);

	rom_decode();
}


static void init_pigskin(void)
{
	MCR_CONFIGURE_SOUND(MCR_WILLIAMS_SOUND);

	/* Pigskin doesn't care too much about this value; currently taken from Tri-Sports */
	mcr68_timing_factor = 115.0 / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	mcr68_sprite_clip = 16;
	mcr68_sprite_xoffset = 0;

	/* expand the sound ROMs */
	memcpy(&memory_region(REGION_CPU2)[0x20000], &memory_region(REGION_CPU2)[0x10000], 0x10000);
	memcpy(&memory_region(REGION_CPU2)[0x40000], &memory_region(REGION_CPU2)[0x30000], 0x10000);
	memcpy(&memory_region(REGION_CPU2)[0x60000], &memory_region(REGION_CPU2)[0x50000], 0x10000);

	rom_decode();
}


static void init_trisport(void)
{
	MCR_CONFIGURE_SOUND(MCR_WILLIAMS_SOUND);

	/* Tri-Sports checks the timing of VBLANK relative to the 493 interrupt */
	/* VBLANK is required to come within 87-119 E clocks (i.e., 870-1190 CPU clocks) */
	/* after the 493 */
	mcr68_timing_factor = 115.0 / (double)(Machine->drv->cpu[0].cpu_clock / 10);

	mcr68_sprite_clip = 0;
	mcr68_sprite_xoffset = 0;

	/* expand the sound ROMs */
	memcpy(&memory_region(REGION_CPU2)[0x20000], &memory_region(REGION_CPU2)[0x10000], 0x10000);
	memcpy(&memory_region(REGION_CPU2)[0x40000], &memory_region(REGION_CPU2)[0x30000], 0x10000);
	memcpy(&memory_region(REGION_CPU2)[0x58000], &memory_region(REGION_CPU2)[0x50000], 0x08000);
	memcpy(&memory_region(REGION_CPU2)[0x60000], &memory_region(REGION_CPU2)[0x50000], 0x10000);

	rom_decode();
}



/*************************************
 *
 *	Game drivers
 *
 *************************************/

GAME( 1984, zwackery, 0,        zwackery, zwackery, zwackery, ROT0,   "Bally Midway", "Zwackery" )
GAME( 1987, xenophob, 0,        xenophob, xenophob, xenophob, ROT0,   "Bally Midway", "Xenophobe" )
GAME( 1987, spyhunt2, 0,        spyhunt2, spyhunt2, spyhunt2, ROT0,   "Bally Midway", "Spy Hunter 2 (rev 2)" )
GAME( 1987, spyhnt2a, spyhunt2, spyhunt2, spyhunt2, spyhunt2, ROT0,   "Bally Midway", "Spy Hunter 2 (rev 1)" )
GAME( 1988, blasted,  0,        xenophob, blasted,  blasted,  ROT0,   "Bally Midway", "Blasted" )
GAME( 1989, archrivl, 0,        archrivl, archrivl, archrivl, ROT0,   "Bally Midway", "Arch Rivals (rev 4.0)" )
GAME( 1989, archriv2, archrivl, archrivl, archrivl, archrivl, ROT0,   "Bally Midway", "Arch Rivals (rev 2.0)" )
GAME( 1989, trisport, 0,        trisport, trisport, trisport, ROT270, "Bally Midway", "Tri-Sports" )
GAME( 1990, pigskin,  0,        pigskin,  pigskin,  pigskin,  ROT0,   "Bally Midway", "Pigskin 621AD" )
