/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*
 * This module encapsulates cell-box handling. Everything else in the program
 * except possibly the evolution function should go through these functions
 * to change the board state. Entry points are:
 *
 * getcell(pp, ptr, dx, dy)       -- get state of cell at x, y in box *ptr
 *
 * setcell(pp, ptr, dx, dy, val)  -- set cell at x, y in box *ptr to state val
 *
 * forgetcell(pp, ptr)            -- cause a box to forget last gen's state
 *
 * int lookcell(context, x, y)    -- get state of cell
 *
 * void chgcell(context, x, y, t, c)	-- set cell to state c
 *
 * void changesize(context, n)    -- change the cellbox allocation size
 *
 * This code knows that cells are packed in 8x8 tiles on a hash list, but
 * only the getcell() and setcell() functions know anything about the
 * internals of cell representation within cell boxes.  The function
 * changesize() changes the size of cellboxes.
 *
 * fetchtile(context, x, y)  -- get tile containing cell (x, y), NULL if none
 * maketile(context, x, y)   -- get tile holding cell (x, y), creating if none
 *
 */

#include <stdio.h>
#include <stddef.h>	/* we need offsetof() */
#include <string.h>	/* for memset(3) */
#include <stdlib.h>	/* for malloc(3) */
#include "defs.h"
#include "data.h"
#include "tile.h"

/*
 * These definitions control the allocation strategy for tiles.  When a
 * context has fewer than HASH_THRESHOLD tiles, tile lookups are done by
 * simple search.  When the threshold is exceeded, createtile() allocates
 * and builds a (rather large) hash table to speed up lookups.  The point
 * here is twofold: first, this avoids the overhead of hash computation
 * and indirection in contexts so small that the simple loop is faster;
 * secondly, it makes contexts for small patterns low-overhead objects
 * (this is significant for the structured-save code, which tosses around
 * lots of contexts).
 *
 * The HASH_THRESHOLD value should really be tuned by profiling; this
 * one is a guess based on staring at the search code. It could
 * probably be larger, but will fit most pattern fragments.
 */
#define HASH_THRESHOLD	9	/* threshold for making a hash table */
#define HASHSIZE	32768	/* range of HASH values */
#define DEADTIME	10	/* nuke a tile when it's been dead this long */
#define HASH(x,y) 	(((x>>3) & 0x7f)<< 7) + ((y>>3) & 0x7f)

static tile *freep2, *freepn;

void chgcell(pattern *context, coord_t x, coord_t y, int t, cell_t c)
/* turn a cell to given state */
{
    coord_t ydx,xdx;
    tile *ptr;
    cell_t oldstate;

    ptr = maketile(context, x,y);
    ydx = y - ptr->y;
    xdx = x - ptr->x;
    if (c)
	ptr->dead = 0;

    if (t == PRESENT)
    {
	oldstate = getcell(context, &ptr->cells, xdx, ydx, t);
	if (c && !oldstate)
	    context->cellcount++;
	else if (!c && oldstate)
	    context->cellcount--;
    }

    setcell(context, &ptr->cells, xdx, ydx, t, c);
}

void changesize(pattern *pp, const int newsize)
/* change the tile allocation size */
{
    if (pp->maxstates != newsize)
    {
	clear_pattern(pp);

#ifdef __UNUSED__
	tile	*ptr, *nptr;
	tile	*freep = pp->maxstates == 2 ? freep2 : freepn;

	for (ptr = freep; ptr; ptr = nptr)
	{
	    nptr = ptr->next;
	    free(ptr);
	}
	freep = (tile *)NULL;
#endif /* __UNUSED__ */

	pp->maxstates = newsize;
	if (newsize == 2)
	    pp->tilesize = offsetof(tile, cells) + sizeof(struct twostate_t);
	else
	    pp->tilesize = offsetof(tile, cells) + sizeof(struct nstate_t);
    }
}

void vacuum(pattern *context)
/* reclaim dead storage */
{
    tile	*ptr, *nextptr;

    for (ptr = context->tiles; ptr; ptr = nextptr)
    {
	nextptr = ptr->next;
	if (ptr->dead > DEADTIME)
	{
	    unsigned long hv=HASH(ptr->x,ptr->y);
	    tile *freep = (context->maxstates == 2) ? freep2 : freepn;

	    if (ptr != context->tiles)
		ptr->fore->next=ptr->next;
	    else
		context->tiles = ptr->next;
	    if (context->hashlist)
	    {
		if (ptr == context->hashlist[hv])
		    context->hashlist[hv] = ptr->hnext;
		else
		    ptr->hfore->hnext=ptr->hnext;
		if (ptr->hnext) ptr->hnext->hfore = ptr->hfore;
	    }
	    if (ptr->next) ptr->next->fore = ptr->fore;
	    if (ptr->rt) ptr->rt->lf = NULL;
	    if (ptr->lf) ptr->lf->rt = NULL;
	    if (ptr->up) ptr->up->dn = NULL;
	    if (ptr->dn) ptr->dn->up = NULL;
	    ptr->next = freep;
	    freep = ptr;
	    context->tilecount--;
	}
    }
}

tile *createtile(pattern *context, coord_t x, coord_t y, unsigned long hv)
/* create a tile to hold cells near life-world coordinates x, y */
{
    tile *ptr;
    tile *freep = (context->maxstates == 2) ? freep2 : freepn;

#ifdef PROF
    create_called++;
#endif /* PROF */

    if (freep == NULL){
#ifdef PROF
	create_null++;
#endif /* PROF */
	if ((ptr = (tile *)malloc(context->tilesize)) ==  (tile *)NULL){
	    perror("create: malloc error while creating tile: ");
	    exit(-1);
	}
    }
    else{
	ptr=freep;
	freep=freep->next;
    }
    memset(ptr, '\0', context->tilesize);

    ptr->next=context->tiles;
    context->tiles=ptr;
    
    if (ptr->next != NULL){
	ptr->next->fore=ptr;
    }	
    if (context->hashlist)
    {
	ptr->hnext=context->hashlist[hv];
	context->hashlist[hv]= ptr;
    
	if (ptr->hnext != NULL){
	    ptr->hnext->hfore=ptr;
	}
    }
    ptr->x = x;
    ptr->y = y;
    context->tilecount++;

    /* this tile creation may trigger building of a hash table */
    if (!context->hashlist && context->tilecount > HASH_THRESHOLD)
    {
	tile	*cptr;

	if ((context->hashlist = (tile **)calloc(sizeof(tile *), HASHSIZE)) ==  (tile **)NULL)
	{
	    perror("create: malloc error while creating hash table: ");
	    exit(-1);
	}

	for (cptr = context->tiles; cptr; cptr = cptr->next)
	{
	    coord_t	hx = cptr->x & 0xfffffff8;
	    coord_t	hy = cptr->y & 0xfffffff8;
	    unsigned 	hv = HASH(hx, hy);

	    cptr->hnext = context->hashlist[hv];
	    context->hashlist[hv] = cptr;
    
	    if (cptr->hnext != NULL){
		ptr->hnext->hfore = cptr;
	    }
	}
    }

    return(ptr);
}


static tile *findtile(const pattern *context, coord_t x, coord_t y, unsigned long hv)
/* find a tile with given coordinates; return NULL if no such tile */
{
    tile	*ptr;

#ifdef PROF
    clink_called++;
#endif /* PROF */
    if (context->hashlist)
    {
	ptr = context->hashlist[hv];

	if (ptr == (tile *)NULL)
	    return((tile *)NULL);
	do {
#ifdef PROF
	    clink_search++;
#endif /* PROF */
	    if ((ptr->x == x) && (ptr->y == y))
		return(ptr);
	    ptr = ptr->hnext;
	} while
		(ptr != (tile *)NULL);
    }
    else
    {
	for (ptr = context->tiles; ptr; ptr = ptr->next)
	{
#ifdef PROF
	    clink_search++;
#endif /* PROF */
	    if ((ptr->x == x) && (ptr->y == y))
		return(ptr);
	}
    }

    return((tile *)NULL);
}

int lookcell(pattern const *context, coord_t x, coord_t y, int t)
/* get the state of a cell from its world coordinates */
{
    tile *ptr;
    coord_t hx, hy;
    unsigned long hv;
    
    hx = x & 0xfffffff8;
    hy = y & 0xfffffff8;
    hv = HASH(hx,hy);
    if ((ptr = findtile(context, hx, hy, hv)))
	return(getcell(context, &ptr->cells, x - hx, y - hy, t));
    else
	return(0);
}

tile *fetchtile(pattern *context, coord_t x, coord_t y)
{
    return(findtile(context, x, y, HASH(x, y)));
}

tile *maketile(pattern *context, coord_t x, coord_t y)
{
    tile *ptr;
    unsigned long hv;    
    
    x &= 0xfffffff8;
    y &= 0xfffffff8;
    hv = HASH(x,y);
    if ((ptr = findtile(context, x, y, hv)))
	return(ptr);
    else
	return(createtile(context, x, y, hv));
}

/*
 * These are the only functions that know about cellbox interiors.
 */

int getcell(const pattern *pp, cellbox *ptr, const int xdx, const int ydx, const int gen)
/* get state of cell xdx, ydx within box *ptr */
{
#if STATEBITS > 1
    if (pp->maxstates > 2)
	return(ptr->nstate.cell[gen][ydx][xdx]);
    else
#endif /* STATEBITS > 1 */
	if (ydx > 3)
	    return((ptr->twostate.lohalf[gen] & 1 << ((ydx - 4)*8 + xdx)) !=0);
	else
	    return((ptr->twostate.hihalf[gen] & 1 << (ydx*8 + xdx)) !=0);
}

void setcell(const pattern *pp, cellbox *ptr, const int xdx, const int ydx, const int gen, const int val)
/* set state of cell xdx, ydx within box *ptr */
{
#if STATEBITS > 1
    if (pp->maxstates > 2)
	ptr->nstate.cell[gen][ydx][xdx] = val;
    else
#endif /* STATEBITS > 1 */
	if (val)
	{
	    if (ydx > 3)
		ptr->twostate.lohalf[gen] |= 1 << ( (ydx - 4)*8  + xdx);
	    else
		ptr->twostate.hihalf[gen] |= 1 << ( ydx*8  + xdx);
	}
	else
	{
	    if (ydx > 3)
		ptr->twostate.lohalf[gen] &= 0xffffffff^(1 << ( (ydx - 4)*8  + xdx));
	    else
		ptr->twostate.hihalf[gen] &= 0xffffffff^(1 << ( ydx*8  + xdx));
	}
}

void forgetcell(const pattern *pp, cellbox *ptr)
/* remove a box's info about last generation's state */
{
#if STATEBITS > 1
    register int dy, dx;

    if (pp->maxstates > 2)
    {
	for (dy = 0; dy < BOXSIZE; dy++)
	    for (dx = 0; dx < BOXSIZE; dx++)
		ptr->nstate.cell[1][dy][dx] = 0;
    }
    else
#endif /* STATEBITS > 1 */
    {
	ptr->twostate.hihalf[1] ^= ptr->twostate.hihalf[1];
	ptr->twostate.lohalf[1] ^= ptr->twostate.lohalf[1];
    }
}

/* tile.c ends here */
