/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*****************************************************************************

This module provides functions for manipulating the pattern contexts.  
It does not interface to the GUI or canvas functions.

void initcells(context)   -- initialize a pattern

void clear_pattern()      -- clear a context, freeing all storage

void savepattern()        -- save entire board state to file

void center(*px, *py)     -- return barycenter of pattern

void median(*px, *py)     -- return median of pattern

This module interfaces with the GUI only through error().

*****************************************************************************/

#include <stdio.h>
#include <limits.h>
#include <string.h>
#include "defs.h"
#include "data.h"
#include "tile.h"

#ifdef __UNUSED__
static dumplocs(pattern *context, char *move)
{
    printf("dumplocs: %s\n", move);
    printf("\tmatrix: txx = %d, tyy = %d, txy = %d, tyx = %d)\n",
	   context->at.rxx, context->at.ryy, context->at.rxy, context->at.ryx);
}
#endif /* __UNUSED__ */

void initcells(pattern *context)
/* initialize the cell hash list, preserve transform */
{
    if (context->pattern_name)
    {
	free(context->pattern_name);
	context->pattern_name = (char *)NULL;
    }
    context->tiles = NULL;
    if (context->hashlist)
	free(context->hashlist);
    context->hashlist = (tile **)NULL;
    context->at.rxx = context->at.ryy = 1;
    context->at.rxy = context->at.ryx = 0;
    context->tilecount = context->cellcount = context->changecount = 0;
    context->generations = 0;
    clearcomments(context);
    while (context->legends)
    {
	legend	*next = context->legends->next;

	free(context->legends->text);
        free(context->legends);

	context->legends = next;
    }
}

void resetload(pattern *context)
/* undo all translations, rotations, and generation steps */
{
    context->at = motion_unity;
    context->at.xtrans = context->obasex;
    context->at.ytrans = context->obasey;

    /* undo generation steps here */

#ifdef __UNUSED__
    dumplocs(context, "reset");
#endif /* __UNUSED__ */
}

void moveload(pattern *context, const int newx, const int newy)
/* Move loaded pattern to given cell */
{
    context->at.xtrans = newx;
    context->at.ytrans = newy;

#ifdef __UNUSED__
    dumplocs(context, "translate");
#endif /* __UNUSED__ */
}

void turnload(pattern *context)
/* Turn loaded pattern 90 degrees about its origin */
{
    int t; 

    t = -context->at.ryx; context->at.ryx = context->at.rxx; context->at.rxx = t;
    t = -context->at.ryy; context->at.ryy = context->at.rxy; context->at.rxy = t;

#ifdef __UNUSED__
    dumplocs(context, "rotate");
#endif /* __UNUSED__ */
} 

void flipload(pattern *context)
/* Flip pattern about its x axis */
{
    context->at.ryx = -context->at.ryx;
    context->at.ryy = -context->at.ryy;

#ifdef __UNUSED__
    dumplocs(context, "flip");
#endif /* __UNUSED__ */
}

int transplant(pattern *from, pattern *to,
	    coord_t x1, coord_t y1, coord_t x2, coord_t y2)
/* transplant all live cells in a bounding box */
{
    coord_t xmin, ymin, xmax, ymax;
    tile *ptr;

    int dx, dy, transcount = 0;

    xmin    = (x1 > x2) ? x2 : x1;
    ymin    = (y1 > y2) ? y2 : y1;
    xmax    = (x1 > x2) ? x1 : x2;
    ymax    = (y1 > y2) ? y1 : y2;

    to->obasex = to->at.xtrans = (xmax + xmin)/2;
    to->obasey = to->at.ytrans = (ymin + ymax)/2;

    for (ptr = from->tiles; ptr != NULL; ptr = ptr->next)
    {
	int	state;

	if (ptr->dead)
	    continue;
	else if (ptr->x + BOXSIZE < xmin
		 || ptr->y + BOXSIZE < ymin 
		 || ptr->x > xmax
		 || ptr->y > ymax)
	    continue;

	for (dx = 0; dx < BOXSIZE; dx++)
	    for (dy = 0; dy < BOXSIZE; dy++)
		if (ptr->x + dx < xmin || ptr->y + dy < ymin 
			 || ptr->x + dx > xmax || ptr->y + dy > ymax)
		    continue;
		else if ((state = getcell(from, &ptr->cells, dx, dy, PRESENT)))
		{
		    setcell(to, &ptr->cells, dx, dy, PRESENT, 0);
		    from->cellcount--;
		    chgcell(to,
			    ptr->x + dx - to->obasex, 
			    ptr->y + dy - to->obasey,
			    PRESENT,
			    state);
		    transcount++;
		}
    }

    return(transcount);
}

void center(pattern *context, coord_t *px, coord_t *py)
/* return the `center of mass' of the live boxes in the context */
{
    double x,y;
    cellcount_t ctr=0;
    tile *ptr;
    x=0.0;
    y=0.0;
    for (ptr = context->tiles; ptr != NULL; ptr = ptr->next){
	if (!ptr->dead){
	    x+= ptr->x;
	    y+= ptr->y;
	    ctr++;
	}
    } 
    if (ctr>0) {
	x/=ctr;
	y/=ctr;
    }
    else {
	x=xorigin;
	y=yorigin;
    }
    *px = x;
    *py = y;
}

void median(pattern *context, coord_t *px, coord_t *py)
/* return the median coordinates of the live cells in the context */
{
    coord_t	*coordxlist, *coordylist;
    cellcount_t	ctr = 0;
    tile *ptr;

    *px = xorigin;
    *py = yorigin;

    if (!(coordxlist=(coord_t*)malloc(sizeof(coord_t)*(context->tilecount+1))))
	return;
    if (!(coordylist=(coord_t*)malloc(sizeof(coord_t)*(context->tilecount+1))))
	return;
    coordxlist++; coordylist++;
    for (ptr = context->tiles; ptr != NULL; ptr = ptr->next){
	if(!ptr->dead)
	{
	    coordxlist[ctr] = ptr->x;
	    coordylist[ctr] = ptr->y;
	    ctr++;
	}
    } 
    if (ctr>0) {
	heapsort(coordxlist,ctr,1);
	heapsort(coordylist,ctr,1);
	*px = coordxlist[ctr/2];
	*py = coordylist[ctr/2];
    }
    coordxlist--; coordylist--;

    free(coordylist);
    free(coordxlist);
}

extern void forget(pattern *pp)
/* forget information about the last generation's state in a context */
{
    tile *ptr;

    for (ptr = pp->tiles; ptr != NULL; ptr = ptr->next)
	if (!ptr->dead)
	    forgetcell(pp, &ptr->cells);
}

void clear_pattern(pattern *pp)
/* free given pattern */
{
    tile *ptr, *nptr;

    initcells(pp);
    ptr = pp->tiles;
    while (ptr)
    {
	nptr = ptr->next;
	free(ptr);
	ptr = nptr;
    }
    pp->tiles = NULL;

    pp->generations = 0;
    pp->tilecount = 0;
    pp->cellcount = 0;
    clearcomments(pp);
}

cellcount_t bounding_box(pattern *context)
/* get the bounding box of a pattern */
{
    int dx, dy;
    cellcount_t cellcount = 0;
    tile *ptr;

    context->box.min.x = COORD_MAX; context->box.min.y = COORD_MAX;
    context->box.max.x = COORD_MIN; context->box.max.y = COORD_MIN;
    cellcount = 0;
    for (ptr = context->tiles; ptr != NULL; ptr = ptr->next)
	if (!ptr->dead)
	    for (dx = 0; dx < BOXSIZE; dx++)
		for (dy = 0; dy < BOXSIZE; dy++)
		    if (getcell(context, &ptr->cells, dx, dy, PRESENT))
		    {
			cellcount++;
			if (ptr->x+dx < context->box.min.x)
			    context->box.min.x = ptr->x+dx;
			if (ptr->y+dy < context->box.min.y)
			    context->box.min.y = ptr->y+dy;
			if (ptr->x+dx > context->box.max.x)
			    context->box.max.x = ptr->x+dx;
			if (ptr->y+dy > context->box.max.y)
			    context->box.max.y = ptr->y+dy;
		    }

    return(cellcount);
}

void randomize(context, lx,ly,dx,dy, density)
pattern *context;
coord_t lx, ly, dx, dy;
{
    coord_t num;

    for (num = dx * dy * (density / 100.0); num; num--)
    {
	chgcell(context, lx + random32()%dx, ly + random32()%dy, PRESENT, 1);
	context->cellcount++;
    }
}

void addcomment(pattern *context, const char *comment)
/* add a comment line */
{
    context->comments = (char **)realloc(context->comments, 
				sizeof(char *) * (context->ncomments + 1));
    context->comments[context->ncomments] = strdup(comment);
    context->ncomments++;
}

void clearcomments(pattern *context)
/* wipe all comments from the context */
{
    int	i;

    for (i = 0; i < context->ncomments; i++)
	free(context->comments[i]);
    free(context->comments);
    context->comments = (char **)NULL;
    context->ncomments = 0;
}

void drawlegend(pattern *context, coord_t x, coord_t y, char *msg)
/* associate a legend with a spot in the pattern */
{
    legend	lg;

    if (context->legends && x == context->legends->x && y == context->legends->y)
    {
	context->legends->text = (char *)realloc(context->legends->text,
	    strlen(context->legends->text) + strlen(msg) + 1);
	strcat(context->legends->text, msg);
    }
    else
    {
	lg.text = strdup(msg);
	lg.x = x; lg.y = y;
	lg.next = context->legends;

	if ((context->legends = (legend *)malloc(sizeof(legend))) == NULL)
	{
	    perror("create: malloc error while creating legend: ");
	    exit(-1);
	}

	memcpy(context->legends, &lg, sizeof(legend));
    }
}

/* pattern.c ends here */
