/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * CMU SUCKS
 */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>

#include "defs.h"
#include "data.h"
#include "tile.h"
#include "icon.h"
#include "cursor_data.h"
#include "cursor_mask.h"
#include "guidata.h"
#include "xcanvas.h"

#ifndef SMALLFONT
#define NORMALFONT "9x15"
#define BOLDFONT "9x15bold"
#define FONTHEIGHT 15
#define FONTWIDTH 9
#else
#define NORMALFONT "8x13"
#define BOLDFONT "8x13bold"
#define FONTHEIGHT 13
#define FONTWIDTH 8
#endif

#define MAINWIN 0

#define LIFEWIN 1	/* the viewport on the Life universe */

#define INPUTWIN 2	/* the command-input window */
#define INPUTXOFF 2
#define INPUTH 20
#define INPUTYOFF FONTHEIGHT
#define ICOORDS(x,y) (INPUTXOFF + (x) * FONTWIDTH), (INPUTYOFF + (y) * FONTHEIGHT)

#define RULEWIN	3	/* the rule window */
#define RULEXOFF 2
#define RULEYOFF FONTHEIGHT
#define CWRULES(x,y) (RULEXOFF + (x) * FONTWIDTH), (RULEYOFF + (y) * FONTHEIGHT)
#define RULEW (FONTWIDTH * 16)

#define COORDWIN 4	/* the coordinate-display window */
#define COORDXOFF 2
#define COORDYOFF FONTHEIGHT
#define CWCOORDS(x,y) (COORDXOFF + (x) * FONTWIDTH), (COORDYOFF + (y) * FONTHEIGHT)
#define COORDW (FONTWIDTH * 17)

#define HELPWIN 5	/* the help window */

/* time-delays for generation evolution */
#define DELAY_FAST	0
#define DELAY_MED	250
#define DELAY_SLOW	500
#define DELAY_INCREMENT	50

/* the GUI's run states */
#define STOP 	0x0	/* single-stepping */
#define HIDE	0x1	/* evolving, no per-generation display */
#define RUN	0x2	/* evolving with per-generation display */

static int bwidth, height, width, cellheight, cellwidth, density;
static bool dispcoord, dispboxes, dispchanges, shrinktofit;
static int runcounter = -1;

#if STATEBITS == 1
#define paintcolor 1
#else
static int paintcolor = 1;
#endif

static struct timeval timeout;
static Window rootw, mainw, helpw, inputw, coordw, rulew;
static int screen;
static unsigned long fcolor, bcolor;
static XEvent event;
static XFontStruct *nfont, *bfont;
static GC ntextgc, btextgc, inputgc;
static KeySym ks;
static int runmode;
static char keybuf[16];
#define INPBUFLEN 255
static char inpbuf[INPBUFLEN];
static XGCValues xgcv;
static unsigned long cblack;
static int delay;

static void DoExpose(const int);
static void getxstring(int minbuflen);
static void getinputwstring(const char  *promptstr);
static void displaystats(void);
static void showcoord(void);
static void benchmark(void);
static void color_button(int location, int color, int large);
static void setcolor(int val, unsigned long x, unsigned long y);
static void redisplay(int force);
static void clear(void);
static void newrules(void);
static void showrules(void);
static void annotate(const char *msg);

static pattern active, tentative;	/* the pattern layers */
static coord_t box1x, box1y;		/* first point seen during boxing */
static coord_t box2x, box2y;		/* last point seen during boxing */
static char imageformat;

static char *strs[] = { 
#include "help.h"
0
};

#define CONTINUE	"Press any key to continue..."

/*****************************************************************************
 *
 * Help and shell command output
 *
 *****************************************************************************/

static void redraw_help(void)
{
    int i;

    XClearWindow(disp, helpw);
    for (i = 0; strs[i]; i++)
	XDrawString(disp,helpw,ntextgc,10,(i+1)*FONTHEIGHT,strs[i],strlen(strs[i]));
}

static void help(void)
{        
    XRaiseWindow(disp, helpw);
    redraw_help();
    getxstring(0);
    inpbuf[0]=0;
    XLowerWindow(disp, helpw);
}

void shellcommand(prestr,userstr,denystr,poststr)
char  *prestr, *userstr, *denystr, *poststr;
{	
#define  MAXLINELEN  255
    char line[MAXLINELEN];
    int h= 0, i ,  j = 2*FONTWIDTH;
    FILE  *fp;

    strcpy(line,prestr); strcat(line,userstr);
    for (h=0, i=strlen(prestr); line[i]; i++)
    {
	for (j=strlen(denystr);j;j--)
	    if (denystr[j-1]==line[i])
		break;
	if (j)
	    h++;
	else
	    line[i-h]=line[i];
    }
    line[i-h]=line[i];
    strcat(line,poststr);

    if (!(fp = popen(line,"r")))
	return;

    XClearWindow(disp, lifew);
    i = 2;
    while  (fgets(line+1, MAXLINELEN-1, fp))
    {
	line[0]=' ';
	line[strlen(line)-1] = '\0';
	XDrawString(disp,lifew,btextgc,1,i*FONTHEIGHT,line,strlen(line));
	i++;
	if (FONTHEIGHT*i > height-INPUTH-bwidth*2)
	{
	    announce(CONTINUE);
	    XMaskEvent(disp, KeyPressMask, &event);
	    XClearWindow(disp,lifew);
	    i = 2;
	}
    }
    pclose(fp);
    if (i > 2)
    {
	announce(CONTINUE);
	XMaskEvent(disp, KeyPressMask, &event);
    }
    redisplay(TRUE);
}

/*****************************************************************************
 *
 * Keyboard handling
 *
 *****************************************************************************/

static void settimeout(unsigned long ms)
{
    char	buf[50];
    struct timeval timeouttemp;

    timeout.tv_sec=ms/1000;
    timeout.tv_usec=(ms%1000)*1000;
    sprintf(buf, "Delay %d msec", delay);
    annotate(buf);

    timeouttemp.tv_sec = timeout.tv_sec;
    timeouttemp.tv_usec = timeout.tv_usec;
    (void) select(32,0,0,0,&timeouttemp);
}

static interactive_load(char *file)
/* load a pattern with center at the cursor */
{
    if (file[strlen(file)-1] == '/')
#define  LIST_FILES_COMMAND  "/bin/ls "
	shellcommand(LIST_FILES_COMMAND,loadirbuf," ;>|<&","*"LOADEXT" 2>&1");
    else
    {
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	loadfile(file, &tentative,
		 MOUSE2LIFE(event.xmotion.x, view_x),
		 MOUSE2LIFE(event.xmotion.y, view_y));
#if STATEBITS > 1
	if (tentative.maxstates > 2)
	    changesize(&active, tentative.maxstates);
#endif /* STATEBITS */

	if (shrinktofit && active.cellcount == 0)
	{
	    for (;;)
	    {
		cellwidth  = SCALEDOWN(width-bwidth*2);
		cellheight = SCALEDOWN(height-INPUTH-bwidth*3);

		if (cellwidth >= (tentative.box.max.x-tentative.box.min.x)
		    && cellheight >= (tentative.box.max.y-tentative.box.min.y))
		    break;

		setscale(--scale);

		moveload(&tentative,
			 view_x + cellwidth/2, view_y + cellheight/2);
	    }
	}

	if (tentative.tiles)
	{
	    char	msgbuf[80];

	    boxpattern(&tentative, LOAD_BOX);

	    sprintf(msgbuf, "(%d lines of comments)", tentative.ncomments);
	    annotate(msgbuf);
	    if (displaymode == DISPLAY_NORMAL)
		redisplay(FALSE);
	    else
	    {
		displaymode = DISPLAY_NORMAL;
		redisplay(TRUE);
	    }
	    showrules();
	}
	else
	    annotate("");
    }
}

int DoKeySymIn(keysym)
KeySym keysym;
{
#define KEYFRAC	4
    switch (keysym)
    {
    case XK_4:
    case XK_KP_4:
    case XK_Left:
	pan(-cellwidth/KEYFRAC, 0);
	break;
    case XK_6:
    case XK_KP_6:
    case XK_Right:
	pan(cellwidth/KEYFRAC, 0);
	break;
    case XK_8:
    case XK_KP_8:
    case XK_Up:
	pan(0, -cellwidth/KEYFRAC);
	break;
    case XK_2:
    case XK_KP_2:
    case XK_Down:
	pan(0, cellwidth/KEYFRAC);
	break;
    case XK_7:
    case XK_KP_7:
	pan(-cellwidth/KEYFRAC, -cellwidth/KEYFRAC);
	break;
    case XK_9:
    case XK_KP_9:
	pan(cellwidth/KEYFRAC, -cellwidth/KEYFRAC);
	break;
    case XK_3:
    case XK_KP_3:
	pan(cellwidth/KEYFRAC, cellwidth/KEYFRAC);
	break;
    case XK_1:
    case XK_KP_1:
	pan(-cellwidth/KEYFRAC, cellwidth/KEYFRAC);
	break;
    case XK_5:
    case XK_KP_5:
	{
	    coord_t bx, by;

	    center(&active, &bx, &by);
	    view_x = bx - cellwidth/2;
	    view_y = by - cellheight/2;
	}
	break;
    case XK_0:
    case XK_KP_0:
	{
	    coord_t bx, by;

	    median(&active, &bx, &by);
	    view_x = bx - cellwidth/2;
	    view_y = by - cellheight/2;
	}
	break;
    case XK_Help:
	help();
	break;
    default:
	return(FALSE);
    }
#undef KEYFRAC

    redisplay(TRUE);    /* could process it */
    return(TRUE);
}

int breakreq(void)
/* poll for a keyboard interrupt by the user */
{
    XComposeStatus stat;
    int breakval = 0;

    annotate("(type X to halt)");
    if (XCheckMaskEvent(disp,KeyPressMask,&event))
    {
	XLookupString(&event.xkey, keybuf, 16, &ks, &stat);
	breakval=(keybuf[0]=='X');
	keybuf[0]='\0';
    } 
    return breakval;
}

void DoKeyIn(kbuf)
char kbuf[16];
{
    char msgbuf[80], *pn;
    unsigned int num;
    pattern *pp;
#define NUMMARKER 4
    static coord_t xmarker[NUMMARKER], ymarker[NUMMARKER];
    static char stashed[PATNAMESIZ];

    if (tentative.tiles)
    {
	pp = &tentative;
	pn = "tentative";
    }
    else
    {
	pp = &active;
	pn = "active";
    }

    /*
     * Command keys are grouped as on the manual page.
     */
    switch(kbuf[0])
    {
	/* Help and Bailing Out */

    case '?':		/* get help */
	help();
	break;

    case 'C':		/* clear pattern */
	boxpattern(&tentative, 0);
	clear();
	annotate("");
	box2x = box1x; box2y = box1y;	/* box was erased */
	redisplay(TRUE);
	break;

    case 'Q':		/* quit */
	exit(0);
	break;

	/* Motion and Viewport Control */

    case 'r':		/* refresh the screen */
	redisplay(TRUE);
	break;

    case 'J':
    {
	coord_t x, y;
	getinputwstring("Jump to x,y: ");
	if (sscanf(inpbuf+1,"%ld,%ld",&x,&y) == 2)
	{
	    view_x += x + xorigin - MOUSE2LIFE(event.xmotion.x,view_x);
	    view_y += y + yorigin - MOUSE2LIFE(event.xmotion.y,view_y);
	    redisplay(TRUE);
	}
    }
    break;

    case '.':		/* center view on cursor */
	view_x += SCALEDOWN(event.xbutton.x - width/2);
	view_y += SCALEDOWN(event.xbutton.y - height/2);
	redisplay(TRUE);
	break;

    case '(':
    case '[':
    case '{':
	for (num = 0; kbuf[0] != "([{"[num]; num++)
	    continue;
	xmarker[num] = MOUSE2LIFE(event.xmotion.x,view_x);
	ymarker[num] = MOUSE2LIFE(event.xmotion.y,view_y);
	sprintf(msgbuf, "Marker %d set to (%ld,%ld)", 
		num, xmarker-xorigin,ymarker-yorigin);
	announce(msgbuf);
	break;

    case ')':
    case ']':
    case '}':
	for (num = 0; kbuf[0] != ")]}"[num]; num++)
	    continue;
	view_x += xmarker[num] - MOUSE2LIFE(event.xmotion.x,view_x);
	view_y += ymarker[num] - MOUSE2LIFE(event.xmotion.y,view_y);
	redisplay(TRUE);
    break;

    case 'O':		/* change origin */
	announce("Origin set to mouse location.");
	xorigin = MOUSE2LIFE(event.xmotion.x,view_x);
	yorigin = MOUSE2LIFE(event.xmotion.y,view_y);
	break;

	/* Scale */

    case '+':		/* zoom in */
    case '=':
	if (scale < 6)
	{
	    int oldx = MOUSE2LIFE(event.xbutton.x, view_x);
	    int oldy = MOUSE2LIFE(event.xbutton.y, view_y);

	    setscale(++scale);
	    cellwidth  = SCALEDOWN(width-bwidth*2);
	    cellheight = SCALEDOWN(height-INPUTH-bwidth*3);
	    view_x += oldx - MOUSE2LIFE(event.xmotion.x,view_x);
	    view_y += oldy - MOUSE2LIFE(event.xmotion.y,view_y);
	    redisplay(TRUE);
	}
	break;

    case '-':		/* zoom out */
	if (scale > -3)
	{
	    int oldx = MOUSE2LIFE(event.xbutton.x, view_x);
	    int oldy = MOUSE2LIFE(event.xbutton.y, view_y);

	    setscale(--scale);
	    cellwidth  = SCALEDOWN(width-bwidth*2);
	    cellheight = SCALEDOWN(height-INPUTH-bwidth*3);
	    view_x += oldx - MOUSE2LIFE(event.xmotion.x,view_x);
	    view_y += oldy - MOUSE2LIFE(event.xmotion.y,view_y);
	    redisplay(TRUE);
	}
	break;

	/* Evolution */

    case 'n':		/* set evolution counter */
	  runcounter = -1;
	  getinputwstring("Number of generations to perform (default = infinite): ");
	  sscanf(inpbuf+1, "%u", &runcounter);
	  if (runcounter == 0)
	      runcounter = -1;
	  break;

    case 'g':		/* evolve pattern */
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	runmode = (runmode==RUN) ? STOP : RUN;
	redisplay(TRUE);
	break;

    case 'o':		/* evolve active pattern 1 step */
	num = (tentative.tiles != NULL);
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	getcheckstats(&active);
	generate(&active);
	redisplay(num);
	break;

    case 'f':		/* set evolution speed to slow */
	settimeout(delay = DELAY_FAST);
	break;

    case 'm':		/* set evolution speed to medium */
	settimeout(delay = DELAY_MED);
	break;

    case 's':		/* set evolution speed to fast */
	settimeout(delay = DELAY_SLOW);
	break;

    case '>':
	if (delay > 0)
	    delay -= DELAY_INCREMENT;
	settimeout(delay);
	break;

    case '<':
	settimeout(delay += DELAY_INCREMENT);
	break;

	/* Reports and Decorations */

    case 'p':		/* set oscillation check */
	getinputwstring("Oscillation check period: ");
	sscanf(inpbuf+1, "%u", &num);
	setcheckperiod(num);
	break;

    case 'M':		/* toggle the mesh */
	dispmesh ^= 1;
	redisplay(TRUE);
      break;

    case 'B':
	if (active.cellcount == 0)
	    announce("Life is extinct.");
	else
	{
	    bounding_box(&active);
	    sprintf(msgbuf,
		    "Life bounds: %ld <= x <= %ld  %ld <= y <= %ld",
	            active.box.min.x-xorigin,
		    active.box.max.x-xorigin,
		    active.box.min.y-yorigin,
		    active.box.max.y-yorigin);
	    announce(msgbuf);
	}
	break;

    case 'c':		/* toggle boxcount display */
	dispboxes ^= 1;
	displaystats();
	break;

    case '$':		/* toggle change count display */
	dispchanges ^= 1;
	displaystats();
	break;

    case '&':		/* toggle running coordinate display */
	dispcoord ^= 1;
	DoExpose(COORDWIN);
	break;

    case 'h':		/* hide (don't show evolution) */
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	if (runmode != HIDE)
	    runmode = HIDE;
	else
	{
	    runmode = STOP;
	    redisplay(TRUE);
	}
	break;

    case '*':		/* toggle false-color display */
	if (active.maxstates == 2)
	{
	    falsecolor ^= 1;
	    redisplay(TRUE);
	}
	break;

    case '~':		/* change display mode */
	displaymode = (displaymode + 1) % DISPLAYMODES;
	sprintf(msgbuf, "Display mode %d", displaymode);
	annotate(msgbuf);
	redisplay(TRUE);
	break;

    case '#':		/* toggle wireframe mode */
	wireframe ^= 1;
	redisplay(TRUE);
	break;

	/* Making Soup */

    case '%':		/* make soup */
	sprintf(msgbuf, "Density in percent (%d%%): ", density);
	getinputwstring(msgbuf);
	sscanf(inpbuf+1, "%i", &density);
	if (box2x != box1x && box2y != box1y)
	{
	    int width, height, xbase, ybase;
	    width    = (box1x > box2x) ? (box1x - box2x) : (box2x - box1x);
	    height   = (box1y > box2y) ? (box1y - box2y) : (box2y - box1y);
	    xbase    = (box1x > box2x) ? box2x : box1x;
	    ybase    = (box1y > box2y) ? box2y : box1y;

	    randomize(&active, xbase, ybase, width, height, density);
	}
	else
	    randomize(&active, view_x,view_y, cellwidth,cellheight, density);
	redisplay(FALSE);
	break;

    case '^':		/* set random-number seed */
    {
	unsigned long  x1, x2 = 0;
	getinputwstring("Number(s) for randomseed: ");
	sscanf(inpbuf+1,"%lu %lu",&x1, &x2);
	randomseed(x2,x1);
    }
    break;

    /* Unstructured pattern saving */

    case 'S':		/* save either active or tentative pattern */
    {
	FILE *savefl;
    
	strcpy(inpbuf,"Save to:");
	strcat(inpbuf, "./");
	XClearWindow(disp,inputw);
	XDrawString(disp, inputw, ntextgc,ICOORDS(0,0),inpbuf, strlen(inpbuf));

	getxstring(8);
	strcpy(msgbuf,checktilda(inpbuf+8));
	inpbuf[0]=0;
	strcat(msgbuf,addlifeext(msgbuf));

	if ((savefl=fopen(msgbuf,"w")) != NULL)
	{
	    savepattern(savefl, pp, imageformat);
	    fclose(savefl);
	}
	else
	    error(strerror(errno));
	break;
    }

    case 'i':		/* set image format */
	  imageformat='\0';
	  getinputwstring("Image format for saving patterns: A D I R M P S (default = I): ");
	  if (strchr("ADIRMPS",inpbuf[1]))
	     imageformat = inpbuf[1];
	  break;

    case 'N':		/* change name of tentative or active for 'S' */
	if (active.pattern_name)
	    sprintf(msgbuf,"Old %s Name: %s, New Name: ", pn,pp->pattern_name);
	else
	    sprintf(msgbuf,"New %s Name: ", pn);
	getinputwstring(msgbuf);
	if (inpbuf[1])
	{
	    if (pp->pattern_name)
	    {
		free(pp->pattern_name);
		pp->pattern_name = (char *)NULL;
	    }
	    pp->pattern_name = strdup(inpbuf+1);
	}
	displaystats();
	break;

	/* Load-script commands */

    case 'l':		/* load from file */
	strcpy(inpbuf,"Load from: ");
	strcat(inpbuf,loadirbuf);
	XClearWindow(disp,inputw);
	XDrawString(disp, inputw, ntextgc,ICOORDS(0,0),inpbuf, strlen(inpbuf));
	getxstring(10);
	strcpy(stashed, inpbuf+11);
	interactive_load(inpbuf+11);
	break;

    case 'L':
	if (!stashed[0])
	    error("No previous load,");
	else
	{
	    sprintf(msgbuf, "Reloading %s", stashed);
	    announce(msgbuf);
	    interactive_load(stashed);
	}
	break;

    case 'G':		/* perform some evolutions on loaded pattern */
	num = 1;
	sprintf(msgbuf, "Evolutions to do on %s pattern (default=1): ", pn);
	getinputwstring(msgbuf);
	sscanf(inpbuf+1, "%i", &num);
	while (num-- > 0)
	{
	    if (divis(num,100))
	    {
		sprintf(msgbuf,	"%d steps left (type X to halt)", pn, num);
		annotate(msgbuf);
	    }
	    if (pp == &active)
		getcheckstats(pp);
	    generate(pp);
	    redisplay(FALSE);
	    if (breakreq())
		break;
	}
	break;

    case 'U':		/* unload pattern */
	if (tentative.tiles)
	    clear_pattern(&tentative);
	boxpattern(&tentative, 0);
	redisplay(TRUE);
	break;

    case 'u':		/* undo post-load manipulations of pattern */
	resetload(&tentative);
	XWarpPointer(disp, lifew, lifew, 0, 0, 0, 0,
		     LIFE2MOUSE(tentative.at.xtrans, view_x),
		     LIFE2MOUSE(tentative.at.ytrans, view_y));
	redisplay(TRUE);
	break;

    case 'I':		/* force confirm of loaded pattern */
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	redisplay(TRUE);
	break;

    case 'W':		/* write load script */
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	strcpy(inpbuf,"Save load script to:");
	num = strlen(inpbuf);
	strcat(inpbuf, "./");
	XClearWindow(disp,inputw);
	XDrawString(disp, inputw, ntextgc,ICOORDS(0,0), inpbuf,strlen(inpbuf));

	getxstring(num);
	saveloadscript(&active, inpbuf + 20, 
		       MOUSE2LIFE(event.xmotion.x,view_x),
		       MOUSE2LIFE(event.xmotion.y,view_y));

	freeloadscript();
	displaystats();
	break;

    case 'D':		/* discard load script */
	freeloadscript();
	if (tentative.tiles)
	{
	    if (tentative.tiles)
		clear_pattern(&tentative);
	    boxpattern(&tentative, 0);
	    redisplay(TRUE);
	    announce("Load script (and latest load) discarded");
	}
	else
	    announce("Load script discarded"); 
	break;

	/* Pattern Comments */

    case 'A':		/* add comment */
	for(;;)
	{
	    getinputwstring("Comment: ");
	    addcomment(pp, inpbuf+1);
	}
	displaystats();
	break;

    case 'K':		/* clear comments */
	clearcomments(pp);
	sprintf(msgbuf, "Comments for %s pattern discarded.\n", pn);
	announce(msgbuf);
	break;

    case 'V':		/* view comments */
	if (!pp->ncomments)
	{
	    sprintf(msgbuf, "No comments attached to %s pattern.", pn);
	    announce(msgbuf);
	}
	else
	{
	    for(num = 0; num < pp->ncomments; num++) {
		char	*last = pp->comments[num] + strlen(pp->comments[num]);
		while (isspace(last[-1]))
		    --last;
		*last = '\0';
		XDrawString(disp, lifew, btextgc, 10, (num+1)*FONTHEIGHT,
			    pp->comments[num], strlen(pp->comments[num]));
	    }
	    XFlush(disp);
	    announce(CONTINUE);
	    XMaskEvent(disp, KeyPressMask, &event);
	    redisplay(TRUE);
	}
	break;

	/* Rules Editing */

    case 'R':		/* change 2-state rules */
        runmode = STOP;
#if STATEBITS > 1
	changesize(&active, 2);
	changesize(&tentative, 2);
#endif /* STATEBITS > 1 */
	newrules();
	DoExpose(INPUTWIN);	/* force the radio buttons on screen */
	break;

#if STATEBITS > 1
    case 'F':		/* load N-state rules */
        runmode = STOP;
	getinputwstring("Load rules from: ");
	if (use_rules(inpbuf+1, &active) == 0)
	    showrules();
	DoExpose(INPUTWIN);	/* force the radio buttons on screen */
	break;
#endif /* STATEBITS > 1 */

#if STATEBITS > 1
    case 'a':		/* add transition to table */
	runmode = STOP;
	{
	    char	*err;

	    (void) strcpy(msgbuf, inpbuf + sizeof("Set transition: ") - 1);
	    if ((err = parse_rule(msgbuf)))
		error(err);
	    else
		announce(msgbuf);
	}
	break;

    case 't':		/* test transition */
#define TESTPROMPT	"Test transition: "
	runmode = STOP;
	{
	    char	s, n1, n2, n3, n4;
	    int	ns;

	    announce(TESTPROMPT);
	    getxstring(strlen(TESTPROMPT));

	    (void) strcpy(msgbuf, inpbuf + sizeof(TESTPROMPT) - 1);
	    if (sscanf(msgbuf, "%c%c%c%c%c", &s, &n1, &n2, &n3, &n4) != 5
		|| !is_state(s)
		|| !is_state(n1) || !is_state(n2)
		|| !is_state(n3) || !is_state(n4))
		announce("This command wants exactly five valid state codes.");
	    else
	    {
		(void) sprintf(msgbuf,"%c%c%c%c%c", s, n1, n2, n3, n4);
		if ((ns = newstate(stoi(s), stoi(n1), stoi(n2), stoi(n3), stoi(n4))) == BADSTATE)
		    (void) strcat(msgbuf, " has no defined outcome.");
		else
		    (void) sprintf(msgbuf + strlen(msgbuf), ": %c", itos(ns));
		announce(msgbuf);
	    }
	}	
	break;
#endif /* STATEBITS > 1 */

	/* Miscellaneous */

    case '!':		/* run shell command */
       	getinputwstring("!");
	for (num = strlen(inpbuf); num>1 ;num--)
	    if (inpbuf[num-1] > ' ')
		break;
	inpbuf[num]='\0';
	for (num=1;inpbuf[num];num++)
	    if (inpbuf[num] > ' ') 
		break;
	if (!strcmp(inpbuf+num,"cd"))
	    chdir(getenv("HOME") );
	else if (!strncmp(inpbuf+num,"cd ",3))
	    chdir(inpbuf+num+3);
	else
	    shellcommand("exec 2>&1 <&- ; ",inpbuf+num,"&","");
	break;

    case '@':		/* benchmark XLife */
	benchmark();
	runmode = STOP;
	redisplay(TRUE);
	break;

    default:
	break;
    }
    kbuf[0]='\0';		/* get rid of old keystroke so shift doesn't bring it back */
}

/*****************************************************************************
 *
 * X event handing
 *
 *****************************************************************************/

static int ClassifyWin(win)
Window win;
{
    if (win == lifew)
	return LIFEWIN;
    if (win == inputw)
	return INPUTWIN;
    if (win == mainw)
	return MAINWIN;
    if (win == rulew)
	return RULEWIN;
    if (win == coordw)
	return COORDWIN;
    if (win == helpw)
	return HELPWIN;
    return 0;	/* -Wall fodder */
}

void DoButton(void)
/* handle a button-press event */
{
#if STATEBITS > 1
    if (ClassifyWin(event.xbutton.window) == INPUTWIN)
	setcolor(-1, event.xbutton.x, event.xbutton.y);
#endif /* STATEBITS > 1 */

    if (ClassifyWin(event.xbutton.window) == LIFEWIN)
	switch(event.xbutton.button)
	{
	case 1:
            if (tentative.tiles)
	    {
		moveload(&tentative, 
			 MOUSE2LIFE(event.xmotion.x,view_x),
			 MOUSE2LIFE(event.xmotion.y,view_y));
		redisplay(TRUE);
	    }
	    else
	    {
		paintcell(event.xbutton.x, event.xbutton.y, paintcolor);
		chgcell(&active,
			MOUSE2LIFE(event.xbutton.x,view_x),
			MOUSE2LIFE(event.xbutton.y,view_y),
			PRESENT,
			paintcolor);
		showcoord();
            }
	    break;
	case 2:
            if (tentative.tiles)
	    {
		flipload(&tentative);
		redisplay(TRUE);
	    }
	    else
	    {
		erasebox();
		box2x = box1x = MOUSE2LIFE(event.xmotion.x, view_x);
		box2y = box1y = MOUSE2LIFE(event.xmotion.y, view_y);		
		showcoord();
		redisplay(FALSE);
            }
	    break;
	case 3:
            if (tentative.tiles)
	    {
		turnload(&tentative);
		redisplay(TRUE);
	    }
	    else
	    {
		paintcell(event.xbutton.x, event.xbutton.y, 0);
		chgcell(&active,
			MOUSE2LIFE(event.xbutton.x,view_x),
			MOUSE2LIFE(event.xbutton.y,view_y),
			PRESENT,
			0); 
		showcoord();
            }
	}

    XFlush(disp);
}

static void DoExpose(const int win)
{
    switch(win)
    {
    case LIFEWIN:
	while (XCheckMaskEvent(disp,ExposureMask,&event))
	    continue;
	redisplay(TRUE);
    case INPUTWIN:
	XClearWindow(disp, inputw);
	XDrawString(disp,inputw, ntextgc, ICOORDS(0,0), inpbuf,strlen(inpbuf));
#if STATEBITS > 1
	if (active.maxstates > 2)
	{
	    int	j;

	    for (j = 0; j < active.maxstates; j++)
		color_button(j, j, 0);
	    setcolor(paintcolor, 0, 0);
	}
#endif
	break;
    case RULEWIN:
	showrules();
	break;
    case HELPWIN:
	redraw_help();
	break;
    case COORDWIN:
	showcoord();
    default:
	break;
    }
}

static void DoMotion(void)
/* handle X motion events */
{
    if (ClassifyWin(event.xmotion.window) == LIFEWIN)
    {
	int x = MOUSE2LIFE(event.xmotion.x,view_x);
	int y = MOUSE2LIFE(event.xmotion.y,view_y);

	if (event.xmotion.state & Button1MotionMask)
	{
	    paintcell(event.xmotion.x, event.xmotion.y, paintcolor);
       	    chgcell(&active, x, y, PRESENT, paintcolor);
	}
	else if (event.xmotion.state & Button2MotionMask)
	{
	    /* erase the old box, draw the new one */
	    erasebox();
	    box2x = x; box2y = y;
       	    drawbox(box1x, box1y, box2x, box2y, SELECT_BOX);
	    redisplay(FALSE);
	}
	else if (event.xmotion.state & Button3MotionMask)
	{
	    paintcell(event.xmotion.x, event.xmotion.y, 0);
	    chgcell(&active, x, y, PRESENT, 0);
	}
	showcoord();
    }
    XFlush(disp);
}

static void DoResize(void)
{
    int owidth = width, oheight = height;

    width = event.xconfigure.width; height = event.xconfigure.height;
    view_x += (owidth - width)/2; view_y += (oheight - height)/2;
    cellwidth = SCALEDOWN(width-bwidth*2);
    cellheight = SCALEDOWN(height-INPUTH-bwidth*3);

    XResizeWindow(disp,inputw, 
                  width-RULEW-dispcoord*(COORDW+bwidth)-bwidth*2, INPUTH);
    XResizeWindow(disp,lifew,width-bwidth*2,height-INPUTH-bwidth*3);
    XMoveWindow(disp,inputw,0,height-INPUTH-bwidth*2);
    XMoveWindow(disp,rulew,width-RULEW-COORDW-bwidth*2,
                            height-INPUTH-bwidth*2);
    XMoveWindow(disp,coordw,width-COORDW-bwidth*2,
                            height-INPUTH-bwidth*2);
    redisplay(TRUE);
}

static void DoRelease()
{
    if (ClassifyWin(event.xbutton.window)==LIFEWIN && event.xbutton.button==2)
	if (!tentative.tiles)
	{
	    initcells(&tentative);
	    if (tentative.pattern_name)
	    {
		free(tentative.pattern_name);
		tentative.pattern_name = (char *)NULL;
	    }

	    /*
	     * If transplant() returns 0, the select box is empty.  Leave it
	     * alone -- user may want to do C or % or some other area command.
	     */
	    if (transplant(&active, &tentative, box1x, box1y, box2x, box2y))
	    {
		erasebox();
		if (displaymode > 0)
		    annotate("Display mode reset to 0.");
		displaymode = DISPLAY_NORMAL;
		redisplay(FALSE);
	    }
	}
}

/*****************************************************************************
 *
 * Interactive input
 *
 *****************************************************************************/

static void getxstring(int minbuflen)
/* fill inpbuf with a line of input, handling other mouse events */
{
    XComposeStatus status;
    int offset=0, buflen, windowlen;
    char tmpinpbuf[INPBUFLEN];

    runmode = STOP;
    for (;;)
    {
	XMaskEvent(disp, KeyPressMask | ButtonPressMask | Button1MotionMask 
		   | PointerMotionMask | Button3MotionMask | ExposureMask 
		   | StructureNotifyMask,&event);
	/* handle other kinds of events if no key is pressed  */
	strcpy(tmpinpbuf,inpbuf);
	switch(event.type)
	{
	case MotionNotify:
            DoMotion();
	    break;
	case ButtonPress:
	    DoButton();
	    break;
	case ConfigureNotify:
	    DoResize();
	    break;
	case Expose:
	    DoExpose(ClassifyWin(event.xexpose.window));
	default:
	    break;
	} 
	strcpy(inpbuf,tmpinpbuf);

	showcoord();

	if (event.type != KeyPress)
	{
	    /* non KeyPress events might write on inputw */
	    XClearWindow(disp, inputw);
	    XDrawString(disp, inputw, ntextgc, ICOORDS(0,0), 
			inpbuf + offset, strlen(inpbuf));
        }
        else
	{ 
	    XLookupString(&event.xkey, keybuf, 16, &ks, &status);
	    
	    if (IsModifierKey(ks))
		continue;

	    /* compute window length for rescrolling */
            windowlen = 
		(width
		 - RULEW
		 - dispcoord * (COORDW+bwidth)-bwidth*2) / FONTWIDTH;
	    switch(ClassifyWin(event.xkey.window))
	    {
	    case INPUTWIN:
	    case LIFEWIN:
	    case HELPWIN:
		if ((ks != XK_Return) && (ks != XK_Linefeed))
		{
		    if ((ks == XK_BackSpace) || (ks == XK_Delete))
		    {
			buflen = strlen(inpbuf);
			if (buflen>minbuflen)
			{
			    inpbuf[buflen - 1] = 0;
			    XClearWindow(disp, inputw);
			    offset = (buflen > windowlen) ? buflen - windowlen : 0;
			    XDrawString(disp, inputw, ntextgc, ICOORDS(0,0), inpbuf + offset, buflen);
			}
		    }
		    else
		    {
			if (ks == '~')
			{
			    inpbuf[minbuflen] = '\0';
			    XClearWindow(disp,inputw);
			}
			strcat(inpbuf, keybuf);
			buflen = strlen(inpbuf);
			if (buflen > INPBUFLEN) inpbuf[INPBUFLEN] = 0;
			offset = (buflen > windowlen) ? buflen - windowlen : 0;
			if (offset) XClearWindow(disp, inputw);
			XDrawString(disp, inputw, ntextgc, ICOORDS(0,0), inpbuf + offset, buflen);
		    }
		}
		else
		{
		    XClearWindow(disp, inputw);
		    return;
		}
	    }
	}
    }
}

static void getinputwstring(const char  *promptstr)
{
    int  i;
    int	minbuflen=strlen(promptstr);

    strcpy(inpbuf,promptstr); 
    XClearWindow(disp,inputw);
    XDrawString(disp, inputw, ntextgc,ICOORDS(0,0),inpbuf, minbuflen);

    getxstring(minbuflen);
    if (minbuflen>1)
	for (i=0; (inpbuf[i+1]=inpbuf[minbuflen+i]); i++);
    else if (!minbuflen)
	for (i=strlen(inpbuf)+1;i;i--)
	    inpbuf[i]=inpbuf[i-1];
    inpbuf[0]=0;
}

/*****************************************************************************
 *
 * Initialization and main loop
 *
 *****************************************************************************/

static void color_alloc(char *argv[], bool reverse, XColor *black, XColor *white)
/* initialize graphics contexts for the Life window */
{
    XGCValues xgcv;
    static XColor cellcolor[MAXCOLORS];
    int	i;
    char	*option;

#define COLOR_NAME_LENGTH	36
    static char statecolors[MAXCOLORS][COLOR_NAME_LENGTH] =
    {
	"black",	/* Cell state 0 and box erase color */
	"white",	/* Cell state 1 */
	"brown",	/* Cell state 2 */
	"blue",		/* Cell state 3 */
	"green",	/* Cell state 4 */
	"red",		/* Cell state 5 */
	"yellow",	/* Cell state 6 */
	"purple",	/* Cell state 7 */
	"orange", 	/* selection box color */
	"cyan",		/* tentative pattern color */
	"magenta",	/* transition box color */
	"dark gray",	/* mesh grid color */
	"light gray",	/* reference grid color */
    };

    static const char *cresources[] =
    {
	"stateColor0", "stateColor1", "stateColor2", "stateColor3",
	"stateColor4", "stateColor5", "stateColor6", "stateColor7",
	"selectBoxColor",
	"loadBoxColor",
	"transitionBoxColor",
	"meshGridColor",
	"refGridColor",
    };

    for (i = 0; i < MAXCOLORS; i++)
    {
	option = XGetDefault(disp, argv[0], cresources[i]);
	if (option)
	    strncpy(statecolors[i], option, COLOR_NAME_LENGTH); 
    }
    if (reverse)
    {
	char *whiteloc = NULL, *blackloc = NULL;

	for (i = 0; i < MAXSTATES; i++)
	{
	    if (strcasecmp(statecolors[i], "Black") == 0)
		blackloc = &statecolors[i][0];
	    if (strcasecmp(statecolors[i], "White") == 0)
		whiteloc = &statecolors[i][0];
	}
	if (whiteloc)
	    strcpy(whiteloc, "Black");
	if (blackloc)
	    strcpy(blackloc, "White");
    }

    xgcv.background = cblack;
    for (i = 0; i < MAXCOLORS; i++) 
    {
	XColor	exact;
	char buf[80];

	if (!XAllocNamedColor(disp, DefaultColormap(disp,screen),
			      statecolors[i], &cellcolor[i], &exact))
	{
	    sprintf(buf,"Color allocation for `%s' failed!\n",statecolors[i]);
	    fatal(buf);
	}
	xgcv.foreground = exact.pixel;
	if ((cellgc[i] = XCreateGC(disp, mainw,
				   GCForeground | GCBackground,
				   &xgcv)) == 0)
	{
	    sprintf(buf,"Color creation for `%s' failed!\n",statecolors[i]);
	    fatal(buf);
	}
    }

    *black = cellcolor[0];
    *white = cellcolor[1];
}

int main(argc, argv)
int argc;
char **argv;
{
    Cursor cursor;
    Pixmap icon,cursor_data,cursor_mask;
    XSizeHints hints;
    XWMHints wm_hints;
    XClassHint class_hints;
    XSetWindowAttributes winat;
    XColor white,black;
    XComposeStatus stat;
    char *option, *geomstring = NULL, *initpat = NULL, *display = NULL;
    struct timeval inputdelay, timeouttemp;
    LoadReq *loadqueue = NULL;
    XTextProperty windowName, iconName;
    unsigned long cwhite;
    char *window_name = "Xlife " VERSION ": a cellular-automaton laboratory";
    char *icon_name = "Xlife " VERSION;
    float grab_percent;
    bool reverse = FALSE;
    int i;

    if (!display && !((display = getenv("DISPLAY"))))
        fatal("No display defined!\n");
    else if (!(disp = XOpenDisplay(display)))
	fatal("Can't open X display!\n");

    if (XDefaultDepth(disp, screen) < STATEBITS)
	fatal("Not enough colors for compiled STATEBITS value!\n");

    option = XGetDefault(disp, argv[0], "borderWidth");
    if (option)
	bwidth = atoi(option);

    option = XGetDefault(disp, argv[0], "grabPercent");
    grab_percent = option ? atoi(option) / 100.0 : 0.8;
    if (grab_percent == 0)
	fatal("Bad value in grabPercent resource.\n");

    option = XGetDefault(disp, argv[0], "saveFormat");
    if (option && strchr("ADIMPRS", option[0]))
	imageformat = option[0];

    option = XGetDefault(disp, argv[0], "reverseBoard");
    if (option && strcmp(option, "true") == 0)
	reverse = TRUE;

    shrinktofit = TRUE;
    option = XGetDefault(disp, argv[0], "autoPlace");
    if (option && strcmp(option, "false") == 0)
	shrinktofit = FALSE;

    option = XGetDefault(disp, argv[0], "falseColorMode");
    if (option && strcmp(option, "true") == 0)
	falsecolor = TRUE;

    option = XGetDefault(disp, argv[0], "wireframeMode");
    if (option && strcmp(option, "true") == 0)
	wireframe = TRUE;

    option = XGetDefault(disp, argv[0], "soupDensity");
    density = option ? atoi(option) : 30;

    option = XGetDefault(disp, argv[0], "gridInterval");
    grid_interval = option ? atoi(option) : -5;

    for (i = 1; i < argc; i++)
    {
	if (strcmp(argv[i], "-geometry") == 0)
	    geomstring = argv[++i];
	else if (*argv[i] == '=')
	    geomstring = argv[i];
	else if (strcmp(argv[i], "-rv") == 0)
	    reverse = TRUE;
	else if (!strcmp(argv[i], "-bw"))
	    bwidth = atoi(argv[++i]);
	else if (!strcmp(argv[i], "-grid"))
	    grid_interval = atoi(argv[++i]);
	else if (!strcmp(argv[i], "-display"))
	    display = argv[++i];
	else if (!strcmp(argv[i], "-?"))
	{
	    fprintf(stdout,"usage: %s [-display disp] [-geometry geom]\n",argv[0]);
	    fprintf(stdout,"       [-bw pixels] [-grid n] [-?]\n");
	    exit(0);
	}
	else if (*argv[i] != '-')
	{
	    initpat = argv[i];
	    break;
	}
    }

    dispmesh = FALSE;
    if (grid_interval > 0)
	dispmesh = TRUE;
    else
	grid_interval *= -1;

    screen = DefaultScreen(disp);
    rootw = RootWindow(disp, screen);
    cwhite = WhitePixel(disp, screen);
    cblack = BlackPixel(disp, screen);

    if (reverse)
    {
	fcolor = cblack;
	bcolor = cwhite;
    }
    else
    {
	fcolor = cwhite;
	bcolor = cblack;
    }

    width = DisplayWidth(disp,screen);
    height = DisplayHeight(disp,screen);

    hints.x = 0;
    hints.y = 0;    
    hints.width = width * grab_percent;
    hints.height = height * grab_percent;    
    hints.flags = PPosition | PSize;

    if (geomstring != NULL)
    {
	int result;

	result = XParseGeometry(geomstring,&hints.x,&hints.y,
				(unsigned int *)&hints.width,
				(unsigned int *)&hints.height);
	if (result & XNegative)
	    hints.x += (DisplayWidth(disp,screen) - hints.width) * grab_percent;

	if (result & YNegative)
	    hints.y += (DisplayHeight(disp,screen) - hints.height) * grab_percent;
	if (result & XValue || result & YValue)
	{
	    hints.flags |= USPosition;
	    hints.flags &= ~PPosition;
	}
	if (result & WidthValue || result & HeightValue)
	{
	    hints.flags |= USSize;
	    hints.flags &= ~PSize;
	}
    }
    
    mainw = XCreateSimpleWindow(disp, rootw,
		0, 0, hints.width, hints.height, 0, fcolor, bcolor);

    if (!mainw)
	fatal("Can't open main window");

    icon = XCreateBitmapFromData(disp, mainw, icon_bits, icon_width, icon_height);

    if (XStringListToTextProperty ( &window_name, 1, &windowName ) == 0 )
    {
        (void) fprintf ( stderr, "%s: structure allocation for windowName failed.\n", argv[0] );
        exit (-1);
    }

    if (XStringListToTextProperty(&icon_name, 1, &iconName) == 0)
    {
        (void) fprintf ( stderr, "%s: structure allocation for iconName failed.\n", argv[0] );
        exit (-1);
    }

    wm_hints.initial_state = NormalState;
    wm_hints.input = True;
    wm_hints.icon_pixmap = icon;
    wm_hints.flags = IconPixmapHint | StateHint | InputHint;

    class_hints.res_name =  argv[0];
    class_hints.res_class =  "Basicwin";

    XSetWMProperties(disp, mainw, &windowName, &iconName, argv, argc, &hints, &wm_hints, &class_hints );

    color_alloc(argv, reverse, &black, &white);

    btextgc = cellgc[1];

    /* text display is forced to black on white */
    xgcv.background = cwhite; xgcv.foreground = cblack;
    ntextgc = XCreateGC(disp, mainw, GCForeground | GCBackground, &xgcv);

    if (!((nfont = XLoadQueryFont(disp, NORMALFONT)) && (bfont = XLoadQueryFont(disp, BOLDFONT))))
	fatal("Can't load font\n");
    XSetFont(disp, ntextgc, nfont->fid);
    XSetFont(disp, btextgc, bfont->fid);

    cursor_data = XCreateBitmapFromData(disp, mainw, cursor_data_bits, cursor_data_width, cursor_data_height);
    cursor_mask = XCreateBitmapFromData(disp, mainw, cursor_mask_bits, cursor_mask_width, cursor_mask_height);
    cursor = XCreatePixmapCursor(disp, cursor_data, cursor_mask, &white, &black, cursor_data_x_hot, cursor_data_y_hot);
    XDefineCursor(disp, mainw, cursor);
    
    xgcv.function = GXcopy; xgcv.plane_mask = 1;
    inputgc = XCreateGC(disp, mainw, GCFunction | GCPlaneMask, &xgcv);

    width = hints.width;
    height = hints.height;

    lifew = XCreateSimpleWindow(disp, mainw,
		0, 0,width-bwidth*2, (height - INPUTH - bwidth*3), 
                bwidth,	fcolor, bcolor);
    helpw = XCreateSimpleWindow(disp, mainw,
		0, 0,width-bwidth*2, (height - bwidth*2), 
                bwidth,	cblack, cwhite);
    rulew = XCreateSimpleWindow(disp, mainw,
		width-COORDW-RULEW-bwidth*2, height-INPUTH-bwidth*2,
		RULEW, INPUTH, 
		bwidth,	cblack, cwhite);
    coordw = XCreateSimpleWindow(disp, mainw,
		width-COORDW-bwidth*2, height-INPUTH-bwidth*2,
                COORDW, INPUTH, 
		bwidth,	cblack, cwhite);
    inputw = XCreateSimpleWindow(disp, mainw,
		0, (height - INPUTH - bwidth*2), 
                width - (COORDW + RULEW + bwidth)-bwidth*2, INPUTH,
		bwidth, cblack, cwhite);

    winat.win_gravity = SouthGravity;
    XChangeWindowAttributes(disp,inputw,CWWinGravity,&winat);

    XSelectInput(disp, mainw, ExposureMask | StructureNotifyMask);
    XSelectInput(disp, inputw, KeyPressMask | ButtonPressMask | ExposureMask);
    XSelectInput(disp, lifew, KeyPressMask | ButtonPressMask | Button1MotionMask | PointerMotionMask | Button3MotionMask | ButtonReleaseMask | ExposureMask);
    XSelectInput(disp, helpw, KeyPressMask | ExposureMask);
    XSelectInput(disp, rulew, ExposureMask);
    XSelectInput(disp, coordw, ExposureMask);

    if ((option = XGetDefault(disp, argv[0], "ruleFile")))
	use_rules(option, &active);
    else
    {
	modify_rules("23/3", &active);
	(void) strcpy(loadirbuf, "life/");
    }

    /* GUI initialization */
    settimeout(delay = DELAY_FAST); annotate("");
    dispcoord = TRUE;
    dispboxes = dispchanges = falsecolor = wireframe = FALSE;
    setscale(scale = 3);
    runmode = STOP;
    view_x = xorigin = 0;
    view_y = yorigin = 0;

    /* life structure initialization */
    changesize(&active, 2);
    changesize(&tentative, 2);
    fileinit();
    initcells(&active);

    /* this has to be done after fileinit() */
    option = XGetDefault(disp, argv[0], "namedPatterns");
    saveinit(option);	/* can handle a null argument */

    XMapWindow(disp, inputw);
    XMapWindow(disp, helpw);
    XMapWindow(disp, lifew);
    XMapWindow(disp, mainw);
    XMapWindow(disp, rulew);
    XMapWindow(disp, coordw);
    XLowerWindow(disp, helpw);

    cellwidth = SCALEDOWN(width-bwidth*2);
    cellheight = SCALEDOWN(height-INPUTH-bwidth*3);

    if (initpat != NULL)
    {
	loadfile(initpat, &tentative,
		 view_x + cellwidth/2, view_y + cellheight/2);
	confirmload(&tentative, &active);
	boxpattern(&tentative, 0);
	runmode = RUN;
    }

    for (;;)
    {
	while (XCheckMaskEvent(disp, KeyPressMask | ButtonPressMask | Button1MotionMask | PointerMotionMask | Button3MotionMask | ButtonReleaseMask | ExposureMask | StructureNotifyMask,&event))
	{
	    switch(event.type)
	    {
	    case KeyPress:
		annotate(""); displaystats();
		XLookupString(&event.xkey, keybuf, 16, &ks, &stat);
		if (!DoKeySymIn(ks))
		    DoKeyIn(keybuf);
		break;
	    case MotionNotify:
		/* don't allow drawing until load is confirmed */
		if (!tentative.tiles)
		    DoMotion();
		break;
	    case ButtonPress:
		annotate(""); displaystats();
		DoButton();
		break;
	    case ButtonRelease:
		DoRelease();
		break;
	    case ConfigureNotify:
		DoResize();
		break;
	    case Expose:
		DoExpose(ClassifyWin(event.xexpose.window));
	    default:
		break;
	    }
        }

	if ((runmode == RUN) || (runmode == HIDE))
	{
	    getcheckstats(&active);
	    generate(&active);
	    if (!active.changecount)
	    {
		annotate("Still-life.");
		runmode = STOP;
	    }
	    else if (active.cellcount == 0)
	    {
		annotate("No active cells.");
		runmode = STOP;
	    }
	    else if ((i = checkcellsperiod(&active)) > 1)
	    {
		char buf[80];

		sprintf(buf, "Possible period-%d oscillator after generation %d.", i, active.generations - 1 - i);
		annotate(buf);
		runmode = STOP;
	    }
	    else if (runcounter-- == 0)
	    {
		annotate("Time up.");
		runmode = STOP;
	    }

	    redisplay(FALSE);

	    if (runmode == RUN)
	    {
		timeouttemp.tv_sec = timeout.tv_sec;
		timeouttemp.tv_usec = timeout.tv_usec;
		(void) select(32,0,0,0,&timeouttemp);
	    }
	}
	else
	{
	    inputdelay.tv_sec = 0;
	    inputdelay.tv_usec = 100000;
	    (void) select(32,0,0,0,&inputdelay);
	}
    }
}

/*****************************************************************************
 *
 * Miscellanea
 *
 *****************************************************************************/

void announce(const char *s)
/* display a string in the input window */
{
    (void) strcpy(inpbuf, s);
    DoExpose(INPUTWIN);
}

void error(const char *s)
/* display a string in the input window and beep */
{
    (void) strcpy(inpbuf, s);
    XBell(disp, 0);
    DoExpose(INPUTWIN);
}

static void showcoord(void)
/* update the coordinate & state display */
{
    /* Coordinate display added by Paul Callahan (callahan@cs.jhu.edu) */
    XClearWindow(disp,coordw);
    if (dispcoord)
    {
	coord_t x,y;
	char pstring[50];

	x = MOUSE2LIFE(event.xmotion.x, view_x); 
	y = MOUSE2LIFE(event.xmotion.y, view_y); 

	sprintf(pstring, "(%ld,%ld)=%d %d",
		x - xorigin, y - yorigin,
		lookcell(&active, x,y,PRESENT),
		scale);
	XDrawString(disp,coordw,ntextgc,CWCOORDS(0,0),pstring,strlen(pstring));
    }
}

static void showrules(void)
/* display the current rules */
{
    char	buf[80], *cp;

    display_rules(buf);
    XClearWindow(disp,rulew);
    XDrawString(disp,rulew,ntextgc,CWRULES(0,0), buf,strlen(buf));
}

static void benchmark(void)
{
    lifetime_t num,count;
    double tm;
    struct timeval start,end;
    struct timezone tmz;

    getinputwstring("Number of generations: ");
    sscanf(inpbuf+1,"%ld", &count);
    
    gettimeofday(&start,&tmz);
    for (num=0;num<count;num++)
	generate(&active);

    gettimeofday(&end,&tmz);
    tm=(((end.tv_sec * 1000000) + end.tv_usec) - ((start.tv_sec * 1000000) + start.tv_usec))/1000000.0;
    sprintf(inpbuf,"Time: %f",tm);
}

static char annotation[80];

static void annotate(const char *msg)
/* post an annotation for the next status display */
{
    strcpy(annotation, msg);
}

static void displaystats(void)
/* display statistics for the current generation. */
{
    sprintf(inpbuf,
	    "Generation: %-6lu Cells: %-6lu ",
	    active.generations,active.cellcount);
    if (dispboxes)
	sprintf(inpbuf+strlen(inpbuf)," Tiles: %-6lu ", active.tilecount);
    if (dispchanges)
	sprintf(inpbuf+strlen(inpbuf)," Changes: %-7lu ", active.changecount);
    if (annotation[0])
    {
	strcat(inpbuf, " ");
	strcat(inpbuf, annotation);
    }

    DoExpose(INPUTWIN);
}

static void clear(void)
/* clear the board or tentative pattern, freeing all storage */
{
    if (tentative.tiles)
    {
	clear_pattern(&tentative);
	freeloadscript();
    }
    else
    {
	clear_pattern(&active);
	runmode = STOP;
	setcheckperiod(-1);
    }
    erasebox();
    redisplay(TRUE);
}

static void newrules(void)
/* prompt for and accept 2-state rules */
{
    char *ptr;
    
    XClearWindow(disp,inputw);
    ptr = inpbuf;
    strcpy(ptr,"Rules: ");
    ptr = ptr + 7;
    display_rules(ptr);
    strcat(ptr,"   New Rules: ");
    XDrawString(disp, inputw, ntextgc,ICOORDS(0,0),inpbuf, strlen(inpbuf));

    ptr = inpbuf + strlen(inpbuf);
    getxstring(strlen(inpbuf));
    modify_rules(ptr, &active);
    showrules();

    inpbuf[0]=0;    
    DoExpose(INPUTWIN);		/* refresh the state picker */
}

#if STATEBITS > 1
static void color_button(int location, int color, int large)
{
    XWindowAttributes inattrs;

    XGetWindowAttributes(disp, inputw, &inattrs);
    XFillRectangle(disp,inputw,cellgc[color],
		   inattrs.width - bwidth
			   - (active.maxstates - location) * (FONTWIDTH + 3)
			   - large,
		   bwidth + 3 - large,
		   FONTWIDTH + large * 2, FONTHEIGHT - 1 + large * 2);
}

static void setcolor(int val, unsigned long x, unsigned long y)
{
    if (val == -1)
    {
	int	left;
	XWindowAttributes inattrs;

	XGetWindowAttributes(disp, inputw, &inattrs);
	left = inattrs.width - bwidth - active.maxstates * (FONTWIDTH + 3);

	if (x < left)
	    return;

	val = (x - left) / (FONTWIDTH + 3);
    }

    color_button(paintcolor, 1, 2); 
    color_button(paintcolor, paintcolor, 0);
    color_button(val, 0, 2);
    color_button(val, val, 0);
    paintcolor = val;
}
#endif /* STATEBITS > 1 */

static void redisplay(int force)
/* re-display the visible part of the board */
{
    tile *ptr;
    coord_t x,y, tentx, tenty;

    displaystats();    
    showcoord();
    if (runmode == HIDE)
	return;

    /* go paint the X canvas for the board */
    if (tentative.tiles)
	bounding_box(&tentative);
    redraw(&active, &tentative, force);

#if STATEBITS > 1
    if (force)
	DoExpose(INPUTWIN);		/* refresh the color picker */
#endif	/* STATEBITS > 1 */
}

int patch_transition(coord_t x, coord_t y,
	    cell_t s, cell_t n1, cell_t n2, cell_t n3, cell_t n4)
{
    char	ns, outbuf[100];
    int		newstate;
    FILE	*fp;

    drawbox(x - 1, y - 1, x + 1, y + 1, TRANSITION_BOX);
    redisplay(FALSE);
    runmode = STOP;
    (void) sprintf(outbuf, "New state for neighborhood %c%c%c%c%c: ",
		   itos(s), itos(n1), itos(n2), itos(n3), itos(n4));
    announce(outbuf);
    for (;;)
    {
	getxstring(strlen(inpbuf));
	ns = inpbuf[strlen(inpbuf) - 1];
	if (!is_state(ns))
	    error("Please supply one valid state code: ");
	else
	    break;
    }
    newstate = stoi(ns);
    (void) make_transition(s, n1, n2, n3, n4, newstate);
    (void) sprintf(outbuf,
		   "%c%c%c%c%c: %c", 
		   itos(s), itos(n1),itos(n2),itos(n3),itos(n4), ns);
    if ((fp = fopen(PATCH_LOG, "a")))
    {
	(void) fprintf(fp, "%c%c%c%c%c%c",
		       itos(s), 
		       itos(n1), itos(n2), itos(n3), itos(n4),
		       itos(newstate));
	stamp("\t#", fp);
	(void) fclose(fp);
    }
    announce(outbuf);
    paintcell(LIFE2MOUSE(x, view_x), LIFE2MOUSE(y, view_y), newstate);
    erasebox();
    redisplay(FALSE);

    return(newstate);
}

/* main.c ends here */
