/*
 * XLife Copyright 1989 Jon Bennett jb7m+@andrew.cmu.edu, jcrb@cs.cmu.edu
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the copyright holders not be used in
 * advertising or publicity pertaining to distribution of the software without
 * specific, written prior permission.  The copyright holders make no
 * representations about the suitability of this software for any purpose.  It
 * is provided "as is" without express or implied warranty.
 *
 * THE COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDERS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

/*****************************************************************************

This module provides displaybox() and trdisplaybox().
Its only interface to the GUI is post().

*****************************************************************************/

#include <stdio.h>
#include "defs.h"
#include "data.h"
#include "tile.h"
#include "guidata.h"

/*
 * This is how we define falsecolor mappings.  The table we want is:
 *
 *  old  new                       result
 *   0    0     (never used)
 *   0    1     born this turn       3 (normally blue)
 *   1    0     died this turn       2 (normally red)
 *   1    1     stable               1 (normally white)
 *   
 * Note that when falsecolor is enabled we have to repaint the
 * whole screen every time.  This is the only way we can be
 * sure to nuke all the red squares that should turn black.
 */
static int pseudocolor(old, new, mask)
u32bits old, new, mask; 
{
    register newstate = ((new & mask) != 0);
    register oldstate = ((old & mask) != 0);
    const int paint[2][2] = {{0, 2}, {3, 1}};

    return(paint[newstate][oldstate]);
}

extern int scale;

static void displayline(oline,line,x,y,force)
/* post changes in a tile's state */
u32bits oline,line;
coord_t	x,y;
bool force;
{
    if (falsecolor)
    {
	if (scale >= 0)
	{
	    post(pseudocolor(oline, line, 0x01), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x02), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x04), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x08), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x10), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x20), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x40), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x80), x, y, POST_ACTIVE);
	}
	else if (scale == -1)
	{
	    post(pseudocolor(oline, line, 0x03), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x0c), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0x30), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0xc0), x, y, POST_ACTIVE);
	}
	else if (scale == -2)
	{
	    post(pseudocolor(oline, line, 0x0f), x, y, POST_ACTIVE);
	    ++x;
	    post(pseudocolor(oline, line, 0xf0), x, y, POST_ACTIVE);
	}
	else if (scale == -3)
	{
	    post(pseudocolor(oline, line, 0xff), x, y, POST_ACTIVE);
	}
    }
    else
    {
	u32bits sel = force ? 0xff : (line ^ oline);

	if (scale >= 0)
	{
	    if (sel & 0x01) post((line & 0x01) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x02) post((line & 0x02) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x04) post((line & 0x04) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x08) post((line & 0x08) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x10) post((line & 0x10) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x20) post((line & 0x20) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x40) post((line & 0x40) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x80) post((line & 0x80) != 0, x, y, POST_ACTIVE);
	}
	else if (scale == -1)
	{
	    if (sel & 0x03) post((line & 0x03) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x0c) post((line & 0x0c) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0x30) post((line & 0x30) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0xc0) post((line & 0xc0) != 0, x, y, POST_ACTIVE);
	}
	else if (scale == -2)
	{
	    if (sel & 0x0f) post((line & 0x0f) != 0, x, y, POST_ACTIVE);
	    ++x;
	    if (sel & 0xf0) post((line & 0xf0) != 0, x, y, POST_ACTIVE);
	}
	else if (scale == -3)
	{
	    if (sel) post(line != 0, x, y, POST_ACTIVE);
	}
    }
}

void displaybox(const pattern *pat, coord_t x, coord_t y, cellbox *ptr, bool force)
/* post changes in a tile's state */
{
#if STATEBITS > 1
    register int i, j;
#endif /* STATEBITS > 1 */

    if (displaymode == DISPLAY_CHANGED)
	force = FALSE;

#if STATEBITS > 1
    if (pat->maxstates > 2)
    {
	for (i = 0; i < BOXSIZE; i++)
	    for (j = 0; j < BOXSIZE; j++)
	    {
		int ns = ptr->nstate.cell[PRESENT][i][j];

		if (force || ns != ptr->nstate.cell[PAST][i][j])
		    post(ns, x+j, y+i, POST_ACTIVE);
	    }
    }
    else
#endif /* STATEBITS > 1 */
    {
	register u32bits live1 = ptr->twostate.hihalf[PRESENT];
	register u32bits live2 = ptr->twostate.lohalf[PRESENT];
	register u32bits olive1 = ptr->twostate.hihalf[PAST];
	register u32bits olive2 = ptr->twostate.lohalf[PAST];

	if (scale >= 0)
	{
	    displayline(olive1,    live1,     x,y, force);
	    ++y;
	    displayline(olive1>>8, live1>>8,  x,y, force);
	    ++y;
	    displayline(olive1>>16,live1>>16, x,y, force);
	    ++y;
	    displayline(olive1>>24,live1>>24, x,y, force);
	    ++y;
	    displayline(olive2,    live2,     x,y, force);
	    ++y;
	    displayline(olive2>>8, live2>>8,  x,y, force);
	    ++y;
	    displayline(olive2>>16,live2>>16, x,y, force);
	    ++y;
	    displayline(olive2>>24,live2>>24, x,y, force);
	}
	else if (scale == -1)
	{
	    displayline(olive1|olive1>>8,     live1|live1>>8,      x,y, force);
	    ++y;
	    displayline(olive1>>16|olive1>>24,live1>>16|live1>>24, x,y, force);
	    ++y;
	    displayline(olive2|olive2>>8,     live2|live2>>8,      x,y, force);
	    ++y;
	    displayline(olive2>>16|olive2>>24,live2>>16|live2>>24, x,y, force);
	}
	else if (scale == -2)
	{
	    displayline(
		olive1|olive1>>8|olive1>>16|olive1>>24,
		live1|live1>>8|live1>>16|live1>>24,
		x,y, force);
	    ++y;
	    displayline(
		olive2|olive2>>8|olive2>>16|olive2>>24,
		live2|live2>>8|live2>>16|live2>>24,
		x,y, force);
	}
	else if (scale == -3)
	{
	    displayline(olive1|olive1>>8|olive1>>16|olive1>>24|
			olive2|olive2>>8|olive2>>16|olive2>>24,
			live1|live1>>8|live1>>16|live1>>24|
			live2|live2>>8|live2>>16|live2>>24,
			x,y, force);
	}
    }
}

static void trdisplayline(pat,oline,line,x,y)
/* post changes in a tile's state; use global geometric transformations */
pattern *pat;
u32bits line,oline;
coord_t x,y;
{
    if (falsecolor)
    {
	if (scale >= 0)
	{
	    post(pseudocolor(oline,line,0x01), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x02), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x04), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x08), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x10), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x20), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x40), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;	
	    post(pseudocolor(oline,line,0x80), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	}
	else if (scale == -1)
	{
	    post(pseudocolor(oline,line,0x03), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x0c), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;
	    post(pseudocolor(oline,line,0x30), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;	
	    post(pseudocolor(oline,line,0xc0), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	}
	else if (scale == -2)
	{
	    post(pseudocolor(oline,line,0x0f), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	    ++x;	
	    post(pseudocolor(oline,line,0xf0), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	}
	else if (scale == -3)
	{
	    post(pseudocolor(oline,line,0xff), TX(pat->at,x,y),TY(pat->at,x,y), POST_TENTATIVE);
	}
    }
    else
    {
	u32bits sel = line ^ oline;

	if (scale >= 0)
	{
	    if (sel & 0x01)
		post((line&0x01)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x02)
		post((line&0x02)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x04)
		post((line&0x04)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x08)
		post((line&0x08)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x10)
		post((line&0x10)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x20)
		post((line&0x20)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x40)
		post((line&0x40)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;	
	    if (sel & 0x80)
		post((line&0x80)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	}
	else if (scale == -1)
	{
	    if (sel & 0x03)
		post((line&0x03)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x0c)
		post((line&0x0c)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0x30)
		post((line&0x30)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0xc0)
		post((line&0xc0)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	}
	else if (scale == -2)
	{
	    if (sel & 0x0f)
		post((line&0x0f)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	    ++x;
	    if (sel & 0xf0)
		post((line&0xf0)!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	}
	else if (scale == -3)
	{
		post(line!=0, TX(pat->at, x,y),TY(pat->at, x,y), POST_TENTATIVE);
	}
    }
}

void trdisplaybox(const pattern *pat, coord_t x, coord_t y, cellbox *ptr)
{
#if STATEBITS > 1
    register int i, j;

    if (pat->maxstates > 2)
    {
	for (i = 0; i < BOXSIZE; i++)
	    for (j = 0; j < BOXSIZE; j++)
	    {
		int ns = ptr->nstate.cell[PRESENT][i][j];

		if (ns != ptr->nstate.cell[PAST][i][j])
		    post(ns, TX(pat->at, x+j,y+i), TY(pat->at, x+j,y+i), POST_TENTATIVE);
	    }
    }
    else
#endif /* STATEBITS > 1 */
    {
	register u32bits live1 = ptr->twostate.hihalf[PRESENT];
	register u32bits live2 = ptr->twostate.lohalf[PRESENT];
	register u32bits olive1 = ptr->twostate.hihalf[PAST];
	register u32bits olive2 = ptr->twostate.lohalf[PAST];

	if (scale >= 0)
	{
	    trdisplayline(pat, olive1,     live1,     x,y);
	    trdisplayline(pat, olive1>>8,  live1>>8,  x,++y);
	    trdisplayline(pat, olive1>>16, live1>>16, x,++y);
	    trdisplayline(pat, olive1>>24, live1>>24, x,++y);
	    trdisplayline(pat, olive2,     live2,     x,++y);
	    trdisplayline(pat, olive2>>8,  live2>>8,  x,++y);
	    trdisplayline(pat, olive2>>16, live2>>16, x,++y);
	    trdisplayline(pat, olive2>>24, live2>>24, x,++y);
	}
	else if (scale == -1)
	{
	    trdisplayline(pat,
			  olive1|olive1>>8, live1|live1>>8,
			  x,y);
	    trdisplayline(pat,
			  olive1>>16|olive1>>24, live1>>16|live1>>24,
			  x,++y);
	    trdisplayline(pat,
			  olive2|olive2>>8, live2|live2>>8,
			  x,++y);
	    trdisplayline(pat,
			  olive2>>16|olive2>>24, live2>>16|live2>>24,
			  x,++y);
	}
	else if (scale == -2)
	{
	    trdisplayline(pat,
			  olive1|olive1>>8|olive1>>16|olive1>>24,
			  live1|live1>>8|live1>>16|live1>>24,
			  x,y);
	    trdisplayline(pat,
			  olive2|olive2>>8|olive2>>16|olive2>>24,
			  live2|live2>>8|live2>>16|live2>>24,
			  x,++y);
	}
	else if (scale == -3)
	{
	    trdisplayline(
			  olive1|olive1>>8|olive1>>16|olive1>>24|
			  olive2|olive2>>8|olive2>>16|olive2>>24,
			  pat,live1|live1>>8|live1>>16|live1>>24|
			  live2|live2>>8|live2>>16|live2>>24,
			  x,y);
	}
    }
}

/* cell.c ends here */
