/*
   xlHtml - converts excel files to Html
   Copyright 1999-2001  Steve Grubb  <linux_4ever@yahoo.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published  by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

 */

#ifndef __BORLANDC__
#include "../config.h"		/* Created by ./configure script */
#include "../support.h"		/* Needs to be before internal.h */
#include "../internal.h"		/* Needs to be before cole */
#include "../cole.h"
#else
#include "../config.h.in"	/* Created by ./configure script */
#include "../support.h"		/* Needs to be before internal.h */
#include "../internal.h"		/* Needs to be before cole */
#include "../cole.h.in"
#include <io.h>			/* for umask */
#endif
#include <stdlib.h>		/* For atof(), calloc() */
#include <string.h>		/* For string functions */
#include <math.h>		/* For fabs() */
#include <ctype.h>		/* For isprint() */
#include <errno.h>

/***************************
*	The next couple of lines are "tunable".
*	They set the limits of several arrays.
****************************/
#define MAX_XFORMATS	64		/* Increments to allocate extended formats */
#define MAX_FONTS	 	32 		/* Increments to allocate fonts */
#define MAX_WORKSHEETS	4		/* Increments to allocate worksheet pages */
#define MAX_COLS		24 		/* Increments to allocate Columns per Worksheet page */
#define MAX_ROWS 		128		/* Increments to allocate Rows per Worksheet page */
#define MAX_STRINGS 	256UL	/* Increments to allocate the string array - */
							/* Used by packed string array Opcode: 0xFC */

/**********************************
*
*	Don't change anything below here...
*
************************************/
#define PRGNAME 		"xlHtml"
#define PRGVER  		"0.2.9.8"
#define WBUFF_SIZE 		1550	/* The working buffer. SB 522+10+4(header) bytes minimum = 536 */
#define MAX_COLORS		65	/* This is the size of the built-in color table */
#define EXCEL95		0x500		/* This is the file stamp for biff7 - Excel 5 & 95 */
#define EXCEL97		0x600		/* This is the file stamp for biff8 - Excel 97 & 2000 */
#ifndef __WIN32__
 #include <sys/stat.h>
 #define GLOBAL_UMASK (S_IXUSR|S_IWGRP|S_IRGRP|S_IXGRP|S_IWOTH|S_IROTH|S_IXOTH)
#else
 #define GLOBAL_UMASK (2)
#endif

static char FileName[2][12] =		/* The section of the Excel Spreadsheet we read in */
{
	"/Workbook",		/* Excel 97 & 2000 */
	"/Book"			/* Everything else ? */
};

/* Used by cole_locate_filename() */
struct str_info {
	char *name;
	size_t size;
};

typedef struct		/* This encapsulates the Unicode String	*/
{
	int uni;				/* Unicode String: 0==ASCII/8859-1, 1==windows-1252, 2==utf-8 */
	unsigned char *str;		/* Characters of string */
	unsigned int len;		/* Length of string */
	unsigned char *fmt_run;	/* formatting run, short pairs: offset, index to font */
	unsigned int crun_cnt;	/* The count of format runs */
}uni_string;

typedef struct 		/* This is everything we need for a cell */
{
	unsigned long xfmt;	/* The high bit will tell us which version 0 =< 2; 1 == 2+ */
	int type;			/* This will record the record type that generated the cell */
	uni_string ustr;	/* The cell's displayed contents */
	int spanned;		/* If 1 don't output */
	int rowspan;		/* rows to span */
	int colspan;		/* columns to span */
	uni_string h_link;	/* If a hyperlinked cell, this is the link*/
}cell;

typedef struct		/* This encapsulates some information about each worksheet */
{
	int spanned;
	int first_row;
	int biggest_row;
	int max_rows;
	int first_col;
	int biggest_col;
	int max_cols;
	uni_string ws_title;
	cell **c_array;
}work_sheet;

typedef struct		/* This is everything we need to know about fonts */
{
	int size;
	int attr;
	int c_idx;
	int bold;
	int super;
	int underline;
	uni_string name;
}font_attr;

typedef struct
{
	uni_string *name;
	int cnt;
}fnt_cnt;

typedef struct		/* This covers the Extended Format records */
{
	unsigned int fnt_idx;
	unsigned int fmt_idx;
	short gen;
	short align;
	short indent;
	short b_style;
	short b_l_color;
	long  b_t_color;
	short cell_color;
}xf_attr;

typedef struct
{
	int fflag;		/* Font Flag */
	int bflag;		/* Bold Flag */
	int iflag;		/* Itallic Flag */
	int sflag;		/* Strike thru flag */
	int uflag;		/* Underline flag */
	int sbflag;		/* Subscript */
	int spflag;		/* Superscript */
}html_attr;

static char colorTab[MAX_COLORS][8] =
{
	"000000",	/* Need to find these first 8 colors! */
	"FFFFFF",
	"FFFFFF",
	"FFFFFF",
	"FFFFFF",
	"FFFFFF",
	"FFFFFF",
	"FFFFFF",
	"FFFFFF",	/*0x08 - This one's Black, too ??? */
	"FFFFFF",	/* This one's normal */
	"FF0000",
	"00FF00",
	"0000FF",
	"FFFF00",
	"FF00FF",
	"00FFFF",
	"800000",	/* 0x10 */
	"008000",
	"000080",
	"808000",
	"800080",
	"008080",
	"C0C0C0",
	"808080",
	"9999FF",	/* 0x18 */
	"993366",
	"FFFFCC",
	"CCFFFF",
	"660066",
	"FF8080",
	"0066CC",
	"CCCCFF",
	"000080",
	"FF00FF",	/* 0x20 */
	"FFFF00",
	"00FFFF",
	"800080",
	"800000",
	"008080",
	"0000FF",
	"00CCFF",	/* 0x28 */
	"CCFFFF",
	"CCFFCC",
	"FFFF99",
	"99CCFF",
	"FF99CC",
	"CC99FF",
	"FFCC99",
	"3366FF",	/* 0x30 */
	"33CCCC",
	"99CC00",
	"FFCC00",
	"FF9900",
	"FF6600",
	"666699",
	"969696",
	"003366",	/* 0x38 */
	"339966",
	"003300",
	"333300",
	"993300",
	"993366",
	"333399",
	"333333",
	"FFFFFF"	/* 0x40 */
};

/* FIXME: Support major languages here...not just English */
static const char month_abbr[12][5] = {	"Jan", "Feb", "Mar", "Apr", "May", "June", 
					"July", "Aug", "Sep", "Oct", "Nov", "Dec" };
static const int ndays[]={31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
static const int ldays[]={31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};


/**************************************************************************
*	Functions and data are declared static to prevent name collisions.
***************************************************************************/
/* Function Prototypes */
static void print_version(void);
COLE_LOCATE_ACTION_FUNC scan_file;
static void OutputPartialTableAscii(void);
static void OutputTableHTML(void);
static void OutputTableXML(void);
static void do_cr(void);
static void main_line_processor(unsigned int, unsigned int, unsigned int, unsigned int, unsigned char);
static void SetupExtraction(void);
static void output_header(void);
static void output_footer(void);
static short getShort(unsigned char *);
static long getLong(unsigned char *);
static void getDouble(unsigned char *, double *);
static void RKtoDouble(unsigned long, double *);
static void decodeBoolErr(unsigned int, int, unsigned char *);
static int IsCellNumeric(cell *);
static int IsCellSafe(cell *);
static int IsCellFormula(cell *);
static void output_cell(cell *, int);
static void output_formatted_data(uni_string *, unsigned int, int, int);
static void PrintFloatComma(char *, int, double);
static void print_as_fraction(double, int);
static void NumToDate(int, int *, int *, int *);
static void FracToTime(char *, int *, int *, int *, int *);
static void trim_sheet_edges(unsigned int);
static void update_default_font(unsigned int);
static void incr_f_cnt(uni_string *);
static int get_default_font(void);
static void update_default_alignment(unsigned int, int);
static int null_string(unsigned char *);
static void OutputString(uni_string *);
static void OutputCharCorrected(unsigned char);
static void update_crun_info(unsigned int *loc, unsigned int *fnt_idx, unsigned int crun_cnt, unsigned char *fmt_run);
static void put_utf8(unsigned short);
static void print_utf8(unsigned short);
static void uni_string_clear(uni_string *);
static int uni_string_comp(uni_string *, uni_string *);
static void output_start_html_attr(html_attr *h, unsigned int, int);
static void output_end_html_attr(html_attr *h);
static void html_flag_init(html_attr *h);
static void output_start_font_attribute(html_attr *h, unsigned int fnt_idx);

/* The array update functions */
static int ws_init(int);
static int realloc_worksheets(void);
static int resize_c_array(work_sheet *, int, int);
static void add_wb_array(int, int, unsigned long, int, int, unsigned char *, int, unsigned int, unsigned char *);
static void update_cell_xf(int, int, unsigned long, int);
static void update_cell_hyperlink(int r, int c, char *hyperlink, int len, int type);
static void add_str_array(int, unsigned char *, int, char *, unsigned int);
static void add_font(int, int, int, int, int, int, int, unsigned char *, int);
static void add_ws_title(int, unsigned char *, int);
static void add_xf_array(short fnt_idx, short fmt_idx, short gen, short align,
	short indent, short b_style, short b_l_color, long  b_t_color, short cell_color);


/* Global data */
static char filename[128];
static int file_version = 0;
static unsigned int next_string=0, next_font=0, next_ws_title=0, next_xf=0;
static unsigned char working_buffer[WBUFF_SIZE];
static unsigned int bufidx, buflast;			/* Needed for working buffer */
static unsigned int grbit=0, crun=0, cch = 0;	/* Needed by the SST Opcode FC */
static unsigned int extrst=0, nonascii = 0;		/* Needed by the SST Opcode FC */
static int sheet_count=-2;					/* Number of worksheets found */
static unsigned int last_opcode = -1;			/* Used for the continue command */
static unsigned int cont_grbit=0, cont_str_array=0;
static uni_string default_font;				/* Font for table */
static int default_fontsize = 3;				/* Default font size for table */
static char *default_alignment = 0;			/* Alignment for table */
static int first_sheet = 0;					/* First worksheet to display */
static int last_sheet = MAX_WORKSHEETS-1;		/* The last worksheet to display */
static int xp=0, xr1=-1, xr2=-1, xc1=-1, xc2=-1;	/* Extraction info... */
static int currency_symbol = '$';				/* What to use for currency */

/* Limits */
static unsigned int max_fonts = MAX_FONTS;
static unsigned int max_xformats = MAX_XFORMATS;
static unsigned long max_strings = MAX_STRINGS;
static unsigned int max_worksheets = MAX_WORKSHEETS;

/* Global arrays */
static xf_attr **xf_array;
static work_sheet **ws_array;
static uni_string **str_array;
static font_attr **font_array;
static fnt_cnt *f_cnt;
static int fnt_size_cnt[7];			/* Html has only 7 sizes... */
static uni_string author;
static char *title = 0;
static char *lastUpdated = 0;

/* Command Line flags */
static int use_colors = 1;				/* Whether or not to use colors in output */
static int aggressive = 0;				/* Aggressive html optimization */
static int formula_warnings = 1;			/* Whether or not to suppress formula warnings */
static int center_tables = 0;				/* Whether or not to center justify tables or leave it left */
static int trim_edges = 0;				/* Whether or not to trim the edges of columns or rows */
static char *default_text_color = "000000";
static char *default_background_color="FFFFFF";
static char *default_image=NULL;			/* Point to background image */
static int Ascii = 0;					/* Whether or not to out ascii instaed of html */
static int Csv = 0;						/* Whether or not to out csv instaed of html */
static int OutputXML = 0;				/* Output as xml */
static int DumpPage = 0;					/* Dump page count & max cols & rows */
static int Xtract = 0;					/* Extract a range on a page. */
static int MultiByte = 0;				/* Output as multibyte */


/* Some Global Flags */
static int notAccurate = 0;			/* Flag used to indicate that stale data was used */
static int NoFormat = 0;				/* Flag used to indicated unimplemented format */
static int NotImplemented = 0;		/* Flag to print unimplemented cell type message */
static int Unsupported = 0;			/* Flag to print unsupported cell type message */
static int DatesR1904 = 0;			/* Flag that the dates are based on McIntosh Dates system */
static int MaxPalExceeded = 0;
static int MaxXFExceeded = 0;
static int MaxFormatsExceeded = 0;
static int MaxColExceeded = 0;
static int MaxRowExceeded = 0;
static int MaxWorksheetsExceeded = 0;
static int MaxStringsExceeded = 0;
static int MaxFontsExceeded = 0;
static int UnicodeStrings = 0;		/* 0==ASCII, 1==windows-1252, 2==uft-8 */



int main (int argc, char **argv)
{
	int i, j, k, f_ptr = 0;
	COLEFS * cfs;
	COLERRNO colerrno;

	/* To test cole_locate_filename() */
	struct str_info info;
	info.size = 108004;
	info.name =  FileName[f_ptr];

	if (argc < 2)
	{
usage:
		fprintf (stderr, "\nxlHtml %s converts excel files (.xls) to Html.\n"
			"Copyright (c) 1999-2001, Steve Grubb. Released under GPL.\n"
			"Usage: "PRGNAME" [-xp:# -xc:#-# -xr:#-# -bc###### -bi???????? -tc######] <FILE>\n"
			"\t-a:  aggressive html optimization\n"
			"\t-asc ascii output for -dp & -x? options\n"
			"\t-csv comma separated value output for -dp & -x? options\n"
			"\t-xml XML output\n"
			"\t-bc: Set default background color - default white\n"
			"\t-bi: Set background image path\n"
			"\t-c:  Center justify tables\n"
			"\t-dp: Dumps page count and max rows & colums per page\n"
			"\t-v:  Prints program version number\n"
			"\t-fw: Suppress formula warnings\n"
			"\t-m:  No encoding for multibyte\n"
			"\t-nc: No Colors - black & white\n"
			"\t-tc: Set default text color - default black\n"
			"\t-te: Trims empty rows & columns at the edges of a worksheet\n"
			"\t-xc: Columns (separated by a dash) for extraction (zero based)\n"
			"\t-xp: Page extracted (zero based)\n"
			"\t-xr: Rows (separated by a dash) to be extracted (zero based)\n",
			PRGVER);

		exit (1);
	}
	else
	{
		strncpy(filename, argv[argc-1], 124);
		filename[124] = 0;
		for (i=1; i<(argc-1); i++)
		{
			if (strcmp(argv[i], "-nc") == 0)
				use_colors = 0;
			else if(strcmp(argv[i], "-xml") == 0 )
				OutputXML = 1;
			else if (strcmp(argv[i], "-asc") == 0)
				Ascii = 1;
			else if (strcmp(argv[i], "-csv") == 0)
			{
				Ascii = 1;
				Csv = 1;
			
			}
			else if (strcmp(argv[i], "-a") == 0)
				aggressive = 1;
			else if (strcmp(argv[i], "-fw") == 0)
				formula_warnings = 0;
			else if (strcmp(argv[i], "-c") == 0)
				center_tables = 1;
			else if (strcmp(argv[i], "-dp") == 0)
				DumpPage = 1;
			else if (strcmp(argv[i], "-m") == 0)
				MultiByte = 1;
			else if (strncmp(argv[i], "-tc", 3) == 0)
			{
				default_text_color = &argv[i][3];
				if (strlen(default_text_color) != 6)
					goto usage;
			}
			else if (strncmp(argv[i], "-bc", 3) == 0)
			{
				default_background_color = &argv[i][3];
				if (strlen(default_background_color) != 6)
					goto usage;
			}
			else if (strncmp(argv[i], "-bi", 3) == 0)
			{
				default_image = &argv[i][3];
				use_colors = 0;
			}
			else if (strncmp(argv[i], "-te", 3) == 0)
			{
				trim_edges = 1;
			}
			else if (strcmp(argv[i], "-v") == 0)
				print_version();
			else if (strncmp(argv[i], "-xc:", 4) == 0)
			{
				if (sscanf(argv[i] + 4, "%d-%d", &xc1, &xc2) != 2) {
					fprintf(stderr, "column range %s not valid, expected -xc:FIRST-LAST\n", argv[i] + 4);
					goto usage;
				}
				Xtract = 1;
				if (xc1 > xc2)
				{
					fprintf(stderr, "last column must be >= the first\n");
					exit(1);
				}
			}
			else if (strncmp(argv[i], "-xp:", 4) == 0)
			{
				Xtract = 1;
				xp = atoi(&(argv[i][4]));
				if (xp < 0)
				{
					fprintf(stderr, "Negative numbers are illegal.\n");
					exit(1);
				}
			}
			else if (strncmp(argv[i], "-xr:", 4) == 0)
			{
				char *ptr, *buf;
				Xtract = 1;
				buf = strdup(argv[i]);
				ptr = strrchr(buf, '-');
				xr2 = atoi(ptr+1);
				*ptr = 0;
				ptr = strchr(buf, ':');
				xr1 = atoi(ptr+1);
				free(buf);
				if (xr1 > xr2)
				{
					fprintf(stderr, "row's 2nd digit must be >= the first\n");
					exit(1);
				}
			}
			else
				goto usage;
		}
		if (strcmp(filename, "-v") == 0)
		{
			print_version();
			exit(1);
		}
	}
	if (Ascii)
	{	/* Disable it if DumpPage or Xtract isn't used... */
		if (!(DumpPage||Xtract))
			Ascii = 0;
	}
	if (Xtract)
		trim_edges = 0;		/* No trimming when extracting... */
	if(OutputXML)
		aggressive = 0;

	/* Init arrays... */
	ws_array = (work_sheet **)malloc(max_worksheets * sizeof(work_sheet*));
	for (i=0; i<(int)max_worksheets; i++)
		ws_array[i] = 0;

	str_array = (uni_string **)malloc(max_strings*sizeof(uni_string *));
	for (i=0; i<(int)max_strings; i++)
		str_array[i] = 0;

	font_array = (font_attr **)malloc(max_fonts * sizeof(font_attr *));
	f_cnt = (fnt_cnt *)malloc(max_fonts * sizeof(fnt_cnt));
	for (i=0; i<(int)max_fonts; i++)
	{	/* I assume these won't fail since we are just starting up... */
		font_array[i] = 0;
		f_cnt[i].name = 0;
	}
	xf_array = (xf_attr **)malloc(max_xformats * sizeof(xf_attr *));
	for (i=0; i<(int)max_xformats; i++)
		xf_array[i] = 0;

	uni_string_clear(&author);
	uni_string_clear(&default_font);
	umask(GLOBAL_UMASK);

	/* If successful, this calls scan_file to extract the work book... */
	cfs = cole_mount (filename, &colerrno);
	if (cfs == NULL)
	{
		fprintf(stderr, "%s: cannot mount %s: ", PRGNAME, filename);
		cole_perror (NULL, colerrno);
		goto usage;
	}
	while (cole_locate_filename (cfs, FileName[f_ptr], &info, scan_file, &colerrno))
	{
		if (f_ptr)
		{	/* Two strikes...we're out! */
			cole_perror (PRGNAME, colerrno);
			if (colerrno == COLE_EFILENOTFOUND)
				fprintf(stderr, "Section: Workbook\n");
			break;
		}
		else
		{
			f_ptr++;
			info.name = FileName[f_ptr];
		}
	}

	if (cole_umount (cfs, &colerrno))
	{
		cole_perror (PRGNAME, colerrno);
		exit (1);
	}

	/* For some reason, this loop core dumps on some */
	/* files from Central Europe...so it's commented out for now */
/*	for (i=0; i<max_strings; i++)
	{
		if (str_array[i])
		{
			if (str_array[i]->str)
				free(str_array[i]->str);
			free(str_array[i]);
		}
	}	*/

	for (i=0; i<(int)max_fonts; i++)
	{
		if (font_array[i])
		{
			if (font_array[i]->name.str)
				free(font_array[i]->name.str);
			free(font_array[i]);
			if (f_cnt[i].name)
			{
				if (f_cnt[i].name->str)
					free(f_cnt[i].name->str);
				free(f_cnt[i].name);
			}
		}
	}
	free(font_array);
	free(f_cnt);
	for (i=0; i<(int)max_worksheets; i++)
	{
		if (ws_array[i])
		{
			if (ws_array[i]->ws_title.str)
				free(ws_array[i]->ws_title.str);
			if (ws_array[i]->c_array)
			{
				for (j=0; j<ws_array[i]->max_rows; j++)
				{
					for (k=0; k<ws_array[i]->max_cols; k++)
					{
						if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k])
						{
							if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->ustr.str)
								free(ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->ustr.str);
							if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->ustr.fmt_run)
								free(ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->ustr.fmt_run);
							if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->h_link.str)
								free(ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->h_link.str);
							free(ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]);
						}
					}
				}
				free(ws_array[i]->c_array);
			}
			free(ws_array[i]);
		}
	}

	for (i=0; i<(int)max_xformats; i++)
	{
		if (xf_array[i])
			free(xf_array[i]);
	}
	free(xf_array);

	if (default_font.str)
		free(default_font.str);
	if (author.str)
		free(author.str);
	if (title)
		free(title);
	if (lastUpdated)
		free(lastUpdated);

	return 0;
}

static void print_version(void)
{
	printf("xlHtml %s\n"
		"Copyright (c) 1999-2001, Steve Grubb. Released under GPL.\n", PRGVER);
	exit(1);
}

void scan_file(COLEDIRENT *cde, void *_info)
{
	unsigned int length=0, opcode=0, version=0, target=0, count = 0;
	unsigned char buf[16];
	COLEFILE *cf;
	COLERRNO err;

	cf = cole_fopen_direntry(cde, &err);
	if (cf == 0)
	{	/* error abort processing */
		cole_perror (PRGNAME, err);
		return;
	}

	/* Read & process the file... */
	while (cole_fread(cf, buf, 1, &err))	
	{
   		if (count > 3)
			main_line_processor(opcode, version, count-4, target, buf[0]);
		else if (count == 0)
		{	/* Init everything */
			length = 0;
			opcode = (unsigned)buf[0];
			target = 80;	/* ficticious number */
		}
		else if (count == 1)
			version = (unsigned)buf[0];
		else if (count == 2)
			length = (unsigned)buf[0];
		else if (count == 3)
		{
			length |= (buf[0]<<8);
			target = length;
		}

		if (count == (target+3))
			count = 0;
		else		
			count++;	
	} 
	cole_fclose(cf, &err);

	if (Ascii)
	{
		if (DumpPage)
		{	/* Output the XLS Parameters */
			int i;
			printf("There are %d pages total.\n", sheet_count+1);
			for (i=0; i<=sheet_count; i++)
			{
				printf("Page:%d Name:%s MaxRow:%d MaxCol:%d\n", i,
					ws_array[i]->ws_title.str, ws_array[i]->biggest_row, ws_array[i]->biggest_col);
			}
		}
		else if (Xtract)
			OutputPartialTableAscii();
	}
	else
	{
		if (DumpPage)
		{	/* Output the XLS Parameters */
			int i;
			output_header();
			printf("<p>There are %d pages total.</p>\n", sheet_count+1);
			for (i=0; i<=sheet_count; i++)
			{
				printf("<p>Page:%d Name:%s MaxRow:%d MaxCol:%d</p>\n", i, 
					ws_array[i]->ws_title.str, ws_array[i]->biggest_row, ws_array[i]->biggest_col);
			}
			output_footer();
		}
		else
		{
			if( OutputXML )
				OutputTableXML();
			else
				OutputTableHTML();
		}
	}
}

static void OutputPartialTableAscii(void)
{
	int i, j, k;
	
	SetupExtraction();

	/* Here's where we dump the Html Page out */
	for (i=first_sheet; i<=last_sheet; i++)	/* For each worksheet */
	{
		if (ws_array[i] == 0)
			continue;
		if ((ws_array[i]->biggest_row == -1)||(ws_array[i]->biggest_col == -1))
			continue;
		if (ws_array[i]->c_array == 0)
			continue;

		/* Now dump the table */
		for (j=ws_array[i]->first_row; j<=ws_array[i]->biggest_row; j++)
		{
			for (k=ws_array[i]->first_col; k<=ws_array[i]->biggest_col; k++)
			{
				int safe, numeric=0;
				cell *c = ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]; /* This stuff happens for each cell... */

				if (c)
				{
					numeric = IsCellNumeric(c);
					if (!numeric && Csv)
						printf("\"");
					safe = IsCellSafe(c);

					if (c->ustr.str)
					{
						if (safe)
							output_formatted_data(&(c->ustr), xf_array[c->xfmt]->fmt_idx, numeric, IsCellFormula(c));
						else
							OutputString(&(c->ustr));
					}
					else if (!Csv)
						printf(" ");	/* Empty cell... */
				}
				else
				{   	/* Empty cell... */
					if (!Csv)
						printf(" ");
				}
				if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k])	/* Honor Column spanning ? */
				{
					if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->colspan != 0)
						k += ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->colspan-1;
				}
				if (!numeric && Csv)
					printf("\"");

				if (Csv && (k < ws_array[i]->biggest_col))
				{	/* big cheat here: quoting everything! */
					putchar(',');	/* Csv Cell Separator */
				}
				else
				{
					if (!Csv)
						putchar('\t');	/* Ascii Cell Separator */
				}
			}
			if (Csv)
				printf("\r\n");
			else
				putchar(0x0A);		/* Row Separator */
		}
		if (!Csv)
			printf("\n\n");		 	/* End of Table 2 LF-CR */
	}
}

static void do_cr(void)
{
	if (!aggressive)
		putchar('\n');
}

static void OutputTableHTML(void)
{
	int i, j, k;

	output_header();
	if (center_tables)
	{
		printf("<CENTER>");
		do_cr();
	}

	SetupExtraction();

	/* Here's where we dump the Html Page out */
	for (i=first_sheet; i<=last_sheet; i++)	/* For each worksheet */
	{
		trim_sheet_edges(i);
		update_default_font(i);
		if (ws_array[i] == 0)
			continue;
		if ((ws_array[i]->biggest_row == -1)||(ws_array[i]->biggest_col == -1))
			continue;
		if (ws_array[i]->c_array == 0)
			continue;

		/* Print its name */
		if (next_ws_title > 0)
		{
			if (ws_array[i]->ws_title.str)
			{
				printf("<H1><CENTER>");
				OutputString(&ws_array[i]->ws_title);
				printf("</CENTER></H1><br>");
				do_cr();
			}
		}

		/* Now dump the table */
		printf("<FONT FACE=\"");
		OutputString(&default_font);
		if (default_fontsize != 3)
			printf("\" SIZE=\"%d", default_fontsize);
		printf("\">");
		do_cr();
		printf("<TABLE BORDER=\"1\" CELLSPACING=\"2\">");
		do_cr();
		for (j=ws_array[i]->first_row; j<=ws_array[i]->biggest_row; j++)
		{
			update_default_alignment(i, j);
			printf("<TR");
			if (strcmp(default_alignment, "left") != 0)
				printf(" ALIGN=\"%s\"", default_alignment);
			if (!aggressive)
				printf(" VALIGN=\"bottom\">\n");
			else
				printf(">");
			for (k=ws_array[i]->first_col; k<=ws_array[i]->biggest_col; k++)
			{
				output_cell(ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k],0); /* This stuff happens for each cell... */
				if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k])
				{
					if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->colspan != 0)
						k += ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->colspan-1;
				}
			}

			if (!aggressive)
				printf("</TR>\n");
		}
		printf("</table></FONT><HR>");
		do_cr();
	}

	if (center_tables)
	{
		printf("</CENTER>");
		do_cr();
	}

	/* Print the author's name in itallics... */
	if (author.str)
	{
		printf("<FONT SIZE=-1><I>Spreadsheet's Author:&nbsp;");
		OutputString(&author);
		printf("</I></FONT><br>");
		do_cr();
	}

	/* Print when & how the file was last updated. */
	printf("<FONT SIZE=-1><I>Last Updated ");
	if (lastUpdated)
		printf("%s&nbsp; ", lastUpdated);
	switch (file_version)
	{
		case EXCEL95:
			printf("with Excel 5.0 or 95");
			break;
		case EXCEL97:
			printf("with Excel 97");
			break;
		default:
			printf("with Excel ????");
			break;
	}
	printf("</I></FONT><br>");
	do_cr();

	/* Next print Disclaimers... */
	if (NoFormat)
	{
		printf("<br>* This cell's format is not supported.<br>");
		do_cr();
	}
	if ((notAccurate)&&(formula_warnings))
	{
		printf("<br>** This cell's data may not be accurate.<br>");
		do_cr();
	}
	if (NotImplemented)
	{
		printf("<br>*** This cell's data type will be supported in the future.<br>");
		do_cr();
	}
	if (Unsupported)
	{
		printf("<br>**** This cell's type is unsupported.<br>");
		do_cr();
	}

	/* Now out exceeded capacity warnings... */
	if (MaxWorksheetsExceeded || MaxRowExceeded || MaxColExceeded || MaxStringsExceeded ||
		MaxFontsExceeded || MaxPalExceeded || MaxXFExceeded || MaxFormatsExceeded )
		printf("<FONT COLOR=\"FF0000\">");
	if (MaxWorksheetsExceeded)
	{
		printf("The Maximum Number of Worksheets was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxRowExceeded)
	{
		printf("The Maximum Number of Rows was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxColExceeded)
	{
		printf("The Maximum Number of Columns was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxStringsExceeded)
	{
		printf("The Maximum Number of Strings was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxFontsExceeded)
	{
		printf("The Maximum Number of Fonts was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxPalExceeded)
	{
		printf("The Maximum Number of Color Palettes was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxXFExceeded)
	{
		printf("The Maximum Number of Extended Formats was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxFormatsExceeded)
	{
		printf("The Maximum Number of Formats was exceeded, you might want to increase it.<br>");
		do_cr();
	}
	if (MaxWorksheetsExceeded || MaxRowExceeded || MaxColExceeded || MaxStringsExceeded ||
		MaxFontsExceeded || MaxPalExceeded || MaxXFExceeded || MaxFormatsExceeded )
		printf("</FONT>");

	printf("&nbsp;<br>");
	do_cr();

	/* Output Credit */
	printf("<hr><FONT SIZE=-1>Created with <a href=\"http://www.xlhtml.org/\">xlHtml %s</a></FONT><br>", PRGVER);
	do_cr();

	/* Output Tail */
	output_footer();
}

static void OutputTableXML(void)
{
	int i, j, k;

        printf( "<?xml version=\"1.0\" encoding=\"" );
        switch (UnicodeStrings)
	{
		case 0:
			printf("iso-8859-1\" ?>\n");		/* Latin-1 */
			break;
		case 1:
			printf("windows-1252\"?>\n");		/* Microsoft */
			break;
		default:
			printf("utf-8\"?>\n");			/* Unicode */
			break;
	}

	SetupExtraction();

	printf( "<excel_workbook>\n" );
	printf( "\t<sheets>\n" );

	/* Here's where we dump the Html Page out */
	for (i=first_sheet; i<=last_sheet; i++)	/* For each worksheet */
	{
		trim_sheet_edges(i);
			 update_default_font(i);
		if (ws_array[i] == 0)
			continue;
		if ((ws_array[i]->biggest_row == -1)||(ws_array[i]->biggest_col == -1))
			continue;
		if (ws_array[i]->c_array == 0)
			continue;

		printf( "\t\t<sheet>\n" );
		printf( "\t\t\t<page>%d</page>\n", i );
		
		/* Print its name */
		if (next_ws_title > 0)
		{
			if (ws_array[i]->ws_title.str)
			{
				printf("\t\t\t<pagetitle>");
				OutputString(&ws_array[i]->ws_title);
				printf("\t\t\t</pagetitle>\n");
			}
		}

		printf( "\t\t\t<firstrow>%d</firstrow>\n", ws_array[i]->first_row );
		printf( "\t\t\t<lastrow>%d</lastrow>\n", ws_array[i]->biggest_row );
		printf( "\t\t\t<firstcol>%d</firstcol>\n", ws_array[i]->first_col );
		printf( "\t\t\t<lastcol>%d</lastcol>\n", ws_array[i]->biggest_col );
		printf( "\t\t\t<rows>\n" );

		for (j=ws_array[i]->first_row; j<=ws_array[i]->biggest_row; j++)
		{
			update_default_alignment(i, j);
			printf("\t\t\t\t<row>\n");
			for (k=ws_array[i]->first_col; k<=ws_array[i]->biggest_col; k++)
			{
				printf("\t\t\t\t\t<cell row=\"%d\" col=\"%d\">", j, k );
				output_cell(ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k], 1); /* This stuff happens for each cell... */
				printf("\t\t\t\t\t</cell>\n" );
				if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k])
				{
					if (ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->colspan != 0)
						k += ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k]->colspan-1;
				}
                                
			}

			printf("\t\t\t\t</row>\n");
		}
		printf( "\t\t\t</rows>\n" );
		printf("\t\t</sheet>\n"); 
	}
     printf( "\t</sheets>\n" );

	/* Print the author's name in itallics... */
	if (author.str)
	{
		printf("\t<author>");
		OutputString(&author);
		printf("\t</author>\n");
	}

	/* Print when & how the file was last updated. */
	if (lastUpdated)
		printf("\t<lastwrite>%s</lastwrite>", lastUpdated);
	printf( "\t<excelversion>" );
	switch (file_version)
	{
		case EXCEL95:
			printf("using Excel 5.0 or 95");
			break;
		case EXCEL97:
			printf("using Excel 97/2000");
			break;
		default:
			printf("using Excel ????");
			break;
	}
	printf("</excelversion>\n");
	
	/* Next print Disclaimers... */
	if (NoFormat)
		printf("\t<noformat>%d</noformat>\n", NoFormat );
	if ((notAccurate)&&(formula_warnings))
		printf("\t<accuracy>%d</accuracy>\n", notAccurate );
	if (NotImplemented)
		printf("\t<notimplemented>%d</notimplemented>\n", NotImplemented );
	if (Unsupported)
		printf("\t<unsupported>%d</unsupported>\n", Unsupported );

	/* Now out exceeded capacity warnings... */
	if (MaxWorksheetsExceeded)
		printf("\t<MaxWorksheetsExceeded>The Maximum Number of Worksheets were exceeded, you might want to increase it.</MaxWorksheetsExceeded>\n ");
	if (MaxRowExceeded)
		printf("\t<MaxRowExceeded>The Maximum Number of Rows were exceeded, you might want to increase it.</MaxRowExceeded>\n ");
	if (MaxColExceeded)
		printf("\t<MaxColExceeded>The Maximum Number of Columns were exceeded, you might want to increase it.</MaxColExceeded>\n");
	if (MaxStringsExceeded)
		printf("\t<MaxStringsExceeded>The Maximum Number of Strings were exceeded, you might want to increase it.</MaxStringsExceeded>\n");
	if (MaxFontsExceeded)
		printf("\t<MaxFontsExceeded>The Maximum Number of Fonts were exceeded, you might want to increase it.</MaxFontsExceeded>\n");
	if (MaxPalExceeded)
		printf("\t<MaxPalExceeded>The Maximum Number of Color Palettes were exceeded, you might want to increase it.</MaxPalExceeded>\n");
	if (MaxXFExceeded)
		printf("\t<MaxXFExceeded>The Maximum Number of Extended Formats were exceeded, you might want to increase it.</MaxXFExceeded>\n");
	if (MaxFormatsExceeded)
		printf("\t<MaxFormatsExceeded>The Maximum Number of Formats were exceeded, you might want to increase it.</MaxFormatsExceeded>\n");

	/* Output Credit */
	printf("\t<tool>Created with xlHtml %s</tool>\n", PRGVER);
	printf("\t<toollink>http://www.xlhtml.org/</toollink>\n");
     printf( "</excel_workbook>\n" );
}

static void SetupExtraction(void)
{
	if (Xtract)
	{	/* Revise the page settings... */
/*		printf("-%d %d %d %d %d<br>\n", xp, xr1, xr2, xc1, xc2); */
		if ((xp >= first_sheet)&&(xp <= last_sheet)&&(xp <= sheet_count))
		{
			first_sheet = xp;
			last_sheet = xp;
			if (xr1 < 0)
			{
				xr1 = ws_array[xp]->first_row;
				xr2 = ws_array[xp]->biggest_row;
			}
			else if ((xr1 >= ws_array[xp]->first_row)&&(xr1 <= ws_array[xp]->biggest_row)
				&&(xr2 >= ws_array[xp]->first_row)&&(xr2 <= ws_array[xp]->biggest_row))
			{
				ws_array[xp]->first_row = xr1;
				ws_array[xp]->biggest_row = xr2;

				if (xc1 < 0)
				{
					xc1 = ws_array[xp]->first_col;
					xc2 = ws_array[xp]->biggest_col;
				}
				else if((xc1 >= ws_array[xp]->first_col)&&(xc1 <= ws_array[xp]->biggest_col)
					&&(xc2 >= ws_array[xp]->first_col)&&(xc2 <= ws_array[xp]->biggest_col))
				{
					ws_array[xp]->first_col = xc1;
					ws_array[xp]->biggest_col = xc2;
				}
				else
				{
					if (Ascii)
						fprintf(stderr, "Error - Col not in range during extraction"
						    " (%d or %d not in [%d..%d])\n", xc1, xc2, ws_array[xp]->first_col, ws_array[xp]->biggest_col);
					else
					{
						printf("Error - Col not in range during extraction.\n");
						output_footer();
					}
					return;
				}
			}
			else
			{
				if (Ascii)
					fprintf(stderr, "Error - Row not in range during extraction"
					    " (%d or %d not in [%d..%d])\n", xr1, xr2, ws_array[xp]->first_row, ws_array[xp]->biggest_row);
				else
				{
					printf("Error - Row not in range during extraction.");
					output_footer();
				}
				return;
			}
		}
		else
		{
			if (Ascii)
				fprintf(stderr, "Error - Page not in range during extraction.");
			else
			{
				printf("Error - Page not in range during extraction.");
				output_footer();
			}
			return;
		}
	}
}

static void output_header(void)
{	/* Ouput Header */
	if (!aggressive)
	{
		printf("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML Transitional//EN\"");
		do_cr();
		printf("\"http://www.w3.org/TR/REC-html40/loose.dtd\">");
		do_cr();
	}
	printf("<HTML><HEAD>");
	do_cr();
	printf("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=");
	switch (UnicodeStrings)
	{
		case 0:
			printf("iso-8859-1\">");		/* Latin-1 */
			break;
		case 1:
			printf("windows-1252\">");	/* Microsoft */
			break;
		default:
			printf("utf-8\">");			/* Unicode */
			break;
	}
	do_cr();

	if (!aggressive)
	{
		printf("<meta name=\"GENERATOR\" content=\"xlHtml\">");
		do_cr();
	}
	printf("<TITLE>");
	if (title)
		printf("%s", title);
	else
		printf("%s", filename);
	printf("</TITLE>");
	do_cr();
	printf("</HEAD>");
	do_cr();
	do_cr();
	printf("<BODY TEXT=\"#%s\" BGCOLOR=\"#%s\"",
				default_text_color, default_background_color);
	if (default_image)
		printf("BACKGROUND=\"%s\"", default_image);
	printf("><br>");
	do_cr();
}

static void output_footer(void)
{
	printf("</BODY></HTML>");
	do_cr();
	fflush(stdout);
}

/*******************************************************************
*	count - is the absolute count in the record
*	last - is the size of the record
*	bufidx - is the index into the working buffer
*	buflast - is the expected length of the working buffer
********************************************************************/
static void main_line_processor(unsigned int opcode, unsigned int version, unsigned int count, unsigned int last, unsigned char data)
{
	unsigned int cont_opcode = 0;
	
	/* If first pass, reset stuff. */
	if (count == 0)
	{
		if (opcode != 0x3C)	/* continue command */
/*		{
			printf("\n* * * * * * CONTINUE * * * * * * * * *\n\n");
		}
		else */
		{	/* Normal path... */
			last_opcode = opcode;
			bufidx = 0;
			buflast = 0;
			cont_str_array = 0;
			memset(working_buffer, 0, WBUFF_SIZE);
		}
	}
	if (opcode == 0x3C)
	{
		opcode = last_opcode;
		cont_opcode = 1;
	}

	/* Abort processing if too big. Next opcode will reset everything. */
	if (bufidx >= WBUFF_SIZE)
	{
		/*printf("OC:%02X C:%04X I:%04X BL:%04X cch:%04X gr:%04X\n", opcode, count, bufidx, buflast, cch, grbit); */
		/*abort(); */
		return;
	}

	switch (opcode)
	{
		case 0x09:	/* BOF */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				if (file_version == 0)
				{	/* File version info can be gathered here...
					 *    4 = Excel version 4
					 * 1280 = Excel version 5
					 * 0500 = Excel 95
					 * 1536 = Excel 97 */
					if (version == 8)
						file_version = getShort(&working_buffer[0]);
					else
						file_version = version;
					if (file_version == EXCEL95)
						use_colors = 0;
/*					printf("Biff:%X\n", file_version); */
				}
				sheet_count++;
			}
			break;
		case 0x01:	/* Blank */
			working_buffer[bufidx++] = data;		
			if (bufidx == last)
			{
				int r, c;
				long f;

				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				if (version == 2)
					f = getShort(&working_buffer[4]);
				else
					f = 0;
				add_wb_array(r, c, f, opcode, 0, " ", 1, 0, 0);
			}
			break;
		case 0x02:	/* Integer */
			working_buffer[bufidx++] = data;		
			if (bufidx == last)
			{
				int r, c, i;
				long f;
				unsigned char temp[32];

				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				if (version == 2)
				{
					f = getShort(&working_buffer[4]);
					i = getShort(&working_buffer[7]);
					sprintf(temp, "%d", i);
				}
				else
				{
					f = 0;
					Unsupported++;
					strcpy(temp, OutputXML ? "<Unsupported/>INT" : "****INT");
				}
				add_wb_array(r, c, f, opcode, 0, temp, strlen(temp), 0, 0);
			}		
			break;
		case 0x03:	/* Number - Float */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int r, c;
				long f;
				double d;
				unsigned char temp[64];

				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				if (version == 2)
				{
					f = getShort(&working_buffer[4]);
					getDouble(&working_buffer[6], &d);
					sprintf(temp, "%.15g", d);
				}
				else
				{	/* Who knows what the future looks like */
					f = 0;
					Unsupported = 1;
					sprintf(temp, "****FPv:%d", version);
				}
				add_wb_array(r, c, f, opcode, 0, temp, strlen(temp), 0, 0);
			}
			break;
		case 0xD6:	/* RString */
			working_buffer[bufidx++] = data;		
			if ((bufidx == 7)&&(buflast == 0))
				buflast = working_buffer[7];
			if (buflast)
			{
				if (bufidx == buflast)
				{
					int r, c;
					long l, f;

					r = getShort(&working_buffer[0]);
					c = getShort(&working_buffer[2]);
					f = getShort(&working_buffer[4]);
					l = getShort(&working_buffer[6]);
					working_buffer[8+l] = 0;

					add_wb_array(r, c, f, opcode, 0, &working_buffer[8],
							strlen(&working_buffer[8]), 0, 0);
				}
			}
			break;
		case 0x04:	/* Label - UNI */
			working_buffer[bufidx++] = data;
			if (file_version == EXCEL95)
			{
				if (bufidx == last)
				{
					int r, c;
					long f;

					r = getShort(&working_buffer[0]);
					c = getShort(&working_buffer[2]);
					f = getShort(&working_buffer[4]);
					working_buffer[bufidx] = 0;

					add_wb_array(r, c, f, opcode, 0, &working_buffer[8],
							strlen(&working_buffer[8]), 0, 0);
				}
			}
			else if (file_version == EXCEL97)
			{
				if ((bufidx == 7)&&(buflast == 0))
					buflast = working_buffer[7];
				if (buflast)
				{
					if (bufidx == buflast)
					{
						int r, c;
						long f;

						r = getShort(&working_buffer[0]);
						c = getShort(&working_buffer[2]);
						if (version == 2)
							f = getShort(&working_buffer[4]);
						else	/* Unknown version */
							f = 0;
						working_buffer[bufidx] = 0;

						add_wb_array(r, c, f, opcode, 0, &working_buffer[8], strlen(&working_buffer[8]), 0, 0);
					}
				}
			}
			break;
		case 0x05:	/* Boolerr */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int r, c;
				long f;
				unsigned char temp[16];

				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				if (version == 2)
				{
					f = getShort(&working_buffer[4]);
					decodeBoolErr(working_buffer[6], working_buffer[7], temp);
					add_wb_array(r, c, f, opcode, 0, temp, strlen(temp), 0, 0);
				}
				else
				{
					f = 0;
					Unsupported = 1;
					strcpy(temp, "****Bool");
					add_wb_array(r, c, f, opcode, 0, temp, strlen(temp), 0, 0);
				}
			}
			break;
			/************
			*	This function has 2 entry points. 1 is the mainline FC opcode.
			*	In this event there are several bytes that setup the type of
			*	strings that will follow. Then there is the continue entry
			*	point which is immediate - e.g location 0.
			*************/
		case 0xFC:	/* Packed String Array A.K.A. SST Shared String Table...UNI */
			if ((count > 7)||(cont_opcode == 1)) /* Skip the 1st 8 locations they are bs */
			{
/*				if ((count == 0)&&(data == 0)&&(buflast))	*/
				if ((count == 0)&&(cont_opcode == 1)&&(buflast))
				{
/*					printf("Adjusting...\n"); */
/*					printf("I:%04X BL:%04X\n", bufidx, buflast); */
					cont_str_array = 1;
					cont_grbit = data;
					return;
				}

				working_buffer[bufidx] = data;
				bufidx++;

				if((cont_str_array)&&(grbit & 0x01)&& !(cont_grbit & 0x01))
				{	/* ASCII -> unicode */
					working_buffer[bufidx] = 0;
					bufidx++;
				}

				if (buflast == 0)	/* Header processor */
				{
					if (bufidx == 0x03)  /* After 3 locations we have length */
					{				    /* and type of chars... */
						cch = getShort(&working_buffer[0]);
						grbit = working_buffer[2];

						if (grbit < 0x04)	/* Normal run */
						{
							nonascii = 0;
							bufidx = 0;
							crun = 0;
							extrst = 0;
							buflast = cch << (grbit & 0x01);

							/* special case for empty strings */
							if (!cch && !buflast)
								add_str_array(0, "", 0, 0, 0);
							else
								memset(working_buffer, 0, WBUFF_SIZE);
						}
					}
					else if (bufidx == 0x05)
					{
						if ((grbit & 0x0C) == 0x08)	/* Rich string only */
						{
							nonascii = 0;
							bufidx = 0;
							crun = getShort(&working_buffer[3]);
							extrst = 0;
							buflast = (cch << (grbit & 0x01)) + (crun*4);
/*							printf("rtbuflast:%X cch%X grbit:%X extrst:%X crun:%X last:%X\n",
										buflast, cch, grbit, extrst, crun, last);
							printf("%02X %02X %02X %02X %02X %02X\n",
							working_buffer[0], working_buffer[1], working_buffer[2],
							working_buffer[3], working_buffer[4], working_buffer[5]); */
							memset(working_buffer, 0, WBUFF_SIZE);
						}
					}
					else if (bufidx == 0x07)
					{
						if ((grbit & 0x0C) == 0x04)	/* Extended string only */
						{
							nonascii = 0;
							bufidx = 0;
							crun = 0;
							extrst = getLong(&working_buffer[3]);
							buflast = (cch << (grbit & 0x01)) + extrst;
/*							printf("esbuflast:%X cch%X grbit:%X extrst:%X last:%X\n",
										buflast, cch, grbit, extrst, last);
							printf("%02X %02X %02X %02X %02X %02X\n",
							working_buffer[0], working_buffer[1], working_buffer[2],
							working_buffer[3], working_buffer[4], working_buffer[5]); */
							memset(working_buffer, 0, WBUFF_SIZE);
						}
					}
					else if (bufidx == 0x09)
					{
						if ((grbit & 0x0C) == 0x0C)
						{
							/* Rich String + Extended String **/
							nonascii = 0;
							bufidx = 0;
							crun = getShort(&working_buffer[3]);
							extrst = getLong(&working_buffer[5]);
							buflast = (cch << (grbit & 0x01)) + extrst + (crun*4);
/*							printf("xrtbuflast:%X cch%X grbit:%X extrst:%X crun:%X last:%X\n",
										buflast, cch, grbit, extrst, crun, last);
							printf("%02X %02X %02X %02X %02X %02X\n",
							working_buffer[0], working_buffer[1], working_buffer[2],
							working_buffer[3], working_buffer[4], working_buffer[5]); */
							memset(working_buffer, 0, WBUFF_SIZE);
						}
					}
/*					printf("*%02X ", data); */
				}
				else	/* payload processor */
				{
/*					if (cont_opcode == 1)
						printf(" %02X", data); */
					if (data > 127)
						nonascii = 1;
					if (bufidx == buflast)
					{
						int uni;
						int len = (cch << (grbit & 0x01));
/*						int i;	*/

						if (grbit & 01)
						{
							uni = 2;
							UnicodeStrings = 2;
						}
						else
							uni = nonascii;
						working_buffer[bufidx] = 0;
/*  						fprintf(stderr,":buflast-"); */
/*                                                  { int i; */
/*  						for (i=0; i<buflast; i++) */
/*                                                    putchar(working_buffer[i]); */
/*  						fprintf(stderr,"\nNext String:%d\n", next_string); */
/*                                                  } */

						if (crun)
							add_str_array(uni, working_buffer, len, working_buffer+len, crun);
						else
							add_str_array(uni, working_buffer, len, 0, 0);
						if (uni > UnicodeStrings)	/* Try to "upgrade" charset */
							UnicodeStrings = uni;
						bufidx = 0;
						buflast = 0;
						cch = 0;
						cont_str_array = 0;
						memset(working_buffer, 0, WBUFF_SIZE);
					}
				}
			} 
			break;
		case 0xFD:	/* String Array Index A.K.A. LABELSST */
			working_buffer[count] = data;
			if (count == (last - 1))
			{
				unsigned long i;
				int r, c, f;

				/* This is byte reversed... */
				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				f = getShort(&working_buffer[4]);
				i = getLong(&working_buffer[6]);
				if (i < next_string)
				{
/*					printf("String used:%d\n", (int)i); */
					if (str_array[i])
					{
						if (str_array[i]->str)
							add_wb_array(
								r, c, f, opcode,
								str_array[i]->uni, str_array[i]->str,
								str_array[i]->len, str_array[i]->crun_cnt, str_array[i]->fmt_run);
					}
				}
				else	
					MaxStringsExceeded = 1;
			}
			break;
		case 0x31:	/* Font */
			working_buffer[bufidx++] = data;
			if (bufidx > 14) /* Address 14 has length in unicode chars */
			{	
				if ((file_version == EXCEL95)&&(bufidx == last))
				{	/* Microsoft doesn't stick to their documentation. Excel 97 is supposed
					   to be 0x0231...but its not. Have to use file_version to separate them. */
					unsigned int i;
					int size, attr, c_idx, b, su, u;

					size = getShort(&working_buffer[0]);
					attr = getShort(&working_buffer[2]);
					c_idx = getShort(&working_buffer[4]);
					b = getShort(&working_buffer[6]);
					su = getShort(&working_buffer[8]);
					u = working_buffer[10];
					buflast = working_buffer[14];
					for (i=0; i<buflast; i++)
						working_buffer[i] = working_buffer[i+15];

					working_buffer[buflast] = 0;
/*					printf("S:%04X A:%04X C:%04X B:%04X SU:%04X U:%02X\n",
							size, attr,c_idx,b,su,u);
					printf("f:%s\n", working_buffer); */
					add_font(size, attr, c_idx, b, su, u, 0, &working_buffer[0], 0);
				}
				else if ((file_version == EXCEL97)&&(bufidx == last))
				{	/* Microsoft doesn't stick to their documentation. Excel 97 is supposed
					   to be 0x0231...but its not. Have to use file_version to separate them. */
					unsigned int i, uni=0, len;
					unsigned int size, attr, c_idx, b, su, u;
							
					size = getShort(&working_buffer[0]);
					attr = getShort(&working_buffer[2]);
					c_idx = getShort(&working_buffer[4]);
					b = getShort(&working_buffer[6]);
					su = getShort(&working_buffer[8]);
					u = working_buffer[10];
					buflast = working_buffer[14];

					for (i=0; i<(buflast-2); i++)
					{	/* This looks at the 2nd byte to see if its unicode... */
						if (working_buffer[(i<<1)+17] != 0)
							uni = 2;
					}
						
					if (uni == 2)
						len = buflast<<1;
					else
						len = buflast;

					if (uni == 0)
					{	
						for (i=0; i<len; i++)
						{
							working_buffer[i] = working_buffer[(i<<1)+16];
							if ((working_buffer[i] > 0x0080U) && (uni == 0))
								uni = 1;
						}
					}
					else
					{
						for (i=0; i<len; i++)
							working_buffer[i] = working_buffer[i+16];
					}

					working_buffer[len] = 0;
					
/*					printf("S:%04X A:%04X C:%04X B:%04X SU:%04X U:%02X\n",
							size, attr,c_idx,b,su,u);
					printf("BL:%d L:%d Uni:%d\n", buflast, len, uni);
					printf("%X %X %X %X\n", working_buffer[15], working_buffer[16], working_buffer[17], working_buffer[18]);
					printf("f:%s\n", working_buffer); */
					add_font(size, attr, c_idx, b, su, u, uni, &working_buffer[0], len);
				}
			}
			break;
		case 0x14:	/* Header */
			break;
		case 0x15:	/* Footer */
			break;
		case 0x06:	/* Formula */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int r, c, f;
				unsigned char calc_val[64];

				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				f = getShort(&working_buffer[4]);
				if ((working_buffer[12] == 0xFF)&&(working_buffer[13] == 0xFF))
				{	/* Formula evaluates to Bool, Err, or String */
					if (working_buffer[6] == 1)				/* Boolean */
					{
						decodeBoolErr(working_buffer[8], 0, calc_val);
						opcode = 0x0105;
					}
					else if (working_buffer[6] == 2)			/* Err */
					{
						decodeBoolErr(working_buffer[8], 1, calc_val);
						opcode = 0x0105;
					}
					else
					{							/* String UNI */
						strcpy(calc_val, OutputXML ? "<NotImplemented/>String Formula"
                                                       : "***String Formula" );
						NotImplemented++;
					}
				}
				else
				{	/* Otherwise...this is a number */
					double n;
					getDouble(&working_buffer[6], &n);
					sprintf(calc_val, "%.15g", n);
					opcode = 0x0103;	/* To fix up OutputCellFormatted... */
				}
				add_wb_array(r, c, f, opcode, 0, calc_val, strlen(calc_val), 0, 0);
			}
			break;
		case 0x5C:	/* Author's name A.K.A. WRITEACCESS */
			working_buffer[bufidx++] = data;
			if ((bufidx == last)&&(author.str == 0))
			{
				if (file_version == EXCEL97)
				{
					author.len = getShort(&working_buffer[0]);
					if ((int)working_buffer[2] & 0x01)
					{
						author.len *= 2;
						author.uni = 2;
					}
					else
						author.uni = 0;
					author.str = (unsigned char *)malloc(author.len+1);
					if (author.str)
					{
						memcpy(author.str, &working_buffer[3], author.len);
						author.str[author.len] = 0;
					}
				}
				else if (file_version == EXCEL95)
				{
					author.len = working_buffer[0];
					author.str = (unsigned char *)malloc(author.len+1);
					if (author.str)
					{
						memcpy(author.str, &working_buffer[1], author.len);
						author.str[author.len] = 0;
					}
					author.uni = 0;
				}
			}
			break;
		case 0x08:	/* Row Data */
			/* There's actually some other interesting things
			   here that we're not collecting. For now, we'll
			   Just get the dimensions of the sheet. */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int i, r, fc, lc, d, xf;
				r = getShort(&working_buffer[0]);
				fc = getShort(&working_buffer[2]);
				lc = getShort(&working_buffer[4]) - 1;
				d = getShort(&working_buffer[12]);
				xf = getShort(&working_buffer[14]);

				if (ws_array[sheet_count] == 0)
					if (ws_init(sheet_count))
						return;

				if (r > ws_array[sheet_count]->biggest_row)
				{
					if (r < ws_array[sheet_count]->max_rows)
						ws_array[sheet_count]->biggest_row = r;
					else
					{	/* Resize the array... */
						if(resize_c_array(ws_array[sheet_count], MAX_ROWS, 0))
						{
							ws_array[sheet_count]->biggest_row = ws_array[sheet_count]->max_rows - 1;
							MaxRowExceeded = 1;
							return;
						}
						else
							ws_array[sheet_count]->biggest_row = r;
					}
				}

				if (lc > ws_array[sheet_count]->biggest_col)
				{
					if (lc < ws_array[sheet_count]->max_cols)
						ws_array[sheet_count]->biggest_col = lc;
					else
					{	/* Resize array... */
						if (resize_c_array(ws_array[sheet_count], 0, MAX_COLS))
						{
							ws_array[sheet_count]->biggest_col = ws_array[sheet_count]->max_cols - 1;
							MaxColExceeded = 1;
							lc = ws_array[sheet_count]->max_cols;
						}
						else
								ws_array[sheet_count]->biggest_col = lc;
					}
				}
				if ((fc < ws_array[sheet_count]->max_cols)&&(d & 0x0080))	/* fGhostDirty flag */
				{
					for (i=fc; i<lc; i++)
					{	/* Set the default attr... */
						update_cell_xf(r, i, xf, opcode);
					}
				}
			}
			break;
		case 0x22:	/* 1904 Flag - MacIntosh Dates or PC Dates */
			working_buffer[bufidx++] = data;
			if (bufidx == 2)
				DatesR1904 = getShort(&working_buffer[0]);
			break;
		case 0x085:	/* BoundSheet */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{	/* This is based on Office 97 info... */
				if ((working_buffer[4] & 0x0F) == 0)
				{	/* Worksheet as opposed to chart, etc */
					int len, uni=0;
					if (file_version == EXCEL97)
					{
						len = working_buffer[6];		/* FIXME: Check this !!! Was GetShort */
						if (working_buffer[7] & 0x01)
						{
							uni = 2;
							len = len<<1;
						}
						if (len != 0)
						{
							working_buffer[8 + len + 1] = 0;
							add_ws_title(uni, &working_buffer[8], len);
						}
					}
					else
					{
						len = working_buffer[6];
						if (len != 0)
						{
							working_buffer[7 + len + 1] = 0;
							add_ws_title(uni, &working_buffer[7], len);
						}
					}
				}
			}
			break;
		case 0x7E:	/* RK Number */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{	/* This is based on Office 97 info... */
				int r, c, f, t;
				long n, n2;		/* Must be signed long !!! */
				double d;
				char temp[64];
				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[2]);
				f = getShort(&working_buffer[4]);
				n = (unsigned long)getLong(&working_buffer[6]);
				t = n & 0x03;
				n2 = n>>2;
				switch (t)
				{
					case 0:
						RKtoDouble(n2, &d);
						sprintf(temp, "%.15g", d);
						break;
					case 1:
						RKtoDouble(n2, &d);
						sprintf(temp, "%.15g", d / 100.0);
						break;
					case 2:
						sprintf(temp, "%d", (int)n2);
						break;
					default:
						d = (float) n2;
						sprintf(temp, "%.15g", d / 100.0 );
						break;
				}
				add_wb_array(r, c, f, opcode, 0, temp, strlen(temp), 0, 0);
			}
			break;
		case 0xBC:		/* Shared Formula's */
/*			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int fr, lr, fc, lc, i, j;
				fr = getShort(&working_buffer[0]);
				lr = getShort(&working_buffer[2]);
				fc = working_buffer[4];
				lc = working_buffer[5];
				for (i=fr; i<=lr; i++)
				{
					for (j=fc; j<=lc; j++)
						add_wb_array(i, j, 0, opcode, 0, "***SHRFORMULA", 13);
				}
				NotImplemented = 1;
			}	*/
			break;
		case 0x21:		/* Arrays */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int fr, lr, fc, lc, i, j;
				fr = getShort(&working_buffer[0]);
				lr = getShort(&working_buffer[2]);
				fc = working_buffer[4];
				lc = working_buffer[5];
				for (i=fr; i<=lr; i++)
				{
					for (j=fc; j<=lc; j++)
						add_wb_array(i, j, 0, opcode, 0, "***Array", 8, 0, 0);
				}
				NotImplemented = 1;
			}
			break;
		case 0xBD:		/* MULRK */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int r, fc, lc, i;
				r = getShort(&working_buffer[0]);
				fc = getShort(&working_buffer[2]);
				lc = getShort(&working_buffer[last-2]);
				for (i=0; i<=(lc-fc); i++)
				{
					int t;
					long n, n2;		/* Must be signed long !!! */
					unsigned long f;
					double d;
					unsigned char temp[64];

					f = getShort(&working_buffer[4+(i*6)]);
					n = getLong(&working_buffer[6+(i*6)]);
					t = n & 0x03;
					n2 = n>>2;
					switch (t)
					{
						case 0:
							RKtoDouble(n2, &d);
							sprintf(temp, "%.15g", (float)d);
							break;
						case 1:
							RKtoDouble(n2, &d);
							sprintf(temp, "%.15g", (float)d / 100.0);
							break;
						case 2:
							sprintf(temp, " %d", (int)n2);
							break;
						default:
							d = (float) n2;
							sprintf(temp, "%.15g",  d / 100.0 );
							break;
					}
/*					printf("%08X %02X %s  %d  %d\n", n2, t, temp, r, fc+i); */
					add_wb_array(r, fc+i, f, opcode, 0, temp, strlen(temp), 0, 0);
				}
			}
			break;
		case 0xBE:		/* MULBLANK */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int r, fc, lc, j, f;
				r = getShort(&working_buffer[0]);
				fc = getShort(&working_buffer[2]);
				lc = getShort(&working_buffer[last-2]);
				for (j=0; j<=(lc-fc); j++)
				{	/* This just stores format strings... */
					f = getShort(&working_buffer[4+(j*2)]);
					add_wb_array(r, fc+j, f, opcode, 0, " ", 1, 0, 0);
				}
			}
			break;
		case 0x18:		/* Name UNI */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				char *ptr;
				working_buffer[bufidx] = 0;
				ptr = (char *)strstr(&working_buffer[15], "LastUpdate");
				if (ptr)
				{
					ptr += 13;
					lastUpdated = (char *)malloc(strlen(ptr)+1);
					if (lastUpdated)
						strcpy(lastUpdated, ptr);
				}
				else
				{
					ptr = (char *)strstr(&working_buffer[15], "Title");
					if (ptr)
					{
						ptr += 8;
						title = (char *)malloc(strlen(ptr)+1);
						if (title)
							strcpy(title, ptr);
					}
				}
			}
			break;
		case 0xE0:		/* Extended format */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				short fnt_idx;
				short fmt_idx;
				short gen;
				short align;
				short indent;
				short b_style;
				short b_l_color;
				long  b_t_color;
				short cell_color;

				fnt_idx = getShort(&working_buffer[0]);
				fmt_idx = getShort(&working_buffer[2]);
				gen = getShort(&working_buffer[4]);
				align  = getShort(&working_buffer[6]);
				indent  = getShort(&working_buffer[8]);
				b_style  = getShort(&working_buffer[10]);
				if (file_version == EXCEL95)
				{
					b_l_color  = 0; 
					b_t_color = 0; 
					cell_color  = getShort(&working_buffer[12]) & (short)0x1FFF;
				}
				else	/* Excel 97 + */
				{
					b_l_color  = getShort(&working_buffer[12]);
					b_t_color = getLong(&working_buffer[14]);
					cell_color  = getShort(&working_buffer[18]);
				}

				/* printf("XF:%02X FG:%02X BG:%02X\n", next_xf, cell_color&0x007F, (cell_color&0x1F80)>>7); */
				/* printf("XF:%02X M:%02X b_t:%04X<br>\n", next_xf, indent, b_t_color); */
				add_xf_array(fnt_idx, fmt_idx, gen, align, indent, b_style,
					b_l_color, b_t_color, cell_color);
			}
			break;
		case 0xE5:		/* CELL MERGE INSTRUCTIONS */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{
				int num, fr, lr, fc, lc, i, j, k;
				ws_array[sheet_count]->spanned = 1;
				num = getShort(&working_buffer[0]);
				if (ws_array[sheet_count]->c_array == 0)
					return;

				for (i=0; i<num; i++)
				{
					cell *c;
					fr = getShort(&working_buffer[2+(i*8)]);
					lr = getShort(&working_buffer[4+(i*8)]);
					fc = getShort(&working_buffer[6+(i*8)]);
					lc = getShort(&working_buffer[8+(i*8)]);
					if (sheet_count < (int)max_worksheets)
					{
						if (ws_array[sheet_count] == 0)
						{
							if (ws_init(sheet_count))
								return;
						}
						if (ws_array[sheet_count]->c_array)
						{
							for(j=fr; j<=lr; j++)
							{	/* For each row */
								for(k=fc; k<=lc; k++)
								{	/* for each column */
									c = ws_array[sheet_count]->c_array[(j*ws_array[sheet_count]->max_cols)+k];
									if (c != 0)
									{
										c->spanned = 1;
										c->rowspan = 0;
										if (k == fc)
											c->colspan = (lc-fc)+1;
										else
											c->colspan = 0;
									}
/*									else
									{	// Need to create one...
										printf("Bad One at:%d %d %d<br>\n", sheet_count, j, k);
									} */
								}
							}
						}
						/* Now reset the first one... */
/*						printf("s:%d fr:%d fc:%d lr:%d lc:%d<br>\n", sheet_count, fr, fc, lr, lc); */
						c = ws_array[sheet_count]->c_array[(fr*ws_array[sheet_count]->max_cols)+fc];
						if (c != 0)
						{
							c->spanned = 0;
							c->rowspan = lr-fr;
							c->colspan = lc-fc;
							if (c->rowspan)
								c->rowspan++;
							if (c->colspan)
								c->colspan++;
						}
					}
				}
			}
			break;
		case 0xB8:	/* Hyperlink */
			working_buffer[bufidx++] = data;
			if (bufidx == last)
			{	/* This is based on Office 97 info... */
				int r, c, off, len, uni_type;
				r = getShort(&working_buffer[0]);
				c = getShort(&working_buffer[4]);
				if (working_buffer[32] == 0xE0)
				{	/* Unicode format */
					len = getLong(&working_buffer[48]);
					off = 52;
					uni_type = 2;
				}
				else
				{	/* Ascii format */
					len = getLong(&working_buffer[50]);
					off = 54;
					uni_type = 0;
				}
				update_cell_hyperlink(r, c, &working_buffer[off], len, uni_type);
			}
			break;
		default:
			break;
	}
}

static short getShort(unsigned char *ptr)
{
	if (ptr == 0)
		return (short)0;

	return (short)((*(ptr+1)<<8)+*ptr);
}

static long getLong(unsigned char *ptr)
{
	if (ptr == 0)
		return 0L;

	return (*(ptr+3)<<24)+(*(ptr+2)<<16)+(*(ptr+1)<<8)+*ptr;
}

#ifndef WORDS_BIGENDIAN				/* Defined in <config.h> */
/* Little Endian - 0x86 family */
static void getDouble(unsigned char *ptr, double *d)
{
	int i;
	double dd;
	unsigned char *t = (unsigned char *)&dd;

	for (i=0; i<sizeof(double); i++)
		*(t+i) = *(ptr+i);

	*d = (double)dd;
}
#else
/* Big Endian version - UltraSparc's, etc. */
static void getDouble (unsigned char *ptr, double *d)
{
	int i;
	double dd;
	unsigned char *t = (char *)&dd;

	for (i=0; i<sizeof(double); i++)
		*(t+i) = *(ptr+sizeof(double) - 1 - i);

	*d = (double)dd;
}
#endif

#ifndef WORDS_BIGENDIAN				/* Defined in <config.h> */
/* Little Endian - 0x86 family */
static void RKtoDouble(unsigned long n, double *d)
{
	*((unsigned long *)d) = 0;
	*((unsigned long *)d+1) = n << 2;
}
#else
/* Big Endian version - UltraSparc's, etc. */
static void RKtoDouble(unsigned long n, double *d)
{
	double dd;
	unsigned long tmp;
	unsigned char *ptr = (char *)&n;

	tmp = (*(ptr+0)<<24)+(*(ptr+1)<<16)+(*(ptr+2)<<8)+(*(ptr+3));

	*((unsigned long *)&dd+1) = 0;
	*((unsigned long *)&dd) = tmp << 2;

	*d = dd;
}
#endif

/* returns 1 on error, 0 on success */
static int ws_init(int i)
{
	int j, k;
	if (i >= (int)max_worksheets)
		return 1;

	ws_array[i] = (work_sheet *)malloc(sizeof(work_sheet));
	if (ws_array[i])
	{
		ws_array[i]->spanned = 0;
		ws_array[i]->first_row = 0;
		ws_array[i]->biggest_row = -1;
		ws_array[i]->max_rows = MAX_ROWS;
		ws_array[i]->first_col = 0;
		ws_array[i]->biggest_col = -1;
		ws_array[i]->max_cols = MAX_COLS;
		uni_string_clear(&ws_array[i]->ws_title);
		ws_array[i]->c_array = (cell **)malloc(MAX_ROWS*MAX_COLS*sizeof(cell *));
		if (ws_array[i]->c_array == 0)
			return 1;
		for (j=0; j<MAX_ROWS; j++)
			for (k=0; k<MAX_COLS; k++)
				ws_array[i]->c_array[(j*ws_array[i]->max_cols)+k] = 0;
	}
	else
		return 1;

	return 0;
}

/* returns 1 on error, 0 on success */
static int realloc_worksheets(void)
{
	work_sheet **tws_array = (work_sheet **)realloc(ws_array,
				(max_worksheets * MAX_WORKSHEETS) * sizeof(work_sheet *));

	if (tws_array == NULL)
		return 1;
	else
	{	/* Next init the array... */
		unsigned int i;

		ws_array = tws_array;

		for (i=max_worksheets; i<max_worksheets+MAX_WORKSHEETS; i++)
			ws_array[i] = 0;

		max_worksheets += MAX_WORKSHEETS;
		last_sheet = max_worksheets - 1;
	}
	return 0;
}

static int resize_c_array(work_sheet *ws, int new_rows, int new_cols)
{
	cell **tc_array;
	if (ws == 0)
		return 1;
	if (ws->c_array == 0)
		return 1;

	tc_array = (cell **)malloc((ws->max_rows+new_rows)*(ws->max_cols+new_cols)*sizeof(cell *));

	if (tc_array == NULL)
		return 1;
	else
	{
		int j, k;

		memset(tc_array, 0, (ws->max_rows+new_rows)*(ws->max_cols+new_cols)*sizeof(cell *));
		for (j=0; j<(ws->max_rows); j++)
		{
			for (k=0; k<ws->max_cols; k++)
				tc_array[(j*(ws->max_cols+new_cols))+k] = ws->c_array[(j*ws->max_cols)+k];
		}
		ws->max_cols += new_cols;
		ws->max_rows += new_rows;
		free(ws->c_array);
		ws->c_array = tc_array;
	}
	return 0;
}

static void add_wb_array(int r, int c, unsigned long xf, int type, int uni,
					unsigned char *str, int len, unsigned int crun_cnt, unsigned char *fmt_run)
{
	work_sheet *ws;

	if ((sheet_count < 0)||(r < 0)||(c < 0))
		return;
	if (sheet_count >= (int)max_worksheets)
	{
		if (realloc_worksheets())
		{
			MaxWorksheetsExceeded = 1;
			return;
		}
	}
	if (ws_array[sheet_count] == 0)
	{
		if (ws_init(sheet_count))
			return;
	}
	if (r >= ws_array[sheet_count]->max_rows)
	{
		int diff = ((r-ws_array[sheet_count]->max_rows)/MAX_ROWS)+1;
		if(resize_c_array(ws_array[sheet_count], MAX_ROWS*diff, 0))
		{
			MaxRowExceeded = 1;
			return;
		}
	}
	if (c >= ws_array[sheet_count]->max_cols)
	{
		int diff = ((c-ws_array[sheet_count]->max_cols)/MAX_COLS)+1;
		if(resize_c_array(ws_array[sheet_count], 0, MAX_COLS*diff))
		{
			MaxColExceeded = 1;
			return;
		}
	}

	ws = ws_array[sheet_count];
	if (ws->c_array[(r*ws->max_cols)+c] == 0)
	{
		ws->c_array[(r*ws->max_cols)+c] = (cell *)malloc(sizeof(cell));
		if (ws->c_array[(r*ws->max_cols)+c])
		{
			if (str)
			{
				ws->c_array[(r*ws->max_cols)+c]->ustr.str = (unsigned char *)malloc(len+1);
				if (ws->c_array[(r*ws->max_cols)+c]->ustr.str)
				{
					memcpy(ws->c_array[(r*ws->max_cols)+c]->ustr.str, str, len);
					ws->c_array[(r*ws->max_cols)+c]->ustr.str[len] = 0;
				}
				ws->c_array[(r*ws->max_cols)+c]->ustr.uni = uni;
				ws->c_array[(r*ws->max_cols)+c]->ustr.len = len;
				if (fmt_run)
				{
					int rlen = crun_cnt*4;

					ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run = malloc(rlen);
					if (ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run)
					{
						memcpy(ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run, fmt_run, rlen);
						ws->c_array[(r*ws->max_cols)+c]->ustr.crun_cnt = crun_cnt;
					}
					else
						ws->c_array[(r*ws->max_cols)+c]->ustr.crun_cnt = 0;
				}
				else
				{
					ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run = 0;
					ws->c_array[(r*ws->max_cols)+c]->ustr.crun_cnt = 0;
				}
			}
			else
			{
				uni_string_clear(&ws->c_array[(r*ws->max_cols)+c]->ustr);
			}

			ws->c_array[(r*ws->max_cols)+c]->xfmt = xf;
			ws->c_array[(r*ws->max_cols)+c]->type = type;
			if (r > ws_array[sheet_count]->biggest_row)
				ws_array[sheet_count]->biggest_row = r;
			if (c > ws_array[sheet_count]->biggest_col)
				ws_array[sheet_count]->biggest_col = c;
			ws->c_array[(r*ws->max_cols)+c]->spanned = 0;
			ws->c_array[(r*ws->max_cols)+c]->rowspan = 0;
			ws->c_array[(r*ws->max_cols)+c]->colspan = 0;
			uni_string_clear(&ws->c_array[(r*ws->max_cols)+c]->h_link);
		}
	}
	else	/* Default attributes already copied */
	{
		if (str)
		{
			if (ws->c_array[(r*ws->max_cols)+c]->ustr.str == 0)
			{
				ws->c_array[(r*ws->max_cols)+c]->ustr.str = (unsigned char *)malloc(len+1);
				if (ws->c_array[(r*ws->max_cols)+c]->ustr.str)
				{
					memcpy(ws->c_array[(r*ws->max_cols)+c]->ustr.str, str, len);
					ws->c_array[(r*ws->max_cols)+c]->ustr.str[len] = 0;
				}
				ws->c_array[(r*ws->max_cols)+c]->ustr.len = len;
				ws->c_array[(r*ws->max_cols)+c]->ustr.uni = uni;
				if (fmt_run)
				{
					int rlen = crun_cnt*4;

					ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run = malloc(rlen);
					if (ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run)
					{
						memcpy(ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run, fmt_run, rlen);
						ws->c_array[(r*ws->max_cols)+c]->ustr.crun_cnt = crun_cnt;
					}
					else
						ws->c_array[(r*ws->max_cols)+c]->ustr.crun_cnt = 0;
				}
				else
				{
					ws->c_array[(r*ws->max_cols)+c]->ustr.fmt_run = 0;
					ws->c_array[(r*ws->max_cols)+c]->ustr.crun_cnt = 0;
				}
			}
		}
		else
		{
			if (ws->c_array[(r*ws->max_cols)+c]->ustr.str == 0)
			{
				ws->c_array[(r*ws->max_cols)+c]->ustr.len = 0;
				ws->c_array[(r*ws->max_cols)+c]->ustr.uni = 0;
			}
		}
		ws->c_array[(r*ws->max_cols)+c]->xfmt = xf;
		ws->c_array[(r*ws->max_cols)+c]->type = type;
		if (r > ws_array[sheet_count]->biggest_row)
			ws_array[sheet_count]->biggest_row = r;
		if (c > ws_array[sheet_count]->biggest_col)
			ws_array[sheet_count]->biggest_col = c;
		ws->c_array[(r*ws->max_cols)+c]->spanned = 0;
		ws->c_array[(r*ws->max_cols)+c]->rowspan = 0;
		ws->c_array[(r*ws->max_cols)+c]->colspan = 0;
	}
}

static void update_cell_xf(int r, int c, unsigned long xf, int type)
{
	work_sheet *ws;

	if ((sheet_count < 0)||(r < 0)||(c < 0))
		return;
	if (sheet_count >= (int)max_worksheets)
	{
		if (realloc_worksheets())
		{
			MaxWorksheetsExceeded = 1;
			return;
		}
	}
	if (ws_array[sheet_count] == 0)
	{
		if (ws_init(sheet_count))
			return;
	}
	if (r >= ws_array[sheet_count]->max_rows)
	{
		int diff = ((r-ws_array[sheet_count]->max_rows)/MAX_ROWS)+1;
		if(resize_c_array(ws_array[sheet_count], MAX_ROWS*diff, 0))
		{
			MaxRowExceeded = 1;
			return;
		}
	}
	if (c >= ws_array[sheet_count]->max_cols)
	{
		int diff = ((c-ws_array[sheet_count]->max_cols)/MAX_COLS)+1;
		if(resize_c_array(ws_array[sheet_count], 0, MAX_COLS*diff))
		{
			MaxColExceeded = 1;
			return;
		}
	}

	ws = ws_array[sheet_count];
	if (ws->c_array[(r*ws->max_cols)+c] == 0)
	{
		ws->c_array[(r*ws->max_cols)+c] = (cell *)malloc(sizeof(cell));
		if (ws->c_array[(r*ws->max_cols)+c])
		{
			uni_string_clear(&ws->c_array[(r*ws->max_cols)+c]->ustr);
			ws->c_array[(r*ws->max_cols)+c]->xfmt = xf;
			ws->c_array[(r*ws->max_cols)+c]->type = type;

			if (r > ws_array[sheet_count]->biggest_row)
				ws_array[sheet_count]->biggest_row = r;
			if (c > ws_array[sheet_count]->biggest_col)
				ws_array[sheet_count]->biggest_col = c;
			ws->c_array[(r*ws->max_cols)+c]->spanned = 0;
			ws->c_array[(r*ws->max_cols)+c]->rowspan = 0;
			ws->c_array[(r*ws->max_cols)+c]->colspan = 0;
			uni_string_clear(&ws->c_array[(r*ws->max_cols)+c]->h_link);
		}
	}
/*	else
	{
		printf("R:%02X C:%02X XF:%02X is:%02X\n",
			r, c, xf, ws->c_array[r][c]->xfmt);
	} */
}

static void update_cell_hyperlink(int r, int c, char *hyperlink, int len, int uni)
{
	work_sheet *ws;

	if ((sheet_count < 0)||(r < 0)||(c < 0))
		return;
	if (sheet_count >= (int)max_worksheets)
	{
		if (realloc_worksheets())
		{
			MaxWorksheetsExceeded = 1;
			return;
		}
	}
	if (ws_array[sheet_count] == 0)
	{
		if (ws_init(sheet_count))
			return;
	}
	if (r >= ws_array[sheet_count]->max_rows)
	{
		int diff = ((r-ws_array[sheet_count]->max_rows)/MAX_ROWS)+1;
		if(resize_c_array(ws_array[sheet_count], MAX_ROWS*diff, 0))
		{
			MaxRowExceeded = 1;
			return;
		}
	}
	if (c >= ws_array[sheet_count]->max_cols)
	{
		int diff = ((c-ws_array[sheet_count]->max_cols)/MAX_COLS)+1;
		if(resize_c_array(ws_array[sheet_count], 0, MAX_COLS*diff))
		{
			MaxColExceeded = 1;
			return;
		}
	}

	ws = ws_array[sheet_count];
	if (ws->c_array[(r*ws->max_cols)+c] == 0)
	{	/* should not get here, but just in case */
		return;
	}
	if (ws->c_array[(r*ws->max_cols)+c]->h_link.str == 0)
	{
		ws->c_array[(r*ws->max_cols)+c]->h_link.str = (unsigned char *)malloc(len);
		if (ws->c_array[(r*ws->max_cols)+c]->h_link.str)
			memcpy(ws->c_array[(r*ws->max_cols)+c]->h_link.str, hyperlink, len);
		ws->c_array[(r*ws->max_cols)+c]->h_link.uni = uni;
		if (uni < 2)
			ws->c_array[(r*ws->max_cols)+c]->h_link.len = len-1;
		else
			ws->c_array[(r*ws->max_cols)+c]->h_link.len = len-2;
	}
/*	else
	{
		printf("R:%02X C:%02X XF:%02X is:%s\n",
			r, c, xf, ws->c_array[r][c]->h_link.str);
	} */
}

static void add_str_array(int uni, unsigned char *str, int len, char *fmt_run, unsigned int crun_cnt)
{

	if ((str == 0)||(len == 0))
	{
		next_string++; /* increment for empty strings, too */
		return;
	}
	if (next_string >= max_strings)
	{
		uni_string **tstr_array;
		size_t new_size = (max_strings + MAX_STRINGS) * sizeof(uni_string *);

		tstr_array = (uni_string **)realloc(str_array, new_size);
		
		if (tstr_array == NULL)
		{
			MaxStringsExceeded = 1;
			fprintf(stderr, "%s: cannot allocate %d bytes for string storage %d: %s",
			    PRGNAME, new_size, errno, strerror(errno));
			return;
		}
		else
		{
			unsigned long i;
			
			str_array = tstr_array;
			
			for (i=max_strings; i<(max_strings + MAX_STRINGS); i++)
				str_array[i] = 0;
			
			max_strings += MAX_STRINGS;
		}
	}
	
	if (str_array[next_string] == 0)
	{
		str_array[next_string] = (uni_string *)malloc(sizeof(uni_string));
		if (str_array[next_string])
		{
			str_array[next_string]->str = (unsigned char *)malloc(len+1);
			if (str_array[next_string]->str)
			{
				memcpy(str_array[next_string]->str, str, len);
				str_array[next_string]->str[len] = 0;
				str_array[next_string]->len = len;
				str_array[next_string]->uni = uni;
				if (crun)
				{
					int rlen = crun_cnt*4;

					str_array[next_string]->fmt_run = malloc(rlen);
					if (str_array[next_string]->fmt_run)
					{
						memcpy(str_array[next_string]->fmt_run, fmt_run, rlen);
						str_array[next_string]->crun_cnt = crun_cnt;
					}
					else
						str_array[next_string]->crun_cnt = 0;
				}
				else
				{
					str_array[next_string]->fmt_run = 0;
					str_array[next_string]->crun_cnt = 0;
				}
			}
		}
	}
	next_string++;
}

static void add_font(int size, int attr, int c_idx, int bold, int super, int underline, 
int uni, unsigned char *n, int len)
{
	if (n == 0)
		return;
	if (next_font >= max_fonts)
	{
		font_attr **tfont_array;
		fnt_cnt *tf_cnt;
		tfont_array = (font_attr **)realloc(font_array, (max_fonts * MAX_FONTS) * sizeof(font_attr *));
		tf_cnt = (fnt_cnt *)realloc(f_cnt, (max_fonts * MAX_FONTS) * sizeof(fnt_cnt));
		
		if ((tf_cnt == NULL) || (tfont_array == NULL))
		{
			MaxFontsExceeded = 1;
			return;
		}
		else
		{	/* Next init the array... */
			unsigned int i;
			
			font_array = tfont_array;
			f_cnt = tf_cnt;
			
			for (i=max_fonts; i<max_fonts+MAX_FONTS; i++)
			{
				font_array[i] = 0;
				f_cnt[i].name = 0;
			}
			max_fonts += MAX_FONTS;
		}
	}

	if (font_array[next_font] == 0)
	{
		font_array[next_font] = (font_attr *)malloc(sizeof(font_attr));
		if (font_array[next_font])
		{
			font_array[next_font]->name.str = (unsigned char *)malloc(len+1);
			if (font_array[next_font]->name.str)
			{
				font_array[next_font]->attr = attr;
				font_array[next_font]->c_idx = c_idx;
				font_array[next_font]->bold = bold;
				font_array[next_font]->super = super;
				font_array[next_font]->underline = underline;
				font_array[next_font]->name.uni = uni;
				memcpy(font_array[next_font]->name.str, n, len);
				font_array[next_font]->name.str[len] = 0;	
				font_array[next_font]->name.len = len;
				
				/* We will "pre-digest" the font size.. */
				if (size >= 0x02D0)		/* 36 pts */
					font_array[next_font]->size = 7;
				else if (size >= 0x01E0)	/* 24 pts */
					font_array[next_font]->size = 6;
				else if (size >= 0x0168)	/* 18 pts */
					font_array[next_font]->size = 5;
				else if (size >= 0x00F0)	/* 12 pts */
					font_array[next_font]->size = 4;
				else if (size >= 0x00C8)	/* 10 pts */
					font_array[next_font]->size = 3;
				else if (size >= 0x00A0)	/* 8 pts */
					font_array[next_font]->size = 2;
				else
					font_array[next_font]->size = 1;
			}
		}
	}
	next_font++;
	if (next_font == 4)		/* Per the doc's - number 4 doesn't exist. */
		next_font++;
}

static void add_ws_title(int uni, unsigned char *n, int len)
{
	if (n == 0)
		return;

	if (next_ws_title >= max_worksheets)
	{
		if (realloc_worksheets())
		{
			MaxWorksheetsExceeded = 1;
			return;
		}
	}

	if (ws_array[next_ws_title] == 0)
	{
		if (ws_init(next_ws_title))
			return;
	}
	if (ws_array[next_ws_title]->ws_title.str == 0)
	{
		ws_array[next_ws_title]->ws_title.str = (unsigned char *)malloc(len+1);
		if (ws_array[next_ws_title]->ws_title.str)
		{
			ws_array[next_ws_title]->ws_title.uni = uni;
			memcpy(ws_array[next_ws_title]->ws_title.str, n, len);
			ws_array[next_ws_title]->ws_title.str[len] = 0;
			ws_array[next_ws_title]->ws_title.len = len;
			ws_array[next_ws_title]->ws_title.crun_cnt = 0;
			ws_array[next_ws_title]->ws_title.fmt_run = 0;
		}
	}
	next_ws_title++;
}

static void add_xf_array(short fnt_idx, short fmt_idx, short gen, short align,
	short indent, short b_style, short b_l_color, long  b_t_color, short cell_color)
{
	if (next_xf >= max_xformats)
	{
		xf_attr **txf_array;
		
		txf_array = (xf_attr **)realloc(xf_array, (max_xformats + MAX_XFORMATS) * sizeof(xf_attr *));
		if (txf_array == NULL)
		{
			MaxXFExceeded = 1;
			return;
		}
		else
		{
			unsigned int i;
			
			xf_array = txf_array;
			
			for (i=max_xformats; i<(max_xformats + MAX_XFORMATS); i++)
				xf_array[i] = 0;
			
			max_xformats += MAX_XFORMATS;
		}
	}

	if (xf_array[next_xf] == 0)
	{
		xf_array[next_xf] = (xf_attr *)malloc(sizeof(xf_attr));
		if (xf_array[next_xf])
		{
			xf_array[next_xf]->fnt_idx = fnt_idx;
			xf_array[next_xf]->fmt_idx = fmt_idx;
			xf_array[next_xf]->gen = gen;
			xf_array[next_xf]->align = align;
			xf_array[next_xf]->indent = indent;
			xf_array[next_xf]->b_style = b_style;
			xf_array[next_xf]->b_l_color = b_l_color;
			xf_array[next_xf]->b_t_color = b_t_color;
			xf_array[next_xf]->cell_color = cell_color;
		}
		next_xf++;
	}
}

static void decodeBoolErr(unsigned int value, int flag, unsigned char *str)
{
	if (str == 0)
		return;
		
	if (flag == 0)
	{
		if (value == 1)
			strcpy(str, "TRUE");
		else
			strcpy(str, "FALSE");		
	}
	else
	{
		switch(value)
		{	
			case 0x00:
				strcpy(str, "#NULL!");		
				break;
			case 0x07:
				strcpy(str, "#DIV/0!");
				break;
			case 0x0F:
				strcpy(str, "#VALUE!");
				break;
			case 0x17:
				strcpy(str, "#REF!");
				break;
			case 0x1D:
				strcpy(str, "#NAME?");
				break;
			case 0x24:
				strcpy(str, "#NUM!");
				break;
			case 0x2A:
				strcpy(str, "#N/A");
				break;
			default:
				strcpy(str, "#ERR");		
				break;
		}
	}
}

static int IsCellNumeric(cell *c)
{
	int ret_val = 0;
	
	switch (c->type & 0x00FF)
	{
		case 0x02:	/* Int */
		case 0x03:	/* Float */
	/*	case 0x06: */	/* Formula */
	/*	case 0x08: */
		case 0x7E:	/* RK */
	/*	case 0xBC: */
	/*	case 0x21: */
		case 0xBD:	/* MulRK */
			ret_val = 1;
			break;
 		default:
			break;
	}
	return ret_val;
}

/*	Safe == 0 not safe at all.
	Safe == 1 extended format is OK
	Safe == 2 Fonts OK */
static int IsCellSafe(cell *c)
{
	int safe = 0;
	
	if (c->xfmt < next_xf)
	{
		if (xf_array[c->xfmt])
		{
			safe = 1;
			if (xf_array[c->xfmt]->fnt_idx < next_font)
			{
				if (font_array[xf_array[c->xfmt]->fnt_idx])
					safe = 2;
			}
		}
	}
	return safe;
}

static int IsCellFormula(cell *c)
{
	if ((c->type > 0x0100)||(c->type == 0x0006))
		return 1;
	else
		return 0;
}

static void output_cell(cell *c, int xml)
{
	html_attr h;

	if (c == NULL)
		printf( xml ? "" : "<TD>&nbsp;");
	else if (c->spanned != 0)
		return;
	else
	{	/* Determine whether or not its of numeric origin.. */
		int numeric = IsCellNumeric(c);	/* 0=Text 1=Numeric */
		html_flag_init(&h);
		if (c->xfmt == 0)
		{	/* Unknown format... */
			printf( xml ? "" : "<TD>");		/* This section doesn't use Unicode */
			if (c->ustr.str)
				OutputString(&(c->ustr));
			else
				printf( xml ? "" : "&nbsp;");
		}
		else
		{	/* This is the BIFF7 & 8 stuff... */
			int safe;
			int nullString = 1;

			safe = IsCellSafe(c);

			if (c->ustr.str)
			{
				if (c->ustr.uni < 2)		/* UNI? */
					nullString = null_string(c->ustr.str);
				else
					nullString = 0;
			}

			/* First take care of text color & alignment */
			printf( xml ? "" : "<TD");
			if ((c->rowspan != 0)||(c->colspan != 0))
			{
				if (c->colspan)
					printf( xml ? "<colspan>%d</colspan>" : " COLSPAN=\"%d\"", c->colspan);
				if (c->rowspan)
					printf( xml ? "<rowspan>%d</rowspan>" : " ROWSPAN=\"%d\"", c->rowspan);
			}
			if ((safe > 0)&&(!nullString))
			{
				switch(xf_array[c->xfmt]->align & 0x0007)
				{	/* Override default table alignment when needed */
					case 2:
					case 6:		/* Center across selection */
						if (strcmp(default_alignment, "center") != 0)
							printf( xml ? "" : " ALIGN=\"center\"");
						break;
					case 0:		/* General alignment */
						if (numeric)						/* Numbers */
						{
							if (strcmp(default_alignment, "right") != 0)
								printf( xml ? "" : " ALIGN=\"right\"");
						}
						else if ((c->type & 0x00FF) == 0x05)
						{			/* Boolean */
							if (strcmp(default_alignment, "center") != 0)
								printf( xml ? "" : " ALIGN=\"center\"");
						}
						else
						{
							if (strcmp(default_alignment, "left") != 0)
								printf( xml ? "" : " ALIGN=\"left\"");
						}
						break;
					case 3:
						if (strcmp(default_alignment, "right") != 0)
							printf( xml ? "" : " ALIGN=\"right\"");
						break;
					case 1:
					default:
						if (strcmp(default_alignment, "left") != 0)
							printf( xml ? "" : " ALIGN=\"left\"");
						break;
				}
				switch((xf_array[c->xfmt]->align & 0x0070)>>4)
				{
					case 0:
						printf( xml ? "" : " VALIGN=\"top\"");
						break;
					case 1:
						printf( xml ? "" : " VALIGN=\"center\"");
						break;
					case 2:		/* General alignment */
						if (safe > 1)
						{
							if ((font_array[xf_array[c->xfmt]->fnt_idx]->super & 0x0003) == 0x0001)
								printf( xml ? "" : " VALIGN=\"top\"");	/* Superscript */
						}
						break;
					default:
						if (safe > 1)
						{
							if ((font_array[xf_array[c->xfmt]->fnt_idx]->super & 0x0003) == 0x0001)
								printf( xml ? "" : " VALIGN=\"top\"");	/* Superscript */
						}
						break;
				}
			}
			/* Next do the bgcolor... BGCOLOR=""   */
			if (safe && use_colors)
			{
				int fgcolor;
/*				int bgcolor = (xf_array[c->xfmt]->cell_color & 0x3F80) >> 7; */
				fgcolor = (xf_array[c->xfmt]->cell_color & 0x007F);
/*				printf("XF:%X BG Color:%d, FG Color:%d", c->xfmt, bgcolor, fgcolor); */

				/* Might be better by blending bg & fg colors?
				   If valid, fgcolor != black and fgcolor != white */
				if ((fgcolor < MAX_COLORS)&&(fgcolor != 0x40)&&(fgcolor != 0x09))
						    if( ! xml ) { printf(" BGCOLOR=\"%s\"", colorTab[fgcolor]); }
			}

			/* Next set the border color... */
			if (safe && use_colors)
			{
				int lcolor, rcolor, tcolor, bcolor;
				lcolor = xf_array[c->xfmt]->b_l_color & 0x007F;
				rcolor = (xf_array[c->xfmt]->b_l_color & 0x3F80) >> 7;
				tcolor = xf_array[c->xfmt]->b_t_color & 0x007F;
				bcolor = (xf_array[c->xfmt]->b_t_color & 0x3F80) >> 7;
				if (((lcolor & rcolor & tcolor & bcolor) == lcolor)&&(lcolor < MAX_COLORS))
				{	/* if they are all the same...do it...that is if its different from BLACK */
					if ((strcmp(colorTab[lcolor], "000000") != 0)&&(strcmp(colorTab[lcolor], "FFFFFF") != 0))
								  if( !xml ) { printf(" BORDERCOLOR=\"%s\"", colorTab[lcolor]); }
				}
			}

			/* Close up the <TD>... */
			printf(xml ? "" : ">");

			/* Next set font properties */
			if (safe > 1 && !xml )
			{
				if (!nullString)
					output_start_font_attribute(&h, xf_array[c->xfmt]->fnt_idx);
			}

			/* Finally, take care of font modifications */
			if ((safe > 1)&&(!nullString))
			{
				if ((font_array[xf_array[c->xfmt]->fnt_idx]->underline&0x0023) > 0)
				{
					if (c->h_link.str)
					{
						printf("<A href=\"");
						OutputString(&(c->h_link));
						printf("\">");
						h.uflag = 2;
					}
					else
					{
						printf("<U>");
						h.uflag = 1;
					}
				}
				output_start_html_attr(&h, xf_array[c->xfmt]->fnt_idx, 0);
			}
			if (c->ustr.str)
			{
				if (safe)
					output_formatted_data(&(c->ustr), xf_array[c->xfmt]->fmt_idx, numeric, IsCellFormula(c));
				else
					OutputString(&(c->ustr));
			}
			else
				printf( xml ? "" : "&nbsp;");
/*			printf(" T:%02X", c->type & 0x00FF); */
		}

		/* Now close the tags... */
		output_end_html_attr(&h);
		if (h.fflag)
			printf("</FONT>");
	}

	if (!aggressive)
		printf( xml ? "" : "</TD>\n");
}

static void output_formatted_data(uni_string *u, unsigned int idx, int numeric, int formula)
{
	if ((idx < max_xformats)&&(u->str))
	{
		if ((formula_warnings)&&(formula))
		{
			if( OutputXML )
				printf( "<NotAccurate/>" );
			else
				printf("** ");
			notAccurate++;
		}
		if (numeric)
		{
			int year, month, date, num;
			double dnum;
			int hr, minu, sec, msec;

/*			printf("idx:%d ", idx);	*/
			switch (idx)
			{
				case 0x00:	/* General */
					dnum = atof(u->str);
					printf("%.15g", dnum);
					break;
				case 0x01:	/* Number 0 */
					dnum = atof(u->str);
					printf("%0.0f", dnum);
					break;
				case 0x02:	/* Number	0.00 */
					dnum = atof(u->str);
					printf("%0.2f", dnum);
					break;
				case 0x03:	/* Number w/comma	0,000 */
					PrintFloatComma("%0.0f", 0, (double)atof(u->str));
					break;
				case 0x04:	/* Number w/comma	0,000.00 */
					PrintFloatComma("%0.2f", 0, (double)atof(u->str));
					break;
				case 0x05:	/* Currency, no decimal */
					PrintFloatComma("%0.0f", 1, (double)atof(u->str));
					break;
				case 0x06:	/* Currency, no decimal Red on Neg */
					PrintFloatComma("%0.0f", 1, (double)atof(u->str));
					break;
				case 0x07:	/* Currency with decimal */
					PrintFloatComma("%0.2f", 1, (double)atof(u->str));
					break;
				case 0x08:	/* Currency with decimal Red on Neg */
					PrintFloatComma("%0.2f", 1, (double)atof(u->str));
					break;
				case 0x09:	/* Percent 0% */
					if (Csv)
						printf("\"");
					dnum = 100.0*atof(u->str);
					printf("%.0f%%", dnum);
					if (Csv)
						printf("\"");
					break;
				case 0x0A:	/* Percent 0.00% */
					if (Csv)
						printf("\"");
					dnum = 100.0*atof(u->str);
					printf("%0.2f%%", dnum);
					if (Csv)
						printf("\"");
					break;
				case 0x0B:	/* Scientific 0.00+E00 */
					if (Csv)
						printf("\"");
					dnum = atof(u->str);
					printf("%0.2E", dnum);
					if (Csv)
						printf("\"");
					break;
				case 0x0C:	/* Fraction 1 number  e.g. 1/2, 1/3 */
					if (Csv)
						printf("\"");
					dnum = atof(u->str);
					print_as_fraction(dnum, 1);
					if (Csv)
						printf("\"");
					break;
				case 0x0D:	/* Fraction 2 numbers  e.g. 1/50, 25/33 */
					if (Csv)
						printf("\"");
					dnum = atof(u->str);
					print_as_fraction(dnum, 2);
					if (Csv)
						printf("\"");
					break;
				case 0x0E:	/* Date: m-d-y */
					if (Csv)
						printf("\"");
					num = atoi(u->str);
					NumToDate(num, &year, &month, &date);
					printf("%d-%d-%02d", month, date, year);
					if (Csv)
						printf("\"");
					break;
				case 0x0F:	/* Date: d-mmm-yy */
					if (Csv)
						printf("\"");
					num = atoi(u->str);
					NumToDate(num, &year, &month, &date);
					printf("%d-%s-%02d", date, month_abbr[month-1], year);
					if (Csv)
						printf("\"");
					break;
				case 0x10:	/* Date: d-mmm */
					if (Csv)
						printf("\"");
					num = atoi(u->str);
					NumToDate(num, &year, &month, &date);
					printf("%d-%s", date, month_abbr[month-1]);
					if (Csv)
						printf("\"");
					break;
				case 0x11:	/* Date: mmm-yy */
					if (Csv)
						printf("\"");
					num = atoi(u->str);
					NumToDate(num, &year, &month, &date);
					printf("%s-%02d", month_abbr[month-1], year);
					if (Csv)
						printf("\"");
					break;
				case 0x12:	/* Time: h:mm AM/PM */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, 0, 0);
					if (hr == 0)
						printf("12:%02d AM", minu);
					else if (hr < 12)
						printf("%d:%02d AM", hr, minu);
					else if (hr == 12)
						printf("12:%02d PM", minu);
					else
						printf("%d:%02d PM", hr-12, minu);
					if (Csv)
						printf("\"");
					break;
				case 0x13:	/* Time: h:mm:ss AM/PM */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, &sec, 0);
					if (hr == 0)
						printf("12:%02d:%02d AM", minu, sec);
					else if (hr < 12)
						printf("%d:%02d:%02d AM", hr, minu, sec);
					else if (hr == 12)
						printf("12:%02d:%02d PM", minu, sec);
					else
						printf("%d:%02d:%02d PM", hr-12, minu, sec);
					if (Csv)
						printf("\"");
					break;
				case 0x14:	/* Time: h:mm */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, 0, 0);
					printf("%d:%02d", hr, minu);
					if (Csv)
						printf("\"");
					break;
				case 0x15:	/* Time: h:mm:ss */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, &sec, 0);
					if ((hr != 0)||(minu != 0)||(sec != 0))
						printf("%d:%02d:%02d", hr, minu, sec);
					else
					{
						if (Ascii)
							putchar(' ');
						else
							printf(OutputXML ? "" : "&nbsp;");
					}
					if (Csv)
						printf("\"");
					break;
				case 0x25:	/* Number with comma, no decimal */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.0f", 0, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x26:	/* Number with comma, no decimal, red on negative */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.0f", 0, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x27:	/* Number with comma & decimal */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.2f", 0, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x28:	/* Number with comma & decimal, red on negative */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.2f", 0, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x29:	/* Number with comma, no decimal */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.2f", 0, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x2a:	/* Currency, no decimal */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.0f", 1, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x2B:	/* Number w/comma & decimal	0,000.00 */
					if (Csv)
						printf("\"");
					PrintFloatComma("%0.2f", 0, (double)atof(u->str));
					if (Csv)
						printf("\"");
					break;
				case 0x2C:	/* Accounting Currency	$0,000.00 */
				{
					double acc_val = atof(u->str);
					if (Csv)
						printf("\"");
					if (acc_val < 0.0)
						PrintFloatComma(" (%0.2f)", 1, fabs(acc_val));
					else
						PrintFloatComma(" %0.2f", 1, acc_val);
					if (Csv)
						printf("\"");
					break;
				}
				case 0x2D:	/* Time: mm:ss */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, &sec, 0);
					printf("%02d:%02d", minu, sec);
					if (Csv)
						printf("\"");
					break;
				case 0x2E:	/* Time: [h]:mm:ss */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, &sec, 0);
					if (hr)
						printf("%d:%02d:%02d", hr, minu, sec);
					else
						printf("%02d:%02d", minu, sec);
					if (Csv)
						printf("\"");
					break;
				case 0x2F:	/* Time: mm:ss.0 */
					if (Csv)
						printf("\"");
					FracToTime(u->str, &hr, &minu, &sec, &msec);
					printf("%02d:%02d.%01d", minu, sec, msec);
					if (Csv)
						printf("\"");
					break;
				case 0x31:	/* Text - if we are here...its a number */
					dnum = atof(u->str);
					printf("%g", dnum);
					break;
				default:	/* Unsupported...but, if we are here, its a number */
					{
						char *ptr = strchr(u->str, '.');
						if( OutputXML )
							printf( "<NoFormat/>" );
						if (ptr)
						{
							dnum = atof(u->str);
							printf("%.15g %s", dnum, OutputXML ? "" : "*" );
						}
						else
							printf("%d %s", atoi(u->str), OutputXML ? "" : "*" );
/*						printf(" F:%02X", idx); */
						NoFormat++ ;
					}
					break;
			}
		}
		else	/* Text data */
			OutputString(u);
	}
	else	/* Error handling just dump it. */
		OutputString(u);
}

static void NumToDate(int num, int *year, int *month, int *day)
{
	int t, i, y = 0;

	num = num%36525;	/* Trim century */
	while (num > (((y%4) == 0) ? 366 : 365))
		num -= ((y++%4) == 0) ? 366 : 365;

	*year = y;
	t = num;
	if (DatesR1904)
		*year += 4;		/* Adjust for McIntosh... */
	if ((*year%4) == 0)
	{	/* Leap Year */
		for (i=0; i<12; i++)
		{
			if (t <= ldays[i])
				break;
			t -= ldays[i];
		}
	}
	else
	{
		for (i=0; i<12; i++)
		{
			if (t <= ndays[i])
				break;
			t -= ndays[i];
		}
	}
	/* Some fixups... */
	*month = 1+i;
	if (t == 0)
		t = 1;
	*day = t;
	*year = *year % 100;
}

static void FracToTime(char *cnum, int *hr, int *minut, int *sec, int *msec)
{
	int Hr, Min, Sec, Msec;
	double fnum, tHr, tMin, tSec, tMsec;

	if (msec)
		fnum = atof(&cnum[0])+(0.05 / 86400.0);	/* Round off to 1/10th seconds */
	else if (sec)
		fnum = atof(&cnum[0])+(0.5 / 86400.0);	/* Round off to seconds */
	else
		fnum = atof(&cnum[0])+(30 / 86400.0);	/* Round off to minutes */
	tHr = 24.0 * fnum;
	Hr = (int)tHr;
	tMin = (tHr - (double)Hr) * 60.0;
	Min = (int)tMin;
	tSec = (tMin - (double)Min) * 60.0;
	Sec = (int)tSec;
	tMsec = (tSec - (double)Sec) * 10.0;
	Msec = (int)tMsec;

	Hr = Hr%24;	/* Fix roll-overs */
	if (hr)
		*hr = Hr;
	if (minut)
		*minut = Min;
	if (sec)
		*sec = Sec;
	if (msec)
		*msec = Msec;
}

static void PrintFloatComma(char *fformat, int is_currency, double d)
{
	int len, int_len, dec_len;
	char *ptr2, buf[64];

	sprintf(buf, fformat, fabs(d));
	len = strlen(buf);
	ptr2 = strchr(buf, '.');
	if (ptr2)
	{
		int_len = ptr2 - buf;
		dec_len = len - int_len;
	}
	else
	{
		int_len = len;
		dec_len = 0;
	}

	if (int_len > 3)
	{	/* we have to do it the hard way... */
		char rbuf[64], buf2[64];
		int neg, i, j, count=0;

		if (d < 0.0)
			neg = 1;
		else
			neg = 0;

		for (i=0, j=len-1; i<len;i++, j--)
			rbuf[i] = buf[j];
		rbuf[len] = 0;
		if (ptr2)
		{
			memcpy(buf2, rbuf, dec_len);
			i = dec_len;
			j = dec_len;
		}
		else
		{
			i = 0;
			j = 0;
		}

		for ( ;i<len;i++, j++)
		{
			buf2[j] = rbuf[i];
			count++;
			if ((count%3)==0)
			{
				if (isdigit(rbuf[i]))
				{
					if (dec_len)
					{
						if (count<(len-dec_len))
							buf2[++j] = ',';
					}
					else
					{
						if (count<len)
							buf2[++j] = ',';
					}
				}
			}
		}
		if (neg)
			buf2[j++] = '-';
		if (is_currency)
			buf2[j++] = (char)currency_symbol;
		buf2[j] = 0;

		len = strlen(buf2);
		for (i=0, j=len-1; i<len;i++, j--)
			buf[i] = buf2[j];
		buf[len] = 0;
		printf("%s", buf);
	}
	else	/* too short for commas, just output it as is. */
	{
		if (is_currency)
			putchar((char)currency_symbol);
		printf(fformat, d);
	}
}

static void print_as_fraction(double d, int digits)
{
	double i, j, w, r, closest=1.0, lim = 9.0;
	int ci=1, cj=1;

	/* Take care of negative sign */
	if (digits == 2)
		lim = 99.0;
	if (d < 0)
		putchar('-');

	/** take care of whole number part */
	w = fabs(d);
	if (w >= 1.0)
	{
		int n = (int)w;
		printf("%d ", n);
		r = w - (double)n;
	}
	else
		r = w;

	/* Get closest fraction - brute force */
	for (j=lim; j>0.0; j--)
	{
		for (i=lim; i>=0.0; i--)
		{
			if ( fabs((i/j)-r) <= closest)
			{
				closest = fabs((i/j)-r);
				ci = (int)i;
				cj = (int)j;
			}
		}
	}

	/* Done, print it... */
	if (ci != 0)
		printf("%d/%d", ci, cj);
}

static void trim_sheet_edges(unsigned int sheet)
{
	cell *ce;
	int r, c, not_done = 1;

	if ((sheet >= max_worksheets)||(ws_array[sheet] == 0)||
		(trim_edges == 0)||(ws_array[sheet]->spanned))
		return;
	if (ws_array[sheet]->c_array == 0)
		return;

	/* First find top edge */
	for (r=ws_array[sheet]->first_row; r<=ws_array[sheet]->biggest_row; r++)
	{
		for (c=ws_array[sheet]->first_col; c<=ws_array[sheet]->biggest_col; c++)
		{	/* This stuff happens for each cell... */
			ce = ws_array[sheet]->c_array[(r*ws_array[sheet]->max_cols)+c];
			if (ce)
			{
				if (ce->ustr.str)
				{
					if (!null_string(ce->ustr.str))
					{
						not_done = 0;
						break;
					}
				}
			}
		}
		if (!not_done) break;
	}
	ws_array[sheet]->first_row = r;

	/* Second Find bottom edge */
	not_done = 1;
	for (r=ws_array[sheet]->biggest_row; r>ws_array[sheet]->first_row; r--)
	{
		for (c=ws_array[sheet]->first_col; c<=ws_array[sheet]->biggest_col; c++)
		{	/* This stuff happens for each cell... */
			ce = ws_array[sheet]->c_array[(r*ws_array[sheet]->max_cols)+c];
			if (ce)
			{
				if (ce->ustr.str)
				{
					if (!null_string(ce->ustr.str))
					{
						not_done = 0;
						break;
					}
				}
			}
		}
		if (!not_done) break;
	}
	ws_array[sheet]->biggest_row = r;

	/* Third find left edge */
	not_done = 1;
	for (c=ws_array[sheet]->first_col; c<=ws_array[sheet]->biggest_col; c++)
	{
		for (r=ws_array[sheet]->first_row; r<=ws_array[sheet]->biggest_row; r++)
		{	/* This stuff happens for each cell... */
			ce = ws_array[sheet]->c_array[(r*ws_array[sheet]->max_cols)+c];
			if (ce)
			{
				if (ce->ustr.str)
				{
					if (!null_string(ce->ustr.str))
					{
						not_done = 0;
						break;
					}
				}
			}
		}
		if (!not_done) break;
	}
	ws_array[sheet]->first_col = c;

	/* Last, find right edge */
	not_done = 1;
	for (c=ws_array[sheet]->biggest_col; c>ws_array[sheet]->first_col; c--)
	{
		for (r=ws_array[sheet]->first_row; r<=ws_array[sheet]->biggest_row; r++)
		{	/* This stuff happens for each cell... */
			ce = ws_array[sheet]->c_array[(r*ws_array[sheet]->max_cols)+c];
			if (ce)
			{
				if (ce->ustr.str)
				{
					if (!null_string(ce->ustr.str))
					{
						not_done = 0;
						break;
					}
				}
			}
		}
		if (!not_done) break;
	}
	ws_array[sheet]->biggest_col = c;
}

/***************
* This function figure's out what the best font & alignment
* is for the current table. It then sets the default_font
* and default_alignment.
****************/
static void update_default_font(unsigned int sheet)
{
	cell *ce;
	int r, c, f;

	if ((sheet >= max_worksheets)||(ws_array[sheet] == 0))
		return;
	if (ws_array[sheet]->c_array == 0)
		return;

	/* Clear the book-keeping info... */
	for (r=0; r<MAX_FONTS; r++)
	{
		f_cnt[r].cnt = 0;
		if (f_cnt[r].name)
		{
			if (f_cnt[r].name->str)
				free(f_cnt[r].name->str);
			free(f_cnt[r].name);
			f_cnt[r].name = 0;
		}
	}
	if (default_font.str)
		free(default_font.str);
	for (r=0; r<7; r++)
		fnt_size_cnt[r] = 0;

	/* Now check each cell to see what its using. */
	for (r=ws_array[sheet]->first_row; r<=ws_array[sheet]->biggest_row; r++)
	{
		for (c=ws_array[sheet]->first_col; c<=ws_array[sheet]->biggest_col; c++)
		{	/* This stuff happens for each cell... */
			ce = ws_array[sheet]->c_array[(r*ws_array[sheet]->max_cols)+c];
			if (ce)
			{
				if ((ce->xfmt < next_xf)&&(ce->ustr.str))
				{
					if (strcmp(ce->ustr.str, "&nbsp;"))
					{
						if (ce->xfmt < next_xf)
						{
							if (xf_array[ce->xfmt])
							{
								unsigned int fn = xf_array[ce->xfmt]->fnt_idx;
								if (fn < next_font)
								{
									if (font_array[fn])
									{
										if (font_array[fn]->name.str)
										{
											/* Here's where we check & increment count... */
											incr_f_cnt(&(font_array[fn]->name));
											if ((font_array[fn]->size < 8)&&(font_array[fn]->size))
												fnt_size_cnt[font_array[fn]->size-1]++;
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	f = get_default_font();
	if (f == -1)
	{
		default_font.str = (char *)malloc(6);
		if (default_font.str)
		{
			strcpy(default_font.str, "Arial");
			default_font.uni = 0;
			default_font.len = 5;
		}
	}
	else
	{
		default_font.str = (char *)malloc(f_cnt[f].name->len+1);
		if (default_font.str)
		{
			memcpy(default_font.str, f_cnt[f].name->str, f_cnt[f].name->len);
			default_font.str[f_cnt[f].name->len] = 0;
			default_font.uni = f_cnt[f].name->uni;
			default_font.len = f_cnt[f].name->len;
		}
	}

	/* Find the font size with the most counts...
	   Just re-using variables, c - max cnt, f = position of max cnt */
	c = 0;
	f = 3;
	for (r=0; r<7; r++)
	{
		if (fnt_size_cnt[r] > c)
		{
			c = fnt_size_cnt[r];
			f = r;
		}
	}
	if (fnt_size_cnt[2] == c)		/* favor size 3... */
		default_fontsize = 3;
	else
		default_fontsize = f+1;

	for (r=0; r<MAX_FONTS; r++)
	{
		if (f_cnt[r].name != 0)
		{
			if (f_cnt[r].name->str != 0)
				free(f_cnt[r].name->str);
			free(f_cnt[r].name);
			f_cnt[r].name= 0;
		}
	}
}

static void incr_f_cnt(uni_string *name)
{
	int i;

	if ((name == 0)||(name->str == 0)||(name->str[0] == 0))
		return;

	for (i=0; i<MAX_FONTS; i++)
	{
		if (f_cnt[i].name)
		{
			if (uni_string_comp(name, f_cnt[i].name) == 0)
				f_cnt[i].cnt++;
		}
		else
		{
			f_cnt[i].name = (uni_string *)malloc(sizeof(uni_string));
			if (f_cnt[i].name)
			{
				f_cnt[i].name->str = (char *)malloc(name->len+1);
				if (f_cnt[i].name->str)
				{
					memcpy(f_cnt[i].name->str, name->str, name->len);
					f_cnt[i].name->str[name->len] = 0;
					f_cnt[i].name->uni = name->uni;
					f_cnt[i].name->len = name->len;
					f_cnt[i].cnt = 1;
					break;
				}
			}
		}
	}
}

static int get_default_font(void)
{
	int i, m = -1;

	for (i=0; i<MAX_FONTS; i++)
	{
		if (f_cnt[i].name)
		{
			if (f_cnt[i].name->str)
			{
				if (f_cnt[i].cnt > m)
					m = i;
			}
		}
	}
	return m;
}

static void update_default_alignment(unsigned int sheet, int row)
{
	int i, left = 0, center = 0, right = 0;
	cell *c;

	if ((sheet >= max_worksheets)||(ws_array[sheet] == 0))
		return;
	if (ws_array[sheet]->c_array == 0)
		return;

	for (i=ws_array[sheet]->first_col; i<=ws_array[sheet]->biggest_col; i++)
	{	/* This stuff happens for each cell... */
		c = ws_array[sheet]->c_array[(row*ws_array[sheet]->max_cols)+i];
		if (c)
		{
			int numeric = IsCellNumeric(c);
			if (c->xfmt == 0)
			{	/* Unknown format... */
				left++;
			}
			else
			{	/* Biff 8 stuff... */
				if ((c->xfmt < next_xf)&&(c->ustr.str))
				{
					if (strcmp(c->ustr.str, "&nbsp;"))
					{
						if (xf_array[c->xfmt])
						{
							switch(xf_array[c->xfmt]->align & 0x0007)
							{	/* Override default table alignment when needed */
								case 2:
								case 6:		/* Center across selection */
									center++;
									break;
								case 0:		/* General alignment */
									if (numeric)						/* Numbers */
										right++;
									else if ((c->type & 0x00FF) == 0x05)	/* Boolean */
										center++;
									else
										left++;
									break;
								case 3:
									right++;
									break;
								case 1:			/* fall through... */
								default:
									left++;
									break;
							}
						}
					}
				}
			}
		}
	}

	if ((center >= left)&&(center >= right))		/* Favor center since its the longest word */
		default_alignment = "center";
	else if ((right >= center)&&(right >= left))
		default_alignment = "right";			/* Favor right since its second longest */
	else
		default_alignment = "left";
}

static int null_string(unsigned char *str)
{	/* FIXME: This function may not be unicode safe */
	unsigned char *ptr;
	if ((str == NULL)||(*str == 0))
		return 1;

	ptr = str;
	while (*ptr != 0)
	{
		if (*ptr++ != ' ')
			return 0;
	}
	return 1;
}

static void OutputString(uni_string *u)
{
	unsigned int i;

	if (u == 0)
		return;

	if (u->uni < 2)
	{
		if (null_string(u->str))
		{
			if (Ascii == 0)
				printf(OutputXML ? "" : "&nbsp;");
			else if (!Csv)
				printf(" ");
		}
		else
		{
			if (Ascii)	/* If Ascii output requested, simply output the string */
			{	/* These are broken up for performance */
				if (Csv)
				{
					for (i=0; i<u->len; i++)
					{
						if (u->str[i] == 0x22)
							printf("\"\"");
						else
							putchar(u->str[i]);
					}
				}
				else
				{
					for (i=0; i<u->len; i++)
						putchar(u->str[i]);
				}
				return;
			}
			if (u->crun_cnt)
			{
				unsigned int loc, fnt_idx, crun_cnt=0;
				int format_changed = 0;
				html_attr h_flags;

				/* read the first format run */
				update_crun_info(&loc, &fnt_idx, crun_cnt, u->fmt_run);
				html_flag_init(&h_flags);
				for (i=0; i<u->len; i++)
				{
					if (i == loc)
					{	/* Time to change formats */
						if (format_changed)
						{	/* if old attributs, close */
							output_end_html_attr(&h_flags);
							if (h_flags.fflag)
								printf("</FONT>");
						}
						else
						{	/* FIXME: Also need to consider that a font may already be set for
							   the cell, in which case a closing tag should be set. */
							format_changed = 1;
						}

						/* set new attr	*/
						output_start_font_attribute(&h_flags, fnt_idx);
						output_start_html_attr(&h_flags, fnt_idx, 1);

						/* get next fmt_run	*/
						if (crun_cnt < u->crun_cnt)
						{
							crun_cnt++;
							update_crun_info(&loc, &fnt_idx, crun_cnt, u->fmt_run);
						}
					}
					OutputCharCorrected(u->str[i]);
				}
				if (format_changed)
				{
					output_end_html_attr(&h_flags);
					if (h_flags.fflag)
						printf("</FONT>");
				}
			}
			else
			{
				for (i=0; i<u->len; i++)
					OutputCharCorrected(u->str[i]);
			}
		}
	}
	else
	{
/*		if (null_string(u->str))		This doesn't work for unicode
		{
			if (Ascii)
     			printf(" ");
			else
				printf("&nbsp;");
		}
		else */
		{
			for (i=0; i<u->len; i+=2)
				print_utf8(getShort(&(u->str[i])));
		}
	}	
}

static void OutputCharCorrected(unsigned char c)
{
	if (MultiByte && (c & 0x80)) 
	{
		putchar(c);
		return;
	}
	switch (c)
	{	/* Special char handlers here... */
		case 0x3C:
			printf("&lt;");
			break;
		case 0x3E:
			printf("&gt;");
			break;
		case 0x26:
			printf("&amp;");
			break;
		case 0x22:
			printf("&quot;");
			break;
		/* Also need to cover 128-159 since MS uses this area... */
		case 0x80:		/* Euro Symbol */
			printf("&#8364;");
			break;
		case 0x82:		/* baseline single quote */
			printf("&#8218;");
			break;
		case 0x83:		/* florin */
			printf("&#402;");
			break;
		case 0x84:		/* baseline double quote */
			printf("&#8222;");
			break;
		case 0x85:		/* ellipsis */
			printf("&#8230;");
			break;
		case 0x86:		/* dagger */
		    printf("&#8224;");
		    break;
		case 0x87:		/* double dagger */
		    printf("&#8225;");
		    break;
		case 0x88:		/* circumflex accent */
		    printf("&#710;");
		    break;
		case 0x89:		/* permile */
		    printf("&#8240;");
		    break;
		case 0x8A:		/* S Hacek */
		    printf("&#352;");
		    break;
		case 0x8B:		/* left single guillemet */
		    printf("&#8249;");
		    break;
		case 0x8C:		/* OE ligature */
		    printf("&#338;");
		    break;
		case 0x8E:		/*  #LATIN CAPITAL LETTER Z WITH CARON */
			printf("&#381;");
			break;
		case 0x91:		/* left single quote ? */
		    printf("&#8216;");
		    break;
		case 0x92:		/* right single quote ? */
		    printf("&#8217;");
		    break;
		case 0x93:		/* left double quote */
		    printf("&#8220;");
		    break;
		case 0x94:		/* right double quote */
		    printf("&#8221;");
		    break;
		case 0x95:		/* bullet */
		    printf("&#8226;");
		    break;
		case 0x96:		/* endash */
		    printf("&#8211;");
		    break;
		case 0x97:		/* emdash */
		    printf("&#8212;");
		    break;
		case 0x98:		/* tilde accent */
		    printf("&#732;");
		    break;
		case 0x99:		/* trademark ligature */
		    printf("&#8482;");
		    break;
		case 0x9A:		/* s Haceks Hacek */
		    printf("&#353;");
		    break;
		case 0x9B:		/* right single guillemet */
		    printf("&#8250;");
		    break;
		case 0x9C:		/* oe ligature */
		    printf("&#339;");
		    break;
		case 0x9F:		/* Y Dieresis */
		    printf("&#376;");
		    break;
		default:
			putchar(c);
			break;
	}
}

static void update_crun_info(unsigned int *loc, unsigned int *fmt_idx, unsigned int crun_cnt, unsigned char *fmt_run)
{
	unsigned short tloc, tfmt_idx;
	int offset = crun_cnt*4;

	tloc = getShort(&fmt_run[offset]);
	tfmt_idx = getShort(&fmt_run[offset+2]);
	*loc = (unsigned int)tloc;
	*fmt_idx = (unsigned int)tfmt_idx;
}

static void put_utf8(unsigned short c)
{
	putchar(0x0080 | ((short)c & 0x003F));
}

static void print_utf8(unsigned short c)
{
	if (c < 0x80)
		OutputCharCorrected(c);
	else if (c < 0x800)
	{
		putchar(0xC0 | (c >>  6));
		put_utf8(c);
	}
	else
	{
		putchar(0xE0 | (c >> 12));
		put_utf8(c >>  6);
		put_utf8(c);
	}
}

static void uni_string_clear(uni_string *str)
{
	if (str == 0)
		return;

	str->str = 0;
	str->uni = 0;
	str->len = 0;
	str->fmt_run = 0;
}

static int uni_string_comp(uni_string *s1, uni_string *s2)
{
	if ((s1 == 0)||(s2 == 0))
		return -1;
	if ((s1->str == 0)||(s2->str == 0))
		return -1;

	if ((s1->uni == s2->uni) && (s1->len == s2->len))
		return memcmp(s1->str, s2->str, s1->len);
	else
		return -1;
}

static void output_start_html_attr(html_attr *h, unsigned int fnt_idx, int do_underlines)
{
	if (fnt_idx < next_font)
	{
		if (((font_array[fnt_idx]->underline&0x0023) > 0)&&(do_underlines))
		{
			printf("<U>");
			h->uflag = 1;
		}
		if (font_array[fnt_idx]->bold >= 0x02BC)
		{
			h->bflag = 1;
			printf("<B>");
		}
		if (font_array[fnt_idx]->attr & 0x0002)
		{
			h->iflag = 1;
			printf("<I>");
		}
		if (font_array[fnt_idx]->attr & 0x0008)
		{
			h->sflag = 1;
			printf("<S>");
		}
		if ((font_array[fnt_idx]->super & 0x0003) == 0x0001)
		{
			h->spflag = 1;
			printf("<SUP>");
		}
		else if ((font_array[fnt_idx]->super & 0x0003) == 0x0002)
		{
			h->sbflag = 1;
			printf("<SUB>");
		}
	}
}

static void output_end_html_attr(html_attr *h)
{
	if (h->sbflag)
	{
		printf("</SUB>");
		h->sbflag = 0;
	}
	else if (h->spflag)
	{
		printf("</SUP>");
		h->spflag = 0;
	}
	if (h->sflag)
	{
		printf("</S>");
		h->sflag = 0;
	}
	if (h->iflag)
	{
		printf("</I>");
		h->iflag = 0;
	}
	if (h->bflag)
	{
		printf("</B>");
		h->bflag = 0;
	}
	if (h->uflag)
	{
		if (h->uflag == 1)
			printf("</U>");
		else
			printf("</A>");
		h->uflag = 0;
	}
}

static void html_flag_init(html_attr *h)
{
	h->fflag = 0;
	h->bflag = 0;
	h->iflag = 0;
	h->sflag = 0;
	h->uflag = 0;
	h->sbflag = 0;
	h->spflag = 0;
}

static void output_start_font_attribute(html_attr *h, unsigned int fnt_idx)
{
	if (uni_string_comp(&default_font, &(font_array[fnt_idx]->name)) != 0)
	{
		h->fflag = 1;
		printf("<FONT FACE=\"");
		OutputString(&(font_array[fnt_idx]->name));
		printf("\"");
	}
	if (font_array[fnt_idx]->c_idx != 0x7FFF)
	{
		char color[8];
		if ((font_array[fnt_idx]->c_idx < MAX_COLORS)&&use_colors)
			strcpy(color, colorTab[font_array[fnt_idx]->c_idx]);
		else
		{	/* Not doing custom palettes yet... */
			strcpy(color, "000000");
		}
		if (strcmp(color, "000000") != 0)
		{
			if (h->fflag)
				printf(" COLOR=\"%s\"", color);
			else
			{
				h->fflag = 1;
				printf("<FONT COLOR=\"%s\"", color);
			}
		}
	}
	if (font_array[fnt_idx]->super & 0x0003)
	{
		if (h->fflag)
			printf(" SIZE=2");		/* Sub & Superscript */
		else
		{
			h->fflag = 1;
			printf("<FONT SIZE=2");		/* Sub & Superscript */
		}
	}
	else
	{
		if (h->fflag)
		{
			if (font_array[fnt_idx]->size != default_fontsize)
				printf(" SIZE=%d", font_array[fnt_idx]->size);
		}
		else
		{
			if (font_array[fnt_idx]->size != default_fontsize)
			{
				h->fflag = 1;
				printf("<FONT SIZE=%d", font_array[fnt_idx]->size);
			}
		}
	}
	if (h->fflag)
		printf(">");
}

