/* GNU xhippo: a GTK-based playlist manager.
   (c) 1998-2000 Adam Sampson, azz@gnu.org

   Please report bugs to bug-xhippo@gnu.org.

   This program is free software, released under the GNU General
   Public License. You may redistribute and/or modify this program
   under the terms of that license as published by the Free Software
   Foundation; either version 2 of the License, or (at your option)
   any later version.
   
   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   To get a copy of the GNU General Public License, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA, or see http://www.gnu.org/.
*/

#ifndef _XHIPPO_H
#define _XHIPPO_H 1

/* --- Other headers --- */

#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <signal.h>
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <time.h>
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#include <sys/stat.h>
#include <glib.h>
#ifdef USEGNOME
#include <gnome.h>
#endif
#ifndef NOGETOPTLONG
#include <getopt.h>
#endif

/* --- Constants --- */

/* A generic string buffer size used throughout xhippo. */
#define STRBUFSIZE 1024

/* --- Types --- */

/* A file type linked list. */
struct _fileaction {
  gchar *extension;        /* The extension of the file. */
  gchar *action;           /* The command to run. */

  struct _fileaction *next;
};
typedef struct _fileaction fileaction;

/* A user command linked list. */
struct _usercommand {
  gchar *description;      /* Description of the command for the menu. */
  gchar *command;          /* The command to run (%s being replaced by
			      the filename of the current song). */

  struct _usercommand *next;
};
typedef struct _usercommand usercommand;

/* Information about a song in the playlist. */
struct _songinfo {
  gchar *name;             /* Its filename. */
  gchar *display_name;     /* The name shown in the list. */
  char played;             /* Whether it has been played. */
  char inforead;           /* Whether the information that comes 
			      from stat (mtime, size) has been read. */
  time_t mtime;            /* When the file was last modified. */
  off_t size;              /* The size of the file. */
  GtkWidget *info_window;  /* The information window; NULL if not up. */
};
typedef struct _songinfo songinfo;

enum _configtype {
  CT_END,                   /* end of list */
  CT_INT,                   /* guint */
  CT_STRING,                /* gchar * */
  CT_FUNCTION1,             /* void function(gchar *arg) */
  CT_FUNCTION2              /* void function(gchar *arg1, gchar *arg2) */
};
typedef enum _configtype configtype;

struct _configcommand {
  gint pass;                /* Pass upon which to read this entry. */
  configtype type;          /* Type of this command. */
  char *command;            /* Name of this command. */
  void *target;
};
typedef struct _configcommand configcommand;

/* --- Globals --- */

/* UI widgets */
extern GtkTooltips *tooltips;
extern GtkWidget *window, *restartbutton, *stopbutton, *buttons_box, 
  *all_box, *savebutton, *sortbutton, *addbutton,
  *filelist, *statusbar, *nextbutton, *pausebutton, *loadbutton,
  *minicheckbox, *randomcheckbox, *fileselector, *list_box,
  *scrollbar, *upbutton, *downbutton, *deletebutton, *popupmenu,
  *up_item, *down_item, *delete_item, *info_item;
extern GtkAdjustment *vadj;
extern GtkAccelGroup *accel_group;

/* Status variables */
extern pid_t childpid;
extern guint contextid, 
  listcount, 
  playing, 
  paused, 
  donotstopflag;
extern gint last_played,
  last_row_hit;
extern gchar *last_player;

/* Configuration options */
extern guint mode_play_on_start, 
  mode_scroll_catchup, 
  mode_left_scroll,
  mode_must_play_all, 
  mode_one_time, 
  mode_show_pid,
  mode_read_id3_tags,
  mode_save_window_pos,
  mode_start_mini,
  mode_play_ordered,
  mode_strip_extension,
  mode_hide_player_output,
  mode_sort_on_load,
  mode_demangle_names,
  mode_playlist_title,
  mode_title_basename,
  mode_no_check_files,
  mode_write_playing,
  mode_skip_path,
  attribute_mini;
extern gchar *attribute_playlist_dir;
extern gint attribute_xpos, 
  attribute_ypos, 
  attribute_width, 
  attribute_height;
extern fileaction *fileactions;
extern usercommand *usercommands;

/* --- Prototypes --- */

songinfo *copy_songinfo(songinfo *src);
void destroy_songinfo(gpointer sinfo);
void insert_row(songinfo *sinfo, gint location);
songinfo *get_songinfo(gint number);
void move_row(gint src, gint dest);
gint filetype(char *file);
void chomp(gchar *line);
void status(char *message);
void update_window(void);
void stop_playing(void);
void pause_playing(void);
void start_playing(gint num);
void pick_next(void);
gchar *read_mp3_tag ( char *filename );
void read_song_info(songinfo *sinfo, struct stat *st);
void add_file(gchar *name);
void save_window_state(void);
void read_window_state(void);
void clear_playlist(void);
int sort_compare_name(const void *a, const void *b);
int sort_compare_mtime(const void *a, const void *b);
void sortplaylist(int (*func)(const void *a, const void *b),
		  char needinfo);
void read_playlist(gchar *name);
void execute_command(gchar *command, gint pass);
void read_config(gchar *name, gint pass);
void read_config_files(gint pass);
void handle_list_select(GtkWidget *gtklist, gint row, gint column,
			GdkEventButton *event, gpointer data);
void handle_list_click(GtkWidget *gtklist, GdkEventButton *event, 
		       gpointer data);
void handle_fileselector_cancel(GtkWidget *widget, gpointer data);
void handle_load_ok(GtkWidget *widget, gpointer data);
void handle_save_ok(GtkWidget *widget, gpointer data);
void handle_add_ok(GtkWidget *widget, gpointer data);
void handle_next(GtkWidget *widget, gpointer data);
void set_mini(void);
void handle_minitoggle(GtkWidget *widget, gpointer data);
void handle_randomtoggle(GtkWidget *widget, gpointer data);
void handle_restart(GtkWidget *widget, gpointer data);
void handle_stop(GtkWidget *widget, gpointer data);
void handle_pause(GtkWidget *widget, gpointer data);
gint handle_info_delete_event(GtkWidget *widget, GdkEvent *event, 
			      gpointer data);
void handle_info_destroy(GtkWidget *widget, gpointer data);
void handle_menu_info(GtkWidget *widget, gpointer data);
void handle_menu_up(GtkWidget *widget, gpointer data);
void handle_menu_down(GtkWidget *widget, gpointer data);
void handle_menu_delete(GtkWidget *widget, gpointer data);
void handle_menu_user(gpointer data);
void get_playlist_dir(gchar *buf, int buflen);
void handle_menu_load(GtkWidget *widget, gpointer data);
void handle_menu_save(GtkWidget *widget, gpointer data);
void handle_menu_add(GtkWidget *widget, gpointer data);
void handle_menu_sort_name(GtkWidget *widget, gpointer data);
void handle_menu_sort_mtime(GtkWidget *widget, gpointer data);
void handle_menu_clear(void);
gint handle_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data);
void handle_destroy(GtkWidget *widget, gpointer data);
void handle_configure(GtkWidget *widget, gpointer data);
gint sigchld_timeout(gpointer data);
void handle_sigchld(int dummy);
#ifdef USEGNOMEMENUS
void handle_about(GtkWidget *widget, void *data);
#endif
void handle_dnd_drop(GtkWidget *widget, GdkDragContext *context,
		     gint x, gint y,
		     GtkSelectionData *data, guint info,
		     guint time);
void print_version(void);
int main(int argc, char **argv);

/* --- Compatibility stuff --- */

#ifndef HAVE_SNPRINTF
/* If the user doesn't have snprintf, use sprintf instead. 
   This is a GNU cpp extension. */
#define snprintf(s, l, f, a...) sprintf(s, f, ##a)
#endif

#ifdef NOSTRCASECMP
/* The system doesn't provide strcasecmp, so we will instead. */
int strcasecmp(char *a, char *b);
#endif

/* Localisation. */

#ifndef LANG_DEUTSCH
/* English locale by Adam Sampson. */
#define STRBADOPTION "Unknown option \"%c\". xhippo -h for help.\n"
#define STRGUESSTYPE "Unable to guess type."
#define STRPLAYINGWITHPID "Playing with %s (pid %d)."
#define STRPLAYINGWITH "Playing with %s."
#define STRNOSTART "Unable to start player \"%s\".\n"
#define STRNOITEMS "No items to pick from!"
#define STRSTOPPED "Stopped."
#define STRDONE "All items played."
#define STRPAUSED "Paused."
#define STRNOFIND "Couldn't find \"%s\".\n"
#define STRNOTFILE "\"%s\" isn't a regular file.\n"
#define STRWELCOME "Welcome to xhippo!"
#define STRMINI "Mini"
#define STRMINITIP "Toggle the size of the window"
#define STRRESTART "Restart"
#define STRRESTARTTIP "Restart the current song"
#define STRSTOP "Stop"
#define STRSTOPTIP "Stop playing"
#define STRPAUSE "Pause"
#define STRPAUSETIP "Pause playing"
#define STRRANDOM "Random"
#define STRNEXT "Next"
#define STRNEXTTIP "Play the next song"
#define STRINFO "Song information"
#define STRUP "Move song up"
#define STRDOWN "Move song down"
#define STRADD "Add song"
#define STRDELETE "Remove song"
#define STRLOAD "Load playlist"
#define STRSAVE "Save playlist"
#define STRSORTNAME "Sort playlist by name"
#define STRSORTMTIME "Sort playlist by mtime"
#define STRCLEAR "Clear playlist"
#define STRNOPLAYLIST "Couldn't open playlist \"%s\"!\n"
#define STRNUMFILES "%d files in list."
#define STRPICKPLAYLIST "Load playlist"
#define STRSAVEPLAYLIST "Save playlist"
#define STRNOWINSTATE "Couldn't write to $HOME/.xhippo/winstate.\n"
#define STRSETHOME "You must set $HOME to your home directory.\n"
#define STRINFOTITLE "Song information"
#define STRINFONAME "Name:"
#define STRINFOFILENAME "Filename:"
#define STRINFOLASTMODIFIED "Last modified:"
#define STRINFOFILESIZE "File size:"
#define STRINFOCLOSE "Close"
#else
/* German locale courtesy of Volker Assmann, volka@bigfoot.de, and
   Markus Kohls, markusk@bnet-ibb.de. */
#define STRBADOPTION "Unbekannte Option \"%c\". Hilfe mit xhippo -h.\n"
#define STRGUESSTYPE "Unbekannter Dateityp."
#define STRPLAYINGWITHPID "Abspielen durch %s (pid %d)."
#define STRPLAYINGWITH "Abspielen durch %s."
#define STRNOSTART "Player \"%s\" funktioniert nicht.\n"
#define STRNOITEMS "Keine Playlist geladen!"
#define STRSTOPPED "Gestoppt."
#define STRDONE "Alle Lieder gespielt."
#define STRPAUSED "Pause."
#define STRNOFIND "Kann \"%s\" nicht finden.\n"
#define STRNOTFILE "\"%s\" ist keine normale Datei.\n"
#define STRWELCOME "Wilkommen im xhippo!"
#define STRMINI "Mini"
#define STRMINITIP "Groesse des Fensters aendern"
#define STRRESTART "Neustart"
#define STRRESTARTTIP "Lied wiederholen."
#define STRSTOP "Stop"
#define STRSTOPTIP "Lied anhalten."
#define STRPAUSE "Pause"
#define STRPAUSETIP "Lied pausieren."
#define STRRANDOM "Zufllig"
#define STRNEXT "Vor"
#define STRNEXTTIP "Naechstes Lied auswaehlen."
#define STRNOPLAYLIST "Konnte Playlist \"%s\" nicht oeffnen!\n"
#define STRNUMFILES "%d Lieder in der Playlist."
#define STRPICKPLAYLIST "Playlist waehlen"
#define STRSAVEPLAYLIST "Playlist speichern"
#define STRNOWINSTATE "Konnte $HOME/.xhippo/winstate nicht schreiben.\n"
#define STRINFO "Information"
#define STRUP "Nach Oben"
#define STRDOWN "Nach Unten"
#define STRLOAD "Laden"
#define STRSAVE "Speichern"
#define STRSORTNAME "Sortieren (Name)"
#define STRSORTMTIME "Sortieren (mtime)"
#define STRCLEAR "Alles loeschen"
#define STRADD "Hinzufuegen"
#define STRDELETE "Loeschen"
#define STRSETHOME "$HOME muss auf Home-Verzeichnis zeigen.\n"
#define STRINFOTITLE "Song Information"
#define STRINFONAME "Name:"
#define STRINFOFILENAME "Filename:"
#define STRINFOLASTMODIFIED "Zuletzt geaendert:"
#define STRINFOFILESIZE "Filegroesse:"
#define STRINFOCLOSE "Close"
#endif

#endif /* _XHIPPO_H */
