/*
   GNU xhippo: a GTK-based playlist manager.
   Copyright 1998, 1999, 2000 Adam Sampson <azz@gnu.org>

   Please report bugs to bug-xhippo@gnu.org.

   This file is part of GNU xhippo.

   GNU xhippo is free software; you can redistribute and/or modify it
   under the terms of that license as published by the Free Software
   Foundation; either version 2 of the License, or (at your option)
   any later version.
   
   GNU xhippo is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with GNU xhippo; see the file COPYING. If not, write to the
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
   MA 02111-1307 USA, or see http://www.gnu.org/.

*/

#include "xhippo.h"

/* Globals. */
pid_t childpid;
guint contextid, 
  listcount = 0, 
  playing = 0, 
  paused = 0, 
  donotstopflag;
gint last_played = -1,
  last_row_hit;
gchar *last_player = NULL;

/* Stop playing. */
void stop_playing() {
  if (!playing) return;
  if (!childpid) return;

  /* Wake up a paused player. */
  if (paused) {
    kill(childpid, SIGCONT);
    paused = 0;
  }

  playing = 0;
  kill(childpid, SIGTERM);

  /* Pause (to avoid the formation of zombies) until the child
     dies. childpid is reset in the SIGCHLD handler. */
  while (childpid) pause();
}

/* Toggle paused status. */
void pause_playing() {
  if (!playing) return;
  if (!childpid) return;

  if (paused) {
    kill(childpid, SIGCONT);
    paused = 0;
  } else {
    kill(childpid, SIGSTOP);
    paused = 1;
  }
}

/* Start playing song number num. Stop the old one first. */
void start_playing(gint num) {
  gchar *name, *type, *thisact, *p;
  int argcount, i;

  if (num < 0 || num >= listcount) return;

  last_played = num;

  /* Stop the previous song. */
  if (donotstopflag) {
    donotstopflag = 0;
  } else {
    stop_playing();
  }

  /* Get the extension of the filename. */
  name = get_songinfo(num)->name;
  type = name + strlen(name);
  while ((type > name) && (*type != '.') && (*type != '/')) {
    --type;
  }

  /* Attempt to find an action that matches the extension (if it had
     one). */
  thisact = NULL;
  if (*type == '.') {
    fileaction *fa;

    ++type;
    for (fa = fileactions; fa; fa = fa->next) {
      if (!strcasecmp(type, fa->extension)) thisact = fa->action;
    }
  }
  if (!thisact) {
    status(STRGUESSTYPE);
    return;
  }
  
  /* Fork off a child process to run the player in. */
  childpid = fork();
  if (!childpid) {
    gchar *command, **argptr, *p;

    /* Build the argptr array for execvp. */
    command = strdup(thisact);
    argcount = 2;
    for (p = command; p; p = strchr(p, ' ')) {
      argcount++;
      p++;
    }
    argptr = (gchar **) malloc(argcount * sizeof(gchar *));
    i = 0;
    for (p = command; p; p = strchr(p, ' ')) {
      if (p != command) *p++ = '\0';
      argptr[i++] = p;
    }
    argptr[i++] = name;
    argptr[i++] = NULL;

    /* Redirect stdout and stderr to /dev/null if necessary. */
    if (mode_hide_player_output) {
      int fd;

      fd = open("/dev/null", O_RDWR);
      dup2(fd, 1); /* stdout */
      dup2(fd, 2); /* stderr */
      close(fd);
    }

    /* At this point we've still got command and argptr allocated. */
    execvp(argptr[0], argptr);
    fprintf(stderr, STRNOSTART, argptr[0]);
    execvp("echo", argptr);
    _exit(20);
  }

  /* Update last_player. */
  if (last_player) free(last_player);
  last_player = strdup(thisact);
  p = strchr(last_player, ' ');
  if (p) *p = '\0';

  playing = 1;
  paused = 0;

  if (mode_write_playing) {
    gchar csname[STRBUFSIZE];
    FILE *f;

    /* Write the title of the song that's currently playing to the
       ~/.xhippo/current_song file. */
    snprintf(csname, STRBUFSIZE, "%s/.xhippo/current_song", 
	     getenv("HOME"));
    if ((f = fopen(csname, "w"))) {
      fprintf(f, "%s\n", get_songinfo(num)->display_name);
      fclose(f);
    }
  }
}

/* Pick and play a random song. */
void pick_next() {
  int newitem, first, i;

  if (listcount) {
    if (mode_play_ordered) {
      newitem = last_played + 1;
      if (newitem == listcount) {
	newitem = 0;
        if (mode_one_time) {
          playing = 0;
          return;
        }
      }
    } else {
      newitem = rand()%listcount; /* FIXME: This is a terrible randomiser. */
      if (mode_must_play_all) {
	first = newitem;
	while (get_songinfo(newitem)->played) {
	  newitem++;
	  if (newitem==listcount) newitem=0;
	  if (newitem==first) {
	    for (i=0; i<listcount; i++) get_songinfo(i)->played = 0;
            if (mode_one_time) {
              playing = 0;
              return; /* do this after setting played back to 0 */
            }
	  }
	}
      }
    }
    gtk_clist_select_row(GTK_CLIST(filelist), newitem, 0);
    /* ... which sends a message anyway and gets handle_list_select called. */
  } else {
    status(STRNOITEMS);
  }
}

/* Routine to read the ID3 tag from an MP3 file, originally supplied
   by Craig Knudsen. */
gchar *read_mp3_tag (char *filename) {
  int fd;
  gchar *trackname = NULL;
  gchar data[100];
  int ret;
  
  fd = open(filename, O_RDONLY);
  ret = lseek(fd, -128, SEEK_END);
  if (ret > 0) {
    ret = read(fd, data, 3);
    if (ret == 3) {
      data[3] = '\0';
      if (strcmp(data, "TAG") == 0) {
        ret = read(fd, data, 26); /* FIXME: Find the spec. */
        if (ret > 0) {
          trackname = strdup(data);
        }
      }
    }
  }
  close(fd);

  return trackname;
}

/* Fill in entries in a songinfo structure that need a stat first. Use
   the information in the provided struct stat if it's not NULL. */
void read_song_info(songinfo *sinfo, struct stat *st) {
  struct stat stnew;

  /* If we've already done this, then don't bother. */
  if (sinfo->inforead) return;

  if (!st) {
    st = &stnew;
    if (stat(sinfo->name, st) < 0) return;
  }
  sinfo->size = st->st_size;
  sinfo->mtime = st->st_mtime;
  sinfo->inforead = 1;
}

/* Called to add a new file to the playlist. */
void add_file(gchar *name) {
    gchar *basename, *dispname, *tag;
    songinfo *sinfo;
    struct stat st;

    if (!mode_no_check_files) {
      /* Note: this uses stat rather than filetype() because we need the
	 other information from the stat structure and this avoids
	 having to stat every file in the playlist twice. */
      if (stat(name, &st) < 0) {
	g_print(STRNOFIND, name);
	return;
      }

      if (!S_ISREG(st.st_mode)) {
	g_print(STRNOTFILE, name);
	return;
      }
    }

    if (mode_skip_path) {
      /* Skip the first mode_skip_path components of the name. */
      gchar *e = name + strlen(name);
      int i = mode_skip_path;

      if (*name == '/') i++;
      for (basename = name; i > 0 && basename < e; basename++) {
	if (*basename == '/') i--;
      }
      if (basename == e) basename = name;
    } else {
      /* Take the basename. */
      basename = name + strlen(name);
      while((basename > name) && (*basename != '/')) --basename; 
    }

    tag = NULL;
    if (mode_read_id3_tags) tag = read_mp3_tag(name);
    if (tag) {
      dispname = strdup(tag);
    } else {
      gchar *p;

      if (basename[0] == '/') ++basename;
      dispname = strdup(basename);

      if (mode_demangle_names) {
	/* Replace underscores with spaces. */
	while ((p = strchr(dispname, '_'))) *p = ' ';

	/* Replace '%20's with spaces. */
	while ((p = strstr(dispname, "%20"))) {
	  *p = ' ';
	  p += 2;
	  while (*(++p - 1)) *(p - 2) = *p;
	}
      }

      if (mode_strip_extension) {
	/* Strip off the extension. */
	p = dispname + strlen(dispname);
	while (p > dispname && *p != '.') --p;
	if (*p == '.') *p = '\0';
      }
    }

    /* Build a songinfo struct. */
    sinfo = malloc(sizeof(songinfo));
    sinfo->name = strdup(name);
    sinfo->display_name = dispname;
    sinfo->played = 0;
    sinfo->inforead = 0;
    sinfo->info_window = NULL;
    if (!mode_no_check_files) read_song_info(sinfo, &st);

    insert_row(sinfo, listcount);

    if (tag) free(tag);
}

/* Clear the playlist. */
void clear_playlist() {
  gtk_clist_clear(GTK_CLIST(filelist));
  last_played = -1;
}

/* Sorting function for playlist entries by display name. */
int sort_compare_name(const void *a, const void *b) {
  return strcmp((*((songinfo **)a))->display_name,
		(*((songinfo **)b))->display_name);
}

/* Sorting function for playlist entries by mtime. */
int sort_compare_mtime(const void *a, const void *b) {
  return (*((songinfo **)a))->mtime - (*((songinfo **)b))->mtime;
}

/* Sort the playlist by the display name using the specified function,
   and getting the song info first if required. */
void sortplaylist(int (*func)(const void *a, const void *b),
		  char needinfo) {
  songinfo **sl;
  gint i, lc = listcount;

  if (needinfo) {
    for (i=0; i<lc; i++) read_song_info(get_songinfo(i), NULL);
  }

  /* Copy the songinfo structs from the playlist into a new list,
     and then sort that. */
  sl = (songinfo **)malloc(lc * sizeof(songinfo *));
  for (i=0; i<lc; i++) sl[i] = copy_songinfo(get_songinfo(i));
  qsort(sl, lc, sizeof(songinfo *), func);
  
  gtk_clist_freeze(GTK_CLIST(filelist));
  clear_playlist();
  for (i=0; i<lc; i++) insert_row(sl[i], i);
  gtk_clist_thaw(GTK_CLIST(filelist));

  free(sl);
}

/* Read a playlist. */
void read_playlist(gchar *name) {
  FILE *f;
  char line[STRBUFSIZE], fullname[STRBUFSIZE], dirname[STRBUFSIZE], 
    *slash, *filename;

  /* Prevent the list from being updated while we load the list. */
  gtk_clist_freeze(GTK_CLIST(filelist));

  f = fopen(name, "r");
  if (!f) {
    g_print(STRNOPLAYLIST, name);
    return;
  }

  /* Work out the dirname if we've got one. */
  slash = strrchr(name, '/');
  if (slash) {
    *slash = '\0';
    snprintf(dirname, STRBUFSIZE, "%s/", name);
    *slash = '/';
  } else {
    *dirname = '\0';
  }
  
  while (fgets(line, STRBUFSIZE, f)) {
    chomp(line);

    /* Work out the full filename including the extra path. */
    filename = (*line == '!') ? line + 1 : line;
    if (*filename != '/') {
      snprintf(fullname, STRBUFSIZE, "%s%s", dirname, filename);
    } else {
      snprintf(fullname, STRBUFSIZE, "%s", filename);
    }
    
    if (*line == '!') {
      /* Allow inclusion of a playlist from a playlist. */
      read_playlist(fullname);
    } else {
      add_file(fullname);
    }
  }

  fclose(f);
  
  if (mode_sort_on_load) sortplaylist(sort_compare_name, FALSE);
  
  /* Set the window title if required. We do this at the end of this
     function so that the name of the highest nested playlist is
     used. */
  if (mode_playlist_title) {
    gchar *s, *t, *prefix = "xhippo " VERSION ": ";

    if (mode_title_basename) {
      s = name + strlen(name);
      while (s > name && *s != '/') s--;
      if (*s == '/') s++;
    } else {
      s = name;
    }

    t = malloc(strlen(s) + strlen(prefix) + 1);
    sprintf(t, "%s%s", prefix, s);
    gtk_window_set_title(GTK_WINDOW(window), t);
    free(t);
  }
  
  gtk_clist_thaw(GTK_CLIST(filelist));
}

/* Copy a songinfo structure. */
songinfo *copy_songinfo(songinfo *src) {
  songinfo *dest = (songinfo *)malloc(sizeof(songinfo));

  dest->name = strdup(src->name);
  dest->display_name = strdup(src->display_name);
  dest->played = src->played;
  dest->inforead = src->inforead;
  dest->mtime = src->mtime;
  dest->size = src->size;
  dest->info_window = src->info_window;

  return dest;
}

/* Destructor for songinfo, and therefore for the row as well. We can
   safely decrement the list count when this gets called. */
void destroy_songinfo(gpointer sinfo) {
  songinfo *s = (songinfo *)sinfo;
  free(s->name);
  free(s->display_name);
  if (s->info_window) gtk_widget_destroy(s->info_window);

  free(sinfo);
  --listcount;
}

/* Insert a row into the list based on a filled-in songinfo structure. 
   Call as insert_row(X, listcount); to append to the end. */
void insert_row(songinfo *sinfo, gint location) {
  gchar *row[1];
  
  row[0] = sinfo->display_name;
  gtk_clist_insert(GTK_CLIST(filelist), location, row);
  gtk_clist_set_row_data_full(GTK_CLIST(filelist), location, sinfo,
			      destroy_songinfo);
  listcount++;
}

/* Get a pointer to a song's songinfo. */
songinfo *get_songinfo(gint number) {
  return (songinfo *)gtk_clist_get_row_data(GTK_CLIST(filelist), number);
}

/* Move a row from one location to another within the list. */
void move_row(gint src, gint dest) {
  songinfo *new;

  gtk_clist_freeze(GTK_CLIST(filelist));

  new = copy_songinfo(get_songinfo(src));
  gtk_clist_remove(GTK_CLIST(filelist), src);
  insert_row(new, dest);
  gtk_clist_thaw(GTK_CLIST(filelist));
}

