/*
 * markerlist.cpp
 *
 * $Id: markerlist.cpp,v 0.4 1999/07/13 17:32:01 thorsten Exp thorsten $
 *
 * This file is part of XGlobe. See README for details.
 *
 * Copyright (C) 1998 Thorsten Scheuermann
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public Licenses as published by
 * the Free Software Foundation.
 *
 * $Log: markerlist.cpp,v $
 * Revision 0.4  1999/07/13 17:32:01  thorsten
 * code beautification
 *
 * Revision 0.3  1999/02/12 17:21:21  thorsten
 * added support for colored markers
 *
 * Revision 0.2  1999/01/06 13:26:48  thorsten
 * minimal change to suppress warning
 *
 * Revision 0.1  1998/12/10 20:00:17  thorsten
 * initial revision
 *
 */

#include "markerlist.h"
#include <qfile.h>
#include <qtextstream.h>

/* ------------------------------------------------------------------------ */

Location::Location(double lon, double lat, const char *name,
                   const QColor &color)
{
  this->lon = lon;
  this->lat = lat;
  this->name = name;
  this->color = color;
}

/* ------------------------------------------------------------------------ */

QString Location::getName()
{
  return name;
}

/* ------------------------------------------------------------------------ */

double Location::getLongitude()
{
  return lon;
}

/* ------------------------------------------------------------------------ */

double Location::getLatitude()
{
  return lat;
}

/* ------------------------------------------------------------------------ */

QColor &Location::getColor()
{
  return color;
}

/* ------------------------------------------------------------------------ */

/*
 * Loads a marker definition file.
 * This file must have the following format:
 *
 * <longitude> <latitude> <name of location>
 * where <longitude> and <latitude> are numbers specifying the geographical
 * position and <name of location> is the place's name in quotes,
 * e.g. "Atlantis".
 * Anything after these three tokens on the same line is ignored.
 * A line with a leading '#' is ignored an can be used for comments.
 */

bool MarkerList::loadMarkerFile(const char *filename)
{
  int         linenum = 0;
  int         numloc = 0;
  double      lon, lat;
  QString     name;
  QString     colorname;
  bool        colorfound = FALSE;
  int         pos1;
  int         pos2;
  QFile f(filename);
  QTextStream t(&f);
  QString     line;
  QColor      color;

  if(!f.open(IO_ReadOnly))
    return FALSE;
  
  while(!t.eof())
  {
    line = t.readLine();
    linenum++;
    line = line.simplifyWhiteSpace();

    if(line.isEmpty())                  // skip empty lines
      continue;
    
    if(((const char *)line)[0] == '#')  // if it's a comment, skip to next line
      continue;

    // read longitude
    pos1 = line.find(' ');
    if(pos1 == -1)
    {
      fprintf(stderr, "Syntax error in marker file \"%s\", line %d.\n",
              filename, linenum);
      return FALSE;
    }
    lat = line.left(pos1).toDouble();

    // read latitude
    pos2 = line.find(' ', pos1+1);
    if(pos2 == -1)
    {
      fprintf(stderr, "Syntax error in marker file \"%s\", line %d.\n",
              filename, linenum);
      return FALSE;
    }
    lon = line.mid(pos1+1, pos2-pos1-1).toDouble();

    // read name
    pos1 = line.find('\"', pos2);
    if(pos1 == -1)
    {
      fprintf(stderr, "Syntax error in marker file \"%s\", line %d.\n",
              filename, linenum);
      return FALSE;
    }
    pos2 = line.find('\"', pos1+1);
    if(pos2 == -1)
    {
      fprintf(stderr, "Syntax error in marker file \"%s\", line %d.\n",
              filename, linenum);
      return FALSE;
    }
    name = line.mid(pos1+1, pos2-pos1-1);

    // read color value

    pos1 = line.find("color=", pos2);
    if(pos1 != -1)
    {
      pos2 = line.find('#', 0);
      if((pos2 == -1) || (pos2 > pos1))
      {
        pos2 = line.find(' ', pos1);
        if(pos2 != -1)
          colorname = line.mid(pos1+6, pos2-pos1-6);
        else
          colorname = line.right(line.length()-pos1-6);
        colorfound = TRUE;
      }
    }
    
    color.setRgb(255, 0, 0);             // the default color is red
    if(colorfound)
    {
      color.setNamedColor((const char *)colorname);
      colorfound = FALSE;
    }
    
    // add Location to list
    append(new Location(lon, lat, name, color));
    numloc++;
    // printf("%.2f %.2f %s\n", lon, lat, (const char *)name);
  }

  //  printf("Read %d location definitions from file \"%s\"\n",
  //          numloc, filename);
  f.close();

  return TRUE;
}

/* ------------------------------------------------------------------------ */
