#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <fx.h>
#include <fxkeys.h>
#include <FXPNGIcon.h>

#include "xfedefs.h"
#include "icons.h"
#include "FileDialog.h"
#include "FileList.h"
#include "Properties.h"
#include "FilePanel.h"
#include "XFileExplorer.h"
#include "InputDialog.h"
#include "BrowseInputDialog.h"
#include "ArchInputDialog.h"
#include "HistInputDialog.h"
#include "File.h"
#include "MessageBox.h"
#include "OverwriteBox.h"
#include "PathLinker.h"

// Clipboard notes :
// The uri-list type used for Xfe is the same as the Gnome uri-list type
// The standard uri-list type is used for KDE and non Gnome / XFCE file managers 
// A special uri-list type that containd only "0" (for copy) or "1" (for cut) is used for KDE compatibility


// Global Variables
extern FXMainWindow *mainWindow;

// Clipboard
extern FXString clipboard;
FXuint clipboard_type=0;


extern char OpenHistory[OPEN_WITH_HIST_SIZE][100];
extern int OpenNum;
#if defined(linux)
extern FXStringDict* fsdevices;
extern FXStringDict* mtdevices;
extern FXbool deb_based;
extern FXbool rpm_based;
#endif

extern FXbool allowPopupScroll;
extern FXuint single_click;


// Map
FXDEFMAP(FilePanel) FilePanelMap[]=
{
	FXMAPFUNC(SEL_CLIPBOARD_LOST,0,FilePanel::onClipboardLost),
	FXMAPFUNC(SEL_CLIPBOARD_GAINED,0,FilePanel::onClipboardGained),
	FXMAPFUNC(SEL_CLIPBOARD_REQUEST,0,FilePanel::onClipboardRequest),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_DIRECTORY_UP,FilePanel::onCmdDirectoryUp),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILTER,FilePanel::onCmdItemFilter),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_GO_HOME,FilePanel::onCmdGoHome),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_GO_TRASH,FilePanel::onCmdGoTrash),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_VIEW,FilePanel::onCmdEdit),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EDIT,FilePanel::onCmdEdit),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PROPERTIES,FilePanel::onCmdProperties),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_XTERM,FilePanel::onCmdXTerm),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_DIR,FilePanel::onCmdNewDir),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_FILE,FilePanel::onCmdNewFile),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_SYMLINK,FilePanel::onCmdNewSymlink),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_COPY,FilePanel::onCmdFileMan),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_CUT,FilePanel::onCmdFileMan),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_COPYTO,FilePanel::onCmdFileMan),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_MOVETO,FilePanel::onCmdFileMan),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_RENAME,FilePanel::onCmdFileMan),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_SYMLINK,FilePanel::onCmdFileMan),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_COPY_CLIPBOARD,FilePanel::onCmdCopyCut),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_CUT_CLIPBOARD,FilePanel::onCmdCopyCut),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PASTE_CLIPBOARD,FilePanel::onCmdPaste),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_TRASH,FilePanel::onCmdFileTrash),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_DELETE,FilePanel::onCmdFileDelete),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_OPEN_WITH,FilePanel::onCmdOpenWith),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_OPEN,FilePanel::onCmdOpen),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_REFRESH,FilePanel::onCmdRefresh),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SHOW_BIG_ICONS,FilePanel::onCmdShow),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SHOW_MINI_ICONS,FilePanel::onCmdShow),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SHOW_DETAILS,FilePanel::onCmdShow),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_TOGGLE_HIDDEN,FilePanel::onCmdToggleHidden),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_TOGGLE_THUMBNAILS,FilePanel::onCmdToggleThumbnails),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SELECT_ALL,FilePanel::onCmdSelect),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_DESELECT_ALL,FilePanel::onCmdSelect),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_SELECT_INVERSE,FilePanel::onCmdSelect),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_ADD_TO_ARCH,FilePanel::onCmdAddToArch),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EXTRACT,FilePanel::onCmdExtract),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EXTRACT_TO_FOLDER,FilePanel::onCmdExtractToFolder),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EXTRACT_HERE,FilePanel::onCmdExtractHere),
	FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,FilePanel::ID_FILELIST,FilePanel::onCmdPopupMenu),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_POPUP_MENU,FilePanel::onCmdPopupMenu),
	FXMAPFUNC(SEL_DOUBLECLICKED,FilePanel::ID_FILELIST,FilePanel::onCmdItemDoubleClicked),
	FXMAPFUNC(SEL_CLICKED,FilePanel::ID_FILELIST,FilePanel::onCmdItemClicked),
	FXMAPFUNC(SEL_FOCUSIN,FilePanel::ID_FILELIST,FilePanel::onCmdItemFocus),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_STATUS,FilePanel::onUpdStatus),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_DIRECTORY_UP,FilePanel::onUpdUp),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_COPY_CLIPBOARD,FilePanel::onUpdMenu),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_CUT_CLIPBOARD,FilePanel::onUpdMenu),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PASTE_CLIPBOARD,FilePanel::onUpdPaste),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PROPERTIES,FilePanel::onUpdMenu),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_TRASH,FilePanel::onUpdFileTrash),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_GO_TRASH,FilePanel::onUpdGoTrash),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_DELETE,FilePanel::onUpdFileDelete),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_MOVETO,FilePanel::onUpdMenu),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_COPYTO,FilePanel::onUpdMenu),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_RENAME,FilePanel::onUpdSelMult),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_EDIT,FilePanel::onUpdOpen),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_VIEW,FilePanel::onUpdOpen),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_OPEN,FilePanel::onUpdOpen),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_ADD_TO_ARCH,FilePanel::onUpdAddToArch),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_SHOW_BIG_ICONS,FilePanel::onUpdShow),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_SHOW_MINI_ICONS,FilePanel::onUpdShow),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_SHOW_DETAILS,FilePanel::onUpdShow),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_TOGGLE_HIDDEN,FilePanel::onUpdToggleHidden),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_TOGGLE_THUMBNAILS,FilePanel::onUpdToggleThumbnails),
#if defined(linux)
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_MOUNT,FilePanel::onCmdMount),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_UMOUNT,FilePanel::onCmdMount),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_MOUNT,FilePanel::onUpdMount),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_UMOUNT,FilePanel::onUpdUnmount),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PKG_QUERY,FilePanel::onCmdPkgQuery),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PKG_INSTALL,FilePanel::onCmdPkgInstall),
	FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PKG_UNINSTALL,FilePanel::onCmdPkgUninstall),
	FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PKG_QUERY,FilePanel::onUpdPkgQuery),
#endif
};

// Object implementation 
FXIMPLEMENT(FilePanel,FXVerticalFrame,FilePanelMap,ARRAYNUMBER(FilePanelMap))

// Construct File Panel
FilePanel::FilePanel(FXWindow *owner, const FXchar* nm,FXComposite *p, FXuint name_size, FXuint size_size, FXuint type_size, FXuint ext_size,
             FXuint modd_size, FXuint user_size, FXuint grou_size, FXuint attr_size, FXuint deldate_size, FXColor listbackcolor, FXColor listforecolor,
			 FXbool smoothscroll,FXuint opts,FXint x,FXint y,FXint w,FXint h):
             FXVerticalFrame(p,opts,x,y,w,h,0,0,0,0)
{
    name=nm;

    // Global container
    FXVerticalFrame* cont=new FXVerticalFrame(this,LAYOUT_FILL_Y|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0, 1,1);
	
	// Container for the path linker
	FXHorizontalFrame* pathframe=new FXHorizontalFrame(cont,LAYOUT_RIGHT|JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 0,0,0,0);
	
	// File list
    
    // Smooth scrolling
	FXuint options;
	if (smoothscroll)
		options=LAYOUT_FILL_X|LAYOUT_FILL_Y|_ICONLIST_MINI_ICONS;
	else
		options=LAYOUT_FILL_X|LAYOUT_FILL_Y|_ICONLIST_MINI_ICONS|SCROLLERS_DONT_TRACK;

	list=new FileList(owner,cont,this,ID_FILELIST,options);
	list->setHeaderSize(0,name_size);
    list->setHeaderSize(1,size_size);
    list->setHeaderSize(2,type_size);
    list->setHeaderSize(3,ext_size);
    list->setHeaderSize(4,modd_size);
    list->setHeaderSize(5,user_size);
    list->setHeaderSize(6,grou_size);
    list->setHeaderSize(7,attr_size);  	
    list->setHeaderSize(8,deldate_size);  	
	list->setTextColor(listforecolor);
	list->setBackColor(listbackcolor);
	
	// Path linker	
    pathlink = new PathLinker(pathframe,list,LAYOUT_FILL_X);
	
	// Status bar
	statusbar=new FXHorizontalFrame(cont,LAYOUT_RIGHT|JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 3,3,3,3);
    statusbar->setTarget(this);
    statusbar->setSelector(FXSEL(SEL_UPDATE,FilePanel::ID_STATUS));
    new FXToggleButton(statusbar,_("\tShow hidden files (Ctrl-F6)"),_("\tHide hidden files (Ctrl-F6)"),showhiddenicon,hidehiddenicon,this->list,
	                   FileList::ID_TOGGLE_HIDDEN,TOGGLEBUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    new FXToggleButton(statusbar,_("\tShow thumbnails (Ctrl-F7)"),_("\tHide thumbnails (Ctrl-F7)"),showthumbicon,hidethumbicon,this->list,
	                   FileList::ID_TOGGLE_THUMBNAILS,TOGGLEBUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    status=new FXLabel(statusbar,_("Status"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN);
	corner=new FXDragCorner(statusbar);

	// Initializations
	selmult=FALSE;
	current=NULL;
    
	// Single click navigation
    single_click=getApp()->reg().readUnsignedEntry("SETTINGS","single_click",SINGLE_CLICK_NONE);
	if (single_click==SINGLE_CLICK_DIR_FILE)
		list->setDefaultCursor(getApp()->getDefaultCursor(DEF_HAND_CURSOR));

	// Dialogs
	operationdialogsingle=NULL;
	operationdialogmultiple=NULL;
	newdirdialog=NULL;
	newfiledialog=NULL;
	newlinkdialog=NULL;
	opendialog=NULL;
	archdialog=NULL;
	filterdialog=NULL;
	
	// Home and trahscan locations
	homelocation=getenv("HOME");
    if(homelocation=="")
        homelocation=ROOTDIR;
	trashlocation=homelocation+PATHSEPSTRING+TRASHPATH;
	
	// Start location (we return to the start location after each chdir)
	startlocation=FXSystem::getCurrentDirectory();

	// Initialize clipboard flags
	clipboard_locked=FALSE;
	paste=FALSE;
	
	// Initialize control flag for right click popup menu
	ctrlflag=FALSE;
	
	// Initialize the Shift-F10 flag
	shiftf10=FALSE;
}


// Create X window
void FilePanel::create()
{
    // Register standard uri-list type
	urilistType=getApp()->registerDragType("text/uri-list");
    
	// Register special uri-list type used for Gnome, XFCE and Xfe 
	xfelistType=getApp()->registerDragType("x-special/gnome-copied-files");
	
	// Register special uri-list type used for KDE 
	kdelistType=getApp()->registerDragType("application/x-kde-cutselection");

	// Register standard UTF-8 text type used for file dialogs 
	utf8Type=getApp()->registerDragType("UTF8_STRING");
	
	// Display or hide path linker
	FXbool show_pathlink=getApp()->reg().readUnsignedEntry("SETTINGS","show_pathlinker",TRUE);
	if (show_pathlink)
		pathlink->show();
	else
		pathlink->hide();
	
	FXVerticalFrame::create();
}


// Destructor
FilePanel::~FilePanel()
{
    delete list;
    delete current;
    delete next;
    delete status;
	delete statusbar;
	delete pathlink;
	delete newfiledialog;
	delete newlinkdialog;
	delete newdirdialog;
	delete opendialog;
	delete archdialog;
	delete filterdialog;
	delete operationdialogsingle;
	delete operationdialogmultiple;
}


// FilePanel Gets Focus
void FilePanel::focus()
{
	if(this == current)
        return;
    pathlink->focus();
    current=this;
    next->unfocus();
    list->setFocus();
}


// FilePanel Loses Focus
void FilePanel::unfocus()
{
	pathlink->unfocus();
    current=next;
    list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),NULL);
}


// Set Pointer to Another FilePanel
void FilePanel::Next(FilePanel* nxt)
{
    next=nxt;
}


// Show or hide drag corner
void FilePanel::showCorner(FXbool show)
{
	if (show)
		corner->show();
	else
		corner->hide();
}
		

// Update location history when changing directory (home, up or double click)
void FilePanel::updateLocation()
{
    FXString item;
	FXint i=0;
	FXComboBox* address=((XFileExplorer*) mainWindow)->getAddressBox();
  	address->setNumVisible(5);
	FXint count=address->getNumItems();
    FXString p=list->getDirectory();
	
	// Remember latest directory in the location address
	if(!count)
	{
		count++;
		address->insertItem(0,address->getText());
	}
    while(i < count)
	{
    	item=address->getItem(i++);
    	if(::streq((const char*)&p[0],(const char*)&item[0]))
		{
			i--;
			break;
		}
	}
    if(i==count)
		address->insertItem(0,list->getDirectory());
}


// We now really do have the clipboard, keep clipboard content
long FilePanel::onClipboardGained(FXObject* sender,FXSelector sel,void* ptr)
{
    FXVerticalFrame::onClipboardGained(sender,sel,ptr);
    return 1;
}


// We lost the clipboard
long FilePanel::onClipboardLost(FXObject* sender,FXSelector sel,void* ptr)
{
    FXVerticalFrame::onClipboardLost(sender,sel,ptr);
	return 1;
}


// Somebody wants our clipboard content
long FilePanel::onClipboardRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

	// Perhaps the target wants to supply its own data for the clipboard
    if(FXVerticalFrame::onClipboardRequest(sender,sel,ptr))
		return 1;

    // Clipboard target is xfelistType (Xfe, Gnome or XFCE)
	if(event->target==xfelistType)
    {	
		// Don't modify the clipboard if we are called from updPaste()
		if (!clipboard_locked)
		{
			// Prepend "copy" or "cut" as in the Gnome way and avoid duplicating these strings
			if (clipboard.find("copy\n")<0 && clipboard.find("cut\n")<0)
			{
				if (clipboard_type==CUT_CLIPBOARD)
					clipboard = "cut\n" + clipboard;
				else
					clipboard = "copy\n" + clipboard;
			}
		
			// Remove the '\r' because Nautilus won't work with it
			// No effect on other apps (thunar, xfe, ...)
			FXint pos;
			while ((pos=clipboard.find("\r"))>0)
					clipboard.erase(pos);
		}
		
		// Return clipboard content
		if(event->target==xfelistType)
		{
			if(!clipboard.empty())
			{
				len=clipboard.length();
				FXMEMDUP(&data,clipboard.text(),FXuchar,len);
				setDNDData(FROM_CLIPBOARD,event->target,data,len);
				
				// Return because xfelistType is not compatible with other types
				return 1;
			}
		}
    }

    // Clipboard target is kdelisType (KDE)
    if(event->target==kdelistType)
    {
		// The only data to be passed in this case is "0" for copy and "1" for cut
		// The uri data are passed using the standard uri-list type
		FXString flag;
		if (clipboard_type==CUT_CLIPBOARD)
			flag = "1";
		else
			flag = "0";

        // Return clipboard content
        if(event->target==kdelistType)
        {
			FXMEMDUP(&data,flag.text(),FXuchar,1);
			setDNDData(FROM_CLIPBOARD,event->target,data,1);
		}
    }

    // Clipboard target is urilistType (KDE apps ; non Gnome, non XFCE and non Xfe apps)  
    if(event->target==urilistType)
    {
		if(!clipboard.empty())
		{
			len=clipboard.length();
			FXMEMDUP(&data,clipboard.text(),FXuchar,len);
			setDNDData(FROM_CLIPBOARD,event->target,data,len);
		
			return 1;
		}
    }
	
    // Clipboard target is utf8Type (to paste file name to the file dialogs)  
	// Note that we only keep the first file name of the clipboard
    if(event->target==utf8Type)
    {
		if(!clipboard.empty())
		{
			FXint beg, end;
			FXString filename, url;

			// Clipboard don't contain copy or cut as first line
			if (clipboard.find("copy\n")<0 && clipboard.find("cut\n")<0)
			{
				end=clipboard.find("\n",0);
				url=clipboard.mid(0,end-1);
				filename=FXPath::name(FXURL::decode(FXURL::fileFromURL(url)));
			}
			
			// Clipboard contains copy or cut as first line, thus skip it
			else
			{
				beg=clipboard.find("\n",0);
				end=clipboard.find("\n",beg+1);
				url=clipboard.mid(beg+1,end-beg-1);
				filename=FXPath::name(FXURL::decode(FXURL::fileFromURL(url)));				
			}
			
			if(!filename.empty())
			{
				len=filename.length();
				FXMEMDUP(&data,filename.text(),FXuchar,len);
				setDNDData(FROM_CLIPBOARD,event->target,data,len);
			
				return 1;
			}
		}
    }
    return 0;
}


// Copy or cut to clipboard
long FilePanel::onCmdCopyCut(FXObject*,FXSelector sel,void*)
{		
    FXString name,curdir;
	
	// Clear clipboard
	clipboard.clear();

    // Clipboard type
	if (FXSELID(sel)==ID_CUT_CLIPBOARD)
		clipboard_type=CUT_CLIPBOARD;
	else
		clipboard_type=COPY_CLIPBOARD;

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();
			
    if (num==0)
		return 0;

	// If exist selected files, use them
	else if (num>=1)
	{
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

    	// Construct the uri list of files and fill the clipboard with it
    	curdir=current->list->getDirectory();
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
        	if (current->list->isItemSelected(u))
        	{
            	name=current->list->getItemText(u).text();
            	name=name.section('\t',0);
            	if (curdir==ROOTDIR)
					clipboard += FXURL::encode(FXURL::fileToURL(curdir+name))+"\r\n";
				else
					clipboard += FXURL::encode(FXURL::fileToURL(curdir+PATHSEPSTRING+name))+"\r\n";
        	}
		}
	}
		
	// Acquire the clipboard
	FXDragType types[4];
	types[0]=xfelistType;
	types[1]=kdelistType;
	types[2]=urilistType;
	types[3]=utf8Type;
	if (acquireClipboard(types,4))
		return 0;
    
	return 1;
}


// Paste file(s) from clipboard
long FilePanel::onCmdPaste(FXObject*,FXSelector sel,void*)
{
    FXuchar *data;
    FXuint len;
	FXint beg, end, pos;
	FXString chaine, url, param;		  
	FXint num=0;
	FXbool from_kde=FALSE;

	// If source is xfelistType (Gnome, XFCE, or Xfe app)
    if(getDNDData(FROM_CLIPBOARD,xfelistType,data,len))
    {
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		
		clipboard=(FXchar*)data;
		
		// Loop over clipboard items
		for(beg=0; beg<clipboard.length(); beg=end+1)
		{
			if((end=clipboard.find("\n",beg))<0)
				end=clipboard.length();
			
			// Obtain item url
			url=clipboard.mid(beg,end-beg);
				
			// Eventually remove the trailing "\r" if any
			if ((pos=url.rfind("\r"))>0)
				url.erase(pos);

			// Process first item
			if (num==0)
			{
				// First item should be "copy" or "cut"
				if (url=="copy")
				{
					clipboard_type=COPY_CLIPBOARD;
					num++;
				}
				else if (url=="cut")
				{
					clipboard_type=CUT_CLIPBOARD;
					num++;
				}
				
				// If first item is not "copy" nor "cut", process it as a normal url
				// and use default clipboard type
				else
				{											
					// Update the param string
					param += FXURL::decode(FXURL::fileFromURL(url)) + "\n";
						
					// Add one more because the first line "copy" or "cut" was not present
					num += 2;
				}
			}
			
			// Process other items
			else
			{
				// Update the param string
				param += FXURL::decode(FXURL::fileFromURL(url)) + "\n";
				num++;
			}
		}

		// Construct the final param string passed to the file management routine
		param = current->list->getDirectory()+"\n" + FXStringVal(num-1) + "\n" + param;
		
		// Copy or cut operation depending on the clipboard type
		switch(clipboard_type)
		{
		case COPY_CLIPBOARD:
			sel=FXSEL(SEL_COMMAND,FilePanel::ID_FILE_COPY);
			break;
		case CUT_CLIPBOARD:
			clipboard.clear();
			sel=FXSEL(SEL_COMMAND,FilePanel::ID_FILE_CUT);
			break;
		}
		paste=TRUE;
		handle(this,sel,strdup(param.text()));
    		
        // Free data pointer
		FXFREE(&data);
		
		// Return here because xfelistType is not compatible with other types
		return 1;
	}

	// If source type is kdelistType (KDE)
    if(getDNDData(FROM_CLIPBOARD,kdelistType,data,len))
    {		
        from_kde=TRUE;
		
		FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		clipboard=(FXchar*)data;
				
		// Obtain clipboard type (copy or cut)
		if (clipboard=="1")
			clipboard_type=CUT_CLIPBOARD;
		else
			clipboard_type=COPY_CLIPBOARD;
		
		FXFREE(&data);
	}
	
	
	// If source type is urilistType (KDE apps ; non Gnome, non XFCE and non Xfe apps)
    if(getDNDData(FROM_CLIPBOARD,urilistType,data,len))
    {		
		// For non KDE apps, set action to copy
		if (!from_kde)
			clipboard_type=COPY_CLIPBOARD;
		
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		clipboard=(FXchar*)data;
				
		// Loop over clipboard items
		for(beg=0; beg<clipboard.length(); beg=end+1)
		{
			if((end=clipboard.find("\n",beg))<0)
				end=clipboard.length();

			// Obtain item url
			url=clipboard.mid(beg,end-beg);

			// Eventually remove the trailing "\r" if any
			if ((pos=url.rfind("\r"))>0)
				url.erase(pos);

			// Update the param string
			param += FXURL::decode(FXURL::fileFromURL(url)) + "\n";
			num++;
		}

		// Construct the final param string passed to the file management routine
		param = current->list->getDirectory()+"\n" + FXStringVal(num) + "\n" + param;
				
		// Copy or cut operation depending on the clipboard type
		switch(clipboard_type)
		{
		case COPY_CLIPBOARD:
			sel=FXSEL(SEL_COMMAND,FilePanel::ID_FILE_COPY);
			break;
		case CUT_CLIPBOARD:
			clipboard.clear();
			sel=FXSEL(SEL_COMMAND,FilePanel::ID_FILE_CUT);
			break;
		}
		paste=TRUE;
		handle(this,sel,strdup(param.text()));
					
		FXFREE(&data);
		return 1;
	}

	return 0;
}


// Double Click on File Item
long FilePanel::onCmdItemDoubleClicked(FXObject* sender,FXSelector sel, void* ptr)
{
	FXString buf, filename, pathname;

 	// Wait cursor
    getApp()->beginWaitCursor();
	
	long item= (long) ptr;
    if(item > -1)
    {
		// File name and path
		filename=list->getItemFilename(item);
		pathname=list->getItemPathname(item);

        // If directory, open the directory
        if(list->isItemDirectory(item))
        {
            // Does not have access
			if(!::isReadExecutable(pathname))
			{
            	MessageBox::error(this,BOX_OK_SU,_("Error"),_(" Permission to: %s denied."), pathname.text());
				getApp()->endWaitCursor();
                return 0;
			}
            if(filename=="..")
                list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
            else
                list->setDirectory(pathname);
            
			// Change directory in tree list  
			((XFileExplorer*) mainWindow)->getDirPanel()->setDirectory(pathname,TRUE);
			current->updatePathLinker();
			
			// Update location history			
			updateLocation();
        }
        else if(list->isItemFile(item))
        {
			// Update associations dictionary
			FileDict *assocdict=new FileDict(getApp());			
			FileAssoc *association=assocdict->findFileBinding(pathname.text());

            if(association)
                if(association->command.section(',',0).text() != NULL)
                {
					buf=association->command.section(',',0)+" "+::quote(pathname);
					runcmd(buf);
                }
                else if(list->isItemExecutable(item)) //executable
                {
                    buf=::quote(pathname);
                    runcmd(buf);
               }    // If regular file return as the selected file
                else
                    current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
            else if(list->isItemExecutable(item))//executable
            {
                buf=::quote(pathname);
                runcmd(buf);
            }    // If regular file return as the selected file
            else
                current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
        }
    }
	
	getApp()->endWaitCursor();

    return 1;
}


// Single click on File Item
long FilePanel::onCmdItemClicked(FXObject* sender,FXSelector sel, void* ptr)
{
	focus();
		
	if (single_click != SINGLE_CLICK_NONE)
	{		
		// In detailed mode, avoid single click when cursor is not over the first column
		FXint x, y;
		FXuint state;
		getCursorPosition(x,y,state);
		FXbool allow=TRUE;
		if (!(list->getListStyle()&(_ICONLIST_BIG_ICONS|_ICONLIST_MINI_ICONS)) && (x-list->getXPosition())>list->getHeaderSize(0))
			allow=FALSE;
		
		// Single click with control or shift
		if (state&(CONTROLMASK|SHIFTMASK))
			return 1;
		
		// Single click without control or shift
		else
		{
			FXString buf, filename, pathname;

			// Wait cursor
			getApp()->beginWaitCursor();
			
			long item= (long) ptr;
			if(item > -1)
			{
				// File name and path
				filename=list->getItemFilename(item);
				pathname=list->getItemPathname(item);

				// If directory, open the directory
				if((single_click != SINGLE_CLICK_NONE) && list->isItemDirectory(item) && allow)
				{
					// Does not have access
					if(!::isReadExecutable(pathname))
					{
						MessageBox::error(this,BOX_OK_SU,_("Error"),_(" Permission to: %s denied."), pathname.text());
						getApp()->endWaitCursor();
						return 0;
					}
					if(filename=="..")
						list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
					else
						list->setDirectory(pathname);
					
					// Change directory in tree list  
					((XFileExplorer*) mainWindow)->getDirPanel()->setDirectory(pathname,TRUE);
					current->updatePathLinker();
					
					// Update location history			
					updateLocation();
				}
				
				// If file, use the association if any
				else if((single_click==SINGLE_CLICK_DIR_FILE) && list->isItemFile(item) && allow)
				{
					// Update associations dictionary
					FileDict *assocdict=new FileDict(getApp());			
					FileAssoc *association=assocdict->findFileBinding(pathname.text());

					if(association)
						if(association->command.section(',',0).text() != NULL)
						{
							buf=association->command.section(',',0)+" "+::quote(pathname);
							runcmd(buf);
						}
						else if(list->isItemExecutable(item)) //executable
						{
							buf=::quote(pathname);
							runcmd(buf);
					   }    // If regular file return as the selected file
						else
							current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
					else if(list->isItemExecutable(item))//executable
					{
						buf=::quote(pathname);
						runcmd(buf);
					}    // If regular file return as the selected file
					else
					{
						getApp()->endWaitCursor();
						current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
					}
				}
			}
			getApp()->endWaitCursor();
		}
	}
    return 1;
}


// Focus when clicked
long FilePanel::onCmdItemFocus(FXObject* sender,FXSelector sel, void* ptr)
{
	focus();
    return 1;
}


// Go to parent directory
long FilePanel::onCmdDirectoryUp(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->getDirPanel()->setDirectory(current->list->getDirectory(),TRUE);
    current->updatePathLinker();
	updateLocation();
    return 1;
}


// Go to home directory
long FilePanel::onCmdGoHome(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->setDirectory(homelocation);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->getDirPanel()->setDirectory(homelocation,TRUE);
    current->updatePathLinker();
	updateLocation();
    return 1;
}


// Go to trash directory
long FilePanel::onCmdGoTrash(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->setDirectory(trashlocation);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->getDirPanel()->setDirectory(trashlocation,TRUE);
    current->updatePathLinker();
	updateLocation();
    return 1;
}



// Copy/Move/Rename/Symlink file(s)
long FilePanel::onCmdFileMan(FXObject* sender,FXSelector sel,void* ptr)
{
	FXint num;
	FXString src, targetdir, target, name, source;
	FXint firstitem=0, lastitem=0;
	
	// Focus on this panel
	current->list->setFocus();
	
	// Confirmation dialog?
	FXbool ask_before_copy=getApp()->reg().readUnsignedEntry("OPTIONS","ask_before_copy",TRUE);

	// If we are we called from the paste command, get the parameters from the pointer
	if (paste)
	{
		// Reset the flag
		paste=FALSE;

		// Get the parameters
		FXString str=(char*)ptr;
		targetdir=str.section('\n',0);
		num=FXUIntVal(str.section('\n',1));
		src=str.after('\n',2);
		
		// If no item, return
		if(num<=0)
			return 0;
	}
	
	// Obtain the parameters from the file panel
	else
	{
		// Current directory
		FXString curdir=current->list->getDirectory();
		
		// Number of selected items
		num=current->list->getNumSelectedItems();
		
		// If no item, return
		if (num<=0)
			return 0;
		
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

		// Obtain the list of source files and the target directory
		for (int u=0; u<current->list->getNumItems(); u++)
		{
			if (current->list->isItemSelected(u))
			{
				if (firstitem==0)
					firstitem=u;
				lastitem=u;
				name=current->list->getItemText(u).text();
				name=name.section('\t',0);
				src += curdir+PATHSEPSTRING+name+"\n";
			}
		}
		targetdir=current->next->list->getDirectory();
				
		if(!current->next->shown() || FXSELID(sel)==ID_FILE_RENAME)
			targetdir=current->list->getDirectory();
	}
				
	// Number of items in the FileList
	FXint numitems=current->list->getNumItems();

	// Go to target directory
	//chdir(targetdir.text());
	
	// Name and directory of the first source file
	source=src.section('\n',0);
	name=FXPath::name(source);
	FXString dir=FXPath::directory(source);
	
	// Initialise target name	
	if (targetdir!=ROOTDIR)
		target=targetdir+PATHSEPSTRING;
	else
		target=targetdir;
	
    // Configure the command, title, message, etc.
    FXIcon *icon=NULL;
	FXString command, title, message;
	if (FXSELID(sel)==ID_FILE_COPY)
    {
        command="copy";
        title=_("Copy");
        icon=copy_bigicon;
        if(num==1)
        {
            message=_("Copy ");
            message+=source;
		}
        else
            message=_("Copy ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_FILE_RENAME)
    {
        command="rename";
        title=_("Rename");
        icon=move_bigicon;
        if(num==1)
        {
            message=_("Rename ");
			message+=name;
            target=name;
			title=_("Rename");
        }
        else
			return 0;
    }
	if (FXSELID(sel)==ID_FILE_COPYTO)
    {
        command="copy";
        title=_("Copy");
        icon=copy_bigicon;
        if(num==1)
        {
            message=_("Copy ");
            message+=source;
		}
        else
            message=_("Copy ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_FILE_MOVETO)
    {
        command="move";
        title=_("Move");
        icon=move_bigicon;
        if(num==1)
        {
            message=_("Move ");
            message+=source;
            title=_("Move");
        }
        else
            message=_("Move ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_FILE_CUT)
    {
        command="move";
        title=_("Move");
        icon=move_bigicon;
        if(num==1)
        {
            message=_("Move ");
            message+=source;
            title=_("Move");
        }
        else
            message=_("Move ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_FILE_SYMLINK)
    {
        command="symlink";
		title=_("Symlink");
        icon=link_bigicon;
        if(num==1)
        {
            message=_("Symlink ");
            message+=source;
        }
        else
            message=_("Symlink ")+FXStringVal(num)+_(" items from: ")+dir;
    }
	
    // File operation dialog, if needed
	if (ask_before_copy || source==target ||  FXSELID(sel)==ID_FILE_COPYTO || FXSELID(sel)==ID_FILE_MOVETO ||  FXSELID(sel)==ID_FILE_RENAME || FXSELID(sel)==ID_FILE_SYMLINK)
	{   
		if (num==1)
		{
			if (operationdialogsingle==NULL)
				operationdialogsingle=new BrowseInputDialog(this,"","",title,_("To:"),icon,BROWSE_INPUT_MIXED);
			operationdialogsingle->setTitle(title);
			operationdialogsingle->setMessage(message);
			operationdialogsingle->setText(target);
			operationdialogsingle->CursorEnd();
			operationdialogsingle->setDirectory(targetdir);
			int rc=1;
			rc=operationdialogsingle->execute(PLACEMENT_CURSOR);
			target=operationdialogsingle->getText();
			if (!rc)
				return 0;
		}
		else
		{
			if (operationdialogmultiple==NULL)
				operationdialogmultiple=new BrowseInputDialog(this,"","",title,_("To folder:"),icon,BROWSE_INPUT_FOLDER);
			operationdialogmultiple->setTitle(title);
			operationdialogmultiple->setMessage(message);
			operationdialogmultiple->setText(target);
			operationdialogmultiple->CursorEnd();
			operationdialogmultiple->setDirectory(targetdir);
			int rc=1;
			rc=operationdialogmultiple->execute(PLACEMENT_CURSOR);
			target=operationdialogmultiple->getText();
			if (!rc)
				return 0;
		}
	}

	// Nothing entered
	if (target=="")
	{
		MessageBox::warning(this,BOX_OK,_("Warning"),_("File name is empty, operation cancelled"));
		return 0;
	}
	
	// Update target and target parent directory
	target=::filePath(target,targetdir);
		
    // Target directory not writable
    if(!::isWritable(targetdir))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"), _("Can't write to %s: Permission denied"), targetdir.text());
        return 0;
    }

	// Multiple sources and non existent destination
	if (num>1 && !::exists(target))
	{
        MessageBox::error(this,BOX_OK,_("Error"),_("Folder %s doesn't exist"),target.text());
        return 0;
	}
		
	// Multiple sources and target is a file
	if (num>1 && ::isFile(target))
	{ 			
        MessageBox::error(this,BOX_OK,_("Error"),_("%s is not a folder"),target.text());
        return 0;
	}

	// Target is a directory and is not writable
    if (::isDirectory(target) && !::isWritable(target))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"), target.text());
        return 0;
    }
	
	// Target is a file and its parent directory is not writable
	if (::isFile(target) && !::isWritable(targetdir))
	{
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),targetdir.text());
        return 0;
	}

	// Target parent directory doesn't exist
	if (!::exists(targetdir))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Folder %s doesn't exist"),targetdir.text());
        return 0;
	}
		
	// Target parent directory is not a directory
	if (!::isDirectory(targetdir))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("%s is not a folder"),targetdir.text());
        return 0;
	}
		
	// One source
	File *f=NULL;
	FXbool ret;
    if(num==1)
	{
		// An empty source file name corresponds to the ".." file
		// Don't perform any file operation on it!
		if (source=="")
			return 0;
		
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
        if(command=="copy")
        {
			f=new File(this,_("File copy"),COPY);
        	f->create();
			ret=f->copy(source,target);
			
			// An error has occurred
			if (ret==0 && !f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the copy file operation!"));
			}
					
			// If action is cancelled in progress dialog
			if (f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("Copy file operation cancelled!"));
			}
        }
        else if(command=="rename")
        {
			f=new File(this,_("File rename"),RENAME);
        	f->create();
			f->rename(source,target);
        }
        else if(command=="move")
        {
			f=new File(this,_("File move"),MOVE);
        	f->create();
			ret=f->move(source,target);

			// An error has occurred
			if (ret==0 && !f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move file operation!"));
			}
					
			// If action is cancelled in progress dialog
			if (f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("Move file operation cancelled!"));
			}
        }
        else if(command=="symlink")
        {
			f=new File(this,_("Symlink"),SYMLINK);
        	f->create();
			f->symlink(source,target);
		}
       	else
           	exit(-1);
									
		getApp()->endWaitCursor();
		delete f;
	}
	
	// Multiple sources
	// Note : rename cannot be used in this case!
    else if(num>1)
    {
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
        if (command=="copy")
			f=new File(this,_("File copy"),COPY);
		else if (command=="move")
			f=new File(this,_("File move"),MOVE);
		else if  (command=="symlink")
			f=new File(this,_("Symlink"),SYMLINK);
			
		f->create();

		// Loop on the multiple files
		for(int i=0;i<num;i++)
        {
			// Individual source file
			source=src.section('\n',i);

			// An empty file name corresponds to the ".." file (why?)
			// Don't perform any file operation on it! 
			if (source!="")
			{
            	if(command=="copy")
				{
					ret=f->copy(source,target);
					
					// An error has occurred
					if (ret==0 && !f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the copy file operation!"));
						break;
					}
					
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Copy file operation cancelled!"));
						break;
					}
				}
            	else if(command=="move")
            	{
					ret=f->move(source,target);
				
					// An error has occurred
					if (ret==0 && !f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move file operation!"));
						break;
					}
					
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Move file operation cancelled!"));
						break;
					}
           	 	}
				else if (command=="symlink")
				{
					ret=f->symlink(source,target);
					
					// An error has occurred
					if (ret==0 && !f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the symlink operation!"));
						break;
					}
					
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Symlink operation cancelled!"));
						break;
					}
				}
            	else
                	exit(-1);
			}
		}
		
		getApp()->endWaitCursor();		
		delete f;
	}	
	
	// Force panel refresh
	onCmdRefresh(0,0,0);
	
	// Enable previous or last selected item for keyboard navigation
	if ((FXSELID(sel)==ID_FILE_MOVETO || FXSELID(sel)==ID_FILE_RENAME) && current->list->getNumItems()<numitems)
	{
		firstitem=(firstitem<1) ? 0 : firstitem-1;
		current->list->enableItem(firstitem);
		current->list->setCurrentItem(firstitem);
	}
	else
	{
		current->list->enableItem(lastitem);
		current->list->setCurrentItem(lastitem);
	}

    return 1;
}



// Trash files from the file list or the tree list
long FilePanel::onCmdFileTrash(FXObject*,FXSelector,void*)
{
	FXint firstitem=0;
	File *f=NULL;
    current->list->setFocus();
	FXString dir=current->list->getDirectory();
    //chdir(dir.text());

    FXbool confirm_del=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_delete",TRUE);
    FXbool confirm_del_emptydir=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_delete_emptydir",TRUE);
	
	// If we don't have permission to write to the parent directory
    if(!::isWritable(dir))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),dir.text());
        return 0;
    }

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();
	
	if (num==0)
		return 0;
	
	// If exist selected files, use them
	else if (num>=1)
	{
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

		if(confirm_del)
    	{
        	FXString message;
			if (num==1)
				message=_("  Move the selected item to trash can?  ");
			else
				message=_("  Move ")+FXStringVal(num)+_(" selected items to trash can?  ");
			MessageBox box(this,_("Confirm Trash"),message,delete_bigicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        	if(box.execute(PLACEMENT_CURSOR) != BOX_CLICKED_OK)
            	return 0;
   		}

		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
   		f=new File(this,_("Move to trash"),DELETE);
   		f->create();
		list->setAllowRefresh(FALSE);	

		// Overwrite initialisations
		FXbool overwrite=FALSE;
		FXbool overwrite_all=FALSE;
		FXbool skip_all=FALSE;
		
		// Trash initialisation
		FXbool trashdel=FALSE;

		// Delete selected files
		FXString filename, pathname;    			
		for(int u=0;u< current->list->getNumItems (); u++)
    	{
        	if (current->list->isItemSelected (u))
        	{

            	// Get index of first selected item
				if (firstitem==0)
					firstitem=u;
				
				// Get file name and path
				filename=current->list->getItemFilename(u);
				pathname=current->list->getItemPathname(u);
	
				// Confirm empty directory deletion
				if (confirm_del & confirm_del_emptydir)
				{
					if (::isEmptyDir(pathname)==0)
					{
						// Dialog to confirm file deletion
						f->allowProgressDialog=FALSE;
						f->hide();

						FXString message=_("Folder ") + pathname + _(" is not empty, move it anyway to trash can?");
						MessageBox box(this,_("Confirm Trash"),message,delete_bigicon,BOX_YES_NO_CANCEL|DECOR_TITLE|DECOR_BORDER);
						FXuint answer=box.execute(PLACEMENT_CURSOR);				
						if(answer == BOX_CLICKED_NO)
							continue;
						else if(answer == BOX_CLICKED_CANCEL)
							goto end;
					}
				}

				// If we don't have permission to write to the file
 				if (!::isWritable(pathname))
    			{
					// Dialog to confirm file deletion
					f->allowProgressDialog=FALSE;
					f->hide();
				
					// Overwrite dialog if necessary
					if (!(overwrite_all | skip_all))
					{
						FXString msg=_("File ") + pathname + _(" is write-protected, move it anyway to trash can?");
						ConfirmOverWriteBox* dlg=new ConfirmOverWriteBox(this,_("Confirm Trash"),msg);
						FXuint answer=dlg->execute(PLACEMENT_SCREEN);
						delete dlg;
						switch(answer)
						{
							case 1:
								overwrite=TRUE;
								break;
							case 2:
								overwrite_all=TRUE;
								break;
							case 3:
								overwrite=FALSE;
								break;
							case 4:
								skip_all=TRUE;
								break;
						}
					}
					if((overwrite | overwrite_all) & !skip_all)
					{
						// Caution!! Don't delete parent directory!!
						if (filename!="..")
						{
							// File or folder is in the trash can
							if (pathname.find(trashlocation.text())==0)
							{
								// Confirm delete messsage
								if (!trashdel)
								{
									f->allowProgressDialog=FALSE;
									f->hide();
									FXString message;
									if (::isDirectory(pathname))
										message=_("Folder is already in the trash can! Definitively delete folder ") + pathname + " ?";
									else
										message=_("File is already in the trash can! Definitively delete file ") + pathname + " ?";	
									MessageBox box(this,_("Delete from trash can"),message,delete_big_permicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
									if(box.execute(PLACEMENT_CURSOR) == BOX_CLICKED_OK)
									{
										trashdel=TRUE;
										f->remove(pathname);
									}
									f->allowProgressDialog=TRUE;
								}
								else
									f->remove(pathname);
							}
							else
							{
								// Create trash can if it doesn't exist
								if (!exists(trashlocation))
									::mkdir(trashlocation.text(),0755);
								
								// Move file or folder to trash can
								FXbool ret=f->move(pathname,trashlocation+PATHSEPSTRING+stampname(pathname));
								
								// An error has occurred
								if (ret==0 && !f->isCancelled)
								{
									f->hide();
									MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move to trash operation!"));
								}
							}
						}
					}
					f->allowProgressDialog=TRUE;
    			}

				// If we have permission to write
				else
				{
					// Caution!! Don't delete parent directory!!
					if (filename!="..")
					{
						// File or folder is in the trash can
						if (pathname.find(trashlocation.text())==0)
						{
							// Confirm delete message
							if (!trashdel)
							{
								f->allowProgressDialog=FALSE;
								f->hide();
								FXString message;
								if (::isDirectory(pathname))
									message=_("Folder is already in the trash can! Definitively delete folder ") + pathname + " ?";
								else
									message=_("File is already in the trash can! Definitively delete file ") + pathname + " ?";											
								MessageBox box(this,_("Delete from trash can"),message,delete_big_permicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
								if(box.execute(PLACEMENT_CURSOR) == BOX_CLICKED_OK)
								{
									trashdel=TRUE;
									f->remove(pathname);
								}
								f->allowProgressDialog=TRUE;
							}
							else
								f->remove(pathname);
						}
						else
						{
							// Create trash can if it doesn't exist
							if (!exists(trashlocation))
								::mkdir(trashlocation.text(),0755);
							
							// Move file or folder to trash can
							FXbool ret=f->move(pathname,trashlocation+PATHSEPSTRING+stampname(pathname));
							
							// An error has occurred
							if (ret==0 && !f->isCancelled)
							{
								f->hide();
								MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move to trash operation!"));
							}
						}
					}
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Move to trash file operation cancelled!"));
						break;
					}
				}
        	}
    	}
end:
		getApp()->endWaitCursor();
		delete f;
	}
	// Force FilePanel and DirPanel refresh
	list->setAllowRefresh(TRUE);	
	onCmdRefresh(0,0,0);

	// Enable last item before the first selected item (for keyboard navigation)
	firstitem=(firstitem<1) ? 0 : firstitem-1;
	current->list->enableItem(firstitem);
	current->list->setCurrentItem(firstitem);
	
	return 1;
}



// Definitively delete files from the file list or the tree list (no trash can)
long FilePanel::onCmdFileDelete(FXObject*,FXSelector,void*)
{
	FXint firstitem=0;
	File *f=NULL;
    current->list->setFocus();
	FXString dir=current->list->getDirectory();
    //chdir(dir.text());

    FXbool confirm_del=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_delete",TRUE);
    FXbool confirm_del_emptydir=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_delete_emptydir",TRUE);
	
	// If we don't have permission to write to the parent directory
    if(!::isWritable(dir))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),dir.text());
        return 0;
    }

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();

    // If no selected files, use the selected folder from the tree list (if any)
	if(num==0)
		return 0;
	
	// If exist selected files, use them
	else if (num>=1)
	{
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

		if(confirm_del)
    	{
        	FXString message;
			if (num==1)
				message=_("  Definitively delete the selected item?  ");
			else
				message=_("  Definitely delete ")+FXStringVal(num)+_(" selected items?  ");
			MessageBox box(this,_("Confirm Delete"),message,delete_big_permicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        	if(box.execute(PLACEMENT_CURSOR) != BOX_CLICKED_OK)
            	return 0;
		}
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
   		f=new File(this,_("File delete"),DELETE);
   		f->create();
		list->setAllowRefresh(FALSE);	

		// Overwrite initialisations
		FXbool overwrite=FALSE;
		FXbool overwrite_all=FALSE;
		FXbool skip_all=FALSE;
		
		// Delete selected files
		FXString filename, pathname;    			
		for(int u=0;u< current->list->getNumItems (); u++)
    	{
        	if (current->list->isItemSelected (u))
        	{
            	// Get index of first selected item
				if (firstitem==0)
					firstitem=u;

            	// Get file name and path
				filename=current->list->getItemFilename(u);
				pathname=current->list->getItemPathname(u);

				// Confirm empty directory deletion
				if (confirm_del & confirm_del_emptydir)
				{
					if (::isEmptyDir(pathname)==0)
					{
						// Dialog to confirm file deletion
						f->allowProgressDialog=FALSE;
						f->hide();

						FXString message=_("Folder ") + pathname + _(" is not empty, delete it anyway?");
						MessageBox box(this,_("Confirm Delete"),message,delete_big_permicon,BOX_YES_NO_CANCEL|DECOR_TITLE|DECOR_BORDER);
						FXuint answer=box.execute(PLACEMENT_CURSOR);				
						if(answer == BOX_CLICKED_NO)
							continue;
						else if(answer == BOX_CLICKED_CANCEL)
							goto end;
					}
				}

				// If we don't have permission to write to the file
 				if (!::isWritable(pathname))
    			{
					// Dialog to confirm file deletion
					f->allowProgressDialog=FALSE;
					f->hide();
				
					// Overwrite dialog if necessary
					if (!(overwrite_all | skip_all))
					{
						FXString msg=_("File ") + pathname + _(" is write-protected, delete it anyway?");
						ConfirmOverWriteBox* dlg=new ConfirmOverWriteBox(this,_("Confirm Delete"),msg);
						FXuint answer=dlg->execute(PLACEMENT_SCREEN);
						delete dlg;
						switch(answer)
						{
							case 1:
								overwrite=TRUE;
								break;
							case 2:
								overwrite_all=TRUE;
								break;
							case 3:
								overwrite=FALSE;
								break;
							case 4:
								skip_all=TRUE;
								break;
						}
					}
					if((overwrite | overwrite_all) & !skip_all)
					{
						// Caution!! Don't delete parent directory!!
						if (filename!="..")
						{
							// Definitively remove file or folder
							f->remove(pathname);
						}
					}
					f->allowProgressDialog=TRUE;
    			}

				// If we have permission to write
				else
				{
					// Caution!! Don't delete parent directory!!
					if (filename!="..")
					{
						// Definitively remove file or folder
						f->remove(pathname);
					}
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Delete file operation cancelled!"));
						break;
					}
				}
        	}
    	}
end:
		getApp()->endWaitCursor();
		delete f;
	}
	// Force FilePanel and DirPanel refresh
	list->setAllowRefresh(TRUE);	
	onCmdRefresh(0,0,0);

	// Enable last item before the first selected item (for keyboard navigation)
	firstitem=(firstitem<1) ? 0 : firstitem-1;
	current->list->enableItem(firstitem);
	current->list->setCurrentItem(firstitem);

	return 1;
}


// Edit file
long FilePanel::onCmdEdit(FXObject*,FXSelector s,void*)
{
    // Wait cursor
    getApp()->beginWaitCursor();

	FXString txteditor=getApp()->reg().readStringEntry("PROGS","txteditor","xfw");
    FXString txtviewer=getApp()->reg().readStringEntry("PROGS","txtviewer","xfv");

    FXString pathname, samecmd, cmd, itemslist=" "; 
	FileAssoc* association;
	FXbool same=TRUE;
	FXbool first=TRUE;
	
	current->list->setFocus();
    //chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
	{
		getApp()->endWaitCursor();
		return 0;
	}
    
	// Update associations dictionary
	FileDict *assocdict=new FileDict(getApp());			
	
	// Check if all files have the same association
	for (FXint u=0; u< current->list->getNumItems (); u++)
	{
		if (current->list->isItemSelected(u))
		{
			// Increment number of selected items
			pathname=current->list->getItemPathname(u);
			association=assocdict->findFileBinding(pathname.text());
		
			// If there is an association
			if (association)
			{
				// Use it to edit/view the files
				if(FXSELID(s)==ID_EDIT)
				{
					cmd=association->command.section(',',2);
					if(cmd.length()==0)
						cmd=txteditor;
				}
				else
				{
					cmd=association->command.section(',',1);
					if(cmd.length()==0)
						cmd=txtviewer;
				}
				if (cmd.text() != NULL)
				{
					
					// First selected item
					if (first)
					{
						samecmd = cmd;
						first=FALSE;
					}
					
					if (samecmd != cmd)
					{
						same=FALSE;
						break;
					}
					
					// List of selected items
					itemslist += ::quote(pathname) + " ";
				}
				else
				{
					same=FALSE;
					break;
				}
			}
			
			// No association
			else
			{
				same=FALSE;
				break;
			}
		}
	}
	
	// Same association for all files : execute the associated or default editor or viewer
	if (same)
	{
		cmd = samecmd + itemslist;
		runcmd(cmd);
	}
	
	// Files have different associations : handle them separately
	else
	{
		for (int u=0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected(u))
			{
				pathname=current->list->getItemPathname(u);
				association=assocdict->findFileBinding(pathname.text());
				
				// If there is an association
				if(association)
				{
					// Use it to edit/view the file
					if(FXSELID(s)==ID_EDIT)
					{
						cmd=association->command.section(',',2);
						if(cmd.length()==0)
							cmd=txteditor;
					}
					else
					{
						cmd=association->command.section(',',1);
						if(cmd.length()==0)
							cmd=txtviewer;
					}
					if(cmd.text() != NULL)
					{
						cmd = cmd + " " + ::quote(pathname);
						runcmd(cmd);
					}
				}
				
				// No association
				else
				{
					if(FXSELID(s)==ID_EDIT)
						cmd=txteditor;
					else
						cmd=txtviewer;
					
					cmd = cmd + " " + ::quote(pathname);
					runcmd(cmd);
				}
			}
		}
	}
	
	getApp()->endWaitCursor();

    return 1;
}


// File or directory properties 
long FilePanel::onCmdProperties(FXObject* sender,FXSelector,void*)
{
    current->list->setFocus();

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);

	// If no selected files in the file list, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->getDirPanel()->getCurrentItem();
		if (!item)
			return 0;
			
		FXString pathname=((XFileExplorer*) mainWindow)->getDirPanel()->getItemPathname(item);
    	FXString path=FXPath::directory(pathname);
    	FXString filename=FXPath::name(pathname);
        PropertiesBox* attrdlg=new PropertiesBox(this,filename,path);
        if(attrdlg->execute(PLACEMENT_SCREEN))
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
            current->updatePathLinker();
        }
        delete attrdlg;
	}

	// There is one selected file in the file list
	else if (num==1)
    {
		// Eventually deselect the '..' directory
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

    	FXString path=current->list->getDirectory();
        FXString filename=current->list->getItemText(itm);
        filename=filename.section('\t',0);
        PropertiesBox* attrdlg=new PropertiesBox(this,filename,path);
        if(attrdlg->execute(PLACEMENT_SCREEN))
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
            current->updatePathLinker();
        }
        delete attrdlg;
    }
    
	// There are multiple selected files in the file list
	else if (num>1)
    {
		chdir(current->list->getDirectory().text());
    	FXString path=current->list->getDirectory();
        FXString *files=new FXString[num];

        int i= 0;
        for (int u= 0; u< current->list->getNumItems (); u++)
            if (current->list->isItemSelected(u))
            {
                files[i]=current->list->getItemText(u).section('\t',0);
                i++;
            }
			
        PropertiesBox* attrdlg=new PropertiesBox(this,files,num,path);
        if(attrdlg->execute(PLACEMENT_SCREEN))
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
            current->updatePathLinker();
        }
        delete attrdlg;
        delete[]files;
		chdir(startlocation.text());
    }
	return 1;
}


// Create new directory
long FilePanel::onCmdNewDir(FXObject*,FXSelector,void*)
{
	FXString dirname="";
	
	// Focus on current panel
	current->list->setFocus();

    FXString dirpath=current->list->getDirectory();
    if(dirpath!=ROOTDIR)
        dirpath+=PATHSEPSTRING;

    if (newdirdialog==NULL)
		newdirdialog=new InputDialog(this,"",_("Create new folder:"),_("New Folder"));		
	newdirdialog->setText("");

	// Accept was pressed
    if(newdirdialog->execute(PLACEMENT_CURSOR))
    {
		dirname=dirpath+newdirdialog->getText();
        if(dirname!=dirpath)
        {
			// Create the new dir according to the current umask
			int mask;
			mask=umask(0);
			umask(mask);
			// Note that the umask value is in decimal (511 means octal 0777)
			FXint ret=::mkdir(dirname.text(),511 & ~mask);
			if (ret==-1)
			{
				if (errno)
					MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't create folder %s: %s",dirname.text(),strerror(errno));
				else
					MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't create folder %s",dirname.text());
    			return 0;
			}
		}							
    }
	
	// Cancel was pressed
	else
		return 0;

	// Force panel refresh
	onCmdRefresh(0,0,0);

	// Enable created item, if any (for keyboard navigation)
	FXString name;
	for (FXint u=0; u< current->list->getNumItems (); u++)
	{
		name=current->list->getItemPathname(u);
		if (name==dirname)
		{
			current->list->enableItem(u);
			current->list->setCurrentItem(u);
			break;
		}
	}

    return 1;
}


// Create new file
long FilePanel::onCmdNewFile(FXObject*,FXSelector,void*)
{
	FXString filename="";
	
	// Focus on current panel
    current->list->setFocus();

    FXString pathname=current->list->getDirectory();
    if(pathname!=ROOTDIR)
        pathname+=PATHSEPSTRING;

    if (newfiledialog==NULL)
		newfiledialog=new InputDialog(this,"",_("Create new file:"),_("New File"),"",NULL,FALSE);
	newfiledialog->setText("");
    
	// Accept was pressed
	if(newfiledialog->execute(PLACEMENT_CURSOR))
    {	
		filename=pathname+newfiledialog->getText();
		FILE *file;
		if(filename!=pathname)
        {
			// Test some error conditions
			if (::exists(filename))
        	{
				MessageBox::error(this,BOX_OK,_("Error"),_("File or folder %s already exists"), filename.text());
        		return 0;
			}
			// Create the new file
			if (!(file=fopen(filename.text(),"w+"))  ||  fclose(file) )
			{
				if (errno)
					MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't create file %s: %s",filename.text(),strerror(errno));
				else
					MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't create file %s",filename.text());
    			return 0;
			}
			// Change the file permissions according to the current umask
			int mask;
			mask=umask(0);
			umask(mask);
			if (chmod(filename.text(), 438 & ~mask) != 0)
			{
				if (errno)
					MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't set permissions in %s: %s",filename.text(),strerror(errno));
				else
					MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't set permissions in %s",filename.text());
			}
		}							
    }
	
	// Cancel was pressed
	else
		return 0;
	
	// Force panel refresh
	onCmdRefresh(0,0,0);

	// Enable created item, if any (for keyboard navigation)
	FXString name;
	for (FXint u=0; u< current->list->getNumItems (); u++)
	{
		name=current->list->getItemPathname(u);
		if (name==filename)
		{
			current->list->enableItem(u);
			current->list->setCurrentItem(u);
			break;
		}
	}
	
	return 1;
}


// Create new symbolic link
long FilePanel::onCmdNewSymlink(FXObject*,FXSelector,void*)
{
	FXString linkname="";
	
	// Focus on current panel
    current->list->setFocus();

    FXString linkpath=current->list->getDirectory();
    if(linkpath!=ROOTDIR)
        linkpath+=PATHSEPSTRING;

    if (newlinkdialog==NULL)
		newlinkdialog=new InputDialog(this,"",_("Create new symbolic link:"),_("New Symlink"),"",NULL,FALSE);
	newlinkdialog->setText("");

	// Accept was pressed
    if(newlinkdialog->execute(PLACEMENT_CURSOR))
    {
		if (newlinkdialog->getText()=="")
		{
			MessageBox::warning(this,BOX_OK,_("Warning"),_("File name is empty, operation cancelled"));
			return 0;
		}
		linkname=linkpath+newlinkdialog->getText();
		File *f;
		if(linkname!=linkpath)
        {
			// Test some error conditions
			if (::exists(linkname))
        	{
				MessageBox::error(this,BOX_OK,_("Error"),_("File or folder %s already exists"), linkname.text());
        		return 0;
			}
			
			// Select target
			FileDialog browse(this,_("Select the symlink refered file or folder"));
			browse.setDirectory(linkpath);
			browse.setSelectMode(SELECT_FILE_MIXED);
			if(browse.execute())
			{
				FXString linksource=browse.getFilename();
				
				// Source does not exist
				if (!::exists(linksource))
				{
					MessageBox::error(this,BOX_OK,_("Error"),_("Symlink source %s does not exist"), linksource.text());
					return 0;
				}
				
				f=new File(this,_("Symlink"),SYMLINK);
				f->create();
				f->symlink(linksource,linkname);
				delete f;
			}
			//else
				//return 0;
		}							
    }

	// Cancel was pressed
	else
		return 0;
	
	// Force panel refresh
	onCmdRefresh(0,0,0);

	// Enable created item, if any (for keyboard navigation)
	FXString name;
	for (FXint u=0; u< current->list->getNumItems (); u++)
	{
		name=current->list->getItemPathname(u);
		if (name==linkname)
		{
			current->list->enableItem(u);
			current->list->setCurrentItem(u);
			break;
		}
	}
	
	return 1;
}


// Open single or multiple files
long FilePanel::onCmdOpen(FXObject*,FXSelector,void*)
{
	// Wait cursor
	getApp()->beginWaitCursor();

    FXString pathname, samecmd, cmd, itemslist=" "; 
	FileAssoc* association;
	FXbool same=TRUE;
	FXbool first=TRUE;
	
	current->list->setFocus();
    //chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
	{
		getApp()->endWaitCursor();
		return 0;
	}
    
	// Update associations dictionary
	FileDict *assocdict=new FileDict(getApp());			
	
	// Check if all files have the same association
	for (FXint u=0; u< current->list->getNumItems (); u++)
	{
		if (current->list->isItemSelected(u))
		{
			// Increment number of selected items
			pathname=current->list->getItemPathname(u);
			association=assocdict->findFileBinding(pathname.text());
		
			if (association)
			{
				cmd = association->command.section(',',0);
				if (cmd.text() != NULL)
				{
					
					// First selected item
					if (first)
					{
						samecmd = cmd;
						first=FALSE;
					}
					
					if (samecmd != cmd)
					{
						same=FALSE;
						break;
					}
					
					// List of selected items
					itemslist += ::quote(pathname) + " ";
				}
				else
				{
					same=FALSE;
					break;
				}
			}
			else
			{
				same=FALSE;
				break;
			}
		}
	}
	
	// Same command for all files : execute it
	if (same)
	{
		cmd = samecmd + itemslist;
		runcmd(cmd);
		
	}
	
	// Files have different commands : handle them separately
	else
	{
		for (int u=0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected(u))
			{
				pathname=current->list->getItemPathname(u);
				association=assocdict->findFileBinding(pathname.text());
				
				// If there is an association
				if(association)
				{
					// Use it to open the file
					cmd = association->command.section(',',0);
					if(cmd.text() != NULL)
					{
						cmd = cmd + " " + ::quote(pathname);
						runcmd(cmd);
					}
					
					// or execute it
					else if(current->list->isItemExecutable(u))
					{
						cmd = ::quote(pathname);
						runcmd(cmd);
					}
					
					// or call the "Open with..." dialog
					else
						current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
				}
				
				// If no association but executable
				else if(current->list->isItemExecutable(u))
				{
					cmd = ::quote(pathname);
					runcmd(cmd);
				}
				
				// Other cases
				else
				{
					getApp()->endWaitCursor();
					current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
				}
			}
		}
	}
	
	getApp()->endWaitCursor();

    return 1;
}

// Open with
long FilePanel::onCmdOpenWith(FXObject*,FXSelector,void*)
{
	char **str=NULL;
    current->list->setFocus();
    int i;
    //chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
		return 0;

    FXString command="";
	if (opendialog==NULL)
		opendialog=new HistInputDialog(this,"",_("Open selected file(s) with:"),_("Open With"), "", NULL, HIST_INPUT_EXECUTABLE_FILE, TRUE, _("A&ssociate"));
	opendialog->setText(command);
    
	// Dialog with history list and associate checkbox
    opendialog->clearItems();
	for(int i=0;i<OpenNum;i++)
        opendialog->appendItem(OpenHistory[i]);
    opendialog->CursorEnd();
	opendialog->setDirectory(ROOTDIR);
    if(opendialog->execute())
    {
        command=opendialog->getText();
		if (command=="")
		{			
			MessageBox::warning(this,BOX_OK,_("Warning"),_("File name is empty, operation cancelled"));
			return 0;
		}
		
        for (int u= 0; u< current->list->getNumItems (); u++)
		{
            if (current->list->isItemSelected(u))
            {
                // Handles "associate" checkbox for "open with..." dialog
                if (opendialog->getOption())
                {
                    FXString filename=current->list->getItemFilename(u);
					FXString ext=FXPath::extension(filename).text();
                    if (ext == "")
						ext=FXPath::name(filename);

					// Convert extension to lower case
					ext=ext.lower();

                    FileAssoc* association=current->list->getItemAssoc(u);

                    if (association)
                    {
                        // Update existing association
                        FXString oldfileassoc=getApp()->reg().readStringEntry("FILETYPES", ext.text(),"");
						oldfileassoc.erase(0, oldfileassoc.section(';',0).section(',',0).length());
                        oldfileassoc.prepend(opendialog->getText());
                        getApp()->reg().writeStringEntry("FILETYPES", ext.text(), oldfileassoc.text());

						// Handle file association
						str=new char*[2];
						str[0]=new char[strlen(ext.text())+1];
						str[1]=new char[strlen(oldfileassoc.text())+1];
						strlcpy(str[0],ext.text(),ext.length()+1);
						strlcpy(str[1],oldfileassoc.text(),oldfileassoc.length()+1);
						mainWindow->handle(this, FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC), str);
                    }
                    else
                    {
                        // New association
						FXString command=opendialog->getText().append(";Document;;;;");
						getApp()->reg().writeStringEntry("FILETYPES", ext.text(), command.text());
                  
						// Handle file association
						str=new char*[2];
						str[0]=new char[strlen(ext.text())+1];
						str[1]=new char[strlen(command.text())+1];
						strlcpy(str[0],ext.text(),ext.length()+1);
						strlcpy(str[1],command.text(),command.length()+1);
						mainWindow->handle(this, FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC),str);
                    }
                }
                // End
				
				FXString pathname=current->list->getItemPathname(u);
				command+=" ";
                command=command+::quote(pathname);
            }
		}
		
		// Run command
		getApp()->beginWaitCursor();
        runcmd(command);

        OpenNum=opendialog->getHistorySize();
        OpenNum++;
        command=opendialog->getText();
        if(OpenNum>OPEN_WITH_HIST_SIZE)
            OpenNum=OPEN_WITH_HIST_SIZE;
        for(i=0;i<OpenNum-1;i++)
            if(::streq(OpenHistory[i],command.text()))
                break;
        if(i==OpenNum-1)
        {
            strlcpy(OpenHistory[0],command.text(),command.length()+1);
            for(i=1;i<OpenNum;i++)
                strlcpy(OpenHistory[i],opendialog->getHistoryItem(i-1).text(),opendialog->getHistoryItem(i-1).length()+1);
        }
        else
            OpenNum--;

		getApp()->endWaitCursor();
    }
    	
	// Force panel refresh
	//onCmdRefresh(0,0,0);
    return 1;
}


long FilePanel::onCmdItemFilter(FXObject*,FXSelector,void*)
{
    current->list->setFocus();
#define FILTER_HIST_SIZE 20

    static char history[FILTER_HIST_SIZE][100];
    static int no=10;
    if (no==10)
	{
		strlcpy(history[0],"*",sizeof(history[0]));
    	strlcpy(history[1],"*.txt",sizeof(history[1]));
    	strlcpy(history[2],"*.c",sizeof(history[2]));
    	strlcpy(history[3],"*.cc",sizeof(history[3]));
    	strlcpy(history[4],"*.cpp",sizeof(history[4]));
    	strlcpy(history[5],"*.h",sizeof(history[5]));
    	strlcpy(history[6],"*.html",sizeof(history[6]));
    	strlcpy(history[7],"*.pdf",sizeof(history[7]));
    	strlcpy(history[8],"*.jpg",sizeof(history[8]));
    	strlcpy(history[9],"*.jpeg",sizeof(history[9]));
	}
    int i;
    FXString pat=list->getPattern();
	if (filterdialog==NULL)
		filterdialog=new HistInputDialog(this,pat,_("Show files:"),_("Filter"),"",NULL,HIST_INPUT_FILE);
    filterdialog->CursorEnd();
    filterdialog->SelectAll();
    filterdialog->clearItems();
	for(int i=0;i<no;i++)
        filterdialog->appendItem(history[i]);

    if(filterdialog->execute() && (pat=filterdialog->getText()) != "")
    {
        list->setPattern(pat);
        no=filterdialog->getHistorySize();
        no++;
        if(no>FILTER_HIST_SIZE)
            no=FILTER_HIST_SIZE;

        for(i=0;i<no-1;i++)
            if(::streq(history[i],pat.text()))
                break;
        if(i==no-1)
        {
            strlcpy(history[0],pat.text(),pat.length()+1);
            for(i=1;i<no;i++)
                strlcpy(history[i],filterdialog->getHistoryItem(i-1).text(),filterdialog->getHistoryItem(i-1).length()+1);
        }
        else
            no--;
    }
    return 1;
}


// Panel context menu
long FilePanel::onCmdPopupMenu(FXObject* o,FXSelector s,void* p)
{
    // FilePanel focus
	focus();

	// Check if control key or Shift-F10 was pressed
	if (p!=NULL)
	{
		FXEvent* event=(FXEvent*)p;
		if (event->state&CONTROLMASK)
			ctrlflag=TRUE;
		if (event->state&SHIFTMASK && event->code==KEY_F10)
			shiftf10=TRUE;
	}
	
	// Use to select the item under cursor when right clicking
	// Only when Shift-F10 was not pressed
    if(!shiftf10 && list->getNumSelectedItems()<=1)
    {
        FXint x,y;
        FXuint state;
        list->getCursorPosition(x,y,state);
 
        int item=list->getItemAt(x,y);

        if (list->getCurrentItem ()>= 0)
            list->deselectItem(list->getCurrentItem());
        if (item>= 0)
        {
            list->setCurrentItem(item);
            if(!list->selectItem(item));            
        }
    }
    
	// If first item (i.e. the '..' item)
	if (list->getNumSelectedItems()==1 && list->isItemSelected(0))
		ctrlflag=TRUE;
	
	// If control flag is set, deselect all items
	if(ctrlflag)
		list->handle(o,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),p);
 
    // Popup menu pane
	FXMenuPane* menu = new FXMenuPane(this);
    FXint x,y;
    FXuint state;
    getRoot()->getCursorPosition(x,y,state);

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);

    // No selection or control flag set
	if(num == 0 || current->ctrlflag)
    {
		// Reset the control flag
		ctrlflag=FALSE;
		
		// Menu items
    	new FXMenuCommand(menu,_("New& file..."),newfileicon,current,FilePanel::ID_NEW_FILE);
    	new FXMenuCommand(menu,_("New f&older..."),newfoldericon,current,FilePanel::ID_NEW_DIR);
    	new FXMenuCommand(menu,_("New s&ymlink..."),newlinkicon,current,FilePanel::ID_NEW_SYMLINK);
        new FXMenuCommand(menu,_("Fi&lter..."),filtericon,current,FilePanel::ID_FILTER);
        new FXMenuSeparator(menu);
		new FXMenuCommand(menu,_("&Paste"),paste_clpicon,current,FilePanel::ID_PASTE_CLIPBOARD);
        new FXMenuSeparator(menu);
        new FXMenuCheck(menu,_("&Hidden files"),current->list,FileList::ID_TOGGLE_HIDDEN);
        new FXMenuCheck(menu,_("Thum&bnails"),current->list,FileList::ID_TOGGLE_THUMBNAILS);
        new FXMenuSeparator(menu);
        new FXMenuRadio(menu,_("B&ig icons"),current->list,IconList::ID_SHOW_BIG_ICONS);
        new FXMenuRadio(menu,_("&Small icons"),current->list,IconList::ID_SHOW_MINI_ICONS);
        new FXMenuRadio(menu,_("&Full file list"),current->list,IconList::ID_SHOW_DETAILS);
        new FXMenuSeparator(menu);
        new FXMenuRadio(menu,_("&Rows"),current->list,FileList::ID_ARRANGE_BY_ROWS);
        new FXMenuRadio(menu,_("&Columns"),current->list,FileList::ID_ARRANGE_BY_COLUMNS);
        new FXMenuCheck(menu,_("Autosize"),current->list,FileList::ID_AUTOSIZE);
        new FXMenuSeparator(menu);
        new FXMenuRadio(menu,_("&Name"),current->list,FileList::ID_SORT_BY_NAME);
        new FXMenuRadio(menu,_("Si&ze"),current->list,FileList::ID_SORT_BY_SIZE);
        new FXMenuRadio(menu,_("&Type"),current->list,FileList::ID_SORT_BY_TYPE);
        new FXMenuRadio(menu,_("E&xtension"),current->list,FileList::ID_SORT_BY_EXT);
        new FXMenuRadio(menu,_("&Date"),current->list,FileList::ID_SORT_BY_TIME);
        new FXMenuRadio(menu,_("&User"),current->list,FileList::ID_SORT_BY_USER);
        new FXMenuRadio(menu,_("&Group"),current->list,FileList::ID_SORT_BY_GROUP);
        new FXMenuRadio(menu,_("Per&missions"),current->list,FileList::ID_SORT_BY_PERM);
        new FXMenuRadio(menu,_("Deletion date"),current->list,FileList::ID_SORT_BY_DELTIME);
        new FXMenuSeparator(menu);
  		new FXMenuCheck(menu,_("Ignore c&ase"),current->list,FileList::ID_SORT_CASE);
   		new FXMenuCheck(menu,_("Dir&ectories first"),current->list,FileList::ID_DIRS_FIRST);
   		new FXMenuCheck(menu,_("Re&verse order"),current->list,FileList::ID_SORT_REVERSE);
   }
    // Non empty selection
	else
    {
		// Deselect the '..' item
		if (current->list->isItemSelected(0))
			current->list->deselectItem(0);

		// Panel submenu items
        FXMenuPane* submenu=new FXMenuPane(this);
    	new FXMenuCommand(submenu,_("Ne&w file..."),newfileicon,current,FilePanel::ID_NEW_FILE);
    	new FXMenuCommand(submenu,_("New f&older..."),newfoldericon,current,FilePanel::ID_NEW_DIR);
    	new FXMenuCommand(submenu,_("New s&ymlink..."),newlinkicon,current,FilePanel::ID_NEW_SYMLINK);
        new FXMenuCommand(submenu,_("Fi&lter..."),filtericon,current,FilePanel::ID_FILTER);
        new FXMenuSeparator(submenu);
		new FXMenuCommand(submenu,_("&Paste"),paste_clpicon,current,FilePanel::ID_PASTE_CLIPBOARD);
        new FXMenuSeparator(submenu);
        new FXMenuCheck(submenu,_("&Hidden files"),current->list,FileList::ID_TOGGLE_HIDDEN);
        new FXMenuCheck(submenu,_("Thum&bnails"),current->list,FileList::ID_TOGGLE_THUMBNAILS);
        new FXMenuSeparator(submenu);
        new FXMenuRadio(submenu,_("B&ig icons"),current->list,IconList::ID_SHOW_BIG_ICONS);
        new FXMenuRadio(submenu,_("&Small icons"),current->list,IconList::ID_SHOW_MINI_ICONS);
        new FXMenuRadio(submenu,_("&Full file list"),current->list,IconList::ID_SHOW_DETAILS);
        new FXMenuSeparator(submenu);
        new FXMenuRadio(submenu,_("&Rows"),current->list,FileList::ID_ARRANGE_BY_ROWS);
        new FXMenuRadio(submenu,_("&Columns"),current->list,FileList::ID_ARRANGE_BY_COLUMNS);
        new FXMenuCheck(submenu,_("Autosize"),current->list,FileList::ID_AUTOSIZE);
        new FXMenuSeparator(submenu);
        new FXMenuRadio(submenu,_("&Name"),current->list,FileList::ID_SORT_BY_NAME);
        new FXMenuRadio(submenu,_("Si&ze"),current->list,FileList::ID_SORT_BY_SIZE);
        new FXMenuRadio(submenu,_("&Type"),current->list,FileList::ID_SORT_BY_TYPE);
        new FXMenuRadio(submenu,_("E&xtension"),current->list,FileList::ID_SORT_BY_EXT);
        new FXMenuRadio(submenu,_("&Date"),current->list,FileList::ID_SORT_BY_TIME);
        new FXMenuRadio(submenu,_("&User"),current->list,FileList::ID_SORT_BY_USER);
        new FXMenuRadio(submenu,_("&Group"),current->list,FileList::ID_SORT_BY_GROUP);
        new FXMenuRadio(submenu,_("Per&missions"),current->list,FileList::ID_SORT_BY_PERM);
        new FXMenuRadio(submenu,_("Deletion date"),current->list,FileList::ID_SORT_BY_DELTIME);
        new FXMenuSeparator(submenu);
  		new FXMenuCheck(submenu,_("Ignore c&ase"),current->list,FileList::ID_SORT_CASE);
   		new FXMenuCheck(submenu,_("Dir&ectories first"),current->list,FileList::ID_DIRS_FIRST);
   		new FXMenuCheck(submenu,_("Re&verse order"),current->list,FileList::ID_SORT_REVERSE);
		new FXMenuCascade(menu,_("Pane&l"),NULL,submenu);
        new FXMenuSeparator(menu);
		
#if defined(linux)
		FXString name=current->list->getItemPathname(itm);
    	if(num==1 && (fsdevices->find(name.text()) || mtdevices->find(name.text()) ) )
		{
			new FXMenuCommand(menu,_("&Mount"),maphosticon,current,FilePanel::ID_MOUNT);
			new FXMenuCommand(menu,_("Unmount"),unmaphosticon,current,FilePanel::ID_UMOUNT);
    		new FXMenuSeparator(menu);
		}
#endif

        FXbool ar=FALSE;
        if(current->list->getItem (itm) && current->list->isItemFile(itm))
        {
			new FXMenuCommand(menu,_("Open &with..."),fileopenicon,current,FilePanel::ID_OPEN_WITH);
            new FXMenuCommand(menu,_("&Open"),fileopenicon,current,FilePanel::ID_OPEN);
			FXString name=current->list->getItemText(itm).section('\t',0);
			
			// Last and before last file extensions
			FXString ext1=name.rafter('.',1);
			FXString ext2=name.rafter('.',2);
			
			// Convert these extensions to lower case
			ext1.lower();			
			ext2.lower();

			// Destination folder name
			FXString extract_to_folder;
			if (ext2=="tar.gz" || ext2=="tar.bz2" || ext2=="tar.z")
				extract_to_folder=_("Extr&act to folder ")+name.section('\t',0).rbefore('.',2);
			else
				extract_to_folder=_("Extr&act to folder ")+name.section('\t',0).rbefore('.',1);

			// Display the extract and package menus according to the archive extensions
			if((num==1) && (ext2=="tar.gz" || ext2=="tar.bz2"))
			{
				ar=TRUE;
				new FXMenuCommand(menu,_("&Extract here"),archexticon,current,FilePanel::ID_EXTRACT_HERE);
				new FXMenuCommand(menu,extract_to_folder,archexticon,current,FilePanel::ID_EXTRACT_TO_FOLDER);
				new FXMenuCommand(menu,_("E&xtract to..."),archexticon,current,FilePanel::ID_EXTRACT);
			}
			else if ((num==1) && (ext1=="gz" || ext1=="bz2" || ext1=="z"))
			{
				ar=TRUE;
				new FXMenuCommand(menu,_("&Extract here"),archexticon,current,FilePanel::ID_EXTRACT_HERE);
			}
			else if((num==1) && (ext1=="tar" || ext1=="tgz" || ext1=="tbz2" || ext1=="zip" || ext1=="7z" || ext1=="lzh" || ext1=="rar" || ext1=="ace"))
			{
				ar=TRUE;
				new FXMenuCommand(menu,_("&Extract here"),archexticon,current,FilePanel::ID_EXTRACT_HERE);
				new FXMenuCommand(menu,extract_to_folder,archexticon,current,FilePanel::ID_EXTRACT_TO_FOLDER);
				new FXMenuCommand(menu,_("E&xtract to..."),archexticon,current,FilePanel::ID_EXTRACT);
			}
#if defined(linux)
			else if(num==1 && (ext1=="rpm" || ext1=="deb"))
			{
				ar=TRUE;
				new FXMenuCommand(menu,_("&View"),packageicon,current,FilePanel::ID_VIEW);
				new FXMenuCommand(menu,_("Install/Up&grade"),packageicon,current,ID_PKG_INSTALL);
				new FXMenuCommand(menu,_("Un&install"),packageicon,current,ID_PKG_UNINSTALL);
			}
#endif
			// Not archive nor package
			if(!ar)
			{
				new FXMenuCommand(menu,_("&View"),viewicon,current,FilePanel::ID_VIEW);
				new FXMenuCommand(menu,_("&Edit"),editicon,current,FilePanel::ID_EDIT);
			}
        }
        if(!ar)
            new FXMenuCommand(menu,_("&Add to archive..."),archaddicon,current,FilePanel::ID_ADD_TO_ARCH);
#if defined(linux)
        if (!ar)
			new FXMenuCommand(menu,_("Packages &query "),packageicon,current,FilePanel::ID_PKG_QUERY);
#endif
        new FXMenuSeparator(menu);
        new FXMenuCommand(menu,_("&Copy"),copy_clpicon,current,FilePanel::ID_COPY_CLIPBOARD);
        new FXMenuCommand(menu,_("C&ut"),cut_clpicon,current,FilePanel::ID_CUT_CLIPBOARD);
		new FXMenuCommand(menu,_("&Paste"),paste_clpicon,current,FilePanel::ID_PASTE_CLIPBOARD);
        new FXMenuSeparator(menu);
        new FXMenuCommand(menu,_("Re&name..."),renameiticon,current,FilePanel::ID_FILE_RENAME);
        new FXMenuCommand(menu,_("Copy &to..."),copy_clpicon,current,FilePanel::ID_FILE_COPYTO);
        new FXMenuCommand(menu,_("&Move to..."),moveiticon,current,FilePanel::ID_FILE_MOVETO);
        new FXMenuCommand(menu,_("Symlin&k to..."),minilinkicon,current,FilePanel::ID_FILE_SYMLINK);
        new FXMenuCommand(menu,_("Mo&ve to trash"),filedeleteicon,current,FilePanel::ID_FILE_TRASH);
        new FXMenuCommand(menu,_("&Delete"),filedelete_permicon,current,FilePanel::ID_FILE_DELETE);
        new FXMenuSeparator(menu);
        new FXMenuCommand(menu,_("P&roperties..."),attribicon,current,FilePanel::ID_PROPERTIES);
	}
    menu->create();
    allowPopupScroll=TRUE;  // Allow keyboard scrolling
	
	menu->popup(NULL,x,y);
	getApp()->runModalWhileShown(menu);
    allowPopupScroll=FALSE;
	
    return 1;
}


// Run Terminal in the selected directory
long  FilePanel::onCmdXTerm(FXObject*,FXSelector,void*)
{
	getApp()->beginWaitCursor();
    chdir(current->list->getDirectory().text());
    FXString cmd=getApp()->reg().readStringEntry("PROGS","xterm","xterm -sb");
    cmd += " &";
    system(cmd.text());
    current->list->setFocus();
	chdir(startlocation.text());
	getApp()->endWaitCursor();
    return 1;
}


// Add files or directory to an archive
long FilePanel::onCmdAddToArch(FXObject* o,FXSelector,void*)
{
    FXString name, ext1, ext2, cmd, archive="";
	File *f;
	
	chdir(current->list->getDirectory().text());

  	// Eventually deselect the '..' directory
	if (current->list->isItemSelected(0))
		current->list->deselectItem(0);

	// Return if nothing is selected
	if(current->list->getNumSelectedItems()==0)
		return 0;
	
	// If only one item is selected, use its name as a starting guess for the archive name
	if (current->list->getNumSelectedItems()==1)
	{
		for (int u= 0; u< current->list->getNumItems(); u++)
			if (current->list->isItemSelected(u))
			{
				name=list->getItemFilename(u);
				break;
			}
		archive=name;
	}
	
    // Initial archive name with full path and default extension
	FXString archpath=current->list->getDirectory();	
	archive = archpath+PATHSEPSTRING+archive+".tar.gz";

	// Archive dialog
	if (archdialog==NULL)
		archdialog=new ArchInputDialog(this,"");	
	archdialog->setText(archive);
	archdialog->CursorEnd();

    if(archdialog->execute())
    {
        if (archdialog->getText()=="")
		{
			MessageBox::warning(this,BOX_OK,_("Warning"),_("File name is empty, operation cancelled"));
			return 0;
		}
		
		// Get string and preserve escape characters
		archive=::quote(archdialog->getText());
		
        // Get extensions of the archive name
        ext1=archdialog->getText().rafter('.',1);
		ext1.lower();
        ext2=archdialog->getText().rafter('.',2);
		ext2.lower();
		
		// Handle different archive formats
		if (ext2=="tar.gz")
            cmd="tar -zcvf "+archive+" ";
		else if (ext2=="tar.bz2")
            cmd="tar -jcvf "+archive+" ";
		else if (ext2=="tar.z")
           	cmd="tar -Zcvf "+archive+" ";
		else if (ext1=="tar")
           	cmd="tar -cvf "+archive+" ";
		else if (ext1=="gz")
			cmd="gzip -v ";				
		else if (ext1=="tgz")
            cmd="tar -zcvf "+archive+" ";
		else if (ext1=="bz2")
			cmd="bzip2 -v ";				
		else if (ext1=="tbz2")
            cmd="tar -jcvf "+archive+" ";
		else if (ext1=="z")
            cmd="compress -v ";
		else if (ext1=="zip")
            cmd="zip -r "+archive+" ";
		else if (ext1=="7z")
            cmd="p7zip ";
		else if (ext1=="rar")
            cmd="rar a "+archive+" ";
		else if (ext1=="lzh")
            cmd="lha a "+archive+" ";
		
		// Default archive format
		else
		{
			archive += ".tar.gz";
			cmd="tar -zcvf "+archive+" ";
		}

        for (int u= 0; u< current->list->getNumItems(); u++)
		{
            if (current->list->isItemSelected(u))
            {
				// Don't include '..' in the list
				name=list->getItemFilename(u);				
				if (name!="..")
				{
                	cmd+=" ";
                	cmd=cmd+::quote(name);
                	cmd+=" ";
				}
            }
		}
		
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
		f=new File(this,_("Create archive"),ARCHIVE);
		f->create();

		// Create archive
		f->archive(archive,cmd);
		chdir(startlocation.text());
 
		getApp()->endWaitCursor();
		delete f; 

		// Force panel refresh
    	onCmdRefresh(0,0,0);
    }
    return 1;
}


// Extract archive
long FilePanel::onCmdExtract(FXObject*,FXSelector,void*)
{
    FXString name, ext1, ext2, cmd, dir, cdir;
	File *f;

    // Current directory
	cdir=current->list->getDirectory();
    
    // File selection dialog
    FileDialog browse(this,_("Select a destination folder"));
	const FXchar *patterns[]=
	{
		_("All Files"),          "*",	NULL
	};
	browse.setDirectory(homelocation);
	browse.setPatternList(patterns);
	browse.setSelectMode(SELECT_FILE_DIRECTORY);

	FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
    if (current->list->getItem(item))
	{
		// Archive name and extensions
		name=current->list->getItemText(item).text();

		ext1=name.section('\t',0).rafter('.',1);
		ext2=name.section('\t',0).rafter('.',2);
		ext1.lower();
		ext2.lower();
		name=::quote(cdir + PATHSEPSTRING + name.section('\t',0));

		// Handle different archive formats
		if (ext2=="tar.gz")
			cmd="tar -zxvf ";
		else if (ext2=="tar.bz2")
			cmd="tar -jxvf ";
		else if (ext2=="tar.z")
			cmd="tar -Zxvf ";
		else if (ext1=="tar")
			cmd="tar -xvf ";
		else if (ext1=="gz")
			cmd="gunzip -v ";				
		else if (ext1=="tgz")
			cmd="tar -zxvf ";
		else if (ext1=="bz2")
			cmd="bunzip2 -v ";				
		else if (ext1=="tbz2")
			cmd="tar -jxvf ";
		else if (ext1=="z")
			cmd="uncompress -v ";
		else if (ext1=="zip")
			cmd="unzip -o ";
		else if (ext1=="7z")
			cmd="p7zip -d ";
		else if (ext1=="rar")
			cmd="rar x -o+ ";
		else if (ext1=="lzh")
			cmd="lha -xf ";
		else if (ext1=="ace")
            cmd="unace x ";
		else
			cmd="tar -zxvf ";

		// Final extract command
		cmd+=name+" ";
				

        // Extract archive
		if(browse.execute())
        {
			dir=browse.getFilename();
			
            if(isWritable(dir))
            {                
				// Wait cursor
				getApp()->beginWaitCursor();
				
				// File object
   				f=new File(this,_("Extract archive"),EXTRACT);
   				f->create();
				   
				// Extract archive
				f->extract(name,dir,cmd);
 
				getApp()->endWaitCursor();
				delete f; 
            }
            else
                MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),dir.text());
        }
	}
	
	// Force panel refresh
	onCmdRefresh(0,0,0);
	
    return 1;
}


// Extract archive to a folder name based on the archive name
long FilePanel::onCmdExtractToFolder(FXObject*,FXSelector,void*)
{
    FXString name, pathname, ext1, ext2, cmd, dirname, dirpath, cdir;
	File *f;

    // Current directory
	cdir=current->list->getDirectory();

	FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
    if (current->list->getItem(item))
	{
		// Archive name and extensions
		name=current->list->getItemText(item).text();
		ext1=name.section('\t',0).rafter('.',1);
		ext2=name.section('\t',0).rafter('.',2);
		ext1.lower();
		ext2.lower();

		// Destination folder name
		if (ext2=="tar.gz" || ext2=="tar.bz2" || ext2=="tar.z")
			dirname=name.section('\t',0).rbefore('.',2);
		else
			dirname=name.section('\t',0).rbefore('.',1);
		
		// Create the new dir according to the current umask
		// Don't complain if directory already exists 
		int mask=umask(0);
		umask(mask);
		dirpath=cdir + PATHSEPSTRING + dirname;
		FXint ret=::mkdir(dirpath.text(),511 & ~mask);
		if (ret==-1 && errno!=EEXIST)
		{
			if (errno)
				MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't create folder %s: %s",dirpath.text(),strerror(errno));
			else
				MessageBox::error(this,BOX_OK_SU,_("Error"),"Can't create folder %s",dirpath.text());
			return 0;
		}
			
		// Archive pathname
		pathname=::quote(cdir + PATHSEPSTRING + name.section('\t',0));
		
		// Handle different archive formats
		if (ext2=="tar.gz")
			cmd="tar -zxvf ";
		else if (ext2=="tar.bz2")
			cmd="tar -jxvf ";
		else if (ext2=="tar.z")
			cmd="tar -Zxvf ";
		else if (ext1=="tar")
			cmd="tar -xvf ";
		else if (ext1=="gz")
			cmd="gunzip -v ";				
		else if (ext1=="tgz")
			cmd="tar -zxvf ";
		else if (ext1=="bz2")
			cmd="bunzip2 -v ";				
		else if (ext1=="tbz2")
			cmd="tar -jxvf ";
		else if (ext1=="z")
			cmd="uncompress -v ";
		else if (ext1=="zip")
			cmd="unzip -o ";
		else if (ext1=="7z")
			cmd="p7zip -d ";
		else if (ext1=="rar")
			cmd="rar x -o+ ";
		else if (ext1=="lzh")
			cmd="lha -xf ";
		else if (ext1=="ace")
            cmd="unace x ";
		else
			cmd="tar -zxvf ";

		// Final extract command
		cmd+=pathname+" ";
									
		// Wait cursor
		getApp()->beginWaitCursor();
			
		// File object
		f=new File(this,_("Extract archive"),EXTRACT);
		f->create();
		   
		// Extract archive
		f->extract(pathname,dirpath,cmd);

		getApp()->endWaitCursor();
		delete f; 
	}
	
	// Force panel refresh
	onCmdRefresh(0,0,0);
	
    return 1;
}



// Extract archive in the current directory
long FilePanel::onCmdExtractHere(FXObject*,FXSelector,void*)
{
    FXString name, ext1, ext2, cmd, cdir;
	File *f;

    // Current directory
	cdir=current->list->getDirectory();

    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
    if (current->list->getItem (item))
	{
		if(isWritable(cdir))
		{
			// Archive name and extensions
			name=current->list->getItemText(item).text();
        	ext1=name.section('\t',0).rafter('.',1);
         	ext2=name.section('\t',0).rafter('.',2);
			ext1.lower();
			ext2.lower();
			name=::quote(cdir + PATHSEPSTRING + name.section('\t',0));

			// Handle different archive formats
			if (ext2=="tar.gz")
            	cmd="tar -zxvf ";
			else if (ext2=="tar.bz2")
            	cmd="tar -jxvf ";
			else if (ext2=="tar.z")
           		cmd="tar -Zxvf ";
			else if (ext1=="tar")
           		cmd="tar -xvf ";
			else if (ext1=="gz")
				cmd="gunzip -v ";				
			else if (ext1=="tgz")
            	cmd="tar -zxvf ";
			else if (ext1=="bz2")
				cmd="bunzip2 -v ";				
			else if (ext1=="tbz2")
            	cmd="tar -jxvf ";
			else if (ext1=="z")
            	cmd="uncompress -v ";
			else if (ext1=="zip")
            	cmd="unzip -o ";
			else if (ext1=="7z")
            	cmd="p7zip -d ";
			else if (ext1=="rar")
            	cmd="rar x -o+ ";
			else if (ext1=="lzh")
            	cmd="lha -xf ";
			else if (ext1=="ace")
				cmd="unace x ";
			else
				cmd="tar -zxvf ";

        	// Final extract command
			cmd+=name+" ";

			// Wait cursor
			getApp()->beginWaitCursor();
					
			// File object
   			f=new File(this,_("Extract archive"),EXTRACT);
   			f->create();
				   
			// Extract archive
			f->extract(name,cdir,cmd);
 
			getApp()->endWaitCursor();
			delete f; 
		}
		else
			MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),cdir.text());
	}
	
	// Force panel refresh
	onCmdRefresh(0,0,0);
	
    return 1;
}

#if defined(linux)
// Install/Upgrade package
long FilePanel::onCmdPkgInstall(FXObject*,FXSelector,void*)
{
    FXString name, path, cmd, dir, cdir;
	File *f;

    cdir=current->list->getDirectory();

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);
    if (current->list->getItem (itm))
	{
        name=current->list->getItemText(itm).text();
		name=name.section('\t',0);
		path=::quote(cdir + PATHSEPSTRING + name);

		// Command to perform
		FXString ext=FXPath::extension(name);
		if(comparecase(ext,"rpm")==0)
			cmd="rpm -Uvh " + path;
		else if(comparecase(ext,"deb")==0)
			cmd="dpkg -i "+ path;
                   
		// Wait cursor
		getApp()->beginWaitCursor();
		
		// File object
		f=new File(this,_("Package Install/Upgrade"),PKG_INSTALL);
		f->create();
				   
		// Install/Upgrade package
		f->pkgInstall(name,cmd);

		getApp()->endWaitCursor();
		delete f; 
      }
    
	// Force panel refresh
	onCmdRefresh(0,0,0);
	
	return 1;
}

// Uninstall package based on its name (package version is ignored)
long FilePanel::onCmdPkgUninstall(FXObject*,FXSelector,void*)
{
    FXString name, cmd, dir, cdir;
	File *f;

    cdir=current->list->getDirectory();

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);
    if (current->list->getItem (itm))
	{
        name=current->list->getItemText(itm).text();
		name=name.section('\t',0);

		// Command to perform
		FXString ext=FXPath::extension(name);
		if(comparecase(ext,"rpm")==0)
		{
			name=name.section('-',0);
			cmd="rpm -e " + name;
		}
		else if(comparecase(ext,"deb")==0)
		{
			name=name.section('_',0);
			cmd="dpkg -r "+ name;
		}
		
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
		f=new File(this,_("Package Uninstall"),PKG_UNINSTALL);
		f->create();

		// Uninstall package
		f->pkgUninstall(name,cmd);

		getApp()->endWaitCursor();
		delete f; 
      }
    
	// Force panel refresh
	onCmdRefresh(0,0,0);
	
	return 1;
}

#endif


// Force FilePanel and DirPanel refresh
long FilePanel::onCmdRefresh(FXObject*,FXSelector,void*)
{
    current->list->setFocus();
    FXString dir=current->list->getDirectory();
    current->list->setDirectory(ROOTDIR);
    current->list->setDirectory(dir);
    current->list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_REFRESH),NULL);
    current->updatePathLinker();
    return 1;
}


// Handle item selection
long FilePanel::onCmdSelect(FXObject* sender,FXSelector sel,void* ptr)
{
    switch(FXSELID(sel))
    {
    case ID_SELECT_ALL:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SELECT_ALL),ptr);
        return 1;
    case ID_DESELECT_ALL:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),ptr);
        return 1;
    case ID_SELECT_INVERSE:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SELECT_INVERSE),ptr);
        return 1;
    }
    return 1;
}



// Handle show commands
long FilePanel::onCmdShow(FXObject* sender,FXSelector sel,void* ptr)
{
    switch(FXSELID(sel))
    {
    case ID_SHOW_BIG_ICONS:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SHOW_BIG_ICONS),ptr);
        break;
    case ID_SHOW_MINI_ICONS:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SHOW_MINI_ICONS),ptr);
        break;
    case ID_SHOW_DETAILS:
        current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_SHOW_DETAILS),ptr);
        break;
    }
    return 1;
}

// Update show commands
long FilePanel::onUpdShow(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg=FXWindow::ID_UNCHECK;
    FXuint style=current->list->getListStyle();
		
    switch(FXSELID(sel))
    {
    case ID_SHOW_BIG_ICONS:
        if(style & _ICONLIST_BIG_ICONS)
            msg=FXWindow::ID_CHECK;
        break;
    case ID_SHOW_MINI_ICONS:
        if(style & _ICONLIST_MINI_ICONS)
            msg=FXWindow::ID_CHECK;
        break;
    case ID_SHOW_DETAILS:
       if(!(style & (_ICONLIST_MINI_ICONS | _ICONLIST_BIG_ICONS)))
            msg=FXWindow::ID_CHECK;
        break;
    }
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}

// Handle toggle hidden command
long FilePanel::onCmdToggleHidden(FXObject* sender,FXSelector sel,void* ptr)
{
	current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_TOGGLE_HIDDEN),ptr);
    return 1;
}

// Update toggle hidden command
long FilePanel::onUpdToggleHidden(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg=FXWindow::ID_UNCHECK;
	FXbool hidden=current->list->shownHiddenFiles();
		
	if(hidden == FALSE)
		msg=FXWindow::ID_CHECK;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}

// Handle toggle thumbnails command
long FilePanel::onCmdToggleThumbnails(FXObject* sender,FXSelector sel,void* ptr)
{
	current->list->handle(sender,FXSEL(SEL_COMMAND,FileList::ID_TOGGLE_THUMBNAILS),ptr);
    return 1;
}

// Update toggle hidden command
long FilePanel::onUpdToggleThumbnails(FXObject* sender,FXSelector sel,void* ptr)
{	
    FXuint msg=FXWindow::ID_UNCHECK;
	FXbool showthumb=current->list->shownThumbnails();
		
	if(showthumb == FALSE)
		msg=FXWindow::ID_CHECK;
    sender->handle(this,FXSEL(SEL_COMMAND,msg),ptr);
    return 1;
}


#if defined(linux)
// Mount/Unmount file systems
long FilePanel::onCmdMount(FXObject*,FXSelector sel,void*)
{
    FXString cmd, msg, text;
	unsigned int op;
	File *f;
	FXString dir;
	
	current->list->setFocus();

	// Use the selected directory in FilePanel if any
	// or use the selected directory in DirPanel
	if (current->list->getNumSelectedItems()==0)
		dir=current->list->getDirectory();
	else
	{
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected(u))
				dir=current->list->getItemPathname(u);
		}
	}		

	// If symbolic link, read the linked directory
	if (::isLink(dir))
		dir=readlink(dir);

    if(FXSELID(sel)==ID_MOUNT)
    {
        op=MOUNT;
		msg=_("Mount");
        cmd="mount ";
    }
    else
    {
        op=UNMOUNT;
		msg=_("Unmount");
        cmd="umount ";
    }
    cmd+=::quote(dir);
    cmd+=" 2>&1";
    chdir(ROOTDIR);

	// Wait cursor
	getApp()->beginWaitCursor();

	// File object
	text=msg + _(" file system...");
	f=new File(this,text.text(),op);
	f->create();
				   
	// Mount/unmount file system
	text=msg + _(" the folder:");
	f->mount(dir,text,cmd,op);
	chdir(startlocation.text());
 
	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		f->hide();
		text=msg + _(" operation cancelled!");
		MessageBox::error(this,BOX_OK,_("Error"),text.text());
		delete f;
		return 0;
	}
					
	getApp()->endWaitCursor();
	delete f; 

	// Force panel refresh
    onCmdRefresh(0,0,0);

    return 1;
}


// Update the Mount button
long FilePanel::onUpdMount(FXObject* o,FXSelector sel,void*)
{
    FXString dir;

   	FXint num=current->list->getNumSelectedItems();
	
	// Use the selected directory in FilePanel if any
	// or use the selected directory in DirPanel
	if (num==0)
		dir=current->list->getDirectory();
	else
	{
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected(u))
				dir=current->list->getItemPathname(u);
		}
	}		

	if(fsdevices->find(dir.text()) && !mtdevices->find(dir.text()) && current->list->getNumItems() && !current->list->isItemSelected(0))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Update the Unmount button
long FilePanel::onUpdUnmount(FXObject* o,FXSelector sel,void*)
{
    FXString dir;
	
   	FXint num=current->list->getNumSelectedItems();
	
	// Use the selected directory in FilePanel if any
	// or use the selected directory in DirPanel
	if (num==0)
		dir=current->list->getDirectory();
	else
	{
		for (int u= 0; u< current->list->getNumItems (); u++)
		{
			if (current->list->isItemSelected (u))
				dir=current->list->getItemPathname(u);
		}
	}		

	if((fsdevices->find(dir.text()) || mtdevices->find(dir.text())) && current->list->getNumItems() && !current->list->isItemSelected(0))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
       o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Query packages data base
long FilePanel::onCmdPkgQuery(FXObject* o,FXSelector sel,void*)
{
	FXString cmd;
	
	// Name of the current selected file
	FXString file=current->list->getCurrentFile();

    // Command to perform
	if (rpm_based)
		cmd="rpm -qf " + ::quote(file);
	else if(deb_based)
		cmd="dpkg -S " + ::quote(file);
	else
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("No compatible package manager (rpm or dpkg) found!"));
		return 0;
	}

	// Query command
   	cmd+=" 2>&1";

	// Wait cursor
	getApp()->beginWaitCursor();

	// Perform the command
	FILE *pcmd=popen(cmd.text(),"r");
	if(!pcmd)
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Failed command: %s"),cmd.text());
		return 0;
	}

	// Get command output
	char text[10000]={0};
	FXString buf;
	while(fgets(text,sizeof(text),pcmd))
		buf+=text;
	snprintf(text,sizeof(text)-1,"%s",buf.text());

	// Close the stream and display error message if any
	if(pclose(pcmd))
	{
		getApp()->endWaitCursor();
		MessageBox::error(this,BOX_OK,_("Error"),"%s",text);
		return 0;
	}
	getApp()->endWaitCursor();
	
	// Get package name (depends on the package manager)
	FXString package=text;
	if (deb_based)
		package=package.section(':',0);

	// Output message
	FXString message=_("File ") + file + _(" belongs to the package: ") + package;
	MessageBox::information(this,BOX_OK,_("Information"),"%s",message.text());

   	return 1;
}


// Update the package query menu 
long FilePanel::onUpdPkgQuery(FXObject* o, FXSelector sel, void*)
{
   	// Menu item is disabled when nothing is selected or multiple selection
	// or when unique selection and the selected item is a directory

	FXint num;
	num=current->list->getNumSelectedItems();
	
	if (num==0 || num>1)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else // num=1
	{
		int item=current->list->getCurrentItem();
		if (current->list->isItemDirectory(item))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}
			
	return 1;
}
#endif // End #if defined(linux)


// Update the status bar and the path linker
long FilePanel::onUpdStatus(FXObject* sender,FXSelector,void*)
{
	// Update the status bar
    int item=-1;
	FXString str, linkto;
    FXchar size[64];
	unsigned long long sz=0;
	FXString hsize=_("0 bytes");
	
    FXString path=list->getDirectory();

   	FXint num=list->getNumSelectedItems();

	// To handle the update rename (ugly, I know)
	if (current==this)
	{
		if (num<=1)
			selmult=FALSE;
		else if (num>1)
			selmult=TRUE;
	}

    item=list->getCurrentItem();
		
    if(num>1)
    {
        for (int u= 0; u< list->getNumItems(); u++)
		{
            if (list->isItemSelected(u) && !list->isItemDirectory(u))
            {
				sz+=list->getItemFileSize(u);
				snprintf(size,sizeof(size)-1,"%llu",sz);
				hsize=::hSize(size);
            }
		}
        str=hsize+_(" in ")+FXStringVal(num)+_(" selected items");
    }
	else
	{
		if(num==0 || item<0)
		{
			num=list->getNumItems();
			str=FXStringVal(num)+_(" Items");
		}
		else // num=1
		{
			FXString string=list->getItemText(item);
			FXString name=string.section('\t',0);
			FXString type=string.section('\t',2);

			FXString date=string.section('\t',4);
			FXString usr=string.section('\t',5);
			FXString grp=string.section('\t',6);
			FXString perm=string.section('\t',7);

			if (type.contains(_("Broken link")))
			{
				linkto=readlink(path+PATHSEPSTRING+name);
				str=hsize+_(" in one selected item")+ " [" + name+"->" + linkto.text() + " | " + type + " | " + date 
				    + " | " + usr + " | " + grp + " | " + perm + "]";
			}
			else if (type.contains(_("Link")))
			{
				linkto=readlink(path+PATHSEPSTRING+name);
				str=hsize+_(" in one selected item")+ " [" + name+"->" + linkto.text() + " | " + type + " | " + date
                    + " | " + usr + " | " + grp + " | " + perm + "]";
			}
			else
			{
				for (int u= 0; u< list->getNumItems(); u++)
				{
					if (list->isItemSelected(u) && !list->isItemDirectory(u))
					{
						sz=list->getItemFileSize(u);
						snprintf(size,sizeof(size)-1,"%llu",sz);
						hsize=::hSize(size);
						break;
					}
				}
				str=hsize+_(" in one selected item")+ " [" + type + " | " + date
					+ " | " + usr + " | " + grp + " | " + perm + "]";
			}
		}
	}

	// Add the filter pattern if any
	if (list->getPattern() != "*" && list->getPattern() != "*.*")
	{
		FXString filter=_(" - Filter: ") + list->getPattern();
		status->setText(str+filter);
	}
	else
		status->setText(str);
	
    return 1;
}


// Update the path link
void FilePanel::updatePathLinker()
{    
	pathlink->setPath(list->getDirectory());
}


// Update the go to parent directory command
long FilePanel::onUpdUp(FXObject* o,FXSelector,void*)
{
    FXButton *button=(FXButton*)o;
    int style=button->getButtonStyle();
    if(style & TOGGLEBUTTON_TOOLBAR)
    {
        if(current->list->getDirectory()!=ROOTDIR)
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
        else
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
    }
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    return 1;
}


// Update the paste button
long FilePanel::onUpdPaste(FXObject* o,FXSelector,void*)
{
    FXuchar *data;
    FXuint len;	
	FXString buf;
	FXbool clipboard_empty=TRUE;
	
	// Lock clipboard to prevent changes in method onCmdRequestClipboard()
	clipboard_locked=TRUE;
	
	// If source is xfelistType (Gnome, XFCE, or Xfe app)
    if (getDNDData(FROM_CLIPBOARD,xfelistType,data,len))
    {
		
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		buf=(FXchar*)data;
		
		// Check if valid clipboard
		if (buf.find("file:/")>=0)
			clipboard_empty=FALSE;
			
        // Free data pointer
		FXFREE(&data);
	}
		
	// If source type is urilistType (KDE apps ; non Gnome, non XFCE and non Xfe apps)
	else if (getDNDData(FROM_CLIPBOARD,urilistType,data,len))
	{		
		FXRESIZE(&data,FXuchar,len+1);
		data[len]='\0';
		buf=(FXchar*)data;
		
		// Check if valid clipboard
		if (buf.find("file:/")>=0)
			clipboard_empty=FALSE;
		
		// Free data pointer
		FXFREE(&data);
	}
		
	// Gray out the paste button, if necessary
	if (clipboard_empty)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);

	// Unlock clipboard
	clipboard_locked=FALSE;

    return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long FilePanel::onUpdMenu(FXObject* o,FXSelector sel,void*)
{
		
   	// Menu item is disabled when nothing or only ".." is selected
	FXint num;
	num=current->list->getNumSelectedItems();
	DirPanel* dirpanel=((XFileExplorer*) mainWindow)->getDirPanel();
	DirItem* item=(DirItem*)dirpanel->getCurrentItem();

	if ((dirpanel->shown() && item))
	{
		if (num==0)
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
		else if (num==1 && current->list->isItemSelected(0))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}
	else
	{
		if (num==0)
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else if (num==1 && current->list->isItemSelected(0))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}

    return 1;
}


// Update file delete menu item and toolbar button
long FilePanel::onUpdFileDelete(FXObject* o,FXSelector sel,void*)
{
	FXbool use_trash_can=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_can",TRUE);
	FXbool use_trash_bypass=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_bypass",FALSE);
	if (!use_trash_can | use_trash_bypass)
	{
		FXint num=current->list->getNumSelectedItems();
		if (num==0)
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else if (num==1 && current->list->isItemSelected(0))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Update file delete menu item and toolbar button
long FilePanel::onUpdFileTrash(FXObject* o,FXSelector sel,void*)
{
	FXbool use_trash_can=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_can",TRUE);
	if (use_trash_can)
	{
		FXint num=current->list->getNumSelectedItems();
		if (num==0)
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else if (num==1 && current->list->isItemSelected(0))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}

// Update go trash menu item and toolbar button
long FilePanel::onUpdGoTrash(FXObject* o,FXSelector sel,void*)
{

	FXbool use_trash_can=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_can",TRUE);
	if (use_trash_can)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}

// Update file open menu
long FilePanel::onUpdOpen(FXObject* o,FXSelector,void*)
{
   	// Menu item is disabled when nothing or a directory (including "..") is selected
    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
	if (num==0)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
	{
		if (current->list->getItem (item) && current->list->isItemFile(item))
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
		else
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	}
    return 1;
}


// Update the status of the menu items that should be disabled when selecting multiple files
long FilePanel::onUpdSelMult(FXObject* o, FXSelector sel, void*)
{
   	// Menu item is disabled when nothing is selected or multiple selection or ".." is only selected
	FXint num;
	num=current->list->getNumSelectedItems();
	DirPanel* dirpanel=((XFileExplorer*) mainWindow)->getDirPanel();
	DirItem* item=(DirItem*)dirpanel->getCurrentItem();

	if (num==0)
	{
		if (!item || !dirpanel->shown())
			o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
		else
    		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	}	
	else if ((current->selmult || num==1) && current->list->isItemSelected(0))
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
    	o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);

	return 1;
}


// Update Add to archive menu
long FilePanel::onUpdAddToArch(FXObject* o,FXSelector,void*)
{
   	// Menu item is disabled when nothing or ".." is selected
    FXint num, item;	
	num=current->list->getNumSelectedItems(&item);
	if (num==0)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);	
	else if (num==1 && current->list->isItemSelected(0))
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    return 1;
}


