// Properties box

#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>
#if defined(linux)
#include <mntent.h>
#endif

#include <fx.h>
#include <fxkeys.h>
#include <FXPNGIcon.h>

#include "xfedefs.h"
#include "icons.h"
#include "DialogBox.h"
#include "FileDialog.h"
#include "OverwriteBox.h"
#include "File.h"
#include "Properties.h"
#include "FilePanel.h"
#include "XFileExplorer.h"
#include "MessageBox.h"


// Global variables
extern FXMainWindow *mainWindow;
extern FXStringDict *fsdevices;


// Map
FXDEFMAP(PermFrame) PermFrameMap[]={};

// Object implementation
FXIMPLEMENT(PermFrame,FXVerticalFrame,PermFrameMap,ARRAYNUMBER(PermFrameMap))

PermFrame::PermFrame(FXComposite* parent,FXObject *target):
        FXVerticalFrame(parent,FRAME_THICK|FRAME_RAISED)
{
    FXHorizontalFrame *accessframe=new FXHorizontalFrame(this,LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXHorizontalFrame *chmodframe=new FXHorizontalFrame(this,LAYOUT_FILL_X|LAYOUT_FILL_Y);
    
	// Permissions
	FXGroupBox *group1=new FXGroupBox(accessframe,_("User"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    ur=new FXCheckButton(group1,_("Read"),target,PropertiesBox::ID_RUSR);
    uw=new FXCheckButton(group1,_("Write"),target,PropertiesBox::ID_WUSR);
    ux=new FXCheckButton(group1,_("Execute"),target,PropertiesBox::ID_XUSR);
	FXGroupBox *group2=new FXGroupBox(accessframe,_("Group"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    gr=new FXCheckButton(group2,_("Read"),target,PropertiesBox::ID_RGRP);
    gw=new FXCheckButton(group2,_("Write"),target,PropertiesBox::ID_WGRP);
    gx=new FXCheckButton(group2,_("Execute"),target,PropertiesBox::ID_XGRP);
    FXGroupBox *group3=new FXGroupBox(accessframe,_("Others"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
    or_=new FXCheckButton(group3,_("Read"),target,PropertiesBox::ID_ROTH);
    ow=new FXCheckButton(group3,_("Write"),target,PropertiesBox::ID_WOTH);
    ox=new FXCheckButton(group3,_("Execute"),target,PropertiesBox::ID_XOTH);

    // Owner
	FXGroupBox *group4=new FXGroupBox(chmodframe,_("Owner"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    new FXLabel(group4,_("User"));
    user=new FXComboBox(group4,15,NULL,0,COMBOBOX_NO_REPLACE);
  	user->setNumVisible(5);
    new FXLabel(group4,_("Group"));
    grp=new FXComboBox(group4,15,NULL,0,COMBOBOX_NO_REPLACE);
  	grp->setNumVisible(5);

	// User names (sorted in ascending order)
    struct passwd* pwde;
    while( (pwde = getpwent()) )
        user->appendItem(pwde->pw_name);
    endpwent();
	user->setSortFunc(FXList::ascending);
	user->sortItems();	

	// Group names (sorted in ascending order)
    struct group* grpe;
    while( (grpe = getgrent()) )
        grp->appendItem(grpe->gr_name);
    endgrent();
	grp->setSortFunc(FXList::ascending);
	grp->sortItems();	

    // Command
	FXGroupBox *group5=new FXGroupBox(chmodframe,_("Command"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXMatrix *matrix5 = new FXMatrix(group5,2,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	cmd_radiotarget.connect(cmd);
	set=new FXRadioButton(matrix5,_("Set marked"),&cmd_radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_SET);
    rec=new FXCheckButton(matrix5,_("Recursively"),NULL,0);
    clear=new FXRadioButton(matrix5,_("Clear marked"),&cmd_radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_CLEAR);
	flt_radiotarget.connect(flt);
    all=new FXRadioButton(matrix5,_("Files and folders"),&flt_radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_ALL);
    add=new FXRadioButton(matrix5,_("Add marked"),&cmd_radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_ADD);
	dironly=new FXRadioButton(matrix5,_("Folders only"),&flt_radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_DIRONLY);
	own=new FXCheckButton(matrix5,_("Owner only"),NULL,0);
    fileonly=new FXRadioButton(matrix5,_("Files only"),&flt_radiotarget,FXDataTarget::ID_OPTION+PropertiesBox::ID_FILEONLY);
}

// Map
FXDEFMAP(PropertiesBox) PropertiesBoxMap[]=
{
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ACCEPT_SINGLE,PropertiesBox::onCmdAcceptSingle),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ACCEPT_MULT,PropertiesBox::onCmdAcceptMult),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_CANCEL,PropertiesBox::onCmdCancel),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_RUSR,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_WUSR,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_XUSR,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_RGRP,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_WGRP,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_XGRP,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ROTH,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_WOTH,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_XOTH,PropertiesBox::onCmdCheck),
	FXMAPFUNC(SEL_UPDATE,0,PropertiesBox::onUpdPerm),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_SET,PropertiesBox::onCmdCommand),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_CLEAR,PropertiesBox::onCmdCommand),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ADD,PropertiesBox::onCmdCommand),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_DIRONLY,PropertiesBox::onCmdFilter),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_FILEONLY,PropertiesBox::onCmdFilter),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_ALL,PropertiesBox::onCmdFilter),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BIG_ICON,PropertiesBox::onCmdBrowseIcon),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_MINI_ICON,PropertiesBox::onCmdBrowseIcon),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BROWSE_OPEN,PropertiesBox::onCmdBrowse),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BROWSE_VIEW,PropertiesBox::onCmdBrowse),
	FXMAPFUNC(SEL_COMMAND,PropertiesBox::ID_BROWSE_EDIT,PropertiesBox::onCmdBrowse),
	FXMAPFUNC(SEL_KEYPRESS,0,PropertiesBox::onCmdKeyPress),
};

// Object implementation
FXIMPLEMENT(PropertiesBox,DialogBox,PropertiesBoxMap,ARRAYNUMBER(PropertiesBoxMap))

// Construct window for one file
PropertiesBox::PropertiesBox(FXWindow *win,FXString file,FXString path): DialogBox(win,_("Properties"), DECOR_TITLE|DECOR_BORDER|DECOR_STRETCHABLE)
{
    unsigned long long filesize, dirsize=0;
    FXString mod,changed,accessed;
	FXString grpid, usrid;	
	FXLabel *sizelabel=NULL;
    struct stat info, linfo;
    FXString type, extension, fileassoc;
    FXbool isLink, isBrokenLink, isDirectory;
    FXString filename;
    char mnttype[64], used[64], avail[64], pctr[64], size[64];
    char buf[MAXPATHLEN+1];
	FXString hsize;
	FXbool ismountpoint=FALSE;
	FILE *p;

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Contents
    FXVerticalFrame *contents=new FXVerticalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Accept
	if (file!="..")
    {
		FXButton *ok = new FXButton(buttons,_("&Accept"),NULL,this,PropertiesBox::ID_ACCEPT_SINGLE,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);
    	ok->addHotKey(KEY_Return);
	}

    // Cancel
    new FXButton(buttons,_("&Cancel"),NULL,this,PropertiesBox::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);

    // First item is General
    new FXTabItem(tabbook,_("&General"),NULL);
    FXPacker *genpack = new FXPacker(tabbook,FRAME_THICK|FRAME_RAISED);
    FXGroupBox *generalgroup=new FXGroupBox(genpack,FXString::null,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXVerticalFrame *generalframe=new FXVerticalFrame(generalgroup,LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Second item is Access Permissions
	FXTabItem *permtab = new FXTabItem(tabbook,_("&Permissions"),NULL);
    perm = new PermFrame(tabbook,this);
	
	// Permission tab is disabled for parent directory
	if (file=="..")
		permtab->disable();

    // Third tab - file associations
    FXTabItem *fassoctab = new FXTabItem(tabbook,_("&File Associations"),NULL);
    FXPacker *fassocpack = new FXPacker(tabbook,FRAME_THICK|FRAME_RAISED);
    FXGroupBox *fassocgroup=new FXGroupBox(fassocpack,FXString::null,FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXVerticalFrame *contassoc=new FXVerticalFrame(fassocgroup,LAYOUT_FILL_X|LAYOUT_FILL_Y);
    FXMatrix *matrix = new FXMatrix(contassoc,3,MATRIX_BY_COLUMNS|LAYOUT_SIDE_TOP|LAYOUT_FILL_X|LAYOUT_FILL_Y);
    fassoctab->disable();

    new FXLabel(matrix,_("Extension:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    ext = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
    new FXLabel(matrix,"",NULL,0);

    new FXLabel(matrix,_("Description:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    descr = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
    new FXLabel(matrix,"",NULL,0);

    new FXLabel(matrix,_("Open:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    open = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
    new FXButton(matrix,_("\tSelect file..."),filedialogicon,this,ID_BROWSE_OPEN,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    int is_ar = FALSE;

    FXString viewlbl=_("View:");
    FXString editlbl=_("Edit:");

	extension=file.rafter('.',1);
	extension=extension.lower();
	if (extension=="gz" || extension=="tgz" || extension=="tar" || extension=="bz2" ||
		extension=="tbz2" || extension=="zip" || extension=="Z" || extension=="lzh" || extension=="rar")
	{
		is_ar = TRUE; // archive
		viewlbl = _("Extract:");
	}
#if defined(linux)
	else if(extension=="rpm")
		editlbl = _("Install/Upgrade:");
#endif	
    new FXLabel(matrix,viewlbl,NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    view = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
    new FXButton(matrix,_("\tSelect file..."),filedialogicon,this,ID_BROWSE_VIEW,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    if(!is_ar)
    {
        new FXLabel(matrix,editlbl,NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
        edit = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
        new FXButton(matrix,_("\tSelect file..."),filedialogicon,this,ID_BROWSE_EDIT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);
    }
    else
        edit = NULL;

    new FXLabel(matrix,_("Big Icon:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    bigic = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
    new FXButton(matrix,_("\tSelect file..."),filedialogicon,this,ID_BIG_ICON,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    new FXLabel(matrix,_("Mini Icon:"),NULL,JUSTIFY_LEFT|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW);
    miniic = new FXTextField(matrix,30,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_FILL_COLUMN|LAYOUT_FILL_ROW|LAYOUT_FILL_X);
    new FXButton(matrix,_("\tSelect file..."),filedialogicon,this,ID_MINI_ICON,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Label
    new FXLabel(generalframe,_("Name"),NULL,JUSTIFY_LEFT);
    input = new FXTextField(generalframe,20,NULL,0,FRAME_THICK|FRAME_SUNKEN|LAYOUT_SIDE_TOP|LAYOUT_FILL_X);

    // Complete filename (with path)
	filename=path+PATHSEPSTRING+file;
	parentdir=path;
	
	// Warn if non UTF-8 file name
	if (!isUtf8(filename.text(),filename.length()))
		new FXLabel(generalframe,_("=> Warning: file name is not UTF-8 encoded!"),NULL,LAYOUT_LEFT|LAYOUT_CENTER_Y|LAYOUT_FILL_ROW);

	// Get file/link stat info
    if(lstatrep(filename.text(),&linfo)!=0)
        return ;

	// Obtain user name
	usrid=FXSystem::userName(linfo.st_uid);

	// Obtain group name
	grpid=FXSystem::groupName(linfo.st_gid);
    perm->user->setText(usrid);
    perm->grp->setText(grpid);
    oldgrp = grpid;
    oldusr = usrid;

	// Test if link or broken link
	// When valid link, set the file name to the refered file
    isLink=S_ISLNK(linfo.st_mode);
	isBrokenLink=FALSE;
	if(isLink)
	{
		// Broken link
		if (statrep(filename.text(),&info)!=0)
			isBrokenLink=TRUE;

		// Get the name of the linked file
		filename=readlink(filename);
	}

	// Obtain stat info on the file itself
	if (statrep(filename.text(),&info)!=0)
	{
		// Or on the link if broken link
		if (isBrokenLink)
			lstatrep(filename.text(),&info);
	}

    orig_mode=(info.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO));

    // Mod time of the file / link
	mod=FXSystem::time("%c",linfo.st_mtime);

    // Change time of the file / link
	changed=FXSystem::time("%c",linfo.st_ctime);

    // Accessed time of the file / link
    accessed=FXSystem::time("%c",linfo.st_atime);;

    // Size of the file / link
    filesize=(unsigned long long)linfo.st_size;

    // Is it a directory?
	isDirectory=S_ISDIR(info.st_mode);
    if(isDirectory)
    {		
		// Directory path
		FXString dupath=FXPath::absolute(path,file);

#if defined(linux)
		FILE *mtab=setmntent("/etc/mtab","r");
        struct mntent *mnt;
        if(mtab)
        {
            while((mnt=getmntent(mtab)))
            {
                if(!::streq(mnt->mnt_type,MNTTYPE_IGNORE) && !::streq(mnt->mnt_type,MNTTYPE_SWAP) && !::streq(mnt->mnt_type,"proc"))
                {
                    if(::streq(mnt->mnt_dir,dupath.text()))
                    {
                        ismountpoint=TRUE;
                        snprintf(buf,sizeof(buf)-1,_("Filesystem (%s)"),mnt->mnt_fsname);
                        type=buf;
						strlcpy(mnttype,mnt->mnt_type,strlen(mnt->mnt_type)+1);
                    }
                }
            }
            endmntent(mtab);
        }
#endif
		// If it is a mount point
		if(ismountpoint)
		{
			// Caution : use the -P option to be POSIX compatible!
			snprintf(buf,sizeof(buf)-1,"df -P --block-size=1 '%s'",filename.text());
			p=popen(buf,"r");
			fgets(buf,sizeof(buf),p);
			fgets(buf,sizeof(buf),p);
			strtok(buf," "); // should check if str!=NULL
			strtok(NULL," ");// get size
			char *pstr;
			pstr=strtok(NULL," ");
			strlcpy(used,pstr,strlen(pstr)+1);// get used
			pstr=strtok(NULL," ");
			strlcpy(avail,pstr,strlen(pstr)+1);// get available
			pstr=strtok(NULL," ");
			strlcpy(pctr,pstr,strlen(pstr)+1);// get percentage
			pclose(p);
		}
			
		// If it is a folder
		else
		{
			type=_("Folder");
    		getApp()->beginWaitCursor();
			strlcpy(buf,dupath.text(),dupath.length()+1);
			dirsize=pathsize(buf);
			snprintf(size,sizeof(size)-1,"%llu",dirsize);
    		getApp()->endWaitCursor();
		}
    }
	
    else if(S_ISCHR(info.st_mode))
        type=_("Character Device");
    else if(S_ISBLK(info.st_mode))
        type=_("Block Device");
    else if(S_ISFIFO(info.st_mode))
        type=_("Named Pipe");
    else if(S_ISSOCK(info.st_mode))
        type=_("Socket");
    
	// Regular file or link
	else
    {
		// Try to use association table	
		extension=FXPath::name(filename).rafter('.',1);
		extension=extension.lower();
		fileassoc=getApp()->reg().readStringEntry("FILETYPES",extension.text(),"");

		// If we have an association
        if(!fileassoc.empty())
        {
            FXString c;
            type=fileassoc.section(';',1);
            ext->setText(extension);
            ext->setEditable(FALSE);
            c=fileassoc.section(';',0);
            descr->setText(fileassoc.section(';',1));
            open->setText(c.section(',',0));
            view->setText(c.section(',',1));
            if(edit)
                edit->setText(c.section(',',2));
            bigic->setText(fileassoc.section(';',2));
            miniic->setText(fileassoc.section(';',3));

            if (!isLink)
				fassoctab->enable();
        }
        else
        {
            ext->setText(extension);
            ext->disable();
			if(info.st_mode&(S_IXUSR|S_IXGRP|S_IXOTH))
                type=_("Executable");
            else
            {
                type=_("Document");
				if (!isLink)
					fassoctab->enable();
            }
        }
    }
	
	// Modify file type for links
	if (isBrokenLink)
		type=_("Broken link");
	else if (isLink)
		type=_("Link to ")+type;

	// Parent directory name not editable
	if (file=="..")
		input->setEditable(FALSE);

	// Root directory name not editable
    if (file=="" & path==ROOTDIR)
	{
		input->setText(ROOTDIR);
		input->setEditable(FALSE);
	}
	else		
		input->setText(file);
    
	input->setFocus();
    input->onCmdSelectAll(0,0,0);

    // Set permissions

    perm->ur->setCheck((orig_mode & S_IRUSR) ? TRUE : FALSE);
    perm->uw->setCheck((linfo.st_mode & S_IWUSR) ? TRUE : FALSE);
    perm->ux->setCheck((linfo.st_mode & S_IXUSR) ? TRUE : FALSE);

    perm->gr->setCheck((linfo.st_mode & S_IRGRP) ? TRUE : FALSE);
    perm->gw->setCheck((linfo.st_mode & S_IWGRP) ? TRUE : FALSE);
    perm->gx->setCheck((linfo.st_mode & S_IXGRP) ? TRUE : FALSE);

    perm->or_->setCheck((linfo.st_mode & S_IROTH) ? TRUE : FALSE);
    perm->ow->setCheck((linfo.st_mode & S_IWOTH) ? TRUE : FALSE);
    perm->ox->setCheck((linfo.st_mode & S_IXOTH) ? TRUE : FALSE);
    perm->set->setCheck();
	perm->all->setCheck();

	FXLabel *mtType=NULL, *mtUsed=NULL, *mtFree=NULL, *fileType=NULL, *fileSize=NULL, *fileChanged=NULL, *fileAccessed=NULL, *fileModified=NULL;

	// Properties are different for mount points
	if (ismountpoint)
	{
  		FXGroupBox *mtgroup=new FXGroupBox(generalframe,_("Mount point"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
  		FXMatrix *mtmatrix=new FXMatrix(mtgroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  		new FXLabel(mtmatrix,_("Mount type:"),NULL,LAYOUT_LEFT);
  		fileType=new FXLabel(mtmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("Used:"),NULL,LAYOUT_LEFT);
  		mtUsed=new FXLabel(mtmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("Free:"),NULL,LAYOUT_LEFT);
  		mtFree=new FXLabel(mtmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("File system:"),NULL,LAYOUT_LEFT);
  		mtType=new FXLabel(mtmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(mtmatrix,_("Location:"),NULL,LAYOUT_LEFT);
  		location=new FXLabel(mtmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	}
	else
	{
  		FXGroupBox *attrgroup=new FXGroupBox(generalframe,_("Properties"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
  		FXMatrix *attrmatrix=new FXMatrix(attrgroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  		new FXLabel(attrmatrix,_("Type:"),NULL,LAYOUT_LEFT);
  		fileType=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		sizelabel=new FXLabel(attrmatrix,_("Total size:"),NULL,LAYOUT_LEFT);
  		fileSize=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(attrmatrix,_("Location:"),NULL,LAYOUT_LEFT);
  		location=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
		if (isLink & !isBrokenLink)
		{
			new FXLabel(attrmatrix,_("Link to:"),NULL,LAYOUT_LEFT);
			linkto=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
		}
		else if(isBrokenLink)
		{
			new FXLabel(attrmatrix,_("Broken link to:"),NULL,LAYOUT_LEFT);
			linkto=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
		}
 		FXGroupBox *timegroup=new FXGroupBox(generalframe,_("File Time"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X);
  		FXMatrix *timematrix=new FXMatrix(timegroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
  		new FXLabel(timematrix,_("Last Modified:"),NULL,LAYOUT_LEFT);
  		fileModified=new FXLabel(timematrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(timematrix,_("Last Changed:"),NULL,LAYOUT_LEFT);
  		fileChanged=new FXLabel(timematrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
  		new FXLabel(timematrix,_("Last Accessed:"),NULL,LAYOUT_LEFT);
  		fileAccessed=new FXLabel(timematrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	}

    // File or mount type
	fileType->setText(type.text());

	// Parent directory
	location->setText(path);
	
	// Refered file for valid or broken link
	if (isLink)
		linkto->setText(filename);
		
	// If directory
	if (isDirectory)
	{
		// if mount point
		if(ismountpoint)
		{
			hsize=::hSize(used);
        	snprintf(size,sizeof(size)-1,"%s (%s)",hsize.text(),pctr);        
			mtUsed->setText(size);
			hsize=::hSize(avail);
			mtFree->setText(hsize);
			mtType->setText(mnttype);
		}
		// if folder
		else
		{
        	fileModified->setText(mod);
        	fileChanged->setText(changed);
        	fileAccessed->setText(accessed);
			hsize=::hSize(size);
        	snprintf(size,sizeof(size)-1,_("%s (%llu bytes)"),hsize.text(),dirsize);        
			fileSize->setText(size);
		}
	}
	// Regular file
    else
    {
        snprintf(size,sizeof(size)-1,"%llu",filesize);
        hsize=::hSize(size);
        snprintf(size,sizeof(size)-1,_("%s (%llu bytes)"),hsize.text(),filesize);        
		sizelabel->setText(_("Size:"));
		fileSize->setText(size);
        fileModified->setText(mod);
        fileChanged->setText(changed);
        fileAccessed->setText(accessed);
    }
    mode = orig_mode;
    perm->cmd=PropertiesBox::ID_SET;
    perm->flt=PropertiesBox::ID_ALL;
    files = &file;
    source = file;
    num = 1;	
}

// Construct window for multiple files
PropertiesBox::PropertiesBox(FXWindow *win,FXString *file,int n,FXString path): DialogBox(win,_("Properties"), DECOR_TITLE|DECOR_BORDER|DECOR_STRETCHABLE)
{
	unsigned long long filesize, dirsize, totalsize=0;
	struct stat info;
	FXString grpid, usrid;
    FXString type, extension, fileassoc;
    char buf[MAXPATHLEN+1];
	int i, isDirectory, nbdirs=0, nbfiles=0, dotdot=0;
	FXbool firstfile=TRUE;

    // Buttons
    FXHorizontalFrame *buttons=new FXHorizontalFrame(this,LAYOUT_SIDE_BOTTOM|LAYOUT_FILL_X,0,0,0,0,10,10,5,5);

    // Contents
    FXVerticalFrame *contents=new FXVerticalFrame(this,LAYOUT_SIDE_TOP|FRAME_NONE|LAYOUT_FILL_X|LAYOUT_FILL_Y);

    // Accept
    FXButton *ok = new FXButton(buttons,_("&Accept"),NULL,this,PropertiesBox::ID_ACCEPT_MULT,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);
    ok->addHotKey(KEY_Return);

    // Cancel
    new FXButton(buttons,_("&Cancel"),NULL,this,DialogBox::ID_CANCEL,FRAME_RAISED|FRAME_THICK|LAYOUT_RIGHT|LAYOUT_CENTER_Y,0,0,0,0,20,20);

    // Switcher
    FXTabBook *tabbook = new FXTabBook(contents,NULL,0,LAYOUT_FILL_X|LAYOUT_FILL_Y|LAYOUT_RIGHT);

    // First item is General
    new FXTabItem(tabbook,_("&General"),NULL);
    FXVerticalFrame *generalframe=new FXVerticalFrame(tabbook,FRAME_THICK|FRAME_RAISED);
	FXGroupBox *attrgroup=new FXGroupBox(generalframe,_("Properties"),GROUPBOX_TITLE_LEFT|FRAME_GROOVE|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	FXMatrix *attrmatrix=new FXMatrix(attrgroup,2,MATRIX_BY_COLUMNS|LAYOUT_FILL_X|LAYOUT_FILL_Y);
	new FXLabel(attrmatrix,_("Selection:"),NULL,LAYOUT_LEFT);
	FXLabel *filesSelected=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	new FXLabel(attrmatrix,_("Type:"),NULL,LAYOUT_LEFT);
	FXLabel *filesType=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	new FXLabel(attrmatrix,_("Total size:"),NULL,LAYOUT_LEFT);
	FXLabel *filesSize=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);
	new FXLabel(attrmatrix,_("Location:"),NULL,LAYOUT_LEFT);
	location=new FXLabel(attrmatrix,FXString::null,NULL,LAYOUT_LEFT|LAYOUT_FILL_COLUMN);

    // Second item is Access Permissions
    new FXTabItem(tabbook,_("&Permissions"),NULL);
    perm = new PermFrame(tabbook,this);

    // Get file/link info of the second file of the list (the first should be ".."
	// This is used as a guess for the username, group and permissions of the whole list
    if(lstatrep(file[1].text(),&info)!=0)
        return ;
	
	// Obtain user name
	usrid=FXSystem::userName(info.st_uid);

	// Obtain group name
	grpid=FXSystem::groupName(info.st_gid);

    orig_mode=(info.st_mode & (S_IRWXU|S_IRWXG|S_IRWXO));

    perm->ur->setCheck((orig_mode & S_IRUSR) ? TRUE : FALSE);
    perm->uw->setCheck((info.st_mode & S_IWUSR) ? TRUE : FALSE);
    perm->ux->setCheck((info.st_mode & S_IXUSR) ? TRUE : FALSE);

    perm->gr->setCheck((info.st_mode & S_IRGRP) ? TRUE : FALSE);
    perm->gw->setCheck((info.st_mode & S_IWGRP) ? TRUE : FALSE);
    perm->gx->setCheck((info.st_mode & S_IXGRP) ? TRUE : FALSE);

    perm->or_->setCheck((info.st_mode & S_IROTH) ? TRUE : FALSE);
    perm->ow->setCheck((info.st_mode & S_IWOTH) ? TRUE : FALSE);
    perm->ox->setCheck((info.st_mode & S_IXOTH) ? TRUE : FALSE);
    perm->add->setCheck();
	perm->all->setCheck();

    perm->user->setText(usrid);
    perm->grp->setText(grpid);

    mode = orig_mode;
    perm->cmd = PropertiesBox::ID_SET;
	perm->flt = PropertiesBox::ID_ALL;
    files = file;
    source = "";
    num = n;

	// Parent directory
	location->setText(path);

	// Start wait cursor
	getApp()->beginWaitCursor();

	// Total size and files type
	for(i=0;i<num;i++)
	{
		if(lstatrep(files[i].text(),&info)!=0)
			continue ;

		// Special case of the ".." directory
		if (files[i]=="..")
		{	
			dotdot=1;
			continue;
		}

		// Is it a directory?
		isDirectory=S_ISDIR(info.st_mode);
		if(isDirectory)
		{
			// Recursive size
			FXString dupath=FXPath::absolute(path,files[i]);
			strlcpy(buf,dupath.text(),dupath.length()+1);
			dirsize=pathsize(buf);
			totalsize+=dirsize;
			nbdirs++;
		}
		else // regular file
		{
			// File size	
    		filesize=(unsigned long long)info.st_size;
			totalsize+=filesize;
			nbfiles++;

			// Try to use association table	
			extension=FXPath::name(files[i]).rafter('.',1);
			extension=extension.lower();
			fileassoc=getApp()->reg().readStringEntry("FILETYPES",extension.text(),"");

			// Keep the first encountered type
			if (firstfile)
			{
				// If we have an association
        		if(!fileassoc.empty())
 					type=fileassoc.section(';',1);
				else
					type=_("Document");
				firstfile=FALSE;
			}
			else
			{
				// If we have an association
        		if (!fileassoc.empty())
				{
					if (fileassoc.section(';',1)!=type)
						type=_("Multiple types");
				}
				else
				{
					if (type!=_("Document"))
						type=_("Multiple types");
				}
			}
		}
	}

	// Special cases of the file type
	if (nbfiles==0)
		type=_("Folder");
	
	if (nbdirs>=1 && nbfiles>=1)
		type=_("Multiple types");
	
	// Display total size
	FXchar size[64];
	FXString hsize;
	snprintf(size,sizeof(size)-1,"%llu",totalsize);
	hsize=::hSize(size);
	snprintf(size,sizeof(size)-1,_("%s (%llu bytes)"),hsize.text(),totalsize);        
	filesSize->setText(size);
	
	// Number of selected files
	snprintf(buf,sizeof(buf)-1,_("%d items including %d folder(s) and %d file(s)"),num,nbdirs+dotdot,nbfiles);
	filesSelected->setText(buf);

	// Display type of selected files
	filesType->setText(type);

	// Stop wait cursor
	getApp()->endWaitCursor();
}


// Make window
void PropertiesBox::create()
{
    DialogBox::create();
    //show(PLACEMENT_SCREEN);
}



// Dialog for multiple selected files
long PropertiesBox::onCmdAcceptMult(FXObject* o,FXSelector s,void* p)
{
    int rc=0,i;
	File *f=NULL;
	char file[MAXPATHLEN];

    DialogBox::onCmdAccept(o,s,p);
    if(mode)
    {
		f = new File(this,_("File permissions"),CHMOD);
		f->create();

		// Wait cursor
		getApp()->beginWaitCursor();

        for(i=0;i<num;i++)
        {
            struct stat info;
            mode_t m;
            if(lstatrep(files[i].text(),&info)!=0)
                continue ;

            switch(perm->cmd)
            {
            	case PropertiesBox::ID_ADD:
                	m=info.st_mode|mode;
                	break;
            	case PropertiesBox::ID_CLEAR:
                	m=info.st_mode & ~mode;
                	break;
            	case PropertiesBox::ID_SET:
                	m=mode;
                	break;
            	default :
					getApp()->endWaitCursor();
					return 1;
            }
            
			if (files[i]!=".." && !perm->own->getCheck())
			{
				rc=f->chmod(strdup(files[i].text()),file,m,perm->rec->getCheck(),perm->dironly->getCheck(),perm->fileonly->getCheck());
				
				// If action is cancelled in progress dialog
				if (f->isCancelled)
				{
					f->hide();
					MessageBox::error(this,BOX_OK,_("Error"),_("Change file(s) permissions cancelled!"));
					delete f;
					getApp()->endWaitCursor();
					return 1;
				}
        		
				// Handle chmod errors
				if(rc)
				{
					if (errno)
						MessageBox::error(this,BOX_OK,_("Error"),_("Chmod in %s failed: %s"), file, strerror(errno));
					else
						MessageBox::error(this,BOX_OK,_("Error"),_("Chmod in %s failed"), file);
					break;
				}
			}
        }
		delete f;
		getApp()->endWaitCursor();
    }

    rc = 0;
    // chown if needed
    if(!perm->grp->getText().empty() || !perm->user->getText().empty())
    {
        f = new File(this,_("File owner"),CHOWN);
        f->create();

		// Wait cursor
		getApp()->beginWaitCursor();

        for(i=0;i<num;i++)
        {
            struct stat info;
            if(lstatrep(files[i].text(),&info)!=0)
                continue ;

            uid_t uid = 32768;
            gid_t gid = 32768;
            struct passwd* pwde;
            while( (pwde = getpwent()) )
                if(perm->user->getText() == pwde->pw_name)
                    uid = pwde->pw_uid;
            endpwent();

            struct group* grpe;
            while( (grpe = getgrent()) )
                if(perm->grp->getText() == grpe->gr_name)
                    gid = grpe->gr_gid;
            endgrent();

			if (files[i]!="..")
            	rc |= f->chown(strdup(files[i].text()),file,uid,gid,perm->rec->getCheck(),perm->dironly->getCheck(),perm->fileonly->getCheck());

			// If action is cancelled in progress dialog
			if (f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("Change owner cancelled!"));
				delete f;
				getApp()->endWaitCursor();
				return 1;
			}
			
			// Handle chmod errors
            if(rc)
            {
				if (errno)
					MessageBox::error(this,BOX_OK,_("Error"),_("Chown in %s failed: %s"), file, strerror(errno));
				else
					MessageBox::error(this,BOX_OK,_("Error"),_("Chown in %s failed"), file);
                break;
            }
        }
		delete f;
		getApp()->endWaitCursor();
    }

	// Redraw the file lists
	((XFileExplorer*) mainWindow)->redrawPanels();

    return 1;
}


// Dialog for single selected file
long PropertiesBox::onCmdAcceptSingle(FXObject* o,FXSelector s,void* p)
{
	char **str=NULL;
    int rc=0;
	File *f=NULL;
	char file[MAXPATHLEN];
    FXString oldfileassoc,fileassoc,op,v,e;

	// Don't proceed any task on the parent directory
	if (::streq(source.text(),".."))
		return 1;

	op = open->getText();
    v = view->getText();
    if(!v.empty())
        v = "," + v;
    if(edit)
    {
        e = edit->getText();
        if(!e.empty())
            e = "," + e;
    }

    fileassoc = ext->getText();
    fileassoc += "=";
    fileassoc += op + v + e + ";";
    fileassoc += descr->getText() + ";";
    fileassoc += bigic->getText() + ";" + miniic->getText() + ";;";

	if (ext->getText()!="")
	{
		oldfileassoc=getApp()->reg().readStringEntry("FILETYPES",ext->getText().text(),"");
    	if(oldfileassoc == "" || fileassoc.section('=',1) != oldfileassoc)
    	{
			FXString command=fileassoc.section('=',1);
        	getApp()->reg().writeStringEntry("FILETYPES",ext->getText().text(),command.text());

			// Handle file association
			str=new char*[2];
			str[0]=new char[strlen(ext->getText().text())+1];
			str[1]=new char[strlen(command.text())+1];
			strlcpy(str[0],ext->getText().text(),ext->getText().length()+1);
			strlcpy(str[1],command.text(),command.length()+1);
			mainWindow->handle(this,FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC),str);
    	}
	}
    
    // Source and target path names
	FXString targetpath;
	FXString sourcepath=parentdir+"/"+source;  
    FXString target=input->getText();
	FXString targetparentdir=FXPath::directory(target);
	if (targetparentdir=="")
	{
		targetparentdir=parentdir;
		targetpath=targetparentdir+"/"+target;
	}
	else
		targetpath=target;
	
	// Nothing entered
	if (target=="")
	{
		MessageBox::warning(this,BOX_OK,_("Warning"),_("File name is empty, operation cancelled"));
		input->setText(source);
		return 0;
	}

    // Rename file if necessary
	if(source != target)
	{
		// Source path is not writable	
		if (!::isWritable(sourcepath))
        	MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),source.text());
		
		// Target parent directory doesn't exist or is not writable
		if (!::exists(targetparentdir))
        {
			MessageBox::error(this,BOX_OK,_("Error"),_("Folder %s doesn't exist"),targetparentdir.text());
        	return 0;
		}
		if (!::isWritable(targetparentdir))
		{
         	MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"),targetparentdir.text());
        	return 0;
		}

		// Rename file or directory
		else
		{
			File *f;
			f=new File(this,_("File rename"),RENAME);
       		f->create();
			f->rename(sourcepath,targetpath);
			delete f;
		}
	}

    // Change perm
    DialogBox::onCmdAccept(o,s,p);
	if (target==".." || target==".")
	{
		input->setText(source);
		return 1;
	}
    // Chmod only if "Owner only" is not checked
	if (!perm->own->getCheck())
    {
        struct stat info;
        mode_t m;
		
        if(lstatrep(targetpath.text(),&info)!=0)
            return 1;
			
        f = new File(this,_("File permissions"),CHMOD);
        f->create();

        switch(perm->cmd)
        {
        	case PropertiesBox::ID_ADD:
            	m=info.st_mode|mode;
            	break;
        	case PropertiesBox::ID_CLEAR:
            	m=info.st_mode & ~mode;
            	break;
        	case PropertiesBox::ID_SET:
            	m=mode;
            	break;
        	default :
            	return 1;
        }

		// Wait cursor
		getApp()->beginWaitCursor();
		
		// Perform chmod on the selected file or directory
		rc=f->chmod(strdup(targetpath.text()),file,m,perm->rec->getCheck(),perm->dironly->getCheck(),perm->fileonly->getCheck());

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,BOX_OK,_("Error"),_("Change file permissions cancelled!"));
			delete f;
			getApp()->endWaitCursor();
			return 1;
		}
		delete f;
		getApp()->endWaitCursor();
        
		// Handle chmod errors
		if (rc)
		{
			if (errno)
				MessageBox::error(this,BOX_OK,_("Error"),_("Chmod in %s failed: %s"), file, strerror(errno));
			else
				MessageBox::error(this,BOX_OK,_("Error"),_("Chmod in %s failed"), file);
		}
    }

    // Chown if needed
    rc=0;
    if (oldgrp != perm->grp->getText() || oldusr != perm->user->getText() || perm->rec->getCheck())
    {
        f = new File(this,_("File owner"),CHOWN);
        f->create();

        uid_t uid = 32768;
        gid_t gid = 32768;
        struct passwd* pwde;
        while( (pwde = getpwent()) )
            if(perm->user->getText() == pwde->pw_name)
                uid = pwde->pw_uid;
        endpwent();

        struct group* grpe;
        while( (grpe = getgrent()) )
            if(perm->grp->getText() == grpe->gr_name)
                gid = grpe->gr_gid;
        endgrent();

		// Wait cursor
		getApp()->beginWaitCursor();

        // Perform chown on the selected file or directory
		rc = f->chown(strdup(targetpath.text()),file,uid,gid,perm->rec->getCheck(),perm->dironly->getCheck(),perm->fileonly->getCheck());

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,BOX_OK,_("Error"),_("Change owner cancelled!"));
			delete f;
			getApp()->endWaitCursor();
			return 1;
		}
		delete f;
		getApp()->endWaitCursor();
		
		// Handle chown errors
        if(rc)
		{
			if (errno)
				MessageBox::error(this,BOX_OK,_("Error"),_("Chown in %s failed: %s"), file, strerror(errno));
			else
				MessageBox::error(this,BOX_OK,_("Error"),_("Chown in %s failed"), file);
		}
    }
	
	// Redraw the file lists
	((XFileExplorer*) mainWindow)->redrawPanels();

    return 1;
}

long PropertiesBox::onCmdCancel(FXObject* o,FXSelector s,void* p)
{
    DialogBox::onCmdCancel(o,s,p);
	
	// Redraw the file lists
	((XFileExplorer*) mainWindow)->redrawPanels();

	return 1;
}

	
	long PropertiesBox::onCmdCommand(FXObject* o,FXSelector s,void* p)
{
    perm->cmd = FXSELID(s);
    return 1;
}

long PropertiesBox::onCmdFilter(FXObject* o,FXSelector s,void* p)
{
    perm->flt = FXSELID(s);
    return 1;
}

long PropertiesBox::onCmdCheck(FXObject* o,FXSelector s,void* p)
{
    FXint xmode=0;
	switch (FXSELID(s))
	{
		case PropertiesBox::ID_RUSR:
			xmode=S_IRUSR;
			break;
		case PropertiesBox::ID_WUSR:
			xmode=S_IWUSR;
			break;
		case PropertiesBox::ID_XUSR:
			xmode=S_IXUSR;
			break;
		case PropertiesBox::ID_RGRP:
			xmode=S_IRGRP;
			break;
		case PropertiesBox::ID_WGRP:
			xmode=S_IWGRP;
			break;
		case PropertiesBox::ID_XGRP:
			xmode=S_IXGRP;
			break;
		case PropertiesBox::ID_ROTH:
			xmode=S_IROTH;
			break;
		case PropertiesBox::ID_WOTH:
			xmode=S_IWOTH;
			break;
		case PropertiesBox::ID_XOTH:
			xmode=S_IXOTH;
			break;
	}
	FXCheckButton* ch = (FXCheckButton*)o;    
	if(!ch->getCheck())
       mode &= ~xmode;
    else
        mode |= xmode;
    return 1;

}


long PropertiesBox::onCmdBrowse(FXObject* o,FXSelector s,void* p)
{
    FileDialog browseProgram(this,_("Select an executable file"));
    const FXchar *patterns[]=
        {
            _("All files"),     "*",NULL
        };
	browseProgram.setFilename(ROOTDIR);
	browseProgram.setPatternList(patterns);
	browseProgram.setSelectMode(SELECT_FILE_EXISTING);
	if(browseProgram.execute())
    {
        char *path=strdup(browseProgram.getFilename().text());
        switch(FXSELID(s))
        {
        	case ID_BROWSE_OPEN:
            	open->setText(FXPath::name(path).text());
            	break;
        	case ID_BROWSE_VIEW:
            	view->setText(FXPath::name(path).text());
            	break;
        	case ID_BROWSE_EDIT:
            	if(edit)
                	edit->setText(FXPath::name(path).text());
            	break;
        }
    }
    return 1;
}


long PropertiesBox::onCmdBrowseIcon(FXObject* o,FXSelector s,void* p)
{
    FXString icon;
    if(FXSELID(s)==ID_BIG_ICON)
        icon = bigic->getText();
    else
        icon= miniic->getText();

	FXString iconpath=getApp()->reg().readStringEntry("SETTINGS","iconpath",DEFAULTICONPATH);
    const FXchar *patterns[]=
        {
            _("PNG Images"),     "*.png",
			_("GIF Images"),     "*.gif",
            _("BMP Images"),     "*.bmp",NULL
        };
	FileDialog browseIcon(this,_("Select an icon file"));
    browseIcon.setFilename(iconpath+PATHSEPSTRING+icon);
    browseIcon.setPatternList(patterns);
	browseIcon.setSelectMode(SELECT_FILE_EXISTING);
    if(browseIcon.execute())
    {
        char *path=strdup(browseIcon.getFilename().text());
        if(!::exists(path))
            return 0;
        if(FXSELID(s)==ID_BIG_ICON)
            bigic->setText(path);
        else
            miniic->setText(path);
    }
    return 1;
}

long PropertiesBox::onUpdPerm(FXObject* o,FXSelector s,void* p)
{
    if(perm->rec->getCheck())
	{
        perm->dironly->enable();
        perm->fileonly->enable();
		perm->all->enable();
	}
    else
	{
		perm->all->disable();
        perm->dironly->disable();
        perm->fileonly->disable();
	}

    if(perm->own->getCheck())
	{
        perm->set->disable();
        perm->clear->disable();
        perm->add->disable();
		perm->ur->disable();
		perm->uw->disable();
		perm->ux->disable();
		perm->gr->disable();
		perm->gw->disable();
		perm->gx->disable();
		perm->or_->disable();
		perm->ow->disable();
		perm->ox->disable();
	}
    else
	{
        perm->set->enable();
        perm->clear->enable();
        perm->add->enable();
		perm->ur->enable();
		perm->uw->enable();
		perm->ux->enable();
		perm->gr->enable();
		perm->gw->enable();
		perm->gx->enable();
		perm->or_->enable();
		perm->ow->enable();
		perm->ox->enable();
	}

    return 1;
}


long PropertiesBox::onCmdKeyPress(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent* event=(FXEvent*)ptr;
    switch(event->code)
    {
    case KEY_Escape:
        handle(this,FXSEL(SEL_COMMAND,ID_CANCEL),NULL);
		return 1;
    case KEY_KP_Enter:
    case KEY_Return:
        handle(this,FXSEL(SEL_COMMAND,ID_ACCEPT_SINGLE),NULL);
		return 1;
    default:
        FXTopWindow::onKeyPress(sender,sel,ptr);
		return 1;
    }
	return 0;
}
