#ifndef FILELIST_H
#define FILELIST_H

#include "StringList.h"


struct FileAssoc;
class  FileDict;
class  FileList;

// File item
class FXAPI FileItem : public FXIconItem
{
    FXDECLARE(FileItem)
    friend class FileList;
protected:
    FileAssoc  *assoc;                      // File association record
	FileItem   *link;                   	// Link to next item
    unsigned long long size;                // File size
    FXTime        date;                     // File time
	FXTime		  deldate;
protected:
    FileItem():assoc(NULL),link(NULL),size(0),date(0)
    {}
protected:
    enum{
        FOLDER     = 64,                        // Directory item
        EXECUTABLE = 128,                       // Executable item
        SYMLINK    = 256,                       // Symbolic linked item
        CHARDEV    = 512,                       // Character special item
        BLOCKDEV   = 1024,                      // Block special item
        FIFO       = 2048,                      // FIFO item
        SOCK       = 4096                       // Socket item
    };
public:
    // Constructor
    FileItem(const FXString& text,FXIcon* bi=NULL,FXIcon* mi=NULL,void* ptr=NULL):FXIconItem(text,bi,mi,ptr),assoc(NULL),link(NULL),size(0),date(0)
    {}

    // Return true if this is a file item
    FXbool isFile() const
    {
        return (state&(FOLDER|BLOCKDEV|CHARDEV|FIFO|SOCK))==0;
    }

    // Return true if this is a directory item
    FXbool isDirectory() const
    {
        return (state&FOLDER)!=0;
    }

    // Return true if this is an executable item
    FXbool isExecutable() const
    {
        return (state&EXECUTABLE)!=0;
    }

    // Return true if this is a symbolic link item
    FXbool isSymlink() const
    {
        return (state&SYMLINK)!=0;
    }

    // Return true if this is a character device item
    FXbool isChardev() const
    {
        return (state&CHARDEV)!=0;
    }

    // Return true if this is a block device item
    FXbool isBlockdev() const
    {
        return (state&BLOCKDEV)!=0;
    }

    // Return true if this is an FIFO item
    FXbool isFifo() const
    {
        return (state&FIFO)!=0;
    }

    // Return true if this is a socket
    FXbool isSocket() const
    {
        return (state&SOCK)!=0;
    }

    // Return the file-association object for this item
    FileAssoc* getAssoc() const
    {
        return assoc;
    }

    // Return the file size for this item
    unsigned long long getSize() const
    {
        return size;
    }

    // Return the date for this item
    FXTime getDate() const
    {
        return date;
    }
};


// File List object
class FXAPI FileList : public FXIconList
{
    FXDECLARE(FileList)
protected:
	FileItem	*list;                  // File item list
	FXint 		 prevIndex;
	FXString     directory;             // Current directory
    FXString     orgdirectory;          // Original directory
    FXString     dropdirectory;         // Drop directory
    FXDragAction dropaction;            // Drop action
    FXString     dragfiles;             // Dragged files
    FileDict	*associations;          // Association table
    FXString     pattern;               // Pattern of file names
    FXuint       matchmode;             // File wildcard match mode
    FXTime       timestamp;             // Time when last refreshed
	FXuint       counter;               // Refresh counter
	FXbool		ignorecase;				// Case sensitivity for file name sorting
	FXbool		allowrefresh;			// Allow or disallow periodic refresh
	FXbool		displaythumbnails;		// Display thumbnails
	FXColor 	*tmpdata;				// Temporary image data
	FXString	trashlocation;          // Location of the trash directory
	FXbool      single;                 // Allow single click navigation for directories
	FXbool		dirsfirst;				// Sort directories first

public:
	StringList		*backhist;          // Back history
	StringList		*forwardhist;       // Forward history

protected:
    FileList()
	{}
    virtual FXIconItem *createItem(const FXString& text,FXIcon *big,FXIcon* mini,void* ptr);
    void listItems();
private:
    FileList(const FileList&);
    FileList &operator=(const FileList&);
public:
	long onCmdRefresh(FXObject*,FXSelector,void*);
    long onCmdRefreshTimer(FXObject*,FXSelector,void*);
    long onOpenTimer(FXObject*,FXSelector,void*);
    long onDNDEnter(FXObject*,FXSelector,void*);
    long onDNDLeave(FXObject*,FXSelector,void*);
    long onDNDMotion(FXObject*,FXSelector,void*);
    long onDNDDrop(FXObject*,FXSelector,void*);
    long onDNDRequest(FXObject*,FXSelector,void*);
    long onBeginDrag(FXObject*,FXSelector,void*);
    long onEndDrag(FXObject*,FXSelector,void*);
    long onDragged(FXObject*,FXSelector,void*);
    long onCmdDirectoryUp(FXObject*,FXSelector,void*);
    long onUpdDirectoryUp(FXObject*,FXSelector,void*);
	long onCmdSortByName(FXObject*,FXSelector,void*);
	long onUpdSortByName(FXObject*,FXSelector,void*);
	long onCmdSortByType(FXObject*,FXSelector,void*);
	long onUpdSortByType(FXObject*,FXSelector,void*);
	long onCmdSortBySize(FXObject*,FXSelector,void*);
	long onUpdSortBySize(FXObject*,FXSelector,void*);
	long onCmdSortByExt(FXObject*,FXSelector,void*);
	long onUpdSortByExt(FXObject*,FXSelector,void*);
	long onCmdSortByTime(FXObject*,FXSelector,void*);
	long onUpdSortByTime(FXObject*,FXSelector,void*);
	long onCmdSortByUser(FXObject*,FXSelector,void*);
	long onUpdSortByUser(FXObject*,FXSelector,void*);
	long onCmdSortByGroup(FXObject*,FXSelector,void*);
	long onUpdSortByGroup(FXObject*,FXSelector,void*);
	long onCmdSortByPerm(FXObject*,FXSelector,void*);
	long onUpdSortByPerm(FXObject*,FXSelector,void*);
	long onCmdSortByDeltime(FXObject*,FXSelector,void*);
	long onUpdSortByDeltime(FXObject*,FXSelector,void*);
	long onCmdSortReverse(FXObject*,FXSelector,void*);
	long onUpdSortReverse(FXObject*,FXSelector,void*);
	long onCmdSortCase(FXObject*,FXSelector,void*);
	long onUpdSortCase(FXObject*,FXSelector,void*);
    long onCmdSetPattern(FXObject*,FXSelector,void*);
    long onUpdSetPattern(FXObject*,FXSelector,void*);
    long onCmdToggleHidden(FXObject*,FXSelector,void*);
    long onUpdToggleHidden(FXObject*,FXSelector,void*);
    long onCmdShowHidden(FXObject*,FXSelector,void*);
    long onUpdShowHidden(FXObject*,FXSelector,void*);
    long onCmdHideHidden(FXObject*,FXSelector,void*);
    long onUpdHideHidden(FXObject*,FXSelector,void*);
	long onCmdHeader(FXObject*,FXSelector,void*);
	long onUpdHeader(FXObject*,FXSelector,void*);
	long onCmdToggleThumbnails(FXObject*,FXSelector,void*);
	long onUpdToggleThumbnails(FXObject* sender,FXSelector,void*);
	long onCmdDirsFirst(FXObject*,FXSelector,void*);
	long onUpdDirsFirst(FXObject*,FXSelector,void*);
	long onCmdDragCopy(FXObject* sender,FXSelector,void*);
	long onCmdDragMove(FXObject* sender,FXSelector,void*);
	long onCmdDragLink(FXObject* sender,FXSelector,void*);
	long onCmdDragReject(FXObject* sender,FXSelector,void*);
	
public:
	static FXint ascending(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descending(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingCase(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingCase(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingType(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingType(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingSize(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingSize(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingExt(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingExt(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingTime(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingTime(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingUser(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingUser(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingGroup(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingGroup(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingPerm(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingPerm(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingDeltime(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingDeltime(const FXIconItem* pa,const FXIconItem* pb);

	static FXint ascendingMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingCaseMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingCaseMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingTypeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingTypeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingSizeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingSizeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingExtMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingExtMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingTimeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingTimeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingUserMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingUserMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingGroupMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingGroupMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingPermMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingPermMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint ascendingDeltimeMix(const FXIconItem* pa,const FXIconItem* pb);
	static FXint descendingDeltimeMix(const FXIconItem* pa,const FXIconItem* pb);
public:
	enum {
		// Note : the order of the 9 following sort IDs must be kept
        ID_SORT_BY_NAME=FXIconList::ID_LAST,
        ID_SORT_BY_SIZE,
        ID_SORT_BY_TYPE,
        ID_SORT_BY_EXT,
        ID_SORT_BY_TIME,
        ID_SORT_BY_USER,
        ID_SORT_BY_GROUP,
        ID_SORT_BY_PERM,   
        ID_SORT_BY_DELTIME,
        ID_SORT_REVERSE,
    	ID_SORT_CASE,
    	ID_DIRS_FIRST,
        ID_DIRECTORY_UP,
        ID_SET_PATTERN,
        ID_SET_DIRECTORY,
        ID_SHOW_HIDDEN,
        ID_HIDE_HIDDEN,
        ID_TOGGLE_HIDDEN,
		ID_TOGGLE_THUMBNAILS,
        ID_REFRESH_TIMER,
		ID_REFRESH,
        ID_OPEN_TIMER,
		ID_DRAG_COPY,
		ID_DRAG_MOVE,
		ID_DRAG_LINK,
		ID_DRAG_REJECT,
        ID_LAST
    };
public:

    // Construct a file list
    FileList(FXComposite *p,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=0,FXint x=0,FXint y=0,FXint w=0,FXint h=0);

    // Create server-side resources
    virtual void create();

	// Scan the current directory and update the items if needed, or if force is TRUE
	void scan(FXbool force=TRUE);

    // Set current file
    void setCurrentFile(const FXString& file);

    // Return current file
    FXString getCurrentFile() const;

    // Set current directory
	void setDirectory(const FXString& path,const FXbool histupdate=TRUE,FXString prevpath= "");

	// Return current directory
    FXString getDirectory() const
    {
        return directory;
    }

    // Change wildcard matching pattern
    void setPattern(const FXString& ptrn);

    // Return wildcard pattern
    FXString getPattern() const
    {
        return pattern;
    }

    // Return TRUE if item is a directory
    FXbool isItemDirectory(FXint index) const;

    // Return TRUE if item is a file
    FXbool isItemFile(FXint index) const;

    // Return TRUE if item is executable
    FXbool isItemExecutable(FXint index) const;

	// Return number of selected items
	FXint getNumSelectedItems(void) const;

	// Return number of selected items and index of first selected item  
	FXint getNumSelectedItems(FXint *index) const;
	
	// Return name of item at index
    FXString getItemFilename(FXint index) const;

    // Return full pathname of item at index
    FXString getItemPathname(FXint index) const;

    // Return file association of item
    FileAssoc* getItemAssoc(FXint index) const;

	// Return file size of the item
	unsigned long long getItemFileSize(FXint index) const;

    // Return wildcard matching mode
    FXuint getMatchMode() const
    {
        return matchmode;
    }

	// Return case sensitivity for file name sorting
	FXbool getIgnoreCase() const
	{
		return ignorecase;
	}

	// Set case sensitivity for file name sorting
	void setIgnoreCase(const FXbool icase)
	{
		ignorecase=icase;
	}

	// Return directory first state for file name sorting
	FXbool getDirsFirst() const
	{
		return dirsfirst;
	}

	// Set directory first state for file name sorting
	void setDirsFirst(const FXbool dfirst)
	{
		dirsfirst=dfirst;
	}

	// Allow or disallow periodic refresh
	void setAllowRefresh(const FXbool allow);

    // Change wildcard matching mode
    void setMatchMode(FXuint mode);

    // Return TRUE if showing hidden files
    FXbool showHiddenFiles() const;

    // Show or hide hidden files
    void showHiddenFiles(FXbool showing);

    // Return TRUE if displaying thumbnails
    FXbool showThumbnails() const;

    // Display or not thumbnails
    void showThumbnails(FXbool display);

    // Return TRUE if showing directories only
    FXbool showOnlyDirectories() const;

    // Show directories only
    void showOnlyDirectories(FXbool shown);

    // Change file associations
    void setAssociations(FileDict* assoc);

    // Return file associations
    FileDict* getAssociations() const
    {
        return associations;
    }

    // Destructor
    virtual ~FileList();
};


#endif
