#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/wait.h>

#include <fox-1.6/fx.h>
#include <fox-1.6/fxkeys.h>
#include <fox-1.6/FXPNGIcon.h>

#include "xfedefs.h"
#include "icons.h"
#include "DirPanel.h"
#include "DirList.h"
#include "Properties.h"
#include "ArchInputDialog.h"
#include "File.h"
#include "XFileExplorer.h"
#include "MessageBox.h"

// Refresh interval for the directory size (ms)
#define DIRSIZE_INTERVAL 1000

// Clipboard
extern FXString clipboard;
extern FXuint clipboard_type;

// Global variables
extern FXMainWindow *mainWindow;
extern FXbool allowPopupScroll;
#if defined(linux)
extern FXStringDict* fsdevices;
extern FXStringDict* mtdevices;
#endif

extern FXbool single_click_diropen;
extern FXbool single_click_fileopen;


// Map
FXDEFMAP(DirPanel) DirPanelMap[]={
								   FXMAPFUNC(SEL_CLIPBOARD_LOST,0,DirPanel::onClipboardLost),
								   FXMAPFUNC(SEL_CLIPBOARD_GAINED,0,DirPanel::onClipboardGained),
								   FXMAPFUNC(SEL_CLIPBOARD_REQUEST,0,DirPanel::onClipboardRequest),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_EXPANDTREE,DirPanel::onExpandTree),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COLLAPSETREE,DirPanel::onCollapseTree),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_PROPERTIES,DirPanel::onCmdProperties),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_ARCHIVE,DirPanel::onCmdAddToArch),
                                   FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,DirPanel::ID_FILELIST,DirPanel::onPopupMenu),
                               	   FXMAPFUNC(SEL_CLICKED,DirPanel::ID_FILELIST,DirPanel::onCmdDirectory),
                               	   FXMAPFUNC(SEL_EXPANDED,DirPanel::ID_FILELIST,DirPanel::onExpand),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_COPY_CLIPBOARD,DirPanel::onCmdCopyCut),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_CUT_CLIPBOARD,DirPanel::onCmdCopyCut),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_PASTE_CLIPBOARD,DirPanel::onCmdPaste),
                               	   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_COPY,DirPanel::onCmdDirMan),
                               	   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_CUT,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_COPYTO,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_MOVE,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_RENAME,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_SYMLINK,DirPanel::onCmdDirMan),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_DELETE,DirPanel::onCmdDirDelete),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_DIR_TRASH,DirPanel::onCmdDirTrash),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_NEW_DIR,DirPanel::onCmdNewDir),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_XTERM,DirPanel::onCmdXTerm),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_TOGGLE_HIDDEN,DirPanel::onCmdToggleHidden),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_TOGGLE_TREE,DirPanel::onCmdToggleTree),
                                   FXMAPFUNC(SEL_TIMEOUT,DirPanel::ID_DIRSIZE_REFRESH,DirPanel::onCmdDirsizeRefresh),
#if defined(linux)
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_MOUNT,DirPanel::onCmdMount),
                                   FXMAPFUNC(SEL_COMMAND,DirPanel::ID_UMOUNT,DirPanel::onCmdMount),
                               	   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_MOUNT,DirPanel::onUpdMount),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_UMOUNT,DirPanel::onUpdUnmount),
#endif
                               	   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_PASTE_CLIPBOARD,DirPanel::onUpdPaste),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TOGGLE_HIDDEN,DirPanel::onUpdToggleHidden),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TOGGLE_TREE,DirPanel::onUpdToggleTree),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_CUT_CLIPBOARD,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_ARCHIVE,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_MOVE,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_RENAME,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_TRASH,DirPanel::onUpdDirTrash),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_DIR_DELETE,DirPanel::onUpdDirDelete),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_NEW_DIR,DirPanel::onUpdMenu),
                                   FXMAPFUNC(SEL_UPDATE,DirPanel::ID_TITLE,DirPanel::onUpdTitle),
                          		};

// Object implementation 
FXIMPLEMENT(DirPanel,FXVerticalFrame,DirPanelMap,ARRAYNUMBER(DirPanelMap))

// Construct Directory Panel
DirPanel::DirPanel(FXComposite *p, FXColor listbackcolor, FXColor listforecolor, FXuint opts, FXint x, FXint y, FXint w, FXint h):
        FXVerticalFrame(p,opts,x,y,w,h,0,0,0,0)
{
	// Construct directory panel
    FXVerticalFrame* cont=new FXVerticalFrame(this,LAYOUT_FILL_Y|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0, 1,1);
	FXPacker* packer=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X|FRAME_RAISED,0,0,0,0, 0,0,0,0);
	new FXLabel(packer,_("Folders"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X);

    list=new DirList(cont,this,ID_FILELIST,LAYOUT_FILL_X|LAYOUT_FILL_Y|TREELIST_BROWSESELECT|TREELIST_SHOWS_LINES|TREELIST_SHOWS_BOXES);
	list->setTextColor(listforecolor);
	list->setBackColor(listbackcolor);

	statusbar=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 3,3,3,3);
    new FXToggleButton(statusbar,_("\tShow hidden folders (Ctrl-F5)"),_("\tHide hidden folders (Ctrl-F5)"),showhiddenicon,hidehiddenicon,this,
	                   ID_TOGGLE_HIDDEN,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    status=new FXLabel(statusbar,_("0 bytes in root"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN);
    status->setTarget(this);
    status->setSelector(FXSEL(SEL_UPDATE,DirPanel::ID_TITLE));

	// Home and trahscan locations
	homelocation=getenv("HOME");
    if(homelocation=="")
        homelocation=ROOTDIR;
	trashlocation=homelocation+PATHSEPSTRING+TRASHPATH;

	// Single click navigation
    single_click_diropen=getApp()->reg().readUnsignedEntry("OPTIONS","single_click_diropen",FALSE);
    single_click_fileopen=getApp()->reg().readUnsignedEntry("OPTIONS","single_click_fileopen",FALSE);
	if (single_click_diropen & single_click_fileopen)
		list->setDefaultCursor(getApp()->getDefaultCursor(DEF_HAND_CURSOR));

	// Dialogs
	newdirdialog=NULL;
	archdialog=NULL;
	operationdialog=NULL;

	// Initialize clipboard flag	
	paste=FALSE;
}


// Destructor
DirPanel::~DirPanel()
{
    getApp()->removeTimeout(this,ID_DIRSIZE_REFRESH);
	delete list;
	delete statusbar;
	delete status;
	delete newdirdialog;
	delete archdialog;
	delete operationdialog;
}

// Create X window
void DirPanel::create()
{	
	// Register standard uri-list type
	urilistType=getApp()->registerDragType("text/uri-list");
    
	// Register special uri-list type used for Gnome, XFCE and Xfe 
	xfelistType=getApp()->registerDragType("x-special/gnome-copied-files");
	
	// Register special uri-list type used for KDE 
	kdelistType=getApp()->registerDragType("application/x-kde-cutselection");

	// Register standard UTF-8 text type used for file dialogs 
	utf8Type=getApp()->registerDragType("UTF8_STRING");

    getApp()->addTimeout(this,ID_DIRSIZE_REFRESH,DIRSIZE_INTERVAL);
	FXVerticalFrame::create();
}

// To pass the expand message to DirList
long DirPanel::onExpand(FXObject*,FXSelector,void* ptr)
{
	list->handle(this,FXSEL(SEL_EXPANDED,0),(void*)ptr);
	return 1;
}


// Change the directory when clicking on the tree list
long DirPanel::onCmdDirectory(FXObject*,FXSelector,void* ptr)
{
    TreeItem *item=(TreeItem*)ptr;
    if(item)
    {
        FXString directory=list->getItemPathname(item);
        if(!::isReadExecutable(directory))
        {
            MessageBox::error(this,BOX_OK_SU,_("Error"),_(" Permission to: %s denied."), directory.text());
            return 0;
        }
		FilePanel* currentpanel=((XFileExplorer*) mainWindow)->lpanel->current;
		FXComboBox* address=((XFileExplorer*) mainWindow)->address;
        currentpanel->list->setDirectory(directory);
        currentpanel->updatePathLinker();
        
		// Remember latest directory in the location address
       	FXString item;
		FXint i=0;
		FXint count=address->getNumItems();
       	FXString p=currentpanel->list->getDirectory();

		if(!count)
		{
			count++;
			address->insertItem(0,address->getText());
		}
       	while(i < count)
		{
       		item=address->getItem(i++);
       		if(::streq((const char*)&p[0],(const char*)&item[0]))
			{
				i--;
				break;
			}
       	}
       	if(i==count)
			address->insertItem(0,currentpanel->list->getDirectory());

    }

	// Manage single click navigation
	if (item && single_click_diropen)
	{
		//if (item->isExpanded())
			//list->handle(this,FXSEL(SEL_COLLAPSED,0),(void*)ptr);
		//else
			list->handle(this,FXSEL(SEL_EXPANDED,0),(void*)ptr);
	}
		
    return 1;
}

// Toggle hidden files
long DirPanel::onCmdToggleHidden(FXObject*,FXSelector,void*)
{
    list->showHiddenFiles(!list->showHiddenFiles());
    return 1;
}


// Update toggle hidden files widget
long DirPanel::onUpdToggleHidden(FXObject* sender,FXSelector,void*)
{
    if(list->showHiddenFiles())
        sender->handle(this,FXSEL(SEL_COMMAND,ID_CHECK),NULL);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,ID_UNCHECK),NULL);
    return 1;
}


long DirPanel::onCmdToggleTree(FXObject* sender,FXSelector sel,void* ptr)
{
	return this->handle(sender,FXSEL(SEL_COMMAND,FXWindow::ID_TOGGLESHOWN),ptr);
}

long DirPanel::onUpdToggleTree(FXObject* sender,FXSelector sel,void* ptr)
{
    if(this->shown())
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_CHECK),ptr);
    else
        sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_UNCHECK),ptr);
    return 1;
}


// Directory list context menu
long DirPanel::onPopupMenu(FXObject* o,FXSelector s,void* p)
{
	// Current item becomes item under cursor
	FXint x, y, xitem, yitem;
    FXuint state;
	getRoot()->getCursorPosition(x,y,state);
	list->getCursorPosition(xitem,yitem,state);   	    
	DirItem* item=(DirItem*)list->getItemAt(xitem,yitem);

	// If no item under cursor, then return
	if (item==NULL) return 0;

	// If item, then set it current and set directory in DirList and FileList
	list->setCurrentItem(item,TRUE);
	FXString dir=list->getItemPathname((TreeItem*)item);
	list->setDirectory(dir,TRUE);
	((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(dir);
	((XFileExplorer*) mainWindow)->lpanel->current->updatePathLinker();
	
    // Popup menu pane
    FXMenuPane menu(this);
    
	// Menu
#if defined(linux)
	if (::isLink(dir))
		dir=readlink(dir);
    if(fsdevices->find(dir.text()) || mtdevices->find(dir.text()))
	{
		new FXMenuCommand(&menu,_("M&ount"),maphosticon,this,ID_MOUNT);
		new FXMenuCommand(&menu,_("Unmoun&t"),unmaphosticon,this,ID_UMOUNT);
    	new FXMenuSeparator(&menu);
	}
#endif
	new FXMenuCommand(&menu,_("New &folder..."),newfoldericon,this,DirPanel::ID_NEW_DIR);
	new FXMenuCommand(&menu,_("&Add to archive..."),archaddicon,this,DirPanel::ID_ARCHIVE);
    new FXMenuSeparator(&menu);
	new FXMenuCommand(&menu,_("&Copy"),copy_clpicon,this,DirPanel::ID_COPY_CLIPBOARD);
	new FXMenuCommand(&menu,_("C&ut"),cut_clpicon,this,DirPanel::ID_CUT_CLIPBOARD);
	new FXMenuCommand(&menu,_("&Paste"),paste_clpicon,this,DirPanel::ID_PASTE_CLIPBOARD);
    new FXMenuSeparator(&menu);
	new FXMenuCommand(&menu,_("Re&name..."),renameiticon,this,DirPanel::ID_DIR_RENAME);
	new FXMenuCommand(&menu,_("Cop&y to..."),copy_clpicon,this,DirPanel::ID_DIR_COPYTO);
	new FXMenuCommand(&menu,_("&Move to..."),moveiticon,this,DirPanel::ID_DIR_MOVE);
	new FXMenuCommand(&menu,_("Symlin&k to..."),minilinkicon,this,DirPanel::ID_DIR_SYMLINK);
	new FXMenuCommand(&menu,_("Mo&ve to trash"),filedeleteicon,this,DirPanel::ID_DIR_TRASH);
	new FXMenuCommand(&menu,_("&Delete"),filedelete_permicon,this,DirPanel::ID_DIR_DELETE);
    new FXMenuSeparator(&menu);
	new FXMenuCheck(&menu,_("&Hidden folders"),this,DirPanel::ID_TOGGLE_HIDDEN);
	new FXMenuCheck(&menu,_("&Reverse order"),list,DirList::ID_SORT_REVERSE);
	new FXMenuCommand(&menu,_("E&xpand tree"),exptreeicon,this,DirPanel::ID_EXPANDTREE);
	new FXMenuCommand(&menu,_("Collap&se tree"),colltreeicon,this,DirPanel::ID_COLLAPSETREE);
    new FXMenuSeparator(&menu);
	new FXMenuCommand(&menu,_("Prop&erties"),attribicon,this,DirPanel::ID_PROPERTIES);
	
    menu.create();
    allowPopupScroll=TRUE;  // Allow keyboard scrolling
	menu.popup(NULL,x,y);
	getApp()->runModalWhileShown(&menu);
    allowPopupScroll=FALSE;
    return 1;
}


// Helper function used to explore the directory tree and expand or collapse it
long DirPanel::exploreUp(DirItem *item, const DirItem *rootitem, const FXint task)
{
	DirItem* parentitem=item;
	
	if (task==ID_EXPANDTREE)
		list->expandTree((TreeItem*)item,TRUE);
	else
		list->collapseTree((TreeItem*)item,TRUE);
	
	item=(DirItem*)item->getFirst();
	if (!item)
		exploreDown(parentitem,rootitem,task);
	else
		exploreUp(item,rootitem,task);
	return 1;	
}


// Helper function used to explore the directory tree and expand or collapse it
long DirPanel::exploreDown(DirItem *item, const DirItem *rootitem, const FXint task)
{
	if (item==rootitem)
		return 1;
	
	DirItem* parentitem=(DirItem*)item->getParent();
	
	if (task==ID_EXPANDTREE)
		list->expandTree((TreeItem*)item,TRUE);
	else
		list->collapseTree((TreeItem*)item,TRUE);
	item=(DirItem*)item->getNext();
	
	if (!item)
		exploreDown(parentitem,rootitem,task);
	else
	{
		if (task==ID_EXPANDTREE)
			list->expandTree((TreeItem*)item,TRUE);
		else
			list->collapseTree((TreeItem*)item,TRUE);
		if (!list->isItemLeaf(item))
			exploreUp(item,rootitem,task);
		else
			exploreDown(item,rootitem,task);
	}
	return 1;		
}
				

// Expand the directory tree under cursor
long DirPanel::onExpandTree(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem* rootitem=(DirItem*)list->getCurrentItem();
	getApp()->beginWaitCursor();
	exploreUp(rootitem,rootitem,ID_EXPANDTREE);
	getApp()->endWaitCursor();
	
	return 1;
}


// Collapse the directory tree under cursor
long DirPanel::onCollapseTree(FXObject* sender,FXSelector sel,void* ptr)
{
    DirItem* rootitem=(DirItem*)list->getCurrentItem();
	getApp()->beginWaitCursor();
	exploreUp(rootitem,rootitem,ID_COLLAPSETREE);
	getApp()->endWaitCursor();
	
	return 1;
}


// Directory properties
long DirPanel::onCmdProperties(FXObject* sender,FXSelector,void* ptr)
{
    // Current item
	DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);
	
	PropertiesBox* attrdlg=new PropertiesBox(this,FXPath::name(pathname),FXPath::directory(pathname));	
   	if(attrdlg->execute(PLACEMENT_SCREEN))
        list->setDirectory(pathname,TRUE);
    delete attrdlg;
   	return 1;
}


// Add files or directory to an archive
long DirPanel::onCmdAddToArch(FXObject* o,FXSelector,void*)
{
	FXString ext1, ext2, cmd, archive;
	File *f;

	// Name and path of the current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString name=list->getItemText(item);
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXPath::directory(pathname);
 
    // Initial archive name with full path and default extension
    archive=parentdir+PATHSEPSTRING+name+".tar.gz";

	// Archive dialog
	if (archdialog==NULL)
		archdialog=new ArchInputDialog(this,"");
	archdialog->setText(archive);
	archdialog->CursorEnd();

    if(archdialog->execute())
    {
        // Get string and preserve escape characters
		archive="\""+archdialog->getText()+"\"";

        // Get extensions of the archive name
        ext1=archdialog->getText().rafter('.',1);
		ext1.lower();
        ext2=archdialog->getText().rafter('.',2);
		ext2.lower();
		
		// Handle different archive formats
		if (ext2=="tar.gz")
            cmd="tar -zcvf "+archive+" ";
		else if (ext2=="tar.bz2")
            cmd="tar -jcvf "+archive+" ";
		else if (ext2=="tar.z")
           	cmd="tar -Zcvf "+archive+" ";
		else if (ext1=="tar")
           	cmd="tar -cvf "+archive+" ";
		else if (ext1=="gz")
			cmd="gzip -v ";				
		else if (ext1=="tgz")
            cmd="tar -zcvf "+archive+" ";
		else if (ext1=="bz2")
			cmd="bzip2 -v ";				
		else if (ext1=="tbz2")
            cmd="tar -jcvf "+archive+" ";
		else if (ext1=="z")
            cmd="compress -v ";
		else if (ext1=="zip")
            cmd="zip -r "+archive+" ";
		else if (ext1=="rar")
            cmd="rar a "+archive+" ";
		else if (ext1=="lzh")
            cmd="lha a "+archive+" ";
		
		// Default archive format
		else
		{
			archive += ".tar.gz";
			cmd="tar -zcvf "+archive+" ";
		}
		
		// Archive command name
		cmd=cmd+"\""+name+"\"";
		
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
		f=new File(this,_("Create archive"),ARCHIVE);
		f->create();

		// Create archive
		chdir(parentdir.text());
		f->archive(archive,cmd);
 
		getApp()->endWaitCursor();
		delete f; 

		// Display parent directory in DirList and FileList
		list->setDirectory(parentdir,TRUE);
		((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
		((XFileExplorer*) mainWindow)->lpanel->current->updatePathLinker();
    }
    return 1;
}


// We now really do have the clipboard, keep clipboard content
long DirPanel::onClipboardGained(FXObject* sender,FXSelector sel,void* ptr)
{
    FXVerticalFrame::onClipboardGained(sender,sel,ptr);
    return 1;
}


// We lost the clipboard, free clipboard content
long DirPanel::onClipboardLost(FXObject* sender,FXSelector sel,void* ptr)
{
    FXVerticalFrame::onClipboardLost(sender,sel,ptr);
	return 1;
}


// Somebody wants our clipboard content
long DirPanel::onClipboardRequest(FXObject* sender,FXSelector sel,void* ptr)
{
    FXEvent *event=(FXEvent*)ptr;
    FXuchar *data;
    FXuint len;

	// Perhaps the target wants to supply its own data for the clipboard
    if(FXVerticalFrame::onClipboardRequest(sender,sel,ptr))
		return 1;

    // Clipboard target is xfelistType (Xfe, Gnome or XFCE)
	if(event->target==xfelistType)
    {	
		// Don't modify the clipboard if we are called from updPaste()
		if (!clipboard_locked)
		{
			// Prepend "copy" or "cut" as in the Gnome way and avoid duplicating these strings
			if (clipboard.find("copy\n")<0 && clipboard.find("cut\n")<0)
			{
				if (clipboard_type==CUT_CLIPBOARD)
					clipboard = "cut\n" + clipboard;
				else
					clipboard = "copy\n" + clipboard;
			}
		
			// Remove the '\r' because Nautilus won't work with it
			// No effect on other apps (thunar, xfe, ...)
			FXint pos;
			while ((pos=clipboard.find("\r"))>0)
					clipboard.erase(pos);
		}
		
        // Return clipboard content
        if(event->target==xfelistType)
        {
			if(!clipboard.empty())
			{
				len=clipboard.length();
				FXMEMDUP(&data,clipboard.text(),FXuchar,len);
				setDNDData(FROM_CLIPBOARD,event->target,data,len);
				
				// Return because xfelistType is not compatible with other types
				return 1;
			}
		}
    }

    // Clipboard target is kdelisType (KDE)
    if(event->target==kdelistType)
    {
		// The only data to be passed in this case is "0" for copy and "1" for cut
		// The uri data are passed using the standard uri-list type
		FXString flag;
		if (clipboard_type==CUT_CLIPBOARD)
			flag = "1";
		else
			flag = "0";

        // Return clipboard content
        if(event->target==kdelistType)
        {
			FXMEMDUP(&data,flag.text(),FXuchar,1);
			setDNDData(FROM_CLIPBOARD,event->target,data,1);
		}
    }

    // Clipboard target is urilistType (KDE apps ; non Gnome, non XFCE and non Xfe apps)  
    if(event->target==urilistType)
    {
		if(!clipboard.empty())
		{
			len=clipboard.length();
			FXMEMDUP(&data,clipboard.text(),FXuchar,len);
			setDNDData(FROM_CLIPBOARD,event->target,data,len);
		
			return 1;
		}
    }

    // Clipboard target is utf8Type (to paste file name to the file dialogs)  
	// Note that we only keep the first file name of the clipboard
    if(event->target==utf8Type)
    {
		if(!clipboard.empty())
		{
			FXint beg, end;
			FXString filename, url;

			// Clipboard don't contain copy or cut as first line
			if (clipboard.find("copy\n")<0 && clipboard.find("cut\n")<0)
			{
				end=clipboard.find("\n",0);
				url=clipboard.mid(0,end-1);
				filename=FXPath::name(FXURL::decode(FXURL::fileFromURL(url)));
			}
			
			// Clipboard contains copy or cut as first line, thus skip it
			else
			{
				beg=clipboard.find("\n",0);
				end=clipboard.find("\n",beg+1);
				url=clipboard.mid(beg+1,end-beg-1);
				filename=FXPath::name(FXURL::decode(FXURL::fileFromURL(url)));				
			}
			
			if(!filename.empty())
			{
				len=filename.length();
				FXMEMDUP(&data,filename.text(),FXuchar,len);
				setDNDData(FROM_CLIPBOARD,event->target,data,len);
			
				return 1;
			}
		}
    }
    return 0;
}


// Copy or cut to clipboard
long DirPanel::onCmdCopyCut(FXObject*,FXSelector sel,void*)
{			
	// Clear clipboard
	clipboard.clear();

    // Clipboard type
	if (FXSELID(sel)==ID_CUT_CLIPBOARD)
		clipboard_type=CUT_CLIPBOARD;
	else
		clipboard_type=COPY_CLIPBOARD;

	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);

	clipboard=FXURL::encode(FXURL::fileToURL(pathname))+"\r\n";
		
	// Acquire the clipboard
	FXDragType types[4];
	types[0]=xfelistType;
	types[1]=kdelistType;
	types[2]=urilistType;
	types[3]=utf8Type;
	if (acquireClipboard(types,4))
		return 0;
    
	return 1;
}


// Paste file(s) from clipboard
long DirPanel::onCmdPaste(FXObject*,FXSelector sel,void*)
{
    FXuchar *data;
    FXuint len;
	FXint beg, end, pos;
	FXString chaine, url, param;		  
	FXint num=0;
	FXbool from_kde=FALSE;
	
    // Target directory
	FXString targetdir=((XFileExplorer*) mainWindow)->lpanel->current->list->getDirectory();
	
	// If source is xfelistType (Gnome, XFCE, or Xfe app)
    if(getDNDData(FROM_CLIPBOARD,xfelistType,data,len))
    {		
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		
		clipboard=(FXchar*)data;
		
		// Loop over clipboard items
		for(beg=0; beg<clipboard.length(); beg=end+1)
		{
			if((end=clipboard.find("\n",beg))<0)
				end=clipboard.length();
			
			// Obtain item url
			url=clipboard.mid(beg,end-beg);
				
			// Eventually remove the trailing "\r" if any
			if ((pos=url.rfind("\r"))>0)
				url.erase(pos);

			// Process first item
			if (num==0)
			{
				// First item should be "copy" or "cut"
				if (url=="copy")
				{
					clipboard_type=COPY_CLIPBOARD;
					num++;
				}
				else if (url=="cut")
				{
					clipboard_type=CUT_CLIPBOARD;
					num++;
				}
				
				// If first item is not "copy" nor "cut", process it as a normal url
				// and use default clipboard type
				else
				{											
					// Update the param string
					param += FXURL::decode(FXURL::fileFromURL(url)) + "\n";
						
					// Add one more because the first line "copy" or "cut" was not present
					num += 2;
				}
			}
			
			// Process other items
			else
			{
				// Update the param string
				param += FXURL::decode(FXURL::fileFromURL(url)) + "\n";
				num++;
			}
		}

		// Construct the final param string passed to the file management routine
		param = targetdir+"\n" + FXStringVal(num-1) + "\n" + param;
		
		// Copy or cut operation depending on the clipboard type
		switch(clipboard_type)
		{
		case COPY_CLIPBOARD:
			sel=FXSEL(SEL_COMMAND,DirPanel::ID_DIR_COPY);
			break;
		case CUT_CLIPBOARD:
			clipboard.clear();
			sel=FXSEL(SEL_COMMAND,DirPanel::ID_DIR_CUT);
			break;
		}
		paste=TRUE;
		this->handle(this,sel,strdup(param.text()));
    		
        // Free data pointer
		FXFREE(&data);
		
		// Return here because xfelistType is not compatible with other types
		return 1;
	}

	// If source type is kdelistType (KDE)
    if(getDNDData(FROM_CLIPBOARD,kdelistType,data,len))
    {		
        from_kde=TRUE;

        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		clipboard=(FXchar*)data;
				
		// Obtain clipboard type (copy or cut)
		if (clipboard=="1")
			clipboard_type=CUT_CLIPBOARD;
		else
			clipboard_type=COPY_CLIPBOARD;
		
		FXFREE(&data);
	}
	
	
	// If source type is urilistType (KDE apps ; non Gnome, non XFCE and non Xfe apps)
    if(getDNDData(FROM_CLIPBOARD,urilistType,data,len))
    {		
		// For non KDE apps, set action to copy
		if (!from_kde)
			clipboard_type=COPY_CLIPBOARD;

        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		clipboard=(FXchar*)data;
				
		// Loop over clipboard items
		for(beg=0; beg<clipboard.length(); beg=end+1)
		{
			if((end=clipboard.find("\n",beg))<0)
				end=clipboard.length();

			// Obtain item url
			url=clipboard.mid(beg,end-beg);

			// Eventually remove the trailing "\r" if any
			if ((pos=url.rfind("\r"))>0)
				url.erase(pos);

			// Update the param string
			param += FXURL::decode(FXURL::fileFromURL(url)) + "\n";
			num++;
		}

		// Construct the final param string passed to the file management routine
		param = targetdir+"\n" + FXStringVal(num) + "\n" + param;
				
		// Copy or cut operation depending on the clipboard type
		switch(clipboard_type)
		{
		case COPY_CLIPBOARD:
			sel=FXSEL(SEL_COMMAND,DirPanel::ID_DIR_COPY);
			break;
		case CUT_CLIPBOARD:
			clipboard.clear();
			sel=FXSEL(SEL_COMMAND,DirPanel::ID_DIR_CUT);
			break;
		}
		paste=TRUE;
		handle(this,sel,strdup(param.text()));
					
		FXFREE(&data);
		return 1;
	}
	return 0;
}


// Copy/Move/Rename/Symlink directory
long DirPanel::onCmdDirMan(FXObject* sender,FXSelector sel,void* ptr)
{
	FXint num;
	FXString src, targetdir, target, name, source;

	// Confirmation dialog?
	FXbool ask_before_copy=getApp()->reg().readUnsignedEntry("OPTIONS","ask_before_copy",TRUE);

	// If we are we called from the paste command, get the parameters from the pointer
	// Multiple sources are allowed
	if (paste)
	{
		// Reset the flag
		paste=FALSE;

		// Get the parameters
		FXString str=(char*)ptr;
		targetdir=str.section('\n',0);
		num=FXUIntVal(str.section('\n',1));
		src=str.after('\n',2);
		source=src.section('\n',0);

		// If no item, return
		if(num<=0)
			return 0;
	}
	
	// Obtain the parameters from the dir panel (only one source)
	else
	{
		// Current item
		DirItem* item=(DirItem*)list->getCurrentItem();
		
		// Number of items
		if (item)
			num=1;
		else
			return 0;
    
		// Source directory
		source=list->getItemPathname((TreeItem*)item);	

		// Target directory
		targetdir=FXPath::directory(source);	
	}

	// Go to target directory
	chdir(targetdir.text());
	
	// Name and directory of the first source file
	name=FXPath::name(source);
	FXString dir=FXPath::directory(source);

	// Initialise target name	
	if (targetdir!=ROOTDIR)
		target=targetdir+PATHSEPSTRING;
	else
		target=targetdir;

    // Configure the command, title, message, etc.
    FXIcon *icon=NULL;
	FXString command, title, message;
	if (FXSELID(sel)==ID_DIR_COPY)
    {
        command="copy";
        title=_("Copy");
        icon=copy_bigicon;
        if(num==1)
        {
            message=_("Copy ");
            message+=source;
		}
        else
            message=_("Copy ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_DIR_RENAME)
    {
        command="rename";
        title=_("Rename");
        icon=move_bigicon;
        if(num==1)
        {
            message=_("Rename ");
			message+=name;
            target=name;
			title=_("Rename");
        }
        else
			return 0;
    }
	if (FXSELID(sel)==ID_DIR_COPYTO)
    {
        command="copy";
        title=_("Copy to");
        icon=copy_bigicon;
        if(num==1)
        {
            message=_("Copy ");
            message+=source;
		}
        else
            message=_("Copy ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_DIR_MOVE)
    {
        command="move";
        title=_("Move");
        icon=move_bigicon;
        if(num==1)
        {
            message=_("Move ");
            message+=source;
            title=_("Move");
        }
        else
            message=_("Move ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_DIR_CUT)
    {
        command="move";
        title=_("Move");
        icon=move_bigicon;
        if(num==1)
        {
            message=_("Move ");
            message+=source;
            title=_("Move");
        }
        else
            message=_("Move ")+FXStringVal(num)+_(" items from: ")+dir;
    }
    if (FXSELID(sel)==ID_DIR_SYMLINK)
    {
        command="symlink";
        title=_("Symlink");
        icon=link_bigicon;
        if(num==1)
        {
            message=_("Symlink ");
            message+=source;
        }
        else
            message=_("Symlink ")+FXStringVal(num)+_(" items from: ")+dir;
    }
	
    // File operation dialog, if needed
	if (ask_before_copy || source==target ||  FXSELID(sel)==ID_DIR_COPYTO || FXSELID(sel)==ID_DIR_MOVE ||  FXSELID(sel)==ID_DIR_RENAME || FXSELID(sel)==ID_DIR_SYMLINK)
	{   
		if (operationdialog==NULL)
			operationdialog=new BrowseInputDialog(this,"","",title,_("To:"),icon);
		operationdialog->setTitle(title);
		operationdialog->setMessage(message);
		operationdialog->setText(target);
		operationdialog->CursorEnd();
		int rc=1;
		rc=operationdialog->execute(PLACEMENT_CURSOR);
		target=operationdialog->getText();
		if (!rc)
			return 0;
	}
	
	// Update target and target parent directory
	target=::filePath(target);
	if (!::isDirectory(target))
		targetdir=FXPath::directory(target);
	else
		targetdir=target;	

    // Target directory not writable
    if(!::isWritable(targetdir))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"), _("Can't write to %s : Permission denied"), targetdir.text());
        return 0;
    }

	// Multiple sources and non existent destination
	if (num>1 & !::exists(target))
	{
        MessageBox::error(this,BOX_OK,_("Error"),_("Folder %s does'nt exist"),target.text());
        return 0;
	}
		
	// Multiple sources and target is a file
	if (num>1 & ::isFile(target))
	{ 			
        MessageBox::error(this,BOX_OK,_("Error"),_("%s is not a folder"),target.text());
        return 0;
	}

	// Target is a directory and is not writable
    if (::isDirectory(target) & !::isWritable(target))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s : Permission denied"), target.text());
        return 0;
    }
	
	// Target is a file and its parent directory is not writable
	if (::isFile(target) & !::isWritable(targetdir))
	{
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s : Permission denied"),targetdir.text());
        return 0;
	}

	// Target parent directory doesn't exist
	if (!::exists(targetdir))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("Folder %s does'nt exist"),targetdir.text());
        return 0;
	}
		
	// Target parent directory is not a directory
	if (!::isDirectory(targetdir))
	{
		MessageBox::error(this,BOX_OK,_("Error"),_("%s is not a folder"),targetdir.text());
        return 0;
	}
		
	// One source
	File *f=NULL;
	FXbool ret;
    if(num==1)
	{
		// An empty source file name corresponds to the ".." file
		// Don't perform any file operation on it!
		if (source=="")
			return 0;
		
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
        if(command=="copy")
        {
			f=new File(this,_("File copy"),COPY);
        	f->create();
			ret=f->copy(source,target);
			
			// An error has occurred
			if (ret==0 && !f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the copy file operation!"));
			}
					
			// If action is cancelled in progress dialog
			if (f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("Copy file operation cancelled!"));
			}
        }
        else if(command=="rename")
        {
			f=new File(this,_("File rename"),RENAME);
        	f->create();
			f->rename(source,target);
        }
        else if(command=="move")
        {
			f=new File(this,_("File move"),MOVE);
        	f->create();
			ret=f->move(source,target);

			// An error has occurred
			if (ret==0 && !f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move file operation!"));
			}
					
			// If action is cancelled in progress dialog
			if (f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("Move file operation cancelled!"));
			}
        }
        else if(command=="symlink")
        {
			f=new File(this,_("Symlink"),SYMLINK);
        	f->create();
			f->symlink(source,target);
		}
       	else
           	exit(-1);
									
		getApp()->endWaitCursor();
		delete f;
	}
	
	// Multiple sources
	// Note : rename cannot be used in this case!
    else if(num>1)
    {
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
        if (command=="copy")
			f=new File(this,_("File copy"),COPY);
		else if (command=="move")
			f=new File(this,_("File move"),MOVE);
		else if  (command=="symlink")
			f=new File(this,_("Symlink"),SYMLINK);
			
		f->create();

		// Loop on the multiple files
		for(int i=0;i<num;i++)
        {
			// Individual source file
			source=src.section('\n',i);

			// An empty file name corresponds to the ".." file (why?)
			// Don't perform any file operation on it! 
			if (source!="")
			{
            	if(command=="copy")
				{
					ret=f->copy(source,target);
					
					// An error has occurred
					if (ret==0 && !f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the copy file operation!"));
						break;
					}
					
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Copy file operation cancelled!"));
						break;
					}
				}
            	else if(command=="move")
            	{
					ret=f->move(source,target);
				
					// An error has occurred
					if (ret==0 && !f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move file operation!"));
						break;
					}
					
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Move file operation cancelled!"));
						break;
					}
           	 	}
				else if (command=="symlink")
				{
					ret=f->symlink(source,target);
					
					// An error has occurred
					if (ret==0 && !f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the symlink operation!"));
						break;
					}
					
					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,BOX_OK,_("Error"),_("Symlink operation cancelled!"));
						break;
					}
				}
            	else
                	exit(-1);
			}
		}
		
		getApp()->endWaitCursor();		
		delete f;
	}	
	
	// Force dirpanel refresh
	list->handle(this,FXSEL(SEL_COMMAND,DirList::ID_REFRESH),NULL);
    return 1;
}


// Delete directory
long DirPanel::onCmdDirDelete(FXObject*,FXSelector,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXPath::directory(pathname);	

    // If we don't have permission to write to the parent directory
    if(!::isWritable(parentdir))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"), parentdir.text());
        return 0;
    }

    FXbool confirm_del=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_delete",TRUE);
	if(confirm_del)
    {
		FXString message=_("Definitively delete folder ") + pathname + " ?";
		MessageBox box(this,_("Confirm Delete"),message,delete_big_permicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        if(box.execute(PLACEMENT_CURSOR) != BOX_CLICKED_OK)
            return 0;
    }

	// File object
   	File* f=new File(this,_("File delete"),DELETE);
   	f->create();

    // If we don't have permission to write to the directory
 	if (!::isWritable(pathname))
    {
		// Dialog to confirm directory deletion
		f->allowPdialog=FALSE;
		f->hide();
		FXString str=_("Folder ") + pathname + _(" is write-protected, definitively delete it anyway?");
		MessageBox box(this,_("Confirm Delete"),str,errorbigicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);        	
		FXuint answer=box.execute(PLACEMENT_SCREEN);
		if(answer == BOX_CLICKED_OK)
		{
			f->remove(pathname); 	

			// If action is cancelled in progress dialog
			if (f->isCancelled)
				MessageBox::error(this,BOX_OK,_("Error"),_("Delete folder operation cancelled!"));
			
			// Return to parent directory in DirList and FileList
			list->setDirectory(parentdir,TRUE);
			((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
			((XFileExplorer*) mainWindow)->lpanel->current->updatePathLinker();
		}
	}

	// If we have permission to write
	else
	{
		f->remove(pathname); 	

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,BOX_OK,_("Error"),_("Delete folder operation cancelled!"));
		}
		// Return to parent directory in DirList and FileList
		list->setDirectory(parentdir,TRUE);
		((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
		((XFileExplorer*) mainWindow)->lpanel->current->updatePathLinker();
	}

	delete f;

	// Force DirPanel and FilePanel refresh
 	//mainWindow->handle(this,FXSEL(SEL_COMMAND,XFileExplorer::ID_REFRESH),NULL);
	list->handle(this,FXSEL(SEL_COMMAND,DirList::ID_REFRESH),NULL);
    return 1;
}


// Move directory to trash can
long DirPanel::onCmdDirTrash(FXObject*,FXSelector,void*)
{
	FXbool ret;
	
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString pathname=list->getItemPathname((TreeItem*)item);	
	FXString parentdir=FXPath::directory(pathname);	

    // If we don't have permission to write to the parent directory
    if(!::isWritable(parentdir))
    {
        MessageBox::error(this,BOX_OK_SU,_("Error"),_("Can't write to %s: Permission denied"), parentdir.text());
        return 0;
    }

    FXbool confirm_del=getApp()->reg().readUnsignedEntry("OPTIONS","confirm_delete",TRUE);
	if(confirm_del)
    {
		FXString message=_("Move folder ") + pathname + " to trash can?";
		MessageBox box(this,_("Confirm Trash"),message,delete_bigicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        if(box.execute(PLACEMENT_CURSOR) != BOX_CLICKED_OK)
            return 0;
    }

	// File object
   	File* f=new File(this,_("Move to trash"),DELETE);
   	f->create();

    // If we don't have permission to write to the directory
 	if (!::isWritable(pathname))
    {
		// Dialog to confirm directory deletion
		f->allowPdialog=FALSE;
		f->hide();
		FXString str=_("Folder ") + pathname + _(" is write-protected, move it to trash can anyway?");
		MessageBox box(this,_("Confirm Trash"),str,errorbigicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);        	
		FXuint answer=box.execute(PLACEMENT_SCREEN);
		if(answer == BOX_CLICKED_OK)
		{
			
			// Folder is in the trash can
			if (pathname.find(trashlocation.text())==0)
			{
				FXString message=_("Folder is already in the trash can! Definitively delete folder ") + pathname + " ?";
				MessageBox box(this,_("Delete from trash can"),message,delete_big_permicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
				if(box.execute(PLACEMENT_CURSOR) == BOX_CLICKED_OK)
					f->remove(pathname);
				f->allowPdialog=TRUE;
			}
			else
			{
				// Create trash can if it doesn't exist
				if (!exists(trashlocation))
					::mkdir(trashlocation.text(),0755);
				
				// Move folder to trash can
				ret=f->move(pathname,trashlocation+PATHSEPSTRING+stampname(pathname));
				
				// An error has occurred
				if (ret==0 && !f->isCancelled)
				{
					f->hide();
					MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move to trash operation!"));
				}
			}

			// If action is cancelled in progress dialog
			if (f->isCancelled)
				MessageBox::error(this,BOX_OK,_("Error"),_("Move to trash folder operation cancelled!"));
			
			// Return to parent directory in DirList and FileList
			list->setDirectory(parentdir,TRUE);
			((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
			((XFileExplorer*) mainWindow)->lpanel->current->updatePathLinker();
		}
	}

	// If we have permission to write
	else
	{
		// Folder is in the trash can
		if (pathname.find(trashlocation.text())==0)
		{
			f->allowPdialog=FALSE;
			f->hide();
			FXString message=_("Folder is already in the trash can! Definitively delete folder ") + pathname + " ?";
       		MessageBox box(this,_("Delete from trash can"),message,delete_big_permicon,BOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
			if(box.execute(PLACEMENT_CURSOR) == BOX_CLICKED_OK)
				f->remove(pathname);
			f->allowPdialog=TRUE;
		}
		else
		{
			// Create trash can if it doesn't exist
			if (!exists(trashlocation))
				::mkdir(trashlocation.text(),0755);
			ret=f->move(pathname,trashlocation+PATHSEPSTRING+stampname(pathname));

			// An error has occurred
			if (ret==0 && !f->isCancelled)
			{
				f->hide();
				MessageBox::error(this,BOX_OK,_("Error"),_("An error has occurred during the move to trash operation!"));
			}
		}

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,BOX_OK,_("Error"),_("Move to trash folder operation cancelled!"));
		}
		// Return to parent directory in DirList and FileList
		list->setDirectory(parentdir,TRUE);
		((XFileExplorer*) mainWindow)->lpanel->current->list->setDirectory(parentdir);
		((XFileExplorer*) mainWindow)->lpanel->current->updatePathLinker();
	}

	delete f;

	// Force DirPanel and FilePanel refresh
 	//mainWindow->handle(this,FXSEL(SEL_COMMAND,XFileExplorer::ID_REFRESH),NULL);
	list->handle(this,FXSEL(SEL_COMMAND,DirList::ID_REFRESH),NULL);
    return 1;
}


// Create new directory
long DirPanel::onCmdNewDir(FXObject*,FXSelector,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();

	FXString dirpath=list->getItemPathname((TreeItem*)item);
    if(dirpath!=ROOTDIR)
        dirpath+=PATHSEPSTRING;
    
    if (newdirdialog==NULL)
		newdirdialog=new InputDialog(this,"",_("Create new folder:"),_("New Folder"));
	newdirdialog->setText("");
    if(newdirdialog->execute(PLACEMENT_CURSOR))
    {
		FXString dirname=dirpath+newdirdialog->getText();		
        if(dirname!=dirpath)
        {
			// Create the new dir according to the current umask
			int mask;
			mask=umask(0);
			umask(mask);
			// note that the umask value is in decimal (511 means octal 0777)
			FXint ret=::mkdir(dirname.text(),511 & ~mask);
			if (ret==-1)
			{
         		MessageBox::error(this,BOX_OK_SU,_("Error"),"%s",strerror(errno));
    			return 0;
			}
		}							
    }
	
	// Force dirpanel refresh
	list->handle(this,FXSEL(SEL_COMMAND,DirList::ID_REFRESH),NULL);
    return 1;
}


// Run Terminal in the selected directory
long DirPanel::onCmdXTerm(FXObject*,FXSelector,void*)
{
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString buf=list->getItemPathname((TreeItem*)item);
	chdir(buf.text());
    FXString cmd=getApp()->reg().readStringEntry("PROGS","xterm","xterm -sb");
    cmd += " &";
    system(cmd.text());
    return 1;
}


#if defined(linux)
// Mount/Unmount directory
long DirPanel::onCmdMount(FXObject*,FXSelector sel,void*)
{
    FXString cmd, msg, text;
	unsigned int op;
	File *f;
	
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

	// If symbolic link
	if (::isLink(dir))
		dir=FXFile::symlink(dir);

    // Select the command and set the appropriate message
    if(FXSELID(sel)==ID_MOUNT)
    {
        op=MOUNT;
		msg=_("Mount");
        cmd="mount ";
    }
    else
    {
        op=UNMOUNT;
		msg=_("Unmount");
        cmd="umount ";
    }
    cmd+="'"+dir+"'";
    cmd+=" 2>&1";
    chdir(ROOTDIR);

	// Wait cursor
	getApp()->beginWaitCursor();

	// File object
	text=msg + _(" file system...");
	f=new File(this,text.text(),op);
	f->create();
				   
	// Mount/unmount file system
	text=msg + _(" the folder :");
	f->mount(dir,text,cmd,op);
 
	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		getApp()->endWaitCursor();
		f->hide();
		text=msg + _(" operation cancelled!");
		MessageBox::error(this,BOX_OK,_("Error"),text.text());
		delete f;
		return 0;
	}					
	
	getApp()->endWaitCursor();
	delete f; 

	// Force dirpanel refresh
	list->handle(this,FXSEL(SEL_COMMAND,DirList::ID_REFRESH),NULL);

    return 1;
}

// Update the Mount menu item
long DirPanel::onUpdMount(FXObject* o,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

    if(fsdevices->find(dir.text()) && !mtdevices->find(dir.text()))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Update the Unmount menu item
long DirPanel::onUpdUnmount(FXObject* o,FXSelector sel,void*)
{
	// Current item
    DirItem* item=(DirItem*)list->getCurrentItem();
	FXString dir=list->getItemPathname((TreeItem*)item);

    if(fsdevices->find(dir.text()) || mtdevices->find(dir.text()))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}
#endif


// Update the paste button
long DirPanel::onUpdPaste(FXObject* o,FXSelector,void*)
{
    FXuchar *data;
    FXuint len;	
	FXString buf;
	FXbool clipboard_empty=TRUE;
	
	// Lock clipboard to prevent changes in method onCmdRequestClipboard()
	clipboard_locked=TRUE;
	
	// If source is xfelistType (Gnome, XFCE, or Xfe app)
    if (getDNDData(FROM_CLIPBOARD,xfelistType,data,len))
    {		
        FXRESIZE(&data,FXuchar,len+1);
        data[len]='\0';
		buf=(FXchar*)data;
		
		// Check if valid clipboard
		if (buf.find("file:/")>=0)
			clipboard_empty=FALSE;
			
        // Free data pointer
		FXFREE(&data);
	}
		
	// If source type is urilistType (KDE apps ; non Gnome, non XFCE and non Xfe apps)
	else if (getDNDData(FROM_CLIPBOARD,urilistType,data,len))
	{		
		FXRESIZE(&data,FXuchar,len+1);
		data[len]='\0';
		buf=(FXchar*)data;
		
		// Test if valid clipboard
		if (buf.find("file:/")>=0)
			clipboard_empty=FALSE;
		
		// Free data pointer
		FXFREE(&data);
	}
		
	// Gray out the paste button, if necessary
	if (clipboard_empty)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);

	// Unlock clipboard
	clipboard_locked=FALSE;

    return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long DirPanel::onUpdMenu(FXObject* o,FXSelector,void*)
{
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
		return 0;
	
	// Path name of the selected item
	FXString dir=list->getItemPathname(item);	
    return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long DirPanel::onUpdDirDelete(FXObject* o,FXSelector,void*)
{
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
		return 0;
	
	// Path name of the selected item
	FXString dir=list->getItemPathname(item);	

 	FXbool use_trash_can=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_can",TRUE);
	FXbool use_trash_bypass=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_bypass",FALSE);
	if (!use_trash_can | use_trash_bypass)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

	return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long DirPanel::onUpdDirTrash(FXObject* o,FXSelector,void*)
{
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
		return 0;
	
	// Path name of the selected item
	FXString dir=list->getItemPathname(item);	

	FXbool use_trash_can=getApp()->reg().readUnsignedEntry("OPTIONS","use_trash_can",TRUE);
	if (use_trash_can)
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

	return 1;
}

// Refresh the directory size in the status bar
long DirPanel::onCmdDirsizeRefresh(FXObject* sender,FXSelector,void*)
{
	unsigned long long dsize;
	char size[64];
	FXString hsize;
	
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
	{
		status->setText(_("0 bytes in root"));
		return 0;
	}
	
	// Path name of the selected item (without trailing '/' except for the root path)
	FXString path=::filePath(list->getItemPathname(item),"");

	// Size of the files present in the directory
	//strlcpy(buf,path.text(),path.length()+1);
	dsize=::dirsize(path.text());

	// Size in human readable form
	snprintf(size,sizeof(size)-1,"%llu",dsize);
	hsize=::hSize(size);

	// Refresh the status label
	FXString string = hsize +  _(" in root");
	status->setText(string);

    // Reset timer again
    getApp()->addTimeout(this,ID_DIRSIZE_REFRESH,DIRSIZE_INTERVAL);

    return 1;
}


// Update the path name in the Window title
long DirPanel::onUpdTitle(FXObject* sender,FXSelector,void*)
{
	
	// Name of the current selected item
	TreeItem* item=(TreeItem*)list->getCurrentItem();
	
	// There is no selected item
	if (item==NULL)
	{
		mainWindow->setTitle("Xfe - ");
		return 0;
	}
	
	// Path name of the selected item
	FXString path=list->getItemPathname(item);	

	// Update the path in the window title
	if(getuid()==0)
		mainWindow->setTitle("Xfe (root) - " + path);
	else
		mainWindow->setTitle("Xfe - " + path);
    return 1;
}
