/*
 *	gadgets.c - gadgets implementation
 *	Copyright (C) 2000 Fred Barnes <frmb2@ukc.ac.uk>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/keysymdef.h>
#include <X11/xpm.h>

#include "support.h"
#include "cfg.h"
#include "interface.h"
#include "error.h"

#define MAX_LOOKUP_STRING (64)

/* structures and stuff */
typedef struct {
	int text_x, text_y;
	char *text;
	int exposed;
	int len;
} LabelPrivate;

typedef struct {
	int text_x, text_y;
	char *text;
	int text_width;
	int cur_len, max_len, cur_pos;
	unsigned long te_pixel;
	int exposed;
	void (*edit_accept)(Gadget *, char *);
	void (*edit_escape)(Gadget *, char *);
	void (*edit_ctrlc)(Gadget *, char *);
} EditPrivate;

typedef struct {
	int text_x, text_y;
	char *text;
	int len;
	int mouse_in, mouse_down;
	int exposed;
	void (*button_press)(Gadget *);
} ButtonPrivate;

/* prototypes */
static Gadget *create_gadget (MyXContext *cxt, char *name, int x, int y, int width, int height);
static void destroy_gadget (Gadget *gadget);
static void allocate_color (MyXContext *cxt, char *colordef, unsigned long *xpixel);
static void panel_expose_event (XEvent *ev, Gadget *gadget);
static void label_expose_event (XEvent *ev, Gadget *gadget);
static void label_show_action (Gadget *gadget);
static void label_hide_action (Gadget *gadget);
static void label_settext_action (Gadget *gadget, char *text);
static void edit_expose_event (XEvent *ev, Gadget *gadget);
static void edit_keypress_event (XEvent *ev, Gadget *gadget);
static void button_expose_event (XEvent *ev, Gadget *gadget);
static void button_mousedown_event (XEvent *ev, Gadget *gadget);
static void button_mouseup_event (XEvent *ev, Gadget *gadget);
static void button_mouseenter_event (XEvent *ev, Gadget *gadget);
static void button_mouseleave_event (XEvent *ev, Gadget *gadget);


/*
 *	Gadget *create_gadget (MyXContext *cxt, char *name, int x, int y, int width, int height)
 *	Creates a new gadget
 */
static Gadget *create_gadget (MyXContext *cxt, char *name, int x, int y, int width, int height)
{
	Gadget *gadget;

	gadget = (Gadget *)xmalloc (sizeof (Gadget));
	gadget->window = None;
	gadget->name = string_dup (name);
	gadget->config = (GadgetConfigData *)xmalloc (sizeof (GadgetConfigData));
	gadget->fg_pixel = BlackPixel (cxt->display, cxt->screen);
	gadget->bg_pixel = BlackPixel (cxt->display, cxt->screen);
	gadget->hi_pixel = gadget->fg_pixel;
	gadget->sh_pixel = gadget->bg_pixel;
	gadget->bg_pixmap = None;
	gadget->x = x;
	gadget->y = y;
	gadget->width = width;
	gadget->height = height;
	gadget->visible = 1;
	gadget->fontinfo = NULL;
	/* gadget->gc */
	gadget->expose_event = NULL;
	gadget->mousedown_event = NULL;
	gadget->mouseup_event = NULL;
	gadget->keypress_event = NULL;
	gadget->mouseenter_event = NULL;
	gadget->mouseleave_event = NULL;
	gadget->show_action = NULL;
	gadget->hide_action = NULL;
	gadget->settext_action = NULL;
	gadget->xcxt = cxt;
	return gadget;
}


/*
 *	void destroy_gadget (Gadget *gadget)
 *	Destroys a gadget
 */
static void destroy_gadget (Gadget *gadget)
{
	if (gadget->config) {
		free (gadget->config);
	}
	if (gadget->name) {
		free (gadget->name);
	}
	free (gadget);
	return;
}


/*
 *	Gadget *create_toplevel (MyXContext *cxt, char *name, int x, int y, int width, int height)
 *	Creates a new top-level gadget
 */
Gadget *create_toplevel (MyXContext *cxt, char *name, int x, int y, int width, int height)
{
	Gadget *gadget;

	gadget = create_gadget (cxt, name, x, y, width, height);
	if (read_config_entry (name, gadget->config, FLD_BACKGROUND)) {
		return NULL;
	}
	allocate_color (cxt, gadget->config->background, &gadget->bg_pixel);
	gadget->window = XCreateSimpleWindow (cxt->display, RootWindow (cxt->display, cxt->screen),
		x, y, width, height, 0, BlackPixel (cxt->display, cxt->screen), gadget->bg_pixel);
	XMapWindow (cxt->display, gadget->window);
	return gadget;
}


/*
 *	void destroy_toplevel (Gadget *gadget)
 *	Destroys a top-level gadget
 */
void destroy_toplevel (Gadget *gadget)
{
	XUnmapWindow (gadget->xcxt->display, gadget->window);
	XDestroyWindow (gadget->xcxt->display, gadget->window);
	destroy_gadget (gadget);
	return;
}


/*
 *	Gadget *create_panel (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height)
 *	Creates a new panel gadget
 */
Gadget *create_panel (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height)
{
	Gadget *gadget;
	XGCValues gc_values;
	int use_pixmap, i;

	gadget = create_gadget (cxt, name, x, y, width, height);
	if (read_config_entry (name, gadget->config, FLD_BACKGROUND | FLD_HIGHLIGHT | FLD_SHADOW)) {
		return NULL;
	}
	/* also allowing a background pixmap on this one */
	if (access (gadget->config->background, F_OK) == 0) {
		/* guess it's a filename then */
		gadget->bg_pixel = BlackPixel (cxt->display, cxt->screen);
		use_pixmap = 1;
	} else if (!strcmp (gadget->config->background, "ParentRelative")) {
		gadget->bg_pixel = BlackPixel (cxt->display, cxt->screen);
		gadget->bg_pixmap = ParentRelative;
		use_pixmap = 1;
	} else {
		allocate_color (cxt, gadget->config->background, &gadget->bg_pixel);
		use_pixmap = 0;
	}
	allocate_color (cxt, gadget->config->highlight, &gadget->hi_pixel);
	allocate_color (cxt, gadget->config->shadow, &gadget->sh_pixel);
	gadget->window = XCreateSimpleWindow (cxt->display, parent->window, x, y, width, height, 0, BlackPixel (cxt->display, cxt->screen), gadget->bg_pixel);
	gc_values.background = gadget->bg_pixel;
	gadget->gc = XCreateGC (cxt->display, gadget->window, GCBackground, &gc_values);
	XSelectInput (cxt->display, gadget->window, ExposureMask);
	XMapWindow (cxt->display, gadget->window);
	gadget->expose_event = panel_expose_event;
	if (use_pixmap) {
		if (gadget->bg_pixmap == None) {
			i = XpmReadFileToPixmap (cxt->display, RootWindow (cxt->display, cxt->screen), gadget->config->background, &gadget->bg_pixmap, NULL, NULL);
			if (i != XpmSuccess) {
				gadget->bg_pixmap = None;
			}
		}
		if (gadget->bg_pixmap != None) {
			XSetWindowBackgroundPixmap (cxt->display, gadget->window, gadget->bg_pixmap);
		}
	}
	return gadget;
}


/*
 *	void destroy_panel (Gadget *gadget)
 *	Destroys a panel gadget
 */
void destroy_panel (Gadget *gadget)
{
	XUnmapWindow (gadget->xcxt->display, gadget->window);
	XDestroyWindow (gadget->xcxt->display, gadget->window);
	XFreeGC (gadget->xcxt->display, gadget->gc);
	if ((gadget->bg_pixmap != None) && (gadget->bg_pixmap != ParentRelative)) {
		XFreePixmap (gadget->xcxt->display, gadget->bg_pixmap);
		gadget->bg_pixmap = None;
	}
	destroy_gadget (gadget);
	return;
}


/*
 *	Gadget *create_label (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height, char *text)
 *	Creates a new label gadget
 */
Gadget *create_label (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height, char *text)
{
	Gadget *gadget;
	XGCValues gc_values;
	LabelPrivate *pdata;
	int parent_pixmap;

	gadget = create_gadget (cxt, name, x, y, width, height);
	if (read_config_entry (name, gadget->config, FLD_FOREGROUND | FLD_BACKGROUND | FLD_FONT | FLD_TEXT)) {
		return NULL;
	}
	allocate_color (cxt, gadget->config->foreground, &gadget->fg_pixel);
	if (parent->bg_pixmap != None) {
		parent_pixmap = 1;
		gadget->bg_pixel = BlackPixel (cxt->display, cxt->screen);
	} else {
		allocate_color (cxt, gadget->config->background, &gadget->bg_pixel);
		parent_pixmap = 0;
	}
	gadget->window = XCreateSimpleWindow (cxt->display, parent->window, x, y, width, height, 0, BlackPixel (cxt->display, cxt->screen), gadget->bg_pixel);
	gadget->fontinfo = XLoadQueryFont (cxt->display, gadget->config->font);
	if (!gadget->fontinfo) {
		gadget->fontinfo = XLoadQueryFont (cxt->display, "fixed");
		if (!gadget->fontinfo) {
			return NULL;
		}
	}
	gc_values.foreground = gadget->fg_pixel;
	gc_values.background = gadget->bg_pixel;
	gc_values.font = gadget->fontinfo->fid;
	gadget->gc = XCreateGC (cxt->display, gadget->window, GCForeground | GCBackground | GCFont, &gc_values);
	pdata = (LabelPrivate *)xmalloc (sizeof (LabelPrivate));
	pdata->text_x = (gadget->width >> 1) - (XTextWidth (gadget->fontinfo, gadget->config->text, strlen (gadget->config->text)) >> 1);
	pdata->text_y = (gadget->height >> 1) + (gadget->fontinfo->ascent >> 1);
	pdata->text = gadget->config->text;
	pdata->len = strlen (pdata->text);
	pdata->exposed = 0;
	gadget->pdata = (void *)pdata;
	XSelectInput (cxt->display, gadget->window, ExposureMask);
	XMapWindow (cxt->display, gadget->window);
	gadget->expose_event = label_expose_event;
	gadget->show_action = label_show_action;
	gadget->hide_action = label_hide_action;
	gadget->settext_action = label_settext_action;
	if (parent_pixmap) {
		XSetWindowBackgroundPixmap (cxt->display, gadget->window, ParentRelative);
	}
	return gadget;
}


/*
 *	void destroy_label (Gadget *gadget)
 *	Destroys a label gadget
 */
void destroy_label (Gadget *gadget)
{
	XUnmapWindow (gadget->xcxt->display, gadget->window);
	XDestroyWindow (gadget->xcxt->display, gadget->window);
	XFreeGC (gadget->xcxt->display, gadget->gc);
	XFreeFont (gadget->xcxt->display, gadget->fontinfo);
	free (gadget->pdata);
	destroy_gadget (gadget);
	return;
}


/*
 *	Gadget *create_edit (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height, int text_max, void (*accept_fcn)(Gadget *, char*))
 *	Creates a new edit gadget
 */
Gadget *create_edit (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height, int text_max, void (*accept_fcn)(Gadget *, char *), void (*escape_fcn)(Gadget *, char *), void (*ctrlc_fcn)(Gadget *, char *))
{
	Gadget *gadget;
	XGCValues gc_values;
	EditPrivate *pdata;

	gadget = create_gadget (cxt, name, x, y, width, height);
	if (read_config_entry (name, gadget->config, FLD_FOREGROUND | FLD_HIGHLIGHT | FLD_SHADOW | FLD_FONT | FLD_TENTRY)) {
		return NULL;
	}
	pdata = (EditPrivate *)xmalloc (sizeof (EditPrivate));
	allocate_color (cxt, gadget->config->foreground, &gadget->fg_pixel);
	allocate_color (cxt, gadget->config->tentry, &pdata->te_pixel);
	allocate_color (cxt, gadget->config->highlight, &gadget->hi_pixel);
	allocate_color (cxt, gadget->config->shadow, &gadget->sh_pixel);
	gadget->window = XCreateSimpleWindow (cxt->display, parent->window, x, y, width, height, 0, BlackPixel (cxt->display, cxt->screen), pdata->te_pixel);
	gadget->fontinfo = XLoadQueryFont (cxt->display, gadget->config->font);
	if (!gadget->fontinfo) {
		gadget->fontinfo = XLoadQueryFont (cxt->display, "fixed");
		if (!gadget->fontinfo) {
			return NULL;
		}
	}
	gc_values.background = pdata->te_pixel;
	gc_values.font = gadget->fontinfo->fid;
	gadget->gc = XCreateGC (cxt->display, gadget->window, GCBackground | GCFont, &gc_values);
	pdata->text_x = 4;
	pdata->text_y = (gadget->height >> 1) + (gadget->fontinfo->ascent >> 1);
	pdata->text = xmalloc (text_max+1);
	pdata->cur_len = 0;
	pdata->cur_pos = 0;
	pdata->max_len = text_max;
	pdata->exposed = 0;
	pdata->text_width = 0;
	pdata->edit_accept = accept_fcn;
	pdata->edit_escape = escape_fcn;
	pdata->edit_ctrlc = ctrlc_fcn;
	memset (pdata->text, '\0', text_max+1);
	XSelectInput (cxt->display, gadget->window, ExposureMask | ButtonPressMask | ButtonReleaseMask);
	XMapWindow (cxt->display, gadget->window);
	gadget->pdata = (void *)pdata;
	gadget->expose_event = edit_expose_event;
	gadget->keypress_event = edit_keypress_event;
	return gadget;
}


/*
 *	void destroy_edit (Gadget *gadget)
 *	destroys an edit gadget
 */
void destroy_edit (Gadget *gadget)
{
	XUnmapWindow (gadget->xcxt->display, gadget->window);
	XDestroyWindow (gadget->xcxt->display, gadget->window);
	XFreeGC (gadget->xcxt->display, gadget->gc);
	XFreeFont (gadget->xcxt->display, gadget->fontinfo);
	free (gadget->pdata);
	destroy_gadget (gadget);
	return;
}


/*
 *	char *text_of_edit (Gadget *gadget)
 *	Returns the text held within and edit gadget
 */
char *text_of_edit (Gadget *gadget)
{
	EditPrivate *pdata;

	pdata = (EditPrivate *)gadget->pdata;
	return pdata->text;
}


/*
 *	Gadget *create_button (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height, char *text, void (*button_fcn)(Gadget *))
 *	Creates a new button gadget
 */
Gadget *create_button (MyXContext *cxt, char *name, Gadget *parent, int x, int y, int width, int height, char *text, void (*button_fcn)(Gadget *))
{
	Gadget *gadget;
	XGCValues gc_values;
	ButtonPrivate *pdata;

	gadget = create_gadget (cxt, name, x, y, width, height);
	if (read_config_entry (name, gadget->config, FLD_FOREGROUND | FLD_BACKGROUND | FLD_HIGHLIGHT | FLD_SHADOW | FLD_FONT | FLD_TEXT)) {
		return NULL;
	}
	pdata = (ButtonPrivate *)xmalloc (sizeof (ButtonPrivate));
	allocate_color (cxt, gadget->config->foreground, &gadget->fg_pixel);
	allocate_color (cxt, gadget->config->background, &gadget->bg_pixel);
	allocate_color (cxt, gadget->config->highlight, &gadget->hi_pixel);
	allocate_color (cxt, gadget->config->shadow, &gadget->sh_pixel);
	gadget->window = XCreateSimpleWindow (cxt->display, parent->window, x, y, width, height, 0, BlackPixel (cxt->display, cxt->screen), gadget->bg_pixel);
	gadget->fontinfo = XLoadQueryFont (cxt->display, gadget->config->font);
	if (!gadget->fontinfo) {
		gadget->fontinfo = XLoadQueryFont (cxt->display, "fixed");
		if (!gadget->fontinfo) {
			return NULL;
		}
	}
	gc_values.background = gadget->bg_pixel;
	gc_values.font = gadget->fontinfo->fid;
	gadget->gc = XCreateGC (cxt->display, gadget->window, GCBackground | GCFont, &gc_values);
	pdata->text_x = (gadget->width >> 1) - (XTextWidth (gadget->fontinfo, gadget->config->text, strlen (gadget->config->text)) >> 1);
	pdata->text_y = (gadget->height >> 1) + (gadget->fontinfo->ascent >> 1);
	pdata->text = gadget->config->text;
	pdata->len = strlen (pdata->text);
	pdata->mouse_in = 0;
	pdata->mouse_down = 0;
	pdata->exposed = 0;
	pdata->button_press = button_fcn;
	XSelectInput (cxt->display, gadget->window, ExposureMask | ButtonPressMask | ButtonReleaseMask | EnterWindowMask | LeaveWindowMask);
	XMapWindow (cxt->display, gadget->window);
	gadget->pdata = (void *)pdata;
	gadget->expose_event = button_expose_event;
	gadget->mousedown_event = button_mousedown_event;
	gadget->mouseup_event = button_mouseup_event;
	gadget->mouseenter_event = button_mouseenter_event;
	gadget->mouseleave_event = button_mouseleave_event;
	return gadget;
}


/*
 *	void destroy_button (Gadget *gadget)
 *	Destroys a button gadget
 */
void destroy_button (Gadget *gadget)
{
	XUnmapWindow (gadget->xcxt->display, gadget->window);
	XDestroyWindow (gadget->xcxt->display, gadget->window);
	XFreeGC (gadget->xcxt->display, gadget->gc);
	XFreeFont (gadget->xcxt->display, gadget->fontinfo);
	free (gadget->pdata);
	destroy_gadget (gadget);
	return;
}


/*
 *	void allocate_color (MyXContext *cxt, char *colordef, unsigned long *xpixel)
 *	Allocates a color
 */
static void allocate_color (MyXContext *cxt, char *colordef, unsigned long *xpixel)
{
	XColor def_theory, def_actual;

	if (!XAllocNamedColor (cxt->display, DefaultColormap (cxt->display, cxt->screen), colordef, &def_theory, &def_actual)) {
		*xpixel = WhitePixel (cxt->display, cxt->screen);
	} else {
		*xpixel = def_theory.pixel;
	}
	return;
}


/*
 *	void panel_expose_event (XEvent *ev, Gadget *gadget)
 *	Called when a panel gets exposed
 */
static void panel_expose_event (XEvent *ev, Gadget *gadget)
{
	XPoint points[3];

	XClearWindow (gadget->xcxt->display, gadget->window);
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->hi_pixel);
	points[0].x = 0, points[0].y = (gadget->height - 1);
	points[1].x = 0, points[1].y = 0;
	points[2].x = (gadget->width - 1), points[2].y = 0;
	XDrawLines (gadget->xcxt->display, gadget->window, gadget->gc, points, 3, CoordModeOrigin);
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->sh_pixel);
	points[0].x = 1, points[0].y = (gadget->height - 1);
	points[1].x = (gadget->width - 1), points[1].y = (gadget->height - 1);
	points[2].x = (gadget->width - 1), points[2].y = 1;
	XDrawLines (gadget->xcxt->display, gadget->window, gadget->gc, points, 3, CoordModeOrigin);
	return;
}


/*
 *	void label_expose_event (XEvent *ev, Gadget *gadget)
 *	Called when a label gets exposed
 */
static void label_expose_event (XEvent *ev, Gadget *gadget)
{
	LabelPrivate *pdata;

	pdata = (LabelPrivate *)gadget->pdata;
	pdata->exposed = 1;
	XClearWindow (gadget->xcxt->display, gadget->window);
	XDrawString (gadget->xcxt->display, gadget->window, gadget->gc, pdata->text_x, pdata->text_y, pdata->text, pdata->len);
	return;
}


/*
 *	void label_show_action (Gadget *gadget)
 *	Shows a label
 */
static void label_show_action (Gadget *gadget)
{
	if (!gadget->visible) {
		XMapWindow (gadget->xcxt->display, gadget->window);
		gadget->visible = 1;
	}
	return;
}


/*
 *	void label_hide_action (Gadget *gadget)
 *	Hides a label
 */
static void label_hide_action (Gadget *gadget)
{
	if (gadget->visible) {
		XUnmapWindow (gadget->xcxt->display, gadget->window);
		gadget->visible = 0;
	}
	return;
}


/*
 *	void label_settext_action (Gadget *gadget, char *text)
 *	Sets the text of the label
 */
static void label_settext_action (Gadget *gadget, char *text)
{
	LabelPrivate *pdata;

	pdata = (LabelPrivate *)gadget->pdata;
	if (pdata->text) {
		free (pdata->text);
		pdata->text = NULL;
	}
	pdata->text = string_dup (text);
	pdata->len = strlen (pdata->text);
	pdata->text_x = (gadget->width >> 1) - (XTextWidth (gadget->fontinfo, pdata->text, pdata->len) >> 1);
	pdata->text_y = (gadget->height >> 1) + (gadget->fontinfo->ascent >> 1);
	if (pdata->exposed) {
		XClearWindow (gadget->xcxt->display, gadget->window);
		XDrawString (gadget->xcxt->display, gadget->window, gadget->gc, pdata->text_x, pdata->text_y, pdata->text, pdata->len);
	}
	return;
}


/*
 *	void edit_expose_event (XEvent *ev, Gadget *gadget)
 *	Called when an edit gadget gets exposed
 */
static void edit_expose_event (XEvent *ev, Gadget *gadget)
{
	EditPrivate *pdata;
	XPoint points[3];

	pdata = (EditPrivate *)gadget->pdata;
	pdata->exposed = 1;
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->sh_pixel);
	points[0].x = 0, points[0].y = (gadget->height - 1);
	points[1].x = 0, points[1].y = 0;
	points[2].x = (gadget->width - 1), points[2].y = 0;
	XDrawLines (gadget->xcxt->display, gadget->window, gadget->gc, points, 3, CoordModeOrigin);
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->hi_pixel);
	points[0].x = 1, points[0].y = (gadget->height - 1);
	points[1].x = (gadget->width - 1), points[1].y = (gadget->height - 1);
	points[2].x = (gadget->width - 1), points[2].y = 1;
	XDrawLines (gadget->xcxt->display, gadget->window, gadget->gc, points, 3, CoordModeOrigin);
	XSetForeground (gadget->xcxt->display, gadget->gc, pdata->te_pixel);
	XFillRectangle (gadget->xcxt->display, gadget->window, gadget->gc, 1, 1, gadget->width - 2, gadget->height - 2);
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->fg_pixel);
	if (pdata->cur_len > 0) {
		XDrawImageString (gadget->xcxt->display, gadget->window, gadget->gc, pdata->text_x, pdata->text_y, pdata->text, pdata->cur_len);
	}
	XDrawLine (gadget->xcxt->display, gadget->window, gadget->gc, pdata->text_width + 4, 4, pdata->text_width + 4, gadget->height - 4);
	return;
}


/*
 *	void edit_keypress_event (XEvent *ev, Gadget *gadget)
 *	Called when a keypress happens on an edit gadget
 */
static void edit_keypress_event (XEvent *ev, Gadget *gadget)
{
	EditPrivate *pdata;
	char buffer [MAX_LOOKUP_STRING];
	int count, blen;
	KeySym keysym;
	int redraw;

	pdata = (EditPrivate *)gadget->pdata;
	if (!pdata->exposed) {
		return;
	}
	redraw = 0;
	switch (ev->type) {
	case KeyPress:
		count = XLookupString (&(ev->xkey), buffer, MAX_LOOKUP_STRING-1, &keysym, NULL);
		buffer[count] = '\0';
		if ((keysym == XK_Return) || (keysym == XK_KP_Enter) || (keysym == XK_Linefeed)) {
			if (pdata->edit_accept) {
				pdata->edit_accept (gadget, pdata->text);
			} else {
				XBell (gadget->xcxt->display, 0);
			}
		} else if ((keysym == XK_c) && (ev->xkey.state & ControlMask)) {
			if (pdata->edit_ctrlc) {
				pdata->edit_ctrlc (gadget, pdata->text);
			} else {
				XBell (gadget->xcxt->display, 0);
			}
		} else if (((keysym>=XK_KP_Enter) && (keysym<=XK_KP_9)) || ((keysym>=XK_space) && (keysym<=XK_asciitilde)) || ((keysym >= XK_F1) && (keysym <= XK_F35))) {
			blen = strlen (buffer);
			if (((pdata->cur_len + blen) >= pdata->max_len) || !blen) {
				XBell (gadget->xcxt->display, 0);
			} else {
				if (pdata->cur_pos == pdata->cur_len) {
					strcat (pdata->text, buffer);
				} else {
					memmove ((pdata->text + pdata->cur_pos + blen), (pdata->text + pdata->cur_pos), strlen (pdata->text + pdata->cur_pos) + 1);
					memcpy (pdata->text + pdata->cur_pos, buffer, blen);
				}
				pdata->cur_len += blen;
				pdata->cur_pos += blen;
				pdata->text_width = XTextWidth (gadget->fontinfo, pdata->text, pdata->cur_pos);
				redraw = 1;
			}
		} else if (keysym == XK_Escape) {
			if (pdata->edit_escape) {
				pdata->edit_escape (gadget, pdata->text);
			} else {
				XBell (gadget->xcxt->display, 0);
			}
		} else if ((keysym >= XK_Shift_L) && (keysym <= XK_Hyper_R)) {
			/* Modifier */
		} else if ((keysym==XK_BackSpace) || (keysym==XK_Delete)) {
			if (pdata->cur_pos > 0) {
				if (pdata->cur_pos == pdata->cur_len) {
					pdata->text[pdata->cur_pos - 1] = '\0';
				} else {
					memmove (pdata->text + (pdata->cur_pos - 1), pdata->text + pdata->cur_pos, strlen (pdata->text + pdata->cur_pos) + 1);
				}
				pdata->cur_len--;
				pdata->cur_pos--;
				pdata->text_width = XTextWidth (gadget->fontinfo, pdata->text, pdata->cur_pos);
				redraw = 1;
			} else {
				XBell (gadget->xcxt->display, 0);
			}
		} else if (keysym == XK_Left) {
			if (pdata->cur_pos > 0) {
				pdata->cur_pos--;
				pdata->text_width = XTextWidth (gadget->fontinfo, pdata->text, pdata->cur_pos);
				redraw = 1;
			}
		} else if (keysym == XK_Right) {
			if (pdata->cur_pos < pdata->cur_len) {
				pdata->cur_pos++;
				pdata->text_width = XTextWidth (gadget->fontinfo, pdata->text, pdata->cur_pos);
				redraw = 1;
			}
		} else if (keysym == XK_End) {
			if (pdata->cur_pos != pdata->cur_len) {
				pdata->cur_pos = pdata->cur_len;
				pdata->text_width = XTextWidth (gadget->fontinfo, pdata->text, pdata->cur_pos);
				redraw = 1;
			}
		} else if (keysym == XK_Home) {
			if (pdata->cur_pos) {
				pdata->cur_pos = 0;
				pdata->text_width = XTextWidth (gadget->fontinfo, pdata->text, pdata->cur_pos);
				redraw = 1;
			}
		}
		if (redraw) {
			edit_expose_event (ev, gadget);
		}
		break;
	case KeyRelease:
		break;
	}
	return;
}


/*
 *	void button_expose_event (XEvent *ev, Gadget *gadget)
 *	Called when button is exposed
 */
static void button_expose_event (XEvent *ev, Gadget *gadget)
{
	ButtonPrivate *pdata;
	XPoint points[3];
	int disp;

	pdata = (ButtonPrivate *)gadget->pdata;
	pdata->exposed = 1;
	if (pdata->mouse_down && pdata->mouse_in) {
		XSetForeground (gadget->xcxt->display, gadget->gc, gadget->sh_pixel);
		disp = 1;
	} else {
		XSetForeground (gadget->xcxt->display, gadget->gc, gadget->hi_pixel);
		disp = -1;
	}
	points[0].x = 0, points[0].y = (gadget->height - 1);
	points[1].x = 0, points[1].y = 0;
	points[2].x = (gadget->width - 1), points[2].y = 0;
	XDrawLines (gadget->xcxt->display, gadget->window, gadget->gc, points, 3, CoordModeOrigin);
	if (pdata->mouse_down && pdata->mouse_in) {
		XSetForeground (gadget->xcxt->display, gadget->gc, gadget->hi_pixel);
	} else {
		XSetForeground (gadget->xcxt->display, gadget->gc, gadget->sh_pixel);
	}
	points[0].x = 1, points[0].y = (gadget->height - 1);
	points[1].x = (gadget->width - 1), points[1].y = (gadget->height - 1);
	points[2].x = (gadget->width - 1), points[2].y = 1;
	XDrawLines (gadget->xcxt->display, gadget->window, gadget->gc, points, 3, CoordModeOrigin);
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->bg_pixel);
	XFillRectangle (gadget->xcxt->display, gadget->window, gadget->gc, 1, 1, gadget->width - 2, gadget->height - 2);
	XSetForeground (gadget->xcxt->display, gadget->gc, gadget->fg_pixel);
	XDrawImageString (gadget->xcxt->display, gadget->window, gadget->gc, pdata->text_x + disp, pdata->text_y + disp, pdata->text, pdata->len);
	return;
}


/*
 *	void button_mousedown_event (XEvent *ev, Gadget *gadget)
 *	Called when mouse button pressed on button
 */
static void button_mousedown_event (XEvent *ev, Gadget *gadget)
{
	ButtonPrivate *pdata;

	pdata = (ButtonPrivate *)gadget->pdata;
	if (!pdata->exposed) {
		return;
	}
	pdata->mouse_down = 1;
	button_expose_event (ev, gadget);
	return;
}


/*
 *	void button_mouseup_event (XEvent *ev, Gadget *gadget)
 *	Called when mouse button released
 */
static void button_mouseup_event (XEvent *ev, Gadget *gadget)
{
	ButtonPrivate *pdata;

	pdata = (ButtonPrivate *)gadget->pdata;
	if (!pdata->exposed) {
		return;
	}
	if (pdata->mouse_in && pdata->mouse_down && pdata->button_press) {
		pdata->button_press (gadget);
	}
	pdata->mouse_down = 0;
	button_expose_event (ev, gadget);
	return;
}


/*
 *	void button_mouseenter_event (XEvent *ev, Gadget *gadget)
 *	Called when mouse enters button
 */
static void button_mouseenter_event (XEvent *ev, Gadget *gadget)
{
	ButtonPrivate *pdata;

	pdata = (ButtonPrivate *)gadget->pdata;
	if (!pdata->exposed) {
		return;
	}
	pdata->mouse_in = 1;
	button_expose_event (ev, gadget);
	return;
}


/*
 *	void button_mouseleave_event (XEvent *ev, Gadget *gadget)
 *	Called when mouse button leaves
 */
static void button_mouseleave_event (XEvent *ev, Gadget *gadget)
{
	ButtonPrivate *pdata;

	pdata = (ButtonPrivate *)gadget->pdata;
	if (!pdata->exposed) {
		return;
	}
	pdata->mouse_in = 0;
	button_expose_event (ev, gadget);
	return;
}





