/*
 *	xdmping.c - XDM pinger (too see if a host will manage us)
 *	Copyright (C) 2000  Fred Barnes
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *	The return codes from this program are:
 *		0 -- host OK and willing to manage
 *		1 -- host not responding (timing out or duff data)
 *		2 -- host OK but not willing to manage
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include "dm.h"
#include <X11/Xdmcp.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <errno.h>

#include <netdb.h>

#define DEFAULT_TIMEOUT		(2)
#define DEFAULT_RETRIES		(1)

static int ping_host (char *progname, struct in_addr *host, int verbose, int timeout, int retries);

/* this is grim and ugly, but kinda sensible.. */
static char slackspace1[512];
static XdmcpBuffer xdmcp_buf, ret_buf;
static XdmcpHeader header, ret_head;
static char slackspace2[512];


/*
 *	int main (int argc, char **argv)
 *	Start here
 */
int main (int argc, char **argv)
{
	int i;
	char **walk;
	int x_timeout, x_retries, x_verbose;
	struct hostent *hp;
	struct in_addr addr;

	if (argc < 2) {
		fprintf (stderr, "Usage: %s <hostname> [-v] [-t timeout] [-r retries]\n", *argv);
		fprintf (stderr, "Options:\n");
		fprintf (stderr, "\t-v\t\tbe verbose\n");
		fprintf (stderr, "\t-t timeout\ttimeout in seconds (default %d)\n", DEFAULT_TIMEOUT);
		fprintf (stderr, "\t-r retries\tnumber of retries before giving up (default %d)\n", DEFAULT_RETRIES);
		exit (EXIT_FAILURE);
	}
	memset (slackspace1, 0, sizeof(slackspace1));
	memset (slackspace2, 0, sizeof(slackspace2));
	x_verbose = 0;
	x_timeout = DEFAULT_TIMEOUT;
	x_retries = DEFAULT_RETRIES;
	for (walk=argv+2, i=2; (i < argc) && *walk; walk++, i++) {
		if (!strcasecmp (*walk, "-v")) {
			x_verbose = 1;
		} else if (!strcasecmp (*walk, "-t")) {
			walk++, i++;
			if (!*walk || (i == argc)) {
				fprintf (stderr, "%s: -t expects an argument\n", *argv);
				exit (EXIT_FAILURE);
			}
			if (!sscanf (*walk, "%d", &x_timeout)) {
				fprintf (stderr, "%s: -t expects integer argument (got %s)\n", *argv, *walk);
				exit (EXIT_FAILURE);
			}
		} else if (!strcasecmp (*walk, "-r")) {
			walk++, i++;
			if (!*walk || (i == argc)) {
				fprintf (stderr, "%s: -r expects an argument\n", *argv);
				exit (EXIT_FAILURE);
			}
			if (!sscanf (*walk, "%d", &x_retries)) {
				fprintf (stderr, "%s: -r expects integer argument (got %s)\n", *argv, *walk);
				exit (EXIT_FAILURE);
			}
		} else {
			fprintf (stderr, "%s: unknown argument %s -- ignoring.\n", *argv, *walk);
		}
	}
	//if (!inet_aton (argv[1], &addr)) {
		hp = gethostbyname (argv[1]);
		if (!hp) {
			fprintf (stderr, "%s: invalid hostname %s\n", *argv, argv[1]);
			exit (EXIT_FAILURE);
		}
		memcpy (&addr.s_addr, hp->h_addr, sizeof(addr.s_addr));
	//}
	inet_ntoa (addr);
	i = ping_host (*argv, &addr, x_verbose, x_timeout, x_retries);
	switch (i) {
	case -1:
	default:
		exit (1);
		break;
	case 0:
		exit (0);
		break;
	case 1:
		exit (2);
		break;
	}
	return 0;
}


/*
 *	int ping_host (char *progname, struct in_addr *host, int verbose, int timeout, int retries)
 *	Attempts to ping the named host (params are obvious)
 *	Returns:
 *		-1 on timeout or other error
 *		0 on host there and willing
 *		1 on host there but not willing
 */
static int ping_host (char *progname, struct in_addr *host, int verbose, int timeout, int retries)
{
	int fd, selret, retcode;
	struct timeval tv;
	struct sockaddr_in sin;
	struct sockaddr r_addr;
	int r_addr_size;
	fd_set read_set;
	ARRAY8 authname, hostname, status;
	ARRAYofARRAY8 aanames;

	header.version = XDM_PROTOCOL_VERSION;
	header.opcode = (CARD16) QUERY;
	header.length = 1;
	aanames.length = 0;
	XdmcpWriteHeader (&xdmcp_buf, &header);
	XdmcpWriteARRAYofARRAY8 (&xdmcp_buf, &aanames);
	fd = socket (AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) {
		fprintf (stderr, "%s: unable to create socket: %s\n", progname, strerror(errno));
		return -1;
	}
	sin.sin_family = AF_INET;
	sin.sin_port = htons (XDM_UDP_PORT);
	memmove (&sin.sin_addr, &host->s_addr, sizeof (host->s_addr));
	retcode = -1;
	while (retries) {
		if (verbose) {
			printf ("contacting %s...", inet_ntoa (*(struct in_addr *)(&sin.sin_addr)));
			fflush (stdout);
		}
		XdmcpFlush (fd, &xdmcp_buf, (XdmcpNetaddr)&sin, sizeof (sin));
		tv.tv_sec = timeout;
		tv.tv_usec = 0;
		do {
			FD_ZERO (&read_set);
			FD_SET (fd, &read_set);
			selret = select (fd+1, &read_set, NULL, NULL, timeout ? &tv : NULL);
		} while ((selret == -1) && (errno == EINTR));
		if (!selret) {
			/* timed out */
			if (verbose) {
				printf (" timed out\n");
			}
			retries--;
		} else if (selret < 0) {
			fprintf (stderr, "%s: select () returned %d (error was %s)\n", progname, selret, strerror(errno));
			return -1;
		} else if (!FD_ISSET (fd, &read_set)) {
			fprintf (stderr, "%s: socket not select()ed.. (bad)\n", progname);
			return -1;
		} else {
			/* read the response */
			if (!XdmcpFill (fd, &ret_buf, (XdmcpNetaddr)&r_addr, &r_addr_size)) {
				printf (" bad packet from %s\n", inet_ntoa (*(struct in_addr *)(&sin.sin_addr)));
				retries--;
			} else if (!XdmcpReadHeader (&ret_buf, &ret_head)) {
				printf (" bad XDMCP header from %s\n", inet_ntoa (*(struct in_addr *)(&sin.sin_addr)));
				retries--;
			} else if (ret_head.version != XDM_PROTOCOL_VERSION) {
				printf (" bad protocol version from %s\n", inet_ntoa (*(struct in_addr *)(&sin.sin_addr)));
				retries--;
			} else {
				switch (ret_head.opcode) {
				case WILLING:
					if (XdmcpReadARRAY8 (&ret_buf, &authname) && XdmcpReadARRAY8 (&ret_buf, &hostname) && XdmcpReadARRAY8 (&ret_buf, &status)) {
						retcode = 0;
						if (verbose) {
							printf ("\n%.*s: %.*s\n", hostname.length, hostname.data, status.length, status.data);
						}
					}
					break;
				case UNWILLING:
					if (XdmcpReadARRAY8 (&ret_buf, &hostname) && XdmcpReadARRAY8 (&ret_buf, &status)) {
						retcode = 1;
						if (verbose) {
							printf ("\n%.*s: %.*s\n", hostname.length, hostname.data, status.length, status.data);
						}
					}
					break;
				}
				break; /* from while (retries) */
			}
		}
	}
	close (fd);
	return retcode;
}



