/*
 *	chooser.c - main chooser app
 *	Copyright (C) 1990 X Consortium
 *	Copyright (C) 2000 Fred Barnes <frmb@ukc.ac.uk>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *	NOTE:
 *	This file is based on the X Consortium's chooser.c, written by
 *	Keith Packard of the MIT X Consortium.
 *	The original copyright notice (verbatim) is below
 */

/*
 * $XConsortium: chooser.c /main/27 1996/01/25 18:45:41 kaleb $
 *
Copyright (c) 1990  X Consortium

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall not be
used in advertising or otherwise to promote the sale, use or other dealings
in this Software without prior written authorization from the X Consortium.
 *
 * Author:  Keith Packard, MIT X Consortium
 */



#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <signal.h>
#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "support.h"
#include "cfg.h"
#include "interface.h"
#include "error.h"
#include "version.h"

#include "dm.h"

#include <X11/Xdmcp.h>

#ifdef SVR4
	#include    <sys/sockio.h>
#endif

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>

#ifdef STREAMSCONN
	#ifdef WINTCP /* NCR with Wollongong TCP */
		#include    <netinet/ip.h>
	#endif
	#include    <stropts.h>
	#include    <tiuser.h>
	#include    <netconfig.h>
	#include    <netdir.h>
#endif


#ifndef ishexdigit
	#define ishexdigit(c)	(isdigit(c) || ('a' <= (c) && (c) <= 'f'))
#endif

#ifdef hpux
	#include <sys/utsname.h>
	#ifdef HAS_IFREQ
		#include <net/if.h>
	#endif
#else
	#ifdef __convex__
		#include <sync/queue.h>
		#include <sync/sema.h>
	#endif
	#include <net/if.h>
#endif /* hpux */

#include    <netdb.h>


static ARRAY8Ptr xdmAddress;
static ARRAY8Ptr  clientAddress;
static int connectionType;
static int child_pid;


typedef struct _hostAddr {
	struct sockaddr *addr;
	int addrlen;
	xdmOpCode type;
} HostAddr;

typedef struct _hostName {
	char *fullname;
	int willing;
	ARRAY8 hostname, status;
	CARD16 connectionType;
	ARRAY8 hostaddr;
} HostName;

static HostAddr sel_hostaddr;
static HostName sel_hostname;

static int socketFD;
static int pingTry;
static int prog_state;

enum PROG_STATES { NOTHING, QUERYING, CONNECTING };


#define PING_INTERVAL	2000
#define TRIES		3

static XdmcpBuffer directBuffer, broadcastBuffer;
static XdmcpBuffer buffer;
static ARRAYofARRAY8 AuthenticationNames;

static int query_host (void);
static int PingHost (void);
static void PingHostAgain (void);
static int AddHostname (ARRAY8Ptr hostname, ARRAY8Ptr status, struct sockaddr *addr, int willing);
static void ReceivePacket (void);
static int RegisterHostaddr (struct sockaddr *addr, int len, xdmOpCode type);
static int RegisterHostname (char *name);
static int InitXDMCP (void);
static int FromHex (char *s, char *d, int len);
static int StringToARRAY8 (ARRAY8Ptr *dest, char *src);
static int check_address (struct in_addr in);
static void run_init_script (void);
static void dump_version_info (FILE *stream);
static void sigchld_handler (int sig);
static void cleanup_child (void);

extern int XdmcpAllocARRAY8 (ARRAY8Ptr, int);

extern GlobalConfigData _globconfig;

/*
 *	int query_host (void)
 *	Queries a host to get info about it
 *	Returns 0 on success -1 on failure
 */
static int query_host (void)
{
	pingTry = 0;
	if (PingHost ()) {
		return -1;
	}
	return 0;
}


/*
 *	int PingHost (void)
 *	Pings the selected host
 *	Returns 0 on success, -1 on failure
 */
static int PingHost (void)
{
	struct timeval tv;

	if (sel_hostaddr.type == QUERY) {
		XdmcpFlush (socketFD, &directBuffer, (XdmcpNetaddr)sel_hostaddr.addr, sel_hostaddr.addrlen);
	} else {
		return -1;
	}
	tv.tv_sec = 2;
	tv.tv_usec = 0;
	if (++pingTry < TRIES) {
		xparts_readtimeout (socketFD, &tv, ReceivePacket, PingHostAgain);
	} else {
		set_error_message ("Timed out");
		xparts_readtimeout (-1, NULL, NULL, NULL);
		return -1;
	}
	return 0;
}


/*
 *	void PingHostAgain (void)
 *	Calls PingHost
 */
static void PingHostAgain (void)
{
	if (PingHost ()) {
		mainloop_terminate (1);
	}
	return;
}


/*
 *	int AddHostname (ARRAY8Ptr hostname, ARRAY8Ptr status, struct sockaddr *addr, int willing)
 *	Called from ReceivePacket
 */
static int AddHostname (ARRAY8Ptr hostname, ARRAY8Ptr status, struct sockaddr *addr, int willing)
{
	ARRAY8 hostAddr;
	CARD16 connectionType;
	int fulllen;

	switch (addr->sa_family) {
	case AF_INET:
		hostAddr.data = (CARD8 *) &((struct sockaddr_in *) addr)->sin_addr;
		hostAddr.length = 4;
		connectionType = FamilyInternet;
		break;
	default:
		hostAddr.data = (CARD8 *) "";
		hostAddr.length = 0;
		connectionType = FamilyLocal;
		break;
	}
	if (hostname->length) {
		switch (addr->sa_family) {
		case AF_INET:
		{
			struct hostent  *hostent;
			char *host;
    	
			hostent = gethostbyaddr ((char *)hostAddr.data, hostAddr.length, AF_INET);
			if (hostent) {
				XdmcpDisposeARRAY8 (hostname);
				host = hostent->h_name;
				XdmcpAllocARRAY8 (hostname, strlen (host));
				memmove( hostname->data, host, hostname->length);
			}
		}
		default:
			break;
		}
	}
	if (!XdmcpAllocARRAY8 (&sel_hostname.hostaddr, hostAddr.length)) {
		free (sel_hostname.fullname);
		return -1;
	}
	memmove (sel_hostname.hostaddr.data, hostAddr.data, hostAddr.length);
	sel_hostname.connectionType = connectionType;
	sel_hostname.hostname = *hostname;
	sel_hostname.willing = willing;
	sel_hostname.status = *status;
	hostname = &sel_hostname.hostname;
	fulllen = hostname->length;
	sel_hostname.fullname = xmalloc (fulllen + status->length + 10);
	if (!sel_hostname.fullname) {
		sel_hostname.fullname = string_dup ("Unknown");
	} else {
		sprintf (sel_hostname.fullname, "%.*s %*.*s", hostname->length, hostname->data, status->length, status->length, status->data);
	}
	return 0;
}


/*
 *	void ReceivePacket (void)
 *	Called when a packet arrives
 */
static void ReceivePacket (void)
{
	XdmcpHeader header;
	ARRAY8 authenticationName;
	ARRAY8 hostname;
	ARRAY8 status;
	int saveHostname = 0;
	struct sockaddr addr;
	int addrlen;

	addrlen = sizeof (addr);
	if (!XdmcpFill (socketFD, &buffer, (XdmcpNetaddr)&addr, &addrlen)) {
		set_error_message ("Bad packet from host");
		xparts_readtimeout (-1, NULL, NULL, NULL);
		prog_state = NOTHING;
		return;
	}
	if (!XdmcpReadHeader (&buffer, &header)) {
		set_error_message ("Bad XDMCP header from host");
		xparts_readtimeout (-1, NULL, NULL, NULL);
		prog_state = NOTHING;
		return;
	}
	if (header.version != XDM_PROTOCOL_VERSION) {
		set_error_message ("Bad protocol version from host");
		xparts_readtimeout (-1, NULL, NULL, NULL);
		prog_state = NOTHING;
		return;
	}
	hostname.data = 0;
	status.data = 0;
	authenticationName.data = 0;
	switch (header.opcode) {
	case WILLING:
		if (XdmcpReadARRAY8 (&buffer, &authenticationName) && XdmcpReadARRAY8 (&buffer, &hostname) && XdmcpReadARRAY8 (&buffer, &status)) {
			if (header.length == 6 + authenticationName.length + hostname.length + status.length) {
				if (AddHostname (&hostname, &status, &addr, header.opcode == (int) WILLING)) {
					saveHostname = 1;
				}
			}
		}
		XdmcpDisposeARRAY8 (&authenticationName);
		break;
	case UNWILLING:
		if (XdmcpReadARRAY8 (&buffer, &hostname) && XdmcpReadARRAY8 (&buffer, &status)) {
			if (header.length == 4 + hostname.length + status.length) {
				if (AddHostname (&hostname, &status, &addr, header.opcode == (int) WILLING)) {
					saveHostname = 1;
				}
			}
		}
		break;
	default:
		break;
	}
	if (!saveHostname) {
		XdmcpDisposeARRAY8 (&hostname);
		XdmcpDisposeARRAY8 (&status);
	}
	xparts_readtimeout (-1, NULL, NULL, NULL);
	mainloop_terminate (0);
	return;
}


/*
 *	int RegisterHostaddr (struct sockaddr *addr, int len, xdmOpCode type)
 *	Registers a host address
 *	Returns 0 on success, -1 on failure
 */
static int RegisterHostaddr (struct sockaddr *addr, int len, xdmOpCode type)
{
	if (sel_hostaddr.addr) {
		free (sel_hostaddr.addr);
		sel_hostaddr.addr = NULL;
	}
	sel_hostaddr.addr = (struct sockaddr *)xmalloc (len);
	if (!sel_hostaddr.addr) {
		return -1;
	}
	memmove ((char *)sel_hostaddr.addr, (char *)addr, len);
	sel_hostaddr.addrlen = len;
	sel_hostaddr.type = type;
	return 0;
}


/*
 *	int RegisterHostname (char *name)
 *	Registers a hostname when typed in
 *	Returns 0 on success, -1 on failure
 */
static int RegisterHostname (char *name)
{
	struct hostent *hostent;
	struct sockaddr_in in_addr;

	if (strlen(name) == 8 && FromHex(name, (char *)&in_addr.sin_addr, strlen(name)) == 0) {
		/* address as hex string, e.g., "12180022" (depreciated) */
		in_addr.sin_family = AF_INET;
	} else if ((in_addr.sin_addr.s_addr = inet_addr(name)) != -1) {
		/* Per RFC 1123, check first for IP address in dotted-decimal form */
		in_addr.sin_family = AF_INET;
	} else {
		hostent = gethostbyname (name);
		if (!hostent) {
			return -1;
		}
		if (hostent->h_addrtype != AF_INET || hostent->h_length != 4) {
			return -1;
		}
		in_addr.sin_family = hostent->h_addrtype;
		memmove( &in_addr.sin_addr, hostent->h_addr, 4);
	}
	in_addr.sin_port = htons (XDM_UDP_PORT);
	#ifdef BSD44SOCKETS
		in_addr.sin_len = sizeof(in_addr);
	#endif
	return RegisterHostaddr ((struct sockaddr *)&in_addr, sizeof (in_addr), QUERY);
}


/*
 *	int InitXDMCP (void)
 *	Initialises bits of the XDMCP stuff
 *	Returns 0 on success, -1 on failure
 */
static int InitXDMCP (void)
{
	int soopts = 1;
	XdmcpHeader header;
	int i;

	header.version = XDM_PROTOCOL_VERSION;
	header.opcode = (CARD16) BROADCAST_QUERY;
	header.length = 1;
	for (i = 0; i < (int)AuthenticationNames.length; i++) {
		header.length += 2 + AuthenticationNames.data[i].length;
	}
	XdmcpWriteHeader (&broadcastBuffer, &header);
	XdmcpWriteARRAYofARRAY8 (&broadcastBuffer, &AuthenticationNames);

	header.version = XDM_PROTOCOL_VERSION;
	header.opcode = (CARD16) QUERY;
	header.length = 1;
	for (i = 0; i < (int)AuthenticationNames.length; i++) {
		header.length += 2 + AuthenticationNames.data[i].length;
	}
	XdmcpWriteHeader (&directBuffer, &header);
	XdmcpWriteARRAYofARRAY8 (&directBuffer, &AuthenticationNames);
	#if defined(STREAMSCONN)
		if ((socketFD = t_open ("/dev/udp", O_RDWR, 0)) < 0) {
			return -1;
		}
		if (t_bind( socketFD, NULL, NULL ) < 0) {
			t_close(socketFD);
			return -1;
		}

		/*
		 * This part of the code looks contrived. It will actually fit in nicely
		 * when the CLTS part of Xtrans is implemented.
		 */
		{
			struct netconfig *nconf;

			if( (nconf=getnetconfigent("udp")) == NULL ) {
				t_unbind(socketFD);
				t_close(socketFD);
				return -1;
			}

			if( netdir_options(nconf, ND_SET_BROADCAST, socketFD, NULL) ) {
				freenetconfigent(nconf);
				t_unbind(socketFD);
				t_close(socketFD);
				return -1;
			}

			freenetconfigent(nconf);
		}
	#else
		if ((socketFD = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
			return -1;
		}
	#endif
	#ifdef SO_BROADCAST
		if (_globconfig.allowbroadcast) {
			soopts = 1;
			if (setsockopt (socketFD, SOL_SOCKET, SO_BROADCAST, (char *)&soopts, sizeof (soopts)) < 0) {
				perror ("setsockopt");
				return -1;
			}
		}
	#endif
    
	return 0;
}


/*
 *	void Choose (void)
 *	Chooses the selected host
 */
static void Choose (void)
{
	if (xdmAddress) {
		struct sockaddr_in  in_addr;
		struct sockaddr	*addr;
		int family;
		int len;
		int fd;
		char buf[1024];
		XdmcpBuffer buffer;
		char *xdm;
		#if defined(STREAMSCONN)
			struct t_call call, rcv;
		#endif

		addr = NULL;
		len = 0;
		xdm = (char *)(xdmAddress->data);
		family = (xdm[0] << 8) + xdm[1];
		switch (family) {
		case AF_INET:
			#ifdef BSD44SOCKETS
				in_addr.sin_len = sizeof(in_addr);
			#endif
			in_addr.sin_family = family;
			memmove( &in_addr.sin_port, xdm + 2, 2);
			memmove( &in_addr.sin_addr, xdm + 4, 4);
			addr = (struct sockaddr *) &in_addr;
			len = sizeof (in_addr);
			break;
		default:
			break;
		}
		#if defined(STREAMSCONN)
			if ((fd = t_open ("/dev/tcp", O_RDWR, NULL)) == -1) {
				fprintf (stderr, "Cannot create response endpoint\n");
				fflush(stderr);
				cleanup_child ();
				exit (REMANAGE_DISPLAY);
			}
			if (t_bind (fd, NULL, NULL) == -1) {
				fprintf (stderr, "Cannot bind response endpoint\n");
				fflush(stderr);
				t_close (fd);
				cleanup_child ();
				exit (REMANAGE_DISPLAY);
			}
			call.addr.buf=(char *)addr;
			call.addr.len=len;
			call.addr.maxlen=len;
			call.opt.len=0;
			call.opt.maxlen=0;
			call.udata.len=0;
			call.udata.maxlen=0;
			if (t_connect (fd, &call, NULL) == -1) {
				t_error ("Cannot connect to xdm\n");
				fflush(stderr);
				t_unbind (fd);
				t_close (fd);
				cleanup_child ();
				exit (REMANAGE_DISPLAY);
			}
		#else
			if ((fd = socket (family, SOCK_STREAM, 0)) == -1) {
				fprintf (stderr, "Cannot create response socket\n");
				cleanup_child ();
				exit (REMANAGE_DISPLAY);
			}
			if (connect (fd, addr, len) == -1) {
				fprintf (stderr, "Cannot connect to xdm\n");
				cleanup_child ();
				exit (REMANAGE_DISPLAY);
			}
		#endif
		buffer.data = (BYTE *) buf;
		buffer.size = sizeof (buf);
		buffer.pointer = 0;
		buffer.count = 0;
		XdmcpWriteARRAY8 (&buffer, clientAddress);
		XdmcpWriteCARD16 (&buffer, (CARD16)connectionType);
		XdmcpWriteARRAY8 (&buffer, &sel_hostname.hostaddr);
		#if defined(STREAMSCONN)
			if( t_snd (fd, (char *)buffer.data, buffer.pointer, 0) < 0 ) {
				fprintf (stderr, "Cannot send to xdm\n");
				fflush(stderr);
				t_unbind (fd);
				t_close (fd);
				cleanup_child ();
				exit (REMANAGE_DISPLAY);
			}
			sleep(5);	/* Hack because sometimes the connection gets
					   closed before the data arrives on the other end. */
			t_snddis (fd,NULL);
			t_unbind (fd);
			t_close (fd);
		#else
			write (fd, (char *)buffer.data, buffer.pointer);
			close (fd);
		#endif
	} else {
		int i;

		printf ("%u\n", sel_hostname.connectionType);
		for (i = 0; i < (int)sel_hostname.hostaddr.length; i++) {
			printf ("%u%s", sel_hostname.hostaddr.data[i], i == sel_hostname.hostaddr.length - 1 ? "\n" : " ");
		}
	}
	return;
}


/*
 *	void host_select_function (char *name)
 *	Called to select a host
 */
static void host_select_function (char *name)
{
	char buffer[256], tbuf[128], *ch, *dh;
	struct hostent *hp;

	if (!name) {
		set_error_message ("invalid hostname");
		return;
	}
	for (ch=name; (*ch==' ') || (*ch=='\t'); ch++);
	for (dh=name + (strlen(name)-1); (*dh==' ') || (*dh=='\t'); *dh = '\0', dh--);
	name = ch;
	if (!strlen(name)) {
		set_error_message ("invalid hostname");
		return;
	}
	hp = gethostbyname (name);
	if (!hp) {
		sprintf (buffer, "invalid hostname %s", name);
		set_error_message (buffer);
		return;
	}
	if (check_address (*(struct in_addr *)(hp->h_addr))) {
		sprintf (buffer, "illegal hostname %s", name);
		set_error_message (buffer);
		return;
	}
	strcpy (tbuf, inet_ntoa (*(struct in_addr *)(hp->h_addr)));
	sprintf (buffer, "trying %s...", tbuf);
	set_status_message (buffer);
	if (RegisterHostname (tbuf)) {
		sprintf (buffer, "invalid hostname %s", name);
		set_error_message (buffer);
		return;
	}
	if (query_host ()) {
		sprintf (buffer, "invalid hostname %s", name);
		set_error_message (buffer);
		return;
	}
	prog_state = CONNECTING;
	return;
}


/*
 *	void host_check_function (char *name)
 *	Called to check a host's availability
 */
static void host_check_function (char *name)
{
	char buffer[256], tbuf[128], *ch, *dh;
	struct hostent *hp;

	if (_globconfig.hidequery) {
		return;
	}
	if (!name) {
		set_error_message ("invalid hostname");
		return;
	}
	for (ch=name; (*ch==' ') || (*ch=='\t'); ch++);
	for (dh=name + (strlen(name)-1); (*dh==' ') || (*dh=='\t'); *dh = '\0', dh--);
	name = ch;
	if (!strlen (name)) {
		set_error_message ("invalid hostname");
		return;
	}
	hp = gethostbyname (name);
	if (!hp) {
		sprintf (buffer, "invalid hostname %s", name);
		set_error_message (buffer);
		return;
	}
	if (check_address (*(struct in_addr *)(hp->h_addr))) {
		sprintf (buffer, "illegal hostname %s", name);
		set_error_message (buffer);
		return;
	}
	strcpy (tbuf, inet_ntoa (*(struct in_addr *)(hp->h_addr)));
	sprintf (buffer, "querying %s...", tbuf);
	set_status_message (buffer);
	if (RegisterHostname (tbuf)) {
		sprintf (buffer, "invalid hostname %s", name);
		set_error_message (buffer);
		return;
	}
	if (query_host ()) {
		sprintf (buffer, "invalid hostname %s", name);
		set_error_message (buffer);
		return;
	}
	prog_state = QUERYING;
	return;
}


/*
 *	void dump_version_info (FILE *stream)
 *	dumps some version information
 */
static void dump_version_info (FILE *stream)
{
	fprintf (stream, "xdmchoose (chooser) version " XDMCHOOSE_VERSION "\n");
	fprintf (stream, "Copyright (C) 2000 Fred Barnes <frmb2@ukc.ac.uk>\n");
	fprintf (stream, "Portions copyright (C) MIT Consortium\n");
	return;
}


/*
 *	void cleanup_child (void)
 *	cleans-up the child (script) process
 */
static void cleanup_child (void)
{
	if (!_globconfig.startscript) {
		return;
	}
	if (child_pid > -1) {
		/* kill, if not already dead */
		kill (child_pid, SIGHUP);
		child_pid = -1;
	}
	/* if the kill-window is set, terminate first window with that name */
	if (_globconfig.startscriptkill) {
		xparts_kill_window (_globconfig.startscriptkill);
	}
	return;
}


/*
 *	int main (int argc, char **argv)
 *	Start here
 */
int main (int argc, char **argv)
{
	int i;
	char **walk;
	int running;

	xdmAddress = (ARRAY8Ptr)NULL;
	clientAddress = (ARRAY8Ptr)NULL;
	connectionType = 0;
	child_pid = 0;
	for (walk = argv+1, i = 1; *walk && (i < argc); i++, walk++) {
		if (!strcasecmp (*walk, "-xdmaddress")) {
			walk++, i++;
			if (!*walk || (i==argc)) {
				fprintf (stderr, "%s: -xdmaddress missing argument\n", *argv);
				exit (EXIT_FAILURE);
			}
			if (StringToARRAY8 (&xdmAddress, *walk)) {
				fprintf (stderr, "%s: unable to convert [%s]\n", *argv, *walk);
				exit (EXIT_FAILURE);
			}
			continue;
		}
		if (!strcasecmp (*walk, "-clientaddress")) {
			walk++, i++;
			if (!*walk || (i == argc)) {
				fprintf (stderr, "%s: -clientaddress missing argument\n", *argv);
				exit (EXIT_FAILURE);
			}
			if (StringToARRAY8 (&clientAddress, *walk)) {
				fprintf (stderr, "%s: unable to convert [%s]\n", *argv, *walk);
				exit (EXIT_FAILURE);
			}
			continue;
		}
		if (!strcasecmp (*walk, "-connectiontype")) {
			walk++, i++;
			if (!*walk || (i == argc)) {
				fprintf (stderr, "%s: -connectiontype missing argument\n", *argv);
				exit (EXIT_FAILURE);
			}
			if (!sscanf (*walk, "%d", &connectionType)) {
				fprintf (stderr, "%s: badly formed integer for -connectiontype [%s]\n", *argv, *walk);
				exit (EXIT_FAILURE);
			}
			continue;
		}
		if (!strcasecmp (*walk, "--version")) {
			dump_version_info (stderr);
			exit (EXIT_SUCCESS);
		}
		/* Just ignore other command line things... */
	}
	if (init_xparts (argv, argc)) {
		cleanup_child ();
		exit (EXIT_FAILURE);
	}
	AuthenticationNames.length = 0;
	InitXDMCP ();
	give_select_function (host_select_function);
	give_check_function (host_check_function);
	run_init_script ();
	close_config_file ();
	running = 1;
	while (running) {
		i = xparts_mainloop ();
		if (i == 1) {
			continue;
		} else if (i == 255) {
			/* ctrl-c pressed */
			running = 0;
			/* cleanup_child (); */
			exit (_globconfig.ctrlcexitcode);
			stop_xparts ();
			continue;
		}
		switch (prog_state) {
		case QUERYING:
			set_status_message (sel_hostname.fullname);
			continue;
			break;
		case CONNECTING:
			if (!sel_hostname.willing) {
				set_error_message ("host unwilling to manage");
			} else {
				Choose ();
				running = 0;
			}
			break;
		default:
			break;
		}
	}
	cleanup_child ();
	_exit (OBEYSESS_DISPLAY);
	stop_xparts ();
	return 0;
}


/*
 *	int FromHex (char *s, char *d, int len)
 *	Converts the hex string in 's' to one in 'd', of length 'len'
 */
static int FromHex (char *s, char *d, int len)
{
	int t;
	int ret = len&1;		/* odd-length hex strings are illegal */

	while (len >= 2) {
		#define HexChar(c)  ('0' <= (c) && (c) <= '9' ? (c) - '0' : (c) - 'a' + 10)

		if (!(ishexdigit(*s))) {
			ret = 1;
		}
		t = HexChar (*s) << 4;
		s++;
		if (!(ishexdigit(*s))) {
			ret = 1;
		}
		t += HexChar (*s);
		s++;
		*d++ = t;
		len -= 2;
	}
	return ret;
}


/*
 *	int StringToARRAY8 (ARRAY8Ptr *dest, char *src)
 *	Converts `src' (a hex-string) to an ARRAY8 (in `dest')
 */
static int StringToARRAY8 (ARRAY8Ptr *dest, char *src)
{
	int len;

	*dest = (ARRAY8Ptr)xmalloc (sizeof (ARRAY8));
	len = strlen (src);
	if (!XdmcpAllocARRAY8 (*dest, len >> 1)) {
		fprintf (stderr, "serious: unable to XdmcpAllocARRAY8 ()\n");
		cleanup_child ();
		exit (EXIT_FAILURE);
	}
	return FromHex (src, (char *)((*dest)->data), len);
}


/*
 *	int check_address (struct in_addr in)
 *	Checks the address for misc. defects
 */
static int check_address (struct in_addr in)
{
	unsigned char *tptr;
	int i;

	tptr = (unsigned char *)&(in.s_addr);
	for (i=0; i<sizeof (in.s_addr); i++) {
		if ((tptr[i] == 0xff) && !_globconfig.allowbroadcast) {
			return -1;
		}
	}
	return 0;
}


/*
 *	void sigchld_handler (int sig)
 *	called when child dies
 */
static void sigchld_handler (int sig)
{
	child_pid = -1;
	return;
}


/*
 *	void run_init_script (void)
 *	Runs an initialisation script
 */
static void run_init_script (void)
{
	int pid;
	struct sigaction chldsig;
	sigset_t tmpset;

	if (!_globconfig.startscript) {
		return;
	}
	chldsig.sa_handler = sigchld_handler;
	sigemptyset (&tmpset);
	chldsig.sa_mask = tmpset;
	chldsig.sa_flags = SA_NOCLDSTOP | SA_RESTART;
	if (sigaction (SIGCHLD, &chldsig, NULL)) {
		return;
	}
	pid = fork ();
	switch (pid) {
	case -1:
		return;
		break;
	case 0:
		execl (_globconfig.startscript, _globconfig.startscript, NULL);
		_exit (-1);
		break;
	default:
		child_pid = pid;
		break;
	}
	return;
}


