/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#define SKIPGNOME
#include "xchat.h"
#include <netdb.h>
#include <arpa/inet.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/wait.h>
#include "plugin.h"

extern GSList *sess_list;
extern struct xchatprefs prefs;

extern void set_server_name (struct server *serv, char *name);
extern void flush_server_queue (struct server *serv);
extern int tcp_send_len (struct server *serv, char *buf, int len);
extern int tcp_send (struct server *serv, char *buf);
extern void clear_user_list (struct session *sess);
extern void PrintText (struct session *sess, unsigned char *text);
extern void read_data (struct server *serv, gint sok);
extern char *errorstring (int err);
extern int waitline (int sok, char *buf, int bufsize);
extern void gui_set_title (struct session *sess);
extern void clear_channel (struct session *sess);
extern void notc_msg (struct session *sess);
extern void notify_cleanup(void);

#ifndef HAVE_INET_ATON

int inet_aton(const char *cp, struct in_addr *inp)
{
   struct in_addr RAddr;

   RAddr.s_addr = inet_addr(cp);
   if (RAddr.s_addr != -1)
   {
	if (inp)
	{
	   *inp = RAddr;
	   return 1;
	}
   }

   return 0;
}
#endif

void 
connecting_fin (struct session *sess)
{
   if (sess)
   {
      struct server *serv = sess->server;
      GSList *list = sess_list;

      sess->server->connecting = 0;
      while (list)              /* check all windows that use this server and
                                   * remove the connecting graph, if it has one. */
      {
         sess = (struct session *) list->data;
         if (sess->server == serv && sess->bar)
         {
            gtk_widget_destroy (sess->bar);
            sess->bar = 0;
            gtk_timeout_remove (sess->server->bartag);
         }
         list = list->next;
      }
   }
}

void 
connected_signal (struct server *serv, gint sok)
{
   struct session *sess = serv->front_session;
   char tbuf[128];
   char outbuf[256];

   waitline (serv->childread, tbuf, sizeof tbuf);

   switch (tbuf[0])
   {
   case '1':                   /* unknown host */
	if(serv->iotag != -1)
	{
   	   gdk_input_remove (serv->iotag);
	   serv->iotag = -1;
	}
      close (serv->childread);
      close (serv->sok);
      waitpid (serv->childpid, NULL, WNOHANG);
      connecting_fin (sess);
      EMIT_SIGNAL (XP_TE_UKNHOST, sess, NULL, NULL, NULL, NULL, 0);
      break;
   case '2':                   /* connection failed */
      waitline (serv->childread, tbuf, 128);
	if(serv->iotag != -1)
	{
   	   gdk_input_remove (serv->iotag);
	   serv->iotag = -1;
	}
      close (serv->childread);
      close (serv->sok);
      waitpid (serv->childpid, NULL, WNOHANG);
      connecting_fin (sess);
      EMIT_SIGNAL (XP_TE_CONNFAIL, sess, errorstring (atoi (tbuf)), NULL, NULL,
                   NULL, 0);
      break;
   case '3':                   /* gethostbyname finished */
      {
         char host[100];
         char ip[100];

         waitline (serv->childread, host, sizeof host);

         waitline (serv->childread, ip, sizeof ip);

         waitline (serv->childread, outbuf, sizeof outbuf);

         EMIT_SIGNAL (XP_TE_CONNECT, sess, host, ip, outbuf, NULL, 0);
         strcpy (serv->hostname, ip);
      }
      break;
   case '4':                   /* success */
	if(serv->iotag != -1)
	{
   	   gdk_input_remove (serv->iotag);
	   serv->iotag = -1;
	}
      close (serv->childread);
      serv->connected = TRUE;
      waitpid (serv->childpid, NULL, WNOHANG);
      connecting_fin (sess);
      serv->iotag = gdk_input_add (serv->sok, GDK_INPUT_EXCEPTION | GDK_INPUT_READ,
                                   (GdkInputFunction) read_data, serv);

      if (!serv->no_login)
      {
         EMIT_SIGNAL (XP_TE_CONNECTED, sess, NULL, NULL, NULL, NULL, 0);
         if (serv->password[0])
         {
            sprintf (outbuf, "PASS %s\r\n", serv->password);
            tcp_send (serv, outbuf);
         }
         sprintf (outbuf, "NICK %s\r\n", serv->nick);
         tcp_send (serv, outbuf);
         sprintf (outbuf, "USER %s %s %s :%s\r\n",
                  prefs.username,
                  (strlen(prefs.local_name) ? prefs.local_name : "localhost"),
                  serv->hostname, prefs.realname);
         tcp_send (serv, outbuf);
      } else
         EMIT_SIGNAL (XP_TE_SERVERCONNECTED, sess, NULL, NULL, NULL, NULL, 0);
      fcntl (serv->sok, F_SETFL, O_NONBLOCK);
	set_server_name(serv, serv->servername);
      break;
   case '5':                   /* prefs ip discovered */
      waitline (serv->childread, prefs.ipaddress, sizeof(prefs.ipaddress));
      inet_aton(prefs.ipaddress, (struct in_addr *)&(prefs.local_ip));
      break;
   case '6':                   /* prefs hostname discovered */
      waitline (serv->childread, prefs.hostname, sizeof(prefs.hostname));
      strcpy(prefs.local_name, prefs.hostname);
      break;
   }
}

int 
check_connecting (struct session *sess)
{
   if (sess->server->connecting)
   {
      char tbuf[256];
      kill (sess->server->childpid, SIGKILL);
      waitpid (sess->server->childpid, NULL, WNOHANG);
      sprintf (tbuf, "%d", sess->server->childpid);
      EMIT_SIGNAL (XP_TE_SCONNECT, sess, tbuf, NULL, NULL, NULL, 0);
	if(sess->server->iotag != -1)
	{
   	   gdk_input_remove (sess->server->iotag);
	   sess->server->iotag = -1;
	}
      close (sess->server->childread);
      close (sess->server->sok);
      connecting_fin (sess);
      return TRUE;
   }
   return FALSE;
}

void 
disconnect_server (struct session *sess, int sendquit)
{
   struct server *serv = sess->server;
   GSList *list;

   if (check_connecting (sess))
      return;

   if (!serv->connected)
   {
      notc_msg (sess);
      return;
   }

   flush_server_queue (serv);

   if (serv->iotag != -1)
   {
	gdk_input_remove (serv->iotag);
	serv->iotag = -1;
   }

   list = sess_list;
   while (list)                 /* print "Disconnected" to each window using this server */
   {
      sess = (struct session *) list->data;
      if (sess->server == serv)
         EMIT_SIGNAL (XP_TE_DISCON, sess, NULL, NULL, NULL, NULL, 0);
      list = list->next;
   }

   if (sendquit)
   {
      char tbuf[128];
      if (!sess->quitreason)
         sess->quitreason = prefs.quitreason;
      snprintf (tbuf, sizeof tbuf, "QUIT :%s\r\n", sess->quitreason);
      sess->quitreason = 0;
      tcp_send (serv, tbuf);
   }
   close (serv->sok);

   serv->sok = -1;
   serv->pos = 0;
   serv->connected = FALSE;
   serv->motd_skipped = FALSE;
   serv->no_login = FALSE;
   serv->servername[0] = 0;

   list = sess_list;
   while (list)
   {
      sess = (struct session *) list->data;
      if (sess->server == serv)
      {
         if (sess->channel[0])
         {
            if (sess->channel[0] == '#' || sess->channel[0] == '&' ||
		    sess->channel[0] == '+' || sess->is_server)
               clear_channel (sess);
         } else
            clear_channel (sess);
      }
      list = list->next;
   }
   notify_cleanup();
}

int 
updatedate_bar (struct session *sess)
{
   static int type = 0;
   static float pos = 0;
   pos += 0.05;
   if (pos >= 0.99)
   {
      if (type == 0)
      {
         type = 1;
         gtk_progress_bar_set_orientation ((GtkProgressBar *) sess->bar, GTK_PROGRESS_RIGHT_TO_LEFT);
      } else
      {
         type = 0;
         gtk_progress_bar_set_orientation ((GtkProgressBar *) sess->bar, GTK_PROGRESS_LEFT_TO_RIGHT);
      }
      pos = 0.05;
   }
   gtk_progress_bar_update ((GtkProgressBar *) sess->bar, pos);
   return 1;
}

void 
connect_server (struct session *sess, char *server, int port, int quiet)
{
   int sok, sw, pid, read_des[2];

   if (!server[0])
      return;

   sess = sess->server->front_session;

   if (sess->server->connected)
      disconnect_server (sess, TRUE);
   else
      check_connecting (sess);

   if (sess->op_box)
   {
      sess->bar = gtk_progress_bar_new ();
      gtk_box_pack_start (GTK_BOX (sess->op_box), sess->bar, 0, 0, 0);
      gtk_widget_show (sess->bar);
      sess->server->bartag = gtk_timeout_add (50, (GtkFunction) updatedate_bar, sess);
   }
   EMIT_SIGNAL (XP_TE_SERVERLOOKUP, sess, server, NULL, NULL, NULL, 0);

   sok = socket (AF_INET, SOCK_STREAM, 0);
   if (sok == -1)
      return;

   sw = 1;
   setsockopt (sok, SOL_SOCKET, SO_LINGER, (char *) &sw, sizeof (sw));
   sw = 1;
   setsockopt (sok, SOL_SOCKET, SO_REUSEADDR, (char *) &sw, sizeof (sw));
   sw = 1;
   setsockopt (sok, SOL_SOCKET, SO_KEEPALIVE, (char *) &sw, sizeof (sw));

   prefs.local_ip = 0;
   prefs.local_name[0] = 0;

   strcpy(sess->server->servername, server);
   sess->server->nickcount = 1;
   sess->server->connecting = TRUE;
   sess->server->sok = sok;
   sess->server->port = port;
   sess->server->end_of_motd = FALSE;
   flush_server_queue (sess->server);

   if (quiet)
      sess->server->no_login = TRUE;

#ifdef __EMX__                  /* if os/2 */
   if (pipe (read_des) < 0)
   {
      printf ("Pipe create error");
      exit (-1);
   }
   setmode (read_des[0], O_BINARY);
   setmode (read_des[1], O_BINARY);
#else
   socketpair (AF_UNIX, SOCK_STREAM, 0, read_des);
#endif

   /*signal (SIGCHLD, SIG_IGN);*/

   switch (pid = fork ())
   {
   case -1:
      return;

   case 0:
      {
         struct sockaddr_in SAddr;
         struct hostent *HostAddr;

         dup2 (read_des[1], 1);
         close (read_des[0]);
         setuid (getuid ());

         if (!prefs.detectip)
         {
            struct sockaddr_in DSAddr;
                  
            memset (&DSAddr, 0, sizeof (DSAddr));
            DSAddr.sin_family = AF_INET;
      
            if (prefs.ipaddress && strlen(prefs.ipaddress)
                && inet_aton (prefs.ipaddress, &(DSAddr.sin_addr))) 
            {
               /* Don't do anything - it worked */
            } else if (prefs.hostname && strlen(prefs.hostname))
            {
               struct hostent *HostAddr;
      
               HostAddr = gethostbyname (prefs.hostname);
               if (HostAddr)
               {
                  DSAddr.sin_addr = *((struct in_addr *) HostAddr->h_addr);
               }
            }
      
            if (DSAddr.sin_addr.s_addr)
            {
               if (!bind(sok, (struct sockaddr *) &DSAddr, sizeof (DSAddr)))
               {
                  printf ("5\n%s\n", inet_ntoa(DSAddr.sin_addr));
                  fflush (stdout);

                  /* doesn't affect master prefs - but helps us here */
                  prefs.local_ip = DSAddr.sin_addr.s_addr;
               }
            }
         }

         if (!prefs.detecthost)
         {
            if (prefs.hostname && strlen(prefs.hostname))
            {
               printf ("6\n%s\n", prefs.hostname);
               fflush (stdout);
            } else if (prefs.local_ip)
            {
               struct hostent *HostAddr;
      
               HostAddr = gethostbyaddr ((char *)&(prefs.local_ip),
                                         sizeof(struct in_addr), AF_INET);
               if (HostAddr)
               {
                  printf ("6\n%s\n", HostAddr->h_name);
                  fflush (stdout);
               }
            }
         }

         HostAddr = gethostbyname (server);
         if (HostAddr)
         {
            printf ("3\n%s\n%s\n%d\n", HostAddr->h_name,
                    inet_ntoa (*((struct in_addr *) HostAddr->h_addr)),
                    port);
            fflush (stdout);
            memset (&SAddr, 0, sizeof (SAddr));
            SAddr.sin_port = htons (port);
            SAddr.sin_family = AF_INET;
            memcpy ((void *) &SAddr.sin_addr, HostAddr->h_addr, HostAddr->h_length);
            if (connect (sok, (struct sockaddr *) &SAddr, sizeof (SAddr)) < 0)
               printf ("2\n%d\n", errno);
            else
            {
               if (prefs.detectip || prefs.detecthost)
               {
                  struct sockaddr_in DSAddr;
                  int len;
         
                  len = sizeof (DSAddr);
                  memset (&DSAddr, 0, len);
                  DSAddr.sin_family = AF_INET;
                  getsockname(sok, (struct sockaddr *)&DSAddr, &len);
         
                  if (prefs.detectip && !prefs.ip_from_server)
                  {
                     printf ("5\n%s\n", inet_ntoa(DSAddr.sin_addr));
                     fflush (stdout);
                  }
         
                  if (prefs.detecthost)
                  {
                     struct hostent *HostAddr;
           
                     HostAddr = gethostbyaddr ((char *)&(DSAddr.sin_addr.s_addr),
                                               sizeof(struct in_addr), AF_INET);
                     if (HostAddr)
                     {
                        printf ("6\n%s\n", HostAddr->h_name);
                        fflush (stdout);
                     }
                  }
               }
               printf ("4\n");
            }
         } else
            printf ("1\n");
         fflush (stdout);
         _exit (0);

   default:
         close (read_des[1]);
         break;
      }
   }
   sess->server->childpid = pid;
   sess->server->iotag = gdk_input_add (read_des[0], GDK_INPUT_READ,
                                        (GdkInputFunction) connected_signal, sess->server);
   sess->server->childread = read_des[0];
}
