#!/usr/bin/env python
# -*- coding: utf-8 -*-

#This file is part of SVGUIEditor, a library implementing an editor for SVGUI
#interfaces
#
#Copyright (C) 2007: Edouard TISSERANT, Laurent BESSARD and Jonathan HURTREL
#
#See COPYING file for copyrights details.
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#General Public License for more details.
#
#You should have received a copy of the GNU General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from xml.dom import minidom
import os, shutil

import wx
from svgui import *

[ITEM_CONTAINER, ITEM_BUTTON, ITEM_TEXT, ITEM_SCROLLBAR, ITEM_ROTATING, 
 ITEM_NOTEBOOK, ITEM_NOTEBOOKCONTENT, ITEM_TRANSFORM] = range(8)

def GetElementType(element):
    if isinstance(element, svgui.controls_Button):
        return ITEM_BUTTON
    elif isinstance(element, svgui.controls_ScrollBar):
        return ITEM_SCROLLBAR
    elif isinstance(element, svgui.controls_TextCtrl):
        return ITEM_TEXT
    elif isinstance(element, svgui.controls_NoteBook):
        return ITEM_NOTEBOOK
    elif isinstance(element, svgui.NoteBook_NoteBookContent):
        return ITEM_NOTEBOOKCONTENT
    elif isinstance(element, svgui.controls_RotatingCtrl):
        return ITEM_ROTATING
    elif isinstance(element, svgui.controls_Transform):
        return ITEM_TRANSFORM
    elif isinstance(element, svgui.Container):
        return ITEM_CONTAINER
    return ITEM_UNEDITABLE

def GetElementName(element):
    if isinstance(element, svgui.controls_Button):
        return "Button"
    elif isinstance(element, svgui.controls_ScrollBar):
        return "ScrollBar"
    elif isinstance(element, svgui.controls_TextCtrl):
        return "TextCtrl"
    elif isinstance(element, svgui.controls_NoteBook):
        return "NoteBook"
    elif isinstance(element, svgui.NoteBook_NoteBookContent):
        return "NoteBookContent"
    elif isinstance(element, svgui.controls_RotatingCtrl):
        return "RotatingCtrl"
    elif isinstance(element, svgui.controls_Transform):
        return "Transform"
    elif isinstance(element, svgui.Container):
        return "Container"
    return None

#-------------------------------------------------------------------------------
#                      SVGUI Program Code Generation Bas Class
#-------------------------------------------------------------------------------    

class SVGUIGenerator:

    def __init__(self, elements, svgfile, xmlfile):
        self.Elements = elements
        if wx.Platform == "__WXMSW__":
            self.SVGFilePath = svgfile.replace("\\", "\\\\")
            self.XMLFilePath = xmlfile.replace("\\", "\\\\")
        else:
            self.SVGFilePath = svgfile
            self.XMLFilePath = xmlfile
        
    def GenerateProgram(self, size, folder, progname="program"):
        pass

class SVGUICGenerator(SVGUIGenerator):

    def GenerateProgramHeaders(self, folder, progname):
        texts = {"h_define" : "%s_H"%progname.upper()}
        
        text = """#ifndef %(h_define)s
#define %(h_define)s

#include <wx/wx.h>
#include <wxSVG/svgctrl.h>
#include <wxSVGXML/svgxml.h>
#include <wxSVG/SVGUIWindow.h>

class Program: public SVGUIWindow
{
  public:
    Program(wxWindow* parent);
    void InitScrollBars();
"""%texts

        for element in self.Elements:
            element_type = GetElementType(element)
            if element_type == ITEM_BUTTON:
                text += "    void On%sClick(wxCommandEvent& event);\n"%element.getname()
            elif element_type in [ITEM_ROTATING, ITEM_SCROLLBAR, ITEM_TRANSFORM]:
                text += "    void On%sChanging(wxScrollEvent& event);\n"%element.getname()
            elif element_type == ITEM_NOTEBOOK:
                text += "    void On%sTabChanged(wxNotebookEvent& event);\n"%element.getname()
            
        text += self.GenerateProgramHeadersPublicVars()
        
        text += "  private:\n"
        
        text += self.GenerateProgramHeadersPrivateVars()
        
        text += """    DECLARE_EVENT_TABLE()
};

class MainFrame: public wxFrame
{
  public:
    MainFrame(wxWindow *parent, const wxString& title, const wxPoint& pos, const wxSize& size, long style = wxDEFAULT_FRAME_STYLE);
    Program* m_svgCtrl;
};

class SVGViewApp: public wxApp
{
  public:
    MainFrame* frame;
    bool OnInit();
};

#endif /*%(h_define)s*/
"""%texts
        
        hfilepath = os.path.join(folder, progname+".h")
        #DEBUG
        #hfilepath = "/taf/Pim/workspace_john/wxsvg/svgui/svgui-test.h"
        hfile = open(hfilepath,"w")
        hfile.write(text)
        hfile.close()
    
    def GenerateProgramHeadersPublicVars(self):
        return ""
    
    def GenerateProgramHeadersPrivateVars(self):
        return ""
    
    def GenerateProgramCFile(self, size, folder, progname):
        text = """//Compile this file with :
//gcc `wx-config-2.6 --cppflags --libs` -lwxsvg program.cpp -o program

#if defined(__GNUG__) && !defined(__APPLE__)
#pragma implementation
#pragma interface
#endif

#include "wx/wxprec.h"

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include "wx/wx.h"
#include <wx/filename.h>
#endif

#ifndef __WXMSW__
#include <locale.h>
#endif

"""
        #text += "#include \"svgui-test.h\"\n" #DEBUG
        text += "#include \"%s.h\"\n"%progname
        text += "#include <wxSVG/svg.h>\n\n\n"
        
        text += self.GenerateGlobalVarsAndFuncs(size)
            
        text += self.GenerateProgramEventTable()
            
        text += self.GenerateProgramInitFrame()
        
        text += self.GenerateProgramInitProgram()
        
        text += """void Program::InitScrollBars()
{
  SVGUIScrollBar* scrollbar;
"""
        
        for element in self.Elements:
            element_type = GetElementType(element)
            if element_type == ITEM_SCROLLBAR:
                text += """  scrollbar = (SVGUIScrollBar*)GetElementById(wxT("%d"));
  scrollbar->Init_ScrollBar(0, 1, 10);
"""%element.getid()
        text +="}\n\n"
        
        text += self.GenerateProgramEventFunctions()
            
        text += self.GenerateProgramPrivateFunctions()
            
        cfilepath = os.path.join(folder, progname + ".cpp")
        #cfilepath = "/taf/Pim/workspace_john/wxsvg/svgui/svgui-test.cpp"#DEBUG
        cfile = open(cfilepath,"w")
        cfile.write(text)
        cfile.close()
            
    def GenerateGlobalVarsAndFuncs(self, size):
        text = """IMPLEMENT_APP(SVGViewApp)

bool SVGViewApp::OnInit()
{
#ifndef __WXMSW__
  setlocale(LC_NUMERIC, "C");
#endif
  //wxLog::SetActiveTarget(new wxLogStderr);
  new MainFrame(NULL, wxT("Program"),"""
        #text+= "    wxDefaultPosition, wxDefaultSize);\n"
        text += """
    wxDefaultPosition, wxSize(%d, %d));
  return true;
}

"""%size

        return text
        
    def GenerateProgramEventTable(self):
        text = "BEGIN_EVENT_TABLE(Program, SVGUIWindow)\n"
        for element in self.Elements:
            element_type = GetElementType(element)
            element_name = element.getname()
            if element_type == ITEM_BUTTON:
                text += "  EVT_BUTTON (SVGUIID(\"%s\"), Program::On%sClick)\n"%(element_name, element_name)
            elif element_type in [ITEM_SCROLLBAR, ITEM_ROTATING, ITEM_TRANSFORM]:
                text += "  EVT_COMMAND_SCROLL_THUMBTRACK (SVGUIID(\"%s\"), Program::On%sChanging)\n"%(element_name, element_name)
                text += "  EVT_SCROLL_THUMBRELEASE (SVGUIID(\"%s\"), Program::On%sChanged)\n"%(element_name, element_name)
            elif element_type == ITEM_NOTEBOOK:
                text += "  EVT_NOTEBOOK_PAGE_CHANGED (SVGUIID(\"%s\"), Program::On%sTabChanged)\n"%(element_name, element_name)
        text += "END_EVENT_TABLE()\n\n"
        return text
        
    def GenerateProgramInitFrame(self):
        text = """MainFrame::MainFrame(wxWindow *parent, const wxString& title, const wxPoint& pos,const wxSize& size, long style): wxFrame(parent, wxID_ANY, title, pos, size, style)
{
  wxFileName apppath(wxTheApp->argv[0]);
  wxFileName svgfilepath(apppath.GetPath(), wxT("%s"));
  wxFileName xmlfilepath(apppath.GetPath(), wxT("%s"));

  m_svgCtrl = new Program(this);
  if (m_svgCtrl->LoadFiles(svgfilepath.GetFullPath(), xmlfilepath.GetFullPath()))
  {
    Show(true);
    m_svgCtrl->SetFocus();
    m_svgCtrl->SetFitToFrame(true);
    m_svgCtrl->InitScrollBars();
    m_svgCtrl->Update();
  }
  else
  {
    printf("Error while opening files\\n");
    exit(0);
  }
}


"""%(self.SVGFilePath, self.XMLFilePath)

        return text
    
    def GenerateProgramInitProgram(self):
        text = "Program::Program(wxWindow* parent):SVGUIWindow(parent)\n{\n"
        text += "}\n\n"
        return text
    
    def GenerateProgramEventFunctions(self):
        text = ""
        
        for element in self.Elements:
            element_type = GetElementType(element)
            element_name = element.getname()
            if element_type == ITEM_BUTTON:
                text += """void Program::On%sClick(wxCommandEvent& event)
{
  SVGUIButton* button = (SVGUIButton*)GetElementByName(wxT("%s"));
  event.Skip();
}

"""%(element_name, element_name)
            elif element_type == ITEM_SCROLLBAR:
                text += """void Program::On%sChanging(wxScrollEvent& event)
{
  SVGUIScrollBar* scrollbar = (SVGUIScrollBar*)GetElementByName(wxT("%s"));
  unsigned int position = scrollbar->GetThumbPosition();
}

void Program::On%sChanged(wxScrollEvent& event)
{
  SVGUIScrollBar* scrollbar = (SVGUIScrollBar*)GetElementByName(wxT("%s"));
  unsigned int position = scrollbar->GetThumbPosition();
}

"""%(element_name, element_name, element_name, element_name)
            elif element_type == ITEM_ROTATING:
                text += """void Program::On%sChanging(wxScrollEvent& event)
{
  SVGUIRotatingCtrl* rotating = (SVGUIRotatingCtrl*)GetElementByName(wxT("%s"));
  double angle = rotating->GetAngle();
}

void Program::On%sChanged(wxScrollEvent& event)
{
  SVGUIRotatingCtrl* rotating = (SVGUIRotatingCtrl*)GetElementByName(wxT("%s"));
  double angle = rotating->GetAngle();
}

"""%(element_name, element_name, element_name, element_name)
            elif element_type == ITEM_TRANSFORM:
                text += """void Program::On%sChanging(wxScrollEvent& event)
{
  SVGUITransform* transform = (SVGUITransform*)GetElementByName(wxT("%s"));
  double xpos = transform->GetX();
  double ypos = transform->GetY();
}

void Program::On%sChanged(wxScrollEvent& event)
{
  SVGUITransform* transform = (SVGUITransform*)GetElementByName(wxT("%s"));
  double xpos = transform->GetX();
  double ypos = transform->GetY();
}

"""%(element_name, element_name, element_name, element_name)

            elif element_type == ITEM_NOTEBOOK:
                text += """void Program::On%sTabChanged(wxNotebookEvent& event)
{
  SVGUINoteBook* notebook = (SVGUINoteBook*)GetElementByName(wxT("%s"));
  unsigned int selected = notebook->GetCurrentPage();
}

"""%(element_name, element_name)
        
        return text
    
    def GenerateProgramPrivateFunctions(self):
        return ""
        
    def GenerateProgram(self, size, folder, progname="program"):
        self.GenerateProgramHeaders(folder, progname)
        self.GenerateProgramCFile(size, folder, progname)



def AppendScriptElement(document, node, script_name):
    script = document.createElement("script")
    script.setAttribute("xlink:href", script_name)
    script.setAttribute("type", "text/javascript")
    node.appendChild(script)

class SVGUIHaxeGenerator(SVGUIGenerator):
    
    def GenerateProgramFile(self, folder, progname):
        text = ""
        for element in self.Elements:
            element_type = GetElementType(element)
            if element_type == ITEM_BUTTON:
                text += "function On%sClick() {\n}\n\n"%element.getname()
            elif element_type in [ITEM_ROTATING, ITEM_SCROLLBAR, ITEM_TRANSFORM]:
                text += "function On%sChanging() {\n}\n\n"%element.getname()
            elif element_type == ITEM_NOTEBOOK:
                text += "function On%sTabChanged() {\n}\n\n"%element.getname()
    
        programfile = open(os.path.join(folder, "%s.js"%progname), 'w')
        programfile.write(text)
        programfile.close()
    
    def GenerateSVGFile(self, folder, progname):
        svgfile = open(self.SVGFilePath, 'r')
        tree = minidom.parse(svgfile)
        svgfile.close()
        
        for child in tree.childNodes:
            if child.nodeType == tree.ELEMENT_NODE and child.nodeName == "svg":
                AppendScriptElement(tree, child, "%s.js"%progname)
                AppendScriptElement(tree, child, "svgui.js")
                AppendScriptElement(tree, child, "resource.js")
        
        svgfile = open(os.path.join(folder, os.path.split(self.SVGFilePath)[1]), 'w')
        svgfile.write(tree.toxml())
        svgfile.close()
    
    def GenerateResourceFile(self, folder):
        xmlfile = open(self.XMLFilePath, 'r')
        lines = xmlfile.readlines()
        xmlfile.close()
        
        text = "js.Boot.__res = {}\n"
        text += "js.Boot.__res[\"fichierXML\"] = \""
        for line in lines:
            text += line.strip().replace("\"", "\\\"")
        text += "\";\n"
         
        resourcefile = open(os.path.join(folder, "resource.js"), 'w')
        resourcefile.write(text)
        resourcefile.close()
    
    def CopySVGUILibrary(self, folder):
        shutil.copy("svgui.js", folder)
    
    def GenerateProgram(self, size, folder, progname="program"):
        self.GenerateProgramFile(folder, progname)
        self.GenerateSVGFile(folder, progname)
        self.GenerateResourceFile(folder)
        self.CopySVGUILibrary(folder)

