#!/usr/bin/env python
# -*- coding: utf-8 -*-

#This file is part of PLCOpenEditor, a library implementing an IEC 61131-3 editor
#based on the plcopen standard. 
#
#Copyright (C): Edouard TISSERANT and Laurent BESSARD
#
#See COPYING file for copyrights details.
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#Lesser General Public License for more details.
#
#You should have received a copy of the GNU General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import wx
from math import *


#-------------------------------------------------------------------------------
#                               Common constants
#-------------------------------------------------------------------------------

"""
Definition of constants for dimensions of graphic elements
"""

# FBD and SFC constants
MIN_MOVE = 5            # Minimum move before starting a element dragging
CONNECTOR_SIZE = 8      # Size of connectors
BLOCK_LINE_SIZE = 20    # Minimum size of each line in a block
HANDLE_SIZE = 6         # Size of the squares for handles
ANCHOR_DISTANCE = 5     # Distance where wire is automativally attached to a connector
POINT_RADIUS = 2        # Radius of the point of wire ends
MIN_SEGMENT_SIZE = 2    # Minimum size of the endling segments of a wire

# LD constants
LD_LINE_SIZE = 40                   # Distance between two lines in a ladder rung
LD_ELEMENT_SIZE = (21, 15)          # Size (width, height) of a ladder element (contact or coil)
LD_WIRE_SIZE = 30                   # Size of a wire between two contact
LD_WIRECOIL_SIZE = 70               # Size of a wire between a coil and a contact
LD_OFFSET = (10, 10)                # Distance (x, y) between each comment and rung of the ladder
LD_COMMENT_DEFAULTSIZE = (600, 40)  # Size (width, height) of a comment box

# SFC constants
SFC_STEP_DEFAULT_SIZE = (40, 30)      # Default size of a SFC step
SFC_TRANSITION_SIZE = (20, 2)         # Size of a SFC transition
SFC_DEFAULT_SEQUENCE_INTERVAL = 80    # Default size of the interval between two divergence branches
SFC_SIMULTANEOUS_SEQUENCE_EXTRA = 20  # Size of extra lines for simultaneous divergence and convergence
SFC_JUMP_SIZE = (12, 13)              # Size of a SFC jump to step
SFC_WIRE_MIN_SIZE = 25                # Size of a wire between two elements
SFC_ACTION_MIN_SIZE = (100, 30)       # Minimum size of an action block line

# Type definition constants for graphic elements
[INPUT, OUTPUT, INOUT] = range(3)
[CONNECTOR, CONTINUATION] = range(2)
[LEFTRAIL, RIGHTRAIL] = range(2)
[CONTACT_NORMAL, CONTACT_REVERSE, CONTACT_RISING, CONTACT_FALLING] = range(4)
[COIL_NORMAL, COIL_REVERSE, COIL_SET, COIL_RESET] = range(4)
[SELECTION_DIVERGENCE, SELECTION_CONVERGENCE, SIMULTANEOUS_DIVERGENCE, SIMULTANEOUS_CONVERGENCE] = range(4)

# Constants for defining the type of dragging that has been selected
[HANDLE_MOVE, HANDLE_RESIZE, HANDLE_POINT, HANDLE_SEGMENT, HANDLE_CONNECTOR] = range(5)

# List of value for resize handle that are valid
VALID_HANDLES = [(1,1), (1,2), (1,3), (2,3), (3,3), (3,2), (3,1), (2,1)]

# Contants for defining the direction of a connector
[EAST, NORTH, WEST, SOUTH] = [(1,0), (0,-1), (-1,0), (0,1)]

# Contants for defining which mode is selected for each view 
[MODE_SELECTION, MODE_BLOCK, MODE_VARIABLE, MODE_CONNECTION, MODE_COMMENT, MODE_WIRE,
 MODE_INITIAL_STEP] = range(7)

"""
Basic vector operations for calculate wire points
"""

# Calculate the scalar product of two vectors
def product(v1, v2):
    return v1[0] * v2[0] + v1[1] * v2[1]

# Create a vector from two points and define if vector must be normal
def vector(p1, p2, normal = True):
    vector = (p2.x - p1.x, p2.y - p1.y)
    if normal:
        return normalize(vector)
    return vector

# Calculate the norm of a given vector
def norm(v):
    return sqrt(v[0] * v[0] + v[1] * v[1])

# Normalize a given vector
def normalize(v):
    v_norm = norm(v)
    # Verifie if it is not a null vector
    if v_norm > 0:
        return (v[0] / v_norm, v[1] / v_norm)
    else:
        return v


"""
Function that calculates the nearest point of the grid defined by scaling for the given point
"""

def GetScaledEventPosition(event, scaling):
    pos = event.GetPosition()
    if scaling:
        pos.x = round(float(pos.x) / float(scaling[0])) * scaling[0]
        pos.y = round(float(pos.y) / float(scaling[1])) * scaling[1]
    return pos


"""
Function that choose a direction during the wire points generation
"""

def DirectionChoice(v_base, v_target, dir_target):
    dir_product = product(v_base, v_target)
    if dir_product < 0:
        return (-v_base[0], -v_base[1])
    elif dir_product == 0 and product(v_base, dir_target) != 0:
        return dir_target
    return v_base


#-------------------------------------------------------------------------------
#                               Viewer Rubberband
#-------------------------------------------------------------------------------

"""
Class that implements a rubberband
"""

class RubberBand:
    
    # Create a rubberband by indicated on which window it must be drawn
    def __init__(self, drawingSurface):
        self.drawingSurface = drawingSurface
        self.Reset()
    
    # Method that initializes the internal attributes of the rubberband
    def Reset(self):
        self.startPoint = None
        self.currentBox = None
        self.lastBox = None
    
    # Method that return if a box is currently edited
    def IsShown(self):
        return self.currentBox != None
    
    # Method that returns the currently edited box
    def GetCurrentExtent(self):
        return self.currentBox
    
    # Method called when a new box starts to be edited
    def OnLeftDown(self, event, scaling):
        pos = GetScaledEventPosition(event, scaling)
        # Save the point for calculate the box position and size
        self.startPoint = pos
        self.currentBox = wxRect(pos.x, pos.y, 0, 0)
        self.drawingSurface.SetCursor(wxStockCursor(wxCURSOR_CROSS))
        self.Redraw()
    
    # Method called when dragging with a box edited
    def OnMotion(self, event, scaling):
        pos = GetScaledEventPosition(event, scaling)
        # Save the last position and size of the box for erasing it
        self.lastBox = wxRect(self.currentBox.x, self.currentBox.y, self.currentBox.width,
            self.currentBox.height)
        # Calculate new position and size of the box 
        if pos.x >= self.startPoint.x:
            self.currentBox.x = self.startPoint.x
            self.currentBox.width = pos.x - self.startPoint.x + 1
        else:
            self.currentBox.x = pos.x
            self.currentBox.width = self.startPoint.x - pos.x + 1
        if pos.y >= self.startPoint.y:
            self.currentBox.y = self.startPoint.y
            self.currentBox.height = pos.y - self.startPoint.y + 1
        else:
            self.currentBox.y = pos.y
            self.currentBox.height = self.startPoint.y - pos.y + 1
        self.Redraw()
    
    # Method called when dragging is stopped
    def OnLeftUp(self, event, scaling):
        self.drawingSurface.SetCursor(wxNullCursor)
        self.lastBox = self.currentBox
        self.currentBox = None
        self.Redraw()

    # Method that erase the last box and draw the new box
    def Redraw(self):
        dc = wxClientDC(self.drawingSurface)
        dc.SetPen(wxPen(wxWHITE, 1, wxDOT))
        dc.SetBrush(wxTRANSPARENT_BRUSH)
        dc.SetLogicalFunction(wxXOR)
        if self.lastBox:
            # Erase last box
            dc.DrawRectangle(self.lastBox.x, self.lastBox.y, self.lastBox.width,
                self.lastBox.height)
        if self.currentBox:
            # Draw current box
            dc.DrawRectangle(self.currentBox.x, self.currentBox.y, self.currentBox.width,
                self.currentBox.height)


#-------------------------------------------------------------------------------
#                           Graphic element base class
#-------------------------------------------------------------------------------

"""
Class that implements a generic graphic element
"""

class Graphic_Element:
    
    # Create a new graphic element
    def __init__(self, parent, id = None):
        self.Parent = parent
        self.Id = id
        self.oldPos = None
        self.Handle = False
        self.Dragging = False
        self.Selected = False
        self.Pos = wxPoint(0, 0)
        self.Size = wxSize(0, 0)
        self.BoundingBox = wxRect(0, 0, 0, 0)
    
    # Make a clone of this element
    def Clone(self):
        return Graphic_Element(self.Parent, self.Id)
    
    # Changes the block position
    def SetPosition(self, x, y):
        self.Pos.x = x
        self.Pos.y = y
        self.RefreshConnected()
        self.RefreshBoundingBox()

    # Returns the block position
    def GetPosition(self):
        return self.Pos.x, self.Pos.y
    
    # Changes the element size
    def SetSize(self, width, height):
        self.Size.SetWidth(width)
        self.Size.SetHeight(height)
        self.RefreshConnectors()
        self.RefreshBoundingBox()

    # Returns the element size
    def GetSize(self):
        return self.Size.GetWidth(), self.Size.GetHeight()
    
    # Refresh the element Bounding Box
    def RefreshBoundingBox(self):
        self.BoundingBox = wxRect(self.Pos.x, self.Pos.y, self.Size[0], self.Size[1])
    
    # Refresh the element connectors position
    def RefreshConnectors(self):
        pass
    
    # Refresh the position of wires connected to element inputs and outputs
    def RefreshConnected(self):
        pass
    
    # Change the parent
    def SetParent(self, parent):
        self.Parent = parent
    
    # Override this method for defining the method to call for deleting this element
    def Delete(self):
        pass
    
    # Returns the Id
    def GetId(self):
        return self.Id
    
    # Returns if the point given is in the bounding box
    def HitTest(self, pt):
        rect = self.BoundingBox
        return rect.InsideXY(pt.x, pt.y)
    
    # Override this method for refreshing the bounding box
    def RefreshBoundingBox(self):
        pass
    
    # Returns the bounding box
    def GetBoundingBox(self):
        return self.BoundingBox
    
    # Change the variable that indicates if this element is selected
    def SetSelected(self, selected):
        self.Selected = selected
    
    # Test if the point is on a handle of this element
    def TestHandle(self, pt):
        # Verify that this element is selected
        if self.Selected:
            # Find if point is on a handle horizontally
            if self.BoundingBox.x - HANDLE_SIZE - 2 <= pt.x < self.BoundingBox.x - 2:
                handle_x = 1
            elif self.BoundingBox.x + (self.BoundingBox.width - HANDLE_SIZE) / 2 <= pt.x < self.BoundingBox.x + (self.BoundingBox.width + HANDLE_SIZE) / 2:
                handle_x = 2
            elif self.BoundingBox.x + self.BoundingBox.width + 2 <= pt.x < self.BoundingBox.x + self.BoundingBox.width + HANDLE_SIZE + 2:
                handle_x = 3
            else:
                handle_x = 0
            # Find if point is on a handle vertically
            if self.BoundingBox.y - HANDLE_SIZE - 2 <= pt.y < self.BoundingBox.y - 2:
                handle_y = 1
            elif self.BoundingBox.y + (self.BoundingBox.height - HANDLE_SIZE) / 2 <= pt.y < self.BoundingBox.y + (self.BoundingBox.height + HANDLE_SIZE) / 2:
                handle_y = 2
            elif self.BoundingBox.y + self.BoundingBox.height - 2 <= pt.y < self.BoundingBox.y + self.BoundingBox.height + HANDLE_SIZE + 2:
                handle_y = 3
            else:
                handle_y = 0
            # Verify that the result is valid
            if (handle_x, handle_y) in VALID_HANDLES:
                return handle_x, handle_y
        return 0, 0
    
    # Method called when a LeftDown event have been generated
    def OnLeftDown(self, event, scaling):
        pos = event.GetPosition()
        # Test if an handle have been clicked
        result = self.TestHandle(pos)
        # Find which type of handle have been clicked,
        # Save a resize event and change the cursor
        if result == (1, 1) or result == (3, 3):
            self.Handle = (HANDLE_RESIZE, result)
            self.Parent.SetCursor(wxStockCursor(wxCURSOR_SIZENWSE))
        elif result == (1, 3) or result == (3, 1):
            self.Handle = (HANDLE_RESIZE, result)
            self.Parent.SetCursor(wxStockCursor(wxCURSOR_SIZENESW))
        elif result == (1, 2) or result == (3, 2):
            self.Handle = (HANDLE_RESIZE, result)
            self.Parent.SetCursor(wxStockCursor(wxCURSOR_SIZEWE))
        elif result == (2, 1) or result == (2, 3):
            self.Handle = (HANDLE_RESIZE, result)
            self.Parent.SetCursor(wxStockCursor(wxCURSOR_SIZENS))
        # If no handle have been clicked, save a move event, and change the cursor
        else:
            self.Handle = (HANDLE_MOVE, None)
            self.Parent.SetCursor(wxStockCursor(wxCURSOR_HAND))
            self.SetSelected(False)
        # Initializes the last position
        self.oldPos = GetScaledEventPosition(event, scaling)
    
    # Method called when a LeftUp event have been generated
    def OnLeftUp(self, event, scaling):
        # If a dragging have been initiated
        if self.Dragging and self.oldPos:
            # Calculate the movement of cursor and refreshes the element state
            pos = GetScaledEventPosition(event, scaling)
            movex = pos.x - self.oldPos.x
            movey = pos.y - self.oldPos.y
            self.ProcessDragging(movex, movey)
            self.RefreshModel()
        self.SetSelected(True)
        self.oldPos = None

    # Method called when a RightUp event have been generated
    def OnRightUp(self, event, scaling):
        self.SetSelected(True)
        self.oldPos = None

    # Method called when a LeftDClick event have been generated
    def OnLeftDClick(self, event, scaling):
        pass
    
    # Method called when a Motion event have been generated
    def OnMotion(self, event, scaling):
        # If the cursor is dragging and the element have been clicked
        if event.Dragging() and self.oldPos:
            # Calculate the movement of cursor
            pos = GetScaledEventPosition(event, scaling)
            movex = pos.x - self.oldPos.x
            movey = pos.y - self.oldPos.y
            # If movement is greater than MIN_MOVE then a dragging is initiated
            if not self.Dragging and (abs(movex) > MIN_MOVE or abs(movey) > MIN_MOVE):
                self.Dragging = True
            # If a dragging have been initiated, refreshes the element state
            if self.Dragging:
                self.oldPos = pos
                self.ProcessDragging(movex, movey)
        # If cursor just pass over the element, changes the cursor if it is on a handle
        else:
            pos = event.GetPosition()
            handle = self.TestHandle(pos)
            if handle == (1, 1) or handle == (3, 3):
                wxCallAfter(self.Parent.SetCursor, wxStockCursor(wxCURSOR_SIZENWSE))
            elif handle == (1, 3) or handle == (3, 1):
                wxCallAfter(self.Parent.SetCursor, wxStockCursor(wxCURSOR_SIZENESW))
            elif handle == (1, 2) or handle == (3, 2):
                wxCallAfter(self.Parent.SetCursor, wxStockCursor(wxCURSOR_SIZEWE))
            elif handle == (2, 1) or handle == (2, 3):
                wxCallAfter(self.Parent.SetCursor, wxStockCursor(wxCURSOR_SIZENS))
            else:
                wxCallAfter(self.Parent.SetCursor, wxNullCursor)
    
    # Moves the element
    def Move(self, dx, dy, exclude = []):
        self.Pos.x += dx
        self.Pos.y += dy
        self.RefreshConnected(exclude)
        self.RefreshBoundingBox()
    
    # Resizes the element from position and size given
    def Resize(self, x, y, width, height):
        self.Move(x, y)
        self.SetSize(width, height)
    
    # Refreshes the element state according to move defined and handle selected
    def ProcessDragging(self, movex, movey):
        handle_type, handle = self.Handle
        # If it is a resize handle, calculate the values from resizing
        if handle_type == HANDLE_RESIZE:
            x, y = 0, 0
            width, height = self.GetSize()
            if handle[0] == 1:
                x = movex
                width -= movex
            elif handle[0] == 3:
                width += movex
            if handle[1] == 1:
                y = movey
                height -= movey
            elif handle[1] == 3:
                height += movey
            # Verify that new size is not lesser than minimum
            min_width, min_height = self.GetMinSize()
            if width >= min_width and height >= min_height:
                self.Resize(x, y, width, height)
        # If it is a move handle, Move this element
        elif handle_type == HANDLE_MOVE:
            self.Move(movex, movey)
    
    # Override this method for defining the method to call for refreshing the model of this element
    def RefreshModel(self, move=True):
        pass
    
    # Draws the handles of this element if it is selected
    def Draw(self, dc):
        if self.Selected:
            dc.SetPen(wxBLACK_PEN)
            dc.SetBrush(wxBLACK_BRUSH)
            dc.DrawRectangle(self.BoundingBox.x - HANDLE_SIZE - 2, self.BoundingBox.y - HANDLE_SIZE - 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x + (self.BoundingBox.width - HANDLE_SIZE) / 2,
                self.BoundingBox.y - HANDLE_SIZE - 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x + self.BoundingBox.width + 2, 
                self.BoundingBox.y - HANDLE_SIZE - 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x + self.BoundingBox.width + 2, 
                self.BoundingBox.y + (self.BoundingBox.height - HANDLE_SIZE) / 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x + self.BoundingBox.width + 2, 
                self.BoundingBox.y + self.BoundingBox.height + 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x + (self.BoundingBox.width - HANDLE_SIZE) / 2, 
                self.BoundingBox.y + self.BoundingBox.height + 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x - HANDLE_SIZE - 2, self.BoundingBox.y + self.BoundingBox.height + 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.DrawRectangle(self.BoundingBox.x - HANDLE_SIZE - 2, self.BoundingBox.y + (self.BoundingBox.height - HANDLE_SIZE) / 2, HANDLE_SIZE, HANDLE_SIZE)
            dc.SetBrush(wxWHITE_BRUSH)


#-------------------------------------------------------------------------------
#                           Group of graphic elements
#-------------------------------------------------------------------------------

"""
Class that implements a group of graphic elements
"""

class Graphic_Group(Graphic_Element):
    
    # Create a new group of graphic elements
    def __init__(self, parent):
        Graphic_Element.__init__(self, parent)
        self.Elements = []
        self.RefreshBoundingBox()
    
    # Destructor
    def __del__(self):
        self.Elements = []
    
    # Make a clone of this group
    def Clone(self):
        clone = Graphic_Group(self.Parent)
        elements = []
        # Makes a clone of all the elements in this group
        for element in self.Elements:
            elements.append(element.Clone())
        clone.SetElements(elements)
        return clone
    
    # Clean this group of elements
    def Clean(self):
        # Clean all the elements of the group
        for element in self.Elements:
            element.Clean()
    
    # Delete this group of elements
    def Delete(self):
        # Delete all the elements of the group
        for element in self.Elements:
            element.Delete()
    
    # Returns if the point given is in the bounding box of one of the elements of this group
    def HitTest(self, pt):
        result = False
        for element in self.Elements:
            result |= element.HitTest(pt)
        return result
    
    # Returns if the element given is in this group
    def IsElementIn(self, element):
        return element in self.Elements
    
    # Change the elements of the group
    def SetElements(self, elements):
        self.Elements = elements
        self.RefreshBoundingBox()
    
    # Returns the elements of the group
    def GetElements(self):
        return self.Elements
    
    # Remove or select the given element if it is or not in the group
    def SelectElement(self, element):
        if element in self.Elements:
            self.Elements.remove(element)
        else:
            self.Elements.append(element)
        self.RefreshBoundingBox()
    
    # Move this group of elements
    def Move(self, movex, movey):
        exclude = []
        for element in self.Elements:
            if isinstance(element, Wire):
                exclude.append(element)
        # Move all the elements of the group
        for element in self.Elements:
            if isinstance(element, Wire):
                element.Move(movex, movey, True)
            else:
                element.Move(movex, movey, exclude)
        self.RefreshBoundingBox()
    
    # Refreshes the bounding box of this group of elements
    def RefreshBoundingBox(self):
        if len(self.Elements) > 0:
            bbox = self.Elements[0].GetBoundingBox()
            minx, miny = bbox.x, bbox.y
            maxx = bbox.x + bbox.width
            maxy = bbox.y + bbox.height
            for element in self.Elements[1:]:
                bbox = element.GetBoundingBox()
                minx = min(minx, bbox.x)
                miny = min(miny, bbox.y)
                maxx = max(maxx, bbox.x + bbox.width)
                maxy = max(maxy, bbox.y + bbox.height)
            self.BoundingBox = wxRect(minx, miny, maxx - minx, maxy - miny)
        else:
            self.BoundingBox = wxRect(0, 0, 0, 0)

    # Forbids to change the group position
    def SetPosition(x, y):
        pass
    
    # Returns the position of this group
    def GetPosition(self):
        return self.BoundingBox.x, self.BoundingBox.y
    
    # Forbids to change the group size
    def SetSize(width, height):
        pass
    
    # Returns the size of this group
    def GetSize(self):
        return self.BoundingBox.width, self.BoundingBox.height

    # Change the variable that indicates if the elemente is selected
    def SetSelected(self, selected):
        for element in self.Elements:
            element.SetSelected(selected)

    # Refreshes the model of all the elements of this group
    def RefreshModel(self):
        for element in self.Elements:
            element.RefreshModel()


#-------------------------------------------------------------------------------
#                         Connector for all types of blocks
#-------------------------------------------------------------------------------

"""
Class that implements a connector for any type of block
"""

class Connector:
    
    # Create a new connector
    def __init__(self, parent, name, type, position, direction, negated = False, edge = "none"):
        self.ParentBlock = parent
        self.Name = name
        self.Type = type
        self.Pos = position
        self.Direction = direction
        self.Wires = []
        self.Negated = negated
        self.Edge = edge
        self.Pen = wxBLACK_PEN
    
    # Change the connector pen
    def SetPen(self, pen):
        self.Pen = pen
    
    # Make a clone of the connector
    def Clone(self):
        return Connector(self.Parent, self.Name, self.Type, wxPoint(self.Pos[0], self.Pos[1]),
                self.Direction, self.Negated)
    
    # Returns the connector parent block
    def GetParentBlock(self):
        return self.ParentBlock
    
    # Returns the connector name
    def GetType(self):
        return self.Type
    
    # Changes the connector name
    def SetType(self, type):
        self.Type = type
    
    # Returns the connector name
    def GetName(self):
        return self.Name
    
    # Changes the connector name
    def SetName(self, name):
        self.Name = name
    
    # Returns the wires connected to the connector
    def GetWires(self):
        return self.Wires
    
    # Returns the parent block Id
    def GetBlockId(self):
        return self.ParentBlock.GetId()
    
    # Returns the connector relative position
    def GetRelPosition(self):
        return self.Pos
    
    # Returns the connector absolute position
    def GetPosition(self, size = True):
        parent_pos = self.ParentBlock.GetPosition()
        # If the position of the end of the connector is asked
        if size:
            x = parent_pos[0] + self.Pos.x + self.Direction[0] * CONNECTOR_SIZE
            y = parent_pos[1] + self.Pos.y + self.Direction[1] * CONNECTOR_SIZE
        else:
            x = parent_pos[0] + self.Pos.x
            y = parent_pos[1] + self.Pos.y
        return wxPoint(x, y)
    
    # Change the connector relative position
    def SetPosition(self, pos):
        self.Pos = pos
    
    # Returns the connector direction
    def GetDirection(self):
        return self.Direction
    
    # Change the connector direction
    def SetDirection(self, direction):
        self.Direction = direction
    
    # Connect a wire to this connector at the last place
    def Connect(self, wire, refresh = True):
        self.InsertConnect(len(self.Wires), wire, refresh)
    
    # Connect a wire to this connector at the place given
    def InsertConnect(self, idx, wire, refresh = True):
        if wire not in self.Wires:
            self.Wires.insert(idx, wire)
            if refresh:
                self.ParentBlock.RefreshModel(False)
    
    # Returns the index of the wire given in the list of connected
    def GetWireIndex(self, wire):
        for i, (tmp_wire, handle) in enumerate(self.Wires):
            if tmp_wire == wire:
                return i
        return None
    
    # Unconnect a wire or all wires connected to the connector
    def UnConnect(self, wire = None, unconnect = True, delete = False):
        i = 0
        found = False
        while i < len(self.Wires) and not found:
            if not wire or self.Wires[i][0] == wire:
                # If Unconnect haven't been called from a wire, disconnect the connector in the wire
                if unconnect:
                    if self.Wires[i][1] == 0:
                        self.Wires[i][0].UnConnectStartPoint(delete)
                    else:
                        self.Wires[i][0].UnConnectEndPoint(delete)
                # Remove wire from connected
                if wire:
                    self.Wires.pop(i)
                    found = True
            i += 1
        # If no wire defined, unconnect all wires
        if not wire:
            self.Wires = []
        self.ParentBlock.RefreshModel(False)
    
    # Returns if connector has one or more wire connected
    def IsConnected(self):
        return len(self.Wires) > 0
    
    # Move the wires connected
    def MoveConnected(self, exclude = []):
        if len(self.Wires) > 0:
            # Calculate the new position of the end point
            parent_pos = self.ParentBlock.GetPosition()
            x = parent_pos[0] + self.Pos.x + self.Direction[0] * CONNECTOR_SIZE
            y = parent_pos[1] + self.Pos.y + self.Direction[1] * CONNECTOR_SIZE
            # Move the corresponding point on all the wires connected
            for wire, index in self.Wires:
                if wire not in exclude:
                    if index == 0:
                        wire.MoveStartPoint(wxPoint(x, y))
                    else:
                        wire.MoveEndPoint(wxPoint(x, y))
    
    # Refreshes the model of all the wires connected
    def RefreshWires(self):
        for wire in self.Wires:
            wire[0].RefreshModel()
    
    # Refreshes the parent block model
    def RefreshParentBlock(self):
        self.ParentBlock.RefreshModel(False)
    
    # Returns the connector negated property
    def IsNegated(self):
        return self.Negated
    
    # Changes the connector negated property
    def SetNegated(self, negated):
        self.Negated = negated
        self.Edge = "none"
    
    # Returns the connector edge property
    def GetEdge(self):
        return self.Edge
    
    # Changes the connector edge property
    def SetEdge(self, edge):
        self.Edge = edge
        self.Negated = False
    
    # Tests if the point given is near from the end point of this connector
    def TestPoint(self, pt, exclude = True):
        parent_pos = self.ParentBlock.GetPosition()
        if not (len(self.Wires) > 0 and self.Direction == WEST and exclude):
            # Calculate a square around the end point of this connector
            x = parent_pos[0] + self.Pos.x + self.Direction[0] * CONNECTOR_SIZE - ANCHOR_DISTANCE
            y = parent_pos[1] + self.Pos.y + self.Direction[1] * CONNECTOR_SIZE - ANCHOR_DISTANCE
            width = ANCHOR_DISTANCE * 2 + abs(self.Direction[0]) * CONNECTOR_SIZE
            height = ANCHOR_DISTANCE * 2 + abs(self.Direction[1]) * CONNECTOR_SIZE
            rect = wxRect(x, y, width, height)
            return rect.InsideXY(pt.x, pt.y)
        return False
    
    # Draws the connector
    def Draw(self, dc):
        dc.SetPen(self.Pen)
        dc.SetBrush(wxWHITE_BRUSH)
        parent_pos = self.ParentBlock.GetPosition()
        if self.Negated:
            # If connector is negated, draw a circle
            xcenter = parent_pos[0] + self.Pos.x + (CONNECTOR_SIZE * self.Direction[0]) / 2
            ycenter = parent_pos[1] + self.Pos.y + (CONNECTOR_SIZE * self.Direction[1]) / 2
            dc.DrawCircle(xcenter, ycenter, CONNECTOR_SIZE / 2)
        else:
            xstart = parent_pos[0] + self.Pos.x
            ystart = parent_pos[1] + self.Pos.y
            if self.Edge == "rising":
                # If connector has a rising edge, draw a right arrow
                dc.DrawLine(xstart, ystart, xstart - 4, ystart - 4)
                dc.DrawLine(xstart, ystart, xstart - 4, ystart + 4)
            elif self.Edge == "falling":
                # If connector has a falling edge, draw a left arrow
                dc.DrawLine(xstart, ystart, xstart + 4, ystart - 4)
                dc.DrawLine(xstart, ystart, xstart + 4, ystart + 4)
            xend = xstart + CONNECTOR_SIZE * self.Direction[0]
            yend = ystart + CONNECTOR_SIZE * self.Direction[1]
            dc.DrawLine(xstart + self.Direction[0], ystart + self.Direction[1], xend, yend)
        # Calculate the position of the text
        text_size = dc.GetTextExtent(self.Name)
        if self.Direction[0] != 0:
            ytext = parent_pos[1] + self.Pos.y - text_size[1] / 2
            if self.Direction[0] < 0:
                xtext = parent_pos[0] + self.Pos.x + 5
            else:
                xtext = parent_pos[0] + self.Pos.x - (text_size[0] + 5)
        if self.Direction[1] != 0:
            xtext = parent_pos[0] + self.Pos.x - text_size[0] / 2
            if self.Direction[1] < 0:
                ytext = parent_pos[1] + self.Pos.y + 5
            else:
                ytext = parent_pos[1] + self.Pos.y - (text_size[1] + 5)
        # Draw the text
        dc.DrawText(self.Name, xtext, ytext)


#-------------------------------------------------------------------------------
#                           Common Wire Element
#-------------------------------------------------------------------------------

"""
Class that implements a wire for connecting two blocks
"""

class Wire(Graphic_Element):
    
    # Create a new wire
    def __init__(self, parent, start = None, end = None):
        Graphic_Element.__init__(self, parent)
        self.StartPoint = start
        self.EndPoint = end
        self.StartConnected = None
        self.EndConnected = None
        # If the start and end points are defined, calculate the wire
        if start and end:
            self.ResetPoints()
            self.GeneratePoints()
        else:
            self.Points = []
            self.Segments = []
        self.SelectedSegment = None
        self.OverStart = False
        self.OverEnd = False
    
    # Destructor of a wire
    def __del__(self):
        self.StartConnected = None
        self.EndConnected = None
    
    # Forbids to change the wire position
    def SetPosition(x, y):
        pass
    
    # Forbids to change the wire size
    def SetSize(width, height):
        pass
    
    # Unconnect the start and end points
    def Clean(self):
        if self.StartConnected:
            self.UnConnectStartPoint()
        if self.EndConnected:
            self.UnConnectEndPoint()
    
    # Delete this wire by calling the corresponding method
    def Delete(self):
        self.Parent.DeleteWire(self)
    
    # Select a segment and not the whole wire. It's useful for Ladder Diagram
    def SetSelectedSegment(self, segment):
        # The last segment is indicated
        if segment == -1:
            segment = len(self.Segments) - 1
        # The selected segment is reinitialised
        if segment == None:
            if self.StartConnected:
                self.StartConnected.SetPen(wxBLACK_PEN)
            if self.EndConnected:
                self.EndConnected.SetPen(wxBLACK_PEN)
        # The segment selected is the first
        elif segment == 0:
            if self.StartConnected:
                self.StartConnected.SetPen(wxRED_PEN)
            if self.EndConnected:
                # There is only one segment
                if len(self.Segments) == 1:
                    self.EndConnected.SetPen(wxRED_PEN)
                else:
                    self.EndConnected.SetPen(wxBLACK_PEN)
        # The segment selected is the last
        elif segment == len(self.Segments) - 1:
            if self.StartConnected:
                self.StartConnected.SetPen(wxBLACK_PEN)
            if self.EndConnected:
                self.EndConnected.SetPen(wxRED_PEN)
        self.SelectedSegment = segment
    
    # Reinitialize the wire points
    def ResetPoints(self):
        if self.StartPoint and self.EndPoint:
            self.Points = [self.StartPoint[0], self.EndPoint[0]]
            self.Segments = [self.StartPoint[1]]
        else:
            self.Points = []
            self.Segments = []
    
    # Refresh the wire bounding box
    def RefreshBoundingBox(self):
        if len(self.Points) > 0:
            # If startpoint or endpoint is connected, save the point radius
            start_radius = end_radius = 0
            if not self.StartConnected:
                start_radius = POINT_RADIUS
            if not self.EndConnected:
                end_radius = POINT_RADIUS
            # Initialize minimum and maximum from the first point
            minx, minbbxx = self.Points[0].x, self.Points[0].x - start_radius
            maxx, maxbbxx = self.Points[0].x, self.Points[0].x + start_radius
            miny, minbbxy = self.Points[0].y, self.Points[0].y - start_radius
            maxy, maxbbxy = self.Points[0].y, self.Points[0].y + start_radius
            # Actualize minimum and maximum with the other points
            for point in self.Points[1:-1]:
                minx, minbbxx = min(minx, point.x), min(minbbxx, point.x)
                maxx, maxbbxx = max(maxx, point.x), max(maxbbxx, point.x)
                miny, minbbxy = min(miny, point.y), min(minbbxy, point.y)
                maxy, maxbbxy = max(maxy, point.y), max(maxbbxy, point.y)
            if len(self.Points) > 1:
                minx, minbbxx = min(minx, self.Points[-1].x), min(minbbxx, self.Points[-1].x - end_radius)
                maxx, maxbbxx = max(maxx, self.Points[-1].x), max(maxbbxx, self.Points[-1].x + end_radius)
                miny, minbbxy = min(miny, self.Points[-1].y), min(minbbxy, self.Points[-1].y - end_radius)
                maxy, maxbbxy = max(maxy, self.Points[-1].y), max(maxbbxy, self.Points[-1].y + end_radius)
            self.Pos = wxPoint(minx, miny)
            self.Size = wxSize(maxx -minx + 1, maxy - miny + 1)
            self.BoundingBox = wxRect(minbbxx, minbbxy, maxbbxx - minbbxx + 1, maxbbxy - minbbxy + 1)
    
    # Refresh the realpoints that permits to keep the proportionality in wire during resizing
    def RefreshRealPoints(self):
        if len(self.Points) > 0:
            self.RealPoints = []
            # Calculate float relative position of each point with the minimum point
            for point in self.Points:
                self.RealPoints.append([float(point.x - self.Pos.x), float(point.y - self.Pos.y)])
    
    # Returns the wire minimum size 
    def GetMinSize(self):
        width = 1
        height = 1
        dir_product = product(self.StartPoint[1], self.EndPoint[1])
        # The directions are opposed
        if dir_product < 0:
            if self.StartPoint[0] != 0:
                width = MIN_SEGMENT_SIZE * 2
            if self.StartPoint[1] != 0:
                height = MIN_SEGMENT_SIZE * 2
        # The directions are the same
        elif dir_product > 0:
            if self.StartPoint[0] != 0:
                width = MIN_SEGMENT_SIZE
            if self.StartPoint[1] != 0:
                height = MIN_SEGMENT_SIZE
        # The directions are perpendiculars
        else:
            width = MIN_SEGMENT_SIZE
            height = MIN_SEGMENT_SIZE
        return width + 1, height + 1
    
    # Returns if the point given is on one of the wire segments
    def HitTest(self, pt):
        test = False
        for i in xrange(len(self.Points) - 1):
            rect = wxRect(0, 0, 0, 0)
            x1, y1 = self.Points[i].x, self.Points[i].y
            x2, y2 = self.Points[i + 1].x, self.Points[i + 1].y
            # Calculate a rectangle around the segment
            rect = wxRect(min(x1, x2) - ANCHOR_DISTANCE, min(y1, y2) - ANCHOR_DISTANCE,
                abs(x1 - x2) + 2 * ANCHOR_DISTANCE, abs(y1 - y2) + 2 * ANCHOR_DISTANCE)
            test |= rect.InsideXY(pt.x, pt.y) 
        return test
    
    # Returns the wire start or end point if the point given is on one of them 
    def TestPoint(self, pt):
        # Test the wire start point
        rect = wxRect(self.Points[0].x - ANCHOR_DISTANCE, self.Points[0].y - ANCHOR_DISTANCE,
            2 * ANCHOR_DISTANCE, 2 * ANCHOR_DISTANCE)
        if rect.InsideXY(pt.x, pt.y):
            return 0
        # Test the wire end point
        if len(self.Points) > 1:
            rect = wxRect(self.Points[-1].x - ANCHOR_DISTANCE, self.Points[-1].y - ANCHOR_DISTANCE,
                2 * ANCHOR_DISTANCE, 2 * ANCHOR_DISTANCE)
            if rect.InsideXY(pt.x, pt.y):
                return -1
        return None
    
    # Returns the wire segment if the point given is on it
    def TestSegment(self, pt, all=False):
        for i in xrange(len(self.Segments)):
            # If wire is not in a Ladder Diagram, first and last segments are excluded
            if 0 < i < len(self.Segments) - 1 or all:
                x1, y1 = self.Points[i].x, self.Points[i].y
                x2, y2 = self.Points[i + 1].x, self.Points[i + 1].y
                # Calculate a rectangle around the segment
                rect = wxRect(min(x1, x2) - ANCHOR_DISTANCE, min(y1, y2) - ANCHOR_DISTANCE,
                    abs(x1 - x2) + 2 * ANCHOR_DISTANCE, abs(y1 - y2) + 2 * ANCHOR_DISTANCE)
                if rect.InsideXY(pt.x, pt.y):
                    return i, self.Segments[i]
        return None
    
    # Define the wire points
    def SetPoints(self, points):
        if len(points) > 1:
            self.Points = [wxPoint(x, y) for x, y in points]
            # Calculate the start and end directions
            self.StartPoint = [None, vector(self.Points[0], self.Points[1])]
            self.EndPoint = [None, vector(self.Points[-1], self.Points[-2])]
            # Calculate the start and end points
            self.StartPoint[0] = wxPoint(self.Points[0].x + CONNECTOR_SIZE * self.StartPoint[1][0], 
                self.Points[0].y + CONNECTOR_SIZE * self.StartPoint[1][1])
            self.EndPoint[0] = wxPoint(self.Points[-1].x + CONNECTOR_SIZE * self.EndPoint[1][0], 
                self.Points[-1].y + CONNECTOR_SIZE * self.EndPoint[1][1])
            self.Points[0] = self.StartPoint[0]
            self.Points[-1] = self.EndPoint[0]
            # Calculate the segments directions
            self.Segments = []
            for i in xrange(len(self.Points) - 1):
                self.Segments.append(vector(self.Points[i], self.Points[i + 1]))
            self.RefreshBoundingBox()
            self.RefreshRealPoints()
    
    # Returns the position of the point indicated
    def GetPoint(self, index):
        if index < len(self.Points):
            return self.Points[index].x, self.Points[index].y
        return None
    
    # Returns a list of the position of all wire points
    def GetPoints(self, invert = False):
        points = self.VerifyPoints()
        points[0] = wxPoint(points[0].x - CONNECTOR_SIZE * self.StartPoint[1][0], 
                points[0].y - CONNECTOR_SIZE * self.StartPoint[1][1])
        points[-1] = wxPoint(points[-1].x - CONNECTOR_SIZE * self.EndPoint[1][0], 
                points[-1].y - CONNECTOR_SIZE * self.EndPoint[1][1])
        # An inversion of the list is asked
        if invert:
            points.reverse()
        return points
    
    # Returns the position of the two selected segment points
    def GetSelectedSegmentPoints(self):
        if self.SelectedSegment != None and len(self.Points) > 1:
            return self.Points[self.SelectedSegment:self.SelectedSegment + 2]
        return []
    
    # Returns if the selected segment is the first and/or the last of the wire
    def GetSelectedSegmentConnections(self):
        if self.SelectedSegment != None and len(self.Points) > 1:
            return self.SelectedSegment == 0, self.SelectedSegment == len(self.Segments) - 1
        return (True, True)
    
    # Returns the connectors on which the wire is connected
    def GetConnected(self):
        connected = []
        if self.StartConnected and self.StartPoint[1] == WEST:
            connected.append(self.StartConnected)
        if self.EndConnected and self.EndPoint[1] == WEST:
            connected.append(self.EndConnected)
        return connected
    
    # Returns the id of the block connected to the first or the last wire point
    def GetConnectedId(self, index):
        if index == 0 and self.StartConnected:
            return self.StartConnected.GetBlockId()
        elif index == -1 and self.EndConnected:
            return self.EndConnected.GetBlockId()
        return None
    
    # Update the wire points position by keeping at most possible the current positions
    def GeneratePoints(self, realpoints = True):
        i = 0
        # Calculate the start enad end points with the minimum segment size in the right direction
        end = wxPoint(self.EndPoint[0].x + self.EndPoint[1][0] * MIN_SEGMENT_SIZE,
            self.EndPoint[0].y + self.EndPoint[1][1] * MIN_SEGMENT_SIZE)
        start = wxPoint(self.StartPoint[0].x + self.StartPoint[1][0] * MIN_SEGMENT_SIZE, 
            self.StartPoint[0].y + self.StartPoint[1][1] * MIN_SEGMENT_SIZE)
        # Evaluate the point till it's the last
        while i < len(self.Points) - 1:
            # The next point is the last
            if i + 1 == len(self.Points) - 1:
                # Calculate the direction from current point to end point
                v_end = vector(self.Points[i], end)
                # The current point is the first
                if i == 0:
                    # If the end point is not in the start direction, a point is added
                    if v_end != self.Segments[0] or v_end == self.EndPoint[1]:
                        self.Points.insert(1, wxPoint(start.x, start.y))
                        self.Segments.insert(1, DirectionChoice((self.Segments[0][1], 
                            self.Segments[0][0]), v_end, self.EndPoint[1]))
                # The current point is the second
                elif i == 1:
                    # The previous direction and the target direction are mainly opposed, a point is added
                    if product(v_end, self.Segments[0]) < 0:
                        self.Points.insert(2, wxPoint(self.Points[1].x, self.Points[1].y))
                        self.Segments.insert(2, DirectionChoice((self.Segments[1][1], 
                            self.Segments[1][0]), v_end, self.EndPoint[1]))
                    # The previous direction and the end direction are the same or they are
                    # perpendiculars and the end direction points towards current segment
                    elif product(self.Segments[0], self.EndPoint[1]) >= 0 and product(self.Segments[1], self.EndPoint[1]) <= 0:
                        # Current point and end point are aligned
                        if self.Segments[0][0] != 0:
                            self.Points[1].x = end.x
                        if self.Segments[0][1] != 0:
                            self.Points[1].y = end.y
                        # If the previous direction and the end direction are the same, a point is added
                        if product(self.Segments[0], self.EndPoint[1]) > 0:
                            self.Points.insert(2, wxPoint(self.Points[1].x, self.Points[1].y))
                            self.Segments.insert(2, DirectionChoice((self.Segments[1][1], 
                                self.Segments[1][0]), v_end, self.EndPoint[1]))
                    else:
                        # Current point is positioned in the middle of start point
                        # and end point on the current direction and a point is added
                        if self.Segments[0][0] != 0:
                            self.Points[1].x = (end.x + start.x) / 2
                        if self.Segments[0][1] != 0:
                            self.Points[1].y = (end.y + start.y) / 2
                        self.Points.insert(2, wxPoint(self.Points[1].x, self.Points[1].y))
                        self.Segments.insert(2, DirectionChoice((self.Segments[1][1], 
                            self.Segments[1][0]), v_end, self.EndPoint[1]))
                else:
                    # The previous direction and the end direction are perpendiculars
                    if product(self.Segments[i - 1], self.EndPoint[1]) == 0:
                        # The target direction and the end direction aren't mainly the same
                        if product(v_end, self.EndPoint[1]) <= 0:
                            # Current point and end point are aligned
                            if self.Segments[i - 1][0] != 0:
                                self.Points[i].x = end.x
                            if self.Segments[i - 1][1] != 0:
                                self.Points[i].y = end.y
                            # Previous direction is updated from the new point
                            if product(vector(self.Points[i - 1], self.Points[i]), self.Segments[i - 1]) < 0:
                                self.Segments[i - 1] = (-self.Segments[i - 1][0], -self.Segments[i - 1][1])
                        else:
                            test = True
                            # If the current point is the third, test if the second 
                            # point can be aligned with the end point
                            if i == 2:
                                test_point = wxPoint(self.Points[1].x, self.Points[1].y)
                                if self.Segments[1][0] != 0:
                                    test_point.y = end.y
                                if self.Segments[1][1] != 0:
                                    test_point.x = end.x
                                test = norm(vector(self.Points[0], test_point, False)) > MIN_SEGMENT_SIZE
                            # The previous point can be aligned
                            if test:
                                self.Points[i].x, self.Points[i].y = end.x, end.y
                                if self.Segments[i - 1][0] != 0:
                                    self.Points[i - 1].y = end.y
                                if self.Segments[i - 1][1] != 0:
                                    self.Points[i - 1].x = end.x
                                self.Segments[i] = (-self.EndPoint[1][0], -self.EndPoint[1][1])
                            else:
                                # Current point is positioned in the middle of previous point
                                # and end point on the current direction and a point is added
                                if self.Segments[1][0] != 0:
                                    self.Points[2].x = (self.Points[1].x + end.x) / 2
                                if self.Segments[1][1] != 0:
                                    self.Points[2].y = (self.Points[1].y + end.y) / 2
                                self.Points.insert(3, wxPoint(self.Points[2].x, self.Points[2].y))
                                self.Segments.insert(3, DirectionChoice((self.Segments[2][1], 
                                    self.Segments[2][0]), v_end, self.EndPoint[1]))
                    else:
                        # Current point is aligned with end point
                        if self.Segments[i - 1][0] != 0:
                            self.Points[i].x = end.x
                        if self.Segments[i - 1][1] != 0:
                            self.Points[i].y = end.y
                        # Previous direction is updated from the new point
                        if product(vector(self.Points[i - 1], self.Points[i]), self.Segments[i - 1]) < 0:
                            self.Segments[i - 1] = (-self.Segments[i - 1][0], -self.Segments[i - 1][1])
                        # If previous direction and end direction are opposed
                        if product(self.Segments[i - 1], self.EndPoint[1]) < 0:
                            # Current point is positioned in the middle of previous point
                            # and end point on the current direction
                            if self.Segments[i - 1][0] != 0:
                                self.Points[i].x = (end.x + self.Points[i - 1].x) / 2
                            if self.Segments[i - 1][1] != 0:
                                self.Points[i].y = (end.y + self.Points[i - 1].y) / 2
                        # A point is added
                        self.Points.insert(i + 1, wxPoint(self.Points[i].x, self.Points[i].y))
                        self.Segments.insert(i + 1, DirectionChoice((self.Segments[i][1], 
                                self.Segments[i][0]), v_end, self.EndPoint[1]))
            else:
                # Current point is the first, and second is not mainly in the first direction
                if i == 0 and product(vector(start, self.Points[1]), self.Segments[0]) < 0:
                    # If first and second directions aren't perpendiculars, a point is added 
                    if product(self.Segments[0], self.Segments[1]) != 0:
                        self.Points.insert(1, wxPoint(start.x, start.y))
                        self.Segments.insert(1, DirectionChoice((self.Segments[0][1], 
                            self.Segments[0][0]), vector(start, self.Points[1]), self.Segments[1]))
                    else:
                        self.Points[1].x, self.Points[1].y = start.x, start.y
                else:
                    # Next point is aligned with current point
                    if self.Segments[i][0] != 0:
                        self.Points[i + 1].y = self.Points[i].y
                    if self.Segments[i][1] != 0:
                        self.Points[i + 1].x = self.Points[i].x
                    # Current direction is updated from the new point
                    if product(vector(self.Points[i], self.Points[i + 1]), self.Segments[i]) < 0:
                        self.Segments[i] = (-self.Segments[i][0], -self.Segments[i][1])
            i += 1
        self.RefreshBoundingBox()
        if realpoints:
            self.RefreshRealPoints()
    
    # Verify that two consecutive points haven't the same position
    def VerifyPoints(self):
        points = [point for point in self.Points]
        segments = [segment for segment in self.Segments]
        i = 1
        while i < len(points) - 1:
            if points[i] == points[i + 1] and segments[i - 1] == segments[i + 1]:
                for j in xrange(2):
                    points.pop(i)
                    segments.pop(i)
            else:
                i += 1
        # If the wire isn't in a Ladder Diagram, save the new point list
        if self.Parent.__class__.__name__ != "LD_Viewer":
            self.Points = [point for point in points]
            self.Segments = [segment for segment in segments]
            self.RefreshBoundingBox()
            self.RefreshRealPoints()
        return points
    
    # Moves all the wire points except the first and the last if they are connected
    def Move(self, dx, dy, endpoints = False):
        for i, point in enumerate(self.Points):
            if endpoints or not (i == 0 and self.StartConnected) and not (i == len(self.Points) - 1 and self.EndConnected):
                point.x += dx
                point.y += dy
        self.StartPoint[0] = self.Points[0]
        self.EndPoint[0] = self.Points[-1]
        self.GeneratePoints()
    
    # Resize the wire from position and size given
    def Resize(self, x, y, width, height):
        if len(self.Points) > 1:
            # Calculate the new position of each point for testing the new size
            minx, miny = self.Pos.x, self.Pos.y
            lastwidth, lastheight = self.Size.width, self.Size.height
            for i, point in enumerate(self.RealPoints):
                # If start or end point is connected, it's not calculate
                if not (i == 0 and self.StartConnected) and not (i == len(self.Points) - 1 and self.EndConnected):
                    if i == 0:
                        dir = self.StartPoint[1]
                    elif i == len(self.Points) - 1:
                        dir = self.EndPoint[1]
                    else:
                        dir = (0, 0)
                    pointx = max(-dir[0] * MIN_SEGMENT_SIZE, min(int(round(point[0] * (width - 1) / float(lastwidth - 1))),
                            width - dir[0] * MIN_SEGMENT_SIZE - 1))
                    pointy = max(-dir[1] * MIN_SEGMENT_SIZE, min(int(round(point[1] * (height - 1) / float(lastheight - 1))),
                            height - dir[1] * MIN_SEGMENT_SIZE - 1))
                    self.Points[i] = wxPoint(minx + x + pointx, miny + y + pointy)
            self.StartPoint[0] = self.Points[0]
            self.EndPoint[0] = self.Points[-1]
            self.GeneratePoints(False)
            # Test if the wire position or size have changed
            if x != 0 and minx == self.Pos.x:
                x = 0
                width = lastwidth
            if y != 0 and miny == self.Pos.y:
                y = 0
                height = lastwidth
            if width != lastwidth and lastwidth == self.Size.width:
                width = lastwidth
            if height != lastheight and lastheight == self.Size.height:
                height = lastheight
            # Calculate the real points from the new size, it's important for
            # keeping a proportionality in the points position with the size
            # duringa resize dragging
            for i, point in enumerate(self.RealPoints):
                if not (i == 0 and self.StartConnected) and not (i == len(self.Points) - 1 and self.EndConnected):
                    point[0] = point[0] * (width - 1) / float(lastwidth - 1)
                    point[1] = point[1] * (height - 1) / float(lastheight - 1)
            # Calculate the correct position of the points from real points
            for i, point in enumerate(self.RealPoints):
                if not (i == 0 and self.StartConnected) and not (i == len(self.Points) - 1 and self.EndConnected):
                    if i == 0:
                        dir = self.StartPoint[1]
                    elif i == len(self.Points) - 1:
                        dir = self.EndPoint[1]
                    else:
                        dir = (0, 0)
                    realpointx = max(-dir[0] * MIN_SEGMENT_SIZE, min(int(round(point[0])),
                            width - dir[0] * MIN_SEGMENT_SIZE - 1))
                    realpointy = max(-dir[1] * MIN_SEGMENT_SIZE, min(int(round(point[1])),
                            height - dir[1] * MIN_SEGMENT_SIZE - 1))
                    self.Points[i] = wxPoint(minx + x + realpointx, miny + y + realpointy)
            self.StartPoint[0] = self.Points[0]
            self.EndPoint[0] = self.Points[-1]
            self.GeneratePoints(False)
    
    # Moves the wire start point and update the wire points
    def MoveStartPoint(self, point):
        if len(self.Points) > 1:
            self.StartPoint[0] = point
            self.Points[0] = point
            self.GeneratePoints()
    
    # Changes the wire start direction and update the wire points
    def SetStartPointDirection(self, dir):
        if len(self.Points) > 1:
            self.StartPoint[1] = dir
            self.Segments[0] = dir
            self.GeneratePoints()
    
    # Rotates the wire start direction by an angle of 90 degrees anticlockwise
    def RotateStartPoint(self):
        self.SetStartPointDirection((self.StartPoint[1][1], -self.StartPoint[1][0]))
    
    # Connects wire start point to the connector given and moves wire start point
    # to given point
    def ConnectStartPoint(self, point, connector):
        if point:
            self.MoveStartPoint(point)
        self.StartConnected = connector
    
    # Unconnects wire start point
    def UnConnectStartPoint(self, delete = False):
        self.StartConnected.UnConnect(self, False)
        if delete:
            self.Delete()
        else:
            self.StartConnected = None
    
    # Moves the wire end point and update the wire points
    def MoveEndPoint(self, point):
        if len(self.Points) > 1:
            self.EndPoint[0] = point
            self.Points[-1] = point
            self.GeneratePoints()

    # Changes the wire end direction and update the wire points
    def SetEndPointDirection(self, dir):
        if len(self.Points) > 1:
            self.EndPoint[1] = dir
            self.GeneratePoints()
            
    # Rotates the wire end direction by an angle of 90 degrees anticlockwise
    def RotateEndPoint(self):
        self.SetEndPointDirection((self.EndPoint[1][1], -self.EndPoint[1][0]))

    # Connects wire end point to the connector given and moves wire end point
    # to given point
    def ConnectEndPoint(self, point, connector):
        if point:
            self.MoveEndPoint(point)
        self.EndConnected = connector
    
    # Unconnects wire end point
    def UnConnectEndPoint(self, delete = False):
        self.EndConnected.UnConnect(self, False)
        if delete:
            self.Delete()
        else:
            self.EndConnected = None
    
    # Moves the wire segment given by its index
    def MoveSegment(self, idx, movex, movey):
        if 0 < idx < len(self.Segments) - 1:
            if self.Segments[idx] in (NORTH, SOUTH):
                self.Points[idx].x += movex
                self.Points[idx + 1].x += movex
            elif self.Segments[idx] in (EAST, WEST):
                self.Points[idx].y += movey
                self.Points[idx + 1].y += movey
            self.GeneratePoints()
    
    # Adds two points in the middle of the handled segment
    def AddSegment(self):
        handle_type, handle = self.Handle
        if handle_type == HANDLE_SEGMENT:
            segment, dir = handle
            pointx = self.Points[segment].x
            pointy = self.Points[segment].y
            if dir[0] != 0:
                pointx = (self.Points[segment].x + self.Points[segment + 1].x) / 2
            if dir[1] != 0:
                pointy = (self.Points[segment].y + self.Points[segment + 1].y) / 2
            self.Points.insert(segment + 1, wxPoint(pointx, pointy))
            self.Segments.insert(segment + 1, (dir[1], dir[0]))
            self.Points.insert(segment + 2, wxPoint(pointx, pointy))
            self.Segments.insert(segment + 2, dir)
            self.GeneratePoints()
    
    # Delete the handled segment by removing the two segment points
    def DeleteSegment(self):
        handle_type, handle = self.Handle
        if handle_type == HANDLE_SEGMENT:
            segment, dir = handle
            for i in xrange(2):
                self.Points.pop(segment)
                self.Segments.pop(segment)
            self.GeneratePoints()
            self.RefreshModel()
            
    # Method called when a LeftDown event have been generated
    def OnLeftDown(self, event, scaling):
        pos = GetScaledEventPosition(event, scaling)
        # Test if a point have been handled
        #result = self.TestPoint(pos)
        #if result != None:
        #    self.Handle = (HANDLE_POINT, result)
        #    self.Parent.SetCursor(wxStockCursor(wxCURSOR_HAND))
        #else:
        # Test if a segment have been handled
        result = self.TestSegment(pos)
        if result != None:
            if result[1] in (NORTH, SOUTH):
                self.Parent.SetCursor(wxStockCursor(wxCURSOR_SIZEWE))
            elif result[1] in (EAST, WEST):
                self.Parent.SetCursor(wxStockCursor(wxCURSOR_SIZENS))
            self.Handle = (HANDLE_SEGMENT, result)
        # Execute the default method for a graphic element
        else:
            Graphic_Element.OnLeftDown(self, event, scaling)
        self.oldPos = pos
    
    # Method called when a RightUp event have been generated
    def OnRightUp(self, event, scaling):
        pos = GetScaledEventPosition(event, scaling)
        # Test if a segment has been handled
        result = self.TestSegment(pos)
        if result != None:
            self.Handle = (HANDLE_SEGMENT, result)
            # Popup the menu with special items for a wire
            self.Parent.PopupWireMenu()
        else:
            # Execute the default method for a graphic element
            Graphic_Element.OnRightUp(self, event, scaling)
    
    # Method called when a LeftDClick event have been generated
    def OnLeftDClick(self, event, scaling):
        self.ResetPoints()
        self.GeneratePoints()
        
    # Method called when a Motion event have been generated
    def OnMotion(self, event, scaling):
        pos = GetScaledEventPosition(event, scaling)
        if not event.Dragging():
            # Test if a segment has been handled
            result = self.TestSegment(pos)
            if result:
                if result[1] in (NORTH, SOUTH):
                    wxCallAfter(self.Parent.SetCursor, wxStockCursor(wxCURSOR_SIZEWE))
                elif result[1] in (EAST, WEST):
                    wxCallAfter(self.Parent.SetCursor, wxStockCursor(wxCURSOR_SIZENS))
            else:
                # Test if a point has been handled
                #result = self.TestPoint(pos)
                #if result != None:
                #    if result == 0 and self.StartConnected:
                #        self.OverStart = True
                #    elif result != 0 and self.EndConnected:
                #        self.OverEnd = True
                #else:
                #    self.OverStart = False
                #    self.OverEnd = False
                # Execute the default method for a graphic element
                Graphic_Element.OnMotion(self, event, scaling)
        else:
            # Execute the default method for a graphic element
            Graphic_Element.OnMotion(self, event, scaling)
    
    # Refreshes the wire state according to move defined and handle selected
    def ProcessDragging(self, movex, movey):
        handle_type, handle = self.Handle
        # A point has been handled
        if handle_type == HANDLE_POINT:
            # Try to connect point to a connector
            new_pos = wxPoint(self.Points[handle].x + movex, self.Points[handle].y + movey)
            connector = self.Parent.FindBlockConnector(new_pos)
            if connector:
                if handle == 0 and self.EndConnected != connector:
                    connector.Connect((self, handle))
                    self.SetStartPointDirection(connector.GetDirection())
                    self.ConnectStartPoint(connector.GetPosition(), connector)
                    self.oldPos = connector.GetPosition()
                    self.Dragging = False
                elif handle != 0 and self.StartConnected != connector:
                    connector.Connect((self, handle))
                    self.SetEndPointDirection(connector.GetDirection())
                    self.ConnectEndPoint(connector.GetPosition(), connector)
                    self.oldPos = connector.GetPosition()
                    self.Dragging = False
                elif handle == 0:
                    self.MoveStartPoint(new_pos)
                else:
                    self.MoveEndPoint(new_pos)
            # If there is no connector, move the point
            elif handle == 0:
                if self.StartConnected:
                    self.UnConnectStartPoint()
                self.MoveStartPoint(new_pos)
            else:
                if self.EndConnected:
                    self.UnConnectEndPoint()
                self.MoveEndPoint(new_pos)
            self.RefreshModel()
        # A segment has been handled, move a segment
        elif handle_type == HANDLE_SEGMENT:
            self.MoveSegment(handle[0], movex, movey)
        # Execute the default method for a graphic element
        else:
            Graphic_Element.ProcessDragging(self, movex, movey)
    
    # Refreshes the wire model
    def RefreshModel(self, move=True):
        if self.StartConnected and self.StartPoint[1] in [WEST, NORTH]:
            self.StartConnected.RefreshParentBlock()
        if self.EndConnected and self.EndPoint[1] in [WEST, NORTH]:
            self.EndConnected.RefreshParentBlock()
    
    # Draws the wire lines and points
    def Draw(self, dc):
        dc.SetPen(wxBLACK_PEN)
        dc.SetBrush(wxBLACK_BRUSH)
        # Draw the start and end points if they are not connected or the mouse is over them
        if len(self.Points) > 0 and (not self.StartConnected or self.OverStart):
            dc.DrawCircle(self.Points[0].x, self.Points[0].y, POINT_RADIUS)
        if len(self.Points) > 1 and (not self.EndConnected or self.OverEnd):
            dc.DrawCircle(self.Points[-1].x, self.Points[-1].y, POINT_RADIUS)
        # Draw the wire lines and the last point (it seems that DrawLines stop before the last point)
        dc.DrawLines(self.Points)
        dc.DrawPoint(self.Points[-1].x, self.Points[-1].y)
        # Draw the segment selected in red
        if self.SelectedSegment != None:
            dc.SetPen(wxRED_PEN)
            dc.DrawLine(self.Points[self.SelectedSegment].x, self.Points[self.SelectedSegment].y,
                        self.Points[self.SelectedSegment + 1].x, self.Points[self.SelectedSegment + 1].y)
            if self.SelectedSegment == len(self.Segments) - 1:
                dc.DrawPoint(self.Points[-1].x, self.Points[-1].y)
        Graphic_Element.Draw(self, dc)


#-------------------------------------------------------------------------------
#                           Graphic comment element
#-------------------------------------------------------------------------------

"""
Class that implements a comment
"""

class Comment(Graphic_Element):

    # Create a new comment
    def __init__(self, parent, content, id = None):
        Graphic_Element.__init__(self, parent)
        self.Id = id
        self.Content = content
        self.Pos = wxPoint(0, 0)
        self.Size = wxSize(0, 0)
    
    # Method for keeping compatibility with others
    def Clean(self):
        pass
    
    # Delete this comment by calling the corresponding method
    def Delete(self):
        self.Parent.DeleteComment(self)
    
    # Refresh the comment bounding box
    def RefreshBoundingBox(self):
        self.BoundingBox = wxRect(self.Pos.x, self.Pos.y, self.Size[0] + 1, self.Size[1] + 1)
    
    # Changes the comment size
    def SetSize(self, width, height):
        self.Size.SetWidth(width)
        self.Size.SetHeight(height)
        self.RefreshBoundingBox()

    # Returns the comment size
    def GetSize(self):
        return self.Size.GetWidth(), self.Size.GetHeight()
    
    # Returns the comment minimum size
    def GetMinSize(self):
        dc = wxClientDC(self.Parent)
        min_width = 0
        min_height = 0
        # The comment minimum size is the maximum size of words in the content
        for line in self.Content.splitlines():
            for word in line.split(" "):
                wordwidth, wordheight = dc.GetTextExtent(word)
                min_width = max(min_width, wordwidth)
                min_height = max(min_height, wordheight)
        return min_width + 20, min_height + 20
    
    # Changes the comment position
    def SetPosition(self, x, y):
        self.Pos.x = x
        self.Pos.y = y
        self.RefreshBoundingBox()

    # Changes the comment content
    def SetContent(self, content):
        self.Content = content
        min_width, min_height = self.GetMinSize()
        self.Size[0] = max(self.Size[0], min_width)
        self.Size[1] = max(self.Size[1], min_height)
        self.RefreshBoundingBox()

    # Returns the comment content
    def GetContent(self):
        return self.Content

    # Returns the comment position
    def GetPosition(self):
        return self.Pos.x, self.Pos.y
    
    # Moves the comment
    def Move(self, dx, dy, connected = True):
        self.Pos.x += dx
        self.Pos.y += dy
        self.RefreshBoundingBox()
    
    # Resizes the comment with the position and the size given
    def Resize(self, x, y, width, height):
        self.Move(x, y)
        self.SetSize(width, height)
    
    # Method called when a RightUp event have been generated
    def OnRightUp(self, event, scaling):
        # Popup the default menu
        self.Parent.PopupDefaultMenu()
    
    # Refreshes the comment model
    def RefreshModel(self, move=True):
        self.Parent.RefreshCommentModel(self)
    
    # Method called when a LeftDClick event have been generated
    def OnLeftDClick(self, event, scaling):
        # Edit the comment content
        self.Parent.EditCommentContent(self)
    
    # Draws the comment and its content
    def Draw(self, dc):
        dc.SetPen(wxBLACK_PEN)
        dc.SetBrush(wxWHITE_BRUSH)
        # Draws the comment shape
        polygon = [wxPoint(self.Pos.x, self.Pos.y), 
                   wxPoint(self.Pos.x + self.Size[0] - 10, self.Pos.y),
                   wxPoint(self.Pos.x + self.Size[0], self.Pos.y + 10),
                   wxPoint(self.Pos.x + self.Size[0], self.Pos.y + self.Size[1] + 1),
                   wxPoint(self.Pos.x, self.Pos.y + self.Size[1] + 1)]
        dc.DrawPolygon(polygon)
        lines = [wxPoint(self.Pos.x + self.Size[0] - 10, self.Pos.y),
                 wxPoint(self.Pos.x + self.Size[0] - 10, self.Pos.y + 10),
                 wxPoint(self.Pos.x + self.Size[0], self.Pos.y + 10)]
        dc.DrawLines(lines)
        # Draws the comment content
        y = self.Pos.y + 10
        for line in self.Content.splitlines():
            first = True
            words = line.split(" ")
            for i, word in enumerate(words):
                if first:
                    test = word
                else:
                    test = linetext + " " + word
                wordwidth, wordheight = dc.GetTextExtent(test)
                if y + wordheight > self.Pos.y + self.Size[1] - 10:
                    break
                if wordwidth < self.Size[0] - 20 and i < len(words) - 1:
                    linetext = test
                    first = False
                else:
                    if wordwidth < self.Size[0] - 20 and i == len(words) - 1:
                        dc.DrawText(test, self.Pos.x + 10, y)
                    else:
                        dc.DrawText(linetext, self.Pos.x + 10, y)
                        if i == len(words) - 1:
                            y += wordheight + 5
                            if y + wordheight > self.Pos.y + self.Size[1] - 10:
                                break
                            dc.DrawText(word, self.Pos.x + 10, y)
                        else:
                            linetext = word
                    y += wordheight + 5
            if y + wordheight > self.Pos.y + self.Size[1] - 10:
                break
        Graphic_Element.Draw(self, dc)
