/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.wst.rdb.data.internal.ui;

import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.eclipse.wst.rdb.data.internal.ui.editor.IExternalTableDataEditor;
import org.eclipse.wst.rdb.data.internal.ui.editor.TableDataEditor;
import org.eclipse.wst.rdb.data.internal.ui.editor.TableDataEditorActionBarContributor;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.containment.ContainmentService;
import org.eclipse.wst.rdb.internal.core.containment.GroupID;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.BinaryStringDataType;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.CharacterStringDataType;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.DataLinkDataType;
import org.eclipse.wst.rdb.internal.models.sql.datatypes.DataType;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.tables.Column;

import com.ibm.icu.util.StringTokenizer;

public class DataUIPlugin extends AbstractUIPlugin
{
    //The shared instance.
    private static DataUIPlugin plugin;
    
    public static final String PLUGIN_ID = "org.eclipse.wst.rdb.data.ui"; //$NON-NLS-1$

    protected TableDataEditorActionBarContributor tableDataEditorContributor;

    /** the list of extensions for the externalTableDataCellEditor extension point */
    protected List externalTableDataEditorExtensions;
   
    /**
     * The constructor.
     */
    public DataUIPlugin()
    {
        super();
        plugin = this;
        listExternalTableDataEditorExtensions();
    }

    /**
     * Returns the shared instance.
     */
    public static DataUIPlugin getDefault()
    {
        return plugin;
    }
    public TableDataEditorActionBarContributor getTableDataEditorContributor() {
        return tableDataEditorContributor;
    }
    public void setTableDataEditorContributor(
            TableDataEditorActionBarContributor tableDataEditorContributor) {
        this.tableDataEditorContributor = tableDataEditorContributor;
    }
    
	/**
	 * Method writeLog.
	 * @param severity - the severity; one of IStatus.OK, IStatus.ERROR, IStatus.INFO, or IStatus.WARNING
	 * @param code - the plug-in-specific status code, or OK
	 * @param message - a human-readable message, localized to the current locale
	 * @param exception- a low-level exception, or null if not applicable
	 */
	public void writeLog(int severity, int code, String message, Throwable exception) {
		if (message == null)
			message = ""; //$NON-NLS-1$
		getLog().log(
				new Status(severity, getBundle().getSymbolicName(), code, message, exception));
	}
    
    public static boolean isGroupIDOK(EObject o)
    {
        if (o==null)
            return false;
        ContainmentService containmentService = RDBCorePlugin.getDefault().getContainmentService(); 
        String groupID = containmentService.getGroupId(o);
        return (groupID != null) && (groupID.startsWith(GroupID.CORE_PREFIX));  
    }
    
    /**
     * Gets the encoding string from the Window->Preferences menu
     * @return the encoding string to be used for saving and loading
     */
    public static String getCharacterEncoding( )
    {
 	   return ResourcesPlugin.getEncoding();
    }
    
	/**
	 * This gets a .gif from the icons folder.
	 */
	public ImageDescriptor getImageDescriptor(String key) {
		try {			
			return ImageDescriptor.createFromURL(					
					new URL(
							getBundle().getEntry("/"),						   //$NON-NLS-1$
							"icons" + java.io.File.separator + key + ".gif")); //$NON-NLS-1$ //$NON-NLS-2$
		}
		catch (Exception e) {
			writeLog(IStatus.ERROR, 0, e.getMessage(), e);
		}
		return null;
	}	

    /**
     * Collects all extensions for the externalTableDataEditor extension point 
     * and stores it in a list.
     */
    protected void listExternalTableDataEditorExtensions() {
        externalTableDataEditorExtensions = new ArrayList();
        IExtensionRegistry registry = Platform.getExtensionRegistry();
        IExtensionPoint extensionPoint = registry.getExtensionPoint(PLUGIN_ID + ".externalTableDataEditor"); //$NON-NLS-1$
        IExtension[] extensions = extensionPoint.getExtensions();
        for (int i = 0; i < extensions.length; i++) {
            IExtension extension = extensions[i];
            IConfigurationElement[] elements = extension.getConfigurationElements();
            for (int j=0; j<elements.length; ++j)
                externalTableDataEditorExtensions.add(new ExternalTableDataEditorExtension(elements[j]));
        }
    }
    
    /**
     * returns the best matching IExternalTableDataEditor or null.
     * @param editor the TableDataEditor
     * @param columnIndex the index of the sql table column 
     * @return the best fitting IExternalTableDataEditor or null if none matches
     */
    public IExternalTableDataEditor newExternalTableDataCellEditor(TableDataEditor editor, int columnIndex){
        // input validation
        if ( (editor==null) || (editor.getSqlTable()==null) || 
             (editor.getSqlTable().getColumns() == null)){
            return null;
        }
        if (columnIndex > editor.getSqlTable().getColumns().size()){
            return null;
        }
        Column sqlCol = (Column)editor.getSqlTable().getColumns().get(columnIndex);
        IExternalTableDataEditor externalEditor = null;
        
        try {
            Database db = sqlCol.getTable().getSchema().getDatabase();
            DataType type = sqlCol.getDataType();
            
            String vendor = db.getVendor();
            String version = db.getVersion();
            String dataType = sqlCol.getDataType().getName();

            // checks for the sqlmodel objects
            // that support a length attribute
            int length = -1;     
            boolean lengthSupported = false;
            if (type instanceof DataLinkDataType){
                lengthSupported = true;
                length = ((DataLinkDataType)type).getLength();
            }
            if (type instanceof BinaryStringDataType){
                lengthSupported = true;
                length = ((BinaryStringDataType)type).getLength();
            }
            if (type instanceof CharacterStringDataType) {
                lengthSupported = true;
                length = ((CharacterStringDataType)type).getLength();
            }
            
            ExternalTableDataEditorExtension element = null;
            
            Iterator it = externalTableDataEditorExtensions.iterator();
            while (it.hasNext()) {
                ExternalTableDataEditorExtension curElement = (ExternalTableDataEditorExtension)it.next();
                boolean matches = curElement.matches(vendor, version, dataType, lengthSupported, length) && (element==null || curElement.getScore()>element.getScore());
                if (matches)  {
                    element = curElement;
                }
            }

            if (element!=null){
                externalEditor = element.createInstance();
            }
        } catch (CoreException ex) {
            writeLog(IStatus.ERROR, 0, ex.getMessage(), ex);
        }
        return externalEditor;
    }
    
}

/**
 * an internal class holding the information of one contribution to 
 * the externalTableDataCellEditor extension point  
 */
class ExternalTableDataEditorExtension {
    protected IConfigurationElement element;
    protected String vendor;
    protected String version;
    protected String dataType;
    protected int length = -10;
    protected int score;
    protected boolean lengthProvided = false;
    
    public ExternalTableDataEditorExtension(IConfigurationElement element){
        this.element = element;
        vendor = element.getAttribute("vendor"); //$NON-NLS-1$
        version = element.getAttribute("version"); //$NON-NLS-1$
        dataType = element.getAttribute("dataType"); //$NON-NLS-1$
        String sLength = element.getAttribute("length"); //$NON-NLS-1$
        try{
            length = Integer.parseInt(sLength);
            lengthProvided = true;
        }catch (NumberFormatException nfe){
        }catch (NullPointerException npe){}
        
        score = ((vendor!=null)?1:0) + ((version!=null)?1:0) + ((dataType!=null)?1:0) + ((length>0)?1:0);
    }
    
    /**
     * returns true if this ExternalTableDataEditorExtension matches the given
     * arguments
     */
    public boolean matches(String vendor, String version, String dataType, boolean lengthSupported, int length){
        
        // no match if we have a length restriction but no datatype with length support
        if (lengthProvided && !lengthSupported) {
            return false;
        }
        // no match if length restriction doesnt apply
        if (lengthProvided && !(this.length <= length)){
            return false;
        }
        // no match if vendor,version,datatype are given but not the same
		if (this.vendor!=null && !contains(this.vendor, vendor))
			return false;
		else if (this.version!=null && !contains(this.version,version))
			return false;
		else if (this.dataType!=null && !contains(this.dataType,dataType))
			return false;
        
        return true;
    }
    
	protected boolean contains(String values, String value)
	{
		StringTokenizer st = new StringTokenizer(values, ",", false); //$NON-NLS-1$
		while (st.hasMoreElements()) {
			String s = st.nextToken();
			if (value.equals(s.trim()))
				return true;
		}
		return false;
	}
    
    public IExternalTableDataEditor createInstance() throws CoreException{
        return (IExternalTableDataEditor)element.createExecutableExtension("class"); //$NON-NLS-1$
    }
    
    public int getScore(){
        return score;
    }
}
