#include <fcntl.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <sys/wait.h>
#include <sys/param.h>
#include <sys/types.h>

#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>
#include <gtk/gtk.h>

#include "../wmgeneral/wmgeneral.h"
#include "../wmgeneral/misc.h"

#include "wmtimer.xpm"

#define CHAR_WIDTH 5
#define CHAR_HEIGHT 7
#define VERSION "2.2"


	/* ---FUNCTIONS--- */
void BlitString (char *name, int x, int y);
void BlitNum (int num, int x, int y);
void DrawTime (int, int, int);

void InitDisplay ();
void ReinitTimer ();
void DecrementTimer ();
void IncrementTimer ();
void ExecAct ();
void usage ();

int configure_wmtimer ();
void callback (GtkWidget * widget, gpointer data);
int delete_event (GtkWidget * widget, GdkEvent * event, gpointer data);
void destroy (GtkWidget * widget, gpointer data);

int atoi ();
char toupper ();
void get_lang ();


	/* ---GLOBALS--- */
static GtkWidget *spinner1;
static GtkWidget *spinner2;
static GtkWidget *spinner3;
static GtkWidget *entry;

char *ProgName;
char cmd[256];
char *command = cmd;
char wminet_mask_bits[64 * 64];
int wminet_mask_width = 64;
int wminet_mask_height = 64;
int mode = -1;			// 0 = alarm @ time     
				// 1 = countdown timer
				// 2 = Chrono paused            
				// 3 = Chrono active

int action = 0;			// 0 = default/bell     1 = command
int tmp_action;
int tmp_mode = 0;
int canceled = -1;		// Keep ReinitTimer() from resetting the 
				// timer to 0 if we enter the config
				// and then cancel
char *tmp_command = NULL;

int ihr = 0, imin = 0, isec = 0;
int oldsec = 0;

	/* ---MAIN--- */
int main (int argc, char *argv[])
{
  int i;
  XEvent Event;
  int buttonStatus = -1;
  long starttime;
  long curtime;
  long nexttime;
  struct tm *time_struct;
  struct tm old_time_struct;

  gtk_init (&argc, &argv);

  command[0] = '\0';

/* Parse Command Line */

  for (i = 1; i < argc; i++) {
    char *arg = argv[i];

    if (*arg == '-') {
      switch (arg[1]) {
	case 'a':
	  mode = 0;
	  break;
	case 'c':
	  mode = 1;
	  break;
	case 'r':
	  mode = 3;
	  break;
	case 'e':
	  strcpy (command, argv[i + 1]);
	  action = 1;
	  break;
	case 't':
	  ihr = atoi (strtok (argv[i + 1], ":"));
	  imin = atoi (strtok (NULL, ":"));
	  isec = atoi (strtok (NULL, ":"));
	  break;
	case 'v':
	  printf ("Version: %s\n", VERSION);
	  _exit (0);
	  break;
	default:
	  usage ();
	  _exit (0);
	  break;
      }
    }
    else {
      if (!(strcmp (arg, ":"))) {
	usage ();
      }
    }
  }

  createXBMfromXPM (wminet_mask_bits, wmtimer_xpm, wminet_mask_width,
		    wminet_mask_height);

  openXwindow (argc, argv, wmtimer_xpm, wminet_mask_bits, wminet_mask_width,
	       wminet_mask_height);

  //setMaskXY(-64, 0);

  AddMouseRegion (0, 18, 49, 45, 59);	/* middle button */
  AddMouseRegion (1, 5, 49, 17, 59);	/* left button   */
  AddMouseRegion (2, 46, 49, 59, 59);	/* right button  */
  AddMouseRegion (3, 2, 2, 58, 47);	/* main area     */
//  AddMouseRegion (3, 6, 2, 60, 18);   /*first bar      */
//  AddMouseRegion (4, 6, 20, 60, 34);  /*first bar      */
//  AddMouseRegion (5, 6, 37, 60, 48);  /*third bar      */


  InitDisplay ();
  ReinitTimer ();

  starttime = time (0);
  nexttime = starttime + 1;

  curtime = time (0);
  time_struct = localtime (&curtime);

  while (1) {
    curtime = time (0);

    waitpid (0, NULL, WNOHANG);

    old_time_struct = *time_struct;
    time_struct = localtime (&curtime);


    if (curtime >= starttime) {
      DrawTime (time_struct->tm_hour, time_struct->tm_min,
		time_struct->tm_sec);
      RedrawWindow ();
      //RedrawWindow ();

      // X Events
      while (XPending (display)) {
	XNextEvent (display, &Event);
	switch (Event.type) {
	  case Expose:
	    RedrawWindow ();
	    break;
	  case DestroyNotify:
	    XCloseDisplay (display);
	    _exit (0);
	    break;
	  case ButtonPress:
	    i = CheckMouseRegion (Event.xbutton.x, Event.xbutton.y);
	    buttonStatus = i;
	    if (buttonStatus == i && buttonStatus >= 0) {
	      switch (buttonStatus) {
		case 0:	// center button
		  if (mode == 0 || mode == 1) {
		    mode = 2;
		    ReinitTimer ();
		  }
		  mode = 2;
		  InitDisplay ();
		  break;
		case 1:	// left arrow button
		  mode = 2;
		  ReinitTimer ();
		  InitDisplay ();
		  break;
		case 2:	// right arrow button
		  if (mode == 0 || mode == 1) {
		    mode = 3;
		    ReinitTimer ();
		  }
		  mode = 3;
		  InitDisplay ();
		  break;
		case 3:	// main area

//            if (fork() == 0) { /* this is the child process */
		  configure_wmtimer ();
//            }


		  ReinitTimer ();
		  InitDisplay ();
		  break;
		case 4:
	      }
	    }

	    break;
	  case ButtonRelease:
	    i = CheckMouseRegion (Event.xbutton.x, Event.xbutton.y);

	    if (buttonStatus == i && buttonStatus >= 0) {
	      switch (buttonStatus) {
		case 0:
		  break;
		case 1:
		  break;
		case 2:
		  break;
		case 3:
		  break;
		case 4:
		  break;
		case 5:
		  break;
	      }
	    }
	    buttonStatus = -1;
	    RedrawWindow ();
	    break;
	}
      }
      usleep (100000L);
    }
  }
  return 0;
}

/********************/
/* DrawTime         */
/********************/

void DrawTime (int hr, int min, int sec)
{

  switch (mode) {
    case 1:
      if (oldsec < sec) {
	DecrementTimer ();
      }
      oldsec = sec;
      break;
    case 3:
      if (oldsec < sec) {
	IncrementTimer ();
      }
      oldsec = sec;
      break;
    case 0:
      if (ihr == hr && imin == min && isec == sec) {
	ExecAct ();
      }
      break;
  }

  BlitNum (hr, 7, 5);
  BlitString (":", 20, 5);
  BlitNum (min, 25, 5);
  BlitString (":", 38, 5);
  BlitNum (sec, 43, 5);

}

void InitDisplay ()
{

  copyXPMArea (13 * 6, 64, 8 * 6, 8, 6, 21);

  switch (mode) {
    case 0:
      BlitString ("Alarm:", 13, 21);
      break;
    case 1:
      BlitString ("Timer:", 13, 21);
      break;
    case 2:
    case 3:
      BlitString ("Chrono:", 12, 21);
      break;
    default:
      BlitString ("WMTIMER", 10, 21);
      break;
  }
}



void ReinitTimer ()
{

  if ((mode == 2 || mode == 3) && (!canceled)) {
    ihr = imin = isec = 0;
  }
  if (canceled)
    canceled = 0;

  BlitNum (ihr, 7, 36);
  BlitString (":", 20, 36);
  BlitNum (imin, 25, 36);
  BlitString (":", 38, 36);
  BlitNum (isec, 43, 36);

}

// Blits a string at given co-ordinates
void BlitString (char *name, int x, int y)
{

  //copyXPMArea (x_get_pos, y_get_pos, x_dist_from_x_pos, y_dist_from_y_pos, x_placement_pos, y_placement_pos);
  // each char/num is 6 units wide & 8 units high, nums are 64 units down, chars are 74 units down


  int i;
  int c;
  int k;

  k = x;
  for (i = 0; name[i]; i++) {
    c = toupper (name[i]);
    if (c >= 'A' && c <= 'Z') {	// its a letter
      c -= 'A';
      copyXPMArea (c * 6, 74, 6, 8, k, y);
      k += 6;
    }
    else {			// its a number or symbol
      c -= '0';
      copyXPMArea (c * 6, 64, 6, 8, k, y);
      k += 6;
    }
  }

}

void BlitNum (int num, int x, int y)
{
  char buf[1024];
  int newx = x;

  if (num > 99) {
    newx -= CHAR_WIDTH;
  }

  if (num > 999) {
    newx -= CHAR_WIDTH;
  }

  sprintf (buf, "%02i", num);
  BlitString (buf, newx, y);
}

void DecrementTimer ()
{
  isec--;
  if (isec == -1) {
    isec = 59;
    imin--;
    if (imin == -1) {
      imin = 59;
      ihr--;
    }
  }

  BlitNum (ihr, 7, 36);
  BlitString (":", 20, 36);
  BlitNum (imin, 25, 36);
  BlitString (":", 38, 36);
  BlitNum (isec, 43, 36);

  if (isec == 0 && imin == 0 && ihr == 0) {
    ExecAct ();
  }
}

void IncrementTimer ()
{

  isec++;
  if (isec == 60) {
    isec = 0;
    imin++;
    if (imin == 60) {
      imin = 0;
      ihr++;
    }
  }

  BlitNum (ihr, 7, 36);
  BlitString (":", 20, 36);
  BlitNum (imin, 25, 36);
  BlitString (":", 38, 36);
  BlitNum (isec, 43, 36);

}

void ExecAct ()
{
  if (action) {
    execCommand (command);
  }
  else {
    printf ("\07");
    fflush (stdout);
  }
  //_exit (0);
  mode = -1;
}


void usage (void)
{
  fprintf (stderr, "\nwmtimer - Josh King <jking@dwave.net>\n\n");

  fprintf (stderr, "usage: ./wmtimer -[a|c] -t <hh:mm:ss> -e <command>\n");
  fprintf (stderr, "       ./wmtimer -r \n\n");

  fprintf (stderr, "    -a    alarm mode, wmtimer will beep/exec command\n");
  fprintf (stderr, "            at specified time\n");
  fprintf (stderr, "    -c    countdowntimer mode, wmtimer will beep/exec\n");
  fprintf (stderr, "	        command when specified time reaches 0 \n");
  fprintf (stderr, "    -e    <command> system bell is default\n");
  fprintf (stderr, "    -r    start in chronograph mode\n");
  fprintf (stderr, "    -t    <hh:mm:ss>\n");
  fprintf (stderr, "    -h    this help screen\n");
  fprintf (stderr, "    -v    print the version number\n");
  fprintf (stderr, "\n");

  _exit (0);
}


	/* Our callback. */
void callback (GtkWidget * widget, gpointer data)
{
  if ((char *) data == "alarm_button")
    tmp_mode = 0;
  if ((char *) data == "timer_button")
    tmp_mode = 1;

  if ((char *) data == "bell_button") {
    tmp_action = 0;
    gtk_entry_set_text (GTK_ENTRY (entry), "");
    gtk_entry_set_editable (GTK_ENTRY (entry), FALSE);
  }

  if ((char *) data == "command_button") {
    tmp_action = 1;
    gtk_entry_set_editable (GTK_ENTRY (entry), TRUE);
    gtk_entry_set_text (GTK_ENTRY (entry), command);
  }
  if ((char *) data == "ok") {
    if (tmp_action == 1)
      strcpy (command, gtk_entry_get_text (GTK_ENTRY (entry)));

    ihr = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinner1));
    imin = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinner2));
    isec = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinner3));

    mode = tmp_mode;
    action = tmp_action;

    //gtk_main_quit ();
  }

  if (!strcmp ((char *) data, "cancel")) {
    canceled = 1;
    ihr = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinner1));
    imin = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinner2));
    isec = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (spinner3));

  }

  if (!strcmp ((char *) data, "clear")) {
    ihr = imin = isec = 0;
    command[0] = '\0';
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spinner1), 0);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spinner2), 0);
    gtk_spin_button_set_value (GTK_SPIN_BUTTON (spinner3), 0);
    gtk_entry_set_text (GTK_ENTRY (entry), command);
  }

}

/* This callback quits the program */
int delete_event (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  return (FALSE);
}

//destroys window 
void destroy (GtkWidget * widget, gpointer data)
{
  gtk_main_quit ();
}


int configure_wmtimer ()
{

  GtkWidget *window;
  GtkWidget *frame;
  GtkWidget *button;
  GtkWidget *box1;
  GtkWidget *box2;
  GtkWidget *sub_vbox;
  GtkWidget *label;
  GtkAdjustment *adj;

  tmp_action = 0;

  // Create a new window
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "Configure");
  gtk_window_set_wmclass (GTK_WINDOW (window), "wmtimerconf", "");

  // Set a handler for delete_event that immediately exits GTK.
  // gtk_signal_connect (GTK_OBJECT (window), "delete_event", GTK_SIGNAL_FUNC (delete_event), NULL);
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
		      GTK_SIGNAL_FUNC (destroy), NULL);

  // Sets the border width of the window.
  gtk_container_set_border_width (GTK_CONTAINER (window), 10);

  // Create Vertical box
  box1 = gtk_vbox_new (FALSE, 0);
  // Add vertical box to main window
  gtk_container_add (GTK_CONTAINER (window), box1);
  gtk_widget_show (box1);

  frame = gtk_frame_new ("Mode");
  gtk_box_pack_start (GTK_BOX (box1), frame, TRUE, TRUE, 2);
  gtk_widget_show (frame);

  box2 = gtk_hbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame), box2);

  // Create Alarm radio button
  button = gtk_radio_button_new_with_label (NULL, "Alarm");
  // When the button is clicked, we call the "callback" function * with a pointer to "button 1" as its argument
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback), (gpointer) "alarm_button");
  gtk_box_pack_start (GTK_BOX (box2), button, FALSE, FALSE, 2);
  gtk_widget_show (button);


  // Create Timer radio button
  button =
    gtk_radio_button_new_with_label (gtk_radio_button_group
				     (GTK_RADIO_BUTTON (button)), "Timer");
  // When the button is clicked, we call the "callback" function * with a pointer to "button 2" as its argument
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback), (gpointer) "timer_button");
  gtk_box_pack_start (GTK_BOX (box2), button, FALSE, FALSE, 2);
  gtk_widget_show (button);
  gtk_widget_show (box2);

  if (tmp_mode == 1)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);

  frame = gtk_frame_new ("Time");
  gtk_box_pack_start (GTK_BOX (box1), frame, TRUE, TRUE, 2);
  gtk_widget_show (frame);

  box2 = gtk_hbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (frame), box2);

  adj = (GtkAdjustment *) gtk_adjustment_new (ihr, 0.0, 24.0, 1.0, 2.0, 0.0);
  spinner1 = gtk_spin_button_new (adj, 0, 0);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner1), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinner1), TRUE);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner1),
				   GTK_SHADOW_OUT);
  gtk_box_pack_start (GTK_BOX (box2), spinner1, FALSE, FALSE, 2);
  gtk_widget_show (spinner1);

  label = gtk_label_new (" : ");
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX (box2), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  adj = (GtkAdjustment *) gtk_adjustment_new (imin, 0.0, 59.0, 1.0, 5.0, 0.0);
  spinner2 = gtk_spin_button_new (adj, 0, 0);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner2), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinner2), TRUE);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner2),
				   GTK_SHADOW_OUT);
  gtk_box_pack_start (GTK_BOX (box2), spinner2, FALSE, FALSE, 2);
  gtk_widget_show (spinner2);

  label = gtk_label_new (" : ");
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX (box2), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  adj = (GtkAdjustment *) gtk_adjustment_new (isec, 0.0, 59.0, 1.0, 5.0, 0.0);
  spinner3 = gtk_spin_button_new (adj, 0, 0);
  gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner3), TRUE);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spinner3), TRUE);
  gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner3),
				   GTK_SHADOW_OUT);
  gtk_box_pack_start (GTK_BOX (box2), spinner3, FALSE, FALSE, 2);
  gtk_widget_show (spinner3);
  gtk_widget_show (box2);

  frame = gtk_frame_new ("Action");
  gtk_box_pack_start (GTK_BOX (box1), frame, TRUE, TRUE, 2);
  gtk_widget_show (frame);

  // Create Vertical box
  sub_vbox = gtk_vbox_new (FALSE, 0);
  // Add vertical box to main window
  gtk_container_add (GTK_CONTAINER (frame), sub_vbox);
  gtk_widget_show (sub_vbox);

  box2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (sub_vbox), box2, TRUE, TRUE, 2);

  // Create Bell radio button
  button = gtk_radio_button_new_with_label (NULL, "System Bell");
  // When the button is clicked, we call the "callback" function * with a pointer to "button 1" as its argument
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback), (gpointer) "bell_button");
  gtk_box_pack_start (GTK_BOX (box2), button, FALSE, FALSE, 2);
  gtk_widget_show (button);


  // Create Command radio button
  button = gtk_radio_button_new_with_label (gtk_radio_button_group
					    (GTK_RADIO_BUTTON (button)),
					    "Command");
  // When the button is clicked, we call the "callback" function * with a pointer to "button 2" as its argument
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback),
		      (gpointer) "command_button");
  gtk_box_pack_start (GTK_BOX (box2), button, FALSE, FALSE, 2);
  gtk_widget_show (button);
  gtk_widget_show (box2);

  box2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (sub_vbox), box2, TRUE, TRUE, 2);


  label = gtk_label_new ("Command: ");
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
  gtk_box_pack_start (GTK_BOX (box2), label, FALSE, FALSE, 2);
  gtk_widget_show (label);

  /* Create "Command" text entry area */
  entry = gtk_entry_new_with_max_length (100);
  gtk_entry_set_editable (GTK_ENTRY (entry), FALSE);
  gtk_signal_connect (GTK_OBJECT (entry), "activate",
		      GTK_SIGNAL_FUNC (callback), entry);
  gtk_box_pack_start (GTK_BOX (box2), entry, FALSE, FALSE, 2);
  gtk_widget_show (entry);
  gtk_widget_show (box2);

  if (action == 1)
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);

  box2 = gtk_hbox_new (FALSE, 0);
  gtk_box_pack_start (GTK_BOX (box1), box2, TRUE, TRUE, 2);

  // Create "Cancel" button
  button = gtk_button_new_with_label ("Cancel");
  // When the button is clicked, we call the "delete_event" function and the program exits
  //gtk_signal_connect (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (delete_event), NULL);
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback), "cancel");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 2);
  gtk_widget_show (button);

  // Create "Clear" button
  button = gtk_button_new_with_label ("Clear");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback), "clear");
  gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 2);
  gtk_widget_show (button);

  // Create "Ok" button
  button = gtk_button_new_with_label ("Ok");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
		      GTK_SIGNAL_FUNC (callback), "ok");
  gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (window));
  gtk_box_pack_start (GTK_BOX (box2), button, TRUE, TRUE, 2);
  gtk_widget_show (button);

  gtk_widget_show (box2);

  gtk_widget_show (window);

  gtk_main ();

  return 0;
}
