#  Download subs for wmtheme

use Socket;
my $ftp_command;
my $hostmatch = '(?i)[a-z](?:[a-z\d-]{0,61}[a-z\d])?(?:\.[a-z](?:[a-z\d-]{0,61}[a-z\d])?)*';

sub dlquery {
  my ($themetxt, $tosite) = @_;
  my (%candidates, @choices, $choice, $yourchoice, $minwidth, $dchoice,
    @dispchoices, $ltime, $maxname, $timepad);
  my $versionwarning = '';

  read_userconfig();
  %candidates = query_to($tosite, $themetxt);

  unless (@choices = sort keys %candidates) {
    print "\n";
    choke("query returned no matches for $themetxt");
  }

  if (@choices == 1) {
    $yourchoice = $choices[0];
    print " found \"$yourchoice\"\n";
    if ($config_wm{version_mismatch} eq 'ask' and
        !$candidates{$yourchoice}{versionok}) {
      print "This theme was made for $wm_name $candidates{$yourchoice}{version}, and may not\n";
      print "work properly with the installed version.  Do you want to proceed? [y/N] ";
      finish("download cancelled.") unless <STDIN> =~ /^[yY]/;
    }
  } else {
    print "\n";
    
    ###  build choices display strings  ###

    $minwidth = getmaxlen(@choices);
      # timepad - width of the timestamp
    $timepad = length(localtime time) - 9;
      # maxname - longest name to allow in display
    $maxname = (screensize())[1] -
      # [index] [. ] <name> [2 spaces] [size] [timestamp] [the * mark] [space at end]
      (length(@choices + 1) + 2 + 2 + 12 + $timepad + 10 + 1);
    $minwidth = $maxname if $minwidth > $maxname;

    foreach $choice (@choices) {
      if (length($choice) > $maxname) {
        $dchoice = substr($choice, 0, $maxname - 5) . '[...]';
      } else {
        $dchoice = $choice . ' ' x ($minwidth - length $choice);
      }

      ($ltime = localtime($candidates{$choice}{updated})) =~ s/\d\d:\d\d:\d\d //;
      $size = bytesdisplay($candidates{$choice}{size});
      $dchoice .= '  ' . $size  . ' ' x (12 - length $size) .
                         $ltime . ' ' x ($timepad - length $ltime);

      unless ($candidates{$choice}{versionok}) {
        $dchoice .= "     [* " . substr($candidates{$choice}{version}, 0, 6) . ']';
        $versionwarning = " ([*] = possible incompatibility)";
      }
      push @dispchoices, $dchoice;
    }

    print "\n" . @choices .
      " themes were returned. Please choose one$versionwarning:\n\n";
    my $uchoice = getuserchoice(@dispchoices);
    finish("theme retrieval aborted") if $uchoice < 0;
    $yourchoice = $choices[$uchoice];
  }

  delete $candidates{$yourchoice}{versionok};
  return ($candidates{$yourchoice}, $yourchoice);
}

sub query_to {
  my ($downloadsite, $themetext) = @_;
  my %qthemes;

  print "Querying $downloadsite... ";

  $xtext = getrdfquery($downloadsite, 'themetxt', $themetext,
    'showmod', 'on', 'numthemes', '50', 'odby', 'download');

  %qthemes = rdfparse($xtext, 'name', $themetext);

  if ($config_wm{version_mismatch} eq 'no') {
    foreach (keys %qthemes) {
      delete $qthemes{$_} unless $qthemes{$_}{versionok};
    }
  }
  %qthemes;
}

sub getrdfquery {
  my ($downloadsite, %queryterms) = @_;
  my ($url, $discard, $response);

  $url = "http://$downloadsite/themes.rdf.phtml?" . makecgiline(%queryterms);

  dbugout("GETRDFQUERY: requesting $url") if $debug;

  ($discard, $response) = downloadfile(parse_url($url));
#  stowfile("wmtheme-dump-" . int(rand() * 500), $discard . $response);
  $response;
}

sub rdfparse {
  my ($xtext, %searchterms) = @_;
  my ($itemtext, $themename, %qthemes, $keepver, $verblock, $vers, $term);

  ITEM: while ($xtext =~ /<item(\s+[^>]*)?>(.+?)<\/item>/gs) {
    $itemtext = $2;
    foreach $term (keys %searchterms) {
      if (entity_text($itemtext, $term) !~ /\Q$searchterms{$term}\E/i) {
        next ITEM;
      }
    }

    $themename = entity_text($itemtext, 'name');
    $qthemes{$themename} = makethemerec();
    $qthemes{$themename}{origname}    = $themename;
    $qthemes{$themename}{id}          = entity_text($itemtext, 'id');
    $qthemes{$themename}{author}      = entity_text($itemtext, 'author');
    $qthemes{$themename}{category}    = entity_text($itemtext, 'category');
    $vers = entity_text($itemtext, 'vers');

      ## fixme -  the latest good version is found here, but it depends
      ##          on the response containing versions in ascending order,
      ##          which they are at the moment, but it's not a great
      ##          assumption.  I don't just want to 'sort' them either

    $keepver = '';
    foreach (split(/,/, $vers)) {
      if (version_ok($_) or !$keepver or !version_ok($keepver)) {
        $keepver = $_
      }
    }

    if ($keepver) {
      $qthemes{$themename}{version} = $keepver;
      $verblock = entity_contents($itemtext, "ver $keepver");
      foreach (qw(link updated size)) {
        $qthemes{$themename}{$_} = entity_text($verblock, $_);
      }
      $qthemes{$themename}{link} .= "&query=download";
      $qthemes{$themename}{versionok} = version_ok($keepver);
      dbugout("RDFPARSE: recorded \"$themename\" for version \"$keepver\"")
        if $debug;
      delete $qthemes{$themename} unless
        $qthemes{$themename}{size} =~ /^\d+/ and
        $qthemes{$themename}{size} > 0;
    } else {
      delete $qthemes{$themename};
    }
  }
  %qthemes;
}

sub entity_text {
  my ($text, $name) = @_;

  my $contents = entity_contents($text, $name);
  $contents =~ s/<\s*(\S*)\b[^>]*>[^<]*<\/\s*\1\b[^>]*>//gis;
  $contents =~ s/&amp;/&/gs;
  $contents || '';
}

sub entity_contents {
  my ($text, $name) = @_;

  if ($text =~ /<\s*\Q$name\E(?:\s+[^>]*)?>(.+?)<\/\s*\Q$name\E(?:\s+[^>]*)?>/si) {
    return $1;
  } else {
    return '';
  }
}


 ########################################################################
 #
 #  DOWNLOADFILE
 #
 #  Grabs a file.  Begins with an HTTP url whose bits are in $dlinfo.
 #
 #  Will follow redirection, use FTP if so redirected, and show a
 #  progress bar if the size of the downloaded file is available.
 #
 #  Has three output modes, depending on the nature of the request
 #  and the filename that's determined when the desired data is
 #  located:
 #
 #   1) No $destination was specified - 
 #
 #      * The file will be placed in a buffer and returned in
 #        a big wad.
 #
 #
 #   2) A $destination was specified, - AND -
 #        -  $extract was false
 #           - OR -
 #        -  $extract was true, but no extraction method could be
 #           determined from the final filename -
 #
 #      * The file will be written into the $destination directory,
 #        with the internally determined filename.
 #
 #
 #   3) A $destination was specified, - AND -
 #      $extract was true             - AND -
 #      An extraction method was determined from the final filename
 #
 #      * The data will be fed through a pipe, that extracts
 #        and writes files into the $destination directory.
 #
 #  INPUTS:
 #
 #    $dlinfo       - a reference to a hash with URL parts
 #    $destination  - optional, a directory in which to save the data
 #    $extract      - boolean, whether to try to extract the data if
 #                    its archive method can be determined
 #    $showprogress - boolean, whether to show a download gauge
 #
 #  OUTPUTS:
 #
 #    A list with two elements:
 #      - The filename
 #      - The data (will be empty if $destination was specified)
 #

sub downloadfile {
  my ($dlinfo, $destination, $extract, $showprogress) = @_;
  my ($socket, $header, $loc);
  
  dbugout("DOWNLOADFILE: getting ", urlstring($dlinfo)) if $debug;

  if ($$dlinfo{protocol} eq 'ftp') {
    $loc = $dlinfo;
  } else {
    ($socket, $header, $loc) = http_follow($dlinfo);
  }

  my (
    $input,            #  Input filehandle - returned socket or pipe from FTP utility
    $size,             #  Number of bytes to retrieve, if known (for gauge)
    $gauge,            #  String, to be displayed gradually during main transfer
    $gauge_index,      #  Position in gauge
    $buffer,           #  Destination for data if no $destination is passed
    $block,            #  a block of data during the transfer
    $blocksize,        #  amount of data to transfer so that the gauge comes out nice
    $extractcmd,       #  command to pipe data through if $extract is true
    $filename,         #  file to write data to if $extract isn't true
    $use_guage,        #  whether the size and gauge were ok, this is used during transfer
    $cols              #  screen columns (width): influences $gauge preparation
    );

  #####
  ##
  ##  SIZE, FILENAME, and INPUT depend on protocol
  ##

  if ($$loc{protocol} eq 'http') {

      ## -  SIZE  - sometimes available in http  - ##

    if ($header =~ /^content-length:\s+(\d+)/im) {
      $size = $1;
    } else {
      $size = 0;
    }

      ## -  FILENAME  - from url or header in http  - ##

    if ($destination) {
      if ($header =~ /^content-disposition:.*filename=\s*(.+?)\s/is) {
        $filename = $1;
      } elsif ($$loc{location} =~ m!/([^/]+)$!) {
        $filename = cgidecode($1);
      } else {
        choke("can't figure out a filename");
      }
    }

      ## -  INPUT  - we get to use this pre-opened socket  - ##

    $input = $socket;

  } elsif ($$loc{protocol} eq 'ftp') {

    close $socket if $socket;

      ## -  SIZE  - not available in ftp  - ##

    $size = 0;
    $showprogress = 0 if $debug;

      ## -  FILENAME  - from url in ftp  - ##

    if ($destination) {
      if ($$loc{location} =~ /([^\/]+)$/) {
        $filename = $1;
      } else {
        choke("can't get filename from ftp url!");
      }
    }

      ## -  INPUT  - must pipe from an ftp command  - ##
      # (not really: fixme)

    $input = makeftpinput($loc);

  } else {
    choke("unhandled protocol \"$$loc{protocol}\"");
  }


  ##
  ##  (end of protocol-dependent business)
  ##
  #####

 ## --- prepare output filehandle (critical if requested) --- ##

  if ($extract and ($extractcmd =
      get_extractcmd($filename, $destination, 1))) {
    dbugout("DOWNLOADFILE: piping through $extractcmd") if $debug;
    open (DEST, "|$extractcmd") or
      choke("can't pipe through \"$extractcmd\": $!");
  } elsif ($destination) {
    dbugout("DOWNLOADFILE: writing to $destination/$filename") if $debug;
    open (DEST, ">$destination/$filename") or
      choke("can't write $destination/$filename: $!");
  }
  local $SIG{PIPE} = sub { choke("child error; terminating.") };


 ## --- prepare progress gauge (no failure mode) --- ##

  $use_gauge = 0;
  if ($showprogress) {
    if ($size) {
      $cols = (screensize())[1];
      $gaugehdr = "($size bytes) [";
      $blocksize = int($size / ($cols - 1 - length($gaugehdr)));
      $blocksize++ if $size / $blocksize > $cols - 2 - length($gaugehdr);
      $gauge = makedlgauge(int($size / $blocksize));
      $use_gauge = 1;
      $gauge_index = 0;
      print $gaugehdr;
    } else {
      print "(length unknown) [";
      $blocksize = 8192;
    }
  } else {
    $blocksize = 8192;
  }


 ## --- transfer data --- ##

  local $SIG{TERM} = sub { 
    close DEST if $destination;
    choke("terminating prematurely.");
  };
  $buffer = '';
  while (read $input, $block, $blocksize) {
    if ($destination) {
      print DEST $block;
      if ($showprogress) {
        $use_gauge ? print substr($gauge, $gauge_index++, 1) : print '.';
      }
    } else {
      $buffer .= $block;
    }
  }
  print "]\n" if $showprogress;


 ## --- clean up & return --- ##

  close $input if $$loc{protocol} eq 'ftp';
  if ($destination) {
    close DEST or choke("error saving download: $!");
  }
  modtree($destination) if $extract;
  ($filename, $buffer);
}


  #########################################################################
  #
  #  HTTP_FOLLOW
  #
  #  Receives a reference to an http url hash.
  #
  #  Makes requests, starting with the passed url, following any HTTP
  #  redirections.
  #
  #  Possible outcomes:
  #
  #    *  The server returns 200:  The socket, header, and URL that
  #       led to this are returned.  The data, if any, is ready to be
  #       read from the socket.
  #
  #    *  The server redirects to a protocol other than HTTP: The
  #       header and URL found are returned (undef is returned in
  #       place of the socket).
  #
  #    *  The server returns an error:  Program aborts.
  #
  
sub http_follow {
  my ($dlinfo, $redirections) = @_;
  my ($httpsock, $header);

  dbugout("HTTP_FOLLOW: getting ", urlstring($dlinfo)) if $debug;

  set_http_proxy($dlinfo);
  ($httpsock, $header) = http_start($dlinfo);

  if ($header =~ /^http\/\S+\s+(\d+)\s+(.+?)\s*$/im) {
    $httpcode = $1;
    $httpresponse = $2;
  } else {
    close $httpsock;
    choke("Malformed HTTP response from $dlinfo->{host}");
  }

  if ($httpcode eq '200') {
    return ($httpsock, $header, $dlinfo);
  } elsif ($httpcode =~ /^30[12357]$/) {
    $redirections ||= 0;
    if ($redirections > 9) {
      choke("server issued $redirections HTTP redirections; quitting");
    } elsif ($header =~ /^location:\s+(\S+)/im) {
      my $newurl = parse_url($1);
      if ($$newurl{protocol} eq 'http') {
        return http_follow($newurl, ++$redirections);
      } else {
        return (undef, $header, $newurl);
      }
    } else {
      choke("$dlinfo->{host} replied \"$httpcode $httpresponse\" but no Location: can be found");
    }
  } else {
    choke("$dlinfo->{host} replied \"$httpcode $httpresponse\"");
  }
}


  #########################################################################
  #
  #  HTTP_START
  #
  #  Initiates an HTTP request.
  #
  #  Receives a reference to a url hash.
  #
  #  Creates a socket and opens a connection to the server; gets the
  #  returned header.  Returns the socket and the header.  Receiving
  #  any remaining data on the socket, if any, is the responsibility of
  #  the caller.
  #

sub http_start {
  my $dlinfo = shift;

  my ($iaddr, $paddr, $rqheader, $header, $line);

  dbugout("HTTP_START: getting ", urlstring($dlinfo)) if $debug;

  $iaddr = gethostbyname($dlinfo->{server}) or
    choke("can't resolve \"$dlinfo->{server}\": $!");

  $paddr = sockaddr_in($$dlinfo{port}, $iaddr) or
    choke("sockaddr_in failure with " . inet_ntoa($paddr) .
      ":$dlinfo->{port}: $!");

  socket(HTTPSOCK, PF_INET, SOCK_STREAM, getprotobyname('tcp')) or
    choke("can't get a socket: $!");

  connect(HTTPSOCK, $paddr) or
    choke("can't connect to $dlinfo->{server}:$dlinfo->{port}: $!");

  select((select(HTTPSOCK), $| = 1)[0]);

  $rqheader =
    "GET $dlinfo->{location} HTTP/1.0\n" .
    "User-Agent: wmtheme/$VERSION ($uname_srm)\n" .
    "Host: $dlinfo->{host}\n" .
    "Accept: */*\n\n";

  dbugout(" ************  REQUEST  ************",
    "$rqheader ==================================") if $debug;

  print HTTPSOCK $rqheader;

  $header = '';
  while ($line = <HTTPSOCK>) {
    $header .= $line;
    last if $line =~ /^[\r\s]*$/;
  }

  dbugout(" ++++++++++++  RESPONSE  ++++++++++++",
    "$header ----------------------------------") if $debug;

  (\*HTTPSOCK, $header);
}


  #########################################################################
  #
  #  PARSE_URL
  #

sub parse_url {
  my ($url, $noncritical) = @_;
  my %parsed;

  dbugout("PARSE_URL: Examining $url") if $debug;
  if ($url =~ m|^(\w+)://($hostmatch)(?::(\d+))?(/.*)|) {
    $parsed{protocol} = lc $1;
    $parsed{server}   = $2;
    $parsed{host}     = $2;
    $parsed{port}     = $3 || getservbyname($parsed{protocol}, 'tcp');
    $parsed{location} = $4 || '/';
    dbugout("PARSE_URL: Result: " . urlstring(\%parsed)) if $debug;
  } else {
    choke("can't parse url $url") unless $noncritical;
    dbugout("PARSE_URL: Can't parse.") if $debug;
  }

  \%parsed;
}

  # See http://www.w3.org/Daemon/User/Proxies/ProxyClients.html

sub set_http_proxy {
  my $dlinfo = shift;
  my ($proxy, $np);

  if ($np = $ENV{no_proxy} || $ENV{NO_PROXY}) {
    foreach (split /[\s,]+/, $np) {
      return if $dlinfo->{server} =~ /\b\Q$_\E$/;
    }
  }

  $proxy = $ENV{http_proxy} || $ENV{HTTP_PROXY} || $ENV{all_proxy} ||
           $ENV{ALL_PROXY} || return;

  if ($proxy =~ m!^(?i:http://)?($hostmatch)(?::(\d+))?!) {
    $dlinfo->{location} =
      "$dlinfo->{protocol}://$dlinfo->{server}:$dlinfo->{port}$dlinfo->{location}";
    $dlinfo->{host}     = $dlinfo->{server};
    $dlinfo->{protocol} = 'http';
    $dlinfo->{server}   = $1;
    $dlinfo->{port}     = $2 || 80;
    dbugout("HTTP_PROXY:  host=$dlinfo->{server}  port=$dlinfo->{port}")
      if $debug;
  } else {
    complain("HTTP_PROXY: can't parse proxy \"$proxy\"");
  }
}


  #########################################################################
  #
  #  URLSTRING
  #

sub urlstring {
  my $url = shift;

  "[$$url{protocol}] :// [$$url{server}] : [$$url{port}] [$$url{location}]";
}

  #########################################################################
  #
  #  MAKEFTPINPUT
  #

sub makeftpinput {
  my $loc = shift;
  my $ftpsource;

  if (!$ftp_command) {
    my $tmpcmd;
    my @ftp_list = qw(wget curl ncftp);  # This defines the order of preference.
    my %ftpcommands = (
      'wget',      '-O - ftp://HOSTLOCATION',
      'curl',      'ftp://HOST/LOCATION'
      );
    my %quietswitches = (
      'wget', '-q',
      'curl', '-sS'
    );

    foreach $tmpcmd(@ftp_list) {
      if (findexe($tmpcmd)) {
        if ($debug) {
          $ftp_command = "$tmpcmd $ftpcommands{$tmpcmd}";
        } else {
          $ftp_command = "$tmpcmd $quietswitches{$tmpcmd} $ftpcommands{$tmpcmd}";
        }
        dbugout("identified FTP command: $ftp_command") if $debug;
        last;
      }
    }
    if (!$ftp_command) {
      choke("no FTP command from [", join(', ', @ftp_list), "] is available");
    }
  }

  $ftpsource = $ftp_command;
  $ftpsource =~ s/HOST/$$loc{server}/ or
    choke("internal error: \"$ftpsource\" didn't contain 'HOST'");
  $ftpsource =~ s/LOCATION/$$loc{location}/ or
    choke("internal error: \"$ftpsource\" didn't contain 'LOCATION'");
  open FTPIN, "$ftpsource|" or
    choke("can't pipe from $ftpsource: $!");
  return \*FTPIN;
}

  #########################################################################
  #
  #  MAKEDLGAUGE
  #

sub makedlgauge {
  my $glength = shift;

  return '.' if $glength < 1;

  my @pctmarkers = qw(0 10 20 30 40 50 60 70 80 90 100);
  my $pctlength = length(join('', @pctmarkers));

  return '.' x $glength if $glength < $pctlength + 10;

  my $smallsect = int(($glength - $pctlength)/ 10);
  my $gauge = $pctmarkers[0];

  $addmarks = substr('385027916', 0, $glength - $pctlength - $smallsect * 10);
  for (my $i = 0; $i < 10;) {
    $gauge .= '.' x ($smallsect + $addmarks =~ /$i/) . $pctmarkers[++$i];
  }

  $gauge;
}

1;

