# linux-lib.pl
# Quota functions for all linux version

# quotas_init()
sub quotas_init
{
if (&has_command("quotaon") && &has_command("quotaoff")) {
	return undef;
	}
else {
	return "The quotas package does not appear to be installed on ".
	       "your system\n";
	}
}

# quotas_supported()
# Returns 1 for user quotas, 2 for group quotas or 3 for both
sub quotas_supported
{
return 3;
}

# free_space(filesystem)
# Returns an array containing  btotal, bfree, ftotal, ffree
sub free_space
{
local(@out, @rv);
$out = `df $_[0]`;
$out =~ /Mounted on\n\S+\s+(\d+)\s+\d+\s+(\d+)/;
push(@rv, ($1, $2));
$out = `df -i $_[0]`;
$out =~ /Mounted on\n\S+\s+(\d+)\s+\d+\s+(\d+)/;
push(@rv, ($1, $2));
return @rv;
}

# quota_can(&mnttab, &fstab)
# Can this filesystem type support quotas?
#  0 = No quota support (or not turned on in /etc/fstab)
#  1 = User quotas only
#  2 = Group quotas only
#  3 = User and group quotas
sub quota_can
{
return ($_[1]->[3] =~ /usrquota/ ? 1 : 0) +
       ($_[1]->[3] =~ /grpquota/ ? 2 : 0);
}

# quota_now(&mnttab, &fstab)
# Are quotas currently active?
#  0 = Not active
#  1 = User quotas active
#  2 = Group quotas active
#  3 = Both active
sub quota_now
{
local $rv = 0;
local $dir = $_[0]->[0];
if ($_[0]->[4]%2 == 1) {
	# test user quotas
	if (-r "$dir/quota.user") {
		$out = `$config{'user_quotaon_command'} $dir 2>&1`;
		if ($out =~ /Device or resource busy/i) {
			# already on..
			$rv += 1;
			}
		elsif ($out =~ /Package not installed/i) {
			# No quota support!
			return 0;
			}
		else {
			# was off.. need to turn on again
			`$config{'user_quotaoff_command'} $dir 2>&1`;
			}
		}
	}
if ($_[0]->[4] > 1) {
	# test group quotas
	if (-r "$dir/quota.group") {
		$out = `$config{'group_quotaon_command'} $dir 2>&1`;
		if ($out =~ /Device or resource busy/i) {
			# already on..
			$rv += 2;
			}
		elsif ($out =~ /Package not installed/i) {
			# No quota support!
			return 0;
			}
		else {
			# was off.. need to turn on again
			`$config{'group_quotaoff_command'} $dir 2>&1`;
			}
		}
	}
return $rv;
}

# quotaon(filesystem, mode)
# Activate quotas and create quota files for some filesystem. The mode can
# be 1 for user only, 2 for group only or 3 for user and group
sub quotaon
{
local($out, $qf, @qfile, $flags);
if ($_[1]%2 == 1) {
	# turn on user quotas
	$qf = "$_[0]/quota.user";
	if (!(-r $qf)) {
		open(QUOTAFILE, "> $qf"); close(QUOTAFILE);
		chmod(0600, $qf);
		&system_logged("$config{'quotacheck_command'} $_[0]");
		}
	$out = &backquote_logged("$config{'user_quotaon_command'} $_[0] 2>&1");
	if ($?) { return $out; }
	}
if ($_[1] > 1) {
	# turn on group quotas
	$qf = "$_[0]/quota.group";
	if (!(-r $qf)) {
		open(QUOTAFILE, "> $qf"); close(QUOTAFILE);
		chmod(0600, $qf);
		&system_logged("$config{'quotacheck_command'} $_[0]");
		}
	$out = &backquote_logged("$config{'group_quotaon_command'} $_[0] 2>&1");
	if ($?) { return $out; }
	}
return undef;
}

# quotaoff(filesystem, mode)
# Turn off quotas for some filesystem
sub quotaoff
{
local($out);
if ($_[1]%2 == 1) {
	$out = &backquote_logged("$config{'user_quotaoff_command'} $_[0] 2>&1");
	if ($?) { return $out; }
	}
if ($_[1] > 1) {
	$out = &backquote_logged("$config{'group_quotaoff_command'} $_[0] 2>&1");
	if ($?) { return $out; }
	}
return undef;
}

# user_filesystems(user)
# Fills the array %filesys with details of all filesystem some user has
# quotas on
sub user_filesystems
{
return &parse_quota_output("$config{'user_quota_command'} $_[0]");
}

# user_filesystems(user)
# Fills the array %filesys with details of all filesystem some group has
# quotas on
sub group_filesystems
{
return &parse_quota_output("$config{'group_quota_command'} $_[0]");
}

sub parse_quota_output
{
local($n, $_, %mtab);
open(MTAB, "/etc/mtab");
while(<MTAB>) {
	@m = split(/\s+/);
	$mtab{$m[0]} = $m[1];
	}
close(MTAB);
open(QUOTA, "$_[0] |");
$n=0; while(<QUOTA>) {
	chop;
	if (/^(Disk|\s+Filesystem)/) { next; }
	if (/^(\S+)$/) {
		# Bogus wrapped line
		$filesys{$n,'filesys'} = $mtab{$1};
		<QUOTA>=~/^.{15}.(.{7}).(.{7}).(.{7}).{8}.(.{7}).(.{7}).(.{7})/;
		$filesys{$n,'ublocks'} = int($1);
		$filesys{$n,'sblocks'} = int($2);
		$filesys{$n,'hblocks'} = int($3);
		$filesys{$n,'ufiles'} = int($4);
		$filesys{$n,'sfiles'} = int($5);
		$filesys{$n,'hfiles'} = int($6);
		$n++;
		}
	elsif (/^(.{15}).(.{7}).(.{7}).(.{7}).{8}.(.{7}).(.{7}).(.{7})/) {
		$filesys{$n,'ublocks'} = int($2);
		$filesys{$n,'sblocks'} = int($3);
		$filesys{$n,'hblocks'} = int($4);
		$filesys{$n,'ufiles'} = int($5);
		$filesys{$n,'sfiles'} = int($6);
		$filesys{$n,'hfiles'} = int($7);
		$dev = $1; $dev =~ s/\s+$//g; $dev =~ s/^\s+//g;
		$filesys{$n,'filesys'} = $mtab{$dev};
		$n++;
		}
	}
close(QUOTA);
return $n;
}

# filesystem_users(filesystem)
# Fills the array %user with information about all users with quotas
# on this filesystem. This may not be all users on the system..
sub filesystem_users
{
return &parse_repquota_output(
	"$config{'user_repquota_command'} $_[0]", "user");
}

sub filesystem_groups
{
return &parse_repquota_output(
	"$config{'group_repquota_command'} $_[0]", "group");
}

sub parse_repquota_output
{
local($rep, @rep, $n, $what, $u, @uinfo);
$what = $_[1];
$rep = `$_[0] 2>&1`;
if ($?) { return -1; }
if ($what eq 'user') {
	setpwent();
	while(@uinfo = getpwent()) {
		$hasu{$uinfo[0]}++;
		}
	endpwent();
	}
else {
	setgrent();
	while(@uinfo = getgrent()) {
		$hasu{$uinfo[0]}++;
		}
	endgrent();
	}
@rep = split(/\n/, $rep); @rep = @rep[3..$#rep];
for($n=0; $n<@rep; $n++) {
	if ($rep[$n] =~ /(\S+)\s*[\-\+]{2}(.{8})(.{8})(.{8}).{7}(.{8})(.{6})(.{6})/) {
		$$what{$n,$what} = $1;
		$$what{$n,'ublocks'} = int($2);
		$$what{$n,'sblocks'} = int($3);
		$$what{$n,'hblocks'} = int($4);
		$$what{$n,'ufiles'} = int($5);
		$$what{$n,'sfiles'} = int($6);
		$$what{$n,'hfiles'} = int($7);
		if ($$what{$n,$what} !~ /^\d+$/ && !$hasu{$$what{$n,$what}}) {
			# User/group name was truncated! Try to find him..
			foreach $u (keys %hasu) {
				if (substr($u, 0, length($$what{$n,$what})) eq
				    $$what{$n,$what}) {
					# found him..
					$$what{$n,$what} = $u;
					last;
					}
				}
			}
		}
	}
return $n;
}

# edit_quota_file(data, filesys, sblocks, hblocks, sfiles, hfiles)
sub edit_quota_file
{
local($rv, $line, %mtab, @m);
open(MTAB, "/etc/mtab");
while(<MTAB>) {
	@m = split(/\s+/);
	$mtab{$m[0]} = $m[1];
	}
close(MTAB);
@line = split(/\n/, $_[0]);
for($i=0; $i<@line; $i++) {
	if ($line[$i] =~ /^(\S+): blocks in use: (\d+), limits \(soft = (\d+), hard = (\d+)\)$/ && $mtab{$1} eq $_[1]) {
		# found lines to change
		$rv .= "$1: blocks in use: $2, limits (soft = $_[2], hard = $_[3])\n";
		$line[++$i] =~ /^\s*inodes in use: (\d+), limits \(soft = (\d+), hard = (\d+)\)$/;
		$rv .= "\tinodes in use: $1, limits (soft = $_[4], hard = $_[5])\n";
		}
	else { $rv .= "$line[$i]\n"; }
	}
return $rv;
}

# quotacheck(filesystem, mode)
# Runs quotacheck on some filesystem
sub quotacheck
{
$out = &backquote_logged("$config{'quotacheck_command'} $_[0] 2>&1");
if ($?) { return $out; }
return undef;
}

# copy_user_quota(user, [user]+)
# Copy the quotas for some user to many others
sub copy_user_quota
{
for($i=1; $i<@_; $i++) {
	$out = &backquote_logged("$config{'user_copy_command'} $_[0] $_[$i] 2>&1");
	if ($?) { return $out; }
	}
return undef;
}

# copy_group_quota(group, [group]+)
# Copy the quotas for some group to many others
sub copy_group_quota
{
for($i=1; $i<@_; $i++) {
	$out = &backquote_logged("$config{'group_copy_command'} $_[0] $_[$i] 2>&1");
	if ($?) { return $out; }
	}
return undef;
}

# get_user_grace(filesystem)
# Returns an array containing  btime, bunits, ftime, funits
# The units can be 0=sec, 1=min, 2=hour, 3=day
sub get_user_grace
{
return &parse_grace_output($config{'user_grace_command'}, $_[0]);
}

# get_group_grace(filesystem)
# Returns an array containing  btime, bunits, ftime, funits
# The units can be 0=sec, 1=min, 2=hour, 3=day
sub get_group_grace
{
return &parse_grace_output($config{'group_grace_command'}, $_[0]);
}

# default_grace()
# Returns 0 if grace time can be 0, 1 if zero grace means default
sub default_grace
{
return 0;
}

sub parse_grace_output
{
local(@rv, %mtab, @m);
open(MTAB, "/etc/mtab");
while(<MTAB>) { @m = split(/\s+/); $mtab{$m[0]} = $m[1]; }
close(MTAB);
$ENV{'EDITOR'} = $ENV{'VISUAL'} = "cat";
open(GRACE, "$_[0] |");
while(<GRACE>) {
	if (/^(\S+): block grace period: (\d+) (\S+), file grace period: (\d+) (\S+)/ && $mtab{$1} eq $_[1]) {
		@rv = ($2, $name_to_unit{$3}, $4, $name_to_unit{$5});
		}
	}
close(GRACE);
return @rv;
}

# edit_grace_file(data, filesystem, btime, bunits, ftime, funits)
sub edit_grace_file
{
local($rv, $line, @m, %mtab);
open(MTAB, "/etc/mtab");
while(<MTAB>) { @m = split(/\s+/); $mtab{$m[0]} = $m[1]; }
close(MTAB);
foreach $line (split(/\n/, $_[0])) {
	if ($line =~ /^(\S+): block grace period: (\d+) (\S+), file grace period: (\d+) (\S+)/ && $mtab{$1} eq $_[1]) {
		# replace this line
		$line = "$1: block grace period: $_[2] $unit_to_name{$_[3]}, file grace period: $_[4] $unit_to_name{$_[5]}";
		}
	$rv .= "$line\n";
	}
return $rv;
}

# grace_units()
# Returns an array of possible units for grace periods
sub grace_units
{
return ($text{'grace_seconds'}, $text{'grace_minutes'}, $text{'grace_hours'},
	$text{'grace_days'});
}

%name_to_unit = ( "second", 0, "seconds", 0,
		  "minute", 1, "minutes", 1,
		  "hour", 2, "hours", 2,
		  "day", 3, "days", 3,
		);
foreach $k (keys %name_to_unit) {
	$unit_to_name{$name_to_unit{$k}} = $k;
	}

1;

