# core.pl
# Defines the core module directives

# core_directives(version)
# Returns ar array of references to associative arrays, each containing
# information about some directive. The keys of each array are:
#  name -	The name of this directive
#  type -	What kind of directive this in. Possible values are
#		0 - Processes and limits
#		1 - Networking and addresses
#		2 - Apache Modules
#		3 - Log files
#		4 - Access control
#		5 - document options
#		6 - MIME types
#		7 - Error handling
#		8 - Users and Groups
#		9 - Miscellaneous
#		10- Aliases and redirects
#		11- CGI programs
#		12- Directory indexing
#		13- Proxying
#		14- SSL
#		15- Perl
#		16- PHP
#  multiple -	Can this directive appear multiple times
#  global -	Can be used in the global server context
#  virtual -	Can be used in a VirtualHost section or in the global section
#  directory -	Can be used in a Directory section context
#  htaccess -	Can be used in a .htaccess file
sub core_directives
{
local($rv);
$rv = [	[ 'AccessFileName', 0, 5, 'virtual', undef, 5 ],
	[ 'ClearModuleList AddModule', 1, 2, 'global' ],
	[ 'AllowOverride', 0, 5, 'directory' ],
	[ 'AuthName', 0, 4, 'directory htaccess', undef, 10 ],
	[ 'AuthType', 0, 4, 'directory htaccess', undef, 8 ],
	[ 'BindAddress Listen Port', 1, 1, 'global' ],
	[ 'ContentDigest', 0, 5, 'virtual directory htaccess' ],
	[ 'CoreDumpDirectory', 0, 9, 'global', 1.3 ],
	[ 'DefaultType', 0, 6, 'virtual directory htaccess' ],
	[ 'DocumentRoot', 0, 5, 'virtual', undef, 10 ],
	[ 'ErrorDocument', 1, 7, 'virtual directory htaccess' ],
	[ 'ErrorLog', 0, 3, 'virtual' ],
	[ 'Group', 0, 8, 'virtual' ],
	[ 'HostNameLookups', 0, 1, 'virtual directory htaccess' ],
	[ 'IdentityCheck', 0, 1, 'virtual directory htaccess' ],
	[ 'KeepAlive MaxKeepAliveRequests', 0, 1, 'global' ],
	[ 'KeepAliveTimeout', 0, 1, 'global' ],
	[ 'ListenBacklog', 0, 1, 'global', 1.2 ],
	[ 'LockFile', 0, 9, 'global' ],
	[ 'LimitRequestBody', 0, 0, 'virtual directory htaccess', 1.302 ],
	[ 'LimitRequestFields', 0, 0, 'global', 1.302 ],
	[ 'LimitRequestFieldsize', 0, 0, 'global', 1.302 ],
	[ 'LimitRequestLine', 0, 0, 'global', 1.302 ],
	[ 'MaxClients', 0, 0, 'global' ],
	[ 'MaxRequestsPerChild', 0, 0, 'global' ],
	[ 'StartServers', 0, 0, 'global' ],
	[ 'MinSpareServers', 0, 0, 'global' ],
	[ 'MaxSpareServers', 0, 0, 'global' ],
	[ 'NameVirtualHost', 1, 1, 'global', 1.3 ],
	[ 'Options', 0, 5, 'virtual directory htaccess', undef, 3 ],
	[ 'PidFile', 0, 9, 'global' ],
	[ 'require', 0, 4, 'directory htaccess', undef, 6 ],
	[ 'RLimitCPU', 0, 0, 'virtual', 1.2 ],
	[ 'RLimitMEM', 0, 0, 'virtual', 1.2 ],
	[ 'RLimitNPROC', 0, 0, 'virtual', 1.2 ],
	[ 'Satisfy', 0, 4, 'directory htaccess', 1.2, 4 ],
	[ 'ScoreBoardFile', 0, 9, 'global', 1.2 ],
	[ 'SendBufferSize', 0, 1, 'global' ],
	[ 'ServerAdmin', 0, 1, 'virtual' ],
	[ 'ServerAlias', 1, 1, 'virtual virtualonly', 1.2 ],
	[ 'ServerName', 0, 1, 'virtual' ],
	[ 'ServerPath', 0, 5, 'virtual' ],
	[ 'ServerType', 0, 9, 'global' ],
	[ 'ServerTokens', 0, 9, 'global', 1.3 ],
	[ 'ServerSignature', 0, 5, 'virtual directory htaccess', 1.3 ],
	[ 'TimeOut', 0, 1, 'global' ],
	[ 'UseCanonicalName', 0, 1, 'virtual directory htaccess', 1.3 ],
	[ 'User', 0, 8, 'virtual', undef, 10 ] ];
return &make_directives($rv, $_[0], "core");
}

# core_handlers(config, version)
# Returns an array of all available handlers
sub core_handlers
{
return ();
}

#########################################################################
# Process and limit directives
sub edit_MaxClients
{
return (1,
	$text{'core_0'},
	&opt_input($_[0]->{'value'}, "MaxClients", $text{'core_12'}, 4));
}
sub save_MaxClients
{
return &parse_opt("MaxClients", '^\d+$',
		  $text{'core_6'});
}

sub edit_MaxKeepAliveRequests
{
return (1,
	$text{'core_1'},
	&opt_input($_[0]->{'value'}, "MaxKeepAliveRequests", $text{'core_13'}, 4));
}
sub save_MaxKeepAliveRequests
{
return &parse_opt("MaxKeepAliveRequests", '^\d+$',
		  $text{'core_7'});
}

sub edit_MaxRequestsPerChild
{
return (1,
	$text{'core_2'},
	&opt_input($_[0]->{'value'}, "MaxRequestsPerChild", $text{'core_14'}, 5));
}
sub save_MaxRequestsPerChild
{
return &parse_opt("MaxRequestsPerChild", '^\d+$',
		  $text{'core_8'});
}

sub edit_MinSpareServers
{
return (1,
	$text{'core_3'},
	&opt_input($_[0]->{'value'},"MinSpareServers",$text{'core_15'}, 4));
}
sub save_MinSpareServers
{
return &parse_opt("MinSpareServers", '^\d+$',
		  $text{'core_9'});
}

sub edit_MaxSpareServers
{
return (1,
	$text{'core_4'},
	&opt_input($_[0]->{'value'},"MaxSpareServers",$text{'core_16'}, 4));
}
sub save_MaxSpareServers
{
return &parse_opt("MaxSpareServers", '^\d+$',
		  $text{'core_10'});
}

sub edit_StartServers
{
return (1,
	$text{'core_5'},
	&opt_input($_[0]->{'value'}, "StartServers", $text{'core_17'}, 4));
}
sub save_StartServers
{
return &parse_opt("StartServers", '^\d+$',
		  $text{'core_11'});
}

sub edit_RLimitCPU
{
return &rlimit_input("RLimitCPU", $text{'core_18'}, $_[0]);
}
sub save_RLimitCPU
{
return &parse_rlimit("RLimitCPU", $text{'core_21'});
}

sub edit_RLimitMEM
{
return &rlimit_input("RLimitMEM", $text{'core_19'}, $_[0]);
}
sub save_RLimitMEM
{
return &parse_rlimit("RLimitMEM", $text{'core_22'});
}

sub edit_RLimitNPROC
{
return &rlimit_input("RLimitNPROC", $text{'core_20'}, $_[0]);
}
sub save_RLimitNPROC
{
return &parse_rlimit("RLimitNPROC", $text{'core_23'});
}

# rlimit_input(name, desc, value)
sub rlimit_input
{
local(@v, $rv);
@v = split(/\s+/, $_[2]->{'value'});
$rv = sprintf "<input type=radio name=$_[0]_mode value=0 %s> $text{'Default'}<br>\n",
	@v ? "" : "checked";
$rv .= sprintf "<input type=radio name=$_[0]_mode value=1 %s>\n",
	@v == 1 ? "checked" : "";
$rv .= sprintf "$text{'core_24'}<input name=$_[0]_soft1 size=5 value=\"%s\"><br>\n",
	@v == 1 ? $v[0] : "";
$rv .= sprintf "<input type=radio name=$_[0]_mode value=2 %s>\n",
	@v == 2 ? "checked" : "";
$rv .= sprintf "$text{'core_25'}<input name=$_[0]_soft2 size=5 value=\"%s\">\n",
	@v == 2 ? $v[0] : "";
$rv .= sprintf "$text{'core_26'}<input name=$_[0]_hard2 size=5 value=\"%s\"><br>\n",
	@v == 2 ? $v[1] : "";
return (1, $_[1], $rv);
}

# parse_rlimit(name, desc)
sub parse_rlimit
{
if ($in{"$_[0]_mode"} == 0) { return ( [ ] ); }
elsif ($in{"$_[0]_mode"} == 1) {
	$in{"$_[0]_soft1"} =~ /^(\d+|max)$/ ||
		&error($in{"$_[0]_soft1"}.&text('core_27', $_[1]));
	return ( [ $in{"$_[0]_soft1"} ] );
	}
elsif ($in{"$_[0]_mode"} == 2) {
	$in{"$_[0]_soft2"} =~ /^(\d+|max)$/ ||
		&error($in{"$_[0]_soft2"}.&text('core_28', $_[1]));
	$in{"$_[0]_hard2"} =~ /^(\d+|max)$/ ||
		&error($in{"$_[0]_hard2"}.&text('core_29', $_[1]));
	return ( [ $in{"$_[0]_soft2"}." ".$in{"$_[0]_hard2"} ] );
	}
}


#########################################################################
# Networking and address directives
sub edit_BindAddress_Listen_Port
{
local($bref, $lref, $pref, @blist, @plist, $inp);
$bref = $_[0]; $lref = $_[1]; $pref = $_[2];
if (@$lref) {
	# listen directives in use.. so BindAddress and Port are unused
	foreach $l (@$lref) {
		if ($l->{'value'} =~ /^(\S+):(\d+)$/) {
			push(@blist, $1); push(@plist, $2);
			}
		elsif ($l->{'value'} =~ /^(\d+)$/) {
			push(@blist, "*"); push(@plist, $1);
			}
		elsif ($l->{'value'} =~ /^(\S+)$/) {
			push(@blist, $1); push(@plist, undef);
			}
		}
	}
else {
	# no listen directives... check for BindAddress
	if (@$bref) { push(@blist, $bref->[@$bref-1]->{'value'}); }
	else { push(@blist, "*"); }
	push(@plist, undef);
	}
$port = @$pref ? $pref->[@$pref-1]->{'value'} : 80;
$inp = "<b>$text{'core_30'}</b> <input name=Port size=6 value=\"$port\"><br>\n";
$inp .=	"<table border>\n".
	"<tr $tb> <td><b>$text{'core_31'}</b></td> <td><b>$text{'core_32'}</b></td> </tr>\n";
for($i=0; $i<@blist+1; $i++) {
	$inp .= sprintf
		"<tr><td><input type=radio name=BindAddress_def_$i value=2 %s>".
		" $text{'core_201'} <input type=radio name=BindAddress_def_$i value=1 %s>".
		" $text{'core_202'} <input type=radio name=BindAddress_def_$i value=0 %s> ".
		"<input name=BindAddress_$i size=20 value=\"%s\"></td>",
		$blist[$i] ? "" : "checked",
		$blist[$i] eq "*" ? "checked" : "",
		$blist[$i] && $blist[$i] ne "*" ? "checked" : "",
		$blist[$i] eq "*" ? "" : $blist[$i];
	$inp .= "<td>".&opt_input($plist[$i], "Port_$i", "$text{'core_34'}", 5)."</td>";
	}
$inp .= "</table>\n";
return (2, $text{'core_35'}, $inp);
}
sub save_BindAddress_Listen_Port
{
local(@blist, @plist, $bdef, $b, $p);

# build list of addresses and ports
for($i=0; defined($in{"Port_$i"}); $i++) {
	$bdef = $in{"BindAddress_def_$i"}; $b = $in{"BindAddress_$i"};
	$pdef = $in{"Port_${i}_def"}; $p = $in{"Port_$i"};
	if ($bdef == 2) { next; }

	if ($bdef) { push(@blist, "*"); }
	elsif ($b =~ /^\S+$/ && gethostbyname($b)) { push(@blist, $b); }
	else { &error(&text('core_36', $b)); }

	if ($pdef) { push(@plist, undef); }
	elsif ($p =~ /^\d+$/) { push(@plist, $p); }
	else { &error(&text('core_37', $p)); }
	}
if (!@blist) { &error($text{'core_38'}); }

# Return directives
$in{'Port'} =~ /^\d+$/ || &error($text{'core_39'});
if (@blist == 1 && !$plist[0]) {
	# Only one address, and the default port
	return ( $blist[0] eq "*" ? [ ] : [ $blist[0] ], [ ], [ $in{'Port'} ] );
	}
else {
	# More than one address, or a non-default port. Must use Listens
	for($i=0; $i<@blist; $i++) {
		if ($blist[$i] ne "*" && $plist[$i]) {
			push(@l, "$blist[$i]:$plist[$i]");
			}
		elsif ($blist[$i] ne "*") { push(@l, $blist[$i]); }
		elsif ($plist[$i]) { push(@l, "*:$plist[$i]"); }
		else { push(@l, $in{'Port'}); }
		}
	return ( [], \@l, [ $in{'Port'} ] );
	}
}

sub edit_KeepAlive_MaxKeepAliveRequests
{
$kref = $_[0]; $mref = $_[1];
if ($_[2]->{'version'} >= 1.2) {
	# two separate directives for keep-alives
	$inp = sprintf
		"<input type=radio name=KeepAlive_def value=0 %s> $text{'core_203'}\n".
		"<input type=radio name=KeepAlive_def value=1 %s> $text{'core_204'}\n".
		"<input type=radio name=KeepAlive_def value=2 %s> ".
		"<input name=KeepAlive size=5 value=\"%s\">",
		$kref->{'value'} =~ /off/i ? "checked" : "",
		$kref->{'value'} !~ /off/i && !$mref ? "checked" : "",
		$mref ? "checked" : "",
		$mref ? $mref->{'value'} : "";
	}
else {
	# only one directive
	$inp = sprintf
		"<input type=radio name=KeepAlive_def value=0 %s> $text{'core_205'}\n".
		"<input type=radio name=KeepAlive_def value=1 %s> $text{'core_206'}\n".
		"<input type=radio name=KeepAlive_def value=2 %s> ".
		"<input name=KeepAlive size=5 value=\"%s\">",
		$kref->{'value'} eq "0" ? "checked" : "",
		$kref ? "" : "checked" ,
		$kref->{'value'} ? "checked" : "",
		$kref->{'value'} ? $kref->{'value'} : "";
	}
return (1, $text{'core_40'}, $inp);
}
sub save_KeepAlive_MaxKeepAliveRequests
{
if ($_[0]->{'version'} >= 1.2) {
	# two separate directives
	if ($in{'KeepAlive_def'} == 0) { return ( [ "off" ], [ ] ); }
	elsif ($in{'KeepAlive_def'} == 1) { return ( [ "on" ], [ ] ); }
	elsif ($in{'KeepAlive'} !~ /^\d+$/) {
		&error(&text('core_41', $in{'KeepAlive'}));
		}
	else { return ( [ "on" ], [ $in{'KeepAlive'} ] ); }
	}
else {
	# only one directive
	if ($in{'KeepAlive_def'} == 0) { return ( [ 0 ], [ ] ); }
	elsif ($in{'KeepAlive_def'} == 1) { return ( [ ], [ ] ); }
	elsif ($in{'KeepAlive'} !~ /^\d+$/) {
		&error(&text('core_78', $in{'KeepAlive'}));
		}
	else { return ( [ $in{'KeepAlive'} ], [ ] ); }
	}
}

sub edit_KeepAliveTimeout
{
return (1,
	$text{'core_55'},
	&opt_input($_[0]->{'value'}, "KeepAliveTimeout", $text{'core_72'}, 5));
}
sub save_KeepAliveTimeout
{
return &parse_opt("KeepAliveTimeout", '^\d+$',
		  $text{'core_67'});
}

sub edit_ListenBacklog
{
return (1,
	$text{'core_56'},
	&opt_input($_[0]->{'value'}, "ListenBacklog", $text{'core_73'}, 4));
}
sub save_ListenBacklog
{
return &parse_opt("ListenBacklog", '^\d+$',
		  $text{'core_68'});
}

sub edit_SendBufferSize
{
return (1,
	$text{'core_57'},
	&opt_input($_[0]->{'value'}, "SendBufferSize", $text{'core_74'}, 4));
}
sub save_SendBufferSize
{
return &parse_opt("SendBufferSize", '^\d+$',
		  $text{'core_69'});
}

sub edit_ServerAdmin
{
return (1, $text{'core_58'},
        &opt_input($_[0]->{'value'}, "ServerAdmin", $text{'core_75'}, 25));
}
sub save_ServerAdmin
{
return &parse_opt("ServerAdmin");
}

sub edit_TimeOut
{
return (1,
	$text{'core_59'},
	&opt_input($_[0]->{'value'}, "TimeOut", $text{'core_76'}, 4));
}
sub save_TimeOut
{
return &parse_opt("TimeOut", '^\d+$',
		  $text{'core_70'});
}

sub edit_UseCanonicalName
{
return (1, $text{'core_60'},
        &choice_input($_[0]->{'value'}, "UseCanonicalName",
	               "", "$text{'core_42'},off", "$text{'core_43'},on", "$text{'core_44'},"));
}
sub save_UseCanonicalName
{
return &parse_choice("UseCanonicalName", "");
}

sub edit_HostNameLookups
{
if ($_[1]->{'version'} >= 1.3) {
	return (1, $text{'core_61'},
		&choice_input($_[0]->{'value'}, "HostNameLookups", "",
		       "$text{'core_45'},off", "$text{'core_46'},on", "$text{'core_47'},double", "$text{'core_48'},"));
	}
else {
	return (1, $text{'core_62'},
		&choice_input($_[0]->{'value'}, "HostNameLookups", "",
			      "$text{'core_49'},on", "$text{'core_50'},off", "$text{'core_51'},"));
	}
}
sub save_HostNameLookups
{
return &parse_choice("HostNameLookups", "");
}

sub edit_IdentityCheck
{
return (1, $text{'core_63'},
	&choice_input($_[0]->{'value'}, "IdentityCheck", "",
		      "$text{'core_52'},on", "$text{'core_53'},off", "$text{'core_54'},"));
}
sub save_IdentityCheck
{
return &parse_choice("IdentityCheck", "");
}

sub edit_ServerAlias
{
local($a, @al);
foreach $a (@{$_[0]}) { push(@al, split(/\s+/, $a->{'value'})); }
return (1, $text{'core_64'},
	sprintf "<textarea name=ServerAlias rows=3 cols=25>%s</textarea>\n",
		join("\n", @al) );
}
sub save_ServerAlias
{
local(@al);
@al = split(/\s+/, $in{'ServerAlias'});
if (@al) { return ( [ join(' ', @al) ] ) }
else { return ( [ ] ); }
}

sub edit_ServerName
{
return (1, $text{'core_65'},
	&opt_input($_[0]->{'value'}, "ServerName", $text{'core_77'}, 25));
}
sub save_ServerName
{
return &parse_opt("ServerName", '^\S+$', $text{'core_71'});
}

sub edit_NameVirtualHost
{
local(@nv, $nv);
foreach $nv (@{$_[0]}) { push(@nv, $nv->{'value'}); }
return (1, $text{'core_66'},
	"<textarea name=NameVirtualHost rows=3 cols=30>".
	join("\n", @nv)."</textarea>");
}
sub save_NameVirtualHost
{
local(@nv, $nv, $addr);
@nv = split(/\s+/, $in{'NameVirtualHost'});
foreach $nv (@nv) {
	if ($nv =~ /^(\S+):(\d+)$/) { $addr = $1; }
	else { $addr = $nv; }
	if (!gethostbyname($addr)) {
		&error(&text('core_79', $addr));
		}
	}
if (@nv) { return ( \@nv ); }
else { return ( [ ] ); }
}

#########################################################################
# Document directives
sub edit_AccessFileName
{
return (1,
        $text{'core_103'},
        &opt_input($_[0]->{'value'}, "AccessFileName", $text{'core_110'}, 20));
}
sub save_AccessFileName
{
if ($_[0]->{'version'} < 1.3) {
	return &parse_opt("AccessFileName", '^(\S+)$',
	                  $text{'core_108'});
	}
else {
	return &parse_opt("AccessFileName", '\S',
			     $text{'core_109'});
	}
}

@AllowOverride_v = ("AuthConfig", "FileInfo", "Indexes", "Limit", "Options");
@AllowOverride_d = ("$text{'core_80'}", "$text{'core_81'}",
                    "$text{'core_82'}", "$text{'core_83'}",
                    "$text{'core_84'}");
sub edit_AllowOverride
{
local($rv, @ov, %over, $rv);
$rv = &choice_input($_[0] ? 0 : 1, "AllowOverride_def", 1,
		 "$text{'core_85'},1", "$text{'core_86'},0");
$rv .= "<table border><tr><td>\n";
if ($_[0]) { @ov = split(/\s+/, $_[0]->{'value'}); }
else { @ov = ("All"); }
foreach $ov (@ov) { $over{$ov}++; }
if ($over{'All'}) { foreach $ov (@AllowOverride_v) { $over{$ov}++; }	}
elsif ($over{'None'}) { %over = (); }

for($i=0; $i<@AllowOverride_v; $i++) {
	$rv .= sprintf "<input type=checkbox name=AllowOverride_%s %s> %s<br>\n",
	        $AllowOverride_v[$i],
	        $over{$AllowOverride_v[$i]} ? "checked" : "",
	        $AllowOverride_d[$i];
	}
$rv .= "</td></tr></table>\n";
return (1, $text{'core_104'}, $rv);
}
sub save_AllowOverride
{
local(@ov, $ov);
if ($in{'AllowOverride_def'}) { return ( [ ] ); }
foreach $ov (@AllowOverride_v) {
	if ($in{"AllowOverride_$ov"}) { push(@ov, $ov); }
	}
if (!@ov) { return ( [ "None" ] ); }
elsif (@ov == @AllowOverride_v) { return ( [ "All" ] ); }
else { return ( [ join(' ', @ov) ] ); }
}

sub edit_ContentDigest
{
return (1,
	$text{'core_105'},
	&choice_input($_[0]->{'value'}, "ContentDigest", "",
		      "$text{'core_87'},on", "$text{'core_88'},off", "$text{'core_89'},"));
}
sub save_ContentDigest
{
return &parse_choice("ContentDigest", "");
}

sub edit_DocumentRoot
{
return (2, $text{'core_106'},
	&opt_input($_[0]->{'words'}->[0], "DocumentRoot", $text{'core_111'}, 40).
	&file_chooser_button("DocumentRoot", 1));
}
sub save_DocumentRoot
{
$in{'DocumentRoot_def'} || -d $in{'DocumentRoot'} ||
	&error(&text('core_112', $in{'DocumentRoot'}));
return &parse_opt("DocumentRoot");
}

sub edit_Options
{
local(@po, @o, $o, %opts, $opts, $po, @pon, $i);
@po = ("ExecCGI", "FollowSymLinks", "Includes", "IncludesNOEXEC",
       "Indexes", "MultiViews", "SymLinksIfOwnerMatch");
@pon = ("$text{'core_90'}", "$text{'core_91'}",
	"$text{'core_92'}", "$text{'core_93'}",
	"$text{'core_94'}", "$text{'core_95'}",
	"$text{'core_96'}");
$opts = &choice_input($_[0] ? 0 : 1, "Options_def", 1, "$text{'core_97'},1",
		      "$text{'core_98'},0")."<br>\n";
@o = split(/\s+/, $_[0]->{'value'});
foreach $o (split(/\s+/, $_[0]->{'value'})) {
	if ($o =~ /^\+(.*)$/) { $opts{$1} = 2; }
	elsif ($o =~ /^\-(.*)$/) { $opts{$1} = 3; }
	else { $opts{$o} = 1; }
	}
if ($opts{'All'}) {
	local($all); $all = $opts{'All'};
	undef(%opts);
	foreach $o (grep {!/MultiViews/} @po) {
		$opts{$o} = $all;
		}
	}
$opts .= "<table border>\n";
$opts .= "<tr $tb> <td><b>$text{'core_195'}</b></td> <td><b>$text{'core_196'}</b></td>\n";
$opts .= "<td><b>$text{'core_197'}</b></td> </tr>\n";
for($i=0; $i<@po; $i++) {
	$po = $po[$i];
	$opts .= "<tr $cb> <td><b>$pon[$i]</b></td> <td>\n";
	$opts .= sprintf "<input type=radio name=$po value=1 %s> $text{'core_99'}\n",
			$opts{$po}==1 ? "checked" : "";
	$opts .= sprintf "<input type=radio name=$po value=0 %s> $text{'core_100'}\n",
			$opts{$po}==0 ? "checked" : "";
	$opts .= "</td> <td>\n";
	$opts .= sprintf "<input type=radio name=$po value=2 %s> $text{'core_101'}\n",
			$opts{$po}==2 ? "checked" : "";
	$opts .= sprintf "<input type=radio name=$po value=3 %s> $text{'core_102'}\n",
			$opts{$po}==3 ? "checked" : "";
	$opts .= "</td> </tr>\n";
	}
$opts .= "</table>\n";
return (2, $text{'core_107'}, $opts);
}
sub save_Options
{
local(@po, $po, @rv);
if ($in{'Options_def'}) { return ( [ ] ); }
@po = ("ExecCGI", "FollowSymLinks", "Includes", "IncludesNOEXEC",
       "Indexes", "MultiViews", "SymLinksIfOwnerMatch");
foreach $po (@po) {
	if ($in{$po} == 1) { push(@rv, $po); }
	elsif ($in{$po} == 2) { push(@rv, "+$po"); }
	elsif ($in{$po} == 3) { push(@rv, "-$po"); }
	}
return @rv ? ( [ join(' ', @rv) ] ) : ( [ "None" ] );
}

sub edit_ServerPath
{
return (2,
	$text{'core_117'},
	&opt_input($_[0]->{'value'}, "ServerPath", $text{'core_120'}, 40).
	&file_chooser_button("ServerPath", 1));
}
sub save_ServerPath
{
return &parse_opt("ServerPath", '^\/\S*$',
		  $text{'core_119'});
}

sub edit_ServerSignature
{
return (1, $text{'core_118'},
	&choice_input($_[0]->{'value'}, "ServerSignature", undef,
		      "$text{'core_113'},Email", "$text{'core_114'},On",
		      "$text{'core_115'},Off", "$text{'core_116'},"));
}
sub save_ServerSignature
{
return &parse_choice("ServerSignature", undef);
}

#########################################################################
# MIME directives directives
sub edit_DefaultType
{
return (1,
	$text{'core_121'},
	&opt_input($_[0]->{'value'}, "DefaultType", $text{'core_123'}, 20));
}
sub save_DefaultType
{
return &parse_opt("DefaultType", '^(\S+)\/(\S+)$',
		  $text{'core_122'});
}

#########################################################################
# Access control directives
sub edit_AuthName
{
my $val = $_[1]->{'version'} >= 1.3 ? $_[0]->{'words'}->[0]
				    : $_[0]->{'value'};
return (1, $text{'core_124'},
        &opt_input($val, "AuthName", $text{'core_126'}, 25));
}
sub save_AuthName
{
return $in{'AuthName_def'}	 ? ( [ ] ) :
       $_[0]->{'version'} >= 1.3 ? ( [ "\"$in{'AuthName'}\"" ] ) :
				   ( [ $in{'AuthName'} ] );
}

sub edit_AuthType
{
local($rv, $a);
$rv = "<select name=AuthType>\n";
foreach $a ("", "Basic", "Digest") {
	$rv .= sprintf "<option %s>$a\n",
	        lc($_[0]->{'value'}) eq lc($a) ? "selected" : "";
	}
$rv .= "</select>";
return (1, $text{'core_125'}, $rv);
}
sub save_AuthType
{
if ($in{'AuthType'}) { return ( [ $in{'AuthType'} ] ); }
else { return ( [ ] ); }
}

sub edit_require
{
local($rv, $mode, $list);
$_[0]->{'value'} =~ /^(\S+)\s*(.*)$/; $mode = $1; $list = $2;
$rv = sprintf
      "<input type=radio name=require_mode value=0 %s> $text{'default'}<br>\n",
      $mode ? "" : "checked";
$rv .= sprintf
      "<input type=radio name=require_mode value=1 %s> $text{'core_198'}:\n",
      $mode eq "user" ? "checked" : "";
$rv .= sprintf
      "<input name=require_user size=20 value=\"%s\"><br>\n",
      $mode eq "user" ? $list : "";
$rv .= sprintf
      "<input type=radio name=require_mode value=2 %s> $text{'core_199'}:\n",
      $mode eq "group" ? "checked" : "";
$rv .= sprintf
      "<input name=require_group size=20 value=\"%s\"><br>\n",
      $mode eq "group" ? $list : "";
$rv .= sprintf
      "<input type=radio name=require_mode value=3 %s> $text{'core_200'}:<br>\n",
      $mode eq "valid-user" ? "checked" : "";
return (1, $text{'core_130'}, $rv);
}
sub save_require
{
if ($in{'require_mode'} == 0) { return ( [ ] ); }
elsif ($in{'require_mode'} == 1) { return ( [ "user $in{'require_user'}" ] ); }
elsif ($in{'require_mode'} == 2) { return ( [ "group $in{'require_group'}" ] ); }
else { return ( [ "valid-user" ] ); }
}

sub edit_Satisfy
{
return (1, $text{'core_131'},
	&choice_input_vert($_[0]->{'value'}, "Satisfy", "", "$text{'core_127'},",
			   "$text{'core_128'},all","$text{'core_129'},any"));
}
sub save_Satisfy
{
return &parse_choice("Satisfy", "");
}

#########################################################################
# Misc. directives
sub edit_CoreDumpDirectory
{
return (1, $text{'core_137'},
	 &opt_input($_[0]->{'value'}, "CoreDumpDirectory", $text{'core_155'}, 20).
	 &file_chooser_button("CoreDumpDirectory", 1));
}
sub save_CoreDumpDirectory
{
return &parse_opt("CoreDumpDirectory", '^\S+$', $text{'core_147'});
}

sub edit_LockFile
{
return (1, $text{'core_138'},
        &opt_input($_[0]->{'value'}, "LockFile", $text{'core_156'}, 20).
        &file_chooser_button("LockFile", 0));
}
sub save_LockFile
{
return &parse_opt("LockFile", '^\S+', $text{'core_148'});
}

sub edit_LimitRequestBody
{
return (1, $text{'core_139'},
	&opt_input($_[0]->{'value'}, "LimitRequestBody", $text{'core_157'}, 8)."bytes");
}
sub save_LimitRequestBody
{
return &parse_opt("LimitRequestBody", '^\d+$', $text{'core_149'});
}

sub edit_LimitRequestFields
{
return (1, $text{'core_140'},
	&opt_input($_[0]->{'value'}, "LimitRequestFields", $text{'core_158'}, 6));
}
sub save_LimitRequestFields
{
return &parse_opt("LimitRequestFields", '^\d+$', $text{'core_150'});
}

sub edit_LimitRequestFieldsize
{
return (1, $text{'core_141'},
	&opt_input($_[0]->{'value'}, "LimitRequestFieldsize", $text{'core_159'}, 6));
}
sub save_LimitRequestFieldsize
{
return &parse_opt("LimitRequestFieldsize", '^\d+$', $text{'core_151'});
}

sub edit_LimitRequestLine
{
return (1, $text{'core_142'},
	&opt_input($_[0]->{'value'}, "LimitRequestLine", $text{'core_160'}, 6));
}
sub save_LimitRequestLine
{
return &parse_opt("LimitRequestLine", '^\d+$', $text{'core_152'});
}

sub edit_PidFile
{
return (1, $text{'core_143'},
        &opt_input($_[0]->{'value'}, "PidFile", $text{'core_161'}, 20).
        &file_chooser_button("PidFile", 0));
}
sub save_PidFile
{
return &parse_opt("PidFile", '^\S+$', $text{'core_153'});
}

sub edit_ScoreBoardFile
{
return (1, $text{'core_144'},
        &opt_input($_[0]->{'value'}, "ScoreBoardFile", $text{'core_162'}, 20).
        &file_chooser_button("ScoreBoardFile", 0));
}
sub save_ScoreBoardFile
{
return &parse_opt("ScoreBoardFile", '^\S+$', $text{'core_154'});
}

sub edit_ServerType
{
return (1, $text{'core_145'},
        &choice_input($_[0]->{'value'}, "ServerType", "standalone",
                      "$text{'core_132'},standalone", "$text{'core_133'},inetd"));
}
sub save_ServerType
{
return &parse_choice("ServerType", "standalone");
}

sub edit_ServerTokens
{
return (1, $text{'core_146'},
	&select_input($_[0]->{'value'}, "ServerTokens", "Full",
		      "$text{'core_134'},Full",
		      "$text{'core_135'},OS", "$text{'core_136'},Min"));
}
sub save_ServerTokens
{
return &parse_select("ServerTokens", "Full");
}

#########################################################################
# User/group directives
sub edit_Group
{
local($rv, @ginfo);
$rv = sprintf "<input type=radio name=Group value=0 %s> Default&nbsp;\n",
       $_[0] ? "" : "checked";
$rv .= sprintf "<input type=radio name=Group value=1 %s>$text{'core_163'}\n",
        $_[0] && $_[0]->{'value'} !~ /^#/ ? "checked" : "";
$rv .= sprintf "<input name=Group_name size=8 value=\"%s\"> %s&nbsp;\n",
	$_[0]->{'value'} !~ /^#/ ? $_[0]->{'value'} : "",
	&group_chooser_button("Group_name", 0);
$rv .= sprintf "<input type=radio name=Group value=2 %s>$text{'core_164'}\n",
        $_[0]->{'value'} =~ /^#/ ? "checked" : "";
$rv .= sprintf "<input name=Group_id size=6 value=\"%s\">\n",
	 $_[0]->{'value'} =~ /^#(.*)$/ ? $1 : "";
return (2, $text{'core_167'}, $rv);
}
sub save_Group
{
if ($in{'Group'} == 0) { return ( [ ] ); }
elsif ($in{'Group'} == 1) { return ( [ $in{'Group_name'} ] ); }
elsif ($in{'Group_id'} !~ /^\-?\d+$/) {
	&error(&text('core_169', $in{'Group_id'}));
	}
else { return ( [ "#$in{'Group_id'}" ] ); }
}

sub edit_User
{
local($rv, @uinfo);
$rv = sprintf "<input type=radio name=User value=0 %s> Default&nbsp;\n",
       $_[0] ? "" : "checked";
$rv .= sprintf "<input type=radio name=User value=1 %s>$text{'core_165'}\n",
        $_[0] && $_[0]->{'value'} !~ /^#/ ? "checked" : "";
$rv .= sprintf "<input name=User_name size=8 value=\"%s\"> %s&nbsp;\n",
	$_[0]->{'value'} !~ /^#/ ? $_[0]->{'value'} : "",
	&user_chooser_button("User_name", 0);
$rv .= sprintf "<input type=radio name=User value=2 %s>$text{'core_166'}\n",
        $_[0]->{'value'} =~ /^#/ ? "checked" : "";
$rv .= sprintf "<input name=User_id size=6 value=\"%s\">\n",
	 $_[0]->{'value'} =~ /^#(.*)$/ ? $1 : "";
return (2, $text{'core_168'}, $rv);
}
sub save_User
{
if ($in{'User'} == 0) { return ( [ ] ); }
elsif ($in{'User'} == 1) { return ( [ $in{'User_name'} ] ); }
elsif ($in{'User_id'} !~ /^\-?\d+$/) {
	&error(&text('core_170', $in{'User_id'}));
	}
else { return ( [ "#$in{'User_id'}" ] ); }
}

#########################################################################
# Error handling directives
sub edit_ErrorDocument
{
local($rv, $len, $i);
$rv = "<table border width=100%>\n";
$rv .= "<tr $tb> <td><b>$text{'core_173'}</b></td> <td><b>$text{'core_174'}</b></td> ".
       "<td><b>$text{'core_175'}</b></td> </tr>\n";
$len = @{$_[0]} + 1;
for($i=0; $i<$len; $i++) {
	$v = $_[0]->[$i]->{'value'};
	if ($v =~ /^(\d+)\s+((http|ftp|gopher):\S+)$/)
		{ $code = $1; $type = 0; $url = $2; }
	elsif ($v =~ /^(\d+)\s+(\/.*)$/) { $code = $1; $type = 0; $url = $2; }
	elsif ($v =~ /^(\d+)\s+"?(.*)$/) { $code = $1; $type = 1; $url = $2; }
	else { $code = ""; $type = 0; $url = ""; }
	$rv .= "<tr $cb>\n";
	$rv .= "<td><input name=ErrorDocument_code_$i size=3 value=$code></td>\n";
	$rv .= "<td><input type=radio name=ErrorDocument_type_$i value=0 ".
	       ($type==0 ? "checked" : "").">$text{'core_171'}\n";
	$rv .= "<input type=radio name=ErrorDocument_type_$i value=1 ".
	       ($type==1 ? "checked" : "").">$text{'core_172'}</td>\n";
	$rv .= "<td><input name=ErrorDocument_url_$i size=40 value=\"$url\"></td>\n";
	$rv .= "</tr>\n";
	}
$rv .= "</table>\n";
return (2, $text{'core_176'}, $rv);
}
sub save_ErrorDocument
{
local($i, $code, $url, @rv);
for($i=0; defined($in{"ErrorDocument_code_$i"}); $i++) {
	$code = $in{"ErrorDocument_code_$i"}; $url = $in{"ErrorDocument_url_$i"};
	if ($code !~ /\S/ && $url !~ /\S/) { next; }
	$code =~ /^\d\d\d$/ || &error(&text('core_177', $code));
	if ($in{"ErrorDocument_type_$i"} == 0) {
		$url =~ /^\S+$/ || &error(&text('core_178', $url));
		push(@rv, "$code $url");
		}
	elsif ($_[0]->{'version'} >= 1.2) { push(@rv, "$code \"$url"); }
	else { push(@rv, "$code $url"); }
	}
return ( \@rv );
}

#########################################################################
# Logging directives
sub edit_ErrorLog
{
if ($_[1]->{'version'} < 1.3) {
	return (1, $text{'core_183'},
		&opt_input($_[0]->{'value'}, "ErrorLog", $text{'core_186'}, 20).
		&file_chooser_button("ErrorLog", 0));
	}
else {
	local $v = $_[0]->{'words'}->[0];
	local $t = !$v ? 3 :
		   $v eq 'syslog' ? 2 :
		   $v =~ /^\|/ ? 1 : 0;
	$rv = sprintf "<input type=radio name=ErrorLog_type value=3 %s>\n",
		$t == 3 ? "checked" : "";
	$rv .= "$text{'core_179'}";
	$rv .= sprintf "<input type=radio name=ErrorLog_type value=2 %s>\n",
		$t == 2 ? "checked" : "";
	$rv .= "$text{'core_180'}<br>\n";
	$rv .= sprintf "<input type=radio name=ErrorLog_type value=0 %s>\n",
		$t == 0 ? "checked" : "";
	$rv .= sprintf "$text{'core_181'}<input name=ErrorLog_file size=25 value='%s'>\n",
		$t == 0 ? $v : "";
	$rv .= sprintf "<input type=radio name=ErrorLog_type value=1 %s>\n",
		$t == 1 ? "checked" : "";
	$rv .= sprintf "$text{'core_182'}<input name=ErrorLog_prog size=25 value='%s'><br>\n",
		$t == 1 ? substr($v, 1) : "";
	return (2, $text{'core_184'}, $rv);
	}
}
sub save_ErrorLog
{
if ($_[0]->{'version'} < 1.3) {
	$in{'ErrorLog_def'} || &allowed_auth_file($in{'ErrorLog'}) ||
		&error($text{'core_187'});
	return &parse_opt("ErrorLog", '^\S+$', $text{'core_185'});
	}
else {
	if ($in{'ErrorLog_type'} == 3) {
		return ( [ ] );
		}
	elsif ($in{'ErrorLog_type'} == 0) {
		$in{'ErrorLog_file'} =~ /\S/ ||
		    &error($text{'core_188'});
		&allowed_auth_file($in{'ErrorLog_file'}) ||
		    &error($text{'core_189'});
		return ( [ $in{'ErrorLog_file'} ] );
		}
	elsif ($in{'ErrorLog_type'} == 1) {
		$in{'ErrorLog_prog'} =~ /\S/ ||
		    &error($text{'core_190'});
		$access{'pipe'} ||
		    &error($text{'core_191'});
		return ( [ "\"|$in{'ErrorLog_prog'}\"" ] );
		}
	else {
		return ( [ "syslog" ] );
		}
	}
}

#########################################################################
# Module directives
sub edit_ClearModuleList_AddModule
{
local($mods, @allmods, $d, %mods, $m, $i, $rv);
($ver, $mods) = &httpd_info($config{'httpd_path'});
@allmods = grep { !/^core$/ } @$mods;
local $conf = &get_config();
foreach $d (&find_directive_struct("LoadModule", $conf)) {
	if ($d->{'words'}->[1] =~ /(mod_\S+)\.so/) {
		push(@allmods, $1);
		}
	elsif ($d->{'words'}->[1] =~ /lib([^\/\s]+)\.so/) {
		push(@allmods, "mod_$1");
		}
	}

if (@{$_[0]}) {
	# Only selected modules have been enabled
	foreach $d (@{$_[1]}) {
		local $modc = $d->{'value'};
		$modc =~ s/\.c$//;
		$mods{$modc} = "checked";
		}
	}
else { foreach $m (@allmods) { $mods{$m} = "checked"; } }
$rv = &choice_input(@{$_[0]} ? 1 : 0, "ClearModuleList", 1,
		    "$text{'core_192'},0", "$text{'core_193'},1")."<br>\n";
$rv .= "<table>\n";
foreach $m (@allmods) {
	if ($i%4 == 0) { $rv .= "<tr>\n"; }
	$rv .= "<td><input name=AddModule type=checkbox value=$m $mods{$m}> $m</td>\n";
	if ($i++%4 == 3) { $rv .= "</tr>\n"; }
	}
$rv .= "</table>\n";
return (2, $text{'core_194'}, $rv);
}
sub save_ClearModuleList_AddModule
{
if ($in{'ClearModuleList'}) {
	local @mods = split(/\0/, $in{'AddModule'});
	return ( [ "" ], [ map { $_.".c" } @mods ] );
	}
else { return ( [ ], [ ] ); }
}

1;

