# redhat-linux-lib.pl
# Networking functions for redhat linux

$net_scripts_dir = "/etc/sysconfig/network-scripts";
$network_config = "/etc/sysconfig/network";
$static_route_config = "/etc/sysconfig/static-routes";

# active_interfaces()
# Returns a list of currently ifconfig'd interfaces
sub active_interfaces
{
local(@rv, @lines, $l);
open(IFC, "ifconfig -a |");
while(<IFC>) {
	s/\r|\n//g;
	if (/^\S+/) { push(@lines, $_); }
	else { $lines[$#lines] .= $_; }
	}
close(IFC);
foreach $l (@lines) {
	local %ifc;
	$l =~ /^([^:\s]+)/; $ifc{'name'} = $1;
	$l =~ /^(\S+)/; $ifc{'fullname'} = $1;
	if ($l =~ /^(\S+):(\d+)/) { $ifc{'virtual'} = $2; }
	if ($l =~ /inet addr:(\S+)/) { $ifc{'address'} = $1; }
	elsif (!$_[0]) { next; }
	if ($l =~ /Mask:(\S+)/) { $ifc{'netmask'} = $1; }
	if ($l =~ /Bcast:(\S+)/) { $ifc{'broadcast'} = $1; }
	if ($l =~ /HWaddr (\S+)/) { $ifc{'ether'} = $1; }
	if ($l =~ /MTU:(\d+)/) { $ifc{'mtu'} = $1; }
	$ifc{'up'}++ if ($l =~ /\sUP\s/);
	$ifc{'edit'} = ($ifc{'name'} !~ /^ppp/);
	$ifc{'index'} = scalar(@rv);
	push(@rv, \%ifc);
	}
return @rv;
}

# activate_interface(&details)
# Create or modify an interface
sub activate_interface
{
local $a = $_[0];
local $cmd = "ifconfig $a->{'name'}";
if ($a->{'virtual'} ne "") { $cmd .= ":$a->{'virtual'}"; }
$cmd .= " $a->{'address'}";
if ($a->{'netmask'}) { $cmd .= " netmask $a->{'netmask'}"; }
if ($a->{'broadcast'}) { $cmd .= " broadcast $a->{'broadcast'}"; }
if ($a->{'mtu'}) { $cmd .= " mtu $a->{'mtu'}"; }
if ($a->{'up'}) { $cmd .= " up"; }
else { $cmd .= " down"; }
local $out = `$cmd 2>&1`;
if ($?) { &error($out); }
if ($a->{'ether'}) {
	$out = `ifconfig $a->{'name'} hw ether $a->{'ether'} 2>&1`;
	if ($?) { &error($out); }
	}
}

# deactivate_interface(&details)
# Shutdown some active interface
sub deactivate_interface
{
local $name = $_[0]->{'name'}.
	      ($_[0]->{'virtual'} ne "" ? ":$_[0]->{'virtual'}" : "");
if ($_[0]->{'virtual'} ne "") {
	# Shutdown virtual interface by setting address to 0
	local $out = `ifconfig $name 0 2>&1`;
	if ($?) { &error($out); }
	}
local ($still) = grep { $_->{'fullname'} eq $name } &active_interfaces();
if ($still) {
	# Old version of ifconfig or non-virtual interface.. down it
	local $out = `ifconfig $name down 2>&1`;
	&error("ifconfig $name down");
	if ($?) { &error($out); }
	}
}

# boot_interfaces()
# Returns a list of interfaces brought up at boot time
sub boot_interfaces
{
local(@rv, $f);
opendir(CONF, $net_scripts_dir);
while($f = readdir(CONF)) {
	next if ($f !~ /^ifcfg-\S+/);
	local (%conf, $b);
	&read_env_file("$net_scripts_dir/$f", \%conf);
	$b->{'fullname'} = $conf{'DEVICE'};
	if ($b->{'fullname'} =~ /(\S+):(\d+)/) {
		$b->{'name'} = $1;
		$b->{'virtual'} = $2;
		}
	else { $b->{'name'} = $b->{'fullname'}; }
	$b->{'up'} = ($conf{'ONBOOT'} eq 'yes');
	$b->{'address'} = $conf{'IPADDR'};
	$b->{'netmask'} = $conf{'NETMASK'};
	$b->{'broadcast'} = $conf{'BROADCAST'};
	$b->{'dhcp'} = ($conf{'BOOTPROTO'} eq 'dhcp');
	$b->{'bootp'} = ($conf{'BOOTPROTO'} eq 'bootp');
	$b->{'edit'} = ($b->{'name'} !~ /^ppp/);
	$b->{'index'} = scalar(@rv);
	push(@rv, $b);
	}
closedir(CONF);
return @rv;
}

# save_interface(&details)
# Create or update a boot-time interface
sub save_interface
{
local(%conf);
local $name = $_[0]->{'virtual'} ne "" ? $_[0]->{'name'}.":".$_[0]->{'virtual'}
				       : $_[0]->{'name'};
&read_env_file("$net_scripts_dir/ifcfg-$name", \%conf);
$conf{'DEVICE'} = $name;
$conf{'IPADDR'} = $_[0]->{'address'};
local($ip1, $ip2, $ip3, $ip4) = split(/\./, $_[0]->{'address'});
$conf{'NETMASK'} = $_[0]->{'netmask'};
local($nm1, $nm2, $nm3, $nm4) = split(/\./, $_[0]->{'netmask'});
if ($_[0]->{'address'} && $_[0]->{'netmask'}) {
	$conf{'NETWORK'} = sprintf "%d.%d.%d.%d",
				($ip1 & int($nm1))&0xff,
				($ip2 & int($nm2))&0xff,
				($ip3 & int($nm3))&0xff,
				($ip4 & int($nm4))&0xff;
	}
else {
	$conf{'NETWORK'} = '';
	}
$conf{'BROADCAST'} = $_[0]->{'broadcast'};
$conf{'ONBOOT'} = $_[0]->{'up'} ? "yes" : "no";
$conf{'BOOTPROTO'} = $_[0]->{'bootp'} ? "bootp" :
		     $_[0]->{'dhcp'} ? "dhcp" : "none";
&write_env_file("$net_scripts_dir/ifcfg-$name", \%conf);
}

# delete_interface(&details)
# Delete a boot-time interface
sub delete_interface
{
local $name = $_[0]->{'virtual'} ne "" ? $_[0]->{'name'}.":".$_[0]->{'virtual'}
				       : $_[0]->{'name'};
unlink("$net_scripts_dir/ifcfg-$name");
}

# iface_type(name)
# Returns a human-readable interface type name
sub iface_type
{
return "PPP" if ($_[0] =~ /^ppp/);
return "SLIP" if ($_[0] =~ /^sl/);
return "PLIP" if ($_[0] =~ /^plip/);
return "Ethernet" if ($_[0] =~ /^eth/);
return "Arcnet" if ($_[0] =~ /^arc/);
return "Token Ring" if ($_[0] =~ /^tr/);
return "Pocket/ATP" if ($_[0] =~ /^atp/);
return "Loopback" if ($_[0] =~ /^lo/);
return "Unknown";
}

# iface_hardware(name)
# Does some interface have an editable hardware address
sub iface_hardware
{
return $_[0] =~ /^eth/;
}

# can_edit(what)
# Can some boot-time interface parameter be edited?
sub can_edit
{
return $_[0] ne "mtu";
}

# valid_boot_address(address)
# Is some address valid for a bootup interface
sub valid_boot_address
{
return &check_ipaddress($_[0]);
}

# get_dns_config()
# Returns a hashtable containing keys nameserver, domain, search & order
sub get_dns_config
{
local $dns;
open(RESOLV, "/etc/resolv.conf");
while(<RESOLV>) {
	s/\r|\n//g;
	if (/nameserver\s+(.*)/) {
		push(@{$dns->{'nameserver'}}, split(/\s+/, $1));
		}
	elsif (/domain\s+(\S+)/) {
		$dns->{'domain'} = [ $1 ];
		}
	elsif (/search\s+(.*)/) {
		$dns->{'domain'} = [ split(/\s+/, $1) ];
		}
	}
close(RESOLV);
open(SWITCH, "/etc/nsswitch.conf");
while(<SWITCH>) {
	s/\r|\n//g;
	if (/^\s*hosts:\s+(.*)/) {
		$dns->{'order'} = $1;
		}
	}
close(SWITCH);
return $dns;
}

# save_dns_config(&config)
# Writes out the resolv.conf and nsswitch.conf files
sub save_dns_config
{
open(RESOLV, "/etc/resolv.conf");
local @resolv = <RESOLV>;
close(RESOLV);
open(RESOLV, ">/etc/resolv.conf");
foreach (@{$_[0]->{'nameserver'}}) {
	print RESOLV "nameserver $_\n";
	}
if ($_[0]->{'domain'}) {
	if ($_[0]->{'domain'}->[1]) {
		print RESOLV "search ",join(" ", @{$_[0]->{'domain'}}),"\n";
		}
	else {
		print RESOLV "domain $_[0]->{'domain'}->[0]\n";
		}
	}
foreach (@resolv) {
	print RESOLV $_ if (!/^\s*(nameserver|domain|search)\s+/);
	}
close(RESOLV);

open(SWITCH, "/etc/nsswitch.conf");
local @switch = <SWITCH>;
close(SWITCH);
open(SWITCH, ">/etc/nsswitch.conf");
foreach (@switch) {
	if (/^\s*hosts:\s+/) {
		print SWITCH "hosts:\t$_[0]->{'order'}\n";
		}
	else { print SWITCH $_; }
	}
close(SWITCH);
}

$max_dns_servers = 3;

# order_input(&dns)
# Returns HTML for selecting the name resolution order
sub order_input
{
if ($_[0]->{'order'} =~ /\[/) {
	# Using a complex resolve list
	return "<input name=order size=45 value=\"$_[0]->{'order'}\">\n";
	}
else {
	# Can select by menus
	local @o = split(/\s+/, $_[0]->{'order'});
	@o = map { s/nis\+/nisplus/; s/yp/nis/; $_; } @o;
	local ($rv, $i, $j);
	local @srcs = ( "", "files", "dns", "nis", "nisplus", "db" );
	local @srcn = ( "", "Hosts", "DNS", "NIS", "NIS+", "DB" );
	for($i=1; $i<@srcs; $i++) {
		local $ii = $i-1;
		$rv .= "<select name=order_$ii>\n";
		for($j=0; $j<@srcs; $j++) {
			$rv .= sprintf "<option value=\"%s\" %s>%s\n",
					$srcs[$j],
					$o[$ii] eq $srcs[$j] ? "selected" : "",
					$srcn[$j];
			}
		$rv .= "</select>\n";
		}
	return $rv;
	}
}

# parse_order(&dns)
# Parses the form created by order_input()
sub parse_order
{
if (defined($in{'order'})) {
	$in{'order'} =~ /\S/ || &error($text{'dns_eorder'});
	$_[0]->{'order'} = $in{'order'};
	}
else {
	local($i, @order);
	for($i=0; defined($in{"order_$i"}); $i++) {
		push(@order, $in{"order_$i"}) if ($in{"order_$i"});
		}
	$_[0]->{'order'} = join(" ", @order);
	}
}

# get_hostname()
sub get_hostname
{
return &get_system_hostname();
}

# save_hostname(name)
sub save_hostname
{
local %conf;
system("hostname $_[0] >/dev/null 2>&1");
open(HOST, ">/etc/HOSTNAME");
print HOST $_[0],"\n";
close(HOST);
&read_env_file($network_config, \%conf);
$conf{'HOSTNAME'} = $_[0];
&write_env_file($network_config, \%conf);
}

# get_domainname()
sub get_domainname
{
local $d = `domainname`;
chop($d);
return $d eq "(none)" ? "" : $d;
}

# save_domainname(domain)
sub save_domainname
{
local %conf;
system("domainname \"$_[0]\" >/dev/null 2>&1");
&read_env_file($network_config, \%conf);
if ($_[0]) {
	$conf{'NISDOMAIN'} = $_[0];
	}
else {
	delete($conf{'NISDOMAIN'});
	}
&write_env_file($network_config, \%conf);
}

sub routing_input
{
# show default router and device
local (%conf, @st, @hr);
&read_env_file($network_config, \%conf);
print "<tr> <td><b>$text{'routes_default'}</b></td> <td>\n";
printf "<input type=radio name=gateway_def value=1 %s> $text{'routes_none'}\n",
	$conf{'GATEWAY'} ? "" : "checked";
printf "<input type=radio name=gateway_def value=0 %s>\n",
	$conf{'GATEWAY'} ? "checked" : "";
printf "<input name=gateway size=15 value=\"%s\"></td> </tr>\n",
	$conf{'GATEWAY'};

print "<tr> <td><b>$text{'routes_device2'}</b></td> <td>\n";
printf "<input type=radio name=gatewaydev_def value=1 %s> $text{'routes_none'}\n",
	$conf{'GATEWAYDEV'} ? "" : "checked";
printf "<input type=radio name=gatewaydev_def value=0 %s>\n",
	$conf{'GATEWAYDEV'} ? "checked" : "";
printf "<input name=gatewaydev size=6 value=\"%s\"></td> </tr>\n",
	$conf{'GATEWAYDEV'};

# show routing
print "<tr> <td><b>$text{'routes_forward'}</b></td> <td>\n";
printf "<input type=radio name=forward value=1 %s> $text{'yes'}\n",
	$conf{'FORWARD_IPV4'} eq "yes" ? "checked" : "";
printf "<input type=radio name=forward value=0 %s> $text{'no'}</td> </tr>\n",
	$conf{'FORWARD_IPV4'} eq "yes" ? "" : "checked";

# get static routes
open(STATIC, $static_route_config);
while(<STATIC>) {
	if (/(\S+)\s+net\s+(\S+)\s+netmask\s+(\S+)\s+gw\s+(\S+)/) {
		push(@st, [ $1, $2, $3, $4 ]);
		}
	elsif (/(\S+)\s+host\s+(\S+)/) {
		push(@hr, [ $1, $2 ]);
		}
	}
close(STATIC);

# show static network routes
print "<tr> <td valign=top><b>$text{'routes_static'}</b></td>\n";
print "<td><table border>\n";
print "<tr $tb> <td><b>$text{'routes_ifc'}</b></td> ",
      "<td><b>$text{'routes_net'}</b></td> ",
      "<td><b>$text{'routes_mask'}</b></td> ",
      "<td><b>$text{'routes_gateway'}</b></td> </tr>\n";
for($i=0; $i<=@st; $i++) {
	local $st = $st[$i];
	print "<tr $cb>\n";
	print "<td><input name=dev_$i size=6 value='$st->[0]'></td>\n";
	print "<td><input name=net_$i size=15 value='$st->[1]'></td>\n";
	print "<td><input name=netmask_$i size=15 value='$st->[2]'></td>\n";
	print "<td><input name=gw_$i size=15 value='$st->[3]'></td>\n";
	print "</tr>\n";
	}
print "</table></td> </tr>\n";

# Show static host routes
print "<tr> <td valign=top><b>$text{'routes_local'}</b></td>\n";
print "<td><table border>\n";
print "<tr $tb> <td><b>$text{'routes_ifc'}</b></td> ",
      "<td><b>$text{'routes_net'}</b></td> </tr>\n";
for($i=0; $i<=@hr; $i++) {
	local $st = $hr[$i];
	print "<tr $cb>\n";
	print "<td><input name=ldev_$i size=6 value='$st->[0]'></td>\n";
	print "<td><input name=lnet_$i size=15 value='$st->[1]'></td>\n";
	print "</tr>\n";
	}
print "</table></td> </tr>\n";
}

sub parse_routing
{
local (%conf, @st);
&read_env_file($network_config, \%conf);
if ($in{'gateway_def'}) { $conf{'GATEWAY'} = ''; }
elsif (!gethostbyname($in{'gateway'})) {
	&error(&text('routes_edefault', $in{'gateway'}));
	}
else { $conf{'GATEWAY'} = $in{'gateway'}; }

if ($in{'gatewaydev_def'}) { $conf{'GATEWAYDEV'} = ''; }
elsif ($in{'gatewaydev'} !~ /^\S+$/) {
	&error(&text('routes_edevice', $in{'gatewaydev'}));
	}
else { $conf{'GATEWAYDEV'} = $in{'gatewaydev'}; }

if ($in{'forward'}) { $conf{'FORWARD_IPV4'} = 'yes'; }
else { $conf{'FORWARD_IPV4'} = 'no'; }

for($i=0; defined($dev = $in{"dev_$i"}); $i++) {
	$net = $in{"net_$i"}; $netmask = $in{"netmask_$i"}; $gw = $in{"gw_$i"};
	next if (!$dev && !$net && !$netmask && !$gw);
	$dev =~ /^\S+$/ || &error(&text('routes_edevice', $dev));
	gethostbyname($net) || &error(&text('routes_enet', $net));
	&check_ipaddress($netmask) || &error(&text('routes_emask', $netmask));
	gethostbyname($gw) || &error(&text('routes_egateway', $gw));
	push(@st, "$dev net $net netmask $netmask gw $gw\n");
	}
for($i=0; defined($dev = $in{"ldev_$i"}); $i++) {
	$net = $in{"lnet_$i"};
	next if (!$dev && !$net);
	$dev =~ /^\S+$/ || &error(&text('routes_edevice', $dev));
	gethostbyname($net) || &error(&text('routes_enet', $net));
	push(@st, "$dev host $net\n");
	}
open(STATIC, ">$static_route_config");
print STATIC @st;
close(STATIC);
&write_env_file($network_config, \%conf);
}

1;

