/*
    webalizer - a web server log analysis program

    Copyright (C) 1997-2001  Bradford L. Barrett (brad@mrunix.net)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

    This software uses the gd graphics library, which is copyright by
    Quest Protein Database Center, Cold Spring Harbor Labs.  Please
    see the documentation supplied with the library for additional
    information and license terms, or visit www.boutell.com/gd/ for the
    most recent version of the library and supporting documentation.
*/

/*********************************************/
/* STANDARD INCLUDES                         */
/*********************************************/

#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>                           /* normal stuff             */
#include <ctype.h>
#include <sys/utsname.h>
#include <sys/times.h>
#include <stddef.h>

/* ensure limits */
#if HAVE_LIMITS_H
# include <limits.h>
#endif
#ifndef UCHAR_MAX
# define UCHAR_MAX ((unsigned char) -1)
#endif

/* ensure getopt */
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

/* ensure sys/types */
#ifndef _SYS_TYPES_H
#include <sys/types.h>
#endif

/* some systems need this */
#ifdef HAVE_MATH_H
#include <math.h>
#endif

/* SunOS 4.x Fix */
#ifndef CLK_TCK
#define CLK_TCK _SC_CLK_TCK
#endif

/* GeoIP stuff */
#ifdef USE_GEOIP
#include <GeoIP.h>
extern GeoIP *gi;
#define GEOIP_OK(r) ((r!=NULL)&&(r[0]!='-')&&(r[1]!='-'))
#endif	/* USE_GEOIP */

#include "webalizer.h"                        /* main header              */
#include "lang.h"
#include "hashtab.h"
#include "preserve.h"
#include "linklist.h"
#include "graphs.h"
#include "output.h"

/* internal function prototypes */
void    write_html_head(char *, FILE *);            /* head of html page   */
void    write_html_tail(FILE *);                    /* tail of html page   */
void    month_links();                              /* Page links          */
void    month_total_table();                        /* monthly total table */
void    month_error_table();                        /* monthly 404 table   */
void    daily_total_table();                        /* daily total table   */
void    hourly_total_table();                       /* hourly total table  */
void    top_sites_table(int);                       /* top n sites table   */
void    top_urls_table(int);                        /* top n URL's table   */
void    top_entry_table(int);                       /* top n entry/exits   */
void    top_refs_table();                           /* top n referrers ""  */
void    top_agents_table();                         /* top n u-agents  ""  */
void    top_ctry_table();                           /* top n countries ""  */
void    top_search_table();                         /* top n search strs   */
void    top_users_table();                          /* top n ident table   */
u_long  load_url_array(  UNODEPTR *);               /* load URL array      */
u_long  load_site_array( HNODEPTR *);               /* load Site array     */
u_long  load_ref_array(  RNODEPTR *);               /* load Refs array     */
u_long  load_agent_array(ANODEPTR *);               /* load Agents array   */
u_long  load_srch_array( SNODEPTR *);               /* load srch str array */
u_long  load_ident_array(INODEPTR *);               /* load ident array    */
int	qs_url_cmph( const void*, const void*);     /* compare by hits     */
int	qs_url_cmpk( const void*, const void*);     /* compare by kbytes   */
int	qs_url_cmpn( const void*, const void*);     /* compare by entrys   */
int	qs_url_cmpx( const void*, const void*);     /* compare by exits    */
int	qs_site_cmph(const void*, const void*);     /* compare by hits     */
int	qs_site_cmpk(const void*, const void*);     /* compare by kbytes   */
int	qs_ref_cmph( const void*, const void*);     /* compare by hits     */
int     qs_agnt_cmph(const void*, const void*);     /* compare by hits     */
int     qs_srch_cmph(const void*, const void*);     /* compare by hits     */
int     qs_ident_cmph(const void*, const void*);    /* compare by hits     */
int     qs_ident_cmpk(const void*, const void*);    /* compare by kbytes   */

/* String replacement function prototypes */
struct TransliterationData {
  const char* TiedString[((size_t)UCHAR_MAX)+1]; };
static void ConstructTransliterationData(struct TransliterationData* data,
  const char* SourceCharacters, const char* const * ReplacementStrings);
static size_t SizeAfterTransliteration(const struct TransliterationData* data,
  const char* begin, const char* end);
static int RawTransliterate(const struct TransliterationData* data,
  const char* begin, const char* end, char* OutputString, char* OutputStringEnd);
extern char* Transliterate(const char* SourceCharacters,
  const char* const * ReplacementStrings, const char* String);

const char *hr_size(unsigned long long int size);   /* human readable size */

int     all_sites_page(u_long, u_long);             /* output site page    */
int     all_urls_page(u_long, u_long);              /* output urls page    */
int     all_refs_page(u_long, u_long);              /* output refs page    */
int     all_agents_page(u_long, u_long);            /* output agents page  */
int     all_search_page(u_long, u_long);            /* output search page  */
int     all_users_page(u_long, u_long);             /* output ident page   */
int     all_errors_page();                          /* output 404 page     */
void    dump_all_sites();                           /* dump sites tab file */
void    dump_all_urls();                            /* dump urls tab file  */
void    dump_all_refs();                            /* dump refs tab file  */
void    dump_all_agents();                          /* dump agents file    */
void    dump_all_users();                           /* dump usernames file */
void    dump_all_search();                          /* dump search file    */

/* define some colors for HTML */
#define WHITE          "#FFFFFF"
#define BLACK          "#000000"
#define RED            "#FF0000"
#define ORANGE         "#FF8000"
#define LTBLUE         "#0080FF"
#define BLUE           "#0000FF"
#define GREEN          "#00FF00"
#define DKGREEN        "#008040"
#define GREY           "#C0C0C0"
#define LTGREY         "#E8E8E8"
#define YELLOW         "#FFFF00"
#define PURPLE         "#FF00FF"
#define CYAN           "#00E0FF"

/* configurable html colors */
#define HITCOLOR       hit_color
#define FILECOLOR      file_color
#define SITECOLOR      site_color
#define KBYTECOLOR     kbyte_color
#define PAGECOLOR      page_color
#define VISITCOLOR     visit_color
#define BACKGNDCOLOR   backgnd_color
#define TEXTCOLOR      text_color
#define LINKCOLOR      link_color
#define VLINKCOLOR     vlink_color
#define ALINKCOLOR     alink_color
#define GRPCOLOR       grp_color
#define HEADLINECOLOR  headline_color
#define COUNTERCOLOR   counter_color

/* sort arrays */
UNODEPTR *u_array      = NULL;                /* Sort array for URL's     */
HNODEPTR *h_array      = NULL;                /* hostnames (sites)        */
RNODEPTR *r_array      = NULL;                /* referrers                */
ANODEPTR *a_array      = NULL;                /* user agents              */
SNODEPTR *s_array      = NULL;                /* search strings           */
INODEPTR *i_array      = NULL;                /* ident strings (username) */
u_long   a_ctr         = 0;                   /* counter for sort array   */

FILE     *out_fp;

/* human readable size warp buffer */
char warpbuf[32][32];
int wb_index = -1;

/*********************************************/
/* WRITE_HTML_HEAD - output top of HTML page */
/*********************************************/

void write_html_head(char *period, FILE *out_fp)
{
   NLISTPTR lptr;                          /* used for HTMLhead processing */

   /* HTMLPre code goes before all else    */
   lptr = html_pre;
   if (lptr==NULL)
   {
      /* Default 'DOCTYPE' header record if none specified */
      fprintf(out_fp,
      "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\">\n\n");
   }
   else
   {
      while (lptr!=NULL)
      {
         fprintf(out_fp,"%s\n",lptr->string);
         lptr=lptr->next;
      }
   }
   /* Standard header comments */
   fprintf(out_fp,"<!--   Generated by Webalizer Xtended %s    -->\n",
      version_xt);
   fprintf(out_fp,"<!--                                          -->\n");
   fprintf(out_fp,"<!--  %s  -->\n",copyright_xt);
   fprintf(out_fp,"<!--   http://www.patrickfrei.ch/webalizer/   -->\n");
   fprintf(out_fp,"<!--                                          -->\n");
   fprintf(out_fp,"<!--   Based on The Webalizer  Ver. %s-%s   -->\n",
      version,editlvl);
   fprintf(out_fp,"<!--                                          -->\n");
   fprintf(out_fp,"<!-- Copyright 1997-2000 Bradford L. Barrett  -->\n");
   fprintf(out_fp,"<!-- (brad@mrunix.net  http://www.mrunix.net) -->\n");
   fprintf(out_fp,"<!--                                          -->\n");
   fprintf(out_fp,"<!-- Distributed under the GNU GPL  Version 2 -->\n");
   fprintf(out_fp,"<!--        Full text may be found at:        -->\n");
   fprintf(out_fp,"<!--     http://www.mrunix.net/webalizer/     -->\n");
   fprintf(out_fp,"<!--                                          -->\n");
   fprintf(out_fp,"<!--  Give the power back to the programmers  -->\n");
   fprintf(out_fp,"<!--   Support the Free Software Foundation   -->\n");
   fprintf(out_fp,"<!--           (http://www.fsf.org)           -->\n");
   fprintf(out_fp,"<!--                                          -->\n");
   fprintf(out_fp,"<!-- *** Generated: %s *** -->\n\n",cur_time());

   fprintf(out_fp,"<HTML>\n<HEAD>\n");
   fprintf(out_fp," <TITLE>%s %s - %s</TITLE>\n",
                  msg_title, hname, period);
   lptr=html_head;
   while (lptr!=NULL)
   {
      fprintf(out_fp,"%s\n",lptr->string);
      lptr=lptr->next;
   }
   fprintf(out_fp,"</HEAD>\n\n");

   lptr = html_body;
   if (lptr==NULL)
      fprintf(out_fp,"<BODY BGCOLOR=\"%s\" TEXT=\"%s\" "   \
              "LINK=\"%s\" VLINK=\"%s\" ALINK=\"%s\">\n",
              BACKGNDCOLOR, TEXTCOLOR, LINKCOLOR, VLINKCOLOR, ALINKCOLOR);
   else
   {
      while (lptr!=NULL)
      {
         fprintf(out_fp,"%s\n",lptr->string);
         lptr=lptr->next;
      }
   }
   fprintf(out_fp,"<H2>%s %s</H2>\n",msg_title, hname);
   fprintf(out_fp,"<SMALL><STRONG>\n%s: %s<BR>\n",msg_hhdr_sp,period);
   fprintf(out_fp,"%s %s<BR>\n",msg_hhdr_gt,cur_time());
   fprintf(out_fp,"<A HREF=\"http://www.patrickfrei.ch/webalizer/\">"\
                   "Webalizer Xtended (%s)</A><BR>\n",version_xt);
#ifdef USE_GEOIP
   if (use_geoip)
      fprintf(out_fp,"with %s<BR>\n", gi_db_info);
#endif	/* USE_GEOIP */
   fprintf(out_fp, "</STRONG></SMALL>\n");
   lptr=html_post;
   while (lptr!=NULL)
   {
      fprintf(out_fp,"%s\n",lptr->string);
      lptr=lptr->next;
   }
   fprintf(out_fp,"<CENTER>\n<HR>\n<P>\n");
}

/*********************************************/
/* WRITE_HTML_TAIL - output HTML page tail   */
/*********************************************/

void write_html_tail(FILE *out_fp)
{
   NLISTPTR lptr;

   fprintf(out_fp,"</CENTER>\n");
   fprintf(out_fp,"<P>\n<HR>\n");
   fprintf(out_fp,"<TABLE WIDTH=\"100%%\" CELLPADDING=0 " \
                  "CELLSPACING=0 BORDER=0>\n");
   fprintf(out_fp,"<TR>\n");
   fprintf(out_fp,"<TD ALIGN=left VALIGN=top>\n");
   fprintf(out_fp,"<SMALL>Generated by "\
                  "<A HREF=\"http://www.patrickfrei.ch/webalizer/\">"\
		  "<STRONG>Webalizer Xtended (%s)</STRONG></A> by Patrick Frei "\
                  "based on \n",version_xt);
   fprintf(out_fp,"<A HREF=\"http://www.mrunix.net/webalizer/\">");
   fprintf(out_fp,"Webalizer Version %s</A>\n",version);
#ifdef USE_GEOIP
   if (use_geoip)
      fprintf(out_fp,"(with " \
                     "<A HREF=\"http://sysd.org/proj/log.php#glzr\">" \
                     "Geolizer</A> patch)\n");
#endif	/* USE_GEOIP */
   fprintf(out_fp,"</SMALL>\n</TD>\n");
   lptr=html_tail;
   if (lptr)
   {
      fprintf(out_fp,"<TD ALIGN=\"right\" VALIGN=\"top\">\n");
      while (lptr!=NULL)
      {
         fprintf(out_fp,"%s\n",lptr->string);
         lptr=lptr->next;
      }
      fprintf(out_fp,"</TD>\n");
   }
   fprintf(out_fp,"</TR>\n</TABLE>\n");

   /* wind up, this is the end of the file */
   fprintf(out_fp,"\n<!-- Webalizer Version %s-%s (Mod: %s) -->\n",
           version,editlvl,moddate);
   fprintf(out_fp,"\n<!--  Webalizer Xtended %s (Mod: %s)   -->\n",
           version_xt,moddate_xt);
   lptr = html_end;
   if (lptr)
   {
      while (lptr!=NULL)
      {
         fprintf(out_fp,"%s\n",lptr->string);
         lptr=lptr->next;
      }
   }
   else fprintf(out_fp,"\n</BODY>\n</HTML>\n");
}

/*********************************************/
/* WRITE_MONTH_HTML - does what it says...   */
/*********************************************/

int write_month_html()
{
   int i;
   char html_fname[256];           /* filename storage areas...       */
   char png1_fname[32];
   char png2_fname[32];

   char buffer[BUFSIZE];           /* scratch buffer                  */
   char dtitle[256];
   char htitle[256];

   if (verbose>1)
      printf("%s %s %d\n",msg_gen_rpt, l_month[cur_month-1], cur_year); 

   /* update history */
   i=cur_month-1;
   hist_month[i] =  cur_month;
   hist_year[i]  =  cur_year;
   hist_hit[i]   =  t_hit;
   hist_files[i] =  t_file;
   hist_page[i]  =  t_page;
   hist_visit[i] =  t_visit;
   hist_site[i]  =  t_site;
   hist_xfer[i]  =  t_xfer/1024;
   hist_ixfer[i] =  t_ixfer/1024;
   hist_oxfer[i] =  t_oxfer/1024;
   hist_fday[i]  =  f_day;
   hist_lday[i]  =  l_day;

   /* fill in filenames */
   snprintf(html_fname, sizeof(html_fname),"usage_%04d%02d.%s",
            cur_year,cur_month,html_ext);
   snprintf(png1_fname, sizeof(png1_fname),"daily_usage_%04d%02d.png",
            cur_year,cur_month);
   snprintf(png2_fname, sizeof(png2_fname),"hourly_usage_%04d%02d.png",
            cur_year,cur_month);

   /* create PNG images for web page */
   if (daily_graph)
   {
      snprintf(dtitle, sizeof(dtitle),"%s %s %d",msg_hmth_du,
               l_month[cur_month-1],cur_year);
      month_graph6 (  png1_fname,          /* filename          */
                      dtitle,              /* graph title       */
                      cur_month,           /* graph month       */
                      cur_year,            /* graph year        */
                      tm_hit,              /* data 1 (hits)     */
                      tm_file,             /* data 2 (files)    */
                      tm_site,             /* data 3 (sites)    */
                      tm_xfer,             /* data 4 (kbytes)   */
                      tm_ixfer,            /* data 5 (kbytes)   */
                      tm_oxfer,            /* data 6 (kbytes)   */
                      tm_page,             /* data 7 (pages)    */
                      tm_visit);           /* data 8 (visits)   */
   }

   if (hourly_graph)
   {
      snprintf(htitle, sizeof(htitle),"%s %s %d",msg_hmth_hu,
               l_month[cur_month-1],cur_year);
      day_graph3(    png2_fname,
                     htitle,
                     th_hit,
                     th_file,
                     th_page );
   }

   /* now do html stuff... */
   /* first, open the file */
   if ( (out_fp=open_out_file(html_fname))==NULL ) return 1;

   snprintf(buffer, sizeof(buffer),"%s %d",l_month[cur_month-1],cur_year);
   write_html_head(buffer, out_fp);
   month_links();
   month_total_table();
   if (ntop_notfound && response[21].count) /* 404 error stuff */
   {
      fprintf(out_fp,"<A NAME=\"ERRORSTATS\"></A>\n");
      month_error_table();
   }

   if (daily_graph || daily_stats)        /* Daily stuff */
   {
      fprintf(out_fp,"<A NAME=\"DAYSTATS\"></A>\n");
      if (daily_graph) fprintf(out_fp,"<IMG SRC=\"%s\" ALT=\"%s\" " \
                  "HEIGHT=500 WIDTH=512><P>\n",png1_fname,dtitle);
      if (daily_stats) daily_total_table();
   }

   if (hourly_graph || hourly_stats)      /* Hourly stuff */
   {
      fprintf(out_fp,"<A NAME=\"HOURSTATS\"></A>\n");
      if (hourly_graph) fprintf(out_fp,"<IMG SRC=\"%s\" ALT=\"%s\" "  \
                     "HEIGHT=256 WIDTH=512><P>\n",png2_fname,htitle);
      if (hourly_stats) hourly_total_table();
   }

   /* Do URL related stuff here, sorting appropriately                      */
   if ( (a_ctr=load_url_array(NULL)) )
   {
    if ( (u_array=malloc(sizeof(UNODEPTR)*(a_ctr))) !=NULL )
    {
     a_ctr=load_url_array(u_array);        /* load up our sort array        */
     if (ntop_urls || dump_urls)
     {
       qsort(u_array,a_ctr,sizeof(UNODEPTR),qs_url_cmph);
       if (ntop_urls) top_urls_table(0);   /* Top URL's (by hits)           */
       if (dump_urls) dump_all_urls();     /* Dump URLS tab file            */
     }
     if (ntop_urlsK)                       /* Top URL's (by kbytes)         */
      {qsort(u_array,a_ctr,sizeof(UNODEPTR),qs_url_cmpk); top_urls_table(1); }
     if (ntop_entry)                       /* Top Entry Pages               */
      {qsort(u_array,a_ctr,sizeof(UNODEPTR),qs_url_cmpn); top_entry_table(0);}
     if (ntop_exit)                        /* Top Exit Pages                */
      {qsort(u_array,a_ctr,sizeof(UNODEPTR),qs_url_cmpx); top_entry_table(1);}
     free(u_array);
    }
    else if (verbose) fprintf(stderr,"%s [u_array]\n",msg_nomem_tu); /* err */
   }

   /* do hostname (sites) related stuff here, sorting appropriately...      */
   if ( (a_ctr=load_site_array(NULL)) )
   {
    if ( (h_array=malloc(sizeof(HNODEPTR)*(a_ctr))) !=NULL )
    {
     a_ctr=load_site_array(h_array);       /* load up our sort array        */
     if (ntop_sites || dump_sites)
     {
       qsort(h_array,a_ctr,sizeof(HNODEPTR),qs_site_cmph);
       if (ntop_sites) top_sites_table(0); /* Top sites table (by hits)     */
       if (dump_sites) dump_all_sites();   /* Dump sites tab file           */
     }
     if (ntop_sitesK)                      /* Top Sites table (by kbytes)   */
     {
       qsort(h_array,a_ctr,sizeof(HNODEPTR),qs_site_cmpk);
       top_sites_table(1);
     }
     free(h_array);
    }
    else if (verbose) fprintf(stderr,"%s [h_array]\n",msg_nomem_ts); /* err */
   }

   /* do referrer related stuff here, sorting appropriately...              */
   if ( (a_ctr=load_ref_array(NULL)) )
   {
    if ( (r_array=malloc(sizeof(RNODEPTR)*(a_ctr))) != NULL)
    {
     a_ctr=load_ref_array(r_array);
     if (ntop_refs || dump_refs)
     {
       qsort(r_array,a_ctr,sizeof(RNODEPTR),qs_ref_cmph);
       if (ntop_refs) top_refs_table();   /* Top referrers table            */
       if (dump_refs) dump_all_refs();    /* Dump referrers tab file        */
     }
     free(r_array);
    }
    else if (verbose) fprintf(stderr,"%s [r_array]\n",msg_nomem_tr); /* err */
   }

   /* do search string related stuff, sorting appropriately...              */
   if ( (a_ctr=load_srch_array(NULL)) )
   {
    if ( (s_array=malloc(sizeof(SNODEPTR)*(a_ctr))) != NULL)
    {
     a_ctr=load_srch_array(s_array);
     if (ntop_search || dump_search)
     {
       qsort(s_array,a_ctr,sizeof(SNODEPTR),qs_srch_cmph);
       if (ntop_search) top_search_table(); /* top search strings table     */
       if (dump_search) dump_all_search();  /* dump search string tab file  */
     }
     free(s_array);
    }
    else if (verbose) fprintf(stderr,"%s [s_array]\n",msg_nomem_tsr);/* err */
   }

   /* do ident (username) related stuff here, sorting appropriately...      */
   if ( (a_ctr=load_ident_array(NULL)) )
   {
    if ( (i_array=malloc(sizeof(INODEPTR)*(a_ctr))) != NULL)
    {
     a_ctr=load_ident_array(i_array);
     if (ntop_users || dump_users)
     {
       qsort(i_array,a_ctr,sizeof(INODEPTR),qs_ident_cmph);
       if (ntop_users) top_users_table(); /* top usernames table            */
       if (dump_users) dump_all_users();  /* dump usernames tab file        */
     }
     free(i_array);
    }
    else if (verbose) fprintf(stderr,"%s [i_array]\n",msg_nomem_ti); /* err */
   }

   /* do user agent related stuff here, sorting appropriately...            */
   if ( (a_ctr=load_agent_array(NULL)) )
   {
    if ( (a_array=malloc(sizeof(ANODEPTR)*(a_ctr))) != NULL)
    {
     a_ctr=load_agent_array(a_array);
     if (ntop_agents || dump_agents)
     {
       qsort(a_array,a_ctr,sizeof(ANODEPTR),qs_agnt_cmph);
       if (ntop_agents) top_agents_table(); /* top user agents table        */
       if (dump_agents) dump_all_agents();  /* dump user agents tab file    */
     }
     free(a_array);
    }
    else if (verbose) fprintf(stderr,"%s [a_array]\n",msg_nomem_ta); /* err */
   }

   if (ntop_ctrys ) top_ctry_table();     /* top countries table            */

   month_links();
   write_html_tail(out_fp);               /* finish up the HTML document    */
   fclose(out_fp);                        /* close the file                 */
   return (0);                            /* done...                        */
}

/*********************************************/
/* MONTH_LINKS - links to other page parts   */
/*********************************************/

void month_links()
{
   char index_fname[256];
   
   fprintf(out_fp,"<SMALL>\n");
   sprintf(index_fname,"index.%s",html_ext);
   fprintf(out_fp, "<A HREF=\"%s\">[%s]</A>\n", index_fname, msg_hlnk_in);
   if (ntop_notfound && response[21].count)
      fprintf(out_fp,"<A HREF=\"#ERRORSTATS\">[%s]</A>\n",msg_hlnk_nf);
   if (daily_stats || daily_graph)
      fprintf(out_fp,"<A HREF=\"#DAYSTATS\">[%s]</A>\n",msg_hlnk_ds);
   if (hourly_stats || hourly_graph)
      fprintf(out_fp,"<A HREF=\"#HOURSTATS\">[%s]</A>\n",msg_hlnk_hs);
   if (ntop_urls || ntop_urlsK)
      fprintf(out_fp,"<A HREF=\"#TOPURLS\">[%s]</A>\n",msg_hlnk_u);
   if (ntop_entry)
      fprintf(out_fp,"<A HREF=\"#TOPENTRY\">[%s]</A>\n",msg_hlnk_en);
   if (ntop_exit)
      fprintf(out_fp,"<A HREF=\"#TOPEXIT\">[%s]</A>\n",msg_hlnk_ex);
   if (ntop_sites || ntop_sitesK)
      fprintf(out_fp,"<A HREF=\"#TOPSITES\">[%s]</A>\n",msg_hlnk_s);
   if (ntop_refs && t_ref)
      fprintf(out_fp,"<A HREF=\"#TOPREFS\">[%s]</A>\n",msg_hlnk_r);
   if (ntop_search && t_ref)
      fprintf(out_fp,"<A HREF=\"#TOPSEARCH\">[%s]</A>\n",msg_hlnk_sr);
   if (ntop_users && t_user)
      fprintf(out_fp,"<A HREF=\"#TOPUSERS\">[%s]</A>\n",msg_hlnk_i);
   if (ntop_agents && t_agent)
      fprintf(out_fp,"<A HREF=\"#TOPAGENTS\">[%s]</A>\n",msg_hlnk_a);
   if (ntop_ctrys)
      fprintf(out_fp,"<A HREF=\"#TOPCTRYS\">[%s]</A>\n",msg_hlnk_c);
   fprintf(out_fp,"</SMALL>\n<P>\n");
}

/*********************************************/
/* MONTH_TOTAL_TABLE - monthly totals table  */
/*********************************************/

void month_total_table()
{
   int i,j,z,days_in_month;
   u_long max_files=0,max_hits=0,max_visits=0,max_pages=0;
   u_long pie_data[10],pie_tmp_data[10];
   char   *pie_legend[10],*pie_tmp_legend[10],pie_title[48],pie_fname[48];
   double max_xfer=0.0,max_ixfer=0.0,max_oxfer=0.0;

   extern int response_graph;  /* include external flag */

   days_in_month=(l_day-f_day)+1;
   for (i=0;i<31;i++)
   {  /* Get max/day values */
      if (tm_hit[i]>max_hits)     max_hits  = tm_hit[i];
      if (tm_file[i]>max_files)   max_files = tm_file[i];
      if (tm_page[i]>max_pages)   max_pages = tm_page[i];
      if (tm_visit[i]>max_visits) max_visits= tm_visit[i];
      if (tm_xfer[i]>max_xfer)    max_xfer  = tm_xfer[i];
      if (tm_ixfer[i]>max_ixfer)  max_ixfer = tm_ixfer[i];
      if (tm_oxfer[i]>max_oxfer)  max_oxfer = tm_oxfer[i];
   }

   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
      table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH COLSPAN=3 ALIGN=center BGCOLOR=\"%s\">"           \
      "%s %s %d</TH></TR>\n",HEADLINECOLOR,msg_mtot_ms,l_month[cur_month-1],
      cur_year);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /* Total Hits */
   fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_th,t_hit);
   /* Total Files */
   fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_tf,t_file);
   /* Total Pages */
   fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s %s</FONT></TD>\n"  \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_h_total, msg_h_pages, t_page);
   /* Total Visits */
   fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s %s</FONT></TD>\n"  \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_h_total, msg_h_visits, t_visit);
   /* Total XFer */
   fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%s</B>"                \
      "</FONT></TD></TR>\n",msg_mtot_tx,hr_size(t_xfer));
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"  \
         "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%s</B>"             \
         "</FONT></TD></TR>\n",msg_mtot_ix,hr_size(t_ixfer));
      fprintf(out_fp,"<TR><TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"  \
         "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%s</B>"             \
         "</FONT></TD></TR>\n",msg_mtot_ox,hr_size(t_oxfer));
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /**********************************************/
   /* Unique Sites */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"                     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_us,t_site);
   /* Unique URL's */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"                     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_uu,t_url);
   /* Unique Referrers */
   if (t_ref != 0)
   fprintf(out_fp,"<TR>"                                                     \
      "<TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"                     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_ur,t_ref);
   /* Unique Usernames */
   if (t_user != 0)
   fprintf(out_fp,"<TR>"                                                     \
      "<TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"                     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_ui,t_user);
   /* Unique Agents */
   if (t_agent != 0)
   fprintf(out_fp,"<TR>"                                                     \
      "<TD WIDTH=380><FONT SIZE=\"-1\">%s</FONT></TD>\n"                     \
      "<TD ALIGN=right COLSPAN=2><FONT SIZE=\"-1\"><B>%lu</B>"               \
      "</FONT></TD></TR>\n",msg_mtot_ua,t_agent);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /**********************************************/
   /* Hourly/Daily avg/max totals */
   fprintf(out_fp,"<TR>"                                                     \
      "<TH WIDTH=380 BGCOLOR=\"%s\"><FONT SIZE=-1 COLOR=\"%s\">.</FONT></TH>\n"\
      "<TH WIDTH=65 BGCOLOR=\"%s\" ALIGN=right>"                             \
      "<FONT SIZE=-1>%s </FONT></TH>\n"                                      \
      "<TH WIDTH=65 BGCOLOR=\"%s\" ALIGN=right>"                             \
      "<FONT SIZE=-1>%s </FONT></TH></TR>\n",
      HEADLINECOLOR,HEADLINECOLOR,HEADLINECOLOR,msg_h_avg,HEADLINECOLOR,
      msg_h_max);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /* Max/Avg Hits per Hour */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                               \
      "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
      "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%lu</B>"                    \
      "</FONT></TD></TR>\n",msg_mtot_mhh, t_hit/(24*days_in_month),mh_hit);
   /* Max/Avg Hits per Day */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                               \
      "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
      "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%lu</B>"                    \
      "</FONT></TD></TR>\n",msg_mtot_mhd, t_hit/days_in_month, max_hits);
   /* Max/Avg Files per Day */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                               \
      "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
      "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%lu</B>"                    \
      "</FONT></TD></TR>\n",msg_mtot_mfd, t_file/days_in_month,max_files);
   /* Max/Avg Pages per Day */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                               \
      "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
      "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%lu</B>"                    \
      "</FONT></TD></TR>\n",msg_mtot_mpd, t_page/days_in_month,max_pages);
   /* Max/Avg Visits per Day */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                               \
      "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
      "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%lu</B>"                    \
      "</FONT></TD></TR>\n",msg_mtot_mvd, t_visit/days_in_month,max_visits);
   /* Max/Avg KBytes per Day */
   fprintf(out_fp,"<TR>"                                                     \
      "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                               \
      "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
      "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%s</B>"                     \
      "</FONT></TD></TR>\n",msg_mtot_mkd,
      hr_size(t_xfer/days_in_month),hr_size(max_xfer));
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TR>"                                                  \
         "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                            \
         "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"\
         "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%s</B>"                  \
         "</FONT></TD></TR>\n",msg_mtot_ikd,
         hr_size(t_ixfer/days_in_month),hr_size(max_ixfer));
      fprintf(out_fp,"<TR>"                                                  \
         "<TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"                            \
         "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"\
         "<TD WIDTH=65 ALIGN=right><FONT SIZE=-1><B>%s</B>"                  \
         "</FONT></TD></TR>\n",msg_mtot_okd,
         hr_size(t_oxfer/days_in_month),hr_size(max_oxfer));
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /**********************************************/
   /* response code totals */
   fprintf(out_fp,"<TR><TH COLSPAN=3 ALIGN=center BGCOLOR=\"%s\">\n"         \
           "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",HEADLINECOLOR,msg_mtot_rc);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   for (i=0;i<TOTAL_RC;i++)
   {
      if (response[i].count != 0) {
         fprintf(out_fp,"<TR><TD><FONT SIZE=\"-1\">%s</FONT></TD>\n"         \
            "<TD ALIGN=right WIDTH=65><FONT SIZE=\"-1\"><B>%lu</B></FONT>"   \
            "</TD>\n<TD ALIGN=right WIDTH=65><FONT SIZE=\"-2\">%3.02f%%"     \
            "</FONT></TD></TR>\n", response[i].desc, response[i].count,
            PCENT( response[i].count, t_hit ) );
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /**********************************************/

   fprintf(out_fp,"</TABLE>\n");
   fprintf(out_fp,"<P>\n");
   /* generate pie chart if needed */
   if (response_graph) {
      for (i=0;i<10;i++) pie_data[i]=0;              /* init data array      */
      z=0;                          /* z: max 10 responses to create the pie */
      for (i=0;i<TOTAL_RC;i++)                      /* process all responses */
      {
         if (response[i].count > 0 && z<10) {
            if (i==0) {       /* process special response[0] that has no "-" */
               pie_data[0]=response[0].count;
               pie_legend[0]=response[0].desc;
               z+=1;
            } else {                                     /* process the rest */
               pie_data[z]=response[i].count;
               size_t legend_size=strcspn(response[i].desc,"-");
               void * tmp_ptr = calloc(legend_size+1,sizeof(char));
               if (tmp_ptr) {
                  pie_legend[z] = (char *) tmp_ptr;
                  pie_tmp_legend[z] = (char *) tmp_ptr;
               } else {
                  fprintf(stderr, "ERROR: Can't allocate enough memory\n");
                  exit(0);
               }
               memcpy(pie_legend[z],response[i].desc,legend_size);
               pie_legend[z][legend_size]='\0';
               z+=1;
            }
         }
      }

      /* Sort the pie chart strings */
      for (i=0;i<(z-1);i++) {
        for (j=0;j<(z-1-i);j++) {
            if (pie_data[j] < pie_data[j+1]) {
               pie_tmp_data[0] = pie_data[j];
               pie_tmp_legend[0] = pie_legend[j];
               pie_data[j] = pie_data[j+1];
               pie_legend[j] = pie_legend[j+1];
               pie_data[j+1] = pie_tmp_data[0];
               pie_legend[j+1] = pie_tmp_legend[0];
            }
         }
      }

      sprintf(pie_title,"%s %s %d",msg_mtot_rc,l_month[cur_month-1],cur_year);
      sprintf(pie_fname,"responses_%04d%02d.png",cur_year,cur_month);
      
      pie_chart(pie_fname,pie_title,t_hit,pie_data,pie_legend);
      
      /* put the image tag in the page */
      fprintf(out_fp,"<IMG SRC=\"%s\" ALT=\"%s\" " \
            "HEIGHT=300 WIDTH=512><P>\n",pie_fname,pie_title);
      
      /* free temp memory */
      /* for (int i=0;i<z;i++) free(pie_legend[i]); */
      /* for (int i=0;i<z;i++) free(pie_tmp_legend[i]); */
      /* free(pie_legend); */
      /* free(pie_tmp_legend); */
   }
}

/*********************************************/
/* MONTH_ERROR_TABLE - monthly 404 table     */
/*********************************************/

void month_error_table()
{
   int i, j;
   u_long tmp_resp_counter;

   /* Alloc memory for the 404 errors with resp_counter size */
   void * tmp_ptr = calloc(resp_counter, sizeof(struct responsetmp_url));
   if (tmp_ptr) {
       respnotfoundtmp = (struct responsetmp_url *) tmp_ptr;
   } else {
       fprintf(stderr, "ERROR: Can't allocate enough memory\n");
       exit(0);
   }
   
   /* Sort the 404 error strings */
   for (i=0;i<(resp_counter-1);i++) {
      for (j=0;j<(resp_counter-1-i);j++) {
         if (respnotfound[j].count < respnotfound[j+1].count) {
            strncpy(respnotfoundtmp[0].respurl,respnotfound[j].respurl,
                  sizeof(respnotfoundtmp[0].respurl)-1);
            respnotfoundtmp[0].respurl[sizeof(respnotfoundtmp[0].respurl)-1]='\0';
            respnotfoundtmp[0].count = respnotfound[j].count;
            strncpy(respnotfound[j].respurl,respnotfound[j+1].respurl,
                  sizeof(respnotfound[j].respurl)-1);
            respnotfound[j].respurl[sizeof(respnotfound[j].respurl)-1]='\0';
            respnotfound[j].count = respnotfound[j+1].count;
            strncpy(respnotfound[j+1].respurl,respnotfoundtmp[0].respurl,
                  sizeof(respnotfound[j+1].respurl)-1);
            respnotfound[j+1].respurl[sizeof(respnotfound[j+1].respurl)-1]='\0';
            respnotfound[j+1].count = respnotfoundtmp[0].count;
         }
      }
   }
   
   /* Print a 404 error table */
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 " \
           "CELLPADDING=1>\n",table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   
   /* Change table header according to ntop_notfound & resp_counter */
   if (ntop_notfound==1) {
      fprintf(out_fp,"<TR><TH COLSPAN=5 BGCOLOR=\"%s\" ALIGN=center>" \
              "Code 404 %s %s %d (%lu %s)</TH></TR>\n",HEADLINECOLOR,msg_mtot_ms,
	      l_month[cur_month-1],cur_year,resp_counter,msg_hlnk_u);
   } else {
      if (resp_counter < ntop_notfound) {
         fprintf(out_fp,"<TR><TH COLSPAN=5 BGCOLOR=\"%s\" ALIGN=center>" \
	         "Code 404 %s %s %d (%s %lu %s %lu %s)</TH></TR>\n",HEADLINECOLOR,
	         msg_mtot_ms,l_month[cur_month-1],cur_year,msg_top_top,
	         resp_counter,msg_top_of,resp_counter,msg_hlnk_u);
      } else {
         fprintf(out_fp,"<TR><TH COLSPAN=5 BGCOLOR=\"%s\" ALIGN=center>" \
	         "Code 404 %s %s %d (%s %i %s %lu %s)</TH></TR>\n",HEADLINECOLOR,
	         msg_mtot_ms,l_month[cur_month-1],cur_year,msg_top_top,
	         ntop_notfound,msg_top_of,resp_counter,msg_hlnk_u);
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>" \
           "<FONT SIZE=\"-1\">#</FONT></TH>\n",COUNTERCOLOR);
   fprintf(out_fp,"<TH COLSPAN=2 BGCOLOR=\"%s\" ALIGN=center>" \
           "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>" \
           "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",PAGECOLOR,msg_h_url);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   /* Show all 404's if ntop_notfound=1 else show preferred number */
   if (resp_counter > ntop_notfound && ntop_notfound!=1) {
      tmp_resp_counter=ntop_notfound;
   } else {
      tmp_resp_counter=resp_counter;
   }
   
   for (i=0;i<tmp_resp_counter;i++) {
      fprintf(out_fp,"<TR><TD ALIGN=center><FONT SIZE=\"-1\"><B>%i</B>" \
              "</FONT></TD>\n",i+1);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B>" \
	      "</FONT></TD>\n",respnotfound[i].count+1);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT>" \
	      "</TD>\n",PCENT( (respnotfound[i].count+1), response[21].count) );
      /* check for a service prefix (ie: http://) */
      if (strstr(respnotfound[i].respurl,"://")!=NULL) {
         fprintf(out_fp,"<TD ALIGN=left><FONT SIZE=-1><A HREF=\"%s\">%s</A>" \
         "</FONT></TD></TR>\n",respnotfound[i].respurl,respnotfound[i].respurl);
      } else {
         if (log_type == LOG_FTP) { /* FTP log? */
            fprintf(out_fp,"<TD ALIGN=left><FONT SIZE=-1>%s</FONT></TD></TR>\n",
               respnotfound[i].respurl);
         } else { /* Web log  */
            if (use_https) {
               /* secure server mode, use https:// */
               fprintf(out_fp,"<TD ALIGN=left><FONT SIZE=-1><A HREF=\"https://%s%s\">%s</A>" \
               "</FONT></TD></TR>\n",hname,respnotfound[i].respurl,respnotfound[i].respurl);
            } else {
               /* otherwise use standard http:// */
               fprintf(out_fp,"<TD ALIGN=left><FONT SIZE=-1><A HREF=\"http://%s%s\">%s</A>" \
               "</FONT></TD></TR>\n",hname,respnotfound[i].respurl,respnotfound[i].respurl);
            }
         }
      }
   }

   /* additionally show all 404's on a single page? */
   if (all_errors) {
      if (all_errors_page()) {
         fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
         fprintf(out_fp,"<TD COLSPAN=6 ALIGN=\"center\">\n");
         fprintf(out_fp,"<FONT SIZE=\"-1\">");
         fprintf(out_fp,"<A HREF=\"./error_%04d%02d.%s\">",
                    cur_year,cur_month,html_ext);
         fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_urls);
      }
   }

   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"</TABLE>\n");
   fprintf(out_fp,"<P>\n");

   free(respnotfoundtmp); /* free temp memory */
}

/*********************************************/
/* ALL_ERRORS_PAGE - HTML page of all 404's  */
/*********************************************/

int all_errors_page()
{
   FILE     *out_fp;
   char     error_fname[256], buffer[256];
   int i;

   /* generate file name */
   sprintf(error_fname,"error_%04d%02d.%s",cur_year,cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(error_fname))==NULL ) return 0;

   sprintf(buffer,"%s %d - %s",l_month[cur_month-1],cur_year,msg_h_url);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");

   fprintf(out_fp," %12s      %s\n",msg_h_hits,msg_h_url);
   fprintf(out_fp,"----------------  --------------------\n\n");

   for (i=0;i<resp_counter;i++) {
      fprintf(out_fp,"%-8lu %6.02f%%  %s\n", respnotfound[i].count+1,
          PCENT( (respnotfound[i].count+1), response[21].count),
          respnotfound[i].respurl);
   }

   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* DAILY_TOTAL_TABLE - daily totals          */
/*********************************************/

void daily_total_table()
{
   int i,j;

   /* Daily stats */
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   /* Daily statistics for ... */
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" COLSPAN=\"%i\" ALIGN=center>"      \
           "%s %s %d</TH></TR>\n",HEADLINECOLOR,j=(!hide_iovol)?17:13,
           msg_dtot_ds,l_month[cur_month-1], cur_year);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TR><TH ALIGN=center BGCOLOR=\"%s\">"                  \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
                  COUNTERCOLOR, msg_h_day,
                  HITCOLOR,     msg_h_hits,
                  FILECOLOR,    msg_h_files,
                  PAGECOLOR,    msg_h_pages,
                  VISITCOLOR,   msg_h_visits,
                  SITECOLOR,    msg_h_sites,
                  KBYTECOLOR,   msg_h_xfer,
                  FILECOLOR,    msg_h_ixfer,
                  HITCOLOR,     msg_h_oxfer);
   }
   else
   {
      fprintf(out_fp,"<TR><TH ALIGN=center BGCOLOR=\"%s\">"                  \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"                       \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"               \
                  "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",                 \
                  COUNTERCOLOR, msg_h_day,
                  HITCOLOR,     msg_h_hits,
                  FILECOLOR,    msg_h_files,
                  PAGECOLOR,    msg_h_pages,
                  VISITCOLOR,   msg_h_visits,
                  SITECOLOR,    msg_h_sites,
                  KBYTECOLOR,   msg_h_xfer);
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   /* skip beginning blank days in a month */
   for (i=0;i<hist_lday[cur_month-1];i++) if (tm_hit[i]!=0) break;
   if (i==hist_lday[cur_month-1]) i=0;

   for (;i<hist_lday[cur_month-1];i++)
   {
      fprintf(out_fp,"<TR><TD ALIGN=center>"                                 \
              "<FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n", i+1);
      fprintf(out_fp,"<TD ALIGN=right>"                                      \
              "<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"                   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              tm_hit[i],PCENT(tm_hit[i],t_hit));
      fprintf(out_fp,"<TD ALIGN=right>"                                      \
              "<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"                   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              tm_file[i],PCENT(tm_file[i],t_file));
      fprintf(out_fp,"<TD ALIGN=right>"                                      \
              "<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"                   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              tm_page[i],PCENT(tm_page[i],t_page));
      fprintf(out_fp,"<TD ALIGN=right>"                                      \
              "<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"                   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              tm_visit[i],PCENT(tm_visit[i],t_visit));
      fprintf(out_fp,"<TD ALIGN=right>"                                      \
              "<FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"                   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              tm_site[i],PCENT(tm_site[i],t_site));
      fprintf(out_fp,"<TD ALIGN=right>"                                      \
              "<FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"                    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              hr_size(tm_xfer[i]),PCENT(tm_xfer[i],t_xfer));
      if (!hide_iovol)
      {
         fprintf(out_fp,"<TD ALIGN=right>"                                   \
              "<FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"                    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
              hr_size(tm_ixfer[i]),PCENT(tm_ixfer[i],t_ixfer));
         fprintf(out_fp,"<TD ALIGN=right>"                                   \
              "<FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"                    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD></TR>\n",
              hr_size(tm_oxfer[i]),PCENT(tm_oxfer[i],t_oxfer));
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"</TABLE>\n");
   fprintf(out_fp,"<P>\n");
}

/*********************************************/
/* HOURLY_TOTAL_TABLE - hourly table         */
/*********************************************/

void hourly_total_table()
{
   int i,j,days_in_month;
   u_long avg_file=0;
   double avg_xfer=0.0,avg_ixfer=0.0,avg_oxfer=0.0;

   days_in_month=(l_day-f_day)+1;

   /* Hourly stats */
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" COLSPAN=\"%i\" ALIGN=center>" \
           "%s %s %d</TH></TR>\n",HEADLINECOLOR,j=(!hide_iovol)?19:13,
           msg_htot_hs,l_month[cur_month-1], cur_year);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TR><TH ALIGN=center ROWSPAN=2 BGCOLOR=\"%s\">" \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
                  COUNTERCOLOR, msg_h_hour,
                  HITCOLOR,     msg_h_hits,
                  FILECOLOR,    msg_h_files,
                  PAGECOLOR,    msg_h_pages,
                  KBYTECOLOR,   msg_h_xfer,
                  FILECOLOR,    msg_h_ixfer,
                  HITCOLOR,     msg_h_oxfer);
   }
   else
   {
      fprintf(out_fp,"<TR><TH ALIGN=center ROWSPAN=2 BGCOLOR=\"%s\">" \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=3>"     \
                  "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",       \
                  COUNTERCOLOR, msg_h_hour,
                  HITCOLOR,     msg_h_hits,
                  FILECOLOR,    msg_h_files,
                  PAGECOLOR,    msg_h_pages,
                  KBYTECOLOR,   msg_h_xfer);
   }
   fprintf(out_fp,"<TR><TH ALIGN=center BGCOLOR=\"%s\">"           \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"     \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n",
                  HITCOLOR, msg_h_avg, HITCOLOR, msg_h_total);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"               \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"     \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n",
                  FILECOLOR, msg_h_avg, FILECOLOR, msg_h_total);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"               \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"     \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n",
                  PAGECOLOR, msg_h_avg, PAGECOLOR, msg_h_total);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"               \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"     \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n",
                  KBYTECOLOR, msg_h_avg, KBYTECOLOR, msg_h_total);
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"            \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"     \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n",
                  FILECOLOR, msg_h_avg, FILECOLOR, msg_h_total);
      fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"            \
                  "<FONT SIZE=\"-2\">%s</FONT></TH>\n"             \
                  "<TH ALIGN=center BGCOLOR=\"%s\" COLSPAN=2>"     \
                  "<FONT SIZE=\"-2\">%s</FONT></TH></TR>\n",
                  HITCOLOR, msg_h_avg, HITCOLOR, msg_h_total);
   }

   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   for (i=0;i<24;i++)
   {
      fprintf(out_fp,"<TR><TD ALIGN=center>"                          \
         "<FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n",i);
      fprintf(out_fp,
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
         th_hit[i]/days_in_month,th_hit[i],
         PCENT(th_hit[i],t_hit));
      fprintf(out_fp,
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
         th_file[i]/days_in_month,th_file[i],
         PCENT(th_file[i],t_file));
      fprintf(out_fp,
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
         th_page[i]/days_in_month,th_page[i],
         PCENT(th_page[i],t_page));
      fprintf(out_fp,
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
         hr_size(th_xfer[i]/days_in_month),hr_size(th_xfer[i]),
         PCENT(th_xfer[i],t_xfer));
      if (!hide_iovol)
      {
         fprintf(out_fp,
            "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n" \
            "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n" \
            "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n",
            hr_size(th_ixfer[i]/days_in_month),hr_size(th_ixfer[i]),
            PCENT(th_ixfer[i],t_ixfer));
         fprintf(out_fp,
            "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n" \
            "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n" \
            "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD></TR>\n",
            hr_size(th_oxfer[i]/days_in_month),hr_size(th_oxfer[i]),
            PCENT(th_oxfer[i],t_oxfer));
      }
      avg_file += th_file[i]/days_in_month;
      avg_xfer += (th_xfer[i]/days_in_month)/1024;
      avg_ixfer+= (th_ixfer[i]/days_in_month)/1024;
      avg_oxfer+= (th_oxfer[i]/days_in_month)/1024;
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* TOP_SITES_TABLE - generate top n table    */
/*********************************************/

void top_sites_table(int flag)
{
   u_long cnt=0, h_reg=0, h_grp=0, h_hid=0, tot_num;
   int i;
   HNODEPTR hptr, *pointer;
#ifdef USE_GEOIP
   int cols=use_geoip?15:14;
   int j, ctry_fnd;
   u_long idx;
   const char *result_name, *result_code;
   char *country = NULL;
   char code[3];
   code[2]='\0';
   const char* array[7]={"&auml;", "&Auml;", "&ouml;", "&Ouml;", "&uuml;", 
                         "&Uuml;", "&szlig;"}; /* new characters */
#else
   int cols=14;
#endif	/* USE_GEOIP */

   cnt=a_ctr; pointer=h_array;
   while(cnt--)
   {
      /* calculate totals */
      switch ((*pointer)->flag)
      {
         case OBJ_REG:   h_reg++;  break;
         case OBJ_GRP:   h_grp++;  break;
         case OBJ_HIDE:  h_hid++;  break;
      }
      pointer++;
   }

   if ( (tot_num=h_reg+h_grp)==0 ) return;              /* split if none    */
   i=(flag)?ntop_sitesK:ntop_sites;                     /* Hits or KBytes?? */
   if (tot_num > i) tot_num = i;                        /* get max to do... */

   if ((!flag) || (flag&&!ntop_sites))                  /* now do <A> tag   */
      fprintf(out_fp,"<A NAME=\"TOPSITES\"></A>\n");

   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if (flag) fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=%d>" \
           "%s %lu %s %lu %s %s %s</TH></TR>\n",
           HEADLINECOLOR, cols, msg_top_top,tot_num,msg_top_of,
           t_site,msg_top_s,msg_h_by,msg_h_xfer);
   else      fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=%d>" \
           "%s %lu %s %lu %s</TH></TR>\n",
           HEADLINECOLOR, cols, msg_top_top, tot_num, msg_top_of, t_site, 
	   msg_top_s);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                    \
          "<FONT SIZE=\"-1\">#</FONT></TH>\n",COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"              \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"              \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_files);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"              \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",KBYTECOLOR,msg_h_xfer);
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"           \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_ixfer);
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"           \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_oxfer);
   }
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>" 
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",VISITCOLOR,msg_h_visits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                        \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",PAGECOLOR,msg_h_hname);
#ifdef USE_GEOIP
   if (use_geoip)
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                     \
             "<FONT SIZE=\"-1\">%s</FONT></TH>\n",PAGECOLOR,msg_h_ctry);
#endif	/* USE_GEOIP */
   fprintf(out_fp,"</TR>\n");
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=h_array; i=0;
   while(tot_num)
   {
      hptr=*pointer++;
      if (hptr->flag != OBJ_HIDE)
      {
         /* shade grouping? */
         if (shade_groups && (hptr->flag==OBJ_GRP))
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">\n", GRPCOLOR);
         else fprintf(out_fp,"<TR>\n");

	 if (!hide_iovol)
         {
            fprintf(out_fp,
              "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
              i+1,hptr->count,
              (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
              (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
              (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hr_size(hptr->ixfer),
              (t_ixfer==0)?0:((float)hptr->ixfer/t_ixfer)*100.0,hr_size(hptr->oxfer),
              (t_oxfer==0)?0:((float)hptr->oxfer/t_oxfer)*100.0,hptr->visit,
              (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0);
	 }
	 else
	 {
            fprintf(out_fp,
              "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
              i+1,hptr->count,
              (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
              (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
              (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hptr->visit,
              (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0);
	 }

         if ((hptr->flag==OBJ_GRP)&&hlite_groups)
             fprintf(out_fp,"<STRONG>%s</STRONG></FONT></TD></TR>\n",
               hptr->string);
#ifdef USE_GEOIP
         else
         {
            fprintf(out_fp,"%s</FONT></TD>", hptr->string);
            if (use_geoip)
            {
               result_name=GeoIP_country_name_by_name(gi, hptr->string);
	       result_code=GeoIP_country_code_by_name(gi, hptr->string);
	       if (!GEOIP_OK(result_code))
	       {
                  country=NULL;
	       }
	       else
	       {
                  code[0]=tolower(result_code[0]);
                  code[1]=tolower(result_code[1]);
                  country=code;
	       }
	       ctry_fnd=0;
	       if (country!=NULL)
	       {
                  idx=ctry_idx(country);
                  for (j=0;ctry[j].desc;j++)
                  {
                     if (idx==ctry[j].idx)
                     {
                        /* Country found. Display it in the currently compiled-in language */
                        ctry_fnd=1;
			fprintf(out_fp,"\n<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">%s"\
                           "</FONT></TD></TR>\n",Transliterate("",array,ctry[j].desc));
                        break;
                     }
                  }
	       }
               if ((!ctry_fnd)||(country==NULL))
               {
                  /* If country is not found in compiled-in language or NULL try to use english as standard language */
                  fprintf(out_fp,"\n<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">%s"\
                     "</FONT></TD></TR>\n",Transliterate("",array,GEOIP_OK(result_name)?result_name:ctry[0].desc));
               }
            }
            else
               fprintf(out_fp,"</TR>\n");
         }
#else
         else fprintf(out_fp,"%s</FONT></TD></TR>\n",
               hptr->string);
#endif	/* USE_GEOIP */
         tot_num--;
         i++;
      }
   }

   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if ((!flag) || (flag&&!ntop_sites))
   {
      if ( (all_sites) && ((h_reg+h_grp)>ntop_sites) )
      {
         if (all_sites_page(h_reg, h_grp))
         {
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
            fprintf(out_fp,"<TD COLSPAN=%d ALIGN=\"center\">\n", cols);
            fprintf(out_fp,"<FONT SIZE=\"-1\">");
            fprintf(out_fp,"<A HREF=\"./site_%04d%02d.%s\">",
                    cur_year,cur_month,html_ext);
            fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_sites);
            if (flag)   /* do we need to sort? */
               qsort(h_array,a_ctr,sizeof(HNODEPTR),qs_site_cmph);
         }
      }
   }
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* ALL_SITES_PAGE - HTML page of all sites   */
/*********************************************/

int all_sites_page(u_long h_reg, u_long h_grp)
{
   HNODEPTR hptr, *pointer;
   char     site_fname[256], buffer[256];
   FILE     *out_fp;
   int      i=(h_grp)?1:0;
#ifdef USE_GEOIP
   int j, ctry_fnd;
   u_long idx;
   const char *result_name, *result_code;
   char *country = NULL;
   char code[3];
   code[2]='\0';
   const char* array[7]={"&auml;", "&Auml;", "&ouml;", "&Ouml;", "&uuml;",
                         "&Uuml;", "&szlig;"}; /* new characters */
#endif	/* USE_GEOIP */

   /* generate file name */
   snprintf(site_fname, sizeof(site_fname),"site_%04d%02d.%s",cur_year,
            cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(site_fname))==NULL ) return 0;

   snprintf(buffer, sizeof(buffer),"%s %d - %s",l_month[cur_month-1],
            cur_year,msg_h_sites);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");

#ifdef USE_GEOIP
   if (!hide_iovol)
   {
      if (use_geoip)
      {
         fprintf(out_fp,"%12s      %12s       %12s         %12s         %12s" \
      		     "       %12s                            %s               "\
		     "                         %s\n",
              msg_h_hits, msg_h_files, msg_h_xfer, msg_h_ixfer, msg_h_oxfer, 
	      msg_h_visits, msg_h_hname, msg_h_ctry);
         fprintf(out_fp,"----------------  ----------------  -------------------  "\
                     "-------------------  -------------------  "	\
		     "----------------  --------------------"	\
		     "----------------------------------------  " \
		     "-------------------------\n\n");
      }
      else
      {
         fprintf(out_fp,"%12s      %12s       %12s         %12s         %12s" \
                     "       %12s         %s\n",
              msg_h_hits, msg_h_files, msg_h_xfer, msg_h_ixfer, msg_h_oxfer, 
              msg_h_visits, msg_h_hname);
         fprintf(out_fp,"----------------  ----------------  -------------------  "\
                     "-------------------  -------------------  "  \
                     "----------------  ---------------\n\n");
      }
   }
   else
   {
      if (use_geoip)
      {
         fprintf(out_fp,"%12s      %12s       %12s" \
      		     "       %12s                            %s               "\
		     "                         %s\n",
              msg_h_hits, msg_h_files, msg_h_xfer, 
	      msg_h_visits, msg_h_hname, msg_h_ctry);
         fprintf(out_fp,"----------------  ----------------  -------------------  "\
		     "----------------  --------------------"	\
		     "----------------------------------------  " \
		     "-------------------------\n\n");
      }
      else
      {
         fprintf(out_fp,"%12s      %12s       %12s" \
                     "       %12s         %s\n",
              msg_h_hits, msg_h_files, msg_h_xfer, msg_h_ixfer, msg_h_oxfer, 
              msg_h_visits, msg_h_hname);
         fprintf(out_fp,"----------------  ----------------  -------------------  "\
                     "----------------  ---------------\n\n");
      }
   }
#else
   if (!hide_iovol)
   {
      fprintf(out_fp,"%12s      %12s       %12s         %12s         %12s" \
                     "       %12s         %s\n",
              msg_h_hits, msg_h_files, msg_h_xfer, msg_h_ixfer, msg_h_oxfer, 
              msg_h_visits, msg_h_hname);
      fprintf(out_fp,"----------------  ----------------  -------------------  "\
                     "-------------------  -------------------  "  \
                     "----------------  " \
		     "---------------\n\n");
   }
   else
   {
      fprintf(out_fp,"%12s      %12s       %12s" \
                     "       %12s         %s\n",
              msg_h_hits, msg_h_files, msg_h_xfer, 
              msg_h_visits, msg_h_hname);
      fprintf(out_fp,"----------------  ----------------  -------------------  "\
                     "----------------  " \
		     "---------------\n\n");
   }
#endif	/* USE_GEOIP */

   /* Do groups first (if any) */
   pointer=h_array;
   while(h_grp)
   {
      hptr=*pointer++;
      if (hptr->flag == OBJ_GRP)
      {
	 if (!hide_iovol)
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  "  \
            "%16s %6.02f%%  %8lu %6.02f%%  %s\n",
            hptr->count,
            (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
            (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
            (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hr_size(hptr->ixfer),
            (t_ixfer==0)?0:((float)hptr->ixfer/t_ixfer)*100.0,hr_size(hptr->oxfer),
            (t_oxfer==0)?0:((float)hptr->oxfer/t_oxfer)*100.0,hptr->visit,
            (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
            hptr->string);
	 }
	 else
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  "  \
            "%8lu %6.02f%%  %s\n",
            hptr->count,
            (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
            (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
            (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hptr->visit,
            (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
            hptr->string);
	 }
         h_grp--;
      }
   }

   if (i) fprintf(out_fp,"\n");

   /* Now do individual sites (if any) */
   pointer=h_array;
   if (!hide_sites) while(h_reg)
   {
      hptr=*pointer++;
      if (hptr->flag == OBJ_REG)
      {
#ifdef USE_GEOIP
         if (use_geoip)
         {
            result_name=GeoIP_country_name_by_name(gi, hptr->string);
	    result_code=GeoIP_country_code_by_name(gi, hptr->string);
	    if (!GEOIP_OK(result_code))
	    {
               country=NULL;
	    }
	    else
	    {
               code[0]=tolower(result_code[0]);
               code[1]=tolower(result_code[1]);
               country=code;
	    }
	    ctry_fnd=0;
	    if (country!=NULL)
	    {
               idx=ctry_idx(country);
               for (j=0;ctry[j].desc;j++)
               {
                  if (idx==ctry[j].idx)
                  {
                     /* Country found. Display it in the currently compiled-in language */
                     ctry_fnd=1;
		     if (!hide_iovol)
		     {
                        fprintf(out_fp,
                        "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  "\
		        "%16s %6.02f%%  %8lu %6.02f%%  %-60s  %s\n",
                        hptr->count,
                        (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
                        (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
                        (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hr_size(hptr->ixfer),
		        (t_ixfer==0)?0:((float)hptr->ixfer/t_ixfer)*100.0,hr_size(hptr->oxfer),
		        (t_oxfer==0)?0:((float)hptr->oxfer/t_oxfer)*100.0,hptr->visit,
                        (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
                        hptr->string,Transliterate("",array,ctry[j].desc));
		     }
		     else
		     {
                        fprintf(out_fp,
                        "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  "\
		        "%8lu %6.02f%%  %-60s  %s\n",
                        hptr->count,
                        (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
                        (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
                        (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hptr->visit,
                        (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
                        hptr->string,Transliterate("",array,ctry[j].desc));
		     }
	             break;
                  }
               }
	    }
	    if ((!ctry_fnd)||(country==NULL))
            {
	       /* If country is not found in compiled-in language or NULL try to use english as standard language */
	       if (!hide_iovol)
	       {
	          fprintf(out_fp,
	          "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  " \
	          "%16s %6.02f%%  %8lu %6.02f%%  %-60s  %s\n",
	          hptr->count,
	          (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
	          (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
	          (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hr_size(hptr->ixfer),
	          (t_ixfer==0)?0:((float)hptr->ixfer/t_ixfer)*100.0,hr_size(hptr->oxfer),
	          (t_oxfer==0)?0:((float)hptr->oxfer/t_oxfer)*100.0,hptr->visit,
	          (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
	          hptr->string,Transliterate("",array,GEOIP_OK(result_name)?result_name:ctry[0].desc));
	       }
	       else
	       {
	          fprintf(out_fp,
	          "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  " \
	          "%8lu %6.02f%%  %-60s  %s\n",
	          hptr->count,
	          (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
	          (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
	          (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hptr->visit,
	          (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
	          hptr->string,Transliterate("",array,GEOIP_OK(result_name)?result_name:ctry[0].desc));
	       }
            }
         }
         else
         {
	    if (!hide_iovol)
	    {
               fprintf(out_fp,
               "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  " \
	       "%16s %6.02f%%  %8lu %6.02f%%  %s\n",
               hptr->count,
               (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
               (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
               (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hr_size(hptr->ixfer),
	       (t_ixfer==0)?0:((float)hptr->ixfer/t_ixfer)*100.0,hr_size(hptr->oxfer),
	       (t_oxfer==0)?0:((float)hptr->oxfer/t_oxfer)*100.0,hptr->visit,
               (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
               hptr->string);
	    }
	    else
	    {
               fprintf(out_fp,
               "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  " \
	       "%8lu %6.02f%%  %s\n",
               hptr->count,
               (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
               (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
               (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hptr->visit,
               (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
               hptr->string);
	    }
         }
#else
         if (!hide_iovol)
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  " \
	    "%16s %6.02f%%  %8lu %6.02f%%  %s\n",
            hptr->count,
            (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
            (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
            (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hr_size(hptr->ixfer),
	    (t_ixfer==0)?0:((float)hptr->ixfer/t_ixfer)*100.0,hr_size(hptr->oxfer),
	    (t_oxfer==0)?0:((float)hptr->oxfer/t_oxfer)*100.0,hptr->visit,
            (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
            hptr->string);
	 }
	 else
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  " \
	    "%8lu %6.02f%%  %s\n",
            hptr->count,
            (t_hit==0)?0:((float)hptr->count/t_hit)*100.0,hptr->files,
            (t_file==0)?0:((float)hptr->files/t_file)*100.0,hr_size(hptr->xfer),
            (t_xfer==0)?0:((float)hptr->xfer/t_xfer)*100.0,hptr->visit,
            (t_visit==0)?0:((float)hptr->visit/t_visit)*100.0,
            hptr->string);
	 }
#endif	/* USE_GEOIP */
         h_reg--;
      }
   }

   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* TOP_URLS_TABLE - generate top n table     */
/*********************************************/

void top_urls_table(int flag)
{
   u_long cnt=0,u_reg=0,u_grp=0,u_hid=0, tot_num;
   int i;
   UNODEPTR uptr, *pointer;

   cnt=a_ctr; pointer=u_array;
   while (cnt--)
   {
      /* calculate totals */
      switch ((*pointer)->flag)
      {
         case OBJ_REG:  u_reg++;  break;
         case OBJ_GRP:  u_grp++;  break;
         case OBJ_HIDE: u_hid++;  break;
      }
      pointer++;
   }

   if ( (tot_num=u_reg+u_grp)==0 ) return;              /* split if none    */
   i=(flag)?ntop_urlsK:ntop_urls;                       /* Hits or KBytes?? */
   if (tot_num > i) tot_num = i;                        /* get max to do... */
   if ((!flag) || (flag&&!ntop_urls))                   /* now do <A> tag   */
      fprintf(out_fp,"<A NAME=\"TOPURLS\"></A>\n");

   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if (flag) fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=10>"\
           "%s %lu %s %lu %s %s %s</TH></TR>\n",
           HEADLINECOLOR,msg_top_top,tot_num,msg_top_of,
           t_url,msg_top_u,msg_h_by,msg_h_xfer);
   else fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=10>"   \
           "%s %lu %s %lu %s</TH></TR>\n",
           HEADLINECOLOR,msg_top_top,tot_num,msg_top_of,t_url,msg_top_u);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                  \
                  "<FONT SIZE=\"-1\">#</FONT></TH>\n",COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"            \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
                  HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"            \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
                  KBYTECOLOR,msg_h_xfer);
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"         \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
                  FILECOLOR,msg_h_ixfer);
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"         \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
                  HITCOLOR,msg_h_oxfer);
   }
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                      \
                  "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
                  PAGECOLOR,msg_h_url);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=u_array; i=0;
   while (tot_num)
   {
      uptr=*pointer++;             /* point to the URL node */
      if (uptr->flag != OBJ_HIDE)
      {
         /* shade grouping? */
         if (shade_groups && (uptr->flag==OBJ_GRP))
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">\n", GRPCOLOR);
         else fprintf(out_fp,"<TR>\n");

         if (!hide_iovol)
         {
            fprintf(out_fp,
               "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n" \
               "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
               "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
               "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"  \
               "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
               "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"  \
               "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
               "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"  \
               "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
               "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
               i+1,uptr->count,
               (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
               hr_size(uptr->xfer),
               (t_xfer==0)?0:((float)uptr->xfer/t_xfer)*100.0,
               hr_size(uptr->ixfer),
               (t_ixfer==0)?0:((float)uptr->ixfer/t_ixfer)*100.0,
               hr_size(uptr->oxfer),
               (t_oxfer==0)?0:((float)uptr->oxfer/t_oxfer)*100.0);
	 }
	 else
	 {
            fprintf(out_fp,
               "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n" \
               "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
               "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
               "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"  \
               "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
               "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
               i+1,uptr->count,
               (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
               hr_size(uptr->xfer),
               (t_xfer==0)?0:((float)uptr->xfer/t_xfer)*100.0);
	 }

         if (uptr->flag==OBJ_GRP)
         {
            if (hlite_groups)
               fprintf(out_fp,"<STRONG>%s</STRONG></FONT></TD></TR>\n",
                uptr->string);
            else fprintf(out_fp,"%s</FONT></TD></TR>\n",uptr->string);
         }
         else 
	 {
            /* check for a service prefix (ie: http://) */
            if (strstr(uptr->string,"://")!=NULL)
               fprintf(out_fp,"<A HREF=\"%s\">%s</A></FONT></TD></TR>\n",
                 uptr->string,uptr->string);
	    else
            {
               if (log_type == LOG_FTP) /* FTP log? */
                   fprintf(out_fp,"%s</FONT></TD></TR>\n",uptr->string);
               else
               {             /* Web log  */
                  if (use_https)
                     /* secure server mode, use https:// */
                     fprintf(out_fp,
                     "<A HREF=\"https://%s%s\">%s</A></FONT></TD></TR>\n",
                      hname,uptr->string,uptr->string);
                   else
                      /* otherwise use standard 'http://' */
                      fprintf(out_fp,
                      "<A HREF=\"http://%s%s\">%s</A></FONT></TD></TR>\n",
                      hname,uptr->string,uptr->string);
               }
            }
	 }
         tot_num--;
         i++;
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if ((!flag) || (flag&&!ntop_urls))
   {
      if ( (all_urls) && ((u_reg+u_grp)>ntop_urls) )
      {
         if (all_urls_page(u_reg, u_grp))
         {
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
            fprintf(out_fp,"<TD COLSPAN=10 ALIGN=\"center\">\n");
            fprintf(out_fp,"<FONT SIZE=\"-1\">");
            fprintf(out_fp,"<A HREF=\"./url_%04d%02d.%s\">",
                    cur_year,cur_month,html_ext);
            fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_urls);
            if (flag)   /* do we need to sort first? */
               qsort(u_array,a_ctr,sizeof(UNODEPTR),qs_url_cmph);
         }
      }
   }
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* ALL_URLS_PAGE - HTML page of all urls     */
/*********************************************/

int all_urls_page(u_long u_reg, u_long u_grp)
{
   UNODEPTR uptr, *pointer;
   char     url_fname[256], buffer[256];
   FILE     *out_fp;
   int      i=(u_grp)?1:0;

   /* generate file name */
   snprintf(url_fname, sizeof(url_fname),"url_%04d%02d.%s",
            cur_year,cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(url_fname))==NULL ) return 0;

   snprintf(buffer, sizeof(buffer),"%s %d - %s",l_month[cur_month-1],
            cur_year,msg_h_url);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");
   
   if (!hide_iovol)
   {
      fprintf(out_fp,"%12s       %12s         %12s          %12s               %s\n",
           msg_h_hits,msg_h_xfer,msg_h_ixfer,msg_h_oxfer,msg_h_url);
      fprintf(out_fp,"----------------  -------------------  -------------------  "\
                  "-------------------  --------------------\n\n");
   }
   else
   {
      fprintf(out_fp," %12s       %12s               %s\n",
           msg_h_hits,msg_h_xfer,msg_h_url);
      fprintf(out_fp,"----------------  -------------------  " \
                  "--------------------\n\n");
   }

   /* do groups first (if any) */
   pointer=u_array;
   while (u_grp)
   {
      uptr=*pointer++;
      if (uptr->flag == OBJ_GRP)
      {
	 if (!hide_iovol)
	 {
            fprintf(out_fp,"%-8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  "  \
                        "%16s %6.02f%%  %s\n",
            uptr->count,
            (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
            hr_size(uptr->xfer),
            (t_xfer==0)?0:((float)uptr->xfer/t_xfer)*100.0,
            hr_size(uptr->ixfer),
            (t_ixfer==0)?0:((float)uptr->ixfer/t_ixfer)*100.0,
            hr_size(uptr->oxfer),
            (t_oxfer==0)?0:((float)uptr->oxfer/t_oxfer)*100.0,
            uptr->string);
	 }
	 else
	 {
            fprintf(out_fp,"%-8lu %6.02f%%  %16s %6.02f%%  %s\n",
            uptr->count,
            (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
            hr_size(uptr->xfer),
            (t_xfer==0)?0:((float)uptr->xfer/t_xfer)*100.0,
            uptr->string);
	 }
         u_grp--;
      }
   }

   if (i) fprintf(out_fp,"\n");

   /* now do invididual sites (if any) */
   pointer=u_array;
   while (u_reg)
   {
      uptr=*pointer++;
      if (uptr->flag == OBJ_REG)
      {
	 if (!hide_iovol)
	 {
            fprintf(out_fp,"%-8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  "  \
                        "%16s %6.02f%%  %s\n",
            uptr->count,
            (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
            hr_size(uptr->xfer),
            (t_xfer==0)?0:((float)uptr->xfer/t_xfer)*100.0,
            hr_size(uptr->ixfer),
            (t_ixfer==0)?0:((float)uptr->ixfer/t_ixfer)*100.0,
            hr_size(uptr->oxfer),
            (t_oxfer==0)?0:((float)uptr->oxfer/t_oxfer)*100.0,
            uptr->string);
	 }
	 else
	 {
            fprintf(out_fp,"%-8lu %6.02f%%  %16s %6.02f%%  %s\n",
            uptr->count,
            (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
            hr_size(uptr->xfer),
            (t_xfer==0)?0:((float)uptr->xfer/t_xfer)*100.0,
            uptr->string);
	 }
         u_reg--;
      }
   }

   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* TOP_ENTRY_TABLE - top n entry/exit urls   */
/*********************************************/

void top_entry_table(int flag)
{
   u_long cnt=0, u_entry=0, u_exit=0, tot_num;
   u_long t_entry=0, t_exit=0;
   int i;
   UNODEPTR uptr, *pointer;

   cnt=a_ctr; pointer=u_array;
   while (cnt--)
   {
      if ((*pointer)->flag == OBJ_REG )
      {
         if ((*pointer)->entry)
            {  u_entry++; t_entry+=(*pointer)->entry; }
         if ((*pointer)->exit)
            { u_exit++;   t_exit += (*pointer)->exit;  }
      }
      pointer++;
   }

   /* calculate how many we have */
   tot_num=(flag)?u_exit:u_entry;
   if (flag) { if (tot_num > ntop_exit ) tot_num=ntop_exit;  }
   else      { if (tot_num > ntop_entry) tot_num=ntop_entry; }

   /* return if none to do */
   if (!tot_num) return;

   if (flag) fprintf(out_fp,"<A NAME=\"TOPEXIT\"></A>\n"); /* do anchor tag */
   else      fprintf(out_fp,"<A NAME=\"TOPENTRY\"></A>\n");

   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=6>"        \
           "%s %lu %s %lu %s</TH></TR>\n",
           HEADLINECOLOR,msg_top_top,tot_num,msg_top_of,
           (flag)?u_exit:u_entry,(flag)?msg_top_ex:msg_top_en);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                  \
                  "<FONT SIZE=\"-1\">#</FONT></TH>\n",
                  COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"            \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
                  HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"            \
                  "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
                  VISITCOLOR,msg_h_visits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                      \
                  "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
                  PAGECOLOR,msg_h_url);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=u_array; i=0;
   while (tot_num)
   {
      uptr=*pointer++;
      if (uptr->flag != OBJ_HIDE)
      {
         fprintf(out_fp,"<TR>\n");
         fprintf(out_fp,
             "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n" \
             "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
             "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
             "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
             "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
             "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
             i+1,uptr->count,
             (t_hit==0)?0:((float)uptr->count/t_hit)*100.0,
             (flag)?uptr->exit:uptr->entry,
             (flag)?((t_exit==0)?0:((float)uptr->exit/t_exit)*100.0)
                   :((t_entry==0)?0:((float)uptr->entry/t_entry)*100.0));

         /* check for a service prefix (ie: http://) */
         if (strstr(uptr->string,"://")!=NULL)
          fprintf(out_fp,
             "<A HREF=\"%s\">%s</A></FONT></TD></TR>\n",
              uptr->string,uptr->string);
	 else
         {
            if (use_https)
            /* secure server mode, use https:// */
             fprintf(out_fp,
                "<A HREF=\"https://%s%s\">%s</A></FONT></TD></TR>\n",
                 hname,uptr->string,uptr->string);
            else
            /* otherwise use standard 'http://' */
             fprintf(out_fp,
                "<A HREF=\"http://%s%s\">%s</A></FONT></TD></TR>\n",
                 hname,uptr->string,uptr->string);
	 }
         tot_num--;
         i++;
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* TOP_REFS_TABLE - generate top n table     */
/*********************************************/

void top_refs_table()
{
   u_long cnt=0, r_reg=0, r_grp=0, r_hid=0, tot_num;
   int i;
   RNODEPTR rptr, *pointer;

   if (t_ref==0) return;        /* return if none to process */

   cnt=a_ctr; pointer=r_array;
   while(cnt--)
   {
      /* calculate totals */
      switch ((*pointer)->flag)
      {
         case OBJ_REG:  r_reg++;  break;
         case OBJ_HIDE: r_hid++;  break;
         case OBJ_GRP:  r_grp++;  break;
      }
      pointer++;
   }

   if ( (tot_num=r_reg+r_grp)==0 ) return;              /* split if none    */
   if (tot_num > ntop_refs) tot_num=ntop_refs;          /* get max to do... */

   fprintf(out_fp,"<A NAME=\"TOPREFS\"></A>\n");
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=4>"         \
           "%s %lu %s %lu %s</TH></TR>\n",
           HEADLINECOLOR, msg_top_top, tot_num, msg_top_of, t_ref, msg_top_r);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                   \
          "<FONT SIZE=\"-1\">#</FONT></TH>\n",
          COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
          HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                       \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
          PAGECOLOR,msg_h_ref);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=r_array; i=0;
   while(tot_num)
   {
      rptr=*pointer++;
      if (rptr->flag != OBJ_HIDE)
      {
         /* shade grouping? */
         if (shade_groups && (rptr->flag==OBJ_GRP))
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">\n", GRPCOLOR);
         else fprintf(out_fp,"<TR>\n");

         fprintf(out_fp,
             "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"  \
             "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
             "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
             "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
             i+1,rptr->count,
             (t_hit==0)?0:((float)rptr->count/t_hit)*100.0);

         if (rptr->flag==OBJ_GRP)
         {
            if (hlite_groups)
               fprintf(out_fp,"<STRONG>%s</STRONG>",rptr->string);
            else fprintf(out_fp,"%s",rptr->string);
         }
         else
         {
            if (rptr->string[0] != '-')
                fprintf(out_fp,"<A HREF=\"%s\" REL=\"nofollow\">%s</A>",
                rptr->string, rptr->string);
            else
            fprintf(out_fp,"%s", rptr->string);
         }
         fprintf(out_fp,"</FONT></TD></TR>\n");
         tot_num--;
         i++;
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if ( (all_refs) && ((r_reg+r_grp)>ntop_refs) )
   {
      if (all_refs_page(r_reg, r_grp))
      {
         fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
         fprintf(out_fp,"<TD COLSPAN=4 ALIGN=\"center\">\n");
         fprintf(out_fp,"<FONT SIZE=\"-1\">");
         fprintf(out_fp,"<A HREF=\"./ref_%04d%02d.%s\">",
                 cur_year,cur_month,html_ext);
         fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_refs);
      }
   }
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* ALL_REFS_PAGE - HTML page of all refs     */
/*********************************************/

int all_refs_page(u_long r_reg, u_long r_grp)
{
   RNODEPTR rptr, *pointer;
   char     ref_fname[256], buffer[256];
   FILE     *out_fp;
   int      i=(r_grp)?1:0;

   /* generate file name */
   snprintf(ref_fname, sizeof(ref_fname),"ref_%04d%02d.%s",cur_year,
            cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(ref_fname))==NULL ) return 0;

   snprintf(buffer, sizeof(buffer),"%s %d - %s",l_month[cur_month-1],
            cur_year,msg_h_ref);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");

   fprintf(out_fp," %12s      %s\n",msg_h_hits,msg_h_ref);
   fprintf(out_fp,"----------------  --------------------\n\n");

   /* do groups first (if any) */
   pointer=r_array;
   while(r_grp)
   {
      rptr=*pointer++;
      if (rptr->flag == OBJ_GRP)
      {
         fprintf(out_fp,"%-8lu %6.02f%%  %s\n",
            rptr->count,
            (t_hit==0)?0:((float)rptr->count/t_hit)*100.0,
            rptr->string);
         r_grp--;
      }
   }

   if (i) fprintf(out_fp,"\n");

   pointer=r_array;
   while(r_reg)
   {
      rptr=*pointer++;
      if (rptr->flag == OBJ_REG)
      {
         fprintf(out_fp,"%-8lu %6.02f%%  <A REL=\"nofollow\">%s</A>\n",
            rptr->count,
            (t_hit==0)?0:((float)rptr->count/t_hit)*100.0,
            rptr->string);
         r_reg--;
      }
   }

   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* TOP_AGENTS_TABLE - generate top n table   */
/*********************************************/

void top_agents_table()
{
   u_long cnt, a_reg=0, a_grp=0, a_hid=0, tot_num;
   int i;
   ANODEPTR aptr, *pointer;

   if (t_agent == 0) return;    /* don't bother if we don't have any */

   cnt=a_ctr; pointer=a_array;
   while(cnt--)
   {
      /* calculate totals */
      switch ((*pointer)->flag)
      {
         case OBJ_REG:   a_reg++;  break;
         case OBJ_GRP:   a_grp++;  break;
         case OBJ_HIDE:  a_hid++;  break;
      }
      pointer++;
   }

   if ( (tot_num=a_reg+a_grp)==0 ) return;              /* split if none    */
   if (tot_num > ntop_agents) tot_num=ntop_agents;      /* get max to do... */

   fprintf(out_fp,"<A NAME=\"TOPAGENTS\"></A>\n");
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=4>"        \
          "%s %lu %s %lu %s</TH></TR>\n",
          HEADLINECOLOR, msg_top_top, tot_num, msg_top_of, t_agent, msg_top_a);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                  \
          "<FONT SIZE=\"-1\">#</FONT></TH>\n",
          COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"            \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
          HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
          PAGECOLOR,msg_h_agent);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=a_array; i=0;
   while(tot_num)
   {
      aptr=*pointer++;
      if (aptr->flag != OBJ_HIDE)
      {
         /* shade grouping? */
         if (shade_groups && (aptr->flag==OBJ_GRP))
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">\n", GRPCOLOR);
         else fprintf(out_fp,"<TR>\n");

         fprintf(out_fp,
             "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n" \
             "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
             "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
             "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
             i+1,aptr->count,
             (t_hit==0)?0:((float)aptr->count/t_hit)*100.0);

         if ((aptr->flag==OBJ_GRP)&&hlite_groups)
            fprintf(out_fp,"<STRONG>%s</STRONG></FONT></TD></TR>\n",
               aptr->string);
         else fprintf(out_fp,"%s</FONT></TD></TR>\n",
               aptr->string);
         tot_num--;
         i++;
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if ( (all_agents) && ((a_reg+a_grp)>ntop_agents) )
   {
      if (all_agents_page(a_reg, a_grp))
      {
         fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
         fprintf(out_fp,"<TD COLSPAN=4 ALIGN=\"center\">\n");
         fprintf(out_fp,"<FONT SIZE=\"-1\">");
         fprintf(out_fp,"<A HREF=\"./agent_%04d%02d.%s\">",
                 cur_year,cur_month,html_ext);
         fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_agents);
      }
   }
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* ALL_AGENTS_PAGE - HTML user agent page    */
/*********************************************/

int all_agents_page(u_long a_reg, u_long a_grp)
{
   ANODEPTR aptr, *pointer;
   char     agent_fname[256], buffer[256];
   FILE     *out_fp;
   int      i=(a_grp)?1:0;

   /* generate file name */
   snprintf(agent_fname, sizeof(agent_fname),"agent_%04d%02d.%s",cur_year,
            cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(agent_fname))==NULL ) return 0;

   snprintf(buffer, sizeof(buffer),"%s %d - %s",l_month[cur_month-1],
            cur_year,msg_h_agent);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");

   fprintf(out_fp," %12s      %s\n",msg_h_hits,msg_h_agent);
   fprintf(out_fp,"----------------  ----------------------\n\n");

   /* do groups first (if any) */
   pointer=a_array;
   while(a_grp)
   {
      aptr=*pointer++;
      if (aptr->flag == OBJ_GRP)
      {
         fprintf(out_fp,"%-8lu %6.02f%%  %s\n",
             aptr->count,
             (t_hit==0)?0:((float)aptr->count/t_hit)*100.0,
             aptr->string);
         a_grp--;
      }
   }

   if (i) fprintf(out_fp,"\n");

   pointer=a_array;
   while(a_reg)
   {
      aptr=*pointer++;
      if (aptr->flag == OBJ_REG)
      {
         fprintf(out_fp,"%-8lu %6.02f%%  %s\n",
             aptr->count,
             (t_hit==0)?0:((float)aptr->count/t_hit)*100.0,
             aptr->string);
         a_reg--;
      }
   }

   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* TOP_SEARCH_TABLE - generate top n table   */
/*********************************************/

void top_search_table()
{
   u_long   cnt,t_val=0, tot_num;
   int      i;
   SNODEPTR sptr, *pointer;
   const char* array[7]={"&auml;", "&Auml;", "&ouml;", "&Ouml;", "&uuml;",
                         "&Uuml;", "&szlig;"}; /* new characters */
   
   if ( (t_ref==0)||(a_ctr==0)) return;   /* don't bother if none to do    */

   cnt=tot_num=a_ctr; pointer=s_array;
   while(cnt--)
   {
      t_val+=(*pointer)->count;
      pointer++;
   }

   if ( tot_num > ntop_search) tot_num=ntop_search;

   fprintf(out_fp,"<A NAME=\"TOPSEARCH\"></A>\n");
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=4>"        \
          "%s %lu %s %lu %s</TH></TR>\n",
          HEADLINECOLOR, msg_top_top, tot_num, msg_top_of, a_ctr, msg_top_sr);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                  \
          "<FONT SIZE=\"-1\">#</FONT></TH>\n",
          COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"            \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",
          HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",
          PAGECOLOR,msg_h_search);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=s_array; i=0;
   while(tot_num)
   {
      sptr=*pointer++;
      fprintf(out_fp,
         "<TR>\n"                                                     \
         "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n" \
         "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"   \
         "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
         i+1,sptr->count,
         (t_val==0)?0:((float)sptr->count/t_val)*100.0);
      fprintf(out_fp,"%s</FONT></TD></TR>\n",
         Transliterate("",array,sptr->string));
      tot_num--;
      i++;
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if ( (all_search) && (a_ctr>ntop_search) )
   {
      if (all_search_page(a_ctr, t_val))
      {
         fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
         fprintf(out_fp,"<TD COLSPAN=4 ALIGN=\"center\">\n");
         fprintf(out_fp,"<FONT SIZE=\"-1\">");
         fprintf(out_fp,"<A HREF=\"./search_%04d%02d.%s\">",
                 cur_year,cur_month,html_ext);
         fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_search);
      }
   }
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* ALL_SEARCH_PAGE - HTML for search strings */
/*********************************************/

int all_search_page(u_long tot_num, u_long t_val)
{
   SNODEPTR sptr, *pointer;
   char     search_fname[256], buffer[256];
   FILE     *out_fp;
   const char* array[7]={"&auml;", "&Auml;", "&ouml;", "&Ouml;", "&uuml;",
                         "&Uuml;", "&szlig;"}; /* new characters */
   
   if (!tot_num) return 0;

   /* generate file name */
   snprintf(search_fname, sizeof(search_fname),"search_%04d%02d.%s",cur_year,
            cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(search_fname))==NULL ) return 0;

   snprintf(buffer, sizeof(buffer),"%s %d - %s",l_month[cur_month-1],
            cur_year,msg_h_search);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");

   fprintf(out_fp," %12s      %s\n",msg_h_hits,msg_h_search);
   fprintf(out_fp,"----------------  ----------------------\n\n");

   pointer=s_array;
   while(tot_num)
   {
      sptr=*pointer++;
      fprintf(out_fp,"%-8lu %6.02f%%  %s\n",
         sptr->count,
         (t_val==0)?0:((float)sptr->count/t_val)*100.0,
         Transliterate("",array,sptr->string));
      tot_num--;
   }
   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* TOP_USERS_TABLE - generate top n table    */
/*********************************************/

void top_users_table()
{
   u_long cnt=0, i_reg=0, i_grp=0, i_hid=0, tot_num;
   int i,j;
   INODEPTR iptr, *pointer;

   cnt=a_ctr; pointer=i_array;
   while(cnt--)
   {
      /* calculate totals */
      switch ((*pointer)->flag)
      {
         case OBJ_REG:   i_reg++;  break;
         case OBJ_GRP:   i_grp++;  break;
         case OBJ_HIDE:  i_hid++;  break;
      }
      pointer++;
   }

   if ( (tot_num=i_reg+i_grp)==0 ) return;              /* split if none    */
   if (tot_num > ntop_users) tot_num = ntop_users;

   fprintf(out_fp,"<A NAME=\"TOPUSERS\"></A>\n");       /* now do <A> tag   */

   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=\"%i\">" \
           "%s %lu %s %lu %s</TH></TR>\n",HEADLINECOLOR,j=(!hide_iovol)?14:10,
           msg_top_top, tot_num, msg_top_of, t_user, msg_top_i);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                   \
          "<FONT SIZE=\"-1\">#</FONT></TH>\n",COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_files);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",KBYTECOLOR,msg_h_xfer);
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_ixfer);
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_oxfer);
   }
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",VISITCOLOR,msg_h_visits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                       \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",PAGECOLOR,msg_h_uname);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");

   pointer=i_array; i=0;
   while(tot_num)
   {
      iptr=*pointer++;
      if (iptr->flag != OBJ_HIDE)
      {
         /* shade grouping? */
         if (shade_groups && (iptr->flag==OBJ_GRP))
            fprintf(out_fp,"<TR BGCOLOR=\"%s\">\n", GRPCOLOR);
         else fprintf(out_fp,"<TR>\n");
         if (!hide_iovol)
         {
            fprintf(out_fp,
              "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
              i+1,iptr->count,
              (t_hit==0)?0:((float)iptr->count/t_hit)*100.0,iptr->files,
              (t_file==0)?0:((float)iptr->files/t_file)*100.0,hr_size(iptr->xfer),
              (t_xfer==0)?0:((float)iptr->xfer/t_xfer)*100.0,hr_size(iptr->ixfer),
              (t_ixfer==0)?0:((float)iptr->ixfer/t_ixfer)*100.0,hr_size(iptr->oxfer),
              (t_oxfer==0)?0:((float)iptr->oxfer/t_oxfer)*100.0,iptr->visit,
              (t_visit==0)?0:((float)iptr->visit/t_visit)*100.0);
	 }
	 else
	 {
	    fprintf(out_fp,
              "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"   \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"  \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"    \
              "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">",
              i+1,iptr->count,
              (t_hit==0)?0:((float)iptr->count/t_hit)*100.0,iptr->files,
              (t_file==0)?0:((float)iptr->files/t_file)*100.0,hr_size(iptr->xfer),
              (t_xfer==0)?0:((float)iptr->xfer/t_xfer)*100.0,iptr->visit,
              (t_visit==0)?0:((float)iptr->visit/t_visit)*100.0);
	 }

         if ((iptr->flag==OBJ_GRP)&&hlite_groups)
             fprintf(out_fp,"<STRONG>%s</STRONG></FONT></TD></TR>\n",
               iptr->string);
         else fprintf(out_fp,"%s</FONT></TD></TR>\n",
               iptr->string);
         tot_num--;
         i++;
      }
   }

   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   if ( (all_users) && ((i_reg+i_grp)>ntop_users) )
   {
      if (all_users_page(i_reg, i_grp))
      {
         fprintf(out_fp,"<TR BGCOLOR=\"%s\">",GRPCOLOR);
         fprintf(out_fp,"<TD COLSPAN=14 ALIGN=\"center\">\n");
         fprintf(out_fp,"<FONT SIZE=\"-1\">");
         fprintf(out_fp,"<A HREF=\"./user_%04d%02d.%s\">",
            cur_year,cur_month,html_ext);
         fprintf(out_fp,"%s</A></TD></TR>\n",msg_v_users);
      }
   }
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* ALL_USERS_PAGE - HTML of all usernames    */
/*********************************************/

int all_users_page(u_long i_reg, u_long i_grp)
{
   INODEPTR iptr, *pointer;
   char     user_fname[256], buffer[256];
   FILE     *out_fp;
   int      i=(i_grp)?1:0;

   /* generate file name */
   snprintf(user_fname, sizeof(user_fname),"user_%04d%02d.%s",cur_year,
            cur_month,html_ext);

   /* open file */
   if ( (out_fp=open_out_file(user_fname))==NULL ) return 0;

   snprintf(buffer, sizeof(buffer),"%s %d - %s",l_month[cur_month-1],
            cur_year,msg_h_uname);
   write_html_head(buffer, out_fp);

   fprintf(out_fp,"<FONT SIZE=\"-1\"></CENTER><PRE>\n");

   if (!hide_iovol)
   {
      fprintf(out_fp,"%12s      %12s       %12s         %12s         %12s        "\
                  "%12s            %s\n",
           msg_h_hits, msg_h_files, msg_h_xfer, msg_h_ixfer, msg_h_oxfer,
           msg_h_visits, msg_h_uname);
      fprintf(out_fp,"----------------  ----------------  -------------------  " \
                  "-------------------  -------------------  "                \
                  "----------------  --------------------\n\n");
   }
   else
   {
      fprintf(out_fp," %12s      %12s      %12s      %12s      %s\n",
           msg_h_hits, msg_h_files, msg_h_xfer, msg_h_visits, msg_h_uname);
      fprintf(out_fp,"----------------  ----------------  -------------------  " \
                  "----------------  --------------------\n\n");
   }

   /* Do groups first (if any) */
   pointer=i_array;
   while(i_grp)
   {
      iptr=*pointer++;
      if (iptr->flag == OBJ_GRP)
      {
	 if (!hide_iovol)
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  " \
            "%16s %6.02f%%  %8lu %6.02f%%  %s\n",
            iptr->count,
            (t_hit==0)?0:((float)iptr->count/t_hit)*100.0,iptr->files,
            (t_file==0)?0:((float)iptr->files/t_file)*100.0,hr_size(iptr->xfer),
            (t_xfer==0)?0:((float)iptr->xfer/t_xfer)*100.0,hr_size(iptr->ixfer),
            (t_ixfer==0)?0:((float)iptr->ixfer/t_ixfer)*100.0,hr_size(iptr->oxfer),
            (t_oxfer==0)?0:((float)iptr->oxfer/t_oxfer)*100.0,iptr->visit,
            (t_visit==0)?0:((float)iptr->visit/t_visit)*100.0,
            iptr->string);
	 }
	 else
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  " \
            "%8lu %6.02f%%  %s\n",
            iptr->count,
            (t_hit==0)?0:((float)iptr->count/t_hit)*100.0,iptr->files,
            (t_file==0)?0:((float)iptr->files/t_file)*100.0,hr_size(iptr->xfer),
            (t_xfer==0)?0:((float)iptr->xfer/t_xfer)*100.0,iptr->visit,
            (t_visit==0)?0:((float)iptr->visit/t_visit)*100.0,
            iptr->string);
	 }
         i_grp--;
      }
   }

   if (i) fprintf(out_fp,"\n");

   /* Now do individual users (if any) */
   pointer=i_array;
   while(i_reg)
   {
      iptr=*pointer++;
      if (iptr->flag == OBJ_REG)
      {
	 if (!hide_iovol)
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  %16s %6.02f%%  "  \
            "%16s %6.02f%%  %8lu %6.02f%%  %s\n",
            iptr->count,
            (t_hit==0)?0:((float)iptr->count/t_hit)*100.0,iptr->files,
            (t_file==0)?0:((float)iptr->files/t_file)*100.0,hr_size(iptr->xfer),
            (t_xfer==0)?0:((float)iptr->xfer/t_xfer)*100.0,hr_size(iptr->ixfer),
            (t_ixfer==0)?0:((float)iptr->ixfer/t_ixfer)*100.0,hr_size(iptr->oxfer),
            (t_oxfer==0)?0:((float)iptr->oxfer/t_oxfer)*100.0,iptr->visit,
            (t_visit==0)?0:((float)iptr->visit/t_visit)*100.0,
            iptr->string);
	 }
	 else
	 {
            fprintf(out_fp,
            "%-8lu %6.02f%%  %8lu %6.02f%%  %16s %6.02f%%  "  \
            "%8lu %6.02f%%  %s\n",
            iptr->count,
            (t_hit==0)?0:((float)iptr->count/t_hit)*100.0,iptr->files,
            (t_file==0)?0:((float)iptr->files/t_file)*100.0,hr_size(iptr->xfer),
            (t_xfer==0)?0:((float)iptr->xfer/t_xfer)*100.0,iptr->visit,
            (t_visit==0)?0:((float)iptr->visit/t_visit)*100.0,
            iptr->string);
	 }
         i_reg--;
      }
   }

   fprintf(out_fp,"</PRE></FONT>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 1;
}

/*********************************************/
/* TOP_CTRY_TABLE - top countries table      */
/*********************************************/

void top_ctry_table()
{
   int i,j,k,x,tot_num=0,tot_ctry=0;
   int ctry_fnd;
   u_long idx;
   HNODEPTR hptr;
   char *domain, *country = NULL;
   u_long pie_data[10];
   char   *pie_legend[10];
   char   pie_title[48];
   char   pie_fname[48];
   const char* array[7]={"&auml;", "&Auml;", "&ouml;", "&Ouml;", "&uuml;",
                         "&Uuml;", "&szlig;"}; /* new characters */
   extern int ctry_graph;  /* include external flag */

#ifdef USE_GEOIP
   const char *result;
   char code[3];
   code[2]='\0';   
#endif	/* USE_GEOIP */

   /* scan hash table adding up domain totals */
   for (i=0;i<MAXHASH;i++)
   {
      hptr=sm_htab[i];
      while (hptr!=NULL)
      {
         if (hptr->flag != OBJ_GRP)   /* ignore group totals */
         {
            domain = hptr->string+strlen(hptr->string)-1;
            while ( (*domain!='.')&&(domain!=hptr->string)) domain--;
            if (domain==hptr->string)
	       country=NULL;
            else if (isdigit((int)*++domain))
            {
#ifdef USE_GEOIP
               if (use_geoip)
               {
                  result=GeoIP_country_code_by_name(gi, hptr->string);
                  if (!GEOIP_OK(result))
                     country=NULL;
                  else
                  {
                     code[0]=tolower(result[0]);
                     code[1]=tolower(result[1]);

                     country=code;
                  }
               }
               else
                  country=NULL;
#else
               country=NULL;
#endif	/* USE_GEOIP */
            }
            else
               country=domain;

            ctry_fnd=0;
            if (country!=NULL)
            {
               idx=ctry_idx(country);
               for (j=0;ctry[j].desc;j++)
               {
                  if (idx==ctry[j].idx)
                  {
                     ctry[j].count+=hptr->count;
                     ctry[j].files+=hptr->files;
                     ctry[j].xfer +=hptr->xfer;
                     ctry[j].ixfer+=hptr->ixfer;
                     ctry[j].oxfer+=hptr->oxfer;
                     ctry_fnd=1;
                     break;
                  }
               }
            }
            if ((!ctry_fnd)||(country==NULL))
            {
#ifdef USE_GEOIP
               if (use_geoip && debug_mode)
                  fprintf(stderr, "--> unresolved country for '%s' (GeoIP says %s:%s)\n",
                          hptr->string,
                          GeoIP_country_code_by_name(gi, hptr->string),
                          GeoIP_country_name_by_name(gi, hptr->string));
#endif	/* USE_GEOIP */
    	       ctry[0].count+=hptr->count;
    	       ctry[0].files+=hptr->files;
    	       ctry[0].xfer +=hptr->xfer;
               ctry[0].ixfer+=hptr->ixfer;
               ctry[0].oxfer+=hptr->oxfer;
            }
         }
         hptr=hptr->next;
      }
   }

   for (i=0;ctry[i].desc;i++)
   {
      if (ctry[i].count!=0) tot_ctry++;
      for (j=0;j<ntop_ctrys;j++)
      {
         if (top_ctrys[j]==NULL) { top_ctrys[j]=&ctry[i]; break; }
         else
         {
            if (ctry[i].count > top_ctrys[j]->count)
            {
               for (x=ntop_ctrys-1;x>j;x--)
                  top_ctrys[x]=top_ctrys[x-1];
               top_ctrys[x]=&ctry[i];
               break;
            }
         }
      }
   }

   /* put our anchor tag first... */
   fprintf(out_fp,"<A NAME=\"TOPCTRYS\"></A>\n");

   /* generate pie chart if needed */
   if (ctry_graph)
   {
      for (i=0;i<10;i++) pie_data[i]=0;             /* init data array      */
      if (ntop_ctrys<10) j=ntop_ctrys; else j=10;   /* ensure data size     */

      for (i=0;i<j;i++)
      {
         pie_data[i]=top_ctrys[i]->count;           /* load the array       */
         pie_legend[i]=top_ctrys[i]->desc;
      }
      snprintf(pie_title, sizeof(pie_title),"%s %s %d",msg_ctry_use,
               l_month[cur_month-1],cur_year);
      snprintf(pie_fname, sizeof(pie_fname),"ctry_usage_%04d%02d.png",
               cur_year,cur_month);

      pie_chart(pie_fname,pie_title,t_hit,pie_data,pie_legend);  /* do it   */

      /* put the image tag in the page */
      fprintf(out_fp,"<IMG SRC=\"%s\" ALT=\"%s\" " \
                  "HEIGHT=300 WIDTH=512><P>\n",pie_fname,
		  Transliterate("",array,pie_title));
   }

   /* Now do the table */
   for (i=0;i<ntop_ctrys;i++) if (top_ctrys[i]->count!=0) tot_num++;
   fprintf(out_fp,"<TABLE WIDTH=510 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=CENTER COLSPAN=\"%i\">"    \
           "%s %d %s %d %s</TH></TR>\n",HEADLINECOLOR,k=(!hide_iovol)?12:8,
           msg_top_top,tot_num,msg_top_of,tot_ctry,msg_top_c);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" ALIGN=center>"                   \
          "<FONT SIZE=\"-1\">#</FONT></TH>\n",COUNTERCOLOR);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_files);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"             \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",KBYTECOLOR,msg_h_xfer);
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"          \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_ixfer);
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center COLSPAN=2>"          \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_oxfer);
   }
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=center>"                       \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",PAGECOLOR,msg_h_ctry);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   for (i=0;i<ntop_ctrys;i++)
   {
      if (top_ctrys[i]->count!=0)
      {
	 if (!hide_iovol)
	 {
            fprintf(out_fp,"<TR>"                                             \
              "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"     \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"     \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"     \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">%s</FONT></TD></TR>\n",
              i+1,top_ctrys[i]->count,
              (t_hit==0)?0:((float)top_ctrys[i]->count/t_hit)*100.0,
              top_ctrys[i]->files,
              (t_file==0)?0:((float)top_ctrys[i]->files/t_file)*100.0,
              hr_size(top_ctrys[i]->xfer),
              (t_xfer==0)?0:((float)top_ctrys[i]->xfer/t_xfer)*100.0,
              hr_size(top_ctrys[i]->ixfer),
              (t_ixfer==0)?0:((float)top_ctrys[i]->ixfer/t_ixfer)*100.0,
              hr_size(top_ctrys[i]->oxfer),
              (t_oxfer==0)?0:((float)top_ctrys[i]->oxfer/t_oxfer)*100.0,
              Transliterate("",array,top_ctrys[i]->desc));
	 }
	 else
	 {
            fprintf(out_fp,"<TR>"                                             \
              "<TD ALIGN=center><FONT SIZE=\"-1\"><B>%d</B></FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%lu</B></FONT></TD>\n"    \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=right><FONT SIZE=\"-1\"><B>%s</B></FONT></TD>\n"     \
              "<TD ALIGN=right><FONT SIZE=\"-2\">%3.02f%%</FONT></TD>\n"      \
              "<TD ALIGN=left NOWRAP><FONT SIZE=\"-1\">%s</FONT></TD></TR>\n",
              i+1,top_ctrys[i]->count,
              (t_hit==0)?0:((float)top_ctrys[i]->count/t_hit)*100.0,
              top_ctrys[i]->files,
              (t_file==0)?0:((float)top_ctrys[i]->files/t_file)*100.0,
              hr_size(top_ctrys[i]->xfer),
              (t_xfer==0)?0:((float)top_ctrys[i]->xfer/t_xfer)*100.0,
              Transliterate("",array,top_ctrys[i]->desc));
	 }
      }
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"</TABLE>\n<P>\n");
}

/*********************************************/
/* DUMP_ALL_SITES - dump sites to tab file   */
/*********************************************/

void dump_all_sites()
{
   HNODEPTR hptr, *pointer;
   FILE     *out_fp;
   char     filename[256];
   u_long   cnt=a_ctr;
#ifdef USE_GEOIP
   const char *result;
#endif	/* USE_GEOIP */

   /* generate file name */
   snprintf(filename, sizeof(filename),"%s/site_%04d%02d.%s",
      (dump_path)?dump_path:".",cur_year,cur_month,dump_ext);

   /* open file */
   if ( (out_fp=open_out_file(filename))==NULL ) return;

   /* need a header? */
   if (dump_header)
   {
#ifdef USE_GEOIP
      if (use_geoip)
         fprintf(out_fp,"%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
          msg_h_hits,msg_h_files,msg_h_xfer,msg_h_ixfer,msg_h_oxfer,
	  msg_h_visits,msg_h_hname,msg_h_ctry);
      else
      fprintf(out_fp,"%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
       msg_h_hits,msg_h_files,msg_h_xfer,msg_h_ixfer,msg_h_oxfer,msg_h_visits,
       msg_h_hname); 
#else
      fprintf(out_fp,"%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
       msg_h_hits,msg_h_files,msg_h_xfer,msg_h_ixfer,msg_h_oxfer,msg_h_visits,
       msg_h_hname); 
#endif	/* USE_GEOIP */
   }

   /* dump 'em */
   pointer=h_array;
   while (cnt)
   {
      hptr=*pointer++;
      if (hptr->flag != OBJ_GRP)
      {
#ifdef USE_GEOIP
         if (use_geoip)
         {
            result=GeoIP_country_name_by_name(gi, hptr->string);
            fprintf(out_fp,
            "%lu\t%lu\t%.0f\t%.0f\t%.0f\t%lu\t%s\t%s\n",
               hptr->count,hptr->files,hptr->xfer/1024,
	       hptr->ixfer/1024,hptr->oxfer/1024,
               hptr->visit,hptr->string,GEOIP_OK(result)?result:ctry[0].desc);
         }
         else
            fprintf(out_fp,
            "%lu\t%lu\t%.0f\t%.0f\t%.0f\t%lu\t%s\n",
               hptr->count,hptr->files,hptr->xfer/1024,
	       hptr->ixfer/1024,hptr->oxfer/1024,
               hptr->visit,hptr->string);
#else
         fprintf(out_fp,
         "%lu\t%lu\t%.0f\t%.0f\t%.0f\t%lu\t%s\n",
            hptr->count,hptr->files,hptr->xfer/1024,
	    hptr->ixfer/1024,hptr->oxfer/1024,
            hptr->visit,hptr->string);
#endif	/* USE_GEOIP */
      }
      cnt--;
   }
   fclose(out_fp);
   return;
}

/*********************************************/
/* DUMP_ALL_URLS - dump all urls to tab file */
/*********************************************/

void dump_all_urls()
{
   UNODEPTR uptr, *pointer;
   FILE     *out_fp;
   char     filename[256];
   u_long   cnt=a_ctr;

   /* generate file name */
   snprintf(filename, sizeof(filename),"%s/url_%04d%02d.%s",
      (dump_path)?dump_path:".",cur_year,cur_month,dump_ext);

   /* open file */
   if ( (out_fp=open_out_file(filename))==NULL ) return;

   /* need a header? */
   if (dump_header)
   {
      fprintf(out_fp,"%s\t%s\t%s\t%s\t%s\n",msg_h_hits,msg_h_xfer,msg_h_ixfer,
                      msg_h_oxfer,msg_h_url);
   }

   /* dump 'em */
   pointer=u_array;
   while (cnt)
   {
      uptr=*pointer++;
      if (uptr->flag != OBJ_GRP)
      {
         fprintf(out_fp,"%lu\t%.0f\t%.0f\t%.0f\t%s\n",
            uptr->count,uptr->xfer/1024,uptr->ixfer/1024,uptr->oxfer/1024,
            uptr->string);
      }
      cnt--;
   }
   fclose(out_fp);
   return;
}

/*********************************************/
/* DUMP_ALL_REFS - dump all refs to tab file */
/*********************************************/

void dump_all_refs()
{
   RNODEPTR rptr, *pointer;
   FILE     *out_fp;
   char     filename[256];
   u_long   cnt=a_ctr;

   /* generate file name */
   snprintf(filename, sizeof(filename),"%s/ref_%04d%02d.%s",
      (dump_path)?dump_path:".",cur_year,cur_month,dump_ext);

   /* open file */
   if ( (out_fp=open_out_file(filename))==NULL ) return;

   /* need a header? */
   if (dump_header)
   {
      fprintf(out_fp,"%s\t%s\n",msg_h_hits,msg_h_ref);
   }

   /* dump 'em */
   pointer=r_array;
   while(cnt)
   {
      rptr=*pointer++;
      if (rptr->flag != OBJ_GRP)
      {
         fprintf(out_fp,"%lu\t%s\n",rptr->count, rptr->string);
      }
      cnt--;
   }
   fclose(out_fp);
   return;
}

/*********************************************/
/* DUMP_ALL_AGENTS - dump agents htab file   */
/*********************************************/

void dump_all_agents()
{
   ANODEPTR aptr, *pointer;
   FILE     *out_fp;
   char     filename[256];
   u_char   cnt=a_ctr;

   /* generate file name */
   snprintf(filename, sizeof(filename),"%s/agent_%04d%02d.%s",
      (dump_path)?dump_path:".",cur_year,cur_month,dump_ext);

   /* open file */
   if ( (out_fp=open_out_file(filename))==NULL ) return;

   /* need a header? */
   if (dump_header)
   {
      fprintf(out_fp,"%s\t%s\n",msg_h_hits,msg_h_agent);
   }

   /* dump 'em */
   pointer=a_array;
   while(cnt)
   {
      aptr=*pointer++;
      if (aptr->flag != OBJ_GRP)
      {
         fprintf(out_fp,"%lu\t%s\n",aptr->count,aptr->string);
      }
      cnt--;
   }
   fclose(out_fp);
   return;
}

/*********************************************/
/* DUMP_ALL_USERS - dump username tab file   */
/*********************************************/

void dump_all_users()
{
   INODEPTR iptr, *pointer;
   FILE     *out_fp;
   char     filename[256];
   u_long   cnt=a_ctr;

   /* generate file name */
   snprintf(filename, sizeof(filename),"%s/user_%04d%02d.%s",
      (dump_path)?dump_path:".",cur_year,cur_month,dump_ext);

   /* open file */
   if ( (out_fp=open_out_file(filename))==NULL ) return;

   /* need a header? */
   if (dump_header)
   {
      fprintf(out_fp,"%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
         msg_h_hits,msg_h_files,msg_h_xfer,msg_h_ixfer,msg_h_oxfer,msg_h_visits,
         msg_h_uname);
   }

   /* dump 'em */
   pointer=i_array;
   while(cnt)
   {
      iptr=*pointer++;
      if (iptr->flag != OBJ_GRP)
      {
         fprintf(out_fp,
         "%lu\t%lu\t%.0f\t%.0f\t%.0f\t%lu\t%s\n",
            iptr->count,iptr->files,iptr->xfer/1024,
	    iptr->ixfer/1024,iptr->oxfer/1024,
            iptr->visit,iptr->string);
      }
      cnt--;
   }
   fclose(out_fp);
   return;
}

/*********************************************/
/* DUMP_ALL_SEARCH - dump search htab file   */
/*********************************************/

void dump_all_search()
{
   SNODEPTR sptr, *pointer;
   FILE     *out_fp;
   char     filename[256];
   u_char   cnt=a_ctr;

   /* generate file name */
   snprintf(filename, sizeof(filename),"%s/search_%04d%02d.%s",
      (dump_path)?dump_path:".",cur_year,cur_month,dump_ext);

   /* open file */
   if ( (out_fp=open_out_file(filename))==NULL ) return;

   /* need a header? */
   if (dump_header)
   {
      fprintf(out_fp,"%s\t%s\n",msg_h_hits,msg_h_search);
   }

   /* dump 'em */
   pointer=s_array;
   while(cnt)
   {
      sptr=*pointer++;
      fprintf(out_fp,"%lu\t%s\n",sptr->count,sptr->string);
      cnt--;
   }
   fclose(out_fp);
   return;
}

/*********************************************/
/* WRITE_MAIN_INDEX - main index.html file   */
/*********************************************/

int write_main_index()
{
   /* create main index file */

   int  i,j,days_in_month;
   int  lyear=0;
   int	s_mth=0;
   double  gt_hit=0.0;
   double  gt_files=0.0;
   double  gt_pages=0.0;
   double  gt_xfer=0.0;
   double  gt_ixfer=0.0;
   double  gt_oxfer=0.0;
   double  gt_visits=0.0;
   char    index_fname[256];
   char    buffer[BUFSIZE];

   if (verbose>1) printf("%s\n",msg_gen_sum);

   snprintf(buffer, sizeof(buffer),"%s %s",msg_main_us,hname);

   for (i=0;i<12;i++)                   /* get last month in history */
   {
      if (hist_year[i]>lyear)
       { lyear=hist_year[i]; s_mth=hist_month[i]; }
      if (hist_year[i]==lyear)
      {
         if (hist_month[i]>=s_mth)
            s_mth=hist_month[i];
      }
   }

   i=(s_mth==12)?1:s_mth+1;

   year_graph6x(   "usage.png",         /* filename          */
                   buffer,              /* graph title       */
                   i,                   /* last month        */
                   hist_hit,            /* data set 1        */
                   hist_files,          /* data set 2        */
                   hist_site,           /* data set 3        */
                   hist_xfer,           /* data set 4        */
                   hist_ixfer,          /* data set 5        */
                   hist_oxfer,          /* data set 6        */
                   hist_page,           /* data set 7        */
                   hist_visit);         /* data set 8        */

   /* now do html stuff... */
   snprintf(index_fname, sizeof(index_fname),"index.%s",html_ext);

   if ( (out_fp=fopen(index_fname,"w")) == NULL)
   {
      if (verbose)
      fprintf(stderr,"%s %s!\n",msg_no_open,index_fname);
      return 1;
   }
   write_html_head(msg_main_per, out_fp);
   /* year graph */
   fprintf(out_fp,"<IMG SRC=\"usage.png\" ALT=\"%s\" "    \
                  "HEIGHT=256 WIDTH=512><P>\n",buffer);
   /* month table */
   fprintf(out_fp,"<TABLE WIDTH=600 BORDER=%i CELLSPACING=1 CELLPADDING=1>\n",
           table_border);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH COLSPAN=\"%i\" BGCOLOR=\"%s\" ALIGN=center>",
          j=(!hide_iovol)?13:11,HEADLINECOLOR);
   fprintf(out_fp,"%s</TH></TR>\n",msg_main_sum);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH ALIGN=left ROWSPAN=2 BGCOLOR=\"%s\">"          \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HEADLINECOLOR,msg_h_mth);
   fprintf(out_fp,"<TH ALIGN=center COLSPAN=4 BGCOLOR=\"%s\">"            \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HEADLINECOLOR,msg_main_da);
   fprintf(out_fp,"<TH ALIGN=center COLSPAN=\"%i\" BGCOLOR=\"%s\">"       \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",j=(!hide_iovol)?8:6,
	  HEADLINECOLOR,msg_main_mt);
   fprintf(out_fp,"<TR><TH ALIGN=center BGCOLOR=\"%s\">"                  \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_files);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",PAGECOLOR,msg_h_pages);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",VISITCOLOR,msg_h_visits);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",SITECOLOR,msg_h_sites);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",KBYTECOLOR,msg_h_xfer);
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                   \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_ixfer);
      fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                   \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HITCOLOR,msg_h_oxfer);
   }
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",VISITCOLOR,msg_h_visits);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",PAGECOLOR,msg_h_pages);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",FILECOLOR,msg_h_files);
   fprintf(out_fp,"<TH ALIGN=center BGCOLOR=\"%s\">"                      \
          "<FONT SIZE=\"-1\">%s</FONT></TH></TR>\n",HITCOLOR,msg_h_hits);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   for (i=0;i<12;i++)
   {
      if (--s_mth < 0) s_mth = 11;
      if ((hist_month[s_mth]==0) && (hist_files[s_mth]==0)) continue;
      days_in_month=(hist_lday[s_mth]-hist_fday[s_mth])+1;
      fprintf(out_fp,"<TR><TD NOWRAP><A HREF=\"usage_%04d%02d.%s\">"      \
                     "<FONT SIZE=\"-1\">%s %d</FONT></A></TD>\n",
                      hist_year[s_mth], hist_month[s_mth], html_ext,
                      s_month[hist_month[s_mth]-1], hist_year[s_mth]);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_hit[s_mth]/days_in_month);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_files[s_mth]/days_in_month);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_page[s_mth]/days_in_month);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_visit[s_mth]/days_in_month);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_site[s_mth]);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%s</FONT></TD>\n",
                      hr_size(hist_xfer[s_mth]*1024));
      if (!hide_iovol)
      {
         fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%s</FONT></TD>\n",
                      hr_size(hist_ixfer[s_mth]*1024));
         fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%s</FONT></TD>\n",
                      hr_size(hist_oxfer[s_mth]*1024));
      }
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_visit[s_mth]);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_page[s_mth]);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD>\n",
                      hist_files[s_mth]);
      fprintf(out_fp,"<TD ALIGN=right><FONT SIZE=\"-1\">%lu</FONT></TD></TR>\n",
                      hist_hit[s_mth]);
      gt_hit   += hist_hit[s_mth];
      gt_files += hist_files[s_mth];
      gt_pages += hist_page[s_mth];
      gt_xfer  += hist_xfer[s_mth];
      gt_ixfer += hist_ixfer[s_mth];
      gt_oxfer += hist_oxfer[s_mth];
      gt_visits+= hist_visit[s_mth];
   }
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"<TR><TH BGCOLOR=\"%s\" COLSPAN=6 ALIGN=left>"          \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HEADLINECOLOR,msg_h_totals);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                       \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HEADLINECOLOR,hr_size(gt_xfer*1024));
   if (!hide_iovol)
   {
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                    \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HEADLINECOLOR,hr_size(gt_ixfer*1024));
      fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                    \
          "<FONT SIZE=\"-1\">%s</FONT></TH>\n",HEADLINECOLOR,hr_size(gt_oxfer*1024));
   }
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                       \
          "<FONT SIZE=\"-1\">%.0f</FONT></TH>\n",HEADLINECOLOR,gt_visits);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                       \
          "<FONT SIZE=\"-1\">%.0f</FONT></TH>\n",HEADLINECOLOR,gt_pages);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                       \
          "<FONT SIZE=\"-1\">%.0f</FONT></TH>\n",HEADLINECOLOR,gt_files);
   fprintf(out_fp,"<TH BGCOLOR=\"%s\" ALIGN=right>"                       \
          "<FONT SIZE=\"-1\">%.0f</FONT></TH></TR>\n",HEADLINECOLOR,gt_hit);
   fprintf(out_fp,"<TR><TH HEIGHT=4></TH></TR>\n");
   fprintf(out_fp,"</TABLE>\n");
   write_html_tail(out_fp);
   fclose(out_fp);
   return 0;
}

/*********************************************/
/* QS_SITE_CMPH - QSort compare site by hits */
/*********************************************/

int qs_site_cmph(const void *cp1, const void *cp2)
{
   u_long  t1, t2;
   t1=(*(HNODEPTR *)cp1)->count;
   t2=(*(HNODEPTR *)cp2)->count;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if hits are the same, we sort by hostname instead */
   return strcmp( (*(HNODEPTR *)cp1)->string,
                  (*(HNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_SITE_CMPK - QSort cmp site by bytes    */
/*********************************************/

int qs_site_cmpk(const void *cp1, const void *cp2)
{
   double t1, t2;
   t1=(*(HNODEPTR *)cp1)->xfer;
   t2=(*(HNODEPTR *)cp2)->xfer;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if xfer bytes are the same, we sort by hostname instead */
   return strcmp( (*(HNODEPTR *)cp1)->string,
                  (*(HNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_URL_CMPH - QSort compare URL by hits   */
/*********************************************/

int qs_url_cmph(const void *cp1, const void *cp2)
{
   u_long  t1, t2;
   t1=(*(UNODEPTR *)cp1)->count;
   t2=(*(UNODEPTR *)cp2)->count;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if hits are the same, we sort by url instead */
   return strcmp( (*(UNODEPTR *)cp1)->string,
                  (*(UNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_URL_CMPK - QSort compare URL by bytes  */
/*********************************************/

int qs_url_cmpk(const void *cp1, const void *cp2)
{
   double t1, t2;
   t1=(*(UNODEPTR *)cp1)->xfer;
   t2=(*(UNODEPTR *)cp2)->xfer;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if xfer bytes are the same, we sort by url instead */
   return strcmp( (*(UNODEPTR *)cp1)->string,
                  (*(UNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_URL_CMPN - QSort compare URL by entry  */
/*********************************************/

int qs_url_cmpn(const void *cp1, const void *cp2)
{
   double t1, t2;
   t1=(*(UNODEPTR *)cp1)->entry;
   t2=(*(UNODEPTR *)cp2)->entry;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if xfer bytes are the same, we sort by url instead */
   return strcmp( (*(UNODEPTR *)cp1)->string,
                  (*(UNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_URL_CMPX - QSort compare URL by exit   */
/*********************************************/

int qs_url_cmpx(const void *cp1, const void *cp2)
{
   double t1, t2;
   t1=(*(UNODEPTR *)cp1)->exit;
   t2=(*(UNODEPTR *)cp2)->exit;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if xfer bytes are the same, we sort by url instead */
   return strcmp( (*(UNODEPTR *)cp1)->string,
                  (*(UNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_REF_CMPH - QSort compare Refs by hits  */
/*********************************************/

int qs_ref_cmph(const void *cp1, const void *cp2)
{
   u_long  t1, t2;
   t1=(*(RNODEPTR *)cp1)->count;
   t2=(*(RNODEPTR *)cp2)->count;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if hits are the same, we sort by referrer URL instead */
   return strcmp( (*(RNODEPTR *)cp1)->string,
                  (*(RNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_AGNT_CMPH - QSort cmp Agents by hits   */
/*********************************************/

int qs_agnt_cmph(const void *cp1, const void *cp2)
{
   u_long  t1, t2;
   t1=(*(ANODEPTR *)cp1)->count;
   t2=(*(ANODEPTR *)cp2)->count;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if hits are the same, we sort by agent string instead */
   return strcmp( (*(ANODEPTR *)cp1)->string,
                  (*(ANODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_SRCH_CMPH - QSort cmp srch str by hits */
/*********************************************/

int qs_srch_cmph(const void *cp1, const void *cp2)
{
   u_long  t1, t2;
   t1=(*(SNODEPTR *)cp1)->count;
   t2=(*(SNODEPTR *)cp2)->count;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if hits are the same, we sort by search string instead */
   return strcmp( (*(SNODEPTR *)cp1)->string,
                  (*(SNODEPTR *)cp2)->string );
}

/*********************************************/
/* QS_IDENT_CMPH - QSort cmp ident by hits   */
/*********************************************/

int qs_ident_cmph(const void *cp1, const void *cp2)
{
   u_long  t1, t2;
   t1=(*(INODEPTR *)cp1)->count;
   t2=(*(INODEPTR *)cp2)->count;
   if (t1!=t2) return (t2<t1)?-1:1;
   /* if hits are the same, sort by ident (username) string instead */
   return strcmp( (*(INODEPTR *)cp1)->string,
                  (*(INODEPTR *)cp2)->string );
}

/*********************************************/
/* LOAD_SITE_ARRAY - load up the sort array  */
/*********************************************/

u_long load_site_array(HNODEPTR *pointer)
{
   HNODEPTR hptr;
   int      i;
   u_long   ctr = 0;

   /* load the array */
   for (i=0;i<MAXHASH;i++)
   {
      hptr=sm_htab[i];
      while (hptr!=NULL)
      {
         if (pointer==NULL) ctr++;       /* fancy way to just count 'em    */
         else *(pointer+ctr++)=hptr;     /* otherwise, really do the load  */
         hptr=hptr->next;
      }
   }
   return ctr;   /* return number loaded */
}

/*********************************************/
/* LOAD_URL_ARRAY - load up the sort array   */
/*********************************************/

u_long load_url_array(UNODEPTR *pointer)
{
   UNODEPTR uptr;
   int      i;
   u_long   ctr = 0;

   /* load the array */
   for (i=0;i<MAXHASH;i++)
   {
      uptr=um_htab[i];
      while (uptr!=NULL)
      {
         if (pointer==NULL) ctr++;       /* fancy way to just count 'em    */
         else *(pointer+ctr++)=uptr;     /* otherwise, really do the load  */
         uptr=uptr->next;
      }
   }
   return ctr;   /* return number loaded */
}

/*********************************************/
/* LOAD_REF_ARRAY - load up the sort array   */
/*********************************************/

u_long load_ref_array(RNODEPTR *pointer)
{
   RNODEPTR rptr;
   int      i;
   u_long   ctr = 0;

   /* load the array */
   for (i=0;i<MAXHASH;i++)
   {
      rptr=rm_htab[i];
      while (rptr!=NULL)
      {
         if (pointer==NULL) ctr++;       /* fancy way to just count 'em    */
         else *(pointer+ctr++)=rptr;     /* otherwise, really do the load  */
         rptr=rptr->next;
      }
   }
   return ctr;   /* return number loaded */
}

/*********************************************/
/* LOAD_AGENT_ARRAY - load up the sort array */
/*********************************************/

u_long load_agent_array(ANODEPTR *pointer)
{
   ANODEPTR aptr;
   int      i;
   u_long   ctr = 0;

   /* load the array */
   for (i=0;i<MAXHASH;i++)
   {
      aptr=am_htab[i];
      while (aptr!=NULL)
      {
         if (pointer==NULL) ctr++;       /* fancy way to just count 'em    */
         else *(pointer+ctr++)=aptr;     /* otherwise, really do the load  */
         aptr=aptr->next;
      }
   }
   return ctr;   /* return number loaded */
}

/*********************************************/
/* LOAD_SRCH_ARRAY - load up the sort array  */
/*********************************************/

u_long load_srch_array(SNODEPTR *pointer)
{
   SNODEPTR sptr;
   int      i;
   u_long   ctr = 0;

   /* load the array */
   for (i=0;i<MAXHASH;i++)
   {
      sptr=sr_htab[i];
      while (sptr!=NULL)
      {
         if (pointer==NULL) ctr++;       /* fancy way to just count 'em    */
         else *(pointer+ctr++)=sptr;     /* otherwise, really do the load  */
         sptr=sptr->next;
      }
   }
   return ctr;   /* return number loaded */
}

/*********************************************/
/* LOAD_IDENT_ARRAY - load up the sort array */
/*********************************************/

u_long load_ident_array(INODEPTR *pointer)
{
   INODEPTR iptr;
   int      i;
   u_long   ctr = 0;

   /* load the array */
   for (i=0;i<MAXHASH;i++)
   {
      iptr=im_htab[i];
      while (iptr!=NULL)
      {
         if (pointer==NULL) ctr++;       /* fancy way to just count 'em    */
         else *(pointer+ctr++)=iptr;     /* otherwise, really do the load  */
         iptr=iptr->next;
      }
   }
   return ctr;   /* return number loaded */
}

/*********************************************/
/* OPEN_OUT_FILE - Open file for output      */
/*********************************************/

FILE *open_out_file(char *filename)
{
   FILE *out_fp;

   /* open the file... */
   if ( (out_fp=fopen(filename,"w")) == NULL)
   {
      if (verbose)
      fprintf(stderr,"%s %s!\n",msg_no_open,filename);
      return NULL;
   }
   return out_fp;
}

/*********************************************/
/* HR_SIZE - Returns a human readable size   */
/*********************************************/

const char *hr_size(unsigned long long int size)
{
   int base=1024;
   char suffixes[]="??KBMBGBTBPBEB";
   float n=size;
   int usesuf=0;

   if (wb_index<0 || wb_index>31)
      wb_index=0;

   while (n>=base && usesuf<=10)
   {
      n/=base;
      usesuf+=2;
   }

   if (usesuf)
      snprintf(warpbuf[wb_index], 31, \
               "%.2f&nbsp;%c%c", n, suffixes[usesuf], suffixes[usesuf+1]);
   else
      snprintf(warpbuf[wb_index], 31, "%d&nbsp;bytes", (unsigned int) size);

   return warpbuf[wb_index++];
}

/*********************************************/
/* Character replacement functions - START   */
/*********************************************/

static void ConstructTransliterationData(struct TransliterationData* data,
                                       const char* SourceCharacters,
                                       const char* const * ReplacementStrings) {
  size_t i;
  for(i=0;i<((size_t)UCHAR_MAX)+1;++i) data->TiedString[i]=NULL;
  while(*SourceCharacters)
    data->TiedString[(unsigned char)*SourceCharacters++] = *ReplacementStrings++;
}

static size_t SizeAfterTransliteration(const struct TransliterationData* data,
                                       const char* begin, const char* end) {
  size_t FinalSize=0;
  
  while(begin!=end) {
    const char* str=data->TiedString[(unsigned char)*begin++];
    if (str)
      FinalSize+=strlen(str);
    else ++FinalSize;
  }
  return FinalSize;
}

static int RawTransliterate(const struct TransliterationData* data,
                            const char* begin, const char* end, 
                            char* OutputString, char* OutputStringEnd) {
/* Note : OutputString can be equal to begin
 * (OutputStringEnd-OutputString) must be the value returned by
 * SizeAfterTransliteration
 * The OutputString buffer (user allocated) must be large enough to contain
 * all these characters
 * The null terminator is not stored
 * returns zero on failure, non-zero otherwise */

  while(end != begin) {
    const char* trans_str;
    --end;
    trans_str=data->TiedString[(unsigned char)*end];
    
    if (trans_str) {
      size_t trans_len=strlen(trans_str);
      if (((size_t)(OutputStringEnd-OutputString)) < trans_len) return 0;
      memcpy(OutputStringEnd-=trans_len, trans_str, trans_len*sizeof(char));
    } else {
      if (OutputStringEnd==OutputString) return 0;
      *--OutputStringEnd=*end;
    }
  }
  return OutputStringEnd==OutputString;
}

extern char* Transliterate(const char* SourceCharacters,
                           const char* const * ReplacementStrings,
                           const char* String) {
/* allocates a block of memory with malloc
 * The caller is responsible of freeing it with free()
 * returns NULL on failure */
 
  struct TransliterationData data;
  size_t InitialSize,FinalSize;
  char* OutputString;
  
  ConstructTransliterationData(&data, SourceCharacters, ReplacementStrings);
  InitialSize=strlen(String);
  FinalSize=SizeAfterTransliteration(&data, String, String+InitialSize);
  
  if (NULL==(OutputString=malloc((FinalSize+1)*sizeof(char))))
    return NULL;
  if (!RawTransliterate(&data, String, String+InitialSize, OutputString, 
      OutputString+FinalSize)) {
    free(OutputString);
    return NULL;
  }
  OutputString[FinalSize]='\0';
  return OutputString;
}

/*********************************************/
/* Character replacement functions - END     */
/*********************************************/
